/*
 *   Copyright (C) 2007-2009 Tristan Heaven <tristanheaven@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include <stdlib.h>
#include <stdbool.h>
#include <gio/gio.h>
#include <gtk/gtk.h>
#include <mhash.h>

#include "properties.h"
#include "properties-hash.h"

#define BUFFER_SIZE (1024 * 256) // File reading buffer

static const hashid hash_ids[HASH_FUNCS_N] = {
	MHASH_MD2, MHASH_MD4, MHASH_MD5,
	MHASH_SHA1, MHASH_SHA224, MHASH_SHA256, MHASH_SHA384, MHASH_SHA512,
	MHASH_RIPEMD128, MHASH_RIPEMD160, MHASH_RIPEMD256, MHASH_RIPEMD320,
	MHASH_HAVAL128, MHASH_HAVAL160, MHASH_HAVAL192, MHASH_HAVAL224, MHASH_HAVAL256,
	MHASH_TIGER128, MHASH_TIGER160, MHASH_TIGER192,
	MHASH_GOST,
	MHASH_WHIRLPOOL,
	MHASH_SNEFRU128, MHASH_SNEFRU256,
	MHASH_CRC32, MHASH_CRC32B,
	MHASH_ADLER32
};

static const char *hash_names[HASH_FUNCS_N] = {
	"MD2", "MD4", "MD5",
	"SHA1", "SHA224", "SHA256", "SHA384", "SHA512",
	"RIPEMD128", "RIPEMD160", "RIPEMD256", "RIPEMD320",
	"HAVAL128", "HAVAL160", "HAVAL192", "HAVAL224", "HAVAL256",
	"TIGER128", "TIGER160", "TIGER192",
	"GOST",
	"WHIRLPOOL",
	"SNEFRU128", "SNEFRU256",
	"CRC32", "CRC32B",
	"ADLER32"
};

int gtkhash_properties_hash_get_pos_from_name(const char *name)
{
	for (int i = 0; i < HASH_FUNCS_N; i++)
		if (g_strcmp0(name, hash_names[i]) == 0)
			return i;

	g_warning("get_pos_from_name: unknown hash name '%s'", name);
	return -1;
}

const char *gtkhash_properties_hash_get_name_from_pos(const int pos)
{
	return hash_names[pos];
}

void gtkhash_properties_hash_set_digest(struct page_s *page, const int pos,
	char *digest)
{
	if (page->hash.funcs[pos].digest)
		g_free(page->hash.funcs[pos].digest);

	page->hash.funcs[pos].digest = digest;
}

const char *gtkhash_properties_hash_get_digest(struct page_s *page,
	const int pos)
{
	if (page->hash.funcs[pos].digest)
		return page->hash.funcs[pos].digest;
	else
		return "";
}

void gtkhash_properties_hash_clear_digests(struct page_s *page)
{
	for (int i = 0; i < HASH_FUNCS_N; i++)
		gtkhash_properties_hash_set_digest(page, i, NULL);
}

unsigned int gtkhash_properties_hash_get_source(struct page_s *page)
{
	g_mutex_lock(page->hash.priv.mutex);
	unsigned int source = page->hash.priv.source;
	g_mutex_unlock(page->hash.priv.mutex);

	return source;
}

static void gtkhash_properties_hash_set_source(struct page_s *page,
	const unsigned int source)
{
	g_mutex_lock(page->hash.priv.mutex);
	page->hash.priv.source = source;
	g_mutex_unlock(page->hash.priv.mutex);
}

void gtkhash_properties_hash_add_source(struct page_s *page)
{
	g_mutex_lock(page->hash.priv.mutex);
	if (G_UNLIKELY(page->hash.priv.source != 0))
		g_critical("source was already added");
	page->hash.priv.source = gdk_threads_add_idle(
		(GSourceFunc)gtkhash_properties_hash_file, page);
	g_mutex_unlock(page->hash.priv.mutex);
}

static void gtkhash_properties_hash_remove_source(struct page_s *page)
{
	g_mutex_lock(page->hash.priv.mutex);
	if (G_UNLIKELY(!g_source_remove(page->hash.priv.source)))
		g_critical("failed to remove source");
	page->hash.priv.source = 0;
	g_mutex_unlock(page->hash.priv.mutex);
}

bool gtkhash_properties_hash_get_stop(struct page_s *page)
{
	g_mutex_lock(page->hash.priv.mutex);
	bool stop = page->hash.priv.stop;
	g_mutex_unlock(page->hash.priv.mutex);

	return stop;
}

void gtkhash_properties_hash_set_stop(struct page_s *page, const bool stop)
{
	g_mutex_lock(page->hash.priv.mutex);
	page->hash.priv.stop = stop;
	g_mutex_unlock(page->hash.priv.mutex);
}

enum hash_state_e gtkhash_properties_hash_get_state(struct page_s *page)
{
	g_mutex_lock(page->hash.priv.mutex);
	enum hash_state_e state = page->hash.priv.state;
	g_mutex_unlock(page->hash.priv.mutex);

	return state;
}

void gtkhash_properties_hash_set_state(struct page_s *page,
	const enum hash_state_e state)
{
	g_mutex_lock(page->hash.priv.mutex);
	page->hash.priv.state = state;
	g_mutex_unlock(page->hash.priv.mutex);
}

static char *gtkhash_properties_hash_end(struct page_s *page, const int pos)
{
	unsigned char *bin = mhash_end_m(page->hash.funcs[pos].thread, g_malloc);
	GString *digest = g_string_sized_new(128);

	for (unsigned int i = 0; i < mhash_get_block_size(hash_ids[pos]); i++)
		g_string_append_printf(digest, "%.2x", bin[i]);

	g_free(bin);

	return g_string_free(digest, false);
}

static void gtkhash_properties_hash_deinit_all(struct page_s *page)
{
	for (int i = 0; i < HASH_FUNCS_N; i++) {
		if (!page->hash.funcs[i].enabled)
			continue;
		if (page->hash.funcs[i].thread == MHASH_FAILED)
			return;
		mhash_deinit(page->hash.funcs[i].thread, NULL);
	}
}

static void gtkhash_properties_hash_file_init(struct page_s *page)
{
	for (int i = 0; i < HASH_FUNCS_N; i++) {
		if (!page->hash.funcs[i].enabled)
			continue;
		page->hash.funcs[i].thread = mhash_init(hash_ids[i]);
		if (page->hash.funcs[i].thread == MHASH_FAILED) {
			g_critical("failed to init func %d", i);
			gtkhash_properties_hash_set_stop(page, true);
			gtkhash_properties_hash_set_state(page, HASH_STATE_END);
			return;
		}
	}

	page->hash.buffer = g_malloc(BUFFER_SIZE);
	page->hash.file = g_file_new_for_uri(page->uri);
	page->hash.func_pos = 0;
	page->hash.just_read = 0;
	page->hash.total_read = 0;
	page->hash.timer = g_timer_new();

	gtkhash_properties_hash_set_state(page, HASH_STATE_OPEN);
}

static void gtkhash_properties_hash_file_open_finish(
	G_GNUC_UNUSED GObject *source, GAsyncResult *res, struct page_s *page)
{
	page->hash.stream = g_file_read_finish(page->hash.file, res, NULL);
	if (!page->hash.stream) {
		g_warning("file_open_finish: failed to open file");
		gtkhash_properties_hash_set_stop(page, true);
	}

	if (gtkhash_properties_hash_get_stop(page))
		if (page->hash.stream)
			gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
		else
			gtkhash_properties_hash_set_state(page, HASH_STATE_END);
	else
		gtkhash_properties_hash_set_state(page, HASH_STATE_GET_SIZE);

	gtkhash_properties_hash_add_source(page);
}

static void gtkhash_properties_hash_file_open(struct page_s *page)
{
	if (gtkhash_properties_hash_get_stop(page)) {
		gtkhash_properties_hash_set_state(page, HASH_STATE_END);
		return;
	}

	gtkhash_properties_hash_remove_source(page);
	g_file_read_async(page->hash.file, G_PRIORITY_DEFAULT, NULL,
		(GAsyncReadyCallback)gtkhash_properties_hash_file_open_finish,
		page);
}

static void gtkhash_properties_hash_file_get_size_finish(
	G_GNUC_UNUSED GObject *source, GAsyncResult *res, struct page_s *page)
{
	GFileInfo *info = g_file_input_stream_query_info_finish(
		page->hash.stream, res, NULL);
	page->hash.file_size = g_file_info_get_size(info);
	g_object_unref(info);

	if (gtkhash_properties_hash_get_stop(page))
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
	else if (page->hash.file_size == 0)
		gtkhash_properties_hash_set_state(page, HASH_STATE_HASH);
	else
		gtkhash_properties_hash_set_state(page, HASH_STATE_READ);

	gtkhash_properties_hash_add_source(page);
}

static void gtkhash_properties_hash_file_get_size(struct page_s *page)
{
	if (gtkhash_properties_hash_get_stop(page)) {
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
		return;
	}

	gtkhash_properties_hash_remove_source(page);
	g_file_input_stream_query_info_async(page->hash.stream,
		G_FILE_ATTRIBUTE_STANDARD_SIZE, G_PRIORITY_DEFAULT, NULL,
		(GAsyncReadyCallback)gtkhash_properties_hash_file_get_size_finish,
		page);
}

static void gtkhash_properties_hash_file_read_finish(
	G_GNUC_UNUSED GObject *source, GAsyncResult *res, struct page_s *page)
{
	page->hash.just_read = g_input_stream_read_finish(
		G_INPUT_STREAM(page->hash.stream), res, NULL);

	if (G_UNLIKELY(page->hash.just_read == -1)) {
		g_warning("file_read_finish: failed to read file");
		gtkhash_properties_hash_set_stop(page, true);
	} else if (G_UNLIKELY(page->hash.just_read == 0)) {
		g_warning("file_read_finish: unexpected EOF");
		gtkhash_properties_hash_set_stop(page, true);
	} else {
		page->hash.total_read += page->hash.just_read;
		if (G_UNLIKELY(page->hash.total_read > page->hash.file_size)) {
			g_warning("file_read_finish: read %" G_GOFFSET_FORMAT
				" bytes more than expected",
				page->hash.total_read - page->hash.file_size);
			gtkhash_properties_hash_set_stop(page, true);
		} else
			gtkhash_properties_hash_set_state(page, HASH_STATE_HASH);
	}

	if (G_UNLIKELY(gtkhash_properties_hash_get_stop(page)))
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);

	gtkhash_properties_hash_add_source(page);
}

static void gtkhash_properties_hash_file_read(struct page_s *page)
{
	if (G_UNLIKELY(gtkhash_properties_hash_get_stop(page))) {
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
		return;
	}

	gtkhash_properties_hash_remove_source(page);
	g_input_stream_read_async(G_INPUT_STREAM(page->hash.stream),
		page->hash.buffer, BUFFER_SIZE, G_PRIORITY_DEFAULT, NULL,
		(GAsyncReadyCallback)gtkhash_properties_hash_file_read_finish,
		page);
}

static void gtkhash_properties_hash_file_hash(struct page_s *page)
{
	if (G_UNLIKELY(gtkhash_properties_hash_get_stop(page))) {
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
		return;
	} else if (page->hash.func_pos < HASH_FUNCS_N) {
		if (page->hash.funcs[page->hash.func_pos].enabled)
			mhash(page->hash.funcs[page->hash.func_pos].thread,
				page->hash.buffer, page->hash.just_read);
		page->hash.func_pos++;
		return;
	}

	page->hash.func_pos = 0;

	gtkhash_properties_hash_set_state(page, HASH_STATE_REPORT);
}

static void gtkhash_properties_hash_file_report(struct page_s *page)
{
	if (page->hash.total_read >= page->hash.file_size)
		gtkhash_properties_hash_set_state(page, HASH_STATE_CLOSE);
	else
		gtkhash_properties_hash_set_state(page, HASH_STATE_READ);

	if (page->hash.file_size == 0)
		return;

	gtk_progress_bar_set_fraction(page->progressbar,
		(double)page->hash.total_read /
		(double)page->hash.file_size);

	if (page->hash.total_read < BUFFER_SIZE * 3)
		return;

	// Update progres bar text...
	int remaining = g_timer_elapsed(page->hash.timer, NULL) /
		page->hash.total_read *
		(page->hash.file_size - page->hash.total_read);
	char *text;
	if (remaining > 60) {
		int minutes = remaining / 60;
		if (minutes == 1)
			text = g_strdup_printf(_("Estimated time remaining: %d minute"),
				minutes);
		else
			text = g_strdup_printf(_("Estimated time remaining: %d minutes"),
				minutes);
	} else {
		if (remaining == 1)
			text = g_strdup_printf(_("Estimated time remaining: %d second"),
				remaining);
		else
			text = g_strdup_printf(_("Estimated time remaining: %d seconds"),
				remaining);
	}
	gtk_progress_bar_set_text(page->progressbar, text);
	g_free(text);
}

static void gtkhash_properties_hash_file_close_finish(
	G_GNUC_UNUSED GObject *source, GAsyncResult *res, struct page_s *page)
{
	if (!g_input_stream_close_finish(G_INPUT_STREAM(page->hash.stream),
		res, NULL))
	{
		g_warning("file_close_finish: failed to close file");
	}

	g_object_unref(page->hash.stream);

	gtkhash_properties_hash_set_state(page, HASH_STATE_END);
	gtkhash_properties_hash_add_source(page);
}

static void gtkhash_properties_hash_file_close(struct page_s *page)
{
	gtkhash_properties_hash_remove_source(page);
	g_input_stream_close_async(G_INPUT_STREAM(page->hash.stream),
		G_PRIORITY_DEFAULT, NULL,
		(GAsyncReadyCallback)gtkhash_properties_hash_file_close_finish,
		page);
}

static void gtkhash_properties_hash_file_end(struct page_s *page)
{
	if (G_UNLIKELY(gtkhash_properties_hash_get_stop(page))) {
		gtkhash_properties_hash_deinit_all(page);
	} else if (page->hash.func_pos < HASH_FUNCS_N) {
		if (page->hash.funcs[page->hash.func_pos].enabled) {
			char *digest = gtkhash_properties_hash_end(page,
				page->hash.func_pos);
			gtkhash_properties_hash_set_digest(page,
				page->hash.func_pos, digest);
		}
		page->hash.func_pos++;
		return;
	}

	gtkhash_properties_hash_set_state(page, HASH_STATE_DONE);
}

static void gtkhash_properties_hash_file_done(struct page_s *page)
{
	g_timer_destroy(page->hash.timer);
	g_free(page->hash.buffer);
	g_object_unref(page->hash.file);

	gtkhash_properties_idle(page);
	gtkhash_properties_hash_set_source(page, 0);
}

bool gtkhash_properties_hash_file(struct page_s *page)
{
	static void (* const hash_state_funcs[])(struct page_s *) = {
		gtkhash_properties_hash_file_init,
		gtkhash_properties_hash_file_open,
		gtkhash_properties_hash_file_get_size,
		gtkhash_properties_hash_file_read,
		gtkhash_properties_hash_file_hash,
		gtkhash_properties_hash_file_report,
		gtkhash_properties_hash_file_close,
		gtkhash_properties_hash_file_end,
		gtkhash_properties_hash_file_done
	};

	const enum hash_state_e state = gtkhash_properties_hash_get_state(page);

	// Call func
	hash_state_funcs[state](page);

	if (state == HASH_STATE_DONE)
		return false;
	else
		return true;
}

void gtkhash_properties_hash_init(struct page_s *page)
{
	page->hash.priv.mutex = g_mutex_new();
	page->hash.priv.source = 0;
	page->hash.priv.state = HASH_STATE_DONE;
	page->hash.priv.stop = false;

	for (int i = 0; i < HASH_FUNCS_N; i++)
		page->hash.funcs[i].digest = NULL;
}

void gtkhash_properties_hash_deinit(struct page_s *page)
{
	gtkhash_properties_hash_clear_digests(page);

	g_mutex_free(page->hash.priv.mutex);
}
