/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright (C) 2002-2010 Rochet Sylvain
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define GTK_ENABLE_BROKEN = 1
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <gtk/gtk.h>
#include <libxml/parser.h>

#include "engine.h"

#include "xmlparse.h"
#include "game.h"
#include "client.h"
#include "interface.h"
#include "global.h"
#include "load.h"
#include "callback.h"
#include "display.h"
#include "trade.h"

/*
ok	server
ok	client
ok	msg
ok	display
dep	updateplayerlist
ok	updategamelist
ok	playerupdate
ok	deleteplayer
ok	estateupdate
ok	cardupdate
ok	estategroupupdate
ok	tradeupdate
ok	auctionupdate
ok	gameupdate
dep	commandlist
ok	configupdate
*/

/*
ok	.gn
ok	.gl
ok	.gj
no-need	.gd
ok	.gp
ok	.gs
ok	.gx
ok	.Tn
ok	.Tc
ok	.Te
ok	.Tm
ok	.Ta
ok	.Tr
ok	.r
ok	.E
ok	.t
ok	.D
no-need	.p
ok	.eb
ok	.ea
ok	.hb
ok	.hs
ok	.em
ok	.ab
ok	.jc
ok	.jp
ok	.jr
ok	.T$
ok	.T%
no-need	.f
ok	.n
no-need	.d
ok	.R
no-need	.pi
*/


void xmlparse_getgamelist_plugger(guint32 connectid, gchar *buffer)  {

	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseMemory(buffer, strlen(buffer) );

	if(doc == NULL)  {

		printf("XML ERROR ON: %s %d\n", connection[connectid]->host, connection[connectid]->port);
		return;
	}

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, SERVER_XMLROOTELEMENT) )  {
		xmlFreeDoc(doc);
		return;
	}

	for( cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"server") )  xmlparse_server(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"updategamelist") )  xmlparse_updategamelist(connectid, doc, cur);
	}

	xmlFreeDoc(doc);
}




void xmlparse_game_plugger(guint32 connectid, gchar *buffer)  {

	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseMemory(buffer, strlen(buffer) );

	if(doc == NULL)   return;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, SERVER_XMLROOTELEMENT) )  {
		xmlFreeDoc(doc);
		return;
	}

	for(cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"gameupdate") )  xmlparse_gameupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"server") )  xmlparse_server(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"client") )  xmlparse_client(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"msg") )  xmlparse_message(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"estateupdate") )  xmlparse_estateupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"playerupdate") )  xmlparse_playerupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"auctionupdate") )  xmlparse_auctionupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"display") )  xmlparse_display(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"configupdate") )  xmlparse_configupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"tradeupdate") )  xmlparse_tradeupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"cardupdate") )  xmlparse_cardupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"estategroupupdate") )  xmlparse_estategroupupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"deleteplayer") )  xmlparse_deleteplayer(connectid, doc, cur);
	}

	xmlFreeDoc(doc);
}




void xmlparse_server(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;

	tmp = xmlGetProp(cur, (xmlChar*)"version");
	if(tmp)  connection[connectid]->server_version = (gchar*)tmp;
}




void xmlparse_client(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;

	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if( tmp )  {
		game->my_playerid = atoi( (gchar*)tmp );
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"cookie");
	if( tmp )  {
		game_write_cookie( (gchar*)tmp );
		g_free(tmp);
	}
}




void xmlparse_gameupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur) {

	xmlChar *tmp;
	struct timeval tv;
	guint32 i;
	gint32 gameid;
	
	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	if(tmp)  {
		gameid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"status");

	if(! tmp) return;

	if( game->gameid && game->gameid != gameid) {
		printf("gameid mismatch, this should not happen\n");
		abort();
	}
	// We only get the status if we are in the game
	game->gameid = gameid;

	if(! xmlStrcmp(tmp, (xmlChar*)"config") ) {

		if(game->status != GAME_STATUS_CONFIG)  {

			game->status = GAME_STATUS_CONFIG;
			interface_create_gameconfigpage();
		}
	}

	if(! xmlStrcmp(tmp, (xmlChar*)"init") )  {

		if(game->status != GAME_STATUS_INIT)  {

			game->status = GAME_STATUS_INIT;
			interface_create_gameinitpage();
			if(! game_load_pngs() ) exit(-1);
		}
	}

	if(! xmlStrcmp(tmp, (xmlChar*)"run") )  {

		if(game->status != GAME_STATUS_RUN)  {

			gettimeofday(&tv, NULL);
			game->start_time = tv.tv_sec;

			game->status = GAME_STATUS_RUN;
			interface_create_gameboardpage();
		}
	}

	if(! xmlStrcmp(tmp, (xmlChar*)"end") )  {

		if(game->status != GAME_STATUS_END)  {

			game->status = GAME_STATUS_END;

			/* remove all commands */
			if(game->CommandsBox)  gtk_widget_destroy(game->CommandsBox);
			game->CommandsBox = 0;
			display_clear_buttons();

			/* destroy all open trades */
			for(i = 0 ; i < MAX_TRADES ; i++)  {

				trade_destroy_slot(i);
			}
		}
	}

	g_free(tmp);
}




void xmlparse_message(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	gchar *type, *text, *author, *value;

	type = (gchar*)xmlGetProp(cur, (xmlChar*)"type");

	if(! g_strncasecmp(type, "chat", 4) )  {

		author = (gchar*)xmlGetProp(cur, (xmlChar*)"author");
		value = (gchar*)xmlGetProp(cur, (xmlChar*)"value");
		if(value && author)  {

			if(! strncmp("[ACTION]", value, 8) )
				text = g_strdup_printf("* %s%s", author, value + 8);
			else if(! strncmp("/me", value, 3) )
				text = g_strdup_printf("* %s%s", author, value + 3);
			else
				text = g_strdup_printf("<%s> %s", author, value);

			text_insert_chat(text, strlen(text));
			g_free(text);
		}
		if(value[0] == '!')  parse_specific_chat_message(value);
		if(author)  g_free(author);
		if(value)  g_free(value);
	}
	else {

		g_strup( type );
		value = (gchar*)xmlGetProp(cur, (xmlChar*)"value");
		text = g_strdup_printf("%s: %s", type, value);
		text_insert_message(text, strlen(text));
		g_free(text);
		g_free(value);
	}

	g_free(type);
}




void xmlparse_estateupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	guint32 color[3], i;
	xmlChar *tmp;
	guint32 id, frame, pic, star;
	gint32 t, u;
	gboolean owner_mortgage_changed = FALSE;

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(!tmp)  return;
	id = atoi( (gchar*)tmp );
	g_free(tmp);
	if(id < 0) return;

	tmp = xmlGetProp(cur, (xmlChar*)"name");
	if(tmp)  game->estate[id].name = (gchar*)tmp;

	tmp = xmlGetProp(cur, (xmlChar*)"color");
	if(tmp) {

		sscanf((gchar*)tmp, "#%2X%2X%2X", &color[0], &color[1], &color[2]);
		game->estate[id].color[0] = color[0];
		game->estate[id].color[1] = color[1];
		game->estate[id].color[2] = color[2];
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"bgcolor");
	if(tmp) {

		sscanf((gchar*)tmp, "#%2X%2X%2X", &color[0], &color[1], &color[2]);
		game->estate[id].bgcolor[0] = color[0];
		game->estate[id].bgcolor[1] = color[1];
		game->estate[id].bgcolor[2] = color[2];
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"owner");
	if(tmp)  {

		t = game->estate[id].owner = atoi((gchar*)tmp);
		g_free(tmp);
		owner_mortgage_changed = TRUE;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"mortgaged");
	if(tmp)  {

		u = game->estate[id].mortgaged = atoi((gchar*)tmp);
		g_free(tmp);
		owner_mortgage_changed = TRUE;
	}

	if(owner_mortgage_changed)  {

		t = game->estate[id].owner;
		u = game->estate[id].mortgaged;

		/* reset unowned cards of this estate to all players */
		for(i = 0 ; i < MAX_PLAYERS ; i++)  {

			if(! global->player[i].playerid) continue;
			if(global->player[i].game != game->gameid)  continue;

			frame = global->player[i].playerlist_cards_frame;
			pic = global->player[i].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			pic_set_alpha(frame, pic, data->playerlist_cards_alphaunowned);
		}

		if(t > 0  &&  !u)  {

			/* star */
			frame = game->board_frame;
			star = global->player[ get_player_slot_with_playerid(t) ].buffer_star;

			pic = game->estate[id].star_alpha;
			pic_set_width(frame, pic, data->pngfile_star_width[star]);
			pic_set_height(frame, pic, data->pngfile_star_height[star]);
			pic_buff(frame, pic, data->pngfile_star_alpha[star]);

			pic = game->estate[id].star_pic;
			pic_set_width(frame, pic, data->pngfile_star_width[star]);
			pic_set_height(frame, pic, data->pngfile_star_height[star]);
			pic_buff(frame, pic, data->pngfile_star_image[star]);
			pic_show(frame, pic);

			/* playerlist card */
			frame = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_frame;
			pic = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			pic_set_bgcolor(frame, pic, data->playerlist_cards_cardbgcolor);
			pic_set_alpha(frame, pic, data->playerlist_cards_alphaowned);
		}
		else  if(t > 0  &&  u)  {

			/* star */
			frame = game->board_frame;
			star = global->player[ get_player_slot_with_playerid(t) ].buffer_star;

			pic = game->estate[id].star_alpha;
			pic_set_width(frame, pic, data->pngfile_star_m_width[star]);
			pic_set_height(frame, pic, data->pngfile_star_m_height[star]);
			pic_buff(frame, pic, data->pngfile_star_m_alpha[star]);

			pic = game->estate[id].star_pic;
			pic_set_width(frame, pic, data->pngfile_star_m_width[star]);
			pic_set_height(frame, pic, data->pngfile_star_m_height[star]);
			pic_buff(frame, pic, data->pngfile_star_m_image[star]);
			pic_show(frame, pic);

			/* playerlist card */
			frame = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_frame;
			pic = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			pic_set_bgcolor(frame, pic, data->playerlist_cards_cardbgcolormortgage);
			pic_set_alpha(frame, pic, data->playerlist_cards_alphamortgage);
		}
		else /* if( t <= 0 ) */ {

			/* star */
			frame = game->board_frame;
			pic_unshow(frame, game->estate[id].star_pic);
		}

		/* update estatelist in trade panel */
		for(i = 0 ; i < MAX_TRADES ; i++)  {

			if(! game->trade[i].open)  continue;
			if(game->trade[i].current_component != TRADE_CURRENT_COMPONENT_ESTATE)  continue;

			trade_rebuild_subcomponent(i);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"houses");
	if(tmp)  {

		t = game->estate[id].houses = atoi((gchar*)tmp);
 		g_free(tmp);

		frame = game->board_frame;

		if(t <= 0)   {

			/* houses */
			pic = game->estate[id].house_pic;
			pic_unshow(frame, pic);
		}
		else  if(t > 0  &&  data->estate[id].type_house == TYPE_HOUSE_HORIZONTAL)  {

			/* houses */
			pic = game->estate[id].house_alpha;
			pic_set_width(frame, pic, data->pngfile_horiz_house_width[t]);
			pic_set_height(frame, pic, data->pngfile_horiz_house_height[t]);
			pic_set_buff(frame, pic, data->pngfile_horiz_house_alpha[t]);

			pic = game->estate[id].house_pic;
			pic_set_width(frame, pic, data->pngfile_horiz_house_width[t]);
			pic_set_height(frame, pic, data->pngfile_horiz_house_height[t]);
			pic_set_buff(frame, pic, data->pngfile_horiz_house_image[t]);
			pic_show(frame, pic);
		}
		else  if(t > 0  &&  data->estate[id].type_house == TYPE_HOUSE_VERTICAL)  {

			/* houses */
			pic = game->estate[id].house_alpha;
			pic_set_width(frame, pic, data->pngfile_vert_house_width[t]);
			pic_set_height(frame, pic, data->pngfile_vert_house_height[t]);
			pic_set_buff(frame, pic, data->pngfile_vert_house_alpha[t]);

			pic = game->estate[id].house_pic;
			pic_set_width(frame, pic, data->pngfile_vert_house_width[t]);
			pic_set_height(frame, pic, data->pngfile_vert_house_height[t]);
			pic_set_buff(frame, pic, data->pngfile_vert_house_image[t]);
			pic_show(frame, pic);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"houseprice");
	if(tmp)  {
		game->estate[id].houseprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"sellhouseprice");
	if(tmp)  {
		game->estate[id].sellhouseprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"mortgageprice");
	if(tmp)  {
		game->estate[id].mortgageprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"unmortgageprice");
	if(tmp)  {
		game->estate[id].unmortgageprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"group");
	if(tmp)  {
		game->estate[id].group = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_be_owned");
	if(tmp)  {
		game->estate[id].can_be_owned = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_toggle_mortgage");
	if(tmp)  {
		game->estate[id].can_toggle_mortgage = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_buy_houses");
	if(tmp)  {
		game->estate[id].can_buy_houses = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_sell_houses");
	if(tmp)  {
		game->estate[id].can_sell_houses = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"money");
	if(tmp)  {
		game->estate[id].money = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"price");
	if(tmp)  {
		game->estate[id].price = atoi((gchar*)tmp);
		g_free(tmp);
	}

	for(i = 0 ; i <= 5 ; i++) {

		xmlChar *key = (xmlChar*)g_strdup_printf("rent%d", i);
		tmp = xmlGetProp(cur, key);

		if(tmp)  {
			game->estate[id].rent[i] = atoi((gchar*)tmp);
			g_free(tmp);
		}
		g_free(key);
	}
}




void xmlparse_playerupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	gint32 playerid, i;
	gint32 id;
	xmlChar *tmp;
	GtkWidget *Combo, *StartButton, *Button;
	GList *list;


	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if(!tmp) return;
 	playerid = atoi((gchar*)tmp);
	g_free(tmp);
	if(!playerid) return;

	id = get_player_slot_with_playerid(playerid);

	/* create new player */
	if(id < 0)  {
		if(game->status <= GAME_STATUS_CONFIG)  {

			if(! game_get_valid_player_slot(&id) ) return;

			memset(&global->player[id], 0, sizeof(_player) );
			global->player[id].playerid = playerid;
		}
		else return;
	}

	if(id >= 0 && game->status > GAME_STATUS_CONFIG && global->player[id].game != game->gameid) return;

	tmp = xmlGetProp(cur, (xmlChar*)"game");
	if(tmp)  {
	
		global->player[id].game = atoi((gchar*)tmp);
		g_free(tmp);
	}


	tmp = xmlGetProp(cur, (xmlChar*)"host");
	if(tmp)  {

		if(global->player[id].host) g_free(global->player[id].host);
		global->player[id].host = (gchar*)tmp;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"master");
	if(tmp)  {

		global->player[id].master = atoi((gchar*)tmp);
		g_free(tmp);

		if(global->player[id].master  &&  game->my_playerid == playerid)
			game->i_am_the_master = TRUE;
		if(!global->player[id].master  &&  game->my_playerid == playerid)
			game->i_am_the_master = FALSE;

		/* set sensitive mode of start button */
		if(game->status == GAME_STATUS_CONFIG)  {

			StartButton = gtk_object_get_data(GTK_OBJECT(global->MainVerticalBox), "start_button");
			gtk_widget_set_sensitive(StartButton, game->i_am_the_master);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"image");
	if(tmp)  {

		if(global->player[id].image) g_free(global->player[id].image);
		global->player[id].image = (gchar*)tmp;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"name");
	if(tmp)  {

		if(global->player[id].name) g_free(global->player[id].name);
		global->player[id].name = g_strndup((gchar*)tmp, MAX_LENGTH_NICK);
		g_free(tmp);

		if(! strcmp(global->player[id].name, "_metaserver_"))  {
			game_delete_player(id);
			return;
		}

		/* playerlist name */
		if(global->player[id].playerlist_LabelNamePlayer)
			gtk_label_set_text(GTK_LABEL(global->player[id].playerlist_LabelNamePlayer), global->player[id].name);

		/* popdown string trade combo */
		Combo = gtk_object_get_data(GTK_OBJECT(global->MainVerticalBox), "trade_combo_listplayer");
		if(Combo)  {

			list = NULL;
			for(i = 0 ; i < MAX_PLAYERS ; i++)  {

				if(! global->player[i].playerid) continue;
				if(global->player[i].game != game->gameid)  continue;
				if(global->player[i].playerid  ==  game->my_playerid) continue;

				list = g_list_append(list, global->player[i].name);
			}
			gtk_combo_set_popdown_strings(GTK_COMBO(Combo), list);
			g_list_free(list);
		}

		/* playerlist in trade panel */
		for(i = 0 ; i < MAX_TRADES ; i++)  {

			if(! game->trade[i].open)  continue;

			trade_rebuild_playerlist(i);
		}

		/* update sub component players in trade panel */
		for(i = 0 ; i < MAX_TRADES ; i++)  {

			if(! game->trade[i].open)  continue;

			trade_rebuild_subcomponent(i);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"money");
	if(tmp)  {

		global->player[id].money = atoi((gchar*)tmp);

		if(global->player[id].playerlist_LabelMoneyPlayer)
			gtk_label_set_text(GTK_LABEL(global->player[id].playerlist_LabelMoneyPlayer), (gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"location");
	if(tmp)  {

		global->player[id].location_to = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"jailed");
	if(tmp)  {
		global->player[id].jailed = atoi((gchar*)tmp);
		g_free(tmp);

		game_update_tokens();
	}

	tmp = xmlGetProp(cur, (xmlChar*)"directmove");
	if(tmp)  {
		global->player[id].directmove = atoi((gchar*)tmp);
		g_free(tmp);

		game_start_move_token(global->player[id].playerid);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"bankrupt");
	if(tmp)  {
		global->player[id].bankrupt = atoi((gchar*)tmp);
		g_free(tmp);

		/* remove token */
		if(global->player[id].bankrupt)
			pic_unshow(game->board_frame, global->player[id].token_pic);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"hasturn");
	if(tmp)  {
		global->player[id].hasturn = atoi((gchar*)tmp);
		g_free(tmp);

		/* set has turn attributes */
		if(global->player[id].hasturn)  {

			if(global->player[id].playerlist_LabelNamePlayer)  {

				gtk_widget_set_style( gtk_object_get_data(GTK_OBJECT(global->player[id].playerlist_AlignementNamePlayer), "namelabel") ,  global->style_player_turn);
				gtk_widget_set_style( gtk_object_get_data(GTK_OBJECT(global->player[id].playerlist_AlignementMoneyPlayer), "moneylabel") ,  global->style_player_turn);
			}

			pic_unset_alpha(game->board_frame, global->player[id].token_pic);
		}
		else  {

			if(global->player[id].playerlist_LabelNamePlayer)  {

				gtk_widget_set_style( gtk_object_get_data(GTK_OBJECT(global->player[id].playerlist_AlignementNamePlayer), "namelabel") ,  global->style_player_normal);
				gtk_widget_set_style( gtk_object_get_data(GTK_OBJECT(global->player[id].playerlist_AlignementMoneyPlayer), "moneylabel") ,  global->style_player_normal);
			}

			if(config->game_token_transparency)
				pic_set_alpha(game->board_frame, global->player[id].token_pic, 0x7f);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_roll");
	if(tmp)  {
		global->player[id].can_roll = atoi((gchar*)tmp);
		g_free(tmp);

		if(game->status == GAME_STATUS_RUN)  {

			Button = gtk_object_get_data(GTK_OBJECT(game->CommandsBox), "roll_button");

			if(global->player[id].can_roll)
				gtk_widget_show_all(Button);
			else
				gtk_widget_hide(Button);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_buyestate");
	if(tmp)  {
		global->player[id].can_buyestate = atoi((gchar*)tmp);
		g_free(tmp);
	}

	game_buildplayerlist();
}




void xmlparse_auctionupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	GtkWidget *text, *entry, *box;
	GtkTextBuffer *textbuff;
	GtkTextIter textiter;
	GtkTextMark *textmark;

	guint32 actor = 0, auctionid, estateid, highbid = 0, highbidder, status;
	gchar *ptr, *message;

	xmlChar *tmp;

	tmp = xmlGetProp(cur, (xmlChar*)"auctionid");
	if(!tmp) return;
	auctionid = atoi((gchar*)tmp);
	g_free(tmp);

	/* initialise auction */
	tmp = xmlGetProp(cur, (xmlChar*)"actor");
	if(tmp)  {
		actor = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(tmp)  {

		estateid = atoi((gchar*)tmp);
		g_free(tmp);

		display_hide();
		interface_create_auctionbox();
		gtk_object_set_data(GTK_OBJECT(game->BoardCenter), "auction_highbid", 0);

		text = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_text");
		entry = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_entry");

		ptr = g_strdup_printf("%d", auctionid);
		gtk_object_set_data_full(GTK_OBJECT(entry), "auctionid", ptr, g_free);

		message = g_strdup_printf("%s auction estate %s", global->player[ get_player_slot_with_playerid( actor ) ].name, game->estate[ estateid ].name);

		textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
		gtk_text_buffer_get_end_iter(textbuff, &textiter);
		gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
		gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);


		g_free(message);
	}

	/* bid */
	tmp = xmlGetProp(cur, (xmlChar*)"highbid");
	if(tmp)  {
		highbid = atoi((gchar*)tmp);
		gtk_object_set_data(GTK_OBJECT(game->BoardCenter), "auction_highbid", GINT_TO_POINTER(highbid));
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"highbidder");
	if(tmp)  {

		highbidder = atoi((gchar*)tmp);
		g_free(tmp);

		message = g_strdup_printf("%s bid %d", global->player[ get_player_slot_with_playerid(highbidder) ].name, highbid);

		text = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_text");

		textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
		gtk_text_buffer_get_end_iter(textbuff, &textiter);
		gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
		gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);

		/* Scroll the text */
		textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
		gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

		g_free(message);
	}

	/* status */
	tmp = xmlGetProp(cur, (xmlChar*)"status");
	if(tmp)  {

		status = atoi((gchar*)tmp);
		g_free(tmp);

		switch( status )  {

			case 1:
				text = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_text");

				message = g_strdup("<- going once ->");
				textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
				gtk_text_buffer_get_end_iter(textbuff, &textiter);
				gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
				gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);
				/* Scroll the text */
				textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
				gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

				g_free(message);
				break;

			case 2:
				text = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auction_text");

				message = g_strdup("\n<- going twice ->");
				textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
				gtk_text_buffer_get_end_iter(textbuff, &textiter);
				gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
				gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);
				/* Scroll the text */
				textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
				gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

				g_free(message);
				break;

			case 3:
				box = gtk_object_get_data(GTK_OBJECT(game->BoardCenter), "auctionbox");
				if(box) gtk_widget_destroy(box);
				gtk_object_set_data(GTK_OBJECT(game->BoardCenter), "auctionbox", 0);
				display_clear_buttons();  /* FIXME: SERVER BUG ! */
				display_unhide();
				break;
		}
	}
}




void xmlparse_display(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlNodePtr cur2;
	gint32 estateid;
	xmlChar *tmp;
	gchar *caption, *command;
	gboolean enable;

	if( game->status != GAME_STATUS_RUN)  return;

	display_init();

	tmp = xmlGetProp(cur, (xmlChar*)"cleartext");
	if(tmp)  {
		if( atoi((gchar*)tmp) )  display_clear_text();
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"clearbuttons");
	if(tmp)  {
		if( atoi((gchar*)tmp) )  display_clear_buttons();
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(tmp)   {

		estateid = atoi((gchar*)tmp);
		g_free(tmp);

		if(estateid < 0)  display_clear_title();
		else  display_estate(estateid);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"text");
	if(tmp)  {

		if( strlen((gchar*)tmp) > 0)  display_text((gchar*)tmp);
		g_free(tmp);
	}

	/* buttons */
	for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		if(game->status == GAME_STATUS_END)  continue;

		if(! xmlStrcmp(cur2->name, (xmlChar*)"button") )  {

			caption = (gchar*)xmlGetProp(cur2, (xmlChar*)"caption");
			command = (gchar*)xmlGetProp(cur2, (xmlChar*)"command");

			tmp = xmlGetProp(cur2, (xmlChar*)"enabled");
			enable = atoi((gchar*)tmp);
			g_free(tmp);

			display_button((gchar*)caption, (gchar*)command, enable);
			g_free(caption);
			g_free(command);
		}
	}
}




void xmlparse_configupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	GtkWidget *CButton;
	xmlNodePtr cur2;
	gchar *text, *title, *command;
	guint32 signalid;
	xmlChar *tmp;

	for( cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		if(! xmlStrcmp(cur2->name, (xmlChar*)"option") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"type");
			if(!tmp)  continue;
			if(xmlStrcmp(tmp, (xmlChar*)"bool") )  {

				g_free(tmp);
				continue;
			}
			g_free(tmp);

			command = (gchar*)xmlGetProp(cur2, (xmlChar*)"command");
			if(!command)  continue;

			/* create it if not created */
			if(! gtk_object_get_data(GTK_OBJECT(game->GameConfigBox), command) )  {

				title = (gchar*)xmlGetProp(cur2, (xmlChar*)"title");
				if(!title)  continue;

				CButton = gtk_check_button_new_with_label(title);
				signalid = gtk_signal_connect(GTK_OBJECT(CButton), "toggled", (GtkSignalFunc)Callback_toggle_boolean_gameoption, NULL);
				gtk_box_pack_start(GTK_BOX(game->GameConfigBox), CButton, FALSE, FALSE, 0);

				/* store signal identifier */
				text = g_strdup_printf("%d", signalid);
				gtk_object_set_data_full(GTK_OBJECT(CButton), "signal", text, g_free);

				/* store command under CButton */
				text = g_strdup(command);
				gtk_object_set_data_full(GTK_OBJECT(CButton), "command", text, g_free);

				/* save the pointer of button for use in the future */
				gtk_object_set_data(GTK_OBJECT(game->GameConfigBox), command, CButton);

				g_free(title);
			}

			/* modify */
			CButton = gtk_object_get_data(GTK_OBJECT(game->GameConfigBox), command);

			tmp = xmlGetProp(cur2, (xmlChar*)"value");
			if(tmp)   {

				/* disconnect signal */
				text = gtk_object_get_data(GTK_OBJECT(CButton), "signal");
				gtk_signal_disconnect(GTK_OBJECT(CButton), atoi(text) );

				if( atoi((gchar*)tmp) )
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CButton), TRUE);
				else
					gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CButton), FALSE);
				g_free(tmp);

				/* reconnect signal */
				signalid = gtk_signal_connect(GTK_OBJECT(CButton), "toggled", (GtkSignalFunc)Callback_toggle_boolean_gameoption, NULL);

				text = g_strdup_printf("%d", signalid);
				gtk_object_set_data_full(GTK_OBJECT(CButton), "signal", text, g_free);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"edit");
			if(tmp)   {

				if( atoi((gchar*)tmp) )
					gtk_widget_set_sensitive(CButton, TRUE);
				else
					gtk_widget_set_sensitive(CButton, FALSE);

				g_free(tmp);
			}

			g_free(command);
		}
	}

	gtk_widget_show_all(game->GameConfigBox);
}




void xmlparse_tradeupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlNodePtr cur2;
	xmlChar *tmp;
	guint32 tradeid = 0, playerid = 0, actor = 0, cardid = 0, estateid = 0;
	guint32 targetplayer = 0, playerfrom = 0, playerto = 0, money = 0;
	gboolean accept = FALSE;

	if( game->status != GAME_STATUS_RUN)  return;

	tmp = xmlGetProp(cur, (xmlChar*)"tradeid");
	if(tmp)  {
		tradeid = atoi((gchar*)tmp);
		g_free(tmp);
	}
	else  return;

	tmp = xmlGetProp(cur, (xmlChar*)"actor");
	if(tmp)   {
		actor = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"type");
	if(tmp)  {

		if(! xmlStrcmp(tmp, (xmlChar*)"rejected") )  {
			trade_destroy(tradeid);
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"completed") )  {
			trade_destroy(tradeid);
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"accepted") )  {
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"new") )  {
			trade_initnew(tradeid);
			g_free(tmp);
		}
	}

	/* create trade panel if wasn't created (server bug I suppose) */
	trade_initnew(tradeid);

	for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		/* -- player -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradeplayer") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"playerid");
			if(tmp)   {
				playerid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"accept");
			if(tmp)   {
				accept = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_player(tradeid, playerid, accept);
		}

		/* -- card -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradecard") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"cardid");
			if(tmp)   {
				cardid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"targetplayer");
			if(tmp)   {
				targetplayer = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_card(tradeid, cardid, targetplayer);
		}

		/* -- estate -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradeestate") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"estateid");
			if(tmp)   {
				estateid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"targetplayer");
			if(tmp)   {
				targetplayer = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_estate(tradeid, estateid, targetplayer);
		}

		/* -- money -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"trademoney") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"playerfrom");
			if(tmp)   {
				playerfrom = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"playerto");
			if(tmp)   {
				playerto = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"money");
			if(tmp)   {
				money = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_money(tradeid, playerfrom, playerto, money);
		}
	}

}




void xmlparse_cardupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 cardid = 0, owner = 0, i, cardslot = 0;

	if( game->status != GAME_STATUS_RUN)  return;

	tmp = xmlGetProp(cur, (xmlChar*)"cardid");
	if(tmp)  {
		cardid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"owner");
	if(tmp)  {
		owner = atoi((gchar*)tmp);
		g_free(tmp);
	}

	/* create new card slot */
	if(owner >= 0)  {

		cardslot = get_card_slot_with_cardid(cardid);
		if(cardslot < 0)  if(! game_get_valid_card_slot(&cardslot) )  return;

		game->card[cardslot].cardid = cardid;
		game->card[cardslot].owner = owner;
	}

	/* destroy a card slot */
	else  {

		cardslot = get_card_slot_with_cardid(cardid);
		if(cardslot < 0)  return;

		game->card[cardslot].cardid = 0;
		game->card[cardslot].owner = 0;
	}

	/* update component list in trade panel */
	for(i = 0 ; i < MAX_TRADES ; i++)  {

		if(! game->trade[i].open)  continue;

		trade_rebuild_component(i);
		trade_rebuild_subcomponent(i);
	}
}




void xmlparse_estategroupupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 groupid = -1;

	tmp = xmlGetProp(cur, (xmlChar*)"groupid");
	if(tmp)  {
		groupid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	if(groupid < 0)  return;

	if(game->group[groupid].name)  g_free(game->group[groupid].name);

	game->group[groupid].name = (gchar*)xmlGetProp(cur, (xmlChar*)"name");
}




void xmlparse_deleteplayer(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 playerid = -1, id;

	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if(tmp)  {
		playerid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	if(playerid < 0) return;

	id = get_player_slot_with_playerid(playerid);
	if(id < 0)  return;

	/* delete a player */
	game_delete_player(id);
	game_buildplayerlist();
}




void xmlparse_updategamelist(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	gchar *txt[10], *host, *port, *name, **text[2];
	gint32 id, test, i;
	xmlNodePtr cur2;
	xmlChar *type;

	if(global->phase != PHASE_GETGAMES)  {

		client_disconnect(connectid);
		return;
	}

	type = xmlGetProp(cur, (xmlChar*)"type");
	if(!type) return;

	if(! xmlStrcmp(type, (xmlChar*)"full") )  {

		gtk_clist_freeze(GTK_CLIST(global->GameCList));
		gameclist_title_showgames();

		host = g_strdup( g_strstrip(connection[connectid]->host) );
		port = g_strdup_printf("%d", connection[connectid]->port);

		/* remove all game from same host & port */
		text[0] = g_malloc0(8192);
		text[1] = g_malloc0(8192);
		for(i = 0 ;   ; i++) {

			test = gtk_clist_get_text(GTK_CLIST(global->GameCList), i, 0, text[0]);
			if(!test) break;
			test = gtk_clist_get_text(GTK_CLIST(global->GameCList), i, 5, text[1]);
			if(!test) break;

			if(! strncmp(host, *text[0], strlen(host))  &&  ! strncmp(port, *text[1], strlen(port)))  {

				gtk_clist_remove(GTK_CLIST(global->GameCList), i);
				i--;
			}
		}
		g_free(text[0]);
		g_free(text[1]);


		for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next )  {

			if(! xmlStrcmp(cur2->name, (xmlChar*)"game") )  {

				xmlChar *tmp;
				
				tmp = xmlGetProp(cur2, (xmlChar*)"canbejoined");
				if(tmp  &&  atoi((gchar*)tmp) == 0 )  {

					g_free(tmp);
					continue;
				}
				if(tmp) g_free(tmp);

				/* ONLY TEMPORARLY ! */
				tmp = xmlGetProp(cur2, (xmlChar*)"gametype");
				if(xmlStrcmp(tmp, (xmlChar*)"city")  &&  xmlStrcmp(tmp, (xmlChar*)"french_city") )  {

					g_free(tmp);
					continue;
				}

				g_free(tmp);

				tmp = xmlGetProp(cur2, (xmlChar*)"id");
				id = atoi((gchar*)tmp);
				g_free(tmp);

				name = (gchar*)xmlGetProp(cur2, (xmlChar*)"name");

				txt[0] = g_strdup(host);
				txt[1] = g_strdup(connection[connectid]->server_version);
				if(id < 0)
					txt[2] = g_strdup_printf("Create a new %s game", name);
				else
					txt[2] = g_strdup_printf("Join %s game #%d", name, id);
				txt[3] = (gchar*)xmlGetProp(cur2, (xmlChar*)"description");
				txt[4] = (gchar*)xmlGetProp(cur2, (xmlChar*)"players");
				txt[5] = g_strdup(port);
				txt[6] = (gchar*)xmlGetProp(cur2, (xmlChar*)"gametype");
				txt[7] = (gchar*)xmlGetProp(cur2, (xmlChar*)"id");

				gtk_clist_append(GTK_CLIST(global->GameCList), txt);

				for(i = 0 ; i < 8 ; i++)
					g_free(txt[i]);
				g_free(name);
			}
		}

		gtk_clist_thaw(GTK_CLIST(global->GameCList));
		g_free(host);
		g_free(port);

		client_disconnect(connectid);
	}

	g_free(type);
}




void xmlparse_metaserver(guint32 connectid, gchar *buffer) {

	gchar *txt[10], *text;
	xmlChar *tmp;
	xmlDocPtr doc;
	xmlNodePtr cur, cur2;
	guint32 i, do_close = TRUE;

	doc = xmlParseMemory(buffer,  strlen(buffer) );

	if(doc == NULL)  return;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {

		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, METASERVER_XMLROOTELEMENT) )  {

		xmlFreeDoc(doc);
		return;
	}

	gtk_clist_freeze(GTK_CLIST(global->GameCList));

	if(connection[connectid]->type == CONNECT_TYPE_META_GETLIST)  {

		gameclist_title_showservers();
		gtk_clist_clear(GTK_CLIST(global->GameCList));
	}
	else if(connection[connectid]->type == CONNECT_TYPE_META_GETGAME)  {

		gameclist_title_showgames();
		gtk_clist_clear(GTK_CLIST(global->GameCList));
	}

	for(cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"metaserver") )  {

			tmp = xmlGetProp(cur, (xmlChar*)"version");
			text = g_strdup_printf("Atlantic metaserver version %s", tmp);
			g_free(tmp);
			text_insert_message(text, strlen(text));
			g_free(text);

			do_close = FALSE;
		}

		if(! xmlStrcmp(cur->name, (xmlChar*)"msg") )  {

			tmp = xmlGetProp(cur, (xmlChar*)"value");
			text = g_strdup_printf("%s", tmp);
			g_free(tmp);
			interface_create_messagewin(text);
			g_free(text);

			do_close = FALSE;
		}


		if(connection[connectid]->type == CONNECT_TYPE_META_GETLIST
			&& !xmlStrcmp(cur->name, (xmlChar*)"server") )  {

			txt[0] = (gchar*)xmlGetProp(cur, (xmlChar*)"host");
			txt[1] = (gchar*)xmlGetProp(cur, (xmlChar*)"port");
			txt[2] = (gchar*)xmlGetProp(cur, (xmlChar*)"version");
			txt[3] = (gchar*)xmlGetProp(cur, (xmlChar*)"users");
			txt[4] = g_strdup("");
			txt[5] = g_strdup("");
			txt[6] = g_strdup("");
			txt[7] = g_strdup("");

			gtk_clist_append(GTK_CLIST(global->GameCList), txt);

			for(i = 0 ; i < 8 ; i++)
				g_free(txt[i]);
		}

		if(connection[connectid]->type == CONNECT_TYPE_META_GETGAME
			&& !xmlStrcmp(cur->name, (xmlChar*)"servergamelist") )  {

			gchar *host, *port, *version, *name;
			gint32 id;
			host = (gchar*)xmlGetProp(cur, (xmlChar*)"host");
			port = (gchar*)xmlGetProp(cur, (xmlChar*)"port");
			version = (gchar*)xmlGetProp(cur, (xmlChar*)"version");

			for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next )  {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"game") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"canbejoined");
					if(tmp  &&  atoi((gchar*)tmp) == 0 )  {

						g_free(tmp);
						continue;
					}
					if(tmp) g_free(tmp);

					/* ONLY TEMPORARLY ! */
					tmp = xmlGetProp(cur2, (xmlChar*)"gametype");
					if(xmlStrcmp(tmp, (xmlChar*)"city")  &&  xmlStrcmp(tmp, (xmlChar*)"french_city") )  {

						g_free(tmp);
						continue;
					}

					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					id = atoi((gchar*)tmp);
					g_free(tmp);

					name = (gchar*)xmlGetProp(cur2, (xmlChar*)"name");

					txt[0] = g_strdup(host);
					txt[1] = g_strdup(version);
					if(id < 0)
						txt[2] = g_strdup_printf("Create a new %s game", name);
					else
						txt[2] = g_strdup_printf("Join %s game #%d", name, id);
					txt[3] = (gchar*)xmlGetProp(cur2, (xmlChar*)"description");
					txt[4] = (gchar*)xmlGetProp(cur2, (xmlChar*)"players");
					txt[5] = g_strdup(port);
					txt[6] = (gchar*)xmlGetProp(cur2, (xmlChar*)"gametype");
					txt[7] = (gchar*)xmlGetProp(cur2, (xmlChar*)"id");

					gtk_clist_append(GTK_CLIST(global->GameCList), txt);

					for(i = 0 ; i < 8 ; i++)
						g_free(txt[i]);
					g_free(name);
				}
			}

			g_free(host);
			g_free(port );
			g_free(version);
		}

	}

	gtk_clist_thaw(GTK_CLIST(global->GameCList));
	xmlFreeDoc(doc);
	if(do_close)  client_disconnect(connectid);
}
