/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright (C) 2002-2010 Rochet Sylvain
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <gtk/gtk.h>

#include <libxml/parser.h>

#include "config.h"

#include "load.h"
#include "global.h"
#include "readpng.h"
#include "parser.h"


/* parse a config file */
gboolean parse_file_config(gchar *filename)  {

	gchar *str, *ident, *tmp;
	struct stat s;
	FILE *file;

	/* detect invalid theme */
	if(stat(filename, &s) < 0) return FALSE;

	/* parse general group */
	file = fopen(filename, "r");
	if (!file) return FALSE;

	str = g_malloc0(512);

	for( fgets(str, 512, file) ; !feof(file) ; fgets(str, 512, file) )  {

		ident = parser_get_identifier(str);
		if(!ident) continue;

		if(! strcmp(ident, "player") )  {

			tmp = parser_get_data(str, "nickname");
			if(config->nickname) g_free(config->nickname);
			config->nickname = tmp;

		} else if(! strcmp(ident, "metaserver2") )  {

			tmp = parser_get_data(str, "host");
			if(config->metaserver_host)  g_free(config->metaserver_host);
			config->metaserver_host = tmp;

			tmp = parser_get_data(str, "port");
			config->metaserver_port = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "autogetgames");
			config->metaserver_autogetgames = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "autogetlist");
			config->metaserver_autogetlist = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "sendclientversion");
			config->metaserver_sendclientversion = atoi( (gchar*)tmp );
			g_free(tmp);

		} else if(! strcmp(ident, "anotherserver") )  {

			tmp = parser_get_data(str, "host");
			if(config->getgames_host) g_free(config->getgames_host);
			config->getgames_host = tmp;

			tmp = parser_get_data(str, "port");
			config->getgames_port = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "autogetgames");
			config->getgames_autogetgames = atoi( (gchar*)tmp );
			g_free(tmp);

		} else if(! strcmp(ident, "buffer") )  {

			tmp = parser_get_data(str, "message");
			config->message_max_lines = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "chat");
			config->chat_max_lines = atoi( (gchar*)tmp );
			g_free(tmp);

		} else if(! strcmp(ident, "interface") )  {

			tmp = parser_get_data(str, "playerlist");
			if(! strcmp( tmp, "left") )
				config->game_playerlist_position = GAME_PLAYERLIST_POS_LEFT;
			if(! strcmp( tmp, "right") )
				config->game_playerlist_position = GAME_PLAYERLIST_POS_RIGHT;
			g_free(tmp);

			tmp = parser_get_data(str, "token_speed");
			config->game_token_animation_speed = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = parser_get_data(str, "token_transparency");
			config->game_token_transparency = atoi( (gchar*)tmp );
			g_free(tmp);
		}

		g_free(ident);
	}

	fclose(file);
	g_free(str);

	return TRUE;
}


/* parse the data file interface.xml */
gboolean xmlparse_file_interface()  {

	xmlDocPtr doc;
	xmlNodePtr cur, cur2;
	guint8 ident = 0xff;
	guint32 color;
	gint32 xoffset, yoffset;
	gchar *filename;
	xmlChar *tmp;

	filename = g_strconcat(PACKAGE_DATA_DIR, "/interface.xml", NULL);
	doc = xmlParseFile(filename);
	g_free(filename);

	if(doc == NULL)  return FALSE;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return FALSE;
	}

	if( xmlStrcmp(cur->name, DATAFILE_XMLROOTELEMENT) )  {
		xmlFreeDoc(doc);
		return FALSE;
	}

	cur = cur->xmlChildrenNode; 
	do {

		/* ===== PNGFILES ===== */
		/* ==================== */

		/* parse pngfile_logo */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_logo") )  {

			data->pngfile_logo_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_logo_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					if( xmlStrcmp(tmp, (xmlChar*)"full")  )  g_free(data->pngfile_logo_filename);
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_board */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_board") )  {

			data->pngfile_board_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_board_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_board_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_board_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_board_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_board_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_tokens */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_tokens") )  {

			data->pngfile_token_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_token_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_token_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_token_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_token_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_token_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_cards */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_cards") )  {

			data->pngfile_card_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_card_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_card_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_card_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_card_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_card_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_stars */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_stars") )  {

			data->pngfile_star_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_star_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_star_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_star_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_star_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_star_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_stars_m (mortgaged) */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_stars_m") )  {

			data->pngfile_star_m_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_star_m_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_star_m_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_star_m_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_star_m_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_star_m_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_commands */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_commands") )  {

			data->pngfile_command_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_command_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_command_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_command_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_command_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_command_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_horiz_houses */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_horiz_houses") )  {

			data->pngfile_horiz_house_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_horiz_house_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_horiz_house_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_horiz_house_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2,(xmlChar*) "width");
					data->pngfile_horiz_house_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_horiz_house_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse pngfile_vert_houses */
		if(! xmlStrcmp(cur->name, (xmlChar*)"pngfile_vert_houses") )  {

			data->pngfile_vert_house_filename = (gchar*)xmlGetProp(cur, (xmlChar*)"filename");

			tmp = xmlGetProp(cur, (xmlChar*)"need_alpha");
			data->pngfile_vert_house_need_alpha = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;
			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"buffer") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->pngfile_vert_house_x[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->pngfile_vert_house_y[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"width");
					data->pngfile_vert_house_width[ident]  = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"height");
					data->pngfile_vert_house_height[ident] = atoi( (gchar*)tmp );
					g_free(tmp);
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}


		/* ===== INTERFACE ===== */
		/* ===================== */

		/* parse board interface data */
		if(! xmlStrcmp(cur->name, (xmlChar*)"display_board") )  {

			data->number_estates = 0;

			tmp = xmlGetProp(cur, (xmlChar*)"width");
			data->board_width = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"height");
			data->board_height = atoi( (gchar*)tmp );
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;

			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"estate") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->estate[ident].x = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->estate[ident].y = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"buffer_board");
					data->estate[ident].buffer_board = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x1token");
					data->estate[ident].x1token = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y1token");
					data->estate[ident].y1token = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x2token");
					data->estate[ident].x2token = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y2token");
					data->estate[ident].y2token = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x1jail");
					data->estate[ident].x1jail = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y1jail");
					data->estate[ident].y1jail = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x2jail");
					data->estate[ident].x2jail = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y2jail");
					data->estate[ident].y2jail = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"xstar");
					data->estate[ident].xstar = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"ystar");
					data->estate[ident].ystar = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"xhouse");
					data->estate[ident].xhouse = data->estate[ident].x + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"yhouse");
					data->estate[ident].yhouse = data->estate[ident].y + atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"type_house");
					if(! xmlStrcmp(tmp, (xmlChar*)"horizontal") )
						data->estate[ident].type_house = TYPE_HOUSE_HORIZONTAL;
					else if(! xmlStrcmp(tmp, (xmlChar*)"vertical") )
						data->estate[ident].type_house = TYPE_HOUSE_VERTICAL;
					else
						data->estate[ident].type_house = TYPE_HOUSE_NONE;
					g_free(tmp);

					data->number_estates++;
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		/* parse player token interface data */
		if(! xmlStrcmp(cur->name, (xmlChar*)"display_playertoken") )  {

			tmp = xmlGetProp(cur, (xmlChar*)"width");
			data->playerlist_token_width = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"height");
			data->playerlist_token_height = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"bgcolor");
			if( atoi( (gchar*)tmp ) < 0 )
				data->playerlist_token_bgcolor = -1;
			else  {
				sscanf((gchar*)tmp, "0x%6X", &color);
				data->playerlist_token_bgcolor = color;
			}
			g_free(tmp);
		}

		/* parse player cards interface data */
		if(! xmlStrcmp(cur->name, (xmlChar*)"display_playercard") )  {

			data->number_playerlist_card = 0;

			tmp = xmlGetProp(cur, (xmlChar*)"width");
			data->playerlist_cards_width = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"height");
			data->playerlist_cards_height = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"xoffset");
			xoffset = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"yoffset");
			yoffset = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"alphaunowned");
			if(! g_strncasecmp((gchar*)tmp, "0x", 2) )  {

				sscanf((gchar*)tmp, "0x%2X", &color);
				data->playerlist_cards_alphaunowned = color;
			}
			else data->playerlist_cards_alphaunowned = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"alphaowned");
			if(! g_strncasecmp((gchar*)tmp, "0x", 2) )  {

				sscanf((gchar*)tmp, "0x%2X", &color);
				data->playerlist_cards_alphaowned = color;
			}
			else data->playerlist_cards_alphaowned = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"alphamortgage");
			if(! g_strncasecmp((gchar*)tmp, "0x", 2) )  {

				sscanf((gchar*)tmp, "0x%2X", &color);
				data->playerlist_cards_alphamortgage = color;
			}
			else data->playerlist_cards_alphamortgage = atoi( (gchar*)tmp );
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"cardbgcolor");
			sscanf((gchar*)tmp, "0x%6X", &color);
			data->playerlist_cards_cardbgcolor = color;
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"cardbgcolormortgage");
			sscanf((gchar*)tmp, "0x%6X", &color);
			data->playerlist_cards_cardbgcolormortgage = color;
			g_free(tmp);

			tmp = xmlGetProp(cur, (xmlChar*)"bgcolor");
			if( atoi((gchar*)tmp) < 0 )
				data->playerlist_cards_bgcolor = -1;
			else  {
				sscanf((gchar*)tmp, "0x%6X", &color);
				data->playerlist_cards_bgcolor = color;
			}
			g_free(tmp);

			cur2 = cur->xmlChildrenNode;

			do {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"card") )  {

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					ident = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"x");
					data->playerlist_card[ident].x = atoi( (gchar*)tmp ) + xoffset;
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"y");
					data->playerlist_card[ident].y = atoi( (gchar*)tmp ) + yoffset;
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"z");
					data->playerlist_card[ident].z = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"buffer_card");
					data->playerlist_card[ident].buffer_card = atoi( (gchar*)tmp );
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"estateid");
					data->playerlist_card[ident].estateid = atoi( (gchar*)tmp );
					g_free(tmp);

					data->number_playerlist_card++;
				}

				cur2 = cur2 -> next;
			} while(cur2 != NULL);
		}

		cur = cur -> next;

	} while(cur != NULL);

	xmlFreeDoc(doc);
	return TRUE;
}


/* read all intro pngs images */
gboolean intro_load_pngs()   {

	png_structp png_ptr;
	png_infop info_ptr;
	_png_imagedata png_imagedata;
	gchar *filename;
	FILE *infile;

	if(data->png_intro_loaded)  return TRUE;

	/* ---- load logo file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_logo_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_logo_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;
	data->logo_width = png_imagedata.width;
	data->logo_height = png_imagedata.height;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	data->pngfile_logo_image = readpng_get_image(&png_imagedata);
	data->pngfile_logo_alpha = readpng_get_alpha(&png_imagedata);

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);

	data->png_intro_loaded = TRUE;
	return TRUE;
}


/* free all intro pngs images */
void intro_free_pngs()  {

	if(! data->png_intro_loaded)  return;

	if(data->pngfile_logo_image) g_free(data->pngfile_logo_image);
	if(data->pngfile_logo_alpha) g_free(data->pngfile_logo_alpha);
	data->pngfile_logo_image = 0;
	data->pngfile_logo_alpha = 0;

	data->png_intro_loaded = FALSE;
}


/* read all game pngs images */
gboolean game_load_pngs()   {

	png_structp png_ptr;
	png_infop info_ptr;
	_png_imagedata png_imagedata;
	gchar *filename;
	FILE *infile;
	guint32 i;

	if(data->png_game_loaded)  return TRUE;

	/* ---- load board file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_board_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_board_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, FALSE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_board_width[i] ; i++)  {

		data->pngfile_board_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_board_x[i], data->pngfile_board_y[i], data->pngfile_board_width[i], data->pngfile_board_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load token file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_token_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_token_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_token_width[i] ; i++)  {

		data->pngfile_token_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_token_x[i], data->pngfile_token_y[i], data->pngfile_token_width[i], data->pngfile_token_height[i]);
		data->pngfile_token_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_token_x[i], data->pngfile_token_y[i], data->pngfile_token_width[i], data->pngfile_token_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load star file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_star_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_star_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_star_width[i] ; i++)  {

		data->pngfile_star_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_star_x[i], data->pngfile_star_y[i], data->pngfile_star_width[i], data->pngfile_star_height[i]);
		data->pngfile_star_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_star_x[i], data->pngfile_star_y[i], data->pngfile_star_width[i], data->pngfile_star_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load star_m file (mortgaged) */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_star_m_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_star_m_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_star_m_width[i] ; i++)  {

		data->pngfile_star_m_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_star_m_x[i], data->pngfile_star_m_y[i], data->pngfile_star_m_width[i], data->pngfile_star_m_height[i]);
		data->pngfile_star_m_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_star_m_x[i], data->pngfile_star_m_y[i], data->pngfile_star_m_width[i], data->pngfile_star_m_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load cards file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_card_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_card_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, FALSE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_card_width[i] ; i++)  {

		data->pngfile_card_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_card_x[i], data->pngfile_card_y[i], data->pngfile_card_width[i], data->pngfile_card_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load command file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_command_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_command_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_command_width[i] ; i++)  {

		data->pngfile_command_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_command_x[i], data->pngfile_command_y[i], data->pngfile_command_width[i], data->pngfile_command_height[i]);
		data->pngfile_command_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_command_x[i], data->pngfile_command_y[i], data->pngfile_command_width[i], data->pngfile_command_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);


	/* ---- load horiz_houses file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_horiz_house_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_horiz_house_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_horiz_house_width[i] ; i++)  {

		data->pngfile_horiz_house_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_horiz_house_x[i], data->pngfile_horiz_house_y[i], data->pngfile_horiz_house_width[i], data->pngfile_horiz_house_height[i]);
		data->pngfile_horiz_house_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_horiz_house_x[i], data->pngfile_horiz_house_y[i], data->pngfile_horiz_house_width[i], data->pngfile_horiz_house_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);



	/* ---- load vert_houses file */
	filename = g_strconcat(PACKAGE_DATA_DIR, "/", data->pngfile_vert_house_filename, NULL);

	if (! (infile = fopen(filename, "rb")) ) {

		fprintf(stderr, "can't open file [%s]\n", filename);
		g_free(filename);
		return FALSE;
	}
	g_free(filename);

	if (! readpng_init(infile, &png_ptr, &info_ptr, &png_imagedata) != 0) {

		fprintf(stderr, "[%s] is not a valid PNG file\n", filename);
		fclose(infile);
		return FALSE;
	}

	if(data->pngfile_vert_house_need_alpha  &&  !png_imagedata.have_alpha) return FALSE;

	if(! readpng_decode_image(&png_ptr, &info_ptr, &png_imagedata, TRUE) )  {

		fprintf(stderr, "Unable to decode PNG image: [%s]\n", filename);
		return FALSE;
	}
	fclose(infile);

	for(i = 0 ; data->pngfile_vert_house_width[i] ; i++)  {

		data->pngfile_vert_house_image[i] = readpng_get_image_zone(&png_imagedata, data->pngfile_vert_house_x[i], data->pngfile_vert_house_y[i], data->pngfile_vert_house_width[i], data->pngfile_vert_house_height[i]);
		data->pngfile_vert_house_alpha[i] = readpng_get_alpha_zone(&png_imagedata, data->pngfile_vert_house_x[i], data->pngfile_vert_house_y[i], data->pngfile_vert_house_width[i], data->pngfile_vert_house_height[i]);
	}

	readpng_cleanup(&png_ptr, &info_ptr, &png_imagedata);

	data->png_game_loaded = TRUE;
	return TRUE;
}


/* free all game pngs images */
void game_free_pngs()  {

	guint32 i;

	if(! data->png_game_loaded)  return;

	for(i = 0 ; data->pngfile_board_width[i] ; i++)  {

		if(data->pngfile_board_image[i])  {
			g_free(data->pngfile_board_image[i]);
			data->pngfile_board_image[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_token_width[i] ; i++)  {

		if(data->pngfile_token_image[i])  {
			g_free(data->pngfile_token_image[i]);
			data->pngfile_token_image[i] = 0;
		}
		if(data->pngfile_token_alpha[i])  {
			g_free(data->pngfile_token_alpha[i]);
			data->pngfile_token_alpha[i] = 0;
		}
	}


	for(i = 0 ; data->pngfile_star_width[i] ; i++)  {

		if(data->pngfile_star_image[i])  {
			g_free(data->pngfile_star_image[i]);
			data->pngfile_star_image[i] = 0;
		}
		if(data->pngfile_star_alpha[i])  {
			g_free(data->pngfile_star_alpha[i]);
			data->pngfile_star_alpha[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_star_m_width[i] ; i++)  {

		if(data->pngfile_star_m_image[i])  {
			g_free(data->pngfile_star_m_image[i]);
			data->pngfile_star_m_image[i] = 0;
		}
		if(data->pngfile_star_m_alpha[i])  {
			g_free(data->pngfile_star_m_alpha[i]);
			data->pngfile_star_m_alpha[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_card_width[i] ; i++)  {

		if(data->pngfile_card_image[i])  {
			g_free(data->pngfile_card_image[i]);
			data->pngfile_card_image[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_command_width[i] ; i++)  {

		if(data->pngfile_command_image[i])  {
			g_free(data->pngfile_command_image[i]);
			data->pngfile_command_image[i] = 0;
		}
		if(data->pngfile_command_alpha[i])  {
			g_free(data->pngfile_command_alpha[i]);
			data->pngfile_command_alpha[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_horiz_house_width[i] ; i++)  {

		if(data->pngfile_horiz_house_image[i])  {
			g_free(data->pngfile_horiz_house_image[i]);
			data->pngfile_horiz_house_image[i] = 0;
		}
		if(data->pngfile_horiz_house_alpha[i])  {
			g_free(data->pngfile_horiz_house_alpha[i]);
			data->pngfile_horiz_house_alpha[i] = 0;
		}
	}

	for(i = 0 ; data->pngfile_vert_house_width[i] ; i++)  {

		if(data->pngfile_vert_house_image[i])  {
			g_free(data->pngfile_vert_house_image[i]);
			data->pngfile_vert_house_image[i] = 0;
		}
		if(data->pngfile_vert_house_alpha[i])  {
			g_free(data->pngfile_vert_house_alpha[i]);
			data->pngfile_vert_house_alpha[i] = 0;
		}
	}

	data->png_game_loaded = FALSE;
}



void init_prototype_gtk_style()  {

	GtkWidget *label;
	GtkStyle  *rcstyle;

	/* create a dummy label so we can fetch the associated rc style */
	label = gtk_label_new ("");
	rcstyle = gtk_rc_get_style (label);
	gtk_widget_destroy (label);

	/* this is needed for some (broken?) gtk themes */
	if (rcstyle == NULL)
		rcstyle = gtk_style_new ();
	else
		gtk_style_ref (rcstyle);


	global->style_normal = gtk_style_copy (rcstyle);

	global->style_player_normal = gtk_style_copy (rcstyle);
	global->style_player_normal->fg[0].red   = 0x0000;
	global->style_player_normal->fg[0].green = 0x0000;
	global->style_player_normal->fg[0].blue  = 0x0000;

	global->style_player_turn = gtk_style_copy (rcstyle);
	global->style_player_turn->fg[0].red   = 0xffff;
	global->style_player_turn->fg[0].green = 0x0000;
	global->style_player_turn->fg[0].blue  = 0x0000;

	global->style_estate_title = gtk_style_copy (rcstyle);
//	global->style_estate_title->font = gdk_font_load("-*-times-*-r-*-*-24-*-*-*-*-*-*-*");
/*	if(! global->style_estate_title->font)
		global->style_estate_title = 0;
*/
	gtk_style_unref (rcstyle);
}

