# utils.py
#
# Copyright 2024 Christopher Talbot
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

from datetime import datetime, timezone
from enum import Enum
from gi.repository import GLib

meshtastic_config_dir = GLib.get_user_config_dir() + '/gtk_meshtastic_client'
messages_file_path = meshtastic_config_dir + '/messages.json'

def idToHex(nodeId):
    return '!' + hex(nodeId)[2:]

time_ago_cutoff = 12
time_go_cutoff_sec = 43200 #12*60*60

def _timeago(delta_secs: int) -> str:
    """ Allow an error offset of 1 minute """
    if delta_secs <= 60 and delta_secs >= -60:
        return "Just Now"
    mins = int(delta_secs // 60)
    if mins == 1:
        return "1 minute ago"
    elif mins <= 60:
        return f"{mins} minutes ago"

    mod_mins = int(mins) % 60
    hours = delta_secs // (60*60)

    if hours == 1 and mod_mins == 1:
        return f"1 hour, 1 minute ago"
    elif hours == 1:
        return f"1 hour, {mod_mins} minutes ago"
    elif mod_mins == 1:
        return f"{hours} hours 1 minute ago"
    elif hours <= time_ago_cutoff:
        return f"{hours} hours {mod_mins} minutes ago"

    return "Over" + str(time_ago_cutoff) + " hours ago"

def getTimeAgo(ts):
    """Format how long ago have we heard from this node (aka timeago)."""
    if ts is None:
        return None
    now = datetime.now()
    timestamp = datetime.fromtimestamp(ts)
    delta = now - timestamp
    delta_secs = int(delta.total_seconds())
    hours = delta_secs // (60*60)
    if hours < time_ago_cutoff:
        return _timeago(delta_secs)
    else:
        return timestamp.ctime()

def getUnixTimestampNow():
    return int(datetime.now(timezone.utc).timestamp())

class MsgDirection(Enum):
    In = 0
    Out = 1

"""
As far as I can tell, this isn't in the
Meshtastic Library
"""
class ChannelRole(Enum):
    Disabled = 0
    Primary = 1
    Secondary = 2
