/*----------------------------------------------------------------------------
 *  imonc.c - imond client for Linux
 *
 *  Creation:	    06.06.2000  fm
 *  Last Update:    10.11.2000  fm
 *
 *  Copyright (c) 2000 Frank Meyer
 *----------------------------------------------------------------------------
 */

#include <errno.h>
#include <stdio.h>

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>

#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>				/* decl of inet_addr()	    */
#include <sys/socket.h>

#include "imonc.h"
#include "timer.h"

/*----------------------------------------------------------------------------
 *  service_connect (host_name, port)
 *  connect to ip=host_name on port=port
 *  returns the fd or '0' if connection failed
 *----------------------------------------------------------------------------
 */
int service_connect (char *	host_name, int	port)
{
    struct sockaddr_in	addr;
    static struct hostent *	host_p;
		static char *last_host_name;
    int			fd;
    int			opt = 1;

    (void) memset ((char *) &addr, 0, sizeof (addr));
		
		if ((addr.sin_addr.s_addr = inet_addr ((char *) host_name)) == INADDR_NONE)
				{
					if (!host_p || strcmp(last_host_name,host_name)) { // quick hack to avoid unnecessary dns lookups
						host_p = gethostbyname (host_name);

						if (! host_p)
							{
								fprintf (stderr, "%s: host not found\n", host_name);
								return (-1);
							}
						if (last_host_name != NULL)
							free(last_host_name);
						last_host_name = strdup(host_name);
				}
					
					(void) memcpy ((char *) (&addr.sin_addr), host_p->h_addr,
												 host_p->h_length);
				}
		
    addr.sin_family  = AF_INET;
    addr.sin_port    = htons ((unsigned short) port);

    if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
			{						/* open socket		    */
				perror ("socket");
				return (-1);
			}

    (void) setsockopt (fd, IPPROTO_TCP, TCP_NODELAY,
											 (char *) &opt, sizeof (opt));

    if (connect (fd, (struct sockaddr *) &addr, sizeof (addr)) != 0)
			{
				(void) close (fd);
				/* [zeank] being q'n'd
				 * if trying to connect to telmond port don't show an error message
				 */
				if (port != TELMOND_PORT)
					perror (host_name);
				return (-1);
			}

    return (fd);
} /* service_connect (host_name, port) */


/*----------------------------------------------------------------------------
 *  service_disconnect (fd)                 - disconnect from service
 *----------------------------------------------------------------------------
 */
void service_disconnect (int fd)
{
    (void) close (fd);
    return;
} /* service_disconnect (fd) */


/*----------------------------------------------------------------------------
 *  send_command (fd, str)                 - send command str via fd
 *----------------------------------------------------------------------------
 */
void send_command (int	fd, char *str)
{
    char    buf[256];
    int	    len = strlen (str);

    sprintf (buf, "%s\r\n", str);
    write (fd, buf, len + 2);

    return;
} /* send_command (fd, str) */


/*----------------------------------------------------------------------------
 *  get_answer (fd)													- get answer from fd
 *----------------------------------------------------------------------------
 */
char *get_answer (int fd)
{
	static char buf[8192];
	char *tmp;
	char c[1];
	int len = 0;
	
	tmp = buf;
	
	alarm (READ_TIMEOUT);
	while (read(fd, c, 1) > 0) {
		len ++;
		if (c[0] == '\n')
			break; /* break at end of line */
		if (c[0] != '\r')
			*tmp++ = c[0];
	}
	alarm (0);
	
	*tmp = '\0';
	
	if (len <= 0)
		{
			/* connection seems to have died */
			fli4l_reconnect (0);
			buf[0] = '\0';
			return (buf);
		}
	
/*     while (len > 1 && (buf[len - 1] == '\n' || buf[len - 1] == '\r')) */
/* 			{ */
/* 				buf[len - 1] = '\0'; */
/* 				len--; */
/* 			} */
		
	if (! strncmp (buf, "OK ", 3))			/* OK xxxx	    */
		{
			return (buf + 3);
		}
	else if (len > 2 && ! strcmp (buf + len - 2, "OK"))
		{
			*(buf + len - 2) = '\0';
			return (buf);
		}
	else if (len == 2 && ! strcmp (buf + len - 2, "OK"))
		{
			return (buf);
		}
		
	buf[0] = '\0';
	return (buf);				/* ERR xxxx	    */
} /* get_answer (fd) */


/*----------------------------------------------------------------------------
 *  get_lastcall (telmond_port)
 *  gets last incoming call from telmond
 *----------------------------------------------------------------------------
 */
char *get_lastcall (int telmond_port)
{
    static char buf[8192];
		static char old_buf[8192];
		static int old_buf_initialized;
    int		len, td;

		if ((td = service_connect (host_name, telmond_port)) <= 0) 
			return ((char *) NULL);

		alarm (READ_TIMEOUT);
    len = read (td, buf, 8192);
		alarm (0);

		new_call = 0;
    if (len <= 0) {
			service_disconnect (td);
			return ((char *) NULL);
		}
		
    while (len > 1 && (buf[len - 1] == '\n' || buf[len - 1] == '\r'))
			{
				buf[len - 1] = '\0';
				len--;
			}
		service_disconnect (td);

		if (!old_buf_initialized) {
			strncpy(old_buf,buf,8192);
			old_buf_initialized = 1;
		}

		if (strncmp(old_buf,buf,8192)) {
			new_call = 1;
			strncpy(old_buf,buf,8192);
		}

		return (buf);

} /* get_lastcall (telmond_port) */

/*----------------------------------------------------------------------------
 *  get_list (fd, command)													- get answer from fd
 *----------------------------------------------------------------------------
 */
GSList *get_list (int fd, char *str)
{
	gchar *buf;
	gchar **strarr;
	GSList *list = NULL;

	buf = get_buf (fd,str);

	strarr = g_strsplit (buf, "\n", 0);
	while (*strarr != NULL)
		{
			if (strncmp (*strarr, "OK", 2) == 0)
				return list;
			else if (strncmp (*strarr, "ERR", 3) == 0)
				return NULL;
			
			list = g_slist_append (list, *strarr);
			
			strarr++;
		}
	return NULL; /* no ok? */
} /* get_list (fd, command) */

/*----------------------------------------------------------------------------
 *  get_buf (fd, command)				- sends command and allocates a dynamic buffer
 *----------------------------------------------------------------------------
 */
gchar *
get_buf (int fd, char *str)
{
	gchar wbuf[64];
	gint len = strlen (str);
  static gchar *buf;
	gint      buffer_size = 0;
	gchar            ch[1];
	gint             n;
	gint             at_bol = 1;             /* flag: at beginning of line   */
	gint             last_line = 0;          /* flag: last line              */
	
	sprintf (wbuf, "%s\r\n", str);
	write (fd, wbuf, len + 2);

	if (buffer_size == 0)
		{
			buffer_size = GRANULARITY;
			buf = malloc (buffer_size);
		}
	
	alarm (READ_TIMEOUT);
	for (n = 0; read (fd, ch, 1) > 0; n++)
    {
			if (ch[0] == '\r')
				{
					n--;
					continue;
				}

        if (n + 1 == buffer_size)
					{
            buffer_size += GRANULARITY;
            buf = realloc (buf, buffer_size);
					}

        buf[n] = ch[0];
				if (at_bol && ch[0] != ' ')
					{
						last_line = 1;
						at_bol = 0;
					}
        else if (ch[0] == '\n')
					{
            at_bol = 1;
						
            if (last_line == 1)
							{
                break;
							}
					}
        else
					{   
						at_bol = 0;
					}
    }
	alarm(0);

	buf[n] = '\0';

  return (buf);
}

/*----------------------------------------------------------------------------
 * hexmd5 (string)      - A small wrapper to the md5 algorithm to use it
 *                        on a string and return some hex digits
 *----------------------------------------------------------------------------
 */
char *
hexmd5(const char *string) {
        static unsigned char md5buffer[16];
        static char tmp[33];
        size_t i;

        md5_buffer(string, strlen(string), md5buffer);

        for (i = 0; i < 16; i++)
                sprintf (tmp + (2 * i), "%02x", md5buffer[i]);

        return tmp;
}

