/* 
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Fluendo MPEG Demuxer plugin.
 *
 * The Initial Developer of the Original Code is Fluendo, S.L.
 * Portions created by Fluendo, S.L. are Copyright (C) 2005
 * Fluendo, S.L. All Rights Reserved.
 *
 * Contributor(s): Wim Taymans <wim@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "gstmpegdefs.h"
#include "gstmpegtsdemux.h"

GST_DEBUG_CATEGORY_STATIC (gstflutsdemux_debug);
#define GST_CAT_DEFAULT (gstflutsdemux_debug)

/* elementfactory information */
static GstElementDetails fluts_demux_details = {
  "MPEG Transport stream demuxer",
  "Codec/Demuxer",
  "Demultiplexes MPEG2 Transport Streams",
  "Wim Taymans <wim@fluendo.com>"
};

/* MPEG2Demux signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  ARG_0,
  ARG_BIT_RATE,
  ARG_MPEG2
      /* FILL ME */
};

static GstStaticPadTemplate sink_template = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/mpegts")
    );

static GstStaticPadTemplate src_template = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/mpeg, "
        "mpegversion = (int) { 1, 2 }, " "systemstream = (boolean) TRUE")
    );


static GstStaticPadTemplate video_template =
GST_STATIC_PAD_TEMPLATE ("video_%02d",
    GST_PAD_SRC,
    GST_PAD_SOMETIMES,
    GST_STATIC_CAPS ("video/mpeg, "
        "mpegversion = (int) { 1, 2 }, " "systemstream = (boolean) FALSE")
    );

static GstStaticPadTemplate audio_template =
GST_STATIC_PAD_TEMPLATE ("audio_%02d",
    GST_PAD_SRC,
    GST_PAD_SOMETIMES,
    GST_STATIC_CAPS ("audio/mpeg, " "mpegversion = (int) 1"
        /* FIXME "layer = (int) { 1, 2 }" */
    )
    );

static GstStaticPadTemplate private_template =
GST_STATIC_PAD_TEMPLATE ("private_%d",
    GST_PAD_SRC,
    GST_PAD_SOMETIMES,
    GST_STATIC_CAPS_ANY);

static void gst_fluts_demux_base_init (GstFluTSDemuxClass * klass);
static void gst_fluts_demux_class_init (GstFluTSDemuxClass * klass);
static void gst_fluts_demux_init (GstFluTSDemux * demux);

static gboolean gst_fluts_demux_is_PMT (GstFluTSDemux *demux, guint16 PID); 

static GstFlowReturn gst_fluts_demux_chain (GstPad *pad, GstBuffer *buffer);

static GstStateChangeReturn gst_fluts_demux_change_state (GstElement * element,
    GstStateChange transition);

static GstElementClass *parent_class = NULL;

/*static guint gst_fluts_demux_signals[LAST_SIGNAL] = { 0 };*/

GType
gst_fluts_demux_get_type (void)
{
  static GType fluts_demux_type = 0;

  if (!fluts_demux_type) {
    static const GTypeInfo fluts_demux_info = {
      sizeof (GstFluTSDemuxClass),
      (GBaseInitFunc) gst_fluts_demux_base_init,
      NULL,
      (GClassInitFunc) gst_fluts_demux_class_init,
      NULL,
      NULL,
      sizeof (GstFluTSDemux),
      0,
      (GInstanceInitFunc) gst_fluts_demux_init,
    };

    fluts_demux_type =
        g_type_register_static (GST_TYPE_ELEMENT, "GstFluTSDemux",
        &fluts_demux_info, 0);

    GST_DEBUG_CATEGORY_INIT (gstflutsdemux_debug, "flutsdemux", 0,
        "MPEG program stream demultiplexer element");
  }

  return fluts_demux_type;
}

static void
gst_fluts_demux_base_init (GstFluTSDemuxClass * klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  klass->sink_template = gst_static_pad_template_get (&sink_template);
  klass->src_template = gst_static_pad_template_get (&src_template);
  klass->video_template = gst_static_pad_template_get (&video_template);
  klass->audio_template = gst_static_pad_template_get (&audio_template);
  klass->private_template = gst_static_pad_template_get (&private_template);

  gst_element_class_add_pad_template (element_class, klass->video_template);
  gst_element_class_add_pad_template (element_class, klass->audio_template);
  gst_element_class_add_pad_template (element_class, klass->private_template);
  gst_element_class_add_pad_template (element_class, klass->sink_template);

  gst_element_class_set_details (element_class, &fluts_demux_details);
}

static void
gst_fluts_demux_class_init (GstFluTSDemuxClass * klass)
{
  GstElementClass *gstelement_class;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gstelement_class = (GstElementClass *) klass;

  gstelement_class->change_state = gst_fluts_demux_change_state;
}

static void
gst_fluts_demux_init (GstFluTSDemux * demux)
{
  GstFluTSDemuxClass *klass = GST_FLUTS_DEMUX_GET_CLASS (demux);

  demux->sinkpad = gst_pad_new_from_template (klass->sink_template, "sink");
  gst_pad_set_chain_function (demux->sinkpad,
			        gst_fluts_demux_chain);
  gst_element_add_pad (GST_ELEMENT (demux), demux->sinkpad);
}

static guint32 crc_tab[256] = {
  0x00000000, 0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b,
  0x1a864db2, 0x1e475005, 0x2608edb8, 0x22c9f00f, 0x2f8ad6d6, 0x2b4bcb61,
  0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd, 0x4c11db70, 0x48d0c6c7,
  0x4593e01e, 0x4152fda9, 0x5f15adac, 0x5bd4b01b, 0x569796c2, 0x52568b75,
  0x6a1936c8, 0x6ed82b7f, 0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3,
  0x709f7b7a, 0x745e66cd, 0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
  0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5, 0xbe2b5b58, 0xbaea46ef,
  0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033, 0xa4ad16ea, 0xa06c0b5d,
  0xd4326d90, 0xd0f37027, 0xddb056fe, 0xd9714b49, 0xc7361b4c, 0xc3f706fb,
  0xceb42022, 0xca753d95, 0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1,
  0xe13ef6f4, 0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d, 0x34867077, 0x30476dc0,
  0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5, 0x2ac12072,
  0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16, 0x018aeb13, 0x054bf6a4,
  0x0808d07d, 0x0cc9cdca, 0x7897ab07, 0x7c56b6b0, 0x71159069, 0x75d48dde,
  0x6b93dddb, 0x6f52c06c, 0x6211e6b5, 0x66d0fb02, 0x5e9f46bf, 0x5a5e5b08,
  0x571d7dd1, 0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
  0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b, 0xbb60adfc,
  0xb6238b25, 0xb2e29692, 0x8aad2b2f, 0x8e6c3698, 0x832f1041, 0x87ee0df6,
  0x99a95df3, 0x9d684044, 0x902b669d, 0x94ea7b2a, 0xe0b41de7, 0xe4750050,
  0xe9362689, 0xedf73b3e, 0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2,
  0xc6bcf05f, 0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34,
  0xdc3abded, 0xd8fba05a, 0x690ce0ee, 0x6dcdfd59, 0x608edb80, 0x644fc637,
  0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb, 0x4f040d56, 0x4bc510e1,
  0x46863638, 0x42472b8f, 0x5c007b8a, 0x58c1663d, 0x558240e4, 0x51435d53,
  0x251d3b9e, 0x21dc2629, 0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5,
  0x3f9b762c, 0x3b5a6b9b, 0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
  0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623, 0xf12f560e, 0xf5ee4bb9,
  0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65, 0xeba91bbc, 0xef68060b,
  0xd727bbb6, 0xd3e6a601, 0xdea580d8, 0xda649d6f, 0xc423cd6a, 0xc0e2d0dd,
  0xcda1f604, 0xc960ebb3, 0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7,
  0xae3afba2, 0xaafbe615, 0xa7b8c0cc, 0xa379dd7b, 0x9b3660c6, 0x9ff77d71,
  0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74, 0x857130c3,
  0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640, 0x4e8ee645, 0x4a4ffbf2,
  0x470cdd2b, 0x43cdc09c, 0x7b827d21, 0x7f436096, 0x7200464f, 0x76c15bf8,
  0x68860bfd, 0x6c47164a, 0x61043093, 0x65c52d24, 0x119b4be9, 0x155a565e,
  0x18197087, 0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
  0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d, 0x2056cd3a,
  0x2d15ebe3, 0x29d4f654, 0xc5a92679, 0xc1683bce, 0xcc2b1d17, 0xc8ea00a0,
  0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb, 0xdbee767c, 0xe3a1cbc1, 0xe760d676,
  0xea23f0af, 0xeee2ed18, 0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4,
  0x89b8fd09, 0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662,
  0x933eb0bb, 0x97ffad0c, 0xafb010b1, 0xab710d06, 0xa6322bdf, 0xa2f33668,
  0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4
};

static guint32 
gst_fluts_demux_calc_crc32 (guint8 *data, guint datalen)
{
  gint i;
  guint32 crc = 0xffffffff;

  for (i=0; i<datalen; i++) {
    crc = (crc << 8) ^ crc_tab[((crc >> 24) ^ *data++) & 0xff];
  }

  return crc;
}

static gboolean 
gst_fluts_demux_fill_stream (GstFluTSStream *stream, guint8 id, guint8 stream_type)
{
  GstPadTemplate *template;
  gchar *name;
  GstFluTSDemuxClass *klass;
  GstFluTSDemux *demux;
  GstCaps *caps;

  if (stream->stream_type && stream->stream_type != stream_type)
    goto wrong_type;

  demux = stream->demux;
  klass = GST_FLUTS_DEMUX_GET_CLASS (demux);

  name = NULL;
  template = NULL;
  caps = NULL;

  switch (stream_type) {
    case ST_VIDEO_MPEG1:
    case ST_VIDEO_MPEG2:
      template = klass->video_template;
      name = g_strdup_printf ("video_%04x", stream->PID);
      caps = gst_caps_new_simple ("video/mpeg",
                  "mpegversion", G_TYPE_INT, stream_type == ST_VIDEO_MPEG1 ? 1 : 2,
                  "systemstream", G_TYPE_BOOLEAN, FALSE,
                  NULL);
      break;
    case ST_AUDIO_MPEG1:
    case ST_AUDIO_MPEG2:
      template = klass->audio_template;
      name = g_strdup_printf ("audio_%04x", stream->PID);
      caps = gst_caps_new_simple ("audio/mpeg",
                  "mpegversion", G_TYPE_INT, 1,
                  NULL);
      break;
    case ST_PRIVATE_SECTIONS:
    case ST_PRIVATE_DATA:
    case ST_MHEG:
    case ST_DSMCC:
    case ST_AUDIO_AAC:
    case ST_VIDEO_MPEG4:
    case ST_VIDEO_H264:
      break;
    case ST_PS_AUDIO_AC3:
      template = klass->audio_template;
      name = g_strdup_printf ("audio_%04x", stream->PID);
      caps = gst_caps_new_simple ("audio/x-private1-ac3", NULL);
      break;
    case ST_PS_AUDIO_DTS:
      template = klass->audio_template;
      name = g_strdup_printf ("audio_%04x", stream->PID);
      caps = gst_caps_new_simple ("audio/x-private1-dts", NULL);
      break;
    case ST_PS_AUDIO_LPCM:
      template = klass->audio_template;
      name = g_strdup_printf ("audio_%04x", stream->PID);
      caps = gst_caps_new_simple ("audio/x-private1-lpcm", NULL);
    case ST_PS_DVD_SUBTITLE:
      break;
    default:
      break;
  }
  if (name == NULL || template == NULL || caps == NULL)
    return FALSE;

  stream->stream_type = stream_type;
  stream->id = id;
  stream->pad = gst_pad_new_from_template (template, name);
  gst_pad_use_fixed_caps (stream->pad);
  gst_pad_set_caps (stream->pad, caps);
  gst_caps_unref (caps);
  g_free (name);

  return TRUE;

wrong_type:
  {
    return FALSE; 
  }
}

static gboolean
gst_fluts_demux_send_new_segment (GstFluTSDemux *demux, GstFluTSStream * stream, gint64 pts)
{
  GstFluTSStream *PCR_stream;
  GstFluTSStream *PMT_stream;
  GstClockTime time;

  /* for the reference start time we need to consult the PCR_PID of the
   * current PMT */
  if (demux->current_PMT == 0)
    goto no_pmt;

  PMT_stream = demux->streams[demux->current_PMT];
  if (PMT_stream == NULL)
    goto no_pmt_stream;

  PCR_stream = demux->streams[PMT_stream->PMT.PCR_PID];
  if (PCR_stream == NULL)
    goto no_pcr;

  if (PCR_stream->base_PCR == -1) {
    GST_DEBUG_OBJECT (demux, "no base PCR, using last PCR %"G_GUINT64_FORMAT,
		    PCR_stream->last_PCR);
    PCR_stream->base_PCR = PCR_stream->last_PCR;
  }
  else {
    GST_DEBUG_OBJECT (demux, "using base PCR %"G_GUINT64_FORMAT,
		    PCR_stream->base_PCR);
  }
  if (PCR_stream->last_PCR == -1) {
    GST_DEBUG_OBJECT (demux, "no last PCR, using PTS %"G_GUINT64_FORMAT, pts);
    PCR_stream->base_PCR = pts;
    PCR_stream->last_PCR = pts;
  }

  time = MPEGTIME_TO_GSTTIME (PCR_stream->base_PCR);

  GST_DEBUG_OBJECT (demux, "segment PTS to (%"G_GUINT64_FORMAT") time: %" 
	    G_GUINT64_FORMAT, PCR_stream->base_PCR, time);

  gst_pad_push_event (stream->pad, 
      gst_event_new_new_segment (FALSE, 1.0, GST_FORMAT_TIME,
	  time, -1, time));

  return TRUE;

  /* ERRORS */
no_pmt:
  {
    GST_DEBUG_OBJECT (demux, "no PMT selected");
    return FALSE;
  }
no_pmt_stream:
  {
    GST_DEBUG_OBJECT (demux, "no PMT stream found");
    return FALSE;
  }
no_pcr:
  {
    GST_DEBUG_OBJECT (demux, "no PCR stream found");
    return FALSE;
  }
}

static GstFlowReturn
gst_fluts_demux_data_cb (GstPESFilter *filter, gboolean first, GstBuffer *buffer, GstFluTSStream * stream)
{
  GstFluTSDemux *demux;
  GstPad *srcpad;
  GstFlowReturn ret;
  gint64 pts;
  GstClockTime time;

  demux = stream->demux;
  srcpad = stream->pad;

  GST_DEBUG_OBJECT (demux, "got data on PID 0x%04x", stream->PID);

  if (first) {
    pts = filter->pts;
    time = MPEGTIME_TO_GSTTIME (pts);
  }
  else {
    time = pts = -1;
  }

  GST_DEBUG_OBJECT (demux, "setting PTS to (%"G_GUINT64_FORMAT") time: %" 
	    G_GUINT64_FORMAT, pts, time);

  GST_BUFFER_TIMESTAMP (buffer) = time;

  /* check if we have a pad already */
  if (srcpad == NULL) {
    /* fill in the last bits of the stream */
    if (!gst_fluts_demux_fill_stream (stream, filter->id, stream->stream_type))
      goto unknown_type;

    srcpad = stream->pad;
    gst_element_add_pad (GST_ELEMENT_CAST (demux), srcpad);

    /* send new_segment */
    gst_fluts_demux_send_new_segment (demux, stream, pts);
  }


  gst_buffer_set_caps (buffer, GST_PAD_CAPS (srcpad));
  ret = gst_pad_push (srcpad, buffer);

  return ret;

  /* ERROR */
unknown_type:
  {
    GST_DEBUG_OBJECT (demux, "got unknown stream id 0x%02x, type 0x%02x", 
		    filter->id, filter->type);
    gst_buffer_unref (buffer);
    return GST_FLOW_NOT_LINKED;
  }
}

static void
gst_fluts_demux_resync_cb (GstPESFilter *filter, GstFluTSStream * stream)
{
  /* does nothing for now */
}

static gboolean
gst_fluts_demux_is_reserved_PID (GstFluTSDemux *demux, guint16 PID)
{
  return (PID >= PID_RESERVED_FIRST) && (PID < PID_RESERVED_LAST);
}

static GstFluTSStream *
gst_fluts_demux_get_stream_for_PID (GstFluTSDemux * demux, guint16 PID)
{
  GstFluTSStream *stream = NULL;

  if (G_UNLIKELY (PID > FLUTS_MAX_PID))
    return NULL;

  stream = demux->streams[PID];
  if (stream == NULL) {
    stream = g_new0 (GstFluTSStream, 1);

    stream->demux = demux;
    stream->PID = PID;
    stream->pad = NULL;
    stream->base_PCR = -1;
    stream->last_PCR = -1;
    stream->PMT.version_number = -1;
    stream->PAT.version_number = -1;

    switch (PID) {
      /* check for fixed mapping */
      case PID_PROGRAM_ASSOCIATION_TABLE:
        stream->PID_type = PID_TYPE_PROGRAM_ASSOCIATION;
        break;
      case PID_CONDITIONAL_ACCESS_TABLE:
        stream->PID_type = PID_TYPE_CONDITIONAL_ACCESS;
        break;
      case PID_NULL_PACKET:
        stream->PID_type = PID_TYPE_NULL_PACKET;
        break;
      default:
	/* mark reserved PIDs */
	if (gst_fluts_demux_is_reserved_PID (demux, PID)) {
          stream->PID_type = PID_TYPE_RESERVED;
	}
	else {
          /* check if PMT found in PAT */
          if (gst_fluts_demux_is_PMT (demux, PID))
            stream->PID_type = PID_TYPE_PROGRAM_MAP;
	  else
            stream->PID_type = PID_TYPE_UNKNOWN;
	}
        break;
    }
    GST_DEBUG_OBJECT (demux, "creating stream %p for PID 0x%04x, PID_type %d", stream, PID, stream->PID_type);

    demux->streams[PID] = stream;
  }

  return stream;
}

/*
 * CA_section() {
 *   table_id                  8 uimsbf   == 0x01
 *   section_syntax_indicator  1 bslbf    == 1
 *   '0'                       1 bslbf    == 0
 *   reserved                  2 bslbf
 *   section_length           12 uimsbf   == 00xxxxx...
 *   reserved                 18 bslbf
 *   version_number            5 uimsbf
 *   current_next_indicator    1 bslbf
 *   section_number            8 uimsbf
 *   last_section_number       8 uimsbf
 *   for (i=0; i<N;i++) {
 *     descriptor()
 *   }
 *   CRC_32                   32 rpchof
 * }
 */
static gboolean
gst_fluts_stream_parse_cat (GstFluTSStream * stream, gboolean pusi, guint8 *data, guint datalen)
{
  GstFluTSDemux *demux;
  guint8 pointer;

  demux = stream->demux;

  if (pusi) {
    pointer = *data++;
    GST_DEBUG_OBJECT (demux, "pointer 0x%02x", pointer);
    datalen--;
  }

  GST_DEBUG_OBJECT (demux, "parsing CA section");
  return TRUE;
}


/*
 * TS_program_map_section() {
 *   table_id                          8 uimsbf   == 0x02
 *   section_syntax_indicator          1 bslbf    == 1
 *   '0'                               1 bslbf    == 0
 *   reserved                          2 bslbf
 *   section_length                   12 uimsbf   == 00xxxxx...
 *   program_number                   16 uimsbf
 *   reserved                          2 bslbf
 *   version_number                    5 uimsbf
 *   current_next_indicator            1 bslbf
 *   section_number                    8 uimsbf
 *   last_section_number               8 uimsbf
 *   reserved                          3 bslbf
 *   PCR_PID                          13 uimsbf
 *   reserved                          4 bslbf
 *   program_info_length              12 uimsbf   == 00xxxxx...
 *   for (i=0; i<N; i++) {
 *     descriptor()
 *   }
 *   for (i=0;i<N1;i++) {
 *     stream_type                     8 uimsbf
 *     reserved                        3 bslbf
 *     elementary_PID                 13 uimsnf
 *     reserved                        4 bslbf
 *     ES_info_length                 12 uimsbf   == 00xxxxx...
 *     for (i=0; i<N2; i++) {
 *       descriptor()
 *     }
 *   }
 *   CRC_32                           32 rpchof
 * }
 */
static gboolean
gst_fluts_stream_parse_pmt (GstFluTSStream *stream, 
		gboolean pusi, guint8 *data, guint datalen)
{
  GstFluTSDemux *demux;
  guint16 section_length;
  gint entries;
  guint32 CRC;
  guint8 pointer;
  GstFluTSPMT *PMT;
  guint version_number;

  demux = stream->demux;

  if (pusi) {
    pointer = *data++;
    GST_DEBUG_OBJECT (demux, "pointer 0x%02x", pointer);
    data += pointer;
    datalen -= (pointer + 1);
  }

  if (*data++ != 0x02)
    goto wrong_id;
  if ((data[0] & 0xc0) != 0x80)
    goto wrong_sync;
  if ((data[0] & 0x0c) != 0x00)
    goto wrong_seclen;

  section_length = GST_READ_UINT16_BE (data);
  section_length &= 0x0fff;
  data += 2;

  if (section_length > 1021)
    goto seclen_too_big;

  if (gst_fluts_demux_calc_crc32 (data-3, section_length+3) != 0)
    goto wrong_crc;

  GST_DEBUG_OBJECT (demux, "PMT section_length: %d", section_length);

  PMT = &stream->PMT;

  /* check if version number changed */
  version_number = (data[2] & 0x3e) >> 1;
  GST_DEBUG_OBJECT (demux, "PMT version_number: %d", version_number);
  if (version_number == PMT->version_number)
    goto same_version;

  stream->PMT.version_number = version_number;

  stream->PMT.program_number = GST_READ_UINT16_BE (data);
  data += 2;
  GST_DEBUG_OBJECT (demux, "PMT program_number: %d", PMT->program_number);

  PMT->current_next_indicator = (*data++ & 0x01) == 0x01;
  GST_DEBUG_OBJECT (demux, "PMT current_next_indicator: %d", PMT->current_next_indicator);

  PMT->section_number = *data++;
  GST_DEBUG_OBJECT (demux, "PMT section_number: %d", PMT->section_number);

  PMT->last_section_number = *data++;
  GST_DEBUG_OBJECT (demux, "PMT last_section_number: %d", PMT->last_section_number);

  PMT->PCR_PID = GST_READ_UINT16_BE (data);
  PMT->PCR_PID &= 0x1fff;
  data += 2;
  GST_DEBUG_OBJECT (demux, "PMT PCR_PID: 0x%04x", PMT->PCR_PID);

  if ((data[0] & 0x0c) != 0x00)
    goto wrong_pilen;

  PMT->program_info_length = GST_READ_UINT16_BE (data);
  PMT->program_info_length &= 0x0fff;
  data += 2;

  /* parse descriptor */
  PMT->program_info = gst_mpeg_descriptor_parse (data, PMT->program_info_length);

  /* skip descriptor */
  data += PMT->program_info_length;
  GST_DEBUG_OBJECT (demux, "PMT program_info_length: %d", PMT->program_info_length);

  entries = section_length - PMT->program_info_length - 9 - 4;

  PMT->entries = g_array_new (FALSE, TRUE, sizeof (GstFluTSPMTEntry));

  while (entries > 0) {
    GstFluTSPMTEntry entry;
    GstFluTSStream *ES_stream;
    guint8 stream_type;
    guint16 ES_info_length;

    stream_type = *data++; 

    entry.PID = GST_READ_UINT16_BE (data);
    entry.PID &= 0x1fff;
    data += 2;
    
    if ((data[0] & 0x0c) != 0x00)
      goto wrong_esilen;

    ES_info_length = GST_READ_UINT16_BE (data);
    ES_info_length &= 0x0fff;
    data += 2;

    /* get/create elementary stream */
    ES_stream = gst_fluts_demux_get_stream_for_PID (demux, entry.PID);
    /* check if PID unknown */
    if (ES_stream->PID_type == PID_TYPE_UNKNOWN) {
      /* set as elementary */
      ES_stream->PID_type = PID_TYPE_ELEMENTARY;
      /* set stream type */
      ES_stream->stream_type = stream_type;
      /* parse descriptor */
      ES_stream->ES_info = gst_mpeg_descriptor_parse (data, ES_info_length);
      /* set adaptor */
      gst_pes_filter_init (&ES_stream->filter);
      gst_pes_filter_set_callbacks (&ES_stream->filter,
		     (GstPESFilterData) gst_fluts_demux_data_cb,
                     (GstPESFilterResync) gst_fluts_demux_resync_cb,
                     ES_stream);
    }
    
    /* skip descriptor */
    data += ES_info_length;

    GST_DEBUG_OBJECT (demux, "  PMT stream_type: %d, PID: 0x%04x (ES_info_len %d)", 
		    stream_type, entry.PID, ES_info_length);

    g_array_append_val (PMT->entries, entry);

    entries -= 5 + ES_info_length;
  }
  CRC = GST_READ_UINT32_BE (data);
  GST_DEBUG_OBJECT (demux, "PMT CRC: 0x%08x", CRC);

  if (demux->current_PMT == 0) {
    GST_DEBUG_OBJECT (demux, "activating first PMT 0x%08x", stream->PID);
    demux->current_PMT = stream->PID;
  }

  return TRUE;

  /* ERRORS */
wrong_crc:
  {
    GST_DEBUG_OBJECT (demux, "wrong crc");
    return FALSE;
  }
same_version:
  {
    GST_DEBUG_OBJECT (demux, "same version as existing PMT");
    return TRUE;
  }
wrong_id:
  {
    GST_DEBUG_OBJECT (demux, "expected table_id == 0, got 0x%02x", data[0]);
    return FALSE;
  }
wrong_sync:
  {
    GST_DEBUG_OBJECT (demux, "expected sync 10, got %02x", data[0]);
    return FALSE;
  }
wrong_seclen:
  {
    GST_DEBUG_OBJECT (demux, "first two bits of section length must be 0, got %02x", data[0]);
    return FALSE;
  }
seclen_too_big:
  {
    GST_DEBUG_OBJECT (demux, "section length > 1021, got %d", section_length);
    return FALSE;
  }
wrong_pilen:
  {
    GST_DEBUG_OBJECT (demux, "first two bits of program_info length must be 0, got %02x", data[0]);
    return FALSE;
  }
wrong_esilen:
  {
    GST_DEBUG_OBJECT (demux, "first two bits of ES_info length must be 0, got %02x", data[0]);
    g_array_free (stream->PMT.entries, TRUE);
    return FALSE;
  }
}

/*
 * private_section() {
 *   table_id                                       8 uimsbf
 *   section_syntax_indicator                       1 bslbf
 *   private_indicator                              1 bslbf
 *   reserved                                       2 bslbf
 *   private_section_length                        12 uimsbf
 *   if (section_syntax_indicator == '0') {
 *     for ( i=0;i<N;i++) {
 *       private_data_byte                          8 bslbf
 *     }
 *   }
 *   else {
 *     table_id_extension                          16 uimsbf
 *     reserved                                     2 bslbf
 *     version_number                               5 uimsbf
 *     current_next_indicator                       1 bslbf
 *     section_number                               8 uimsbf
 *     last_section_number                          8 uimsbf
 *     for ( i=0;i<private_section_length-9;i++) {
 *       private_data_byte                          8 bslbf
 *     }
 *     CRC_32                                      32 rpchof
 *   }
 * }
 */
#if 0 /* JS - Commented out until it's used to satisfy -Werror */
static gboolean
gst_fluts_demux_parse_private_section (GstFluTSDemux * demux, gboolean pusi, guint8 *data, guint datalen)
{
  guint8 pointer;

  if (pusi) {
    pointer = *data++;
    GST_DEBUG_OBJECT (demux, "pointer 0x%02x", pointer);
    data += pointer;
    datalen -= (pointer + 1);
  }

  GST_DEBUG_OBJECT (demux, "parsing private section");
  return  TRUE;
}
#endif

/*
 * adaptation_field() {
 *   adaptation_field_length                              8 uimsbf
 *   if(adaptation_field_length >0) {
 *     discontinuity_indicator                            1 bslbf
 *     random_access_indicator                            1 bslbf
 *     elementary_stream_priority_indicator               1 bslbf
 *     PCR_flag                                           1 bslbf
 *     OPCR_flag                                          1 bslbf
 *     splicing_point_flag                                1 bslbf
 *     transport_private_data_flag                        1 bslbf
 *     adaptation_field_extension_flag                    1 bslbf
 *     if(PCR_flag == '1') {
 *       program_clock_reference_base                    33 uimsbf
 *       reserved                                         6 bslbf
 *       program_clock_reference_extension                9 uimsbf
 *     }
 *     if(OPCR_flag == '1') {
 *       original_program_clock_reference_base           33 uimsbf
 *       reserved                                         6 bslbf
 *       original_program_clock_reference_extension       9 uimsbf
 *     }
 *     if (splicing_point_flag == '1') {
 *       splice_countdown                                 8 tcimsbf
 *     }
 *     if(transport_private_data_flag == '1') {
 *       transport_private_data_length                    8 uimsbf
 *       for (i=0; i<transport_private_data_length;i++){
 *         private_data_byte                              8 bslbf
 *       }
 *     }
 *     if (adaptation_field_extension_flag == '1' ) {
 *       adaptation_field_extension_length                8 uimsbf
 *       ltw_flag                                         1 bslbf
 *       piecewise_rate_flag                              1 bslbf
 *       seamless_splice_flag                             1 bslbf
 *       reserved                                         5 bslbf
 *       if (ltw_flag == '1') {
 *         ltw_valid_flag                                 1 bslbf
 *         ltw_offset                                    15 uimsbf
 *       }
 *       if (piecewise_rate_flag == '1') {
 *         reserved                                       2 bslbf
 *         piecewise_rate                                22 uimsbf
 *       }
 *       if (seamless_splice_flag == '1'){
 *         splice_type                                    4 bslbf
 *         DTS_next_AU[32..30]                            3 bslbf
 *         marker_bit                                     1 bslbf
 *         DTS_next_AU[29..15]                           15 bslbf
 *         marker_bit                                     1 bslbf
 *         DTS_next_AU[14..0]                            15 bslbf
 *         marker_bit                                     1 bslbf
 *       }
 *       for ( i=0;i<N;i++) {
 *         reserved                                       8 bslbf
 *       }
 *     }
 *     for (i=0;i<N;i++){
 *       stuffing_byte                                    8 bslbf
 *     }
 *   }
 * }
 */
static gboolean
gst_fluts_demux_parse_adaptation_field (GstFluTSStream *stream,
		guint8 *data, guint *consumed)
{
  GstFluTSDemux *demux;
  guint8 length;

  demux = stream->demux;

  length = *data++;

  GST_DEBUG_OBJECT (demux, "parsing adaptation field, length %d", length);

  if (length > 0) {
    guint8 flags = *data++;

    GST_DEBUG_OBJECT (demux, "flags 0x%02x", flags);

    /* PCR_flag */
    if (flags & 0x10) {
      guint32 pcr1;
      guint16 pcr2;
      guint64 pcr, pcr_ext;

      pcr1 = GUINT32_FROM_BE (*(guint32 *) data);
      pcr2 = GUINT16_FROM_BE (*(guint16 *) (data + 4));
      pcr = ((guint64) pcr1) << 1;
      pcr |= (pcr2 & 0x8000) >> 15;
      pcr_ext = (pcr2 & 0x01ff);
      if (pcr_ext)
        pcr = (pcr * 300 + pcr_ext % 300) / 300;
      GST_DEBUG_OBJECT (demux, "have PCR %" G_GUINT64_FORMAT, pcr);
      stream->last_PCR = pcr;
      data += 6;
    }
    /* OPCR_flag */
    if (flags & 0x08) {
      guint32 opcr1;
      guint16 opcr2;
      guint64 opcr, opcr_ext;

      opcr1 = GUINT32_FROM_BE (*(guint32 *) data);
      opcr2 = GUINT16_FROM_BE (*(guint16 *) (data + 4));
      opcr = ((guint64) opcr1) << 1;
      opcr |= (opcr2 & 0x8000) >> 15;
      opcr_ext = (opcr2 & 0x01ff);
      if (opcr_ext)
        opcr = (opcr * 300 + opcr_ext % 300) / 300;
      GST_DEBUG_OBJECT (demux, "have OPCR %" G_GUINT64_FORMAT, opcr);
      stream->last_OPCR = opcr;
      data += 6;
    }
    /* splicing_point_flag */
    if (flags & 0x04) {
      guint8 splice_countdown;
      splice_countdown = *data++;
      GST_DEBUG_OBJECT (demux, "have splicing point, countdown %d", splice_countdown);
    }
    /* transport_private_data_flag */
    if (flags & 0x02) {
      guint8 plength = *data++;
      GST_DEBUG_OBJECT (demux, "have private data, length: %d", plength);
      data += plength;
    }
    /* adaptation_field_extension_flag */
    if (flags & 0x01) {
      GST_DEBUG_OBJECT (demux, "have field extension");
    }
  }

  *consumed = length + 1;
  return TRUE;
}

/*
 * program_association_section() {
 *   table_id                               8 uimsbf   == 0x00
 *   section_syntax_indicator               1 bslbf    == 1
 *   '0'                                    1 bslbf    == 0
 *   reserved                               2 bslbf
 *   section_length                        12 uimsbf   == 00xxxxx...
 *   transport_stream_id                   16 uimsbf
 *   reserved                               2 bslbf
 *   version_number                         5 uimsbf
 *   current_next_indicator                 1 bslbf
 *   section_number                         8 uimsbf
 *   last_section_number                    8 uimsbf
 *   for (i=0; i<N;i++) {
 *     program_number                      16 uimsbf
 *     reserved                             3 bslbf
 *     if(program_number == '0') {
 *       network_PID                       13 uimsbf
 *     }
 *     else {
 *       program_map_PID                   13 uimsbf
 *     }
 *   }
 *   CRC_32                                32 rpchof
 * }
 */
static gboolean
gst_fluts_stream_parse_pat (GstFluTSStream *stream, 
		gboolean pusi, guint8 *data, guint datalen)
{
  GstFluTSDemux *demux;
  guint16 section_length;
  gint entries;
  guint32 CRC;
  guint8 pointer;
  guint version_number;
  GstFluTSPAT *PAT;

  demux = stream->demux;

  if (pusi) {
    pointer = *data++;
    GST_DEBUG_OBJECT (demux, "pointer 0x%02x", pointer);
    data += pointer;
    datalen -= (pointer + 1);
  }

  if (*data++ != 0x00)
    goto wrong_id;
  if ((data[0] & 0xc0) != 0x80)
    goto wrong_sync;
  if ((data[0] & 0x0c) != 0x00)
    goto wrong_seclen;

  section_length = GST_READ_UINT16_BE (data);
  section_length &= 0x0fff;
  data += 2;
  GST_DEBUG_OBJECT (demux, "PAT section_length: %d", section_length);

  if (section_length > 1021)
    goto seclen_too_big;

  if (gst_fluts_demux_calc_crc32 (data-3, section_length+3) != 0)
    goto wrong_crc;

  PAT = &stream->PAT;

  version_number = (data[0] & 0x3e) >> 1;
  GST_DEBUG_OBJECT (demux, "PAT version_number: %d", version_number);
  if (version_number == PAT->version_number)
    goto same_version;

  PAT->version_number = version_number;

  PAT->transport_stream_id = GST_READ_UINT16_BE (data);
  data += 2;
  GST_DEBUG_OBJECT (demux, "PAT stream_id: %d", PAT->transport_stream_id);

  PAT->current_next_indicator = (*data++ & 0x01) == 0x01;
  PAT->section_number = *data++;
  PAT->last_section_number = *data++;

  GST_DEBUG_OBJECT (demux, "PAT current_next_indicator: %d", PAT->current_next_indicator);
  GST_DEBUG_OBJECT (demux, "PAT section_number: %d", PAT->section_number);
  GST_DEBUG_OBJECT (demux, "PAT last_section_number: %d", PAT->last_section_number);

  PAT->entries = g_array_new (FALSE, TRUE, sizeof (GstFluTSPATEntry));

  /* 5 bytes after section length a 4 bytes CRC */
  entries = (section_length - 9) / 4;
  while (entries--) {
    GstFluTSPATEntry entry;
    GstFluTSStream *PMT_stream;

    entry.program_number = GST_READ_UINT16_BE (data);
    data += 2;
    entry.PID = GST_READ_UINT16_BE (data);
    entry.PID &= 0x1fff;
    data += 2;

    /* get/create elementary stream */
    PMT_stream = gst_fluts_demux_get_stream_for_PID (demux, entry.PID);
    if (PMT_stream->PID_type != PID_TYPE_PROGRAM_MAP) {
      /* set as program map */
      PMT_stream->PID_type = PID_TYPE_PROGRAM_MAP;
    }

    g_array_append_val (PAT->entries, entry);

    GST_DEBUG_OBJECT (demux, "  PAT program: %d, PID 0x%04x", entry.program_number, entry.PID);
  }
  CRC = GST_READ_UINT32_BE (data);
  GST_DEBUG_OBJECT (demux, "PAT CRC: 0x%08x", CRC);

  return TRUE;

  /* ERRORS */
wrong_crc:
  {
    GST_DEBUG_OBJECT (demux, "wrong crc");
    return FALSE;
  }
same_version:
  {
    GST_DEBUG_OBJECT (demux, "same version as existing PAT");
    return TRUE;
  }
wrong_id:
  {
    GST_DEBUG_OBJECT (demux, "expected table_id == 0, got %02x", data[0]);
    return FALSE;
  }
wrong_sync:
  {
    GST_DEBUG_OBJECT (demux, "expected sync 10, got %02x", data[0]);
    return FALSE;
  }
wrong_seclen:
  {
    GST_DEBUG_OBJECT (demux, "first two bits of section length must be 0, got %02x", data[0]);
    return FALSE;
  }
seclen_too_big:
  {
    GST_DEBUG_OBJECT (demux, "section length > 1021, got %d", section_length);
    return FALSE;
  }
}

static gboolean
gst_fluts_demux_is_PMT (GstFluTSDemux *demux, guint16 PID)
{
  GstFluTSStream *stream;
  GstFluTSPAT *PAT;
  gint i;

  /* get the PAT */
  stream = demux->streams[PID_PROGRAM_ASSOCIATION_TABLE];
  if (stream == NULL || stream->PAT.entries == NULL)
    return FALSE;

  PAT = &stream->PAT;

  for (i = 0; i < PAT->entries->len; i++) {
    GstFluTSPATEntry *entry;

    entry = &g_array_index (PAT->entries, GstFluTSPATEntry, i);
    if (!entry)
      continue;

    if (entry->PID == PID)
      return TRUE;
  }
  return FALSE;
}

/*
 * transport_packet(){
 *   sync_byte                                                               8 bslbf == 0x47
 *   transport_error_indicator                                               1 bslbf
 *   payload_unit_start_indicator                                            1 bslbf
 *   transport _priority                                                     1 bslbf
 *   PID                                                                    13 uimsbf
 *   transport_scrambling_control                                            2 bslbf
 *   adaptation_field_control                                                2 bslbf
 *   continuity_counter                                                      4 uimsbf
 *   if(adaptation_field_control=='10' || adaptation_field_control=='11'){
 *     adaptation_field()
 *   }
 *   if(adaptation_field_control=='01' || adaptation_field_control=='11') {
 *     for (i=0;i<N;i++){
 *       data_byte                                                           8 bslbf
 *     }
 *   }
 * }
 */
static GstFlowReturn
gst_fluts_demux_parse_stream (GstFluTSDemux *demux, GstFluTSStream * stream, 
		const guint8 *data, guint datalen)
{
  GstFlowReturn ret;
  gboolean transport_error_indicator;
  gboolean payload_unit_start_indicator;
  gboolean transport_priority;
  guint16 PID;
  guint8 transport_scrambling_control;
  guint8 adaptation_field_control;
  guint8 continuity_counter;

  transport_error_indicator = (data[0] & 0x80) == 0x80;
  payload_unit_start_indicator = (data[0] & 0x40) == 0x40;
  transport_priority = (data[0] & 0x20) == 0x20;
  PID = stream->PID;
  transport_scrambling_control = (data[2] & 0xc0) >> 6;
  adaptation_field_control = (data[2] & 0x30) >> 4;
  continuity_counter = data[2] & 0x0f;

  data += 3;
  datalen -= 3;

  GST_DEBUG_OBJECT (demux, "afc 0x%x, pusi %d, PID 0x%04x", 
		  adaptation_field_control, payload_unit_start_indicator, PID);

  ret = GST_FLOW_OK;

  /* packets with adaptation_field_control == 0 must be skipped */
  if (adaptation_field_control == 0) 
    goto skip;

  /* parse adaption field if any */
  if (adaptation_field_control & 0x2) {
    guint consumed;

    if (!gst_fluts_demux_parse_adaptation_field (stream, (guint8 *) data, &consumed))
      goto done;

    data += consumed;
    datalen -= consumed;
  }

  if (adaptation_field_control & 0x1) { 
    /* now parse based on the stream type */
    switch (stream->PID_type) {
      case PID_TYPE_PROGRAM_ASSOCIATION:
	gst_fluts_stream_parse_pat (stream, 
		payload_unit_start_indicator, (guint8 *) data, datalen);
        break;
      case PID_TYPE_CONDITIONAL_ACCESS:
        gst_fluts_stream_parse_cat (stream, 
		payload_unit_start_indicator, (guint8 *) data, datalen);
        break;
      case PID_TYPE_PROGRAM_MAP:
	gst_fluts_stream_parse_pmt (stream, 
		payload_unit_start_indicator, (guint8 *) data, datalen);
        break;
      case PID_TYPE_NULL_PACKET:
	GST_DEBUG_OBJECT (demux, "skipping PID 0x%04x", PID);
        break;
      case PID_TYPE_UNKNOWN:
	GST_DEBUG_OBJECT (demux, "skipping unknown PID 0x%04x", PID);
        break;
      case PID_TYPE_ELEMENTARY:
      {
	GstBuffer *buf;

	if (payload_unit_start_indicator) {
          GST_DEBUG_OBJECT (demux, "new PES start for PID 0x%04x", PID);
	  gst_pes_filter_drain (&stream->filter);
	}
        buf = gst_buffer_new_and_alloc (datalen);
        memcpy (GST_BUFFER_DATA (buf), data, datalen);

        ret = gst_pes_filter_push (&stream->filter, buf);
	if (ret == GST_FLOW_NOT_LINKED)
	  ret = GST_FLOW_OK;
        break;
      }
    }
  }

done:
  return ret;

skip:
  {
    GST_DEBUG_OBJECT (demux, "skipping, adaptation_field_control == 0");
    return GST_FLOW_OK;
  }
}

static GstFlowReturn
gst_fluts_demux_parse_transport_packet (GstFluTSDemux * demux)
{
  GstFlowReturn ret;
  const guint8 *data;
  guint16 PID;
  guint datalen;
  GstFluTSStream *stream;

  if (!(data = gst_adapter_peek (demux->adapter, 188)))
    goto need_more_data;

  if (*data++ != MPEG_TS_SYNC_BYTE)
    goto lost_sync;

  /* we consumed the sync byte */
  datalen = 187;

  /* get PID */
  PID = ((data[0] & 0x1f) << 8) | data[1];

  /* get the stream. */
  stream = gst_fluts_demux_get_stream_for_PID (demux, PID);

  /* parse the stream */
  ret = gst_fluts_demux_parse_stream (demux, stream, data, datalen);

  gst_adapter_flush (demux->adapter, 188);

  return ret;

  /* ERRORS */
need_more_data:
  {
    GST_DEBUG_OBJECT (demux, "need more data");
    return GST_FLOW_NEED_MORE_DATA;
  }
lost_sync:
  {
    GST_DEBUG_OBJECT (demux, "lost sync");
    gst_adapter_flush (demux->adapter, 1);
    return GST_FLOW_LOST_SYNC;
  }
}

static gboolean
gst_fluts_demux_resync (GstFluTSDemux *demux)
{
  const guint8 *data;
  gint avail;
  guint8 code;
  gint offset;
  gboolean found;

  avail = gst_adapter_available (demux->adapter);
  if (avail < 1)
    goto need_data;

  data = gst_adapter_peek  (demux->adapter, avail);

  /* read currect code */
  code = *data;
  GST_DEBUG ("resync code = %02x", code);

  /* we are at byte 1 */
  offset = 1;

  found = code == MPEG_TS_SYNC_BYTE;

  while (offset < avail && !found) {
    code = data[offset++];
    GST_DEBUG ("  code = %02x", code);
    found = code == MPEG_TS_SYNC_BYTE;
  }

  if (offset > 1)
    gst_adapter_flush (demux->adapter, offset-1);

  return found;

need_data:
  {
    return FALSE;
  }
}

static GstFlowReturn
gst_fluts_demux_chain (GstPad *pad, GstBuffer *buffer)
{
  GstFluTSDemux *demux = GST_FLUTS_DEMUX (gst_pad_get_parent (pad));
  GstFlowReturn ret = GST_FLOW_OK;
  
  gst_adapter_push (demux->adapter, buffer);

  /* align adapter data to sync boundary */
  while (gst_fluts_demux_resync (demux)) {
    ret = gst_fluts_demux_parse_transport_packet (demux);
    switch (ret) {
      case GST_FLOW_NEED_MORE_DATA:
	ret = GST_FLOW_OK;
	goto done;
      case GST_FLOW_LOST_SYNC:
	ret = GST_FLOW_OK;
	break;
      case GST_FLOW_OK:
      default:
	break;
    }
  }
done:
  gst_object_unref (demux);

  return ret;
}

static GstStateChangeReturn
gst_fluts_demux_change_state (GstElement * element, GstStateChange transition)
{
  GstFluTSDemux *demux = GST_FLUTS_DEMUX (element);
  GstStateChangeReturn result;


  switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
      demux->adapter = gst_adapter_new();
      break;
    case GST_STATE_CHANGE_READY_TO_PAUSED:
      break;
    default:
      break;
  }

  result = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    case GST_STATE_CHANGE_READY_TO_NULL:
      g_object_unref (demux->adapter);
      break;
    default:
      break;
  }

  return result;
}

gboolean
gst_fluts_demux_plugin_init (GstPlugin * plugin)
{
  if (!gst_element_register (plugin, "flutsdemux",
      GST_RANK_PRIMARY, GST_TYPE_FLUTS_DEMUX))
    return FALSE;

  return TRUE;
}
