//*****************************************************************************
//                              PnlGnuCapTR.cpp                               *
//                             -----------------                              *
// Started     : 22/02/2004                                                   *
// Last Update : 16/05/2011                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapTR.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapTR, PnlAnaBase )

  EVT_BUTTON( ID_BTN_SETUP, PnlGnuCapTR::OnBtnSetup )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapTR::PnlGnuCapTR( wxWindow * poWin )
                        : PnlAnaBase( poWin ), m_oDlgCfgGEN( poWin )
{
  bSetSimrEng( eSIMR_GNUCAP );
  bSetAnaType( eCMD_TR );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapTR::~PnlGnuCapTR( )
{
}

//*****************************************************************************
// Createe the display objects.

void  PnlGnuCapTR::Create( void )
{
  wxStaticBox * poSbxSigSrc;
#ifndef LAYOUT_MNGR
  int           ix, iy;

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" Time Sweep ") );
#endif // LAYOUT_MNGR
  m_oPnlStart  .bSetName( wxT("Start Time") );
  m_oPnlStop   .bSetName( wxT("Stop Time") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( eUNITS_TIME );
  m_oPnlStop .bSetUnitsType( eUNITS_TIME );
  m_oPnlStep .bSetUnitsType( eUNITS_TIME );

  // Create and add generator controls
  poSbxSigSrc = new wxStaticBox( this, ID_UNUSED, wxT(" Signal Source "),
                        wxPoint( 6, 175 ),   wxSize( 241, 55 ) );
  m_oChoSrcName.Create( this, ID_CHO_SRCNAME,
                        wxPoint( 18, 195 ),  wxSize( 107, 29 ) );
  m_oBtnSetup  .Create( this, ID_BTN_SETUP, wxT("Setup ..."),
                        wxPoint( 130, 195 ), wxSize( 105, 29 ) );
  poSbxSigSrc->SetOwnFont( FONT_SLANT );

  PnlAnaBase::CreateInitC( ); // Create and add the initial conditions controls
  PnlAnaBase::CreateTemp( );  // Create and add the analysis temp. controls

//  PnlAnaBase::DoLayout( );    // Layout the panel's GUI objects

#ifdef LAYOUT_MNGR
  SetSwpPnlLbl( wxT(" Time Sweep ") );
#endif // LAYOUT_MNGR
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapTR::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )    bRtn = FALSE;

  // Set the default sweep values
  m_oPnlStart.bSetValue( (float)   0.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)   1.0 );
  m_oPnlStart.bSetUnits( wxT("msec") );
  m_oPnlStop .bSetUnits( wxT("msec") );
  m_oPnlStep .bSetUnits( wxT("msec") );

  // Set default initial condition values
  if( ! bSetInitC( eINITC_WARM ) ) bRtn = FALSE;

  // Clear the signal generator configuration dialog
  m_oDlgCfgGEN.bClear( );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapTR::bLoad( SimnGnuCap & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VI") );

  // Go no further if the TR command isn't valid
  if( ! roSimn.m_oCmdTR.bIsValid( ) )                        return( FALSE );

  // Set the sweep source
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdTR.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdTR.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdTR.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the initial conditions
  if( roSimn.m_oCmdTR.m_eInitC != eINITC_NONE )
    m_oRbxInitC.SetSelection( roSimn.m_oCmdTR.m_eInitC );

  // Load the generator setup dialog
  if( ! m_oDlgCfgGEN.bSetValues( roSimn.m_oCmdGEN ) )        bRtn = FALSE;

  if( roSimn.eGetAnaType( ) == eCMD_TR )
  {
    // Set the parameters to derive
    m_oCbxVoltage.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] );
    m_oCbxCurrent.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] );
    m_oCbxPower  .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] );
    m_oCbxResist .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] );
  }

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdTR.m_osTempC ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapTR::bSave( SimnGnuCap & roSimn )
{
  wxString  os1;

  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdTR.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdTR.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdTR.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the initial conditions
  roSimn.m_oCmdTR.m_eInitC = (eInitCType) m_oRbxInitC.GetSelection( );

  // Set the component to be used as a signal source
  if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
  {
    // Get the unmodified signal source component, set it as the signal source
    os1 = m_oChoSrcName.GetStringSelection( );
    (Component &) roSimn.m_oCpntSwpSrc = roSimn.roGetCpnt( os1 );
    roSimn.m_oCpntSwpSrc.m_osValue = wxT("GENERATOR(1)");
    // Get the GENERATOR command from the Generator Setup dialog
    m_oDlgCfgGEN.bGetValues( roSimn.m_oCmdGEN );
    if( ! roSimn.m_oCmdGEN.bIsValid( ) )
    {
      os1 = wxT("The signal source ")
          + roSimn.m_oCmdGEN.rosGetErrMsg( ) + wxT(".   ");
      SetErrMsg( os1 );
    }
  }
  else
  {
    roSimn.m_oCpntSwpSrc.bClear( );
    roSimn.m_oCmdGEN    .bSetDefaults( );
  }

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_TR;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdTR.m_osTempC = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdTR.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdTR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdTR.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Generator component setup button control event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapTR::OnBtnSetup( wxCommandEvent & roEvtCmd )
{
  m_oDlgCfgGEN.CenterOnParent( );
  m_oDlgCfgGEN.ShowModal( );
}

//*****************************************************************************
