//*****************************************************************************
//                                Component.cpp                               *
//                               ---------------                              *
//  Started     : 14/05/2004                                                  *
//  Last Update : 04/06/2007                                                  *
//  Copyright   : (C) 2004 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/Component.hpp"

//*****************************************************************************
// Constructor.

Component::Component( const wxChar * psDefn )
{
  Clear( );
  
  bSetDefn( psDefn );
}

//*****************************************************************************
// Destructor.

Component::~Component( )
{
}

//*****************************************************************************
// Parse a component definition string extracted from a net list.
//
// Argument List:
//   psDefn - A string containing a component definition
//
// Return Values:
//   TRUE  - Success (The component definition was    parsed successfully)
//   FALSE - Failure (The component definition wasn't parsed successfully)

bool  Component::bParse( const wxChar * psDefn )
{
  wxStringTokenizer  ostk1( psDefn );
  wxString           os1=psDefn;
  size_t             szt1, szt2;

  // Do preliminary component definition validity checks
  if( os1.Length( ) < 7 )                return( FALSE );  // Test line length
  if( isalpha( os1.GetChar( 0 ) ) == 0 ) return( FALSE );  // Test first char.
  if( ostk1.CountTokens( ) < 4 )         return( FALSE );  // Test field count

  // Determine the component type
  switch( toupper( os1.GetChar( 0 ) ) )
  {
    case wxT('C'): m_eType = eCPNT_CAP;    break; // Capacitor
    case wxT('D'): m_eType = eCPNT_DIODE;  break; // Diode
    case wxT('E'): m_eType = eCPNT_VCVS;   break; // Voltage Ctrled Voltage Src
    case wxT('F'): m_eType = eCPNT_CCCS;   break; // Current Ctrled Current Src
    case wxT('G'): m_eType = eCPNT_VCCS;   break; // Voltage Ctrled Current Src
    case wxT('H'): m_eType = eCPNT_CCVS;   break; // Current Ctrled Voltage Src
    case wxT('I'): m_eType = eCPNT_ICS;    break; // Independent Current Source
    case wxT('J'): m_eType = eCPNT_JFET;   break; // Junction Field-Effect Transistor
    case wxT('K'): m_eType = eCPNT_CIND;   break; // Coupled (Mutual) Inductors
    case wxT('L'): m_eType = eCPNT_IND;    break; // Inductor
    case wxT('M'): m_eType = eCPNT_MOS;    break; // Metal-Oxide Semiconductor FET
    case wxT('Q'): m_eType = eCPNT_BJT;    break; // Bipolar Junction Transistor
    case wxT('R'): m_eType = eCPNT_RES;    break; // Resistor
    case wxT('S'): m_eType = eCPNT_VCSW;   break; // Voltage Controlled Switch
    case wxT('T'): m_eType = eCPNT_TLINE;  break; // Transmission Line
    case wxT('U'): m_eType = eCPNT_LOGIC;  break; // Logic Device
    case wxT('V'): m_eType = eCPNT_IVS;    break; // Independent Voltage Source
    case wxT('W'): m_eType = eCPNT_CCSW;   break; // Current Controlled Switch
    case wxT('X'): m_eType = eCPNT_SUBCKT; break; // Sub-circuit
    case wxT('Y'): m_eType = eCPNT_ADM;    break; // Admittance
    default      : m_eType = eCPNT_NONE;          // None type selected
  }

  // Extract the component name
  m_osName = ostk1.GetNextToken( );

  // Extract the nodes the component is connected to
  szt2 = (size_t) ostk1.CountTokens( );
  for( szt1=1; szt1<szt2; szt1++ )
  {
    m_oasNodes.Add( ostk1.GetNextToken( ) );
    if( szt1 >= 2 )
    {
      if( m_eType == eCPNT_CAP   ) break; // Capacitor
      if( m_eType == eCPNT_DIODE ) break; // Diode
      if( m_eType == eCPNT_ICS   ) break; // Independent Current Source
      if( m_eType == eCPNT_IND   ) break; // Inductor
      if( m_eType == eCPNT_RES   ) break; // Resistor
      if( m_eType == eCPNT_IVS   ) break; // Independent Voltage Source
      if( m_eType == eCPNT_ADM   ) break; // Admittance
    }
  }

  // Extract the component value
  m_osValue = ostk1.GetString( );

  // Set the state of m_bIsOk flag
  m_bIsOk = TRUE;
  if( m_eType == eCPNT_NONE      ) m_bIsOk = FALSE;
  if( m_osName.IsEmpty( )        ) m_bIsOk = FALSE;
  if( m_oasNodes.GetCount( ) < 2 ) m_bIsOk = FALSE;
  if( m_osValue.IsEmpty( )       ) m_bIsOk = FALSE;

  return( bIsOk( ) );
}

//*****************************************************************************
// Clear the object attributes.

void  Component::Clear( void )
{
  m_osDefn  .Clear( );
  m_osName  .Clear( );
  m_oasNodes.Clear( );
  m_osValue .Clear( );
  m_eType = eCPNT_NONE;
  m_bIsOk = FALSE;
}

//*****************************************************************************
// Set the component definition string extracted from a net list.
//
// Argument List:
//   psDefn - A string containing a component definition
//
// Return Values:
//   TRUE  - Success (The component definition was    valid)
//   FALSE - Failure (The component definition wasn't valid)

bool  Component::bSetDefn( const wxChar * psDefn )
{
  Clear( );
  
  if( ! bParse( psDefn ) ) return( FALSE );

  m_osDefn = psDefn;

  return( TRUE );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List:
//   psPrefix - A prefix to every line displayed (usually just spaces)

void  Component::Print( const wxChar * psPrefix )
{
  wxString  osPrefix;
  size_t    sz1;

  if( psPrefix != NULL ) osPrefix = psPrefix;
  else                   osPrefix = wxT("");

  std::cout << osPrefix.mb_str( ) << "m_osName   : " << m_osName.mb_str( ) << '\n';
  std::cout << osPrefix.mb_str( ) << "m_oasNodes : ";
  if( m_oasNodes.GetCount( ) > 0 )
  {
    sz1 = 0;
    while( 1 )
    {
      std::cout << m_oasNodes.Item( sz1 ).mb_str( );
      if( ++sz1 >= m_oasNodes.GetCount( ) ) break;
      std::cout << ", ";
    }
  }
  else std::cout << "None";
  std::cout << '\n';
  std::cout << osPrefix.mb_str( ) << "m_osValue  : " << m_osValue.mb_str( ) << '\n';
  std::cout << osPrefix.mb_str( ) << "m_osDefn   : " << m_osDefn.mb_str( )  << '\n';
}

//*****************************************************************************
