package Gscan2pdf::Dialog::Scan::Sane;

use warnings;
use strict;
use Gscan2pdf::Dialog::Scan;
use Glib qw(TRUE FALSE);   # To get TRUE and FALSE
use Sane 0.05;             # To get SANE_NAME_PAGE_WIDTH & SANE_NAME_PAGE_HEIGHT
use Gscan2pdf::Frontend::Sane;
use Locale::gettext 1.05;    # For translations
use feature "switch";

# logger duplicated from Gscan2pdf::Dialog::Scan
# to ensure that SET_PROPERTIES gets called in both places
use Glib::Object::Subclass Gscan2pdf::Dialog::Scan::, properties => [
 Glib::ParamSpec->scalar(
  'logger',                              # name
  'Logger',                              # nick
  'Log::Log4perl::get_logger object',    # blurb
  [qw/readable writable/]                # flags
 ),
];

my $SANE_NAME_SCAN_TL_X   = SANE_NAME_SCAN_TL_X;
my $SANE_NAME_SCAN_TL_Y   = SANE_NAME_SCAN_TL_Y;
my $SANE_NAME_SCAN_BR_X   = SANE_NAME_SCAN_BR_X;
my $SANE_NAME_SCAN_BR_Y   = SANE_NAME_SCAN_BR_Y;
my $SANE_NAME_PAGE_HEIGHT = SANE_NAME_PAGE_HEIGHT;
my $SANE_NAME_PAGE_WIDTH  = SANE_NAME_PAGE_WIDTH;
my ( $d, $d_sane, $logger, $tooltips );

sub INIT_INSTANCE {
 my $self = shift;
 $tooltips = Gtk2::Tooltips->new;
 $tooltips->enable;

 $d      = Locale::gettext->domain(Glib::get_application_name);
 $d_sane = Locale::gettext->domain('sane-backends');
 return $self;
}

sub SET_PROPERTY {
 my ( $self, $pspec, $newval ) = @_;
 my $name   = $pspec->get_name;
 my $oldval = $self->get($name);
 $self->{$name} = $newval;
 if (( defined($newval) and defined($oldval) and $newval ne $oldval )
  or ( defined($newval) xor defined($oldval) ) )
 {
  if ( $name eq 'logger' ) { $logger = $self->get('logger') }
 }
 $self->SUPER::SET_PROPERTY( $pspec, $newval );
 return;
}

# Run Sane->get_devices

sub get_devices {
 my ($self) = @_;

 my $pbar;
 my $hboxd = $self->{hboxd};
 Gscan2pdf::Frontend::Sane->get_devices(
  sub {

   # Set up ProgressBar
   $pbar = Gtk2::ProgressBar->new;
   $pbar->set_pulse_step(.1);
   $pbar->set_text( $d->get('Fetching list of devices') );
   $hboxd->pack_start( $pbar, TRUE, TRUE, 0 );
   $hboxd->hide_all;
   $hboxd->show;
   $pbar->show;
  },
  sub {
   $pbar->pulse;
  },
  sub {
   my ($data) = @_;
   $pbar->destroy;
   my @device_list = @{$data};
   use Data::Dumper;
   $logger->info( "Sane->get_devices returned: ", Dumper( \@device_list ) );
   if ( @device_list == 0 ) {
    $self->signal_emit( 'process-error', 'get_devices',
     $d->get('No devices found') );
    $self->destroy;
    undef $self;
    return FALSE;
   }
   $self->set( 'device-list', \@device_list );
   $hboxd->show_all;
  }
 );
 return;
}

# Scan device-dependent scan options

sub scan_options {
 my ($self) = @_;

 # Remove any existing pages
 while ( $self->{notebook}->get_n_pages > 1 ) {
  $self->{notebook}->remove_page(-1);
 }

 # Ghost the scan button whilst options being updated
 $self->{sbutton}->set_sensitive(FALSE) if ( defined $self->{sbutton} );

 my $signal;
 Gscan2pdf::Frontend::Sane->open_device(
  device_name      => $self->get('device'),
  started_callback => sub {
   $self->signal_emit( 'started-process', $d->get('Opening device') );
  },
  running_callback => sub {
   $self->signal_emit( 'changed-progress', -1, undef );
  },
  finished_callback => sub {
   $self->signal_emit( 'finished-process', 'open_device' );
   Gscan2pdf::Frontend::Sane->find_scan_options(
    sub {    # started callback
     $self->signal_emit( 'started-process', $d->get('Retrieving options') );
    },
    sub {    # running callback
     $self->signal_emit( 'changed-progress', -1, undef );
    },
    sub {    # finished callback
     my ($data) = @_;
     my $options = Gscan2pdf::Scanner::Options->new_from_data($data);
     $self->_initialise_options($options);

     $self->signal_emit( 'finished-process', 'find_scan_options' );

     # This fires the reloaded-scan-options signal,
     # so don't set this until we have finished
     $self->set( 'available-scan-options', $options );
    },
    sub {    # error callback
     my ($message) = @_;
     $self->signal_emit( 'process-error', 'find_scan_options',
      $d->get( 'Error retrieving scanner options: ' . $message ) );
     $self->destroy;
    }
   );
  },
  error_callback => sub {
   my ($message) = @_;
   $self->signal_emit( 'process-error', 'open_device',
    $d->get( 'Error opening device: ' . $message ) );
   $self->destroy;
  }
 );

 return;
}

sub _initialise_options {    ## no critic (ProhibitExcessComplexity)
 my ( $self, $options ) = @_;
 $logger->debug( "Sane->get_option_descriptor returned: ", Dumper($options) );

 my ( $group, $vbox, $hboxp );
 my $num_dev_options = $options->num_options;

 # We have hereby removed the active profile and paper,
 # so update the properties without triggering the signals
 $self->{profile}       = undef;
 $self->{paper_formats} = undef;
 $self->{paper}         = undef;

 delete $self->{combobp};    # So we don't carry over from one device to another
 for ( my $i = 1 ; $i < $num_dev_options ; ++$i ) {
  my $opt = $options->by_index($i);

  # Notebook page for group
  if ( $opt->{type} == SANE_TYPE_GROUP or not defined($vbox) ) {
   $vbox = Gtk2::VBox->new;
   $group =
       $opt->{type} == SANE_TYPE_GROUP
     ? $d_sane->get( $opt->{title} )
     : $d->get('Scan Options');
   $self->{notebook}->append_page( $vbox, $group );
   next;
  }

  next unless ( $opt->{cap} & SANE_CAP_SOFT_DETECT );

  # Widget
  my ( $widget, $val );
  $val = $opt->{val};

  # Define HBox for paper size here
  # so that it can be put before first geometry option
  if ( _geometry_option($opt) and not defined($hboxp) ) {
   $hboxp = Gtk2::HBox->new;
   $vbox->pack_start( $hboxp, FALSE, FALSE, 0 );
  }

  # HBox for option
  my $hbox = Gtk2::HBox->new;
  $vbox->pack_start( $hbox, FALSE, TRUE, 0 );
  $hbox->set_sensitive(FALSE)
    if ( $opt->{cap} & SANE_CAP_INACTIVE
   or not $opt->{cap} & SANE_CAP_SOFT_SELECT );

  if ( $opt->{max_values} < 2 ) {

   # Label
   if ( $opt->{type} != SANE_TYPE_BUTTON ) {
    my $label = Gtk2::Label->new( $d_sane->get( $opt->{title} ) );
    $hbox->pack_start( $label, FALSE, FALSE, 0 );
   }

   # CheckButton
   if ( $opt->{type} == SANE_TYPE_BOOL )
   {    ## no critic (ProhibitCascadingIfElse)
    $widget = Gtk2::CheckButton->new;
    $widget->set_active(TRUE) if ($val);
    $widget->{signal} = $widget->signal_connect(
     toggled => sub {
      my $value = $widget->get_active;
      $self->set_option( $opt, $value );
     }
    );
   }

   # Button
   elsif ( $opt->{type} == SANE_TYPE_BUTTON ) {
    $widget = Gtk2::Button->new( $d_sane->get( $opt->{title} ) );
    $widget->{signal} = $widget->signal_connect(
     clicked => sub {
      $self->set_option( $opt, $val );
     }
    );
   }

   # SpinButton
   elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_RANGE ) {
    my $step = 1;
    $step = $opt->{constraint}{quant} if ( $opt->{constraint}{quant} );
    $widget = Gtk2::SpinButton->new_with_range( $opt->{constraint}{min},
     $opt->{constraint}{max}, $step );

    # Set the default
    $widget->set_value($val)
      if ( defined $val and not $opt->{cap} & SANE_CAP_INACTIVE );
    $widget->{signal} = $widget->signal_connect(
     'value-changed' => sub {
      my $value = $widget->get_value;
      $self->set_option( $opt, $value );
     }
    );
   }

   # ComboBox
   elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_STRING_LIST
    or $opt->{constraint_type} == SANE_CONSTRAINT_WORD_LIST )
   {
    $widget = Gtk2::ComboBox->new_text;
    my $index = 0;
    for ( my $i = 0 ; $i < @{ $opt->{constraint} } ; ++$i ) {
     $widget->append_text( $d_sane->get( $opt->{constraint}[$i] ) );
     $index = $i if ( defined $val and $opt->{constraint}[$i] eq $val );
    }

    # Set the default
    $widget->set_active($index) if ( defined $index );
    $widget->{signal} = $widget->signal_connect(
     changed => sub {
      my $i = $widget->get_active;
      $self->set_option( $opt, $opt->{constraint}[$i] );
     }
    );
   }

   # Entry
   elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_NONE ) {
    $widget = Gtk2::Entry->new;

    # Set the default
    $widget->set_text($val)
      if ( defined $val and not $opt->{cap} & SANE_CAP_INACTIVE );
    $widget->{signal} = $widget->signal_connect(
     activate => sub {
      my $value = $widget->get_text;
      $self->set_option( $opt, $value );
     }
    );
   }
  }
  else {    # $opt->{max_values} > 1
   $widget = Gtk2::Button->new( $d_sane->get( $opt->{title} ) );
   $widget->{signal} = $widget->signal_connect(
    clicked => \&multiple_values_button_callback,
    [ $self, $opt ]
   );
  }

  $self->_pack_widget( $widget, [ $options, $opt, $hbox, $hboxp ] );
 }

 # Set defaults
 my $sane_device = Gscan2pdf::Frontend::Sane->device();

 # Show new pages
 for ( my $i = 1 ; $i < $self->{notebook}->get_n_pages ; $i++ ) {
  $self->{notebook}->get_nth_page($i)->show_all;
 }

 $self->{sbutton}->set_sensitive(TRUE);
 $self->{sbutton}->grab_focus;
 return;
}

# Return true if we have a valid geometry option

sub _geometry_option {
 my ($opt) = @_;
 return
       ( $opt->{type} == SANE_TYPE_FIXED or $opt->{type} == SANE_TYPE_INT )
   and ( $opt->{unit} == SANE_UNIT_MM or $opt->{unit} == SANE_UNIT_PIXEL )
   and ( $opt->{name} =~
/^(?:$SANE_NAME_SCAN_TL_X|$SANE_NAME_SCAN_TL_Y|$SANE_NAME_SCAN_BR_X|$SANE_NAME_SCAN_BR_Y|$SANE_NAME_PAGE_HEIGHT|$SANE_NAME_PAGE_WIDTH)$/x
   );
}

sub _create_paper_widget {
 my ( $self, $options, $hboxp ) = @_;

 # Only define the paper size once the rest of the geometry widgets
 # have been created
 if (
      defined( $options->{box}{$SANE_NAME_SCAN_BR_X} )
  and defined( $options->{box}{$SANE_NAME_SCAN_BR_Y} )
  and defined( $options->{box}{$SANE_NAME_SCAN_TL_X} )
  and defined( $options->{box}{$SANE_NAME_SCAN_TL_Y} )
  and ( not defined $options->by_name(SANE_NAME_PAGE_HEIGHT)
   or defined( $options->{box}{$SANE_NAME_PAGE_HEIGHT} ) )
  and ( not defined $options->by_name(SANE_NAME_PAGE_WIDTH)
   or defined( $options->{box}{$SANE_NAME_PAGE_WIDTH} ) )
  and not defined( $self->{combobp} )
   )
 {

  # Paper list
  my $label = Gtk2::Label->new( $d->get('Paper size') );
  $hboxp->pack_start( $label, FALSE, FALSE, 0 );

  $self->{combobp} = Gtk2::ComboBox->new_text;
  $self->{combobp}->append_text( $d->get('Manual') );
  $self->{combobp}->append_text( $d->get('Edit') );
  $tooltips->set_tip( $self->{combobp},
   $d->get('Selects or edits the paper size') );
  $hboxp->pack_end( $self->{combobp}, FALSE, FALSE, 0 );
  $self->{combobp}->set_active(0);
  $self->{combobp}->signal_connect(
   changed => sub {

    if ( $self->{combobp}->get_active_text eq $d->get('Edit') ) {
     $self->edit_paper;
    }
    elsif ( $self->{combobp}->get_active_text eq $d->get('Manual') ) {
     for (
      ( SANE_NAME_SCAN_TL_X, SANE_NAME_SCAN_TL_Y,
       SANE_NAME_SCAN_BR_X,   SANE_NAME_SCAN_BR_Y,
       SANE_NAME_PAGE_HEIGHT, SANE_NAME_PAGE_WIDTH
      )
       )
     {
      $options->{box}{$_}->show_all if ( defined $options->{box}{$_} );
     }
    }
    else {
     my $paper   = $self->{combobp}->get_active_text;
     my $formats = $self->get('paper-formats');
     if ( defined( $options->by_name(SANE_NAME_PAGE_HEIGHT) )
      and defined( $options->by_name(SANE_NAME_PAGE_WIDTH) ) )
     {
      $options->by_name(SANE_NAME_PAGE_HEIGHT)->{widget}
        ->set_value( $formats->{$paper}{y} + $formats->{$paper}{t} );
      $options->by_name(SANE_NAME_PAGE_WIDTH)->{widget}
        ->set_value( $formats->{$paper}{x} + $formats->{$paper}{l} );
     }

     $options->by_name(SANE_NAME_SCAN_TL_X)->{widget}
       ->set_value( $formats->{$paper}{l} );
     $options->by_name(SANE_NAME_SCAN_TL_Y)->{widget}
       ->set_value( $formats->{$paper}{t} );
     $options->by_name(SANE_NAME_SCAN_BR_X)->{widget}
       ->set_value( $formats->{$paper}{x} + $formats->{$paper}{l} );
     $options->by_name(SANE_NAME_SCAN_BR_Y)->{widget}
       ->set_value( $formats->{$paper}{y} + $formats->{$paper}{t} );
     Glib::Idle->add(
      sub {
       for (
        ( SANE_NAME_SCAN_TL_X, SANE_NAME_SCAN_TL_Y,
         SANE_NAME_SCAN_BR_X,   SANE_NAME_SCAN_BR_Y,
         SANE_NAME_PAGE_HEIGHT, SANE_NAME_PAGE_WIDTH
        )
         )
       {
        $options->{box}{$_}->hide_all if ( defined $options->{box}{$_} );
       }
      }
     );

     # Do this last, as it fires the changed-paper signal
     $self->set( 'paper', $paper );
    }
   }
  );
 }
 return;
}

sub _pack_widget {
 my ( $self, $widget, $data ) = @_;
 my ( $options, $opt, $hbox, $hboxp ) = @$data;
 if ( defined $widget ) {
  $opt->{widget} = $widget;
  if ( $opt->{type} == SANE_TYPE_BUTTON or $opt->{max_values} > 1 ) {
   $hbox->pack_end( $widget, TRUE, TRUE, 0 );
  }
  else {
   $hbox->pack_end( $widget, FALSE, FALSE, 0 );
  }
  $tooltips->set_tip( $widget, $d_sane->get( $opt->{desc} ) );

  # Look-up to hide/show the box if necessary
  $options->{box}{ $opt->{name} } = $hbox
    if ( _geometry_option($opt) );

  $self->_create_paper_widget( $options, $hboxp );

 }
 else {
  $logger->warn("Unknown type $opt->{type}");
 }
 return;
}

# Update the sane option in the thread
# If necessary, reload the options,
# and walking the options tree, update the widgets

sub set_option {
 my ( $self, $option, $val ) = @_;

 my $current = $self->{current_scan_options};

 # Cache option
 push @$current, { $option->{name} => $val };

 # Note any duplicate options, keeping only the last entry.
 my %seen;

 my $j = $#{$current};
 while ( $j > -1 ) {
  my ($opt) =
    keys( %{ $current->[$j] } );
  $seen{$opt}++;
  if ( $seen{$opt} > 1 ) {
   splice @$current, $j, 1;
  }
  $j--;
 }
 $self->{current_scan_options} = $current;

 my $signal;
 my $options = $self->get('available-scan-options');
 Gscan2pdf::Frontend::Sane->set_option(
  index            => $option->{index},
  value            => $val,
  started_callback => sub {
   $self->signal_emit( 'started-process', sprintf $d->get('Setting option %s'),
    $option->{name} );
  },
  running_callback => sub {
   $self->signal_emit( 'changed-progress', -1, undef );
  },
  finished_callback => sub {
   my ($data) = @_;
   $self->update_options( Gscan2pdf::Scanner::Options->new_from_data($data) )
     if ($data);

   # We can carry on applying defaults now, if necessary.
   $self->signal_emit( 'finished-process', 'set_option' );

   # Unset the profile unless we are actively setting it
   $self->set( 'profile', undef ) unless ( $self->{setting_profile} );

   $self->signal_emit( 'changed-scan-option', $option->{name}, $val );

   #   $self->signal_emit( 'changed-current-scan-options',
   #    $self->get('current-scan-options') );
  }
 );
 return;
}

# If setting an option triggers a reload, we need to update the options

sub update_options {
 my ( $self, $options ) = @_;

 # walk the widget tree and update them from the hash
 $logger->debug( "Sane->get_option_descriptor returned: ", Dumper($options) );

 my ( $group, $vbox );
 my $num_dev_options = $options->num_options;
 for ( my $i = 1 ; $i < $num_dev_options ; ++$i ) {
  my $widget = $self->get('available-scan-options')->by_index($i)->{widget};

  # could be undefined for !($opt->{cap} & SANE_CAP_SOFT_DETECT)
  if ( defined $widget ) {
   my $opt   = $options->by_index($i);
   my $value = $opt->{val};
   $widget->signal_handler_block( $widget->{signal} );

   # HBox for option
   my $hbox = $widget->parent;
   $hbox->set_sensitive( ( not $opt->{cap} & SANE_CAP_INACTIVE )
      and $opt->{cap} & SANE_CAP_SOFT_SELECT );

   if ( $opt->{max_values} < 2 ) {

    # CheckButton
    if ( $opt->{type} == SANE_TYPE_BOOL )
    {    ## no critic (ProhibitCascadingIfElse)
     $widget->set_active($value)
       if ( $self->value_for_active_option( $value, $opt ) );
    }

    # SpinButton
    elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_RANGE ) {
     my ( $step, $page ) = $widget->get_increments;
     $step = 1;
     $step = $opt->{constraint}{quant} if ( $opt->{constraint}{quant} );
     $widget->set_range( $opt->{constraint}{min}, $opt->{constraint}{max} );
     $widget->set_increments( $step, $page );
     $widget->set_value($value)
       if ( $self->value_for_active_option( $value, $opt ) );
    }

    # ComboBox
    elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_STRING_LIST
     or $opt->{constraint_type} == SANE_CONSTRAINT_WORD_LIST )
    {
     $widget->get_model->clear;
     my $index = 0;
     for ( my $i = 0 ; $i < @{ $opt->{constraint} } ; ++$i ) {
      $widget->append_text( $d_sane->get( $opt->{constraint}[$i] ) );
      $index = $i if ( defined $value and $opt->{constraint}[$i] eq $value );
     }
     $widget->set_active($index) if ( defined $index );
    }

    # Entry
    elsif ( $opt->{constraint_type} == SANE_CONSTRAINT_NONE ) {
     $widget->set_text($value)
       if ( $self->value_for_active_option( $value, $opt ) );
    }
   }
   $widget->signal_handler_unblock( $widget->{signal} );
  }
 }
 return;
}

# Set options to profile referenced by hashref

sub set_current_scan_options {
 my ( $self, $profile ) = @_;

 return unless ( defined $profile );

 # Move them first to a dummy array, as otherwise it would be self-modifying
 my $defaults;

 # Config::General flattens arrays with 1 entry to scalars,
 # so we must check for this
 if ( ref($profile) ne 'ARRAY' ) {
  push @$defaults, $profile;
 }
 else {
  @$defaults = @$profile;
 }

 # Give the GUI a chance to catch up between settings,
 # in case they have to be reloaded.
 # Use the 'changed-scan-option' signal to trigger the next loop
 my $i = 0;

 my ( $changed_scan_signal, $changed_paper_signal );
 $changed_scan_signal = $self->signal_connect(
  'changed-scan-option' => sub {
   my ( $widget, $name, $val ) = @_;

   # for reasons I don't understand, without walking the reference tree,
   # parts of $default are undef
   Gscan2pdf::Dialog::Scan::my_dumper($defaults);
   my ( $ename, $eval ) = each( %{ $defaults->[$i] } );

   # don't check $eval against $val, just in case they are different
   if ( $ename eq $name ) {
    $i++;
    $i =
      $self->set_option_emit_signal( $i, $defaults, $changed_scan_signal,
     $changed_paper_signal );
   }
  }
 );
 $changed_paper_signal = $self->signal_connect(
  'changed-paper' => sub {
   my ( $widget, $val ) = @_;

   # for reasons I don't understand, without walking the reference tree,
   # parts of $default are undef
   Gscan2pdf::Dialog::Scan::my_dumper($defaults);
   my ( $ename, $eval ) = each( %{ $defaults->[$i] } );

   if ( $eval eq $val ) {
    $i++;
    $i =
      $self->set_option_emit_signal( $i, $defaults, $changed_scan_signal,
     $changed_paper_signal );
   }
  }
 );
 $i =
   $self->set_option_emit_signal( $i, $defaults, $changed_scan_signal,
  $changed_paper_signal );
 return;
}

# Set option widget

sub set_option_widget {
 my ( $self, $i, $profile ) = @_;

 while ( $i < @$profile ) {

  # for reasons I don't understand, without walking the reference tree,
  # parts of $profile are undef
  Gscan2pdf::Dialog::Scan::my_dumper( $profile->[$i] );
  my ( $name, $val ) = each( %{ $profile->[$i] } );

  if ( $name eq 'Paper size' ) {
   $self->set( 'paper', $val );
   return $self->set_option_widget( $i + 1, $profile );
  }

  # As scanimage and scanadf rename the geometry options,
  # we have to map them back to the original names
  given ($name) {
   when ('l') {
    $name = SANE_NAME_SCAN_TL_X;
    $profile->[$i] = { $name => $val };
   }
   when ('t') {
    $name = SANE_NAME_SCAN_TL_Y;
    $profile->[$i] = { $name => $val };
   }
   when ('x') {
    $name = SANE_NAME_SCAN_BR_X;
    my $l = $self->get_option_from_profile( 'l', $profile );
    $l = $self->get_option_from_profile( SANE_NAME_SCAN_TL_X, $profile )
      unless ( defined $l );
    $val += $l if ( defined $l );
    $profile->[$i] = { $name => $val };
   }
   when ('y') {
    $name = SANE_NAME_SCAN_BR_Y;
    my $t = $self->get_option_from_profile( 't', $profile );
    $t = $self->get_option_from_profile( SANE_NAME_SCAN_TL_Y, $profile )
      unless ( defined $t );
    $val += $t if ( defined $t );
    $profile->[$i] = { $name => $val };
   }
  }

  my $options = $self->get('available-scan-options');
  my $opt     = $options->by_name($name);
  my $widget  = $opt->{widget};

  if ( ref($val) eq 'ARRAY' ) {
   $self->set_option( $opt, $val );

   # when INFO_INEXACT is implemented, so that the value is reloaded,
   # check for it here, so that the reloaded value is not overwritten.
   $opt->{val} = $val;
  }
  else {
   given ($widget) {
    when ( $widget->isa('Gtk2::CheckButton') ) {
     if ( $widget->get_active != $val ) {
      $widget->set_active($val);
      return $i;
     }
    }
    when ( $widget->isa('Gtk2::SpinButton') ) {
     if ( $widget->get_value != $val ) {
      $widget->set_value($val);
      return $i;
     }
    }
    when ( $widget->isa('Gtk2::ComboBox') ) {
     if ( $opt->{constraint}[ $widget->get_active ] ne $val ) {
      my $index;
      for ( my $j = 0 ; $j < @{ $opt->{constraint} } ; ++$j ) {
       $index = $j if ( $opt->{constraint}[$j] eq $val );
      }
      $widget->set_active($index) if ( defined $index );
      return $i;
     }
    }
    when ( $widget->isa('Gtk2::Entry') ) {
     if ( $widget->get_text ne $val ) {
      $widget->set_text($val);
      return $i;
     }
    }
   }
  }
  ++$i;
 }

 return;
}

sub scan {
 my ($self) = @_;

 # Get selected number of pages
 my $npages = $self->get('num-pages');

 # Gscan2pdf::Frontend::Sane uses -1 for all
 # Gscan2pdf::Dialog::Sane uses 0 for all, as -1 puts 999 in spinbox
 $npages = -1 if ( $npages == 0 );

 my $start = $self->get('page-number-start');
 my $step  = $self->get('page-number-increment');
 $npages = $self->get('max-pages')
   if ( $npages > 0 and $step < 0 );

 if ( $start == 1 and $step < 0 ) {
  $self->signal_emit( 'process-error', 'scan',
   $d->get('Must scan facing pages first') );
  return TRUE;
 }

 my $i = 1;
 Gscan2pdf::Frontend::Sane->scan_pages(
  dir              => $self->get('dir'),
  format           => "out%d.pnm",
  npages           => $npages,
  start            => $start,
  step             => $step,
  started_callback => sub {
   $self->signal_emit( 'started-process',
    Gscan2pdf::Dialog::Scan::make_progress_string( $i, $npages ) );
  },
  running_callback => sub {
   my ($progress) = @_;
   $self->signal_emit( 'changed-progress', $progress, undef );
  },
  finished_callback => sub {
   $self->signal_emit( 'finished-process', 'scan_pages' );
  },
  new_page_callback => sub {
   my ($n) = @_;
   $self->signal_emit( 'new-scan', $n );
   $self->signal_emit( 'changed-progress', 0,
    Gscan2pdf::Dialog::Scan::make_progress_string( ++$i, $npages ) );
  },
  error_callback => sub {
   my ($msg) = @_;
   $self->signal_emit( 'process-error', 'scan_pages', $msg );
  }
 );
 return;
}

sub cancel_scan {
 Gscan2pdf::Frontend::Sane->cancel_scan;
 $logger->info("Cancelled scan");
 return;
}

1;

__END__
