package org.codehaus.groovy.antlr.treewalker;

import junit.framework.TestCase;

import org.codehaus.groovy.antlr.parser.GroovyTokenTypes;
import org.custommonkey.xmlunit.Diff;
import org.custommonkey.xmlunit.XMLUnit;

/**
 * Testcases for the antlr AST visitor that prints groovy source code nodes as Freemind compatible XML.
 */
public class MindMapPrinterTest extends TestCase implements GroovyTokenTypes{

    private static final String HEADER = "<map version='0.7.1'><node TEXT='AST'>";
    private static final String FOOTER = "</node></map>";

    protected void setUp() throws Exception {
        XMLUnit.setIgnoreWhitespace(true);
    }

    public void testAbstract() throws Exception {
        assertXmlEquals(
                HEADER +
                        "<node TEXT='MODIFIERS &lt;" + MODIFIERS +"&gt;' POSITION='right' COLOR=\"#000000\">" +
                        "  <node TEXT='public  &lt;" + LITERAL_public +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "</node>" +
                        "<node TEXT='abstract  &lt;" + ABSTRACT +"&gt;' POSITION='right' COLOR=\"#006699\"></node>" +
                        "<node TEXT='CLASS_DEF &lt;" + CLASS_DEF +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "<node TEXT='an identifier &lt;" + IDENT +"&gt; : Foo' POSITION='right' COLOR=\"#006699\"></node>" +
                        "<node TEXT='EXTENDS_CLAUSE &lt;" + EXTENDS_CLAUSE +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "<node TEXT='IMPLEMENTS_CLAUSE &lt;" + IMPLEMENTS_CLAUSE +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "<node TEXT='OBJBLOCK &lt;" + OBJBLOCK +"&gt;' POSITION='right' COLOR=\"#006699\"></node>" +
                        FOOTER,
                pretty("public abstract class Foo{}"));

    }

    public void testArrayDeclarator() throws Exception {
        assertXmlEquals(
                HEADER +
                        "<node TEXT='VARIABLE_DEF &lt;" + VARIABLE_DEF +"&gt; : primes' POSITION='right' COLOR=\"#000000\">" +
                        "  <node TEXT='TYPE &lt;" + TYPE +"&gt;' POSITION='right' COLOR=\"#17178B\">" +
                        "    <node TEXT='ARRAY_DECLARATOR &lt;" + ARRAY_DECLARATOR +"&gt; : [' POSITION='right' COLOR=\"#000000\">" +
                        "      <node TEXT='int  &lt;" + LITERAL_int +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "    </node>" +
                        "  </node>" +
                        "  <node TEXT='an identifier &lt;" + IDENT +"&gt; : primes' POSITION='right' COLOR=\"#006699\"></node>" +
                        "  <node TEXT='=  &lt;" + ASSIGN +"&gt;' POSITION='right' COLOR=\"#000000\">" +
                        "    <node TEXT='new  &lt;" + LITERAL_new +"&gt;' POSITION='right' COLOR=\"#17178B\">" +
                        "      <node TEXT='int  &lt;" + LITERAL_int +"&gt;' POSITION='right' COLOR=\"#17178B\"></node>" +
                        "      <node TEXT='ARRAY_DECLARATOR &lt;" + ARRAY_DECLARATOR +"&gt; : [' POSITION='right' COLOR=\"#000000\">" +
                        "        <node TEXT='a numeric literal &lt;" + NUM_INT +"&gt; : 5' POSITION='right' COLOR=\"#006699\"></node>" +
                        "      </node>" +
                        "    </node>" +
                        "  </node>" +
                        "</node>" +
                        FOOTER,
                pretty("int[] primes = new int[5]"));
    }

    public void testRegexMatch() throws Exception {
        assertXmlEquals(
                HEADER +
                        "<node TEXT='if  &lt;" + LITERAL_if +"&gt;' POSITION='right' COLOR=\"#17178B\">" +
                        "  <node TEXT='an identifier &lt;" + IDENT +"&gt; : foo' POSITION='right' COLOR=\"#006699\"></node>" +
                        "  <node TEXT='==~  &lt;" + REGEX_MATCH +"&gt;' POSITION='right' COLOR=\"#000000\"></node>" +
                        "  <node TEXT='a string literal &lt;" + STRING_LITERAL +"&gt; : bar' POSITION='right' COLOR=\"#008000\"></node>" +
                        "</node>" +
                        "<node TEXT='SLIST &lt;" + SLIST +"&gt; : {' POSITION='right' COLOR=\"#006699\"></node>" +
                        FOOTER,
                pretty("if (foo==~\"bar\"){}"));
    }

    private void assertXmlEquals(String expected, String actual) throws Exception {
        Diff diff = new Diff(expected, actual);
        assertTrue(diff.toString(), diff.similar());
    }

    private String pretty(String input) throws Exception {
        TraversalTestHelper traverser = new TraversalTestHelper();
        return traverser.traverse(input, MindMapPrinter.class);
    }

}
