/*
 * squish theme module
 *
 * Basically, this follows default module method(theme-default.c).
 * The main difference is that this shows squish animation in button push
 * event.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <math.h>
#include <gnome.h>

#include "groach.h"


/* Data structure definitions */	
/* These are derived from theme-default.c */
typedef struct _DefaultMoveEnv DefaultMoveEnv;
typedef struct _AngleInfo AngleInfo;

struct _DefaultMoveEnv {
	gboolean turn_left;
	guint steps;/* number of steps until recomputation */
	AngleInfo *angle_infos;/* array of AngleInfo for all directions */
};

struct _AngleInfo {
	gdouble sine;
	gdouble cosine;
};

/* Private variables */
#define NUM_BLOOD_COLORS	3
static GdkGC *s_gc[NUM_BLOOD_COLORS];
static const char *color_name[NUM_BLOOD_COLORS] = { 
	"purple", "green4", "HotPink"
};

/* Private function declarations */
static void move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data);
static void turn_gmove(GroMove *gmove);
static void show_squish(GdkWindow *window, gdouble x, gdouble y);

/**
 * theme_init:
 * Called once for one module.
 **/
gint
theme_init(const GroController *controller)
{
	int i;

	for (i = 0; i < NUM_BLOOD_COLORS; i++) {
		GdkColor blood;

		s_gc[i] = gdk_gc_new(controller->gro_win->window);
		/* allocate blood color, and set it as foreground color */
		gdk_color_parse(color_name[i], &blood);
		gdk_colormap_alloc_color(gdk_colormap_get_system(), &blood, FALSE, TRUE);
		gdk_gc_set_foreground(s_gc[i], &blood);
		gdk_gc_set_subwindow(s_gc[i], GDK_INCLUDE_INFERIORS);
	}

	return 1;
}

/**
 * theme_finalize:
 * Called once for one module.
 **/
gint
theme_finalize(const GroController *controller)
{
	int i;
	
	for (i = 0; i < NUM_BLOOD_COLORS; i++) {
		gdk_gc_unref(s_gc[i]);
	}
	return 1;
}

/**
 * move_init:
 * Called once for each GroMove.
 **/
gint
move_init(const GroController *controller, GroMove *gmove)
{
	GroVector init_pos;
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;
	int nd;

	move_env = g_new(DefaultMoveEnv, 1);
	gmove->move_env = move_env;/* Set it for later use */
	move_env->turn_left = RAND_INT(100) >= 50;
	move_env->steps = RAND_INT(200);
	angle_infos = g_new(AngleInfo, gmove->num_direct);
	move_env->angle_infos = angle_infos;
	for (nd = 0; nd < gmove->num_direct; nd++) {
		gdouble angle;

		angle = 2 * M_PI * nd / gmove->num_direct;
		angle_infos[nd].sine = sin(angle);
		angle_infos[nd].cosine = cos(angle);
	}

	/* Initialize */
	gro_move_change_gstat(gmove, GRO_STAT_WAKE);
	gro_move_change_direct(gmove, RAND_INT(gmove->num_direct));

	/* Initial position is decided with randum number */
	init_pos.ix = RAND_INT(gcontroller_window_width(controller) - gmove_width(gmove));
	init_pos.iy = RAND_INT(gcontroller_window_height(controller) - gmove_height(gmove));
	gro_move_move(gmove, controller, &init_pos);
	
	gtk_signal_connect(GTK_OBJECT(gmove), "push",
					   GTK_SIGNAL_FUNC(move_push_cb),
					   NULL);

	return 1;
}

/**
 * move_compute:
 * Called for each GroMove by a timer.
 **/
GroMoveRet
move_compute(const GroController *controller, GroMove *gmove, const GdkRegion *vis_region, GroVector *ret_vec)
{
	DefaultMoveEnv *move_env = gmove->move_env;
	const AngleInfo *angle_infos = move_env->angle_infos;
	GdkRectangle tmp_gmove_rect = gmove->cur_rect;/* local copy */
	gint ix;
	gint iy;

	g_return_val_if_fail(gmove->cur_gstat != GRO_STAT_DEAD, GRO_RET_DONT_MOVE);
	
	/* Every theme is recommended to call this at first */
	DONTCARE_hidden_gmove(gmove, vis_region);
	
	/* Compute a new position temporarily to check whether it is
	   within the window and to check collisions with other gmoves. */
	ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
	iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	tmp_gmove_rect.x += ix;
	tmp_gmove_rect.y += iy;

	if (is_rect_in_gcontroller_window(controller, &tmp_gmove_rect) == TRUE) {
		const GList *other_gmoves = controller->gmove_list;
		const GList *node;

		if (move_env->steps-- <= 0) {
			turn_gmove(gmove);
			move_env->steps = RAND_INT(200);
		}

		/* Detect collision and avoid it.
		   This is intended to follow xroach's method. */
		for (node = other_gmoves; node; node = node->next) {
			GroMove *other = node->data;
			if (gmove == other)
				continue;
			if (is_rect_intersect(&tmp_gmove_rect, &other->cur_rect)) {
				turn_gmove(gmove);
				break;
			}
		}

		/* Recompute the move vector after turn */
		ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
		iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	} else {
		/* If a new position is out of the window, turn it */
		turn_gmove(gmove);
		ix = iy = 0;
    }
	
	ret_vec->ix = ix;
	ret_vec->iy = iy;

	return GRO_RET_MOVE;
}

/**
 * move_finalize:
 * Called once for each GroMove.
 **/
gint
move_finalize(const GroController *controller, GroMove *gmove)
{
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;

	gtk_signal_disconnect_by_func(GTK_OBJECT(gmove),
								  GTK_SIGNAL_FUNC(move_push_cb),
								  NULL);
	move_env = gmove->move_env;
	angle_infos = move_env->angle_infos;
	g_free(angle_infos);
	g_free(move_env);

	return 1;
}



/* ---The followings are private functions--- */
/* Show squish animation, and make it dead.
   This is different from default module. */
static void
move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data)
{
	gdk_beep();
	show_squish(gcontroller_gdkwindow(controller),
				gmove_center_x(gmove), gmove_center_y(gmove));
	gro_move_change_gstat(gmove, GRO_STAT_DEAD);
}

/* This is derived from theme-default.c */
static void
turn_gmove(GroMove *gmove)
{
	const DefaultMoveEnv *move_env = gmove->move_env;
	gint cur_direct = gmove->cur_direct;/* not guint */
	int turn_base;

	turn_base = gmove->num_direct / 8;	/* heuristic */
    if (move_env->turn_left == TRUE) {
		cur_direct += RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct >= gmove->num_direct)
			cur_direct -= gmove->num_direct;
    } else {
		cur_direct -= RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct < 0)
			cur_direct += gmove->num_direct;
    }
	gro_move_change_direct(gmove, cur_direct);
}

/**
 * show_squish:
 * Many magic numbers! In fact, they are heuristic and no meaning.
 * Move random portions from inside to outside.
 */
static void
show_squish(GdkWindow *window, gdouble x, gdouble y)
{
	int i;

	for (i = 0; i < 10; i++) {
		int j;
		for (j = 0; j < 30; j++) {
			int range = (i * i / 10) + 10;/* 10 - 20 */
			int src_x = x + RAND_INT(range*2) - range;
			int src_y = y + RAND_INT(range*2) - range;
			int dest_x = x + ((src_x - x) * (i < 5 ? 4 : 8));
			int dest_y = y + ((src_y - y) * (i < 5 ? 4 : 8));
			int w = RAND_INT(i < 5 ? 8 : 4);
			int h = RAND_INT(i < 5 ? 8 : 4);
			int cn = RAND_INT(NUM_BLOOD_COLORS);/* color number */
			
			gdk_window_copy_area(window, s_gc[cn], dest_x, dest_y,
								 window, src_x, src_y, w, h);
			if (j == 0)
				gdk_draw_rectangle(window, s_gc[cn], TRUE, src_x, src_y, w, h);
		}
	}
}
