/*
 * Theme search paths module
 * Refer to theme-search.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <dirent.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
/* excerpted from autoconf (GNU)info */
#if defined(HAVE_DIRENT_H)
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if defined(HAVE_SYS_NDIR_H)
#  include <sys/ndir.h>
# endif
# if defined(HAVE_SYS_DIR_H)
#  include <sys/dir.h>
# endif
# if defined(HAVE_NDIR_H)
#  include <ndir.h>
# endif
#endif

#include <gnome.h>

#include "groach.h"/* for THEME_FILE_NAME */

#include "types.h"
#include "theme-search.h"


#define SYS_THEME_SEARCH  "groach/theme"
#define USER_THEME_SEARCH  ".groach/theme"	/* $HOME/.groach/theme */

/* Data structure definitions */
struct _ThemeSearchPrivate {
	GSList *spath_list;/* search path list */
};


/**
 * theme_search_new:
 * paths is path list, concatenated with ':'(G_SEARCHPATH_SEPARATOR).
 **/
ThemeSearch*
theme_search_new(char *paths)
{
	ThemeSearch *theme_search;
	ThemeSearchPrivate *privat;
	char *sys_theme_search;
	char *user_theme_search;
	char *def_theme_search;

	theme_search = g_new(ThemeSearch, 1);
	
	privat = g_new(ThemeSearchPrivate, 1);
	theme_search->privat = privat;
	privat->spath_list = NULL;

	/* Do the best effort */
	/* I can't pass static strings, because they might be passed to strtok(). */
	def_theme_search = g_strdup("./:./theme/:../theme/");
	theme_search_prepend_paths(theme_search, def_theme_search);
	g_free(def_theme_search);
	
	sys_theme_search = gnome_datadir_file(SYS_THEME_SEARCH);
	theme_search_prepend_paths(theme_search, sys_theme_search);
	g_free(sys_theme_search);

	user_theme_search = gnome_util_prepend_user_home(USER_THEME_SEARCH);
	theme_search_prepend_paths(theme_search, user_theme_search);
	g_free(user_theme_search);

	if (paths)
		theme_search_prepend_paths(theme_search, paths);
	
	return theme_search;
}

/**
 * theme_search_delete:
 **/
void
theme_search_delete(ThemeSearch *theme_search)
{
	ThemeSearchPrivate *privat;
	GSList *list;

	g_return_if_fail(theme_search != NULL);
	privat = theme_search->privat;

	for (list = privat->spath_list; list; list = list->next) {
		g_free(list->data);
	}
	g_slist_free(privat->spath_list);
	
	g_free(privat);
	g_free(theme_search);
}

/**
 * theme_search_prepend_paths:
 * paths is path list, concatenated with ':'(G_SEARCHPATH_SEPARATOR).
 **/
void
theme_search_prepend_paths(ThemeSearch *theme_search, char *paths)
{
	ThemeSearchPrivate *privat;
	char *ptr;
#if defined(HAVE_STRTOK_R)
	char *last;
#endif

	g_return_if_fail(theme_search != NULL);
	if (paths == NULL || paths[0] == '\0')
		return;
	privat = theme_search->privat;

#if defined(HAVE_STRSEP)
	for (ptr = strsep(&paths, G_SEARCHPATH_SEPARATOR_S); ptr; ptr = strsep(&paths, G_SEARCHPATH_SEPARATOR_S)) {
#elif defined(HAVE_STRTOK_R)
	for (ptr = strtok_r(paths, G_SEARCHPATH_SEPARATOR_S, &last); ptr; ptr = strtok_r(last, G_SEARCHPATH_SEPARATOR_S, &last)) {
#else
	for (ptr = strtok(paths, G_SEARCHPATH_SEPARATOR_S); ptr; ptr = strtok(NULL, G_SEARCHPATH_SEPARATOR_S)) {
#endif
		privat->spath_list = g_slist_prepend(privat->spath_list, g_strdup(ptr));
	}
}

/**
 * theme_search_find_path:
 * For example, pass in "default", and return
 * "/usr/share/groach/theme/default".
 * Returned buffer should be free'd by the caller.
 **/
char*
theme_search_find_path(ThemeSearch *theme_search, const char *theme_name)
{
	ThemeSearchPrivate *privat;
	GSList *list;

	g_return_val_if_fail(theme_search != NULL, NULL);
	g_return_val_if_fail(theme_name != NULL && theme_name[0] != '\0', NULL);
	privat = theme_search->privat;
	
	/* For consistency, even if theme_name is already absolute path,
	   allocate memory and return it */
	if (theme_name[0] == G_DIR_SEPARATOR)
		return g_strdup(theme_name);
	
	for (list = privat->spath_list; list; list = list->next) {
		char *path;
		path = g_concat_dir_and_file(list->data, theme_name);
#ifdef DEBUG
		g_print("theme searching... %s\n", path);
#endif
		if (g_file_test(path, G_FILE_TEST_ISDIR))
			return path;
		else
			g_free(path);
	}
	return NULL;
}

/**
 * theme_search_enum_themes:
 * Enumerate theme names, by searching in search paths.
 * Note, returned GList and its content should be free'd by the caller.
 **/
GList*
theme_search_enum_themes(ThemeSearch *theme_search)
{
	ThemeSearchPrivate *privat;
	GList *theme_enum = NULL;
	GSList *list;
	DIR *dir;
	struct dirent *dent;

	g_return_val_if_fail(theme_search != NULL, NULL);

	privat = theme_search->privat;

	for (list = privat->spath_list; list; list = list->next) {
		char *search_path = list->data;/* search path */
		char *dir_path;
		
		if ((dir = opendir(search_path)) == NULL)
			continue;/* ignore error */
		while ((dent = readdir(dir)) != NULL) {
			if (dent->d_ino == 0)
				continue;
			dir_path = g_concat_dir_and_file(search_path, dent->d_name);
			if (g_file_test(dir_path, G_FILE_TEST_ISDIR)) {
				char *config_path;/* theme config file path */
				config_path = g_concat_dir_and_file(dir_path, THEME_FILE_NAME);
				if (g_file_test(config_path, G_FILE_TEST_ISFILE)) {
					theme_enum = g_list_prepend(theme_enum, g_strdup(dent->d_name));
				}
				g_free(config_path);
			}
			g_free(dir_path);
		}
		closedir(dir);
	}

	return theme_enum;
}
