#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# pylint: disable-msg=C0302
"""
grml2usb
~~~~~~~~

This script installs a Grml system (either a running system or ISO[s]) to a USB device

:copyright: (c) 2009-2019 by Michael Prokop <mika@grml.org>
:license: GPL v2 or any later version
:bugreports: http://grml.org/bugs/

"""


import fileinput
import glob
import logging
import os
import os.path
import re
import shutil
import subprocess
import sys
import tempfile
import uuid
from inspect import isclass, isroutine
from optparse import OptionParser

# The line following this line is patched by debian/rules and tarball.sh.
PROG_VERSION = "***UNKNOWN***"

# when running from inside git, try to report version information via git-describe
try:
    git_dir = os.path.abspath(os.path.dirname(sys.argv[0]))
    with open(os.devnull, "w") as devnull:
        PROG_VERSION = (
            subprocess.check_output(
                ["git", "-C", git_dir, "describe", "--always", "--dirty"],
                stderr=devnull,
            )
            .strip()
            .decode("utf-8", errors="replace")
            + " (git)"
        )
except Exception:
    pass

# global variables
MOUNTED = set()  # register mountpoints
TMPFILES = set()  # register tmpfiles
GRML_FLAVOURS = set()  # which flavours are being installed?
GRML_DEFAULT = None
UUID = None
SYSLINUX_LIBS = [
    "/usr/lib/syslinux/modules/bios/",  # Debian
    "/usr/lib/syslinux/bios/",  # Arch Linux
]
GRUB_INSTALL = None

RE_PARTITION = re.compile(r"([a-z/]*?)(\d+)$")
RE_P_PARTITION = re.compile(r"(.*?\d+)p(\d+)$")
RE_LOOP_DEVICE = re.compile(r"/dev/loop\d+$")


def syslinux_warning(option, opt, opt_parser):
    """A helper function for printing a warning about deprecated option"""
    # pylint: disable-msg=W0613
    sys.stderr.write(
        "Note: the --syslinux option is deprecated as syslinux "
        "is grml2usb's default. Continuing anyway.\n"
    )
    setattr(opt_parser.values, option.dest, True)


# if grub option is set, unset syslinux option
def grub_option(option, opt, opt_parser):
    """A helper function adjusting other option values"""
    # pylint: disable-msg=W0613
    setattr(opt_parser.values, option.dest, True)
    setattr(opt_parser.values, "syslinux", False)


# cmdline parsing
USAGE = "Usage: %prog [options] <[ISO[s] | /run/live/medium]> </dev/sdX#>\n\
\n\
%prog installs Grml ISO[s] to an USB device to be able to boot from it.\n\
Make sure you have at least one Grml ISO or a running Grml system (/run/live/medium),\n\
grub or syslinux and root access.\n\
\n\
Run %prog --help for usage hints, further information via: man grml2usb"

# pylint: disable-msg=C0103
# pylint: disable-msg=W0603
parser = OptionParser(usage=USAGE)
parser.add_option(
    "--bootoptions",
    dest="bootoptions",
    action="append",
    type="string",
    help="use specified bootoptions as default",
)
parser.add_option(
    "--bootloader-only",
    dest="bootloaderonly",
    action="store_true",
    help="do not copy files but just install a bootloader",
)
parser.add_option(
    "--copy-only",
    dest="copyonly",
    action="store_true",
    help="copy files only but do not install bootloader",
)
parser.add_option(
    "--dry-run", dest="dryrun", action="store_true", help="avoid executing commands"
)
parser.add_option(
    "--fat16",
    dest="fat16",
    action="store_true",
    help="format specified partition with FAT16",
)
parser.add_option(
    "--force",
    dest="force",
    action="store_true",
    help="force any actions requiring manual interaction",
)
parser.add_option(
    "--grub",
    dest="grub",
    action="callback",
    callback=grub_option,
    help="install grub bootloader instead of (default) syslinux",
)
parser.add_option(
    "--grub-mbr",
    dest="grubmbr",
    action="store_true",
    help="install grub into MBR instead of (default) PBR",
)
parser.add_option(
    "--mbr-menu",
    dest="mbrmenu",
    action="store_true",
    help="enable interactive boot menu in MBR",
)
parser.add_option(
    "--quiet",
    dest="quiet",
    action="store_true",
    help="do not output anything but just errors on console",
)
parser.add_option(
    "--remove-bootoption",
    dest="removeoption",
    action="append",
    help="regex for removing existing bootoptions",
)
parser.add_option(
    "--rw-blockdev",
    dest="rwblockdev",
    action="store_true",
    help="enforce read-write mode on involved block devices",
)
parser.add_option(
    "--skip-addons",
    dest="skipaddons",
    action="store_true",
    help="do not install /boot/addons/ files",
)
parser.add_option(
    "--skip-bootflag",
    dest="skipbootflag",
    action="store_true",
    help="do not try to check whether the destination has the boot flag set",
)
parser.add_option(
    "--skip-grub-config",
    dest="skipgrubconfig",
    action="store_true",
    help="skip generation of grub configuration files",
)
parser.add_option(
    "--skip-mbr",
    dest="skipmbr",
    action="store_true",
    help="do not install a master boot record (MBR) on the device",
)
parser.add_option(
    "--skip-syslinux-config",
    dest="skipsyslinuxconfig",
    action="store_true",
    help="skip generation of syslinux configuration files",
)
parser.add_option(
    "--skip-usb-check",
    dest="skipusbcheck",
    action="store_true",
    help="skip check to verify whether given device is removable",
)
parser.add_option(
    "--syslinux",
    dest="syslinux",
    action="callback",
    default=True,
    callback=syslinux_warning,
    help="install syslinux bootloader (deprecated as it's the default)",
)
parser.add_option(
    "--syslinux-mbr",
    dest="syslinuxmbr",
    action="store_true",
    help="install syslinux master boot record (MBR) instead of default",
)
parser.add_option(
    "--syslinux-libs",
    dest="syslinuxlibs",
    action="append",
    default=[],
    help="syslinux modules directory path",
)
parser.add_option(
    "--tmpdir",
    dest="tmpdir",
    default="/tmp",
    help="directory to be used for temporary files",
)
parser.add_option(
    "--verbose", dest="verbose", action="store_true", help="enable verbose mode"
)
parser.add_option(
    "-v",
    "--version",
    dest="version",
    action="store_true",
    help="display version and exit",
)
(options, args) = parser.parse_args()


GRML2USB_BASE = "/usr/share/grml2usb"
if not os.path.isdir(GRML2USB_BASE):
    GRML2USB_BASE = os.path.dirname(os.path.realpath(__file__))


class HodorException(Exception):
    """Throw exception if the exact error is not known and not fatal.

    @Exception: message"""


class CriticalException(Exception):
    """Throw critical exception if the exact error is not known but fatal.

    @Exception: message"""


class VerifyException(Exception):
    """Throw critical exception if there is an fatal error when verifying something.

    @Exception: message"""


def cleanup():
    """Cleanup function to make sure there aren't any mounted devices left behind."""

    def del_failed(_fn, path, _exc):
        logging.warning("Deletion of %s failed in temporary folder", path)

    logging.info("Cleaning up before exiting...")
    proc = subprocess.Popen(["sync"])
    proc.wait()

    for device in MOUNTED.copy():
        try:
            unmount(device, "")
            logging.debug("Unmounted %s" % device)
        except Exception:
            logging.debug("RuntimeError while umount %s, ignoring" % device)

    for tmppath in TMPFILES.copy():
        try:
            if os.path.isdir(tmppath) and not os.path.islink(tmppath):
                # symbolic links to directories are ignored
                # without the check it will throw an OSError
                shutil.rmtree(tmppath, onerror=del_failed)
                logging.debug("temporary directory %s deleted" % tmppath)
                unregister_tmpfile(tmppath)
            elif os.path.isfile:
                os.unlink(tmppath)
                logging.debug("temporary file %s deleted" % tmppath)
                unregister_tmpfile(tmppath)
        except Exception:
            msg = "RuntimeError while removing temporary %s, ignoring"
            logging.debug(msg % tmppath)


def register_tmpfile(path):
    """
    register tmpfile
    """

    TMPFILES.add(path)


def unregister_tmpfile(path):
    """
    remove registered tmpfile
    """

    try:
        TMPFILES.remove(path)
    except KeyError:
        pass


def register_mountpoint(target):
    """register specified target in a set() for handling clean exiting

    @target: destination target of mountpoint
    """

    MOUNTED.add(target)


def unregister_mountpoint(target):
    """unregister specified target in a set() for handling clean exiting

    @target: destination target of mountpoint
    """

    if target in MOUNTED:
        MOUNTED.remove(target)


def get_function_name(obj):
    """Helper function for use in execute() to retrieve name of a function

    @obj: the function object
    """
    if not (isroutine(obj) or isclass(obj)):
        obj = type(obj)
    return obj.__module__ + "." + obj.__name__


def set_rw(device):
    if not options.rwblockdev:
        return

    logging.debug("executing: blockdev --setrw %s", device)
    proc = subprocess.Popen(["blockdev", "--setrw", device])
    proc.wait()
    if proc.returncode != 0:
        raise Exception("error executing blockdev on %s" % device)


def execute(f, *exec_arguments):
    """Wrapper for executing a command. Either really executes
    the command (default) or when using --dry-run commandline option
    just displays what would be executed."""
    # usage: execute(subprocess.Popen, (["ls", "-la"]))
    if options.dryrun:
        # pylint: disable-msg=W0141
        logging.debug(
            "dry-run only: %s(%s)",
            get_function_name(f),
            ", ".join(map(repr, exec_arguments)),
        )
    else:
        # pylint: disable-msg=W0142
        return f(*exec_arguments)


def is_exe(fpath):
    """Check whether a given file can be executed

    @fpath: full path to file
    @return:"""
    return os.path.exists(fpath) and os.access(fpath, os.X_OK)


def which(program):
    """Check whether a given program is available in PATH

    @program: name of executable"""
    fpath = os.path.split(program)[0]
    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file

    return None


def get_defaults_file(iso_mount, flavour, name):
    """get the default file for syslinux"""
    bootloader_dirs = ["/boot/isolinux/", "/boot/syslinux/"]
    for directory in bootloader_dirs:
        for name in name, "%s_%s" % (get_flavour_filename(flavour), name):
            if os.path.isfile(iso_mount + directory + name):
                return (directory, name)
    return ("", "")


def search_file(
    filename,
    search_path="/bin" + os.pathsep + "/usr/bin",
    lst_return=False,
    required=False,
):
    """Given a search path, find file

    @filename: name of file to search for
    @search_path: path where searching for the specified filename
    @lst_return: return list of matching files instead one file"""
    paths = search_path.split(os.pathsep)
    current_dir = ""  # make pylint happy :)
    retval = []

    def match_file(cwd):
        """Helper function ffor testing if specified file exists in cwd

        @cwd: current working directory
        """
        return os.path.exists(os.path.join(cwd, filename))

    for path in paths:
        current_dir = path
        if match_file(current_dir):
            retval.append(os.path.abspath(os.path.join(current_dir, filename)))
            if not lst_return:
                break
        for current_dir, _directories, _files in os.walk(path):
            if match_file(current_dir):
                retval.append(os.path.abspath(os.path.join(current_dir, filename)))
                if not lst_return:
                    break

    if required and not retval:
        raise CriticalException(
            "Required file %s not found in %s" % (filename, search_path)
        )

    if lst_return:
        return retval
    elif retval:
        return retval[0]
    else:
        return None


def check_uid_root():
    """Check for root permissions"""
    if not os.geteuid() == 0:
        raise CriticalException("please run this script with uid 0 (root).")


# for usage inside check_boot_flag
def get_partition_for_path(path):
    import parted

    boot_dev, x = get_device_from_partition(path)

    d = parted.getDevice(boot_dev)
    disk = parted.Disk(d)
    return disk.getPartitionByPath(path)


def check_boot_flag(device):
    if os.path.isdir(device):
        logging.debug(
            "Device %s is a directory, skipping check for boot flag." % device
        )
        return

    boot_dev, x = get_device_from_partition(device)

    logging.info("Checking for boot flag")
    try:
        import parted

        part = get_partition_for_path(device)
        if part is None:
            raise HodorException("parted could not find partition")
        if part.getFlag(parted.PARTITION_BOOT):
            logging.debug("boot flag is enabled on %s" % device)
            return
        else:
            logging.debug("boot flag is NOT enabled on %s" % device)
            raise VerifyException(
                "Device %s does not have the boot flag set. "
                "Please enable it to be able to boot." % device
            )
    except ImportError:
        raise VerifyException(
            "Could not import parted to verify boot flag on %s, please make sure python3-parted is installed."
            % device
        )


def mkfs_fat16(device):
    """Format specified device with VFAT/FAT16 filesystem.

    @device: partition that should be formated"""

    if options.dryrun:
        logging.info("Would execute mkfs.vfat -F 16 %s now.", device)
        return 0

    logging.info("Formating partition with fat16 filesystem")
    logging.debug("mkfs.vfat -F 16 %s", device)
    proc = subprocess.Popen(["mkfs.vfat", "-n", "GRML", "-F", "16", device])
    proc.wait()
    if proc.returncode != 0:
        raise CriticalException("error executing mkfs.vfat")


def generate_main_syslinux_config():
    """Generate main configuration for use in syslinux.cfg"""

    return """\
label -
menu label Default boot modes:
menu disable
include defaults.cfg

menu end
menu separator

# flavours:
label -
menu label Additional boot entries for:
menu disable
include additional.cfg

menu separator
include options.cfg
include addons.cfg

label help
  include promptname.cfg
  config prompt.cfg
  text help
                                        Jump to old style isolinux prompt
                                        featuring further information
                                        regarding available boot options.
  endtext


include hiddens.cfg
"""


def generate_flavour_specific_syslinux_config(grml_flavour):
    """Generate flavour specific configuration for use in syslinux.cfg

    @grml_flavour: name of grml flavour the configuration should be generated for"""

    return """\
menu begin grml %(grml_flavour)s
    menu title %(display_name)s
    label mainmenu
    menu label ^Back to main menu...
    menu exit
    menu separator
    # include config for boot parameters from disk
    include %(grml_flavour)s_grml.cfg
    menu hide
menu end
""" % {
        "grml_flavour": grml_flavour,
        "display_name": get_flavour_filename(grml_flavour),
    }


def install_grub(device):
    """Install grub on specified device.

    @mntpoint: mountpoint of device where grub should install its files to
    @device: partition where grub should be installed to"""

    if options.dryrun:
        logging.info(
            "Would execute grub-install [--root-directory=mount_point] %s now.", device
        )
    else:
        device_mountpoint = tempfile.mkdtemp(prefix="grml2usb")
        register_tmpfile(device_mountpoint)
        try:
            # If using --grub-mbr then make sure we install grub in MBR instead of PBR
            if options.grubmbr:
                logging.debug("Using option --grub-mbr ...")
                grub_device, x = get_device_from_partition(device)
            else:
                grub_device = device

            set_rw(device)
            mount(device, device_mountpoint, "")

            logging.info("Installing grub as bootloader")
            for opt in ["--", "--force"]:
                set_rw(device)
                set_rw(grub_device)
                logging.debug(
                    "%s --recheck --no-floppy --target=i386-pc --root-directory=%s %s %s",
                    GRUB_INSTALL,
                    device_mountpoint,
                    opt,
                    grub_device,
                )
                proc = subprocess.Popen(
                    [
                        GRUB_INSTALL,
                        "--recheck",
                        "--no-floppy",
                        "--target=i386-pc",
                        "--root-directory=%s" % device_mountpoint,
                        opt,
                        grub_device,
                    ],
                    stdout=open(os.devnull, "r+"),
                )
                proc.wait()
                if proc.returncode == 0:
                    break

            if proc.returncode != 0:
                # raise Exception("error executing grub-install")
                logging.critical(
                    "Fatal: error executing grub-install "
                    "(please check the grml2usb FAQ or drop the --grub option)"
                )
                logging.critical(
                    "Note:  if using grub2 consider using "
                    "the --grub-mbr option as grub considers PBR problematic."
                )
                cleanup()
                sys.exit(1)
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            cleanup()
            sys.exit(1)
        finally:
            unmount(device_mountpoint, "")
            os.rmdir(device_mountpoint)
            unregister_tmpfile(device_mountpoint)


def install_syslinux(device):
    """Install syslinux on specified device.

    @device: partition where syslinux should be installed to"""

    if options.dryrun:
        logging.info("Would install syslinux as bootloader on %s", device)
        return 0

    set_rw(device)

    # syslinux -d boot/isolinux /dev/sdb1
    logging.info("Installing syslinux as bootloader")
    logging.debug("syslinux -d boot/syslinux %s", device)
    proc = subprocess.Popen(["syslinux", "-d", "boot/syslinux", device])
    proc.wait()
    if proc.returncode != 0:
        raise CriticalException(
            "Error executing syslinux (either try --fat16 or use grub?)"
        )


def install_bootloader(device):
    """Install bootloader on specified device.

    @device: partition where bootloader should be installed to"""

    # by default we use grub, so install syslinux only on request
    if options.grub:
        try:
            install_grub(device)
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            cleanup()
            sys.exit(1)
    else:
        try:
            install_syslinux(device)
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            cleanup()
            sys.exit(1)


def install_mbr(mbrtemplate, device, partition, ismirbsdmbr=True):
    """install 'mbr' master boot record (MBR) on a device

    Retrieve the partition table from "device", install an MBR from the
    "mbrtemplate" file, set the "partition" (0..3) active, and install the
    result back to "device".

    @mbrtemplate: default MBR file (must be a valid MBR file of at least 440
    (or 439 if ismirbsdmbr) bytes)

    @device: name of a file assumed to be a hard disc (or USB stick) image, or
    something like "/dev/sdb"

    @partition: must be a number between 0 and 3, inclusive

    @ismirbsdmbr: if true then ignore the active flag, set the mirbsdmbr
    specific flag to 0/1/2/3 and set the MBR's default value accordingly. If
    false then leave the mirbsdmbr specific flag set to FFh, set all
    active flags to 0 and set the active flag of the partition to 80h.  Note:
    behaviour of mirbsdmbr: if flag = 0/1/2/3 then use it, otherwise search for
    the active flag."""

    logging.info("Installing default MBR")

    if not os.path.isfile(mbrtemplate):
        logging.error(
            "Error installing MBR (either try --syslinux-mbr or "
            'install missing file "%s"?)',
            mbrtemplate,
        )
        raise CriticalException("%s can not be read." % mbrtemplate)

    if partition is not None and ((partition < 0) or (partition > 3)):
        logging.warning("Cannot activate partition %d", partition)
        partition = None

    if ismirbsdmbr:
        nmbrbytes = 439
    else:
        nmbrbytes = 440

    tmpf = tempfile.NamedTemporaryFile()

    logging.debug("executing: dd if='%s' of='%s' bs=512 count=1", device, tmpf.name)
    proc = subprocess.Popen(
        ["dd", "if=%s" % device, "of=%s" % tmpf.name, "bs=512", "count=1"],
        stderr=open(os.devnull, "r+"),
    )
    proc.wait()
    if proc.returncode != 0:
        raise Exception("error executing dd (first run)")

    logging.debug(
        "executing: dd if=%s of=%s bs=%s count=1 conv=notrunc",
        mbrtemplate,
        tmpf.name,
        nmbrbytes,
    )
    proc = subprocess.Popen(
        [
            "dd",
            "if=%s" % mbrtemplate,
            "of=%s" % tmpf.name,
            "bs=%s" % nmbrbytes,
            "count=1",
            "conv=notrunc",
        ],
        stderr=open(os.devnull, "r+"),
    )
    proc.wait()
    if proc.returncode != 0:
        raise Exception("error executing dd (second run)")

    mbrcode = tmpf.file.read(512)
    if len(mbrcode) < 512:
        raise EOFError("MBR size (%d) < 512" % len(mbrcode))

    if partition is not None:
        if ismirbsdmbr:
            mbrcode = (
                mbrcode[0:439]
                + chr(partition).encode("latin-1")
                + mbrcode[440:510]
                + b"\x55\xAA"
            )
        else:
            actives = [b"\x00", b"\x00", b"\x00", b"\x00"]
            actives[partition] = b"\x80"
            mbrcode = (
                mbrcode[0:446]
                + actives[0]
                + mbrcode[447:462]
                + actives[1]
                + mbrcode[463:478]
                + actives[2]
                + mbrcode[479:494]
                + actives[3]
                + mbrcode[495:510]
                + b"\x55\xAA"
            )

    tmpf.file.seek(0)
    tmpf.file.truncate()
    tmpf.file.write(mbrcode)
    tmpf.file.close()

    set_rw(device)

    logging.debug(
        "executing: dd if='%s' of='%s' bs=512 count=1 conv=notrunc,fsync",
        tmpf.name,
        device,
    )
    proc = subprocess.Popen(
        [
            "dd",
            "if=%s" % tmpf.name,
            "of=%s" % device,
            "bs=512",
            "count=1",
            "conv=notrunc,fsync",
        ],
        stderr=open(os.devnull, "r+"),
    )
    proc.wait()
    if proc.returncode != 0:
        raise Exception("error executing dd (third run)")
    del tmpf

    logging.debug("Probing device via 'blockdev --rereadpt %s'", device)
    proc = subprocess.Popen(["blockdev", "--rereadpt", device])
    proc.wait()
    if proc.returncode != 0:
        raise Exception(
            "Couldn't execute blockdev on '%s' (install util-linux?)", device
        )

    set_rw(device)


def mount(source, target, mount_options):
    """Mount specified source on given target

    @source: name of device/ISO that should be mounted
    @target: directory where the ISO should be mounted to
    @options: mount specific options"""

    # note: options.dryrun does not work here, as we have to
    # locate files and identify the grml flavour

    for x in open("/proc/mounts", "r").readlines():
        if x.startswith(source):
            raise CriticalException(
                (
                    "Error executing mount: {0} already mounted - "
                    "please unmount before invoking grml2usb"
                ).format(source)
            )

    if os.path.isdir(source):
        logging.debug("Source %s is not a device, therefore not mounting.", source)
        return 0

    logging.debug("mount %s %s %s", mount_options, source, target)
    proc = subprocess.Popen(["mount"] + list(mount_options) + [source, target])
    proc.wait()
    if proc.returncode != 0:
        raise CriticalException(
            "Error executing mount (no filesystem on the partition?)"
        )
    else:
        logging.debug("register_mountpoint(%s)", target)
        register_mountpoint(target)


def unmount(target, unmount_options):
    """Unmount specified target

    @target: target where something is mounted on and which should be unmounted
    @options: options for umount command"""

    # make sure we unmount only already mounted targets
    target_unmount = False
    mounts = open("/proc/mounts").readlines()
    mountstring = re.compile(".*%s.*" % re.escape(os.path.realpath(target)))
    for line in mounts:
        if re.match(mountstring, line):
            target_unmount = True

    if not target_unmount:
        logging.debug("%s not mounted anymore", target)
    else:
        logging.debug("umount %s %s", list(unmount_options), target)
        proc = subprocess.Popen(["umount"] + list(unmount_options) + [target])
        proc.wait()
        if proc.returncode != 0:
            raise Exception("Error executing umount")
        else:
            logging.debug("unregister_mountpoint(%s)", target)
            unregister_mountpoint(target)


def extract_device_name(device):
    """Extract the device name of a given path

    @device: device name, like /dev/sda1 or /dev/sda
    """
    return re.match(r"/dev/(.*?)\d*$", device).group(1)


def check_for_usbdevice(device):
    """Check whether the specified device is a removable USB device

    @device: device name, like /dev/sda1 or /dev/sda
    """

    usbdevice = extract_device_name(device)
    # newer systems:
    usbdev = os.path.realpath("/sys/class/block/" + usbdevice + "/removable")
    if not os.path.isfile(usbdev):
        # Ubuntu with kernel 2.6.24 for example:
        usbdev = os.path.realpath("/sys/block/" + usbdevice + "/removable")

    if os.path.isfile(usbdev):
        is_usb = open(usbdev).readline()
        if is_usb.find("1"):
            return 0

    return 1


def check_for_fat(partition):
    """Check whether specified partition is a valid VFAT/FAT16 filesystem

    @partition: device name of partition"""

    if not os.access(partition, os.R_OK):
        raise CriticalException(
            "Failed to read device %s"
            " (wrong UID/permissions or device/directory not present?)" % partition
        )

    try:
        filesystem = (
            subprocess.check_output(
                ["/sbin/blkid", "-s", "TYPE", "-o", "value", partition]
            )
            .decode()
            .rstrip()
        )

        if filesystem != "vfat":
            raise CriticalException(
                "Partition %s does not contain a FAT16 filesystem. "
                "(Use --fat16 or run mkfs.vfat %s)" % (partition, partition)
            )

    except OSError:
        raise CriticalException(
            "Sorry, /sbin/blkid not available (install util-linux?)"
        )


def mkdir(directory):
    """Simple wrapper around os.makedirs to get shell mkdir -p behaviour"""

    # just silently pass as it's just fine it the directory exists
    if not os.path.isdir(directory):
        try:
            os.makedirs(directory)
        # pylint: disable-msg=W0704
        except OSError:
            pass


def exec_rsync(source, target):
    """Simple wrapper around rsync to install files

    @source: source file/directory
    @target: target file/directory"""
    logging.debug("Source: %s / Target: %s", source, target)
    proc = subprocess.Popen(["rsync", "-rlptDH", "--inplace", source, target])
    proc.wait()
    if proc.returncode == 12:
        logging.critical("Fatal: No space left on device")
        cleanup()
        sys.exit(1)

    if proc.returncode != 0:
        logging.critical("Fatal: could not install %s", source)
        cleanup()
        sys.exit(1)


def write_uuid(target_file):
    """Generates an returns uuid and write it to the specified file

    @target_file: filename to write the uuid to
    """

    fileh = open(target_file, "w")
    uid = str(uuid.uuid4())
    fileh.write(uid)
    fileh.close()
    return uid


def get_uuid(target):
    """Get the uuid of the specified target. Will generate an uuid if none exist.

    @target: directory/mountpoint containing the grml layout
    """

    conf_target = target + "/conf/"
    uuid_file_name = conf_target + "/bootid.txt"
    if os.path.isdir(conf_target):
        if os.path.isfile(uuid_file_name):
            uuid_file = open(uuid_file_name, "r")
            uid = uuid_file.readline().strip()
            uuid_file.close()
            return uid
        else:
            return write_uuid(uuid_file_name)
    else:
        execute(mkdir, conf_target)
        return write_uuid(uuid_file_name)


def get_shortname(grml_flavour):
    """Get shortname based from grml_flavour name. The rules applied are the same as in grml-live
    @grml_flavour: flavour name which shold be translated to shortname"""

    return re.sub(r"[,._-]", "", grml_flavour)


def copy_system_files(grml_flavour, iso_mount, target):
    """copy grml's main files (like squashfs, kernel and initrd) to a given target

    @grml_flavour: name of grml flavour the configuration should be generated for
    @iso_mount: path where a grml ISO is mounted on
    @target: path where grml's main files should be copied to"""

    squashfs = search_file(grml_flavour + ".squashfs", iso_mount)
    if squashfs is None:
        logging.error("error locating squashfs file")
        raise CriticalException(
            "squashfs file not found, please check that your iso is not corrupt"
        )
    else:
        squashfs_target = target + "/live/" + grml_flavour + "/"
        execute(mkdir, squashfs_target)
    exec_rsync(squashfs, squashfs_target + grml_flavour + ".squashfs")

    for prefix in grml_flavour + "/", "":
        filesystem_module = search_file(prefix + "filesystem.module", iso_mount)
        if filesystem_module:
            break
    if filesystem_module is None:
        logging.error("error locating filesystem.module file")
        raise CriticalException("filesystem.module not found")
    else:
        exec_rsync(filesystem_module, squashfs_target + "filesystem.module")

    shortname = get_shortname(grml_flavour)
    if os.path.isdir(iso_mount + "/boot/" + shortname):
        exec_rsync(iso_mount + "/boot/" + shortname, target + "/boot")
    else:
        kernel = search_file("vmlinuz", iso_mount)
        if kernel is None:
            # compat for releases < 2011.12
            kernel = search_file("linux26", iso_mount)

        if kernel is None:
            logging.error("error locating kernel file")
            raise CriticalException("Kernel not found")

        source = os.path.dirname(kernel) + "/"
        dest = target + "/" + os.path.dirname(kernel).replace(iso_mount, "") + "/"
        execute(mkdir, dest)
        exec_rsync(source, dest)


def copy_grml_files(grml_flavour, iso_mount, target):
    """copy some minor grml files to a given target

    @grml_flavour: the current grml_flavour
    @iso_mount: path where a grml ISO is mounted on
    @target: path where grml's main files should be copied to"""

    grml_target = target + "/grml/"
    execute(mkdir, grml_target)

    grml_prefixe = ["GRML", "grml"]
    for prefix in grml_prefixe:
        filename = "{0}/{1}/{2}".format(iso_mount, prefix, grml_flavour)
        if os.path.exists(filename):
            exec_rsync(filename, grml_target)
            break
    else:
        logging.warning(
            "Warning: could not find flavour directory for %s ", grml_flavour
        )


def copy_addons(iso_mount, target):
    """copy grml's addons files (like allinoneimg, bsd4grml,..) to a given target

    @iso_mount: path where a grml ISO is mounted on
    @target: path where grml's main files should be copied to"""

    addons = target + "/boot/addons/"
    execute(mkdir, addons)

    for addon_file in glob.glob(iso_mount + "/boot/addons/*"):
        filename = os.path.basename(addon_file)
        src_file = iso_mount + "/boot/addons/" + os.path.basename(addon_file)
        logging.debug("Copying addon file %s" % filename)
        exec_rsync(src_file, addons)


def build_loopbackcfg(target):
    """Generate GRUB's loopback.cfg based on existing config files.

    @target: target directory
    """

    grub_dir = "/boot/grub/"
    mkdir(os.path.join(target, grub_dir))

    f = open(target + grub_dir + "loopback.cfg", "w")

    f.write("# grml2usb generated grub2 configuration file\n")
    f.write("source /boot/grub/header.cfg\n")

    for defaults in glob.glob(
        target + os.path.sep + grub_dir + os.path.sep + "*_default.cfg"
    ):
        sourcefile = defaults.split(target + os.path.sep)[1]
        logging.debug("Found source file" + sourcefile)
        os.path.isfile(defaults) and f.write("source " + sourcefile + "\n")

    for ops in glob.glob(
        target + os.path.sep + grub_dir + os.path.sep + "*_options.cfg"
    ):
        sourcefile = ops.split(target + os.path.sep)[1]
        logging.debug("Found source file" + sourcefile)
        os.path.isfile(ops) and f.write("source " + sourcefile + "\n")

    f.write("source /boot/grub/addons.cfg\n")
    f.write("source /boot/grub/footer.cfg\n")
    f.close()


def glob_and_copy(filepattern, dst):
    """Glob on specified filepattern and copy the result to dst

    @filepattern: globbing pattern
    @dst: target directory
    """
    for name in glob.glob(filepattern):
        copy_if_exist(name, dst)


def search_and_copy(filename, search_path, dst):
    """Search for the specified filename at searchpath and copy it to dst

    @filename: filename to look for
    @search_path: base search file
    @dst: destionation to copy the file to
    """
    file_location = search_file(filename, search_path)
    copy_if_exist(file_location, dst)


def copy_if_exist(filename, dst):
    """Copy filename to dst if filename is set.

    @filename: a filename
    @dst: dst file
    """
    if filename and (os.path.isfile(filename) or os.path.isdir(filename)):
        exec_rsync(filename, dst)


def copy_bootloader_files(iso_mount, target, grml_flavour):
    """Copy grml's bootloader files to a given target

    @iso_mount: path where a grml ISO is mounted on
    @target: path where grml's main files should be copied to
    @grml_flavour: name of the current processed grml_flavour
    """

    syslinux_target = target + "/boot/syslinux/"
    execute(mkdir, syslinux_target)

    grub_target = target + "/boot/grub/"
    execute(mkdir, grub_target)

    logo = search_file("logo.16", iso_mount, required=True)
    exec_rsync(logo, syslinux_target + "logo.16")

    bootx64_efi = search_file("bootx64.efi", iso_mount)
    if bootx64_efi:
        mkdir(target + "/efi/boot/")
        exec_rsync(bootx64_efi, target + "/efi/boot/bootx64.efi")

    efi_img = search_file("efi.img", iso_mount)
    if efi_img:
        mkdir(target + "/boot/")
        exec_rsync(efi_img, target + "/boot/efi.img")
        handle_secure_boot(target, efi_img)

    execute(mkdir, target + "/conf/")
    glob_and_copy(iso_mount + "/conf/bootfile_*", target + "/conf/")

    for ffile in ["f%d" % number for number in range(1, 11)]:
        search_and_copy(ffile, iso_mount, syslinux_target + ffile)

    # avoid the "file is read only, overwrite anyway (y/n) ?" question
    # of mtools by syslinux ("mmove -D o -D O s:/ldlinux.sys $target_file")
    if os.path.isfile(syslinux_target + "ldlinux.sys"):
        os.unlink(syslinux_target + "ldlinux.sys")

    (source_dir, name) = get_defaults_file(iso_mount, grml_flavour, "default.cfg")
    (source_dir, defaults_file) = get_defaults_file(iso_mount, grml_flavour, "grml.cfg")

    if not source_dir:
        raise CriticalException(
            "file default.cfg could not be found.\n"
            "Note:  this grml2usb version requires an ISO generated by grml-live >=0.9.24 ...\n"
            "       ... either use grml releases >=2009.10 or switch to an older grml2usb version."
        )

    if not os.path.exists(iso_mount + "/boot/grub/footer.cfg"):
        logging.warning(
            "Warning: Grml releases older than 2011.12 support only one flavour in grub."
        )

    for expr in (
        name,
        "distri.cfg",
        defaults_file,
        "grml.png",
        "hd.cfg",
        "isolinux.cfg",
        "isolinux.bin",
        "isoprompt.cfg",
        "options.cfg",
        "prompt.cfg",
        "vesamenu.cfg",
        "grml.png",
        "*.c32",
    ):
        glob_and_copy(iso_mount + source_dir + expr, syslinux_target)

    for modules_dir in options.syslinuxlibs + SYSLINUX_LIBS:
        if os.path.isdir(modules_dir):
            for filename in glob.glob1(syslinux_target, "*.c32"):
                copy_if_exist(os.path.join(modules_dir, filename), syslinux_target)
            break

    # copy the addons_*.cfg file to the new syslinux directory
    glob_and_copy(iso_mount + source_dir + "addon*.cfg", syslinux_target)

    search_and_copy(
        "hidden.cfg", iso_mount + source_dir, syslinux_target + "new_" + "hidden.cfg"
    )

    # copy all grub files from ISO
    glob_and_copy(iso_mount + "/boot/grub/*", grub_target)

    # finally (after all GRUB files have been installed) build static loopback.cfg
    build_loopbackcfg(target)


def install_iso_files(grml_flavour, iso_mount, device, target):
    """Copy files from ISO to given target

    @grml_flavour: name of grml flavour the configuration should be generated for
    @iso_mount: path where a grml ISO is mounted on
    @device: device/partition where bootloader should be installed to
    @target: path where grml's main files should be copied to"""

    global GRML_DEFAULT
    GRML_DEFAULT = GRML_DEFAULT or grml_flavour
    if options.dryrun:
        return 0
    elif not options.bootloaderonly:
        logging.info("Copying files. This might take a while....")
        try:
            copy_system_files(grml_flavour, iso_mount, target)
            copy_grml_files(grml_flavour, iso_mount, target)
        except CriticalException as error:
            logging.critical("Execution failed: %s", error)
            sys.exit(1)

    if not options.skipaddons:
        if not search_file("addons", iso_mount):
            logging.info("Could not find addons, therefore not installing.")
        else:
            copy_addons(iso_mount, target)

    if not options.copyonly:
        copy_bootloader_files(iso_mount, target, grml_flavour)

        if not options.dryrun:
            handle_bootloader_config(grml_flavour, device, target)

    # make sure we sync filesystems before returning
    logging.info("Synching data (this might take a while)")
    proc = subprocess.Popen(["sync"])
    proc.wait()


def get_device_from_partition(partition):
    device = partition
    partition_number = None
    if partition[-1].isdigit() and not RE_LOOP_DEVICE.match(partition):
        m = RE_P_PARTITION.match(partition) or RE_PARTITION.match(partition)
        if m:
            device = m.group(1)
            partition_number = int(m.group(2)) - 1
    return (device, partition_number)


def get_flavour(flavour_str):
    """Returns the flavour of a grml version string"""
    return re.match(r"[\w-]*", flavour_str).group()


def identify_grml_flavour(mountpath):
    """Get name of grml flavour

    @mountpath: path where the grml ISO is mounted to
    @return: name of grml-flavour"""

    version_files = search_file("grml-version", mountpath, lst_return=True)

    if not version_files:
        if mountpath.startswith("/run/live/medium"):
            logging.critical("Error: could not find grml-version file.")
            logging.critical(
                "Looks like your system is running from RAM but required files are not available."
            )
            logging.critical(
                "Please either boot without toram=... or use boot option toram instead of toram=..."
            )
        else:
            logging.critical("Error: could not find grml-version file.")
        cleanup()
        sys.exit(1)

    flavours = []
    logging.debug("version_files = %s", version_files)
    for version_file in version_files:
        tmpfile = None
        try:
            tmpfile = open(version_file, "r")
            for line in tmpfile.readlines():
                flavours.append(get_flavour(line))
        finally:
            if tmpfile:
                tmpfile.close()

    return flavours


def get_bootoptions(grml_flavour):
    """Returns bootoptions for specific flavour

    @grml_flavour: name of the grml_flavour
    """
    # do NOT write "None" in kernel cmdline
    if not options.bootoptions:
        bootopt = ""
    else:
        bootopt = " ".join(options.bootoptions)
    bootopt = bootopt.replace("%flavour", grml_flavour)
    return bootopt


def handle_grub_config(grml_flavour, device, target):
    """Main handler for generating grub (v1 and v2) configuration

    @grml_flavour: name of grml flavour the configuration should be generated for
    @device: device/partition where grub should be installed to
    @target: path of grub's configuration files"""

    global UUID

    logging.debug("Updating grub configuration")

    grub_target = target + "/boot/grub/"

    bootid_re = re.compile(r"bootid=[\w_-]+")
    live_media_path_re = re.compile(r"live-media-path=[\w_/-]+")

    bootopt = get_bootoptions(grml_flavour)

    remove_regexes = []
    option_re = re.compile(r"(.*/boot/.*(linux26|vmlinuz).*)")

    if options.removeoption:
        for regex in options.removeoption:
            remove_regexes.append(re.compile(regex))

    shortname = get_shortname(grml_flavour)
    for filename in glob.glob(grub_target + "*.cfg"):
        for line in fileinput.input(filename, inplace=1):
            line = line.rstrip("\r\n")
            if option_re.search(line):
                line = bootid_re.sub("", line)
                if shortname in filename:
                    line = live_media_path_re.sub("", line)
                    line = line.rstrip() + " live-media-path=/live/%s/ " % (
                        grml_flavour
                    )
                if bootopt.strip():
                    line = line.replace(" {} ".format(bootopt.strip()), " ")
                    if line.endswith(bootopt):
                        line = line[: -len(bootopt)]
                line = line.rstrip() + r" bootid=%s %s " % (UUID, bootopt)
                for regex in remove_regexes:
                    line = regex.sub(" ", line)
            print(line)
        fileinput.close()


def initial_syslinux_config(target):
    """Generates initial syslinux configuration

    @target path of syslinux's configuration files"""

    target = target + "/"
    filename = target + "grmlmain.cfg"
    if os.path.isfile(target + "grmlmain.cfg"):
        return
    data = open(filename, "w")
    data.write(generate_main_syslinux_config())
    data.close()

    filename = target + "hiddens.cfg"
    data = open(filename, "w")
    data.write("include hidden.cfg\n")
    data.close()


def add_entry_if_not_present(filename, entry):
    """Write entry into filename if entry is not already in the file

    @filename: name of the file
    @entry: data to write to the file
    """
    data = open(filename, "a+")
    data.seek(0)
    for line in data:
        if line == entry:
            break
    else:
        data.write(entry)

    data.close()


def get_flavour_filename(flavour):
    """Generate a iso9960 save filename out of the specified flavour

    @flavour: grml flavour
    """
    return flavour.replace("-", "_")


def adjust_syslinux_bootoptions(src, flavour):
    """Adjust existing bootoptions of specified syslinux config to
    grml2usb specific ones, e.g. change the location of the kernel...

    @src: config file to alter
    @flavour: grml flavour
    """

    append_re = re.compile(r"^(\s*append.*/boot/.*)$", re.I)
    # flavour_re = re.compile("(label.*)(grml\w+)")
    default_re = re.compile(r"(default.cfg)")
    bootid_re = re.compile(r"bootid=[\w_-]+")
    live_media_path_re = re.compile(r"live-media-path=[\w_/-]+")

    bootopt = get_bootoptions(flavour)

    regexe = []
    option_re = None
    if options.removeoption:
        option_re = re.compile(r"/boot/.*/(initrd.gz|initrd.img)")

        for regex in options.removeoption:
            regexe.append(re.compile(r"%s" % regex))

    for line in fileinput.input(src, inplace=1):
        # line = flavour_re.sub(r'\1 %s-\2' % flavour, line)
        line = default_re.sub(r"%s-\1" % flavour, line)
        line = bootid_re.sub("", line)
        line = live_media_path_re.sub("", line)
        line = append_re.sub(r"\1 live-media-path=/live/%s/ " % flavour, line)
        line = append_re.sub(r"\1 boot=live %s " % bootopt, line)
        line = append_re.sub(r"\1 %s=%s " % ("bootid", UUID), line)
        if option_re and option_re.search(line):
            for regex in regexe:
                line = regex.sub(" ", line)
        sys.stdout.write(line)
    fileinput.close()


def adjust_labels(src, replacement):
    """Adjust the specified labels in the syslinux config file src with
    specified replacement
    """
    label_re = re.compile(r"^(\s*label\s*) ([a-zA-Z0-9_-]+)", re.I)
    for line in fileinput.input(src, inplace=1):
        line = label_re.sub(replacement, line)
        sys.stdout.write(line)
    fileinput.close()


def add_syslinux_entry(filename, grml_flavour):
    """Add includes for a specific grml_flavour to the specified filename

    @filename: syslinux config file
    @grml_flavour: grml flavour to add
    """

    entry_filename = "option_%s.cfg" % grml_flavour
    entry = "include %s\n" % entry_filename

    add_entry_if_not_present(filename, entry)
    path = os.path.dirname(filename)

    data = open(path + "/" + entry_filename, "w")
    data.write(generate_flavour_specific_syslinux_config(grml_flavour))
    data.close()


def modify_filenames(grml_flavour, target, filenames):
    """Replace the standard filenames with the new ones

    @grml_flavour: grml-flavour strin
    @target: directory where the files are located
    @filenames: list of filenames to alter
    """
    grml_filename = get_flavour_filename(grml_flavour)
    for filename in filenames:
        old_filename = "%s/%s" % (target, filename)
        new_filename = "%s/%s_%s" % (target, grml_filename, filename)
        os.rename(old_filename, new_filename)


def remove_default_entry(filename):
    """Remove the default entry from specified syslinux file

    @filename: syslinux config file
    """
    default_re = re.compile(r"^(\s*menu\s*default\s*)$", re.I)
    for line in fileinput.input(filename, inplace=1):
        if default_re.match(line):
            continue
        sys.stdout.write(line)
    fileinput.close()


def handle_syslinux_config(grml_flavour, target):
    """Main handler for generating syslinux configuration

    @grml_flavour: name of grml flavour the configuration should be generated for
    @target: path of syslinux's configuration files"""

    logging.debug("Generating syslinux configuration")
    syslinux_target = target + "/boot/syslinux/"
    # should be present via  copy_bootloader_files(), but make sure it exits:
    execute(mkdir, syslinux_target)
    syslinux_cfg = syslinux_target + "syslinux.cfg"

    # install main configuration only *once*, no matter how many ISOs we have:
    syslinux_config_file = open(syslinux_cfg, "w")
    syslinux_config_file.write("timeout 300\n")
    syslinux_config_file.write("include vesamenu.cfg\n")
    syslinux_config_file.close()

    prompt_name = open(syslinux_target + "promptname.cfg", "w")
    prompt_name.write("menu label S^yslinux prompt\n")
    prompt_name.close()

    initial_syslinux_config(syslinux_target)
    flavour_filename = get_flavour_filename(grml_flavour)

    if search_file("default.cfg", syslinux_target):
        modify_filenames(grml_flavour, syslinux_target, ["grml.cfg", "default.cfg"])

    filename = search_file("new_hidden.cfg", syslinux_target)

    # process hidden file
    if not search_file("hidden.cfg", syslinux_target):
        new_hidden = syslinux_target + "hidden.cfg"
        os.rename(filename, new_hidden)
        adjust_syslinux_bootoptions(new_hidden, grml_flavour)
    else:
        new_hidden_file = "%s/%s_hidden.cfg" % (syslinux_target, flavour_filename)
        os.rename(filename, new_hidden_file)
        adjust_labels(new_hidden_file, r"\1 %s-\2" % grml_flavour)
        adjust_syslinux_bootoptions(new_hidden_file, grml_flavour)
        entry = "include %s_hidden.cfg\n" % flavour_filename
        add_entry_if_not_present("%s/hiddens.cfg" % syslinux_target, entry)

    new_default = "%s_default.cfg" % (flavour_filename)
    entry = "include %s\n" % new_default
    defaults_file = "%s/defaults.cfg" % syslinux_target
    new_default_with_path = "%s/%s" % (syslinux_target, new_default)
    new_grml_cfg = "%s/%s_grml.cfg" % (syslinux_target, flavour_filename)

    if os.path.isfile(defaults_file):

        # remove default menu entry in menu
        remove_default_entry(new_default_with_path)

        # adjust all labels for additional isos
        adjust_labels(new_default_with_path, r"\1 %s" % grml_flavour)
        adjust_labels(new_grml_cfg, r"\1 %s-\2" % grml_flavour)

    # always adjust bootoptions
    adjust_syslinux_bootoptions(new_default_with_path, grml_flavour)
    adjust_syslinux_bootoptions(new_grml_cfg, grml_flavour)

    add_entry_if_not_present("%s/defaults.cfg" % syslinux_target, entry)

    add_syslinux_entry("%s/additional.cfg" % syslinux_target, flavour_filename)


def handle_secure_boot(target, efi_img):
    """Provide secure boot support by extracting files from /boot/efi.img

    @target: path where grml's main files should be copied to
    @efi_img: path to the efi.img file that includes the files for secure boot
    """

    mkdir(target + "/efi/boot/")
    efi_mountpoint = tempfile.mkdtemp(
        prefix="grml2usb", dir=os.path.abspath(options.tmpdir)
    )
    logging.debug("efi_mountpoint = %s" % efi_mountpoint)
    register_tmpfile(efi_mountpoint)

    try:
        logging.debug(
            "mount(%s, %s, ['-o', 'ro', '-t', 'vfat']" % (efi_img, efi_mountpoint)
        )
        mount(efi_img, efi_mountpoint, ["-o", "ro", "-t", "vfat"])
    except CriticalException as error:
        logging.critical("Fatal: %s", error)
        sys.exit(1)

    grub_cfg = search_file("grub.cfg", efi_mountpoint + "/boot/grub/")
    logging.debug("grub_cfg = %s" % grub_cfg)
    if not grub_cfg:
        logging.info(
            "No /boot/grub/grub.cfg found inside EFI image, looks like Secure Boot support is missing."
        )
    else:
        mkdir(target + "/boot/grub/x86_64-efi/")
        logging.debug(
            "exec_rsync(%s, %s + '/boot/grub/x86_64-efi/grub.cfg')" % (grub_cfg, target)
        )
        exec_rsync(grub_cfg, target + "/boot/grub/x86_64-efi/grub.cfg")

        logging.debug(
            "exec_rsync(%s + '/EFI/BOOT/grubx64.efi', %s + '/efi/boot/grubx64.efi')'"
            % (efi_mountpoint, target)
        )
        exec_rsync(
            efi_mountpoint + "/EFI/BOOT/grubx64.efi", target + "/efi/boot/grubx64.efi"
        )

        # NOTE - we're overwriting /efi/boot/bootx64.efi from copy_bootloader_files here
        logging.debug(
            "exec_rsync(%s + '/EFI/BOOT/bootx64.efi', %s + '/efi/boot/bootx64.efi')'"
            % (efi_mountpoint, target)
        )
        exec_rsync(
            efi_mountpoint + "/EFI/BOOT/bootx64.efi", target + "/efi/boot/bootx64.efi"
        )

    try:
        unmount(efi_mountpoint, "")
        logging.debug("Unmounted %s" % efi_mountpoint)
        os.rmdir(efi_mountpoint)
        logging.debug("Removed directory %s" % efi_mountpoint)
    except Exception:
        logging.critical("RuntimeError while umount %s" % efi_mountpoint)
        sys.exit(1)


def handle_bootloader_config(grml_flavour, device, target):
    """Main handler for generating bootloader's configuration

    @grml_flavour: name of grml flavour the configuration should be generated for
    @device: device/partition where bootloader should be installed to
    @target: path of bootloader's configuration files"""

    global UUID
    UUID = get_uuid(target)
    if options.skipsyslinuxconfig:
        logging.info("Skipping generation of syslinux configuration as requested.")
    else:
        try:
            handle_syslinux_config(grml_flavour, target)
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            sys.exit(1)

    if options.skipgrubconfig:
        logging.info("Skipping generation of grub configuration as requested.")
    else:
        try:
            handle_grub_config(grml_flavour, device, target)
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            sys.exit(1)


def install(image, device):
    """Install a grml image to the specified device

    @image: directory or is file
    @device: partition or directory to install the device
    """
    iso_mountpoint = image
    remove_image_mountpoint = False
    if os.path.isdir(image):
        if options.force or os.path.exists(os.path.join(image, "live")):
            logging.info("Using %s as install base", image)
        else:
            q = input(
                "%s does not look like a Grml system. "
                "Do you really want to use this image? y/N " % image
            )
            if q.lower() == "y":
                logging.info("Using %s as install base", image)
            else:
                logging.info("Skipping install base %s", image)
    else:
        logging.info("Using ISO %s", image)
        iso_mountpoint = tempfile.mkdtemp(
            prefix="grml2usb", dir=os.path.abspath(options.tmpdir)
        )
        register_tmpfile(iso_mountpoint)
        remove_image_mountpoint = True
        try:
            mount(image, iso_mountpoint, ["-o", "loop,ro", "-t", "iso9660"])
        except CriticalException as error:
            logging.critical("Fatal: %s", error)
            sys.exit(1)

    try:
        install_grml(iso_mountpoint, device)
    finally:
        if remove_image_mountpoint:
            try:
                remove_mountpoint(iso_mountpoint)
            except CriticalException:
                cleanup()
                raise


def install_grml(mountpoint, device):
    """Main logic for copying files of the currently running Grml system.

    @mountpoint: directory where currently running live system resides (usually /run/live/medium)
    @device: partition where the specified ISO should be installed to"""

    device_mountpoint = device
    if os.path.isdir(device):
        logging.info("Specified device is a directory, therefore not mounting.")
        remove_device_mountpoint = False
    else:
        device_mountpoint = tempfile.mkdtemp(prefix="grml2usb")
        register_tmpfile(device_mountpoint)
        remove_device_mountpoint = True
        try:
            set_rw(device)
            mount(device, device_mountpoint, ["-o", "utf8,iocharset=iso8859-1"])
        except CriticalException:
            mount(device, device_mountpoint, "")
    try:
        grml_flavours = identify_grml_flavour(mountpoint)
        for flavour in set(grml_flavours):
            if not flavour:
                logging.warning("No valid flavour found, please check your iso")
            logging.info('Identified grml flavour "%s".', flavour)
            install_iso_files(flavour, mountpoint, device, device_mountpoint)
            GRML_FLAVOURS.add(flavour)
    finally:
        if remove_device_mountpoint:
            remove_mountpoint(device_mountpoint)


def remove_mountpoint(mountpoint):
    """remove a registered mountpoint"""

    try:
        unmount(mountpoint, "")
        if os.path.isdir(mountpoint):
            os.rmdir(mountpoint)
            unregister_tmpfile(mountpoint)
    except CriticalException:
        cleanup()
        raise


def handle_mbr(device):
    """Main handler for installing master boot record (MBR)

    @device: device where the MBR should be installed to"""

    if options.dryrun:
        logging.info("Would install MBR")
        return 0

    mbr_device, partition_number = get_device_from_partition(device)
    if partition_number is None:
        logging.warning("Could not detect partition number, not activating partition")

    # if we get e.g. /dev/loop1 as device we don't want to put the MBR
    # into /dev/loop of course, therefore use /dev/loop1 as mbr_device
    if mbr_device == "/dev/loop":
        mbr_device = device
        logging.info(
            "Detected loop device - using %s as MBR device therefore", mbr_device
        )

    mbrcode = GRML2USB_BASE + "/mbr/mbrldr"
    if options.syslinuxmbr:
        mbrcode = ""
        mbr_locations = (
            "/usr/lib/syslinux/mbr.bin",
            "/usr/lib/syslinux/bios/mbr.bin",
            "/usr/share/syslinux/mbr.bin",
        )
        for mbrpath in mbr_locations:
            if os.path.isfile(mbrpath):
                mbrcode = mbrpath
                break

        if not mbrcode:
            str_locations = " or ".join(['"%s"' % x for x in mbr_locations])
            logging.error("Cannot find syslinux MBR, install it at %s)", str_locations)
            raise CriticalException(
                "syslinux MBR  can not be found at %s." % str_locations
            )
    elif options.mbrmenu:
        mbrcode = GRML2USB_BASE + "/mbr/mbrldr"

    try:
        install_mbr(mbrcode, mbr_device, partition_number, True)
    except (IOError, Exception) as error:
        logging.critical("Execution failed: %s", error)
        sys.exit(1)


def handle_vfat(device):
    """Check for FAT specific settings and options

    @device: device that should checked / formated"""

    # make sure we have mkfs.vfat available
    if options.fat16:
        if not which("mkfs.vfat") and not options.copyonly and not options.dryrun:
            logging.critical("Sorry, mkfs.vfat not available. Exiting.")
            logging.critical("Please make sure to install dosfstools.")
            sys.exit(1)

        if options.force:
            print("Forcing mkfs.fat16 on %s as requested via option --force." % device)
        else:
            # make sure the user is aware of what he is doing
            f = input(
                "Are you sure you want to format the specified partition with fat16? y/N "
            )
            if f == "y" or f == "Y":
                logging.info(
                    "Note: you can skip this question using the option --force"
                )
            else:
                sys.exit(1)
        try:
            mkfs_fat16(device)
        except CriticalException as error:
            logging.critical("Execution failed: %s", error)
            sys.exit(1)

    # check for vfat filesystem
    if device is not None and not os.path.isdir(device) and options.syslinux:
        try:
            check_for_fat(device)
        except CriticalException as error:
            logging.critical("Execution failed: %s", error)
            sys.exit(1)

    if options.skipusbcheck:
        logging.info(
            "Not checking for removable USB device as requested via option --skip-usb-check."
        )
        return

    if (
        not os.path.isdir(device)
        and not check_for_usbdevice(device)
        and not options.force
    ):
        print(
            "Warning: the specified device %s does not look like a removable usb device."
            % device
        )
        f = input("Do you really want to continue? y/N ")
        if f.lower() != "y":
            sys.exit(1)


def handle_compat_warning(device):
    """Backwards compatible checks

    @device: device that should be checked"""

    # make sure we can replace old grml2usb script and warn user when using old way of life:
    if (
        device.startswith("/mnt/external")
        or device.startswith("/mnt/usb")
        and not options.force
    ):
        print("Warning: the semantics of grml2usb has changed.")
        print("Instead of using grml2usb /path/to/iso %s you might" % device)
        print("want to use grml2usb /path/to/iso /dev/... instead.")
        print("Please check out the grml2usb manpage for details.")
        f = input("Do you really want to continue? y/N ")
        if f.lower() != "y":
            sys.exit(1)


def handle_logging():
    """Log handling and configuration"""

    if options.verbose and options.quiet:
        parser.error("please use either verbose (--verbose) or quiet (--quiet) option")

    FORMAT = "%(message)s"
    if options.verbose:
        FORMAT = "%(asctime)-15s %(message)s"
        logging.basicConfig(level=logging.DEBUG, format=FORMAT)
    elif options.quiet:
        logging.basicConfig(level=logging.CRITICAL, format=FORMAT)
    else:
        logging.basicConfig(level=logging.INFO, format=FORMAT)


def handle_bootloader(device):
    """wrapper for installing bootloader

    @device: device where bootloader should be installed to"""

    # Install bootloader only if not using the --copy-only option
    if options.copyonly:
        logging.info(
            "Not installing bootloader and its files as requested via option copyonly."
        )
    elif os.path.isdir(device):
        logging.info("Not installing bootloader as %s is a directory.", device)
    else:
        install_bootloader(device)


def check_options(opts):
    """Check compatibility of provided user opts

    @opts option dict from OptionParser
    """
    if opts.grubmbr and not opts.grub:
        raise CriticalException("--grub-mbr requires --grub option.")

    if opts.copyonly and opts.grub:
        raise CriticalException("Cannot use --copy-only and --grub at the same time.")


def check_programs():
    """check if all needed programs are installed"""
    if options.grub:
        global GRUB_INSTALL
        GRUB_INSTALL = which("grub-install") or which("grub2-install")
        if not GRUB_INSTALL:
            logging.critical(
                "Fatal: grub-install not available (please install the "
                "grub package or drop the --grub option)"
            )
            sys.exit(1)

    if options.syslinux:
        if not which("syslinux"):
            logging.critical(
                "Fatal: syslinux not available (please install the "
                "syslinux package or use the --grub option)"
            )
            sys.exit(1)

    if not which("rsync"):
        logging.critical("Fatal: rsync not available, can not continue - sorry.")
        sys.exit(1)


def load_loop():
    """Runs modprobe loop and throws away its output"""
    if not which("modprobe"):
        logging.critical("Fatal: modprobe not available, can not continue - sorry.")
        logging.critical("Hint: is /sbin missing in PATH?")
        sys.exit(1)

    proc = subprocess.Popen(
        ["modprobe", "loop"], stdout=subprocess.PIPE, stderr=subprocess.PIPE
    )
    proc.communicate()


def main():
    """Main invocation"""

    try:
        if options.version:
            print(os.path.basename(sys.argv[0]) + " " + PROG_VERSION)
            sys.exit(0)

        if len(args) < 2:
            parser.error("invalid usage")

        # log handling
        handle_logging()

        # make sure we have the appropriate permissions
        check_uid_root()

        check_options(options)

        load_loop()

        logging.info("Executing grml2usb version %s", PROG_VERSION)

        if options.dryrun:
            logging.info("Running in simulation mode as requested via option dry-run.")

        check_programs()

        # specified arguments
        device = os.path.realpath(args[len(args) - 1])
        isos = args[0 : len(args) - 1]

        if not os.path.isdir(device):
            if device[-1:].isdigit():
                if int(device[-1:]) > 4 or device[-2:].isdigit():
                    logging.warning(
                        "Warning: installing on partition number >4, booting *might* fail depending on your system."
                    )

        # provide upgrade path
        handle_compat_warning(device)

        if not options.skipbootflag:
            check_boot_flag(device)

        # check for vfat partition
        handle_vfat(device)

        # main operation (like installing files)
        for iso in isos:
            install(iso, device)

        # install mbr
        is_superfloppy = not device[-1:].isdigit()
        if is_superfloppy:
            logging.info("Detected superfloppy format - not installing MBR")

        if not options.skipmbr and not os.path.isdir(device) and not is_superfloppy:
            handle_mbr(device)

        handle_bootloader(device)

        logging.info(
            "Note: grml flavour %s was installed as the default booting system.",
            GRML_DEFAULT,
        )

        for flavour in GRML_FLAVOURS:
            logging.info(
                "Note: you can boot flavour %s using '%s' on the commandline.",
                flavour,
                flavour,
            )

        # finally be polite :)
        logging.info(
            "Finished execution of grml2usb (%s). Have fun with your Grml system.",
            PROG_VERSION,
        )

    except Exception as error:
        logging.critical("Fatal: %s", str(error))
        if options.verbose:
            logging.exception("Exception:")
        sys.exit(1)


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        logging.info("Received KeyboardInterrupt")
        cleanup()

# END OF FILE ##################################################################
# vim:foldmethod=indent expandtab ai ft=python tw=120 fileencoding=utf-8
