/*
 * GridTools
 *
 * Copyright (c) 2014-2019, ETH Zurich
 * All rights reserved.
 *
 * Please, refer to the LICENSE file in the root directory.
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include <gridtools/stencil_composition/icosahedral_grids/icosahedral_topology.hpp>

#include <gtest/gtest.h>

#include <gridtools/common/defs.hpp>
#include <gridtools/stencil_composition/stencil_composition.hpp>
#include <gridtools/tools/backend_select.hpp>

using namespace gridtools;

using icosahedral_topology_t = icosahedral_topology<backend_t>;

template <int... Select>
using layout_t = icosahedral_topology_t::layout_t<selector<Select...>>;

TEST(icosahedral_topology, layout) {
#ifdef GT_BACKEND_CUDA
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 1>, layout_map<3, 2, 1, 0>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 0>, layout_map<2, 1, 0, -1>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 0, 1, 1>, layout_map<2, -1, 1, 0>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 0, 1, 1>, layout_map<3, 2, -1, 1, 0>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 1, 1, 1>, layout_map<5, 4, 3, 2, 1, 0>>::value), "ERROR");
#else
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 1>, layout_map<0, 1, 2, 3>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 0>, layout_map<0, 1, 2, -1>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 0, 1, 1>, layout_map<0, -1, 1, 2>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 0, 1, 1>, layout_map<1, 2, -1, 3, 0>>::value), "ERROR");
    GT_STATIC_ASSERT((std::is_same<layout_t<1, 1, 1, 1, 1, 1>, layout_map<2, 3, 4, 5, 0, 1>>::value), "ERROR");
#endif
}

TEST(icosahedral_topology, make_storage) {

    icosahedral_topology_t grid(4, 6, 7);
    icosahedral_topology_t::meta_storage_t<icosahedral_topology_t::edges, halo<0, 0, 0, 0>, selector<1, 1, 1, 1>> x(
        1, 2, 3, 4);
    {
        auto astorage =
            grid.template make_storage<icosahedral_topology_t::edges, double, halo<0, 0, 0, 0>, selector<1, 1, 1, 1>>(
                "turu");
        auto ameta = *astorage.get_storage_info_ptr();

        ASSERT_EQ(ameta.total_length<0>(), 4);
        ASSERT_EQ(ameta.total_length<1>(), 3);
        ASSERT_EQ(ameta.total_length<2>(), 6);
        ASSERT_EQ(ameta.total_length<3>(), 7);
#ifdef GT_BACKEND_MC
        // 3rd dimension is padded for MC
        ASSERT_EQ(ameta.padded_length<0>(), 4);
        ASSERT_EQ(ameta.padded_length<1>(), 3);
        ASSERT_EQ(ameta.padded_length<2>(), 6);
        ASSERT_EQ(ameta.padded_length<3>(), 8);
#endif
#ifdef GT_BACKEND_CUDA
        // 3rd dimension is padded for CUDA
        ASSERT_EQ(ameta.padded_length<0>(), 4);
        ASSERT_EQ(ameta.padded_length<1>(), 3);
        ASSERT_EQ(ameta.padded_length<2>(), 6);
        ASSERT_EQ(ameta.padded_length<3>(), 32);
#endif
    }
    {
        auto astorage = grid.template make_storage<icosahedral_topology_t::edges,
            double,
            halo<0, 0, 0, 0, 0, 0>,
            selector<1, 1, 1, 1, 1, 1>>("turu", 8, 9);
        auto ameta = *astorage.get_storage_info_ptr();

        ASSERT_EQ(ameta.total_length<0>(), 4);
        ASSERT_EQ(ameta.total_length<1>(), 3);
        ASSERT_EQ(ameta.total_length<2>(), 6);
        ASSERT_EQ(ameta.total_length<3>(), 7);
#ifdef GT_BACKEND_MC
        // 3rd dimension is padded for MC
        ASSERT_EQ(ameta.padded_length<3>(), 8);
#endif
#ifdef GT_BACKEND_CUDA
        ASSERT_EQ(ameta.padded_length<3>(), 32);
#endif
        ASSERT_EQ(ameta.total_length<4>(), 8);
        ASSERT_EQ(ameta.total_length<5>(), 9);
    }
}
