/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/*
 * C Implementation: grib_compare
 *
 * Author: Enrico Fucile <enrico.fucile@ecmwf.int>
 *
 *
 */

#include "grib_tools.h"

GRIB_INLINE static int grib_inline_strcmp(const char* a,const char* b) {
  if (*a != *b) return 1;
  while((*a!=0 && *b!=0) &&  *(a) == *(b) ) {a++;b++;}
  return (*a==0 && *b==0) ? 0 : 1;
}


static int compare_handles(grib_handle* h1,grib_handle* h2,grib_runtime_options* options);
static int in_blacklist(const char* name,grib_runtime_options* options);
static int compare_values(grib_handle* h1,grib_handle *h2,const char *name);
double maxAbsoluteError = 1.0e-6;
double headerError=1.0e-24;
int usePackingError=1;
int error=0;
int count=1;


grib_option grib_options[]={
/*  {id, args, help}, on, command_line, value*/
    {"b:",0,0,1,1,0},
    {"c:",0,0,0,1,0},
    {"e:",0,0,0,1,0},
    {"w:",0,0,0,1,0},
    {"f",0,0,0,1,0},
    {"F",0,0,1,0,0},
    {"q",0,0,1,0,0},
    {"M",0,0,1,0,0},
    {"I",0,0,1,0,0},
    {"V",0,0,0,1,0},
    {"7",0,0,0,1,0},
    {"v",0,0,0,1,0}
};

grib_handle* h1=NULL;

char* grib_tool_description=
         "Compares the grib messages contained in two files one by one in the same order."
     "\n\tIf some differences are found it fails returning an error code."
     "\n\tAll the keys are compared except those listed with the -b option and those that are specific "
     "\n\tof a different grib edition. Also data values are compared and are considered as different "
     "\n\tif their maximum absolute difference is greater than the absolute error, that by default is 0.000001. "
     "\n\tThe value used by the absolute error can be set with te -e option.";
char* grib_tool_name="grib_compare";
char* grib_tool_usage="[options] "
                      "grib_file grib_file";

int grib_options_count=sizeof(grib_options)/sizeof(grib_option);

int main(int argc, char *argv[]) { return grib_tool(argc,argv);}

int grib_tool_before_getopt(grib_runtime_options* options) {
  return 0;
}

int grib_tool_init(grib_runtime_options* options) {

  options->infile_extra->file=fopen(options->infile_extra->name,"r");

  if (!options->infile_extra->file) {
    perror(options->infile_extra->name);
    exit(1);
  }

  if (grib_options_on("e:")) {
    maxAbsoluteError = atof(grib_options_get_option("e:"));
    usePackingError=0;
  } else {
    usePackingError=1;
    maxAbsoluteError = 0.000001;
  }

  options->strict=1;
  return 0;
}

int grib_tool_new_file_action(grib_runtime_options* options) {
   return 0;
}

int grib_tool_new_handle_action(grib_runtime_options* options, grib_handle* h) {
  int err=0;
  char shortName[254]={0,};
  char levelType[254]={0,};
  char level[254]={0,};
  size_t len=254;
  char stepRange[254]={0,};
  h1=grib_handle_new_from_file(h->context,options->infile_extra->file,&err);
  if (!h1 || err!= GRIB_SUCCESS) {
    printf("unable to get a new grib message form %s\n",options->infile_extra->name);
    printf("%s\n",grib_get_error_message(err));
    exit(err);
  }

  len=254;
  grib_get_string(h,"shortName",shortName,&len);
  len=254;
  grib_get_string(h,"stepRange",stepRange,&len);
  len=254;
  grib_get_string(h,"levelType",levelType,&len);
  len=254;
  grib_get_string(h,"level",level,&len);
  
  printf("-- count=%d shortName=%s stepRange=%s levelType=%s level=%s --\n",
         count++,shortName,stepRange,levelType,level);
  
  if(compare_handles(h1,h,options)) error++;

  grib_handle_delete(h1);
  return 0;
}

int grib_tool_skip_handle(grib_runtime_options* options, grib_handle* h) {
  int err=0;
  h1=grib_handle_new_from_file(h->context,options->infile_extra->file,&err);
  if (!h1 || err!= GRIB_SUCCESS) {
    printf("unable to get a new grib message form %s\n",options->infile_extra->name);
    printf("%s\n",grib_get_error_message(err));
    exit(err);
  }
  grib_handle_delete(h1);
  grib_handle_delete(h);
  count++;
  return 0;
}

void grib_tool_print_key_values(grib_runtime_options* options,grib_handle* h) {
  grib_print_key_values(options,h);
}

int grib_tool_finalise_action(grib_runtime_options* options) {
  if (error !=0) exit(1);
  return 0;
}

static int compare_values(grib_handle* h1,grib_handle *h2,const char *name) {
  size_t len1 = 0;
  size_t len2 = 0;
  int err;
  int err1;
  int err2;
  int type1,type2;

  char *sval1 = NULL,*sval2 = NULL;
  unsigned char *uval1 = NULL,*uval2 = NULL;
  double *dval1 = NULL, *dval2 = NULL;
  long *lval1 = NULL, *lval2 = NULL;
  int failed=0;
  double maxdiff=0;
  double packingError1,packingError2;
  double tolerance=maxAbsoluteError;

  if((err = grib_get_native_type(h1,name,&type1)) != GRIB_SUCCESS)
  {
    printf("Oops... cannot get type of [%s] in 1st field: %s\n",name,grib_get_error_message(err));
    return err;
  }

  if((err = grib_get_native_type(h2,name,&type2)) != GRIB_SUCCESS)
  {
    if(err == GRIB_NOT_FOUND)
    {
      printf("[%s] not found in 2nd field\n",name);
      return err;
    }

    printf("Oops... cannot get type of [%s] in 2nd field: %s\n",name,grib_get_error_message(err));
    return err;
  }

  if(type1 != type2)
  {
    printf("Warning, [%s] has different types: 1st field: [%s], 2nd field: [%s]\n",
        name,grib_get_type_name(type1),grib_get_type_name(type2));
    /* return GRIB_TYPE_MISMATCH; */
  }

  if(type1 == GRIB_TYPE_LABEL)
    return err;

  if(type1 == GRIB_TYPE_SECTION)
    return err;


  if((err = grib_get_size(h1,name,&len1)) != GRIB_SUCCESS)
  {
    printf("Oops... cannot get size of [%s] in 1st field: %s\n",name,grib_get_error_message(err));
    return err;
  }

  if((err = grib_get_size(h2,name,&len2)) != GRIB_SUCCESS)
  {
    if(err == GRIB_NOT_FOUND)
    {
      printf("[%s] not found in 2nd field\n",name);
      return err;
    }

    printf("Oops... cannot get size of [%s] in 2nd field: %s\n",name,grib_get_error_message(err));
    return err;
  }

  if(len1 != len2)
  {
    printf("[%s] has different size: 1st field: %ld, 2nd field: %ld\n",name,(long)len1,(long)len2);
    return GRIB_COUNT_MISMATCH;
  }

  switch(type1)
  {
    case GRIB_TYPE_STRING:

      if (len1==0) len1=512;
      if (len2==0) len2=512;
      sval1 = grib_context_malloc(h1->context,len1*sizeof(char));
      sval2 = grib_context_malloc(h2->context,len2*sizeof(char));

      if((err1 = grib_get_string(h1,name,sval1,&len1)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get string value of [%s] in 1nd field: %s\n",
          name,grib_get_error_message(err1));
      }

      if((err2 = grib_get_string(h2,name,sval2,&len2)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get string value of [%s] in 2nd field: %s\n",
          name,grib_get_error_message(err2));
      }

      if(err1 == GRIB_SUCCESS && err2 == GRIB_SUCCESS)
      {
        if(grib_inline_strcmp(sval1,sval2) != 0)
        {
          printf("[%s] string values are different: [%s] and [%s]\n",
            name,sval1,sval2);
          err1 = GRIB_VALUE_MISMATCH;
        }
      }

      grib_context_free(h1->context,sval1);
      grib_context_free(h2->context,sval2);

      if(err1) return err1;
      if(err2) return err2;

      break;

    case GRIB_TYPE_LONG:

      lval1 = grib_context_malloc(h1->context,len1*sizeof(long));
      lval2 = grib_context_malloc(h2->context,len2*sizeof(long));

      if((err1 = grib_get_long_array(h1,name,lval1,&len1)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get long value of [%s] in 1nd field: %s\n",
          name,grib_get_error_message(err1));
      }

      if((err2 = grib_get_long_array(h2,name,lval2,&len2)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get long value of [%s] in 2nd field: %s\n",
          name,grib_get_error_message(err2));
      }

      if(err1 == GRIB_SUCCESS && err2 == GRIB_SUCCESS)
      {
        int i;
        for(i = 0; i < len1; i++)
          if(lval1[i] != lval2[i])
          {
              if(len1 == 1)
                printf("[%s] long  values are different: [%ld] and [%ld]\n",
                  name,lval1[i],lval2[i]);
              else
                printf("[%s] long value %d of %ld are different: [%ld] and [%ld]\n",
                  name,i,(long)len1,lval1[i],lval2[i]);

            err1 = GRIB_VALUE_MISMATCH;
            break;
          }
      }

      grib_context_free(h1->context,lval1);
      grib_context_free(h2->context,lval2);

      if(err1) return err1;
      if(err2) return err2;
      break;

    case GRIB_TYPE_DOUBLE:
      dval1 = grib_context_malloc(h1->context,len1*sizeof(double));
      dval2 = grib_context_malloc(h2->context,len2*sizeof(double));
 
      if (!grib_inline_strcmp(name,"values")) {
        packingError1=0;
        packingError2=0;
        if (usePackingError) {
          err1=grib_get_double(h1,"packingError",&packingError1);
          err2=grib_get_double(h2,"packingError",&packingError2);
          
          if(err1 == GRIB_SUCCESS && err2 == GRIB_SUCCESS) {
            tolerance= packingError1 > packingError2 ? packingError1 : packingError2;
            if (packingError1 != packingError2) 
              printf("WARNING: packingError are different : [%g] [%g]\n",
                      packingError1,packingError2);
          }
        } else tolerance=maxAbsoluteError;
      } else 
        tolerance=headerError;
      
      if((err1 = grib_get_double_array(h1,name,dval1,&len1)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get double value of [%s] in 1nd field: %s\n",
          name,grib_get_error_message(err1));
      }

      if((err2 = grib_get_double_array(h2,name,dval2,&len2)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get double value of [%s] in 2nd field: %s\n",
          name,grib_get_error_message(err2));
      }
      
      if(err1 == GRIB_SUCCESS && err2 == GRIB_SUCCESS)
      {
        int i,imaxdiff,countdiff;
        double diff,diff1,diff2;
        failed=0;
        maxdiff=0;
        imaxdiff=0;
        diff1=0;
        diff2=0;
        countdiff=1;
        for(i = 0; i < len1; i++) {
          if(fabs(dval1[i]-dval2[i]) > tolerance) {
            if(len1 == 1) {
                printf("[%s] double values are different: [%.20e] and [%.20e], diff: %.10e\n",
                  name,dval1[i],dval2[i],dval1[i] - dval2[i]);
                break;
              } else {
                failed=1;
                diff=fabs(dval1[i]-dval2[i]);
                diff1=dval1[i];
                diff2=dval2[i];
                countdiff++;
                if (maxdiff < diff) {maxdiff=diff;imaxdiff=i;}
              }
            err1 = GRIB_VALUE_MISMATCH;
          }
        }

        if (failed) {  
          printf("[%s] %ld values, max abs. diff.: %g ",name,(long)len1,maxdiff);
          if (usePackingError && packingError1!=0)
            printf("\n\tpackingError: [%g] [%g]",packingError1,packingError2);
          else
            printf("tolerance=%g",tolerance);
          printf("\n\t(value1[%d]=%.20e value2[%d]=%.20e) %d values different\n",
                  imaxdiff,dval1[imaxdiff],imaxdiff,
                  dval2[imaxdiff],countdiff);
        }
      }

      grib_context_free(h1->context,dval1);
      grib_context_free(h2->context,dval2);

      if(err1) return err1;
      if(err2) return err2;
      break;

    case GRIB_TYPE_BYTES:

      uval1 = grib_context_malloc(h1->context,len1*sizeof(unsigned char));
      uval2 = grib_context_malloc(h2->context,len2*sizeof(unsigned char));

      if((err1 = grib_get_bytes(h1,name,uval1,&len1)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get bytes value of [%s] in 1nd field: %s\n",
          name,grib_get_error_message(err1));
      }

      if((err2 = grib_get_bytes(h2,name,uval2,&len2)) != GRIB_SUCCESS)
      {
        printf("Oops... cannot get bytes value of [%s] in 2nd field: %s\n",
          name,grib_get_error_message(err2));
      }

      if(err1 == GRIB_SUCCESS && err2 == GRIB_SUCCESS)
      {
        if(memcmp(uval1,uval2,len1) != 0)
        {
        int i;
        for(i = 0; i < len1; i++)
          if(uval1[i] != uval2[i])
          {
              if(len1 == 1)
                printf("[%s] bute values are different: [%02x] and [%02x]\n",
                  name,uval1[i],uval2[i]);
              else
                printf("[%s] byte value %d of %ld are different: [%02x] and [%02x]\n",
                  name,i,(long)len1,uval1[i],uval2[i]);

            err1 = GRIB_VALUE_MISMATCH;
            break;
          }
          err1 = GRIB_VALUE_MISMATCH;
        }
      }

      grib_context_free(h1->context,uval1);
      grib_context_free(h2->context,uval2);

      if(err1) return err1;
      if(err2) return err2;
      break;

    case GRIB_TYPE_LABEL:
      break;

    default:
      printf("Cannot compare [%s], unsoported type %d\n",name,type1);
      return GRIB_UNABLE_TO_COMPARE_ACCESSORS;
      break;
  }

  return GRIB_SUCCESS;

}

static int in_blacklist(const char* name,grib_runtime_options* options) {
    int i=0;

    if (!grib_options_on("b:")) return 0;

    for (i=0;i < options->set_values_count; i++)
      if (!grib_inline_strcmp(name,options->set_values[i].name)) return 1;

    return 0;
}

static int compare_handles(grib_handle* h1,grib_handle* h2,grib_runtime_options* options)
{
  int err = 0;
  int i=0;
  const char* name=NULL;
  grib_keys_iterator* iter  = NULL;

  if ( grib_options_on("c:") ) {
    for (i=0; i< options->compare_count; i++) {
      if( compare_values(h1,h2,options->compare[i].name))
        err++;
    }
  } else {
    iter=grib_keys_iterator_new(h1,
        GRIB_KEYS_ITERATOR_SKIP_FUNCTION |
        GRIB_KEYS_ITERATOR_SKIP_READ_ONLY |
      GRIB_KEYS_ITERATOR_SKIP_DUPLICATES,NULL);

    if (!iter) {
      printf("ERROR: unable to get iterator begin\n");
      exit(1);
    }

    while(grib_keys_iterator_next(iter))
    {
      name=grib_keys_iterator_get_name(iter);

      if( !in_blacklist(name,options) && compare_values(h1,h2,name))
        err++;

    }

    grib_keys_iterator_delete(iter);
  }
  return err;
}
