/**
* Copyright 2005-2007 ECMWF
* 
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/*
//
// C  Interface: grib_associative_array
//
// Description: associative array
//
//
// Author: Enrico Fucile <enrico.fucile@ecmwf>
//
//
//
*/

#ifndef grib_associative_array_H
#define grib_associative_array_H


#define GRIB_ARRAY_SUCCESS             0
#define GRIB_ARRAY_END                 1
#define GRIB_ARRAY_BEGIN               2
#define GRIB_ARRAY_NOT_FOUND           3
#define GRIB_ARRAY_NOT_IMPLEMENTED     4
#define GRIB_ARRAY_ERROR               5
#define GRIB_ARRAY_MALLOC_ERROR        6
#define GRIB_ARRAY_FOUND               7

#define GRIB_RUNTIMETYPE_SUCCESS       0
#define GRIB_RUNTIMETYPE_WRONG_TYPE    1
#define GRIB_RUNTIMETYPE_LONG          2
#define GRIB_RUNTIMETYPE_DOUBLE        3
#define GRIB_RUNTIMETYPE_CHAR          4
#define GRIB_RUNTIMETYPE_VOID_POINTER  5
#define GRIB_RUNTIMETYPE_MALLOC_ERROR  7
#define GRIB_RUNTIMETYPE_ERROR         8
#define GRIB_RUNTIMETYPE_NULL_POINTER  9

typedef struct grib_associative_array grib_associative_array;
typedef struct grib_associative_array_el grib_associative_array_el;
typedef struct grib_associative_array_iterator grib_associative_array_iterator;
typedef struct grib_runtime_type grib_runtime_type;

/* runtime type */
struct grib_runtime_type {
  void* value;
  size_t size;
  int type;
  grib_context* context;
};

/* Runtime type constructor/destructor/copy constructor */
grib_runtime_type* grib_runtimetype_new(grib_context*);
void grib_runtimetype_destroy(grib_runtime_type*);
int grib_runtimetype_copy(grib_runtime_type*,grib_runtime_type*);

/* Runtime type set functions */
int grib_runtimetype_set_long(grib_runtime_type* ,long* ,size_t);
int grib_runtimetype_set_double(grib_runtime_type* ,double* ,size_t);
int grib_runtimetype_set_char(grib_runtime_type* ,char*, size_t);
int grib_runtimetype_set(grib_runtime_type* ,void*, int, size_t);

/* Runtime type get functions */
int grib_runtimetype_get_long(grib_runtime_type* ,long**,size_t* );
int grib_runtimetype_get_double(grib_runtime_type* ,double**,size_t* );
int grib_runtimetype_get_char(grib_runtime_type* ,char**,size_t*);
int grib_runtimetype_get(grib_runtime_type* ,void**,size_t*);
int grib_runtimetype_get_type(grib_runtime_type*,int*);

/* associative array element */
struct grib_associative_array_el {
  char* key;
  grib_runtime_type* value;
  grib_context *context;
};

/* associative array element Constructor/Destructor */
grib_associative_array_el* grib_associative_array_el_new(grib_context*);
void grib_associative_array_el_destroy(grib_associative_array_el*);

/* associative array structure */
struct grib_associative_array {
  grib_associative_array_el *el;
  grib_associative_array *prev;
  grib_associative_array *next;
  grib_context *context;
  int* pcount;
};

/* iterator through the associative array elements */
struct grib_associative_array_iterator {
  grib_associative_array **par;
  grib_context *context;
};

/*  Constructor/Destructor  */
grib_associative_array* grib_associative_array_new(grib_context *gc);
void grib_associative_array_destroy(grib_associative_array* ar);

/* Set functions */
int grib_associative_array_set(grib_associative_array* ar,char* key,grib_runtime_type* value);
int grib_associative_array_set_long(grib_associative_array* ar,char* key,long* value,size_t size);
int grib_associative_array_set_double(grib_associative_array* ar,char* key,double* dvalue,size_t size);
int grib_associative_array_set_char(grib_associative_array* ar,char* key,char* value,size_t size);

/* Push functions */
int grib_associative_array_push(grib_associative_array* ar,char* key,grib_runtime_type* value);
int grib_associative_array_push_long(grib_associative_array* ar,char* key,long* value,size_t size);
int grib_associative_array_push_double(grib_associative_array* ar,char* key,double* value,size_t size);
int grib_associative_array_push_char(grib_associative_array* ar,char* key,char* value,size_t size);

/* Get functions */
int grib_associative_array_get(grib_associative_array* ar,const char* key,grib_runtime_type** value);
int grib_associative_array_get_long(grib_associative_array* ar,const char* key,long** value,size_t* size);
int grib_associative_array_get_double(grib_associative_array* ar,const char* key,double** value,size_t* size);
int grib_associative_array_get_char(grib_associative_array* ar,const char* key,char** value,size_t* size);

/* Loads a key=values file into an associative array. */
int grib_file_to_array(FILE *in,grib_associative_array* ar);

#endif
