/*
Gregorio GregorioTeX output format.
Copyright (C) 2006-2010 Elie Roux <elie.roux@telecom-bretagne.eu>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file
 * @brief The plugin which writes a GregorioTeX score.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gregorio/struct.h>
#include <gregorio/unicode.h>
#if ALL_STATIC == 0
#include <gregorio/plugin.h>
#endif
#include <gregorio/messages.h>
#include <gregorio/characters.h>

#include "gregoriotex.h"

#if ALL_STATIC == 0
DECLARE_PLUGIN (gregoriotex)
{
.id = "gtex",.name = "gregoriotex",.description =
    "GregorioTeX output plugin",.author =
    "Elie Roux <elie.roux@telecom-bretagne.eu>",.file_extension =
    "tex",.type = GREGORIO_PLUGIN_OUTPUT,.write = write_score};
#endif

/// the value indicating to GregorioTeX that there is no flat
#define NO_KEY_FLAT 'a'

gregoriotex_status *status = NULL;

#if ALL_STATIC == 0
void
write_score (FILE * f, gregorio_score * score)
#else
void
gregoriotex_write_score (FILE * f, gregorio_score * score)
#endif
{
  gregorio_character *first_text;
  status = malloc (sizeof (gregoriotex_status));
  status -> bottom_line = 0;
  status -> to_modify_note = NULL;
  // a char that will contain 1 if it is the first syllable and 0 if not. It is for the initial.
  char first_syllable = 0;
  char clef_letter;
  int clef_line;
  char clef_flat = NO_KEY_FLAT;
  gregorio_syllable *current_syllable;
  // the current line (as far as we know), it is always 0, it can be 1 in the case of the first line of a score with a two lines initial
  unsigned char line = 0;
  gregorio_line *first_line;

  if (!f)
    {
      gregorio_message (_
			("call with NULL file"),
			"gregoriotex_write_score", ERROR, 0);
      return;
    }

  if (score->number_of_voices != 1)
    {
      gregorio_message (_
			("gregoriotex only works in monophony (for the moment)"),
			"gregoriotex_write_score", ERROR, 0);
    }
  if (score->name)
    {
      fprintf (f, "%% Name: %s\n", score->name);
    }
  if (score->si.author)
    {
      fprintf (f, "%% Author: %s\n", score->si.author);
    }
  if (score->gabc_copyright)
    {
      fprintf (f, "%% The copyright of this gabc is: %s\n",
	       score->gabc_copyright);
    }
  if (score->score_copyright)
    {
      fprintf (f, "%% The copyright of the score is: %s\n",
	       score->score_copyright);
    }

  fprintf (f, "%% File generated by gregorio %s\n", VERSION);
  fprintf (f, "\\begingregorioscore%%\n");
  // if necessary, we add some bottom space to the first line
  first_line = (gregorio_line *) malloc (sizeof (gregorio_line));
  gregoriotex_getlineinfos (score->first_syllable, first_line);
  // here we assume that if there is chironomy somewhere, there will be chironomy on the first line
  if (first_line->ictus)
    {
      fprintf (f, "\\greactivatechironomy %%\n");
      fprintf (f, "\\greinsertchiroline %%\n");
    }
  if (first_line->additional_bottom_space != 0
      || first_line->translation != 0)
    {
      fprintf (f, "\\grefirstlinebottomspace{%u}{%u}%%\n",
	       first_line->additional_bottom_space, first_line->translation);
    }
  free (first_line);
  // we select the good font
  if (score->gregoriotex_font)
    {
      if (!strcmp (score->gregoriotex_font, "gregorio"))
	{
	  fprintf (f, "\\setgregoriofont{gregorio}%%\n");
	}
      if (!strcmp (score->gregoriotex_font, "parmesan"))
	{
	  fprintf (f, "\\setgregoriofont{parmesan}%%\n");
	}
      if (!strcmp (score->gregoriotex_font, "greciliae"))
	{
	  fprintf (f, "\\setgregoriofont{greciliae}%%\n");
	}
      if (!strcmp (score->gregoriotex_font, "gregoria"))
	{
	  fprintf (f, "\\setgregoriofont{gregoria}%%\n");
	}
    }
// first we draw the initial (first letter) and the initial key
  if (score->initial_style == NO_INITIAL)
    {
      fprintf (f, "\\grenoinitial %%\n");
    }
  else
    {
      if (score->initial_style == BIG_INITIAL)
	{
	  fprintf (f, "\\gresetbiginitial %%\n");
	  line = 1;
	}
      first_text = gregorio_first_text (score);
      if (first_text)
	{
	  fprintf (f, "\\greinitial{");
	  gregorio_write_initial (first_text, f,
				  (&gtex_write_verb),
				  (&gtex_print_char),
				  (&gtex_write_begin),
				  (&gtex_write_end),
				  (&gtex_write_special_char));
	  fprintf (f, "}%%\n");
	  first_syllable = SKIP_FIRST_LETTER;
	}
    }
  if (score->mode != 0)
    {
      fprintf (f, "\\gregorianmode{%d}%%\n", score->mode);
    }
  if (score->si.manuscript_reference)
    {
      fprintf (f, "\\grescorereference{%s}%%\n",
	       score->si.manuscript_reference);
    }
  if (score->first_voice_info)
    {
      gregoriotex_write_voice_info (f, score->first_voice_info);
    }
  fprintf (f, "\\grebeginnotes %%\n");
  if (score->first_voice_info)
    {
      gregorio_det_step_and_line_from_key (score->first_voice_info->
					   initial_key, &clef_letter,
					   &clef_line);
      if (score->first_voice_info->flatted_key == FLAT_KEY)
	{
	  clef_flat = gregoriotex_clef_flat_height (clef_letter, clef_line);
	}
      else
	{
	  clef_flat = NO_KEY_FLAT;
	}
    }
  else
    {
      clef_letter = 'c';
      clef_line = 3;
      clef_flat = NO_KEY_FLAT;
    }
  fprintf (f, "\\gresetinitialclef{%c}{%d}{%c}%%\n", clef_letter, clef_line,
	   clef_flat);
  current_syllable = score->first_syllable;
  while (current_syllable)
    {
      gregoriotex_write_syllable (f, current_syllable,
				  &first_syllable, &line);
      current_syllable = current_syllable->next_syllable;
    }
  fprintf (f, "\\endgregorioscore %%\n\\endinput %%\n");
  free(status);
}

/** 
 * This now does nothing useful, since the manuscript_reference is
 * now part of the score info.  But we keep it here in case it may
 * be needed in future.
 */
void
gregoriotex_write_voice_info (FILE * f, gregorio_voice_info * voice_info)
{
  if (!voice_info)
    {
      return;
    }
}

// this function indicates if the syllable is the last of the line. If it's the
// last of the score it returns 0, as it's handled another way
unsigned char
gregoriotex_is_last_of_line (gregorio_syllable * syllable)
{
  gregorio_element *current_element = NULL;
  if (!(syllable->next_syllable))
    {
      return 0;
    }
  if ((syllable->next_syllable->elements)[0]
      && (syllable->next_syllable->elements)[0]->type == GRE_END_OF_LINE)
    {
      // the next syllable start by an end of line
      return 1;
    }
  current_element = (syllable->elements)[0];
  while (current_element)
    {
      if (current_element->type == GRE_END_OF_LINE)
	{
	  // we return 1 only if the end of line is the last element
	  if (!(current_element->next))
	    {
	      return 1;
	    }
	  else
	    {
	      return 0;
	    }
	}
      else
	{
	  current_element = current_element->next;
	}
    }
  return 0;
}

void
gregoriotex_write_syllable (FILE * f,
			    gregorio_syllable * syllable,
			    char *first_syllable, unsigned char *line_number)
{
  gregorio_element *current_element;
  gregorio_line *line;

  if (!syllable)
    {
      return;
    }
  /* first we check if the syllable is only a end of line.
     If it is the case, we don't print anything but a comment (to be able to read it if we read GregorioTeX).
     The end of lines are treated separately in GregorioTeX, it is buit inside the TeX structure. */
  if ((syllable->elements) && (syllable->elements)[0])
    {

      if ((syllable->elements)[0]->type == GRE_END_OF_LINE)
	{
	  line = (gregorio_line *) malloc (sizeof (gregorio_line));
	  gregoriotex_getlineinfos (syllable->next_syllable, line);
	  if (line->additional_bottom_space == 0
	      && line->additional_top_space == 0 && line->translation == 0 && line->abovelinestext == 0)
	    {
	      if ((syllable->elements)[0]->element_type != GRE_END_OF_PAR)
		{
		  fprintf (f, "%%\n%%\n\\grenewline %%\n%%\n%%\n");
		}
	      else
		{
		  fprintf (f, "%%\n%%\n\\grenewparline %%\n%%\n%%\n");
		}
	    }
	  else
	    {
	      if ((syllable->elements)[0]->element_type != GRE_END_OF_PAR)
		{
		  fprintf (f,
			   "%%\n%%\n\\grenewlinewithspace{%u}{%u}{%u}{%u}%%\n%%\n%%\n",
			   line->additional_top_space,
			   line->additional_bottom_space, line->translation, line->abovelinestext);
		}
	      else
		{
		  fprintf (f,
			   "%%\n%%\n\\grenewparlinewithspace{%u}{%u}{%u}{%u}%%\n%%\n%%\n",
			   line->additional_top_space,
			   line->additional_bottom_space, line->translation, line->abovelinestext);
		}

	    }
	  if (line->ictus)
	    {
	      fprintf (f, "\\greinsertchiroline %%\n%%\n");
	    }
	  free (line);
	  if (*line_number == 1)
	    {
	      fprintf (f, "\\greadjustthirdline %%\n");
	      *line_number = 0;
	    }
	  return;
	}
      if ((syllable->elements)[0]->type == GRE_BAR)
	{
	  if (!syllable->next_syllable && !syllable->text
	      && (syllable->elements)[0]->element_type == B_DIVISIO_FINALIS)
	    {
	      fprintf (f, "\\grefinaldivisiofinalis{0}%%\n");
	      if ((syllable->elements)[0]->next 
	          && (syllable->elements)[0]->next->type == GRE_TEXVERB_ELEMENT
	          && (syllable->elements)[0]->next->texverb)
	        {
            	  fprintf (f,
		   "%% verbatim text at element level:\n%s%%\n%% end of verbatim text\n",
		   (syllable->elements)[0]->next->texverb);
	        }
	      return;
	    }
	  if (!syllable->next_syllable && !syllable->text
	      && (syllable->elements)[0]->element_type == B_DIVISIO_MAIOR)
	    {
	      fprintf (f, "\\grefinaldivisiomaior{0}%%\n");
	      if ((syllable->elements)[0]->next 
	          && (syllable->elements)[0]->next->type == GRE_TEXVERB_ELEMENT
	          && (syllable->elements)[0]->next->texverb)
	        {
            	  fprintf (f,
		   "%% verbatim text at element level:\n%s%%\n%% end of verbatim text\n",
		   (syllable->elements)[0]->next->texverb);
	        }
	      return;
	    }
	  else
	    {
	      fprintf (f, "\\grebarsyllable");
	    }
	}
      else
	{
	  fprintf (f, "\\gresyllable");
	}
    }
  else
    {
      fprintf (f, "\\gresyllable");
    }
  gregoriotex_write_text (f, syllable->text, first_syllable);
  if (syllable->position == WORD_END
      || syllable->position == WORD_ONE_SYLLABLE || !syllable->text
      || !syllable->next_syllable
      || !syllable->next_syllable->type == GRE_END_OF_LINE
      || !syllable->next_syllable->type == GRE_END_OF_PAR)
    {
      fprintf (f, "{1}");
    }
  else
    {
      fprintf (f, "{0}");
    }
  if (syllable->next_syllable)
    {
      gregoriotex_write_next_first_text (f, syllable->next_syllable->text);
      fprintf (f, "{%d}{",
	       gregoriotex_syllable_first_type (syllable->next_syllable));
    }
  else
    {
      fprintf (f, "{}{}{0}{");
    }
  if (syllable->translation)
    {
      fprintf (f, "%%\n\\grewritetranslation{");
      gregoriotex_write_translation (f, syllable->translation);
      fprintf (f, "}%%\n");
    }
  if (syllable->abovelinestext)
    {
      fprintf (f, "%%\n\\gresettextabovelines{%s}%%\n", syllable->abovelinestext);
    }
  if (gregoriotex_is_last_of_line (syllable) != 0)
    {
      fprintf (f, "%%\n\\grelastofline %%\n");
    }
  if (!syllable->next_syllable)
    {
      fprintf (f, "%%\n\\grelastofscore %%\n");
    }
  fprintf (f, "}{%%\n");

  current_element = (syllable->elements)[0];
  while (current_element)
    {
      if (current_element->type == GRE_SPACE)
	{
	  switch (current_element->element_type)
	    {
	    case SP_ZERO_WIDTH:
	      fprintf (f, "\\greendofelement{3}{1}%%\n");
	      break;
	    case SP_LARGER_SPACE:
	      fprintf (f, "\\greendofelement{1}{0}%%\n");
	      break;
	    case SP_GLYPH_SPACE:
	      fprintf (f, "\\greendofelement{2}{0}%%\n");
	      break;
	    case SP_GLYPH_SPACE_NB:
	      fprintf (f, "\\greendofelement{2}{1}%%\n");
	      break;
	    case SP_LARGER_SPACE_NB:
	      fprintf (f, "\\greendofelement{1}{1}%%\n");
	      break;
	    case SP_NEUMATIC_CUT_NB:
	      fprintf (f, "\\greendofelement{0}{1}%%\n");
	      break;
	    default:
	      break;
	    }
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_TEXVERB_ELEMENT
	  && current_element->texverb)
	{
	  fprintf (f,
		   "%% verbatim text at element level:\n%s%%\n%% end of verbatim text\n",
		   current_element->texverb);
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_ALT
	  && current_element->texverb)
	{
	  fprintf (f,
		   "\\gresettextabovelines{%s}%%\n",
		   current_element->texverb);
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_C_KEY_CHANGE)
	{
	  if (current_element->previous &&
	      current_element->previous->type == GRE_BAR)
	    {
	      if (current_element->additional_infos == FLAT_KEY)
		{
		  // the third argument is 0 or 1 according to the need for a space before the clef
		  fprintf (f, "\\grechangeclef{c}{%d}{0}{%c}%%\n",
			   current_element->element_type - 48,
			   gregoriotex_clef_flat_height ('c',
							 current_element->
							 element_type - 48));
		}
	      else
		{
		  fprintf (f, "\\grechangeclef{c}{%d}{0}{%c}%%\n",
			   current_element->element_type - 48, NO_KEY_FLAT);
		}
	    }
	  else
	    {
	      if (current_element->additional_infos == FLAT_KEY)
		{
		  // the third argument is 0 or 1 according to the need for a space before the clef
		  fprintf (f, "\\grechangeclef{c}{%d}{1}{%c}%%\n",
			   current_element->element_type - 48,
			   gregoriotex_clef_flat_height ('c',
							 current_element->
							 element_type - 48));
		}
	      else
		{
		  fprintf (f, "\\grechangeclef{c}{%d}{1}{%c}%%\n",
			   current_element->element_type - 48, NO_KEY_FLAT);
		}
	    }
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_F_KEY_CHANGE)
	{
	  if (current_element->previous &&
	      current_element->previous->type == GRE_BAR)
	    {
	      if (current_element->additional_infos == FLAT_KEY)
		{
		  // the third argument is 0 or 1 according to the need for a space before the clef
		  fprintf (f, "\\grechangeclef{f}{%d}{0}{%c}%%\n",
			   current_element->element_type - 48,
			   gregoriotex_clef_flat_height ('f',
							 current_element->
							 element_type - 48));
		}
	      else
		{
		  fprintf (f, "\\grechangeclef{f}{%d}{0}{%c}%%\n",
			   current_element->element_type - 48, NO_KEY_FLAT);
		}
	    }
	  else
	    {
	      if (current_element->additional_infos == FLAT_KEY)
		{
		  // the third argument is 0 or 1 according to the need for a space before the clef
		  fprintf (f, "\\grechangeclef{f}{%d}{1}{%c}%%\n",
			   current_element->element_type - 48,
			   gregoriotex_clef_flat_height ('f',
							 current_element->
							 element_type - 48));
		}
	      else
		{
		  fprintf (f, "\\grechangeclef{f}{%d}{1}{%c}%%\n",
			   current_element->element_type - 48, NO_KEY_FLAT);
		}
	    }
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_CUSTO)
	{
	  // we also print an unbreakable larger space before the custo
	  fprintf (f, "\\greendofelement{1}{1}%%\n\\grecusto{%c}%%\n",
		   current_element->element_type);
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_BAR)
	{
	  if (current_element->next)
	    {
	      gregoriotex_write_bar (f, current_element->element_type,
				     current_element->additional_infos,
				     INSIDE_BAR);
	    }
	  else
	    {
	      gregoriotex_write_bar (f, current_element->element_type,
				     current_element->additional_infos,
				     SYLLABLE_BAR);
	    }
	  current_element = current_element->next;
	  continue;
	}
      if (current_element->type == GRE_END_OF_LINE)
	{
	  line = (gregorio_line *) malloc (sizeof (gregorio_line));
	  // here we suppose we don't have two linebreaks in the same syllable
	  gregoriotex_getlineinfos (syllable->next_syllable, line);
	  if (line->additional_bottom_space == 0
	      && line->additional_top_space == 0 && line->translation == 0 && line->abovelinestext == 0)
	    {
	      if (current_element->element_type != GRE_END_OF_PAR)
		{
		  fprintf (f, "%%\n%%\n\\grenewline %%\n%%\n%%\n");
		}
	      else
		{
		  fprintf (f, "%%\n%%\n\\grenewparline %%\n%%\n%%\n");
		}
	    }
	  else
	    {
	      if (current_element->element_type != GRE_END_OF_PAR)
		{
		  fprintf (f,
			   "%%\n%%\n\\grenewlinewithspace{%u}{%u}{%u}{%u}%%\n%%\n%%\n",
			   line->additional_top_space,
			   line->additional_bottom_space, line->translation, line->abovelinestext);
		}
	      else
		{
		  fprintf (f,
			   "%%\n%%\n\\grenewparlinewithspace{%u}{%u}{%u}{%u}%%\n%%\n%%\n",
			   line->additional_top_space,
			   line->additional_bottom_space, line->translation, line->abovelinestext);
		}
	    }
	  if (line->ictus)
	    {
	      fprintf (f, "\\greinsertchiroline %%\n%%\n");
	    }
	  free (line);
	  if (*line_number == 1)
	    {
	      fprintf (f, "\\greadjustthirdline %%\n");
	      *line_number = 0;
	    }
	  current_element = current_element->next;
	  continue;
	}
// there current_element->type is GRE_ELEMENT
      gregoriotex_write_element (f, syllable, current_element);
      if (current_element->next 
      && (current_element->next->type == GRE_ELEMENT
      || current_element->next->next && current_element->next->type == GRE_ALT
         && current_element->next->next->type == GRE_ELEMENT
      )
      )
	{
	  fprintf (f, "\\greendofelement{0}{0}%%\n");
	}
      current_element = current_element->next;
    }
  fprintf (f, "}%%\n");
  if (syllable->position == WORD_END
      || syllable->position == WORD_ONE_SYLLABLE || !syllable->text)
    {
      fprintf (f, "%%\n");
    }
}

/*!
 * @brief This is an arbitrary maximum of notes we consider to affect the syllable's text.
 * 
 * We automatically lower the textline if the chant notes are below the staff, so that they do not overlap.
 * However, after a certain point (defined by this value), the notes extend far beyond the letters of the text,
 * and thus lowering the text does not need to be considered.
 *
 * Used by gregoriotex_getlineinfos()
 */
#define NUMBER_OF_NOTES 6

/*!
 * @brief function filling the gregorio_line (see gregoriotex.h) struct with the infos on the line following syllable
 *
 */
void
gregoriotex_getlineinfos (gregorio_syllable * syllable, gregorio_line * line)
{
  gregorio_element *element;
  gregorio_glyph *glyph;
  gregorio_note *note;
  unsigned char i;		// a counter to know at which note we are in a syllable
  // the idea behind it is that after the 6th note (arbitrary number), we can
  // consider that the bottom notes won't be bothering the text, because
  // they won't be above it.

  if (line == NULL)
    {
      gregorio_message (_
			("call with NULL pointer"),
			"gregoriotex_write_score", ERROR, 0);
      return;
    }

  line->additional_top_space = 0;
  line->additional_bottom_space = 0;
  line->translation = 0;
  line->ictus = 0;
  line->abovelinestext = 0;

  if (syllable == NULL)
    {
      // we allow the call with NULL syllable
      return;
    }

  while (syllable)
    {
      i = 0;
      if (syllable->translation)
	{
	  line->translation = 1;
	}
      if (syllable->abovelinestext)
	{
	  line->abovelinestext = 1;
	}
      element = (syllable->elements)[0];
      while (element)
	{
	  if (element->type == GRE_END_OF_LINE)
	    {
	      return;
	    }
	  if (element->type == GRE_ALT)
	    {
	      line->abovelinestext = 1;
	    }
	  if (element->type == GRE_BAR)
	    {
	      switch (element->additional_infos)
		{
		case _ICTUS_A:
		case _ICTUS_T:
		case _V_EPISEMUS_ICTUS_A:
		case _V_EPISEMUS_ICTUS_T:
		  line->ictus = 1;
		  break;
		default:
		  break;
		}
	      element = element->next;
	      continue;
	    }
	  if (element->type != GRE_ELEMENT)
	    {
	      element = element->next;
	      continue;
	    }
	  glyph = element->first_glyph;
	  while (glyph)
	    {
	      if (glyph->type != GRE_GLYPH)
		{
		  glyph = glyph->next;
		  continue;
		}
	      note = glyph->first_note;
	      while (note)
		{
		  i = i + 1;
		  if (note->rare_sign == _ICTUS_A
		      || note->rare_sign == _ICTUS_T)
		    {
		      line->ictus = 1;
		    }
		  switch (note->pitch)
		    {
		    case 'a':
		      if (line->additional_bottom_space < 3
			  && i < NUMBER_OF_NOTES)
			{
			  // the idea is to put an extra space when a low note has a vertical episemus
			  if (note->signs >= _V_EPISEMUS)
			    {
			      line->additional_bottom_space = 4;
			    }
			  else
			    {
			      line->additional_bottom_space = 3;
			    }
			}
		      break;
		    case 'b':
		      if (line->additional_bottom_space < 2
			  && i < NUMBER_OF_NOTES)
			{
			  if (note->signs >= _V_EPISEMUS)
			    {
			      line->additional_bottom_space = 3;
			    }
			  else
			    {
			      line->additional_bottom_space = 2;
			    }
			}
		      break;
		    case 'c':
		      if (line->additional_bottom_space < 1
			  && i < NUMBER_OF_NOTES)
			{
			  if (note->signs >= _V_EPISEMUS)
			    {
			      line->additional_bottom_space = 2;
			    }
			  else
			    {
			      line->additional_bottom_space = 1;
			    }
			}
		      break;
		    case 'm':
		      if (line->additional_top_space < 3)
			{
			  line->additional_top_space = 3;
			}
		      break;
		    case 'l':
		      if (line->additional_top_space < 2)
			{
			  line->additional_top_space = 2;
			}
		      break;
		    case 'k':
		      if (line->additional_top_space < 1)
			{
			  line->additional_top_space = 1;
			}
		      break;
		    default:
		      break;
		    }
		  note = note->next;
		}
	      glyph = glyph->next;
	    }
	  element = element->next;
	}
      syllable = syllable->next_syllable;
    }
}

/// @todo: What is this comment -here- for? --> we will need type for one thing for the moment : type=first_syllable=1 when it is the first syllable (for the initial).

/*!
 * @brief Prints the beginning of each text style
 *
 */
void
gtex_write_begin (FILE * f, unsigned char style)
{
  switch (style)
    {
    case ST_ITALIC:
      fprintf (f, "\\greitalic{");
      break;
    case ST_SMALL_CAPS:
      fprintf (f, "\\gresmallcaps{");
      break;
    case ST_BOLD:
      fprintf (f, "\\greboldfont{");
      break;
    case ST_FORCED_CENTER:
    case ST_CENTER:
      fprintf (f, "}{");
      break;
    case ST_TT:
      fprintf (f, "\\grett{");
      break;
    case ST_UNDERLINED:
      fprintf (f, "\\greul{");
      break;
    default:
      break;
    }
}

/**
 * @brief Ends each text style
 */
void
gtex_write_end (FILE * f, unsigned char style)
{
  switch (style)
    {
    case ST_FORCED_CENTER:
    case ST_CENTER:
      fprintf (f, "}{");
      break;
    case ST_INITIAL:
      break;
    default:
      fprintf (f, "}");
      break;
    }
}

/// a specific function for writing ends of the two first parts of the text of the next syllable
void
gtex_write_end_for_two (FILE * f, unsigned char style)
{
  fprintf (f, "}");
}

/*!
 * @brief Writes GregorioTeX special characters.
 *
 * This function takes the special characters as input (i.e. from gabc representation), and writes them in
 * GregorioTeX form.
 */
void
gtex_write_special_char (FILE * f, grewchar * special_char)
{
  if (!gregorio_wcsbufcmp (special_char, "A/"))
    {
      fprintf (f, "\\Abar{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "%"))
    {
      fprintf (f, "\\%%{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "R/"))
    {
      fprintf (f, "\\Rbar{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "V/"))
    {
      fprintf (f, "\\Vbar{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "'æ"))
    {
      fprintf (f, "\\'\\ae{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "'ae"))
    {
      fprintf (f, "\\'\\ae{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "'œ"))
    {
      fprintf (f, "\\'\\oe{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "'oe"))
    {
      fprintf (f, "\\'\\oe{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "ae"))
    {
      fprintf (f, "\\ae{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "oe"))
    {
      fprintf (f, "\\oe{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "*"))
    {
      fprintf (f, "\\grestar{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "+"))
    {
      fprintf (f, "\\gredagger{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "-"))
    {
      fprintf (f, "\\zerhyph{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "\\"))
    {
      fprintf (f, "\\textbackslash{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "&"))
    {
      fprintf (f, "\\&{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "#"))
    {
      fprintf (f, "\\#{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "_"))
    {
      fprintf (f, "\\_{}");
      return;
    }
  if (!gregorio_wcsbufcmp (special_char, "~"))
    {
      fprintf (f, "\\gretilde{}");
      return;
    }
}

void
gtex_write_verb (FILE * f, grewchar * first_char)
{
  gregorio_print_unistring (f, first_char);
}

void
gtex_print_char (FILE * f, grewchar to_print)
{
  switch (to_print)
    {
    case L'*':
      fprintf (f, "\\grestar{}");
      break;
    case L'%':
      fprintf (f, "\\%%{}");
      break;
    case L'\\':
      fprintf (f, "\\textbackslash{}");
      break;
    case L'&':
      fprintf (f, "\\&{}");
      break;
    case L'#':
      fprintf (f, "\\#{}");
      break;
    case L'+':
      fprintf (f, "\\gredagger{}");
      break;
    case L'_':
      fprintf (f, "\\_{}");
      break;
    case L'-':
      fprintf (f, "\\grehyph{}");
      break;
    default:
      gregorio_write_one_tex_char (f, to_print);
      break;
    }
  return;
}

void
gregoriotex_write_text (FILE * f, gregorio_character * text,
			char *first_syllable)
{
  if (text == NULL)
    {
      fprintf (f, "{}{}{}");
      return;
    }
  fprintf (f, "{");
  gregorio_write_text (*first_syllable, text, f,
		       (&gtex_write_verb),
		       (&gtex_print_char),
		       (&gtex_write_begin),
		       (&gtex_write_end), (&gtex_write_special_char));
  if (*first_syllable)
    {
      *first_syllable = 0;
    }
  fprintf (f, "}");
}

/*
 * @brief Writes the translation.
 * 
 * There is no special handling of translation text; that is, we just print the entire string of text
 * under the normal text line, without considering any special centering or linebreaks. 
 */
void
gregoriotex_write_translation (FILE * f, gregorio_character * translation)
{
  if (translation == NULL)
    {
      return;
    }
  gregorio_write_text (0, translation, f,
		       (&gtex_write_verb),
		       (&gtex_print_char),
		       (&gtex_write_begin),
		       (&gtex_write_end), (&gtex_write_special_char));
}

// a function to write only the two first syllables of the next syllables
void
gregoriotex_write_next_first_text (FILE * f,
				   gregorio_character * current_character)
{
  gregorio_character *first_character = current_character;
  gregorio_character *next_character;
  if (current_character == NULL)
    {
      fprintf (f, "{}{}");
      return;
    }
  // big dirty hack to have only the first two syllables printed : we cut the thing just after the ST_CENTER closing
  while (current_character)
    {
      if (current_character->is_character == 0
	  && (current_character->cos.s.style == ST_CENTER
	      || current_character->cos.s.style == ST_FORCED_CENTER)
	  && current_character->cos.s.type == ST_T_END)
	{
	  next_character = current_character->next_character;
	  current_character->next_character = NULL;

	  fprintf (f, "{");
	  gregorio_write_text (0, first_character, f,
			       (&gtex_write_verb),
			       (&gtex_print_char),
			       (&gtex_write_begin),
			       (&gtex_write_end_for_two),
			       (&gtex_write_special_char));
	  current_character->next_character = next_character;
	  return;
	}
      else
	{
	  current_character = current_character->next_character;
	}
    }
}

// here we absolutely need to pass the syllable as an argument, because we will need the next note, that may be contained in the next syllable

void
gregoriotex_write_element (FILE * f,
			   gregorio_syllable * syllable,
			   gregorio_element * element)
{
  gregorio_glyph *current_glyph = element->first_glyph;
  while (current_glyph)
    {
      if (current_glyph->type == GRE_SPACE)
	{
// we assume here that it is a SP_ZERO_WIDTH, the only one a glyph can be
	  fprintf (f, "\\greendofglyph{1}%%\n");
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_TEXVERB_GLYPH && current_glyph->texverb)
	{
	  fprintf (f,
		   "%% verbatim text at glyph level:\n%s%%\n%% end of verbatim text\n",
		   current_glyph->texverb);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_FLAT)
	{
	  fprintf (f, "\\greflat{%c}{0}%%\n", current_glyph->glyph_type);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_NATURAL)
	{
	  fprintf (f, "\\grenatural{%c}{0}%%\n", current_glyph->glyph_type);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_SHARP)
	{
	  fprintf (f, "\\gresharp{%c}{0}%%\n", current_glyph->glyph_type);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_BAR)
	{
	  gregoriotex_write_bar (f, current_glyph->glyph_type,
				 current_glyph->liquescentia, INSIDE_BAR);
	  current_glyph = current_glyph->next;
	  continue;
	}
      if (current_glyph->type == GRE_TEXVERB_GLYPH && current_glyph->texverb)
	{
	  fprintf (f, "%s", current_glyph->texverb);
	  current_glyph = current_glyph->next;
	  continue;
	}
// at this point glyph->type is GRE_GLYPH
      gregoriotex_write_glyph (f, syllable, element, current_glyph);
      if (current_glyph->next && current_glyph->next->type == GRE_GLYPH)
	{
	  if (is_puncta_inclinata (current_glyph->next->glyph_type)
	      || current_glyph->next->glyph_type == G_PUNCTA_INCLINATA)
	    {
	      fprintf (f, "\\greendofglyph{9}%%\n");
	    }
	  else
	    {
	      fprintf (f, "\\greendofglyph{0}%%\n");
	    }
	}
      current_glyph = current_glyph->next;
    }
}

// a function to compute the height of the flat of a key
// the flat is always on the line of the 

char
gregoriotex_clef_flat_height (char step, int line)
{
  switch (step)
    {
    case C_KEY:
      switch (line)
	{
	case 1:
	  return 'c';
	  break;
	case 2:
	  return 'e';
	  break;
	case 3:
	  return 'g';
	  break;
	case 4:
	  return 'i';
	  break;
	default:
	  gregorio_message (_("unknown line number"),
			    "gregoriotex_clef_flat_height", ERROR, 0);
	  return 'g';
	  break;
	}
      break;
    case F_KEY:
      switch (line)
	{
	case 1:
	  return 'g';
	  break;
	case 2:
	  return 'i';
	  break;
	case 3:
	  return 'd';
	  break;
	case 4:
	  return 'f';
	  break;
	default:
	  gregorio_message (_("unknown line number"),
			    "gregoriotex_clef_flat_height", ERROR, 0);
	  return 'g';
	  break;
	}
      break;
    default:
      gregorio_message (_("unknown clef type"),
			"gregoriotex_clef_flat_height", ERROR, 0);
      return 'g';
      break;
    }
}


void
gregoriotex_write_bar (FILE * f, char type, char signs, char inorsyllable)
{
  // the type number of function vepisemusorrare
  char typenumber = 26;
  if (inorsyllable == INSIDE_BAR)
    {
      fprintf (f, "\\grein");
    }
  else
    {
      fprintf (f, "\\gre");
    }
  switch (type)
    {
    case B_VIRGULA:
      fprintf (f, "virgula");
      typenumber = 26;
      break;
    case B_DIVISIO_MINIMA:
      fprintf (f, "divisiominima");
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR:
      fprintf (f, "divisiominor");
      typenumber = 25;
      break;
    case B_DIVISIO_MAIOR:
      fprintf (f, "divisiomaior");
      typenumber = 25;
      break;
    case B_DIVISIO_FINALIS:
      fprintf (f, "divisiofinalis");
      typenumber = 27;
      break;
    case B_DIVISIO_MINOR_D1:
      fprintf (f, "dominica{1}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR_D2:
      fprintf (f, "dominica{2}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR_D3:
      fprintf (f, "dominica{3}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR_D4:
      fprintf (f, "dominica{4}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR_D5:
      fprintf (f, "dominica{5}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    case B_DIVISIO_MINOR_D6:
      fprintf (f, "dominica{6}");
      // TODO: a true typenumber
      typenumber = 25;
      break;
    default:
      gregorio_message (_("unknown bar type"),
			"gregoriotex_write_bar", ERROR, 0);
      break;
    }
  switch (signs)
    {
    case _V_EPISEMUS:
      fprintf (f, "{\\grebarvepisemus{%d}}%%\n", typenumber);
      break;
    case _ICTUS_A:
      fprintf (f, "{\\greictusa{%d}}%%\n", typenumber);
      break;
    case _ICTUS_T:
      fprintf (f, "{\\greictust{%d}}%%\n", typenumber);
      break;
    case _V_EPISEMUS_ICTUS_A:
      fprintf (f, "{\\grebarvepisemusictusa{%d}}%%\n", typenumber);
      break;
    case _V_EPISEMUS_ICTUS_T:
      fprintf (f, "{\\grebarvepisemusictust{%d}}%%\n", typenumber);
      break;
    case _H_EPISEMUS:
      fprintf (f, "{\\grebarbrace{%d}}%%\n", typenumber);
      break;
    case _V_EPISEMUS_H_EPISEMUS:
      fprintf (f, "{\\grebarbrace{%d}\\grebarvepisemus{%d}}%%\n", typenumber, typenumber);
      break;
    case _H_EPISEMUS_ICTUS_A:
      fprintf (f, "{\\grebarbrace{%d}\\greictusa{%d}}%%\n", typenumber, typenumber);
      break;
    case _H_EPISEMUS_ICTUS_T:
      fprintf (f, "{\\grebarbrace{%d}\\greictust{%d}}%%\n", typenumber, typenumber);
      break;
    case _V_EPISEMUS_H_EPISEMUS_ICTUS_A:
      fprintf (f, "{\\grebarbrace{%d}\\grebarvepisemusictusa{%d}}%%\n", typenumber, typenumber);
      break;
    case _V_EPISEMUS_H_EPISEMUS_ICTUS_T:
      fprintf (f, "{\\grebarbrace{%d}\\grebarvepisemusictust{%d}}%%\n", typenumber, typenumber);
      break;
    default:
      fprintf (f, "{}%%\n", typenumber);
      break;
    }
}

void
gregoriotex_write_glyph (FILE * f,
			 gregorio_syllable * syllable,
			 gregorio_element * element, gregorio_glyph * glyph)
{
  unsigned int glyph_number = 0;
// glyph number is the number of the glyph in the fonte, it is discussed in later comments
// type is the type of the glyph. Understand the type of the glyph for gregoriotex, for the alignement between text and notes. (AT_ONE_NOTE, etc.)
  int type = 0;
  // the type of the glyph, in the sense of the shape (T_PES, etc.)
  char gtype = 0;
  char next_note_pitch = 0;
  gregorio_note *current_note;
  if (!glyph)
    {
      gregorio_message (_
			("called with NULL pointer"),
			"gregoriotex_write_glyph", ERROR, 0);
      return;
    }
  if (!glyph->first_note)
    {
      gregorio_message (_
			("called with glyph without note"),
			"gregoriotex_write_glyph", ERROR, 0);
      return;
    }
  next_note_pitch = gregorio_determine_next_pitch (syllable, element, glyph);
  current_note = glyph->first_note;
// first we check if it is really a unique glyph in gregoriotex... the glyphs that are not a unique glyph are : trigonus and pucta inclinata in general, and torculus resupinus and torculus resupinus flexus, so we first divide the glyph into real gregoriotex glyphs
  switch (glyph->glyph_type)
    {
    case G_TRIGONUS:
    case G_PUNCTA_INCLINATA:
    case G_2_PUNCTA_INCLINATA_DESCENDENS:
    case G_3_PUNCTA_INCLINATA_DESCENDENS:
    case G_4_PUNCTA_INCLINATA_DESCENDENS:
    case G_5_PUNCTA_INCLINATA_DESCENDENS:
    case G_2_PUNCTA_INCLINATA_ASCENDENS:
    case G_3_PUNCTA_INCLINATA_ASCENDENS:
    case G_4_PUNCTA_INCLINATA_ASCENDENS:
    case G_5_PUNCTA_INCLINATA_ASCENDENS:
      while (current_note)
	{
	  gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
	  gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element, syllable,
				   current_note);
	  current_note = current_note->next;
	}
      break;
    case G_SCANDICUS:
    case G_ANCUS:
      if (glyph->liquescentia == L_DEMINUTUS
	  || glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS)
	{
	  gregoriotex_determine_number_and_type (glyph, element, &type,
						 &gtype, &glyph_number);
	  fprintf (f, "\\greglyph{\\char %d}{%c}{%c}{%d}", glyph_number,
		   glyph->first_note->pitch, next_note_pitch, type);
	  gregoriotex_write_signs (f, gtype, glyph, element, syllable,
				   glyph->first_note);
	}
      else
	{
	  while (current_note)
	    {
	      gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
	      gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element,
				       syllable, current_note);
	      current_note = current_note->next;
	    }
	}
      break;
    case G_TORCULUS_RESUPINUS_FLEXUS:
      gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
      gregoriotex_write_signs (f, T_ONE_NOTE_TRF, glyph, element, syllable,
			       glyph->first_note);
      glyph->glyph_type = G_PORRECTUS_FLEXUS_NO_BAR;
      // tricky to have the good position for these glyphs
      glyph->first_note = current_note->next;
      gregoriotex_determine_number_and_type (glyph, element, &type,
					     &gtype, &glyph_number);
//TODO : fusion functions
      fprintf (f, "\\greglyph{\\char %d}{%c}{%c}{%d}", glyph_number,
	       glyph->first_note->pitch, next_note_pitch, type);
      gregoriotex_write_signs (f, gtype, glyph, element, syllable,
			       glyph->first_note);
      glyph->first_note = current_note;
      glyph->glyph_type = G_TORCULUS_RESUPINUS_FLEXUS;
      break;
    case G_BIVIRGA:
    case G_TRIVIRGA:
      while (current_note)
	{
	  gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
	  gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element, syllable,
				   current_note);
	  current_note = current_note->next;
	  if (current_note)
	    {
	      fprintf (f, "\\greendofglyph{4}%%\n");
	    }
	}
      break;
    case G_TRISTROPHA:
    case G_TRISTROPHA_AUCTA:
    case G_DISTROPHA:
    case G_DISTROPHA_AUCTA:
      while (current_note)
	{
	  gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
	  gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element, syllable,
				   current_note);
	  current_note = current_note->next;
	  if (current_note)
	    {
	      fprintf (f, "\\greendofglyph{5}%%\n");
	    }
	}
      break;
    case G_PUNCTUM:
      if (glyph->first_note->shape != S_ORISCUS
        && glyph->first_note->shape != S_ORISCUS_AUCTUS
        && glyph->first_note->shape != S_ORISCUS_DEMINUTUS)
      {
      switch (glyph->liquescentia)
	{
	case L_AUCTUS_ASCENDENS:
	  glyph->first_note->shape = S_PUNCTUM_AUCTUS_ASCENDENS;
	  break;
	case L_AUCTUS_DESCENDENS:
	case L_AUCTA:
	  glyph->first_note->shape = S_PUNCTUM_AUCTUS_DESCENDENS;
	  break;
	case L_DEMINUTUS:
	case L_INITIO_DEBILIS:
	  glyph->first_note->shape = S_PUNCTUM_DEMINUTUS;
	default:
	  break;
	}
	}
    case G_PUNCTUM_INCLINATUM:
    case G_VIRGA:
    case G_STROPHA:
    case G_STROPHA_AUCTA:
      gregoriotex_write_note (f, glyph->first_note, glyph, element, next_note_pitch);
      gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element, syllable,
			       current_note);
      break;
    default:
      // special case of the torculus resupinus which first note is not a punctum
      if (glyph->glyph_type == G_TORCULUS_RESUPINUS
	  && current_note->shape != S_PUNCTUM)
	{
	  gregoriotex_write_note (f, current_note, glyph, element, next_note_pitch);
	  gregoriotex_write_signs (f, T_ONE_NOTE, glyph, element, syllable,
				   glyph->first_note);
	  // tricky to have the good position for these glyphs
	  glyph->first_note = current_note->next;
	  glyph->glyph_type = G_PORRECTUS_NO_BAR;
	  gregoriotex_determine_number_and_type (glyph, element, &type,
						 &gtype, &glyph_number);
//TODO : fusion functions
	  fprintf (f, "\\greglyph{\\char %d}{%c}{%c}{%d}", glyph_number,
		   glyph->first_note->pitch, next_note_pitch, type);
	  gregoriotex_write_signs (f, gtype, glyph, element, syllable,
				   glyph->first_note);
	  glyph->glyph_type = G_TORCULUS_RESUPINUS;
	  glyph->first_note = current_note;
	}
      else
	{
	  gregoriotex_determine_number_and_type (glyph, element, &type,
						 &gtype, &glyph_number);
	  fprintf (f, "\\greglyph{\\char %d}{%c}{%c}{%d}", glyph_number,
		   glyph->first_note->pitch, next_note_pitch, type);
	  gregoriotex_write_signs (f, gtype, glyph, element, syllable,
				   glyph->first_note);
	  break;
	}
    }
}

/*

A function that write the signs of a glyph, which has the type type (T_*, not G_*, which is in the glyph->glyph_type), and (important), we start only at the note current_note. It is due to the way we call it : if type is T_ONE_NOTE, we just do the signs on current_note, not all. This is the case for example for the first note of the torculus resupinus, or the G_*_PUNCTA_INCLINATA.

*/

//small helper
#define _found()\
	  if (found == 0)\
	    {\
	      found = 1;\
	      fprintf (f, "%%\n");\
	    }

#define _end_loop()\
      if (type == T_ONE_NOTE || type == T_ONE_NOTE_TRF)\
	{\
	  break;\
	}\
      else\
	{\
	  current_note = current_note->next;\
	  i++;\
	}

void
gregoriotex_write_signs (FILE * f, char type,
			 gregorio_glyph * glyph,
			 gregorio_element * element,
			 gregorio_syllable * syllable,
			 gregorio_note * note)
{
  // i is the number of the note for which we are typesetting the sign.
  int i = 1;
  // a dumb char
  char block_hepisemus = 0;
  unsigned char found = 0;
  gregorio_note *current_note = note;
  while (current_note)
    {
      // we start by the additional lines
      if (current_note->pitch < 'c')
	{
	  	  if (found == 0)
	    {
	      found = 1;
	      fprintf (f, "%%\n{%%\n");
	    }
	  gregoriotex_write_additional_line (f, glyph, i, type,
					     TT_BOTTOM, current_note);
		status -> bottom_line = 1;
	}
      if (current_note->pitch > 'k')
	{
	  	  if (found == 0)
	    {
	      found = 1;
	      fprintf (f, "%%\n{%%\n");
	    }
	  gregoriotex_write_additional_line (f, glyph, i, type,
					     TT_TOP, current_note);
	}
     if (current_note->texverb)
    {
	  	  if (found == 0)
	    {
	      found = 1;
	      fprintf (f, "%%\n{%%\n");
	    }
          fprintf (f,
             "%% verbatim text at note level:\n%s%%\n%% end of verbatim text\n",
             current_note->texverb);
    }
	  _end_loop();
	}
	  if (found == 0)
  {
    fprintf (f, "{}{");
  }
    else
  {
	  fprintf (f, "}{");
	}
	found = 0;
	i=1;
	current_note = note;
	// now a first loop for the choral signs, because high signs must be taken into account before any hepisemus
	while (current_note)
	  {
      if (current_note->choral_sign)
        {
	        _found();
	        gregoriotex_write_choral_sign(f, glyph, type, i, current_note, 0);
        }
     _end_loop();
	  }
	// a loop for rare signs, vertical episemus, horizontal episemus and ictus
	i=1;
	current_note = note;
	while (current_note)
	 {
      // we continue with the hepisemus
      if (current_note->h_episemus_type != H_NO_EPISEMUS && block_hepisemus == 0)
	{
	  // if it is a porrectus or a porrectus flexus, we check if the episemus is on the two first notes:
	  _found();
	  if ((type == T_PORRECTUS || type == T_PORRECTUSFLEXUS
	       || type == T_PORRECTUSFLEXUS_NOBAR
	       || type == T_PORRECTUS_NOBAR) && current_note->next
	      && simple_htype (current_note->next->h_episemus_type) !=
	      H_NO_EPISEMUS && i == 1)
	    {
	      gregoriotex_write_hepisemus (f, glyph, element, syllable,
					   HEPISEMUS_FIRST_TWO,
					   type, current_note);
	      block_hepisemus = 1;
	    }
	  else
	    {
	      if (type == T_TORCULUS_RESUPINUS && current_note->next
		  && simple_htype (current_note->next->h_episemus_type) !=
		  H_NO_EPISEMUS && i == 2)
		{
		  gregoriotex_write_hepisemus (f, glyph, element, syllable,
					       HEPISEMUS_FIRST_TWO,
					       type, current_note);
		  block_hepisemus = 1;
		}
	      else
		{
		  gregoriotex_write_hepisemus (f, glyph, element, syllable, i,
					       type, current_note);
		}
	    }
	}
      // write_rare also writes the vepisemus
      if (current_note->rare_sign)
	{
	  _found();
	  gregoriotex_write_rare (f, glyph, i, type,
				  current_note, current_note->rare_sign);
	}
	  if (current_note->signs != _NO_SIGN)
	    {
	      _found();
	    }
      switch (current_note->signs)
	{
	case _V_EPISEMUS:
	case _V_EPISEMUS_PUNCTUM_MORA:
	case _V_EPISEMUS_AUCTUM_DUPLEX:
	  if (current_note->rare_sign != _ICTUS_A
	      && current_note->rare_sign != _ICTUS_T)
	    {
	      gregoriotex_write_vepisemus (f, glyph, i, type, current_note);
	    }
	  break;
	default:
	  break;
	}
      // why is this if there?...
      if (!current_note->rare_sign)
	{
	  if (block_hepisemus == 2)
	    {
	      block_hepisemus = 0;
	    }
	  if (block_hepisemus == 1)
	    {
	      block_hepisemus = 2;
	    }
	}
    _end_loop()
  // final loop for choral signs and punctum mora
    }
  i = 1;
  current_note = note;
  while (current_note)
    {
      switch (current_note->signs)
	{
	case _PUNCTUM_MORA:
	case _V_EPISEMUS_PUNCTUM_MORA:
	  gregoriotex_write_punctum_mora (f, glyph, type, current_note);
	  break;
	case _AUCTUM_DUPLEX:
	case _V_EPISEMUS_AUCTUM_DUPLEX:
	  gregoriotex_write_auctum_duplex (f, glyph, current_note);
	  break;
	default:
	  break;
	}
	  if (current_note->choral_sign)
	    {
	      _found();
        gregoriotex_write_choral_sign(f, glyph, type, i, current_note, 1);
	    }
	  _end_loop();
    } 
  fprintf(f, "}%%\n");
}

void
gregoriotex_write_choral_sign (FILE * f,
				 gregorio_glyph * glyph, char type, int i,
				 gregorio_note * current_note, char low)
{
  // 0 in the normal case (sign above the note), 1 in the case of it's next to the note (same height as a punctum)
  unsigned char low_sign = 0;
  char bottom = 0;
  char number = 0;
  char height = 0;
  // a temp value
  char kind_of_pes = 0;
  gregorio_note *tmpnote=NULL;
  
    switch (glyph->glyph_type)
	{
	case G_FLEXUS:
	case G_TORCULUS:
	case G_TORCULUS_RESUPINUS_FLEXUS:
	case G_PORRECTUS_FLEXUS:
	  if (!current_note->next)
	    {
	      low_sign = 1;
	    }
	  break;
	case G_PES:
	case G_PORRECTUS:
	case G_TORCULUS_RESUPINUS:
	  if (!current_note->next)
	    {
        break;
	    }
	  kind_of_pes = 1;
	  if (current_note->shape != S_QUILISMA)
	    {
	      low_sign = 1;
	    }
	  break;
	default:
	  break;
	}

  // the low choral signs must be typeset after the punctum, whereas the high must be typeset before the h episemus
  if ((low_sign == 1 && low == 0) || (low_sign == 0 && low == 1))
    {
      return;
    }

  if (low_sign ==  0)
    {
      // let's cheat a little
      current_note -> h_episemus_top_note = current_note->pitch;
      gregoriotex_find_sign_number (glyph, i,
				type, TT_H_EPISEMUS, current_note,
				&number, &height, &bottom);
		  if (is_on_a_line(current_note->pitch))
		    {
		      fprintf(f, "\\grehighchoralsign{%c}{%s}{%d}%%\n", current_note->pitch, current_note->choral_sign, number);
		    }
		  else
		    {
		      fprintf(f, "\\grehighchoralsign{%c}{%s}{%d}%%\n", current_note->pitch+2, current_note->choral_sign, number);
		    }
		  if (simple_htype(current_note->h_episemus_type) != H_NO_EPISEMUS)
		    {
		      tmpnote = current_note;
          while(tmpnote)
            {
              if (simple_htype(current_note->h_episemus_type) != H_NO_EPISEMUS)
                {
                  if (is_on_a_line(tmpnote->h_episemus_top_note))
                    {
                      tmpnote->h_episemus_top_note = tmpnote-> h_episemus_top_note + 1;
                    }
                  else
                    {
                      tmpnote->h_episemus_top_note = tmpnote-> h_episemus_top_note + 2;
                    }
                }
              tmpnote = tmpnote -> next;
            }
          tmpnote = current_note -> previous;
          while(tmpnote)
            {
              if (simple_htype(current_note->h_episemus_type) != H_NO_EPISEMUS)
                {
                  if (is_on_a_line(tmpnote->h_episemus_top_note))
                    {
                      tmpnote->h_episemus_top_note = tmpnote-> h_episemus_top_note + 1;
                    }
                  else
                    {
                      tmpnote->h_episemus_top_note = tmpnote-> h_episemus_top_note + 2;
                    }
                }
              tmpnote = tmpnote -> previous;
            }
		    }
    }
  else
    {
      // very approximative euristic, some things may have to be adapted here...
      if (is_on_a_line(current_note->pitch))
        {
          if (kind_of_pes = 1 && current_note->pitch - current_note->next->pitch == -1)
            {
              fprintf(f, "\\grelowchoralsign{%c}{%s}{1}%%\n", current_note->pitch, current_note->choral_sign);
              return;
            }
          if (current_note->previous && (current_note->previous->signs == _PUNCTUM_MORA || current_note->previous->signs == _V_EPISEMUS_PUNCTUM_MORA))
            {
              fprintf(f, "\\grelowchoralsign{%c}{%s}{1}%%\n", current_note->pitch, current_note->choral_sign);
              return;
            }
        }
      else
        {
          fprintf(f, "\\grelowchoralsign{%c}{%s}{0}%%\n", current_note->pitch, current_note->choral_sign);
        }
    }
}

/*!
 * @brief Writes augmentum duplexes (double dots)
 *
 * We suppose we are on the last note. \n
 * The algorithm is the following: if there is a previous note, we consider that the two puncta of the augumentum duplex must correspond to the last note and the previous note.
 * If we are adding to a single note glyph, which would be weird but sure why not, we just typeset two puncta spaced of 2.
 */
void
gregoriotex_write_auctum_duplex (FILE * f,
				 gregorio_glyph * glyph,
				 gregorio_note * current_note)
{
  char pitch = current_note->pitch;
  char previous_pitch = 0;
// second_pitch is the second argument of the \augmentumduplex macro, that's what this function is all about.
  char second_pitch = 0;
  // this variable will be set to 1 if we are on the note before the last note of a podatus or a porrectus or a torculus resupinus
  unsigned char special_punctum = 0;
  if (current_note->previous)
    {
      if (current_note->previous->pitch - current_note->pitch == -1
	  || current_note->previous->pitch - current_note->pitch == 1)
	{
	  special_punctum = 1;
	}
      second_pitch = current_note->previous->pitch;
      previous_pitch = second_pitch;
    }

  if (!previous_pitch || previous_pitch == pitch)
    {
      if (is_on_a_line (pitch))
	{
	  second_pitch = pitch;
	  special_punctum = 1;
	}
      else
	{
	  second_pitch = pitch + 1;
	}
    }
  // the first argument should always be the lowest one, that's what we do here:
  if (pitch > second_pitch)
    {
      previous_pitch = pitch;
      pitch = second_pitch;
      second_pitch = previous_pitch;
    }
  // maybe the third argument should be changed
  fprintf (f, "\\greaugmentumduplex{%c}{%c}{%d}%%\n", pitch, second_pitch,
	   special_punctum);
}

/**
 * @brief Adds a dot.
 *
 * Writes \c \\grepunctummora in the gtex file, with the appropriate arguments. You might think this function
 * more straightforward than it actually is...
 */
void
gregoriotex_write_punctum_mora (FILE * f,
				gregorio_glyph * glyph,
				char type, gregorio_note * current_note)
{
  // in this if we consider that the puncta are only on the last two notes (maybe it would be useful to consider it more entirely, but it would be really weird...)
  // the variable that will be set to 1 if we have to shift the punctum inclinatum before the last note
  unsigned char shift_before = 0;
  // this variable will be set to 1 if we are on the note before the last note of a podatus or a porrectus or a torculus resupinus
  unsigned char special_punctum = 0;
  // 0 if space is normal, 1 if there should be no space after a punctum
  unsigned char no_space = 0;
  // the pitch where to set the punctum
  char pitch = current_note->pitch;
  // a variable to know if we are on a punctum inclinatum or not
  unsigned char punctum_inclinatum = 0;
  // a temp variable
  gregorio_note *tmpnote;
  // first: the very special case where type == T_ONE_NOTE_TRF, the punctum is at a strange place:
  if (type == T_ONE_NOTE_TRF)
    {
      fprintf (f, "\\grepunctummora{%c}{1}{0}{0}%%\n", current_note->pitch);
    }
  // we go into this switch only if it is the note before the last note
  if (current_note->next)
    {
      switch (glyph->glyph_type)
	{
	case G_FLEXUS:
	case G_TORCULUS:
	case G_TORCULUS_RESUPINUS_FLEXUS:
	case G_PORRECTUS_FLEXUS:
	  if (glyph->liquescentia != L_DEMINUTUS
	      && glyph->liquescentia != L_DEMINUTUS_INITIO_DEBILIS)
	    {
	      shift_before = 1;
	    }
	  break;
	case G_PES:
	  if ((current_note->shape != S_PUNCTUM
	       && current_note->shape != S_QUILISMA)
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS
	      || glyph->liquescentia == L_AUCTUS_ASCENDENS
	      || glyph->liquescentia == L_AUCTUS_ASCENDENS_INITIO_DEBILIS
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS_INITIO_DEBILIS)
	    {
	      shift_before = 1;
	      // fine tuning
	      if (current_note->next->pitch - current_note->pitch == 1)
		{
		  if (is_on_a_line (current_note->pitch))
		    {
		      special_punctum = 1;
		    }
		  else
		    {
		      pitch = current_note->pitch - 1;
		    }
		}
	    }
	  else
	    {
	      // case for f.g
	      if (current_note->next->pitch - current_note->pitch == 1)
		{
		  special_punctum = 1;
		}
	    }
	  break;
	case G_PES_QUADRATUM:
	  shift_before = 1;
	  if (current_note->next->pitch - current_note->pitch == 1)
	    {
	      if (is_on_a_line (current_note->pitch))
		{
		  special_punctum = 1;
		}
	      else
		{
		  pitch = current_note->pitch - 1;
		}
	    }
	  break;
	case G_PORRECTUS:
	case G_TORCULUS_RESUPINUS:
	  // this case is only for the note before the previous note
	  if ((current_note->next->pitch - current_note->pitch == -1
	       || current_note->next->pitch - current_note->pitch == 1)
	      && !(current_note->next->next))
	    special_punctum = 1;
	  break;
	default:
	  break;
	}
    }
  // we enter here in any case
  switch (glyph->glyph_type)
    {
        case G_TRIGONUS:
        case G_PUNCTA_INCLINATA:
        case G_2_PUNCTA_INCLINATA_DESCENDENS:
        case G_3_PUNCTA_INCLINATA_DESCENDENS:
        case G_4_PUNCTA_INCLINATA_DESCENDENS:
        case G_5_PUNCTA_INCLINATA_DESCENDENS:
          if (! current_note->next)
            {
              special_punctum = 1;
            }
          break;
        default:
          break;
    }
  if (current_note -> shape == S_PUNCTUM_INCLINATUM)
    {
      punctum_inclinatum = 1;
    }
  if (current_note -> shape == S_PUNCTUM_INCLINATUM_DEMINUTUS)
    {
      punctum_inclinatum = 1;
    }
    
//when the punctum mora is on a note on a line, and the prior note is on the space immediately above, the dot is placed on the space below the line instead
  if (current_note->previous
      && (current_note->previous->pitch - current_note->pitch == 1)
      && is_on_a_line (current_note->pitch)
      && (current_note->previous->signs == _PUNCTUM_MORA
	    || current_note->previous->signs == _V_EPISEMUS_PUNCTUM_MORA
	    || current_note -> previous-> choral_sign))
    {
      special_punctum = 1;
    }
  
  if (shift_before == 1)
    {
      if (current_note->next->pitch - current_note->pitch == -1
	  || current_note->next->pitch - current_note->pitch == 1)
	{
	  fprintf (f, "\\grepunctummora{%c}{3}{%d}{%d}%%\n", pitch,
		   special_punctum, punctum_inclinatum);
	}
      else
	{
	  fprintf (f, "\\grepunctummora{%c}{2}{%d}{%d}%%\n", pitch,
		   special_punctum, punctum_inclinatum);
	}
      return;
    }

// There are two special cases. The first: if the next glyph is a ZERO_WIDTH_SPACE, and the current glyph is a PES, and the punctum mora is on the first note, and the first note of the next glyph is at least two (or three depending on something) pitches higher than the current note. You'll all have understood, this case is quite rare... but when it appears, we pass 1 as a second argument of \punctummora so that it removes the space introduced by the punctummora.
  if (glyph->glyph_type == G_PODATUS && glyph->next
      && glyph->next->type == GRE_SPACE
      && glyph->next->glyph_type == SP_ZERO_WIDTH
      && current_note->next && glyph->next->next
      && glyph->next->next->type == GRE_GLYPH
      && glyph->next->next->first_note
      && (glyph->next->next->first_note->pitch - current_note->pitch > 1))
    {
      fprintf (f, "\\grepunctummora{%c}{1}{%d}{%d}%%\n", pitch, special_punctum, punctum_inclinatum);
      return;
    }
  // if there is a punctum or a auctum dumplex on a note after, we put a zero-width punctum
  tmpnote = current_note->next;
  while (tmpnote)
    {
      if (tmpnote->signs == _PUNCTUM_MORA || tmpnote->signs == _AUCTUM_DUPLEX
	  || tmpnote->signs == _V_EPISEMUS_PUNCTUM_MORA
	  || tmpnote->signs == _V_EPISEMUS_AUCTUM_DUPLEX || tmpnote -> choral_sign)
	{
	  no_space = 1;
	  break;
	}
      tmpnote = tmpnote->next;
    }

// the normal operation
  fprintf (f, "\\grepunctummora{%c}{%d}{%d}{%d}%%\n", pitch, no_space,
	   special_punctum, punctum_inclinatum);
}

/**
 * @brief A function that writes the good \c \\hepisemus in GregorioTeX. 
 * @param i The position of the note in the glyph.
 */
void
gregoriotex_write_hepisemus (FILE * f,
			     gregorio_glyph *
			     current_glyph,
			     gregorio_element * current_element,
			     gregorio_syllable * current_syllable,
			     int i, char type, gregorio_note * current_note)
{

  char height = 0;
  char no_bridge_height = 0;
  char number = 0;
  char ambitus = 0;
  char bottom = 0;
  char next_height = -1;
  // a helper value containing 0 if we must not change the height (case of the height already modified by a previous hepisemus bridge
  char do_not_change_height = 0;
  gregorio_note *next_note = NULL;

  if (!current_note || current_note->h_episemus_type == H_NO_EPISEMUS)
    {
      return;
    }

  gregoriotex_find_sign_number (current_glyph, i,
				type, TT_H_EPISEMUS, current_note,
				&number, &height, &bottom);

  no_bridge_height = height;

  if (status->to_modify_note && status->to_modify_note == current_note)
    {
      do_not_change_height = 1;
      height = status->to_modify_h_episemus;
      // we also modify the next note if necessary
      if (current_note->next && simple_htype(current_note->next->h_episemus_type) != H_NO_EPISEMUS)
        {
          status->to_modify_note = current_note->next;
        }
      else
        {
          status->to_modify_note = NULL;
        }
    }

  next_height = gregoriotex_find_next_hepisemus_height (current_glyph, current_note, current_element, &next_note);

  if (simple_htype(current_note->h_episemus_type) != H_NO_EPISEMUS && (!current_note->next 
  || current_note->next->shape == S_PUNCTUM_INCLINATUM || current_note->next->shape == S_PUNCTUM_INCLINATUM_DEMINUTUS || current_note->next->shape == S_PUNCTUM_INCLINATUM_AUCTUS) && (!current_note->previous || simple_htype(current_note->previous->h_episemus_type) == H_NO_EPISEMUS) && bottom == 0 && next_height != -1)
    {
      if (height == next_height || do_not_change_height == 0 && (height == next_height -1 && is_on_a_line(height)))
        {
          fprintf (f, "\\grehepisemusbridge{%c}{}{}%%\n", next_height);
        }
      else
        {
          if (height == next_height + 1 && is_on_a_line(next_height))
            {
              status->to_modify_h_episemus = height;
              status->to_modify_note = next_note;
              fprintf (f, "\\grehepisemusbridge{%c}{}{}%%\n", height);
            }
          next_height = height;
        }
    }
  else
    {
      next_height = height;
    }

  if (current_note->next)
    {
      ambitus = current_note->pitch - current_note->next->pitch;
    }
  if (has_bottom (current_note->h_episemus_type))
    {
      fprintf (f, "\\grehepisemusbottom{%c}{%d}{%d}%%\n",
	       current_note->pitch - 1, number, ambitus);
      if (bottom != 1
	  && simple_htype (current_note->h_episemus_type) != H_NO_EPISEMUS)
	{
	  fprintf (f, "\\grehepisemus{%c}{%d}{%d}{%c}%%\n", no_bridge_height, number,
		   ambitus, next_height);
	}
      return;
    }
  if (bottom == 1)
    {
      fprintf (f, "\\grehepisemusbottom{%c}{%d}{%d}%%\n", no_bridge_height, number,
	       ambitus);
    }
  else
    {
      fprintf (f, "\\grehepisemus{%c}{%d}{%d}{%c}%%\n", no_bridge_height, number, ambitus, next_height);
    }
}

// a function to find the next horizontal episemus height (returns -1 if none interesting)

char
gregoriotex_find_next_hepisemus_height (gregorio_glyph *glyph, gregorio_note *note,
   gregorio_element *element, gregorio_note **final_note)
{
  char  i = 1;
  char height = 0;
  char number = 0;
  char bottom = 0;
  int type = 0;
  char gtype = 0;
  unsigned int glyph_number = 0;
  
  if (note && note->shape == S_PUNCTUM_INCLINATUM && note->next && 
   (note->next-> shape == S_PUNCTUM_INCLINATUM
    || note->next-> shape == S_PUNCTUM_INCLINATUM_DEMINUTUS
    || note->next-> shape == S_PUNCTUM_INCLINATUM_AUCTUS)   
    && simple_htype(note->next->h_episemus_type) != H_NO_EPISEMUS)
    {
       return note->next->h_episemus_top_note + 1;
    }
  note = NULL;
  if(glyph->next && glyph->next->type == GRE_GLYPH || glyph->next && glyph->next->next && glyph->next->next->type == GRE_GLYPH)
    {
      if (glyph->next->type != GRE_GLYPH)
        {
          glyph = glyph->next;
        }
      glyph = glyph->next;
      note = glyph->first_note;
      if (glyph->glyph_type == G_PES)
        {
          note = note->next;
          i = 2;
          gtype = T_PES;
        }
      else
        {
          gregoriotex_determine_number_and_type (glyph, element, &type,
						 &gtype, &glyph_number);
        }
      if (simple_htype(note->h_episemus_type) != H_NO_EPISEMUS)
        { 
          gregoriotex_find_sign_number (glyph, i,
				        gtype, TT_H_EPISEMUS, note,
				        &number, &height, &bottom);
				  if (bottom == 0)
				    {
				      *final_note = note;
              return height;
            }
          else
            {
              return -1;
            }
        }
      else
        {
          return -1;
        }
    }
  element = element->next;
  if (!element)
    {
      return -1;
    }
  if (element->type == GRE_SPACE && (element->element_type == SP_NEUMATIC_CUT || element->element_type == SP_LARGER_SPACE || element->element_type == SP_NEUMATIC_CUT_NB || element->element_type == SP_LARGER_SPACE_NB))
    {
      element=element->next;
    }
  if (!element || element->type != GRE_ELEMENT)
    {
      return -1; 
    }
  glyph = element->first_glyph;
  if (!glyph || !glyph->first_note)
    {
      return -1;
    }
  note = glyph->first_note;
  if (glyph->glyph_type == G_PES)
    {
      note = note->next;
      i = 2;
      gtype = T_PES;
    }
  else
    {
      gregoriotex_determine_number_and_type (glyph, element, &type,
				 &gtype, &glyph_number);
    }
  if (simple_htype(note->h_episemus_type) != H_NO_EPISEMUS)
    { 
      gregoriotex_find_sign_number (glyph, i,
		        gtype, TT_H_EPISEMUS, note,
		        &number, &height, &bottom);
				  if (bottom == 0)
				    {
				      *final_note = note;
              return height;
            }
          else
            {
              return -1;
            }
    }
  else
    {
      return -1;
    }
}

// a macro to write an additional line bottom_or_top is bottom, or top...

void
gregoriotex_write_additional_line (FILE * f,
				   gregorio_glyph *
				   current_glyph,
				   int i, char type,
				   char bottom_or_top,
				   gregorio_note * current_note)
{

  char height = 0;
  char number = 0;
  char ambitus = 0;

  if (!current_note)
    {
      return;
    }

  // patch to get a line under the full glyph in the case of dbc (for example)

  switch (type)
    {
    case T_PORRECTUSFLEXUS:
    case T_PORRECTUSFLEXUS_NOBAR:
    case T_PORRECTUS:
    case T_PORRECTUS_NOBAR:
      if (i == 1)
	{
	  i = HEPISEMUS_FIRST_TWO;
	}
      if (i == 2)
	{
	  if (current_note->previous->pitch > 'b')
	    {
	      i = HEPISEMUS_FIRST_TWO;
	    }
	  else
	    // we don't need to add twice the same line
	    {
	      return;
	    }
	}
      if (i == 3)
	{
	  // here the line has also been already added before
	  return;
	}
      break;
    case T_TORCULUS_RESUPINUS:
      if (i == 2)
	{
	  i = HEPISEMUS_FIRST_TWO;
	}
      if (i == 3 && current_note->previous->pitch > 'b')
	{
	  if (current_note->previous->pitch > 'b')
	    {
	      i = HEPISEMUS_FIRST_TWO;
	    }
	  else
	    {
	      return;
	    }
	}
      if (i == 4)
	{
	  // here the line has also been already added before
	  return;
	}
      break;
    default:
      break;
    }

  gregoriotex_find_sign_number (current_glyph, i,
				type, TT_H_EPISEMUS, current_note,
				&number, &height, NULL);

  if (i == HEPISEMUS_FIRST_TWO)
    {
      // here we must compare the first note of the big bar with the second one
      // but it may be tricky sometimes, because of the previous patch
      if (current_note->previous &&
	  current_note->previous->pitch > current_note->pitch)
	{
	  ambitus = current_note->previous->pitch - current_note->pitch;
	}
      else
	{
	  ambitus = current_note->pitch - current_note->next->pitch;
	}
    }
  fprintf (f, "\\greadditionalline{%d}{%d}{%d}%%\n", number, ambitus,
	   bottom_or_top);
}


/*

a function that writes the good value of \vepisemus in GregorioTeX. i is the position of the note in the glyph

*/


void
gregoriotex_write_vepisemus (FILE * f,
			     gregorio_glyph *
			     current_glyph,
			     int i, char type, gregorio_note * current_note)
{

  char height = 0;
  char number = 0;

  gregoriotex_find_sign_number (current_glyph, i,
				type, TT_V_EPISEMUS, current_note,
				&number, &height, NULL);
  if (current_note->pitch == 'a')
    {
      height = 'z';
    }
  fprintf (f, "\\grevepisemus{%c}{%d}%%\n", height, number);
}


/*
a function that writes the rare signs in GregorioTeX. i is the position of the note in the glyph
*/

void
gregoriotex_write_rare (FILE * f,
			gregorio_glyph *
			current_glyph,
			int i, char type,
			gregorio_note * current_note, char rare)
{

  char height = 0;
  char number = 0;

  // we set TT_V_EPISEMUS because the only height we have to calculate is this one
  gregoriotex_find_sign_number (current_glyph, i,
				type, TT_V_EPISEMUS, current_note,
				&number, &height, NULL);

  switch (rare)
    {
    case _ACCENTUS:
      fprintf (f, "\\greaccentus{%c}{%d}%%\n", current_note->pitch, number);
      break;
    case _ACCENTUS_REVERSUS:
      fprintf (f, "\\grereversedaccentus{%c}{%d}%%\n", current_note->pitch, number);
      break;
    case _CIRCULUS:
      fprintf (f, "\\grecirculus{%c}{%d}%%\n", current_note->pitch, number);
      break;
    case _SEMI_CIRCULUS:
      fprintf (f, "\\gresemicirculus{%c}{%d}%%\n", current_note->pitch, number);
      break;
    case _SEMI_CIRCULUS_REVERSUS:
      fprintf (f, "\\grereversedsemicirculus{%c}{%d}%%\n", current_note->pitch, number);
      break;
      // the cases of the bar signs are dealt in another function (write_bar)
    case _ICTUS_A:
      switch (current_note->signs)
	{
	case _V_EPISEMUS:
	case _V_EPISEMUS_PUNCTUM_MORA:
	case _V_EPISEMUS_AUCTUM_DUPLEX:
	  fprintf (f, "\\grevepisemusictusa{%c}{%d}%%\n", height, number);
	  break;
	default:
	  fprintf (f, "\\greictusa{%d}%%\n", number);
	  break;
	}
      break;
    case _ICTUS_T:
      switch (current_note->signs)
	{
	case _V_EPISEMUS:
	case _V_EPISEMUS_PUNCTUM_MORA:
	case _V_EPISEMUS_AUCTUM_DUPLEX:
	  fprintf (f, "\\grevepisemusictust{%c}{%d}%%\n", height, number);
	  break;
	default:
	  fprintf (f, "\\greictust{%d}%%\n", number);
	  break;
	}
      break;
    default:
      break;
    }
}

#define number_note_before_last_note() \
  if ((current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS || current_glyph->liquescentia == L_DEMINUTUS) && current_note->next)\
    {\
      *number = 3;\
    }\
  else \
      {\
      if ((current_note->pitch - current_note->next->pitch) == 1 || (current_note->pitch - current_note->next->pitch) == -1)\
    {\
      *number = 17;\
    }\
    else \
    {\
      *number = 2;\
    }\
    }

// num can be 0 or 18 according if the last note is a standard punctum or a smaller punctum (for pes, porrectus and torculus resupinus
#define number_last_note(num) \
  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS || current_glyph->liquescentia == L_DEMINUTUS)\
    {\
      /* may seem strange, but it is unlogical to typeset a small horizontal episemus at the end of a flexus deminutus */\
      *number = 1;\
    }\
  else \
    {\
      *number = num;\
    }

#define normal_height() \
  if (sign_type == TT_H_EPISEMUS) {\
    *height=current_note->h_episemus_top_note+1;\
  }\
  else {\
    *height=current_note->pitch -1;\
  }

#define normal_height_long_first() \
  if (sign_type == TT_H_EPISEMUS) {\
    *height=current_note->h_episemus_top_note+1;\
  }\
  else {\
    if (sign_type == TT_V_EPISEMUS)\
      {\
        *height=current_note->pitch -2;\
      }\
    else \
      {\
        *height=current_note->pitch -1;\
      }\
  }

//same as before, but for one note and then another one higher, when the sign is on the last
#define normal_height_top()\
  if (sign_type == TT_H_EPISEMUS) {\
    *height=current_note->h_episemus_top_note+1;\
  }\
  else {\
    if (current_note->previous && current_note -> pitch - current_note -> previous -> pitch > 2)\
      {\
        *height=current_note->pitch - 1;\
      }\
    else\
      {\
        *height=current_note->pitch + 2;\
      }\
  }

// case of one note and then one lower, when the sign is on the first
#define height_layered_notes()\
  if (sign_type == TT_H_EPISEMUS)\
    {\
      *height=current_note->h_episemus_top_note+1;\
    }\
  else\
    {\
      if ((current_note->pitch - current_note->next->pitch) == 1 || (current_note->pitch - current_note->next->pitch) == -1)\
      {\
        *height=current_note->pitch + 2;\
      }\
      else\
      {\
        *height=current_note->pitch -1;\
      }\
    }

// case of one note and then one higher, on the same vertical axis,
// when the sign is on the first
#define normal_height_bottom()\
  if (sign_type == TT_H_EPISEMUS) \
    {\
  /* we check if the previous or the next note has an horizontal episemus\
  // if it is the case, we use this height. If not, we put the episemus under the note*/\
      if ((!current_note->previous || simple_htype(current_note->previous -> h_episemus_type) == H_NO_EPISEMUS)\
          && (!current_note->next || simple_htype(current_note->next->h_episemus_type) == H_NO_EPISEMUS))\
        {\
          *height=current_note->pitch - 1;\
           if (bottom){\
             *bottom=1;\
           }\
        }\
      else \
        {\
          *height=current_note->h_episemus_top_note+1;\
        }\
    }\
  else\
    {\
      *height=current_note->pitch -1;\
    }

// case of one note and then one higher, on the same vertical axis,
// when the sign is on the first, special case of the pes
#define normal_height_bottom_pes()\
  if (sign_type == TT_H_EPISEMUS) \
    {\
       *height=current_note->pitch - 1;\
       if (bottom) {\
         *bottom=1;\
       }\
    }\
  else\
    {\
      *height=current_note->pitch -1;\
    }

// a function that finds the good sign (additional line, vepisemus or hepisemus) number, according to the gregoriotex convention (described in gregoriotex.tex)
// this function is REALLY a pain in the ass, but it is sadly necessary
void
gregoriotex_find_sign_number (gregorio_glyph * current_glyph,
			      int i, char type, char sign_type,
			      gregorio_note * current_note,
			      char *number, char *height, char *bottom)
{
  switch (type)
    {
    case T_PES:
    case T_PESQUILISMA:
/* in the case of a pes, we put the episemus just under the bottom note */
      if (i == 1)
	{
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 1;
	    }
	  else
	    {
	      if (current_note->shape == S_QUILISMA)
		{
		  *number = 15;
		}
	      else
		{
		  *number = 0;
		}
	    }
	  normal_height_bottom_pes ();
	}
      else
	{			/* i=2 */
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      *number = 1;
	    }
	  else
	    {
	      *number = 18;
	    }
	  normal_height_top ();
	}
      break;
    case T_PESQUADRATUM:
    case T_PESQUASSUS:
    case T_PESQUILISMAQUADRATUM:
      if (i == 1)
	{
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 7;
	    }
	  else
	    {
	      if (current_note->shape == S_ORISCUS)
		{
		  *number = 19;
		}
	      if (current_note->shape == S_QUILISMA)
		{
		  *number = 20;
		}
	      else
		{
		  *number = 6;
		}
	    }
	  normal_height_bottom ();
	}
      else
	{			/* i=2 */
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      *number = 7;
	    }
	  else
	    {
	      *number = 0;
	    }
	  normal_height ();
	}
      break;
    case T_FLEXUS:
    case T_FLEXUS_LONGQUEUE:
    case T_FLEXUS_ORISCUS:
      switch (i)
	{
	case 1:
	  number_note_before_last_note ();
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      height_layered_notes ();
	    }
	  else
	    {
	      normal_height ();
	    }
	  break;
	default:		/* i=2 */
	  number_last_note (0);
	  normal_height_bottom ();
	  break;
	}
      break;
    case T_PORRECTUSFLEXUS:
    case T_PORRECTUSFLEXUS_NOBAR:
      switch (i)
	{
	case HEPISEMUS_FIRST_TWO:
	  // special case, called when the horizontal episemus is on the fist two notes of a glyph. We consider current_note to be the first note.
	  if (!current_note->next)
	    {
	      *number = 0;
	    }
	  *number = 10;
	  normal_height ();
	  break;
	case 1:
	  *number = 6;
	  normal_height_long_first ();
	  break;
	case 2:
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      *number = 5;
	    }
	  else
	    {
	      *number = 4;
	    }
	  normal_height_bottom ();
	  break;
	case 3:
	  if (current_note->pitch - current_note->next->pitch != 1)
	    {
	      number_note_before_last_note ();
	    }
	  if ((current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	       || current_glyph->liquescentia == L_DEMINUTUS)
	      && current_note->next)
	    {
	      *number = 3;
	    }
	  else
	    {
	      *number = 2;
	    }
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      height_layered_notes ();
	    }
	  else
	    {
	      normal_height ();
	    }
	  break;
	default:
	  number_last_note (0);
	  normal_height_bottom ();
	  break;
	}
      break;
    case T_TORCULUS_RESUPINUS:
      switch (i)
	{
	case HEPISEMUS_FIRST_TWO:
	  // special case, called when the horizontal episemus is on the fist two notes of a glyph. We consider current_note to be the second note. in the case of the toruculus resupinus, it are the notes two and three. Warning, this MUST NOT be called if the porrectus is deminutus.
	  if (!current_note->next)
	    {
	      return;
	    }
	  *number = 11;
	  normal_height ();
	  break;
	case 1:
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 7;
	    }
	  else
	    {
	      *number = 6;
	    }
	  normal_height_bottom ();
	  break;
	case 2:
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 23;
	    }
	  else
	    {
	      if (current_note->pitch - current_note->previous->pitch == 1)
		{
		  *number = 22;
		}
	      else
		{
		  *number = 21;
		}
	    }
	  normal_height_long_first ();
	  break;
	case 3:
	  // you might think number_note_before_last_note more appropriate, but in the current fonts the third note of the torculus resupinus is v aligned with the last note
	  number_last_note (18);
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      height_layered_notes ();
	    }
	  else
	    {
	      normal_height_bottom ();
	    }
	  break;
	default:
	  number_last_note (18);
	  normal_height_top ();
	  break;
	}
      break;
    case T_PORRECTUS:
    case T_PORRECTUS_NOBAR:
      switch (i)
	{
	case HEPISEMUS_FIRST_TWO:
	  // special case, called when the horizontal episemus is on the fist two notes of a glyph. We consider current_note to be the first note. Warning, this MUST NOT be called if the porrectus is deminutus.
	  if (!current_note->next)
	    {
	      *number = 0;
	    }
	  *number = 9;
	  normal_height ()break;
	case 1:
	  *number = 6;
	  normal_height_long_first ();
	  break;
	case 2:
	  if ((current_glyph->liquescentia ==
	       L_DEMINUTUS_INITIO_DEBILIS
	       || current_glyph->liquescentia == L_DEMINUTUS)
	      && current_note->next)
	    {
	      *number = 3;
	    }
	  else
	    {
	      *number = 0;
	    }
	  normal_height_bottom ();
	  break;
	default:
	  number_last_note (1);
	  normal_height_top ();
	  break;
	}
      break;
    case T_SCANDICUS:
      switch (i)
	{
	case 1:
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 7;
	    }
	  else
	    {
	      *number = 6;
	    }
	  normal_height ();
	  break;
	case 2:
	  number_note_before_last_note ();
	  normal_height ();
	  break;
	default:
	  number_last_note (0);
	  normal_height_top ();
	  break;
	}
      break;
    case T_ANCUS:
    case T_ANCUS_LONGQUEUE:
      switch (i)
	{
	case 1:
	  *number = 6;
	  normal_height ();
	  break;
	case 2:
	  number_note_before_last_note ();
	  height_layered_notes ();
	  break;
	default:
	  number_last_note (0);
	  normal_height ();
	  break;
	}
      break;
    case T_TORCULUS:
    case T_TORCULUS_QUILISMA:
      switch (i)
	{
	case 1:
	  if (current_glyph->liquescentia >= L_INITIO_DEBILIS)
	    {
	      *number = 7;
	    }
	  else
	    {
	      if (type == T_TORCULUS_QUILISMA)
		{
		  *number = 20;
		}
	      else
		{
		  *number = 6;
		}
	    }
	  normal_height_bottom ();
	  break;
	case 2:
	  number_note_before_last_note ();
	  if (current_glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS
	      || current_glyph->liquescentia == L_DEMINUTUS)
	    {
	      height_layered_notes ();
	    }
	  else
	    {
	      normal_height ();
	    }
	  break;
	default:
	  number_last_note (0);
	  normal_height_bottom ();
	  break;
	}
      break;
    default:			/* case T_ONE_NOTE */
      normal_height ();
      switch (current_note->shape)
	{
	case S_PUNCTUM_INCLINATUM_DEMINUTUS:
	  *number = 13;
	  break;
	case S_PUNCTUM_INCLINATUM_AUCTUS:
	case S_PUNCTUM_INCLINATUM:
	  *number = 12;
	  break;
	case S_STROPHA:
	  *number = 14;
	  break;
	case S_QUILISMA:
	  *number = 15;
	  break;
	case S_ORISCUS:
	case S_ORISCUS_AUCTUS:
	case S_ORISCUS_DEMINUTUS:
	  *number = 16;
	  break;
	case S_LINEA_PUNCTUM:
	case S_LINEA_PUNCTUM_CAVUM:
	  *number = 24;
	  break;
	case S_LINEA:
	  *number = 0;
	  break;
	default:
	  number_last_note (0);
	  break;
	}
      break;
    }

}

/*

the different numbers of the liquescentiae are:
'nothing':0,
'initiodebilis':1,
'deminutus':2,
'auctusascendens':3,
'auctusdescendens':4,
'initiodebilisdeminutus':5,
'initiodebilisauctusascendens':6,
'initiodebilisauctusdescendens':7

if it is an auctus, which may be ascendens or descendens, by default we consider it as an ascendens

they also are and must be the same as in squarize.py.

to obtain the glyph number, we just do 512 * glyphtype + liq_factor * liquescentia + x

where liq_factor is 64 for short types (pes and flexus) and 256 for long types

where x is a number related to the differences between te heights of the notes:

if i is the difference between the two first notes, j between the second and the third and k the difference betweend the third and the fourth, x = i + 5 * j + 25 * k

*/

unsigned int
gregoriotex_determine_liquescentia_number (unsigned int
					   factor,
					   unsigned char
					   type, char liquescentia)
{
  if (liquescentia == L_AUCTA)
    {
      liquescentia = L_AUCTUS_ASCENDENS;
    }
  if (liquescentia == L_AUCTA_INITIO_DEBILIS)
    {
      liquescentia = L_AUCTUS_ASCENDENS_INITIO_DEBILIS;
    }
  switch (type)
    {
    case L_ALL:
      break;
    case L_NO_INITIO:
      if (liquescentia >= L_INITIO_DEBILIS)
	{
	  liquescentia = liquescentia - L_INITIO_DEBILIS;
	}
      break;
    case L_ONLY_DEMINUTUS:
      if (liquescentia != L_DEMINUTUS
	  && liquescentia != L_DEMINUTUS_INITIO_DEBILIS)
	{
	  liquescentia = L_NO_LIQUESCENTIA;
	}
      break;
    case L_ONLY_AUCTUS:
      if (liquescentia != L_AUCTUS_ASCENDENS
	  && liquescentia != L_AUCTUS_DESCENDENS)
	{
	  liquescentia = L_NO_LIQUESCENTIA;
	}
    case L_UNDET_AUCTUS:
      if (liquescentia == L_AUCTUS_DESCENDENS)
	{
	  liquescentia = L_AUCTUS_ASCENDENS;
	}
      if (liquescentia == L_AUCTUS_DESCENDENS_INITIO_DEBILIS)
	{
	  liquescentia = L_AUCTUS_ASCENDENS_INITIO_DEBILIS;
	}
      break;
    default:
      return 0;
    }

  //now we convert liquescentia into the good GregorioTeX liquescentia numbers

  switch (liquescentia)
    {
    case L_NO_LIQUESCENTIA:
      liquescentia = GL_NO_LIQUESCENTIA;
      break;
    case L_DEMINUTUS:
      liquescentia = GL_DEMINUTUS;
      break;
    case L_AUCTUS_ASCENDENS:
      liquescentia = GL_AUCTUS_ASCENDENS;
      break;
    case L_AUCTA:
    case L_AUCTUS_DESCENDENS:
      liquescentia = GL_AUCTUS_DESCENDENS;
      break;
    case L_INITIO_DEBILIS:
      liquescentia = GL_INITIO_DEBILIS;
      break;
    case L_DEMINUTUS_INITIO_DEBILIS:
      liquescentia = GL_DEMINUTUS_INITIO_DEBILIS;
      break;
    case L_AUCTUS_ASCENDENS_INITIO_DEBILIS:
      liquescentia = GL_AUCTUS_ASCENDENS_INITIO_DEBILIS;
      break;
    case L_AUCTA_INITIO_DEBILIS:
    case L_AUCTUS_DESCENDENS_INITIO_DEBILIS:
      liquescentia = GL_AUCTUS_DESCENDENS_INITIO_DEBILIS;
      break;
    default:
      liquescentia = GL_NO_LIQUESCENTIA;
      break;
    }

  return factor * liquescentia;
}

// a helper macro for the following function
#define whileglyph(prevornext) \
        while(glyph)\
          {\
            if (glyph->type == GRE_GLYPH)\
              {\
                note = glyph->first_note;\
                while (note)\
                  {\
                    if (note->pitch < 'c')\
                      {\
                        return 1;\
                      }\
                    note = note->next;\
                  }\
              }\
            glyph = glyph->prevornext;\
          }

// a function that determines if we must use a long queue or not (less easy that it might seem)

unsigned char
gregoriotex_is_long(char pitch, gregorio_glyph *current_glyph, gregorio_element *current_element)
{
  gregorio_note *note;
  gregorio_glyph *glyph = current_glyph->next;
  gregorio_element *element = current_element->next;
  switch (pitch)
    {
      case 'b':
      case 'f':
      case 'h':
      case 'j':
      case 'l':
        return 1;
      case 'd':
        //we first look forward to see if there is a note underneath c
        whileglyph(next);
        if (element && element->type == GRE_SPACE && (element->element_type == SP_NEUMATIC_CUT || element->element_type == SP_LARGER_SPACE || element->element_type == SP_NEUMATIC_CUT_NB || element->element_type == SP_LARGER_SPACE_NB))
          {
            element=element->next;
          }
        if (element && element->type == GRE_ELEMENT)
          { 
            glyph = element->first_glyph;
            whileglyph(next);
          }
        // and now something completely different
        glyph = current_glyph->previous;
        element = current_element->previous;
        whileglyph(previous);
        if (element && element->type == GRE_SPACE && (element->element_type == SP_NEUMATIC_CUT || element->element_type == SP_LARGER_SPACE || element->element_type == SP_NEUMATIC_CUT_NB || element->element_type == SP_LARGER_SPACE_NB))
          {
            element=element->previous;
          }
        if (element && element->type == GRE_ELEMENT)
          { 
            glyph = element->first_glyph;
            whileglyph(next);
          }
        return 0;
      default:
        return 0;
    }
}

// finaly the function that calculates the number of the glyph. It also calculates the type, used for determining the position of signs. Type is very basic, it is only the global dimensions : torculus, one_note, etc.

void
gregoriotex_determine_number_and_type
  (gregorio_glyph * glyph, gregorio_element *element, int *type, char *gtype, unsigned int *glyph_number)
{
  unsigned int temp = 0;
  char pitch = 0;
  char liquescentia;
  if (!glyph)
    {
      gregorio_message (_
			("called with NULL pointer"),
			"gregoriotex_determine_number_and_type", ERROR, 0);
      return;
    }
  if (!glyph->first_note)
    {
      gregorio_message (_
			("called with a glyph that has no note"),
			"gregorio_tex_determine_number_and_type", ERROR, 0);
      return;
    }
  liquescentia = glyph->liquescentia;
  /* commented, but must be there for the font gregoria (as there is no auctus descendens). TODO : having a variable telling the font
     if (liquescentia == L_AUCTUS_ASCENDENS)
     {
     glyph->liquescentia = L_AUCTUS_DESCENDENS;
     }
     if (liquescentia == L_AUCTUS_ASCENDENS_INITIO_DEBILIS)
     {
     glyph->liquescentia = L_AUCTUS_DESCENDENS_INITIO_DEBILIS;
     } */

  switch (glyph->glyph_type)
    {
    case G_PODATUS:
      pitch = glyph->first_note->next->pitch;
      switch (glyph->first_note->shape)
	{
	case S_QUILISMA:
	  *type = AT_QUILISMA;
	  // the next if is because we made the choice that AUCTUS shapes look like pes quadratum.
	  if (glyph->liquescentia == L_AUCTUS_ASCENDENS
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS_INITIO_DEBILIS
	      || glyph->liquescentia == L_AUCTUS_ASCENDENS_INITIO_DEBILIS)
	    {
	      *gtype = T_PESQUILISMAQUADRATUM;
	    }
	  else
	    {
	      *gtype = T_PESQUILISMA;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_NO_INITIO,
						       glyph->liquescentia);
	  break;
	case S_ORISCUS:
	  *type = AT_ORISCUS;
	  // TODO: we could factorize this code
	  if (glyph->liquescentia == L_NO_LIQUESCENTIA && gregoriotex_is_long(pitch, glyph, element) == 1)
	    {
	      *gtype = T_PESQUASSUS_LONGQUEUE;
	    }
	  else
	    {
	      *gtype = T_PESQUASSUS;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_NO_INITIO,
						       glyph->liquescentia);
	  break;
	default:
	  *type = AT_ONE_NOTE;
	  if (glyph->liquescentia == L_AUCTUS_ASCENDENS
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS
	      || glyph->liquescentia == L_AUCTUS_DESCENDENS_INITIO_DEBILIS
	      || glyph->liquescentia == L_AUCTUS_ASCENDENS_INITIO_DEBILIS)
	    {
	      *gtype = T_PESQUADRATUM;
	    }
	  else
	    {
	      *gtype = T_PES;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_ALL,
						       glyph->liquescentia);
	  break;
	}
      break;
    case G_PES_QUADRATUM:
      pitch = glyph->first_note->next->pitch;
      switch (glyph->first_note->shape)
	{
	case S_QUILISMA:
	  *type = AT_QUILISMA;
	  if (glyph->liquescentia == L_NO_LIQUESCENTIA && gregoriotex_is_long(pitch, glyph, element) == 1)
	    {
	      *gtype = T_PESQUILISMAQUADRATUM_LONGQUEUE;
	    }
	  else
	    {
	      *gtype = T_PESQUILISMAQUADRATUM;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_NO_INITIO,
						       glyph->liquescentia);
	  break;
	case S_ORISCUS:
	  *type = AT_ORISCUS;
	  if (glyph->liquescentia == L_NO_LIQUESCENTIA && gregoriotex_is_long(pitch, glyph, element) == 1)
	    {
	      *gtype = T_PESQUASSUS_LONGQUEUE;
	    }
	  else
	    {
	      *gtype = T_PESQUASSUS;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_NO_INITIO,
						       glyph->liquescentia);
	  break;
	default:
	  *type = AT_ONE_NOTE;
	  if (glyph->liquescentia == L_NO_LIQUESCENTIA && gregoriotex_is_long(pitch, glyph, element) == 1)
	    {
	      *gtype = T_PESQUADRATUM_LONGQUEUE;
	    }
	  else
	    {
	      *gtype = T_PESQUADRATUM;
	    }
	  temp =
	    TYPE_FACTOR * (*gtype) +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_ALL,
						       glyph->liquescentia);
	  break;
	}
      break;
    case G_FLEXA:
      pitch = glyph->first_note->pitch;
      if (glyph->liquescentia == L_DEMINUTUS)
	{
	  *type = AT_FLEXUS_DEMINUTUS;
	}
      else
	{
	  if (pitch - glyph->first_note->next->pitch == 1)
	    {
	      *type = AT_FLEXUS_1;
	    }
	  else
	    {
	      *type = AT_FLEXUS;
	    }
	}
      if (glyph->first_note->shape == S_ORISCUS)
	{
	  *gtype = T_FLEXUS_ORISCUS;
	  temp =
	    TYPE_FACTOR * T_FLEXUS_ORISCUS +
	    gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
						       L_ONLY_AUCTUS,
						       glyph->liquescentia);
	}
      else
	{
	  if (is_short (pitch, glyph, element))
	    {
	      *gtype = T_FLEXUS;
	      temp =
		TYPE_FACTOR * T_FLEXUS +
		gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
							   L_NO_INITIO,
							   glyph->
							   liquescentia);
	    }
	  else
	    {
	      *gtype = T_FLEXUS_LONGQUEUE;
	      temp =
		TYPE_FACTOR * T_FLEXUS_LONGQUEUE +
		gregoriotex_determine_liquescentia_number (S_LIQ_FACTOR,
							   L_NO_INITIO,
							   glyph->
							   liquescentia);
	    }
	}
      break;
    case G_TORCULUS:
      if (glyph->first_note->shape == S_QUILISMA)
	{
	  *type = AT_QUILISMA;
	  *gtype = T_TORCULUS_QUILISMA;
	  temp =
	    TYPE_FACTOR * T_TORCULUS_QUILISMA +
	    gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						       L_NO_INITIO,
						       glyph->liquescentia);
	}
      else
	{
	  *type = AT_ONE_NOTE;
	  *gtype = T_TORCULUS;
	  temp =
	    TYPE_FACTOR * T_TORCULUS +
	    gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR, L_ALL,
						       glyph->liquescentia);
	}
      break;
    case G_TORCULUS_RESUPINUS_FLEXUS:
      // not sure about that... TODO: check
      *type = AT_ONE_NOTE;
      *gtype = T_TORCULUS_RESUPINUS;
      temp = 0;
      break;
    case G_PORRECTUS:
      *type = AT_PORRECTUS;
      *gtype = T_PORRECTUS;
      temp =
	TYPE_FACTOR * T_PORRECTUS +
	gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						   L_ONLY_DEMINUTUS,
						   glyph->liquescentia);
      break;
    case G_TORCULUS_RESUPINUS:
      *type = AT_ONE_NOTE;
      *gtype = T_TORCULUS_RESUPINUS;
      temp =
	TYPE_FACTOR * T_TORCULUS_RESUPINUS +
	gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						   L_ONLY_DEMINUTUS,
						   glyph->liquescentia);
      break;
    case G_PORRECTUS_FLEXUS:
      *type = AT_PORRECTUS;
      *gtype = T_PORRECTUSFLEXUS;
      temp =
	TYPE_FACTOR * T_PORRECTUSFLEXUS +
	gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						   L_NO_INITIO,
						   glyph->liquescentia);
      break;
    case G_PORRECTUS_NO_BAR:
      *type = AT_PORRECTUS;
      *gtype = T_PORRECTUS_NOBAR;
      temp =
	TYPE_FACTOR * T_PORRECTUS_NOBAR +
	gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						   L_ONLY_DEMINUTUS,
						   glyph->liquescentia);
      break;
    case G_PORRECTUS_FLEXUS_NO_BAR:
      *type = AT_PORRECTUS;
      *gtype = T_PORRECTUSFLEXUS_NOBAR;
      temp =
	TYPE_FACTOR * T_PORRECTUSFLEXUS_NOBAR +
	gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						   L_NO_INITIO,
						   glyph->liquescentia);
      break;
    case G_ANCUS:
      if (glyph->liquescentia == L_DEMINUTUS
	  || glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS)
	{
	  if (pitch - glyph->first_note->next->pitch == 1)
	    {
	      *type = AT_FLEXUS_1;
	    }
	  else
	    {
	      *type = AT_FLEXUS;
	    }
	  if (is_short (pitch, glyph, element))
	    {
	      *gtype = T_ANCUS;
	      temp = TYPE_FACTOR * T_ANCUS +
		gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
							   L_ONLY_DEMINUTUS,
							   glyph->
							   liquescentia);
	    }
	  else
	    {
	      *gtype = T_ANCUS_LONGQUEUE;
	      temp = TYPE_FACTOR * T_ANCUS_LONGQUEUE +
		gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
							   L_ONLY_DEMINUTUS,
							   glyph->
							   liquescentia);
	    }
	}
      else
	{
	  //TODO...
	  *type = AT_ONE_NOTE;
	}
      break;
    case G_SCANDICUS:
      if (glyph->liquescentia == L_DEMINUTUS
	  || glyph->liquescentia == L_DEMINUTUS_INITIO_DEBILIS)
	{
	  *type = AT_ONE_NOTE;
	  *gtype = T_SCANDICUS;
	  temp = TYPE_FACTOR * T_SCANDICUS +
	    gregoriotex_determine_liquescentia_number (L_LIQ_FACTOR,
						       L_ONLY_DEMINUTUS,
						       glyph->liquescentia);;
	}
      else
	{
	  // TODO: do it really...
	  *type = AT_ONE_NOTE;
	}
      break;
    case G_ONE_NOTE:
    case G_PUNCTUM_INCLINATUM:
    case G_TRIGONUS:
    case G_PUNCTA_INCLINATA:
    case G_2_PUNCTA_INCLINATA_DESCENDENS:
    case G_3_PUNCTA_INCLINATA_DESCENDENS:
    case G_4_PUNCTA_INCLINATA_DESCENDENS:
    case G_5_PUNCTA_INCLINATA_DESCENDENS:
    case G_2_PUNCTA_INCLINATA_ASCENDENS:
    case G_3_PUNCTA_INCLINATA_ASCENDENS:
    case G_4_PUNCTA_INCLINATA_ASCENDENS:
    case G_5_PUNCTA_INCLINATA_ASCENDENS:
    case G_PUNCTUM:
    case G_STROPHA:
    case G_VIRGA:
    case G_STROPHA_AUCTA:
    case G_DISTROPHA:
    case G_DISTROPHA_AUCTA:
    case G_TRISTROPHA:
    case G_TRISTROPHA_AUCTA:
    case G_BIVIRGA:
    case G_TRIVIRGA:
      *type = AT_ONE_NOTE;
      break;
    default:
      gregorio_message (_
			("called with unknown glyph"),
			"gregoriotex_determine_number_and_type", ERROR, 0);
      break;
    }

  *glyph_number = gregoriotex_determine_interval (glyph);
  *glyph_number = temp + (*glyph_number);
  // we change to the original liquescentia
  glyph->liquescentia = liquescentia;
  // we fix *type with initio_debilis
  if (*type == AT_ONE_NOTE)
    {
      if (is_initio_debilis (liquescentia))
	{
	  *type = AT_INITIO_DEBILIS;
	}
    }

}

unsigned int
gregoriotex_determine_interval (gregorio_glyph * glyph)
{
  gregorio_note *current_note;
  unsigned int current;
// then we start making our formula
  char first;
  char second;
  if (!glyph)
    {
      gregorio_message (_
			("called with NULL pointer"),
			"gregoriotex_determine_interval", ERROR, 0);
      return 0;
    }
  if (!glyph->first_note)
    {
      gregorio_message (_
			("called with a glyph that have no note"),
			"gregoriotex_determine_interval", ERROR, 0);
      return 0;
    }
  current_note = glyph->first_note;
  if (!current_note->next)
    {
      return 0;
    }
  first = current_note->pitch;
  second = current_note->next->pitch;
  if (first < second)
    {
      current = second - first;
    }
  else
    {
      current = first - second;
    }
  current_note = current_note->next;
  if (!current_note->next)
    {
      return current;
    }
  first = current_note->pitch;
  second = current_note->next->pitch;
  if (first < second)
    {
      current = 5 * (second - first) + current;
    }
  else
    {
      current = 5 * (first - second) + current;
    }
  current_note = current_note->next;
  if (!current_note->next)
    {
      return current;
    }
  first = current_note->pitch;
  second = current_note->next->pitch;
  if (first < second)
    {
      current = 25 * (second - first) + current;
    }
  else
    {
      current = 25 * (first - second) + current;
    }
  return current;
}

/* function used when the glyph is only one note long, the glyph number are simply the following:
* 01: c clef
* 02: f clef
* 03: c clef for key changes
* 04: f clef for key changes
* 05: flat (oriented to the top)
* 06: flat (oriented to the bottom)
* 07: natural
* 85: sharp
* 86: sharp hole
* 08: virgula
* 09: divisio minima
* 10: divisio minor
* 11: divisio maior
* 12: divisio finalis
* 83: virgula minor oriented to the top
* 84: virgula minor oriented to the bottom
* 13: punctum deminutum
* 14: punctum mora
* 15: auctum duplex
* 16: circumflexus
* 17: punctum
* 18: punctum quadratum
* 19: punctum inclinatum
* 20: stropha
* 21: stropha aucta
* 22: virga
* 23: virga_short_bar
* 24: left virga
* 25: left virga_short_bar
* 26: quilisma
* 27: oriscus
* 28: oriscus auctus
* 91: oriscus deminutus
* 31: punctum inclinatum auctum
* 32: punctum inclinatum deminutus
* 33: vertical episemus
* 34: punctum cavum
* 35: linea punctum
* 36: linea puctum cavum
* 37: circulus
* 38: semi-curculus
* 39: accentus
* 75: punctum cavum alt
* 76: linea punctum cavum alt
* 77: punctum cavum hole
* 78: punctum cavum alt hole
* 79: linea punctum cavum hole
* 80: linea punctum cavum alt hole
* 81: flat hole
* 82: natural hole
* 69: reversed accentus
* 70: reversed semi-circulus
* 72: punctum auctus ascendens
* 73: punctum auctus descendens
* 74: smaller punctum for pes, porrectus and torculus resupinus (theorically never used alone, but necessary for some measures
* 87: linea

* 60: custo for bass notes (oriented to the top)
* 61: custo for bass notes (oriented to the top) with short bar
* 62: custo for bass notes (oriented to the top) with middle bar (for the lowest note)
* 63: custo for high notes (oriented to the bottom)
* 64: custo for high notes (oriented to the bottom) with short bar
* 65: custo for high notes (oriented to the bottom) with middle bar (for the highest note)

*/

// and the different types of horizontal episemus:
//* 40: horizontal episemus, width of a punctum
#define H_PUNCTUM 40
//* 41: horizontal episemus, width of a flexus debilis
#define H_FLEXUS 41
//* 42: horizontal episemus, width of an initio debilis
#define H_INITIO 42
//* 43: horizontal episemus, width of a punctum inclinatum
#define H_INCLINATUM 43
//* 44: horizontal episemus, width of a punctum inclinatum deminutus
#define H_INCLINATUM_DEMINUTUS 44
//* 45: horizontal episemus, width of a stropha
#define H_STROPHA 45
//* 46: horizontal episemus, width of a porrectus with ambitus of 1
#define H_PORRECTUS1 46
//* 47: horizontal episemus, width of a porrectus with ambitus of 2
#define H_PORRECTUS2 47
//* 48: horizontal episemus, width of a porrectus with ambitus of 3
#define H_PORRECTUS3 48
//* 49: horizontal episemus, width of a porrectus with ambitus of 4
#define H_PORRECTUS4 49
//* 50: horizontal episemus, width of a porrectus with ambitus of 5
#define H_PORRECTUS5 50
//* 51: horizontal episemus, width of a porrectus flexus with ambitus of 1
#define H_PORRECTUS_FLEXUS1 51
//* 52: horizontal episemus, width of a porrectus flexus with ambitus of 2
#define H_PORRECTUS_FLEXUS2 52
//* 53: horizontal episemus, width of a porrectus flexus with ambitus of 3
#define H_PORRECTUS_FLEXUS3 53
//* 54: horizontal episemus, width of a porrectus flexus with ambitus of 4
#define H_PORRECTUS_FLEXUS4 54
//* 55: horizontal episemus, width of a porrectus flexus with ambitus of 5
#define H_PORRECTUS_FLEXUS5 55
// * 56: horizontal episemus, width of a quilisma
#define H_QUILISMA 56
// * 57: horizontal episemus, width of an oriscus
#define H_ORISCUS 57
// * 58: horizontal episemus width of a small punctum for pes, porrectus and torculus resupinus
#define H_SMALL_PUNCTUM 58

void
gregoriotex_write_note (FILE * f, gregorio_note * note, gregorio_glyph *glyph, gregorio_element *element, char next_note_pitch)
{
  unsigned int glyph_number;
  char temp;
  // type in the sense of GregorioTeX alignment type
  int type = AT_ONE_NOTE;
  if (!note)
    {
      gregorio_message (_
			("called with NULL pointer"),
			"gregoriotex_write_note", ERROR, 0);
      return;
    }

  gregoriotex_determine_note_number_and_type (note, glyph, element, &type, &glyph_number);
// special things for puncta inclinata
  if (note->shape == S_PUNCTUM_INCLINATUM)
    {
      if (note->previous)
	{
//means that it is the first note of the puncta inclinata sequence
	  temp = note->previous->pitch - note->pitch;
	  //if (temp < -1 || temp > 1)
	  switch (temp)		//we switch on the range of the inclinata
	    {
	      // this will look somewhat strange if temp is negative... to be aligned then,
	      //the second note should really shift differently
	    case -2:
	    case 2:
	      fprintf (f, "\\greendofglyph{10}%%\n");
	      break;
	    case -3:
	    case 3:
	      fprintf (f, "\\greendofglyph{11}%%\n");
	      break;
	    case -4:
	    case 4:		//not sure we ever need to consider a larger ambitus here
	      fprintf (f, "\\greendofglyph{11}%%\n");
	      break;
	    default:
	      fprintf (f, "\\greendofglyph{3}%%\n");
	      break;
	    }
	}
    }
  if (note->shape == S_PUNCTUM_INCLINATUM_DEMINUTUS)
    {
      if (note->previous)
	{
//means that it is the first note of the puncta inclinata sequence
	  temp = note->previous->pitch - note->pitch;
	  if (temp < -2 || temp > 2)
	    {
	      fprintf (f, "\\greendofglyph{11}%%\n");
	    }
	  else
	    {
	      if (note->previous
		  && note->previous->shape == S_PUNCTUM_INCLINATUM_DEMINUTUS)
		{
		  if (temp < -1 || temp > 1)
		    //really if the ambitus = 3rd at this point
		    {
		      fprintf (f, "\\greendofglyph{10}%%\n");
		    }
		  else
		    {
		      fprintf (f, "\\greendofglyph{8}%%\n");
		    }
		}
	      else
		{
		  //puncta inclinatum followed by puncta inclinatum debilis
		  fprintf (f, "\\greendofglyph{7}%%\n");
		}
	    }
	}
    }
  if (note->shape == S_PUNCTUM_INCLINATUM_AUCTUS)
    {
      if (note->previous)
	{
//means that it is the first note of the puncta inclinata sequence
	  temp = note->previous->pitch - note->pitch;
	  if (temp < -1 || temp > 1)
	    {
	      fprintf (f, "\\greendofglyph{1}%%\n");
	    }
	  else
	    {
	      // we approximate that it is the same space
	      fprintf (f, "\\greendofglyph{3}%%\n");
	    }
	}
    }
  switch (note->shape)
    {
    case S_PUNCTUM_CAVUM:
      fprintf (f, "\\grepunctumcavum{%c}{%c}{%d}", note->pitch,
	       next_note_pitch, type);
      break;
    case S_LINEA_PUNCTUM_CAVUM:
      fprintf (f, "\\grelineapunctumcavum{%c}{%c}{%d}", note->pitch,
	       next_note_pitch, type);
      break;
    case S_LINEA:
      fprintf (f, "\\grelinea{%c}{%c}{%d}", note->pitch,
	       next_note_pitch, type);
      break;
    default:
      fprintf (f, "\\greglyph{\\char %d}{%c}{%c}{%d}",
	       glyph_number, note->pitch, next_note_pitch, type);
      break;
    }
}

void
  gregoriotex_determine_note_number_and_type
  (gregorio_note * note, gregorio_glyph *glyph, gregorio_element *element, int *type, unsigned int *glyph_number)
{
  if (!note)
    {
      gregorio_message (_
			("called with NULL pointer"),
			"gregoriotex_determine_note_number_and_type", ERROR, 0);
      return;
    }

  *type = AT_ONE_NOTE;
  switch (note->shape)
    {
    case S_PUNCTUM_INCLINATUM:
      *glyph_number = 19;
      *type = AT_PUNCTUM_INCLINATUM;
      break;
    case S_PUNCTUM_INCLINATUM_DEMINUTUS:
      *glyph_number = 32;
      break;
    case S_PUNCTUM_INCLINATUM_AUCTUS:
      *glyph_number = 31;
      break;
    case S_PUNCTUM:
      *glyph_number = 17;
      break;
    case S_PUNCTUM_AUCTUS_ASCENDENS:
      *glyph_number = 72;
      break;
    case S_PUNCTUM_AUCTUS_DESCENDENS:
      *glyph_number = 73;
      break;
    case S_PUNCTUM_DEMINUTUS:
      *glyph_number = 13;
      break;
    case S_PUNCTUM_CAVUM:
      *glyph_number = 34;
      break;
    case S_LINEA:
      *glyph_number = 87;
      break;
    case S_LINEA_PUNCTUM:
      *glyph_number = 35;
      break;
    case S_LINEA_PUNCTUM_CAVUM:
      *glyph_number = 36;
      break;
    case S_VIRGA:
      if (is_short (note->pitch, glyph, element))
	{
	  *glyph_number = 23;
	}
      else
	{
	  *glyph_number = 22;
	}
      break;
    case S_ORISCUS:
      *type = AT_ORISCUS;
      *glyph_number = 27;
      break;
    case S_ORISCUS_AUCTUS:
      *type = AT_ORISCUS;
      *glyph_number = 28;
      break;
    case S_ORISCUS_DEMINUTUS:
      *type = AT_ORISCUS;
      *glyph_number = 91;
      break;
    case S_QUILISMA:
      *type = AT_QUILISMA;
      *glyph_number = 26;
      break;
    case S_STROPHA:
	    if (glyph->liquescentia == L_AUCTA)
	      {
	        *glyph_number = 21;
	      }
	    else
	      {
          *glyph_number = 20;
	      }
      *type = AT_STROPHA;
      break;
    case S_STROPHA_AUCTA:
      *type = AT_STROPHA;
      *glyph_number = 21;
      break;
    default:
      gregorio_message (_
			("called with unknown shape"),
			"gregoriotex_determine_note_number_and_type", ERROR, 0);
      return;
      break;
    }

}

int
gregoriotex_syllable_first_type (gregorio_syllable * syllable)
{
  int type = 0;
  char gtype = 0;
  unsigned int number = 0;
  // alteration says if there is a flat or a natural first in the next syllable, see gregoriotex.tex for more details
  int alteration = 0;
  gregorio_glyph *glyph;
  gregorio_element *element;
  if (!syllable)
    {
      gregorio_message (_
			("called with a NULL argument"),
			"gregoriotex_syllable_first_type", ERROR, 0);
    }
  element = syllable->elements[0];
  while (element)
    {
      if (element->type == GRE_BAR)
	{
	  switch (element->element_type)
	    {
	    case B_NO_BAR:
	    case B_VIRGULA:
	      type = 10;
	      break;
	    case B_DIVISIO_MINIMA:
	    case B_DIVISIO_MINOR:
	    case B_DIVISIO_MAIOR:
	    case B_DIVISIO_MINOR_D1:
	    case B_DIVISIO_MINOR_D2:
	    case B_DIVISIO_MINOR_D3:
	    case B_DIVISIO_MINOR_D4:
	    case B_DIVISIO_MINOR_D5:
	    case B_DIVISIO_MINOR_D6:
	      type = 11;
	      break;
	    case B_DIVISIO_FINALIS:
	      type = 12;
	      break;
	    default:
	      type = 0;
	      break;
	    }
	  return type;
	}
      if (element->type == GRE_ELEMENT && element->first_glyph)
	{
	  glyph = element->first_glyph;
	  while (glyph)
	    {
	      if (glyph->type == GRE_FLAT && alteration == 0)
		{
		  alteration = 20;
		}
	      if (glyph->type == GRE_NATURAL && alteration == 0)
		{
		  alteration = 40;
		}
	      if (glyph->type == GRE_SHARP && alteration == 0)
		{
		  alteration = 60;
		}
	      if (glyph->type == GRE_GLYPH && glyph->first_note)
		{
		  switch (glyph->glyph_type)
		    {
		    case G_TRIGONUS:
		    case G_PUNCTA_INCLINATA:
		    case G_2_PUNCTA_INCLINATA_DESCENDENS:
		    case G_3_PUNCTA_INCLINATA_DESCENDENS:
		    case G_4_PUNCTA_INCLINATA_DESCENDENS:
		    case G_5_PUNCTA_INCLINATA_DESCENDENS:
		    case G_2_PUNCTA_INCLINATA_ASCENDENS:
		    case G_3_PUNCTA_INCLINATA_ASCENDENS:
		    case G_4_PUNCTA_INCLINATA_ASCENDENS:
		    case G_5_PUNCTA_INCLINATA_ASCENDENS:
		    case G_PUNCTUM:
		    case G_STROPHA:
		    case G_VIRGA:
		    case G_STROPHA_AUCTA:
		    case G_DISTROPHA:
		    case G_DISTROPHA_AUCTA:
		    case G_TRISTROPHA:
		    case G_TRISTROPHA_AUCTA:
		    case G_BIVIRGA:
		    case G_TRIVIRGA:
		      gregoriotex_determine_note_number_and_type
			(glyph->first_note, glyph, element, &type, &number);
		      break;
		    default:
		      gregoriotex_determine_number_and_type
			(glyph, element, &type, &gtype, &number);
		      break;
		    }
		  return type + alteration;
		}
	      glyph = glyph->next;
	    }
	}
      element = element->next;
    }
  return 0;
}
