/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <unistd.h>
#include <signal.h>
#include "grdcfile.h"
#include "grdcplug.h"
#include "grdcplug_rdp.h"

G_DEFINE_TYPE (GrdcPlugRdp, grdc_plug_rdp, GRDC_TYPE_PLUG)

static void
grdc_plug_rdp_plug_added (GtkSocket *socket, GrdcPlugRdp *gp_rdp)
{
    grdc_plug_emit_signal (GRDC_PLUG (gp_rdp), "connect");
}

static void
grdc_plug_rdp_plug_removed (GtkSocket *socket, GrdcPlugRdp *gp_rdp)
{
}

static void
grdc_plug_rdp_set_error (GrdcPlugRdp *gp_rdp)
{
    gchar *buffer;
    gint len;

    buffer = GRDC_PLUG (gp_rdp)->error_message;
    lseek (gp_rdp->error_fd, 0, SEEK_SET);
    len = read (gp_rdp->error_fd, buffer, MAX_ERROR_LENGTH);
    buffer[len] = '\0';

    GRDC_PLUG (gp_rdp)->has_error = TRUE;
}

static void
grdc_plug_rdp_child_exit (GPid pid, gint status, gpointer data)
{
    GrdcPlugRdp *gp_rdp;

    gp_rdp = GRDC_PLUG_RDP (data);
    g_spawn_close_pid (pid);

    if (status != 0 && status != SIGTERM)
    {
        grdc_plug_rdp_set_error (gp_rdp);
    }
    close (gp_rdp->output_fd);
    close (gp_rdp->error_fd);

    grdc_plug_emit_signal (GRDC_PLUG (gp_rdp), "disconnect");
}

static gboolean
grdc_plug_rdp_open_connection (GrdcPlug *gp, GrdcFile *grdcfile)
{
    GrdcPlugRdp *gp_rdp = GRDC_PLUG_RDP (gp);
    gchar *argv[50];
    gchar buf[200];
    gchar *pos;
    gint argc;
    gint i;
    GError *error = NULL;
    gboolean ret;
    GtkWidget *dialog;

    gtk_widget_set_size_request (GTK_WIDGET (gp), grdcfile->resolution_width, grdcfile->resolution_height);

    argc = 0;
    argv[argc++] = g_strdup ("rdesktop");

    if (grdcfile->username && grdcfile->username[0] != '\0')
    {
        argv[argc++] = g_strdup ("-u");
        argv[argc++] = g_strdup (grdcfile->username);
    }

    if (grdcfile->domain && grdcfile->domain[0] != '\0')
    {
        argv[argc++] = g_strdup ("-d");
        argv[argc++] = g_strdup (grdcfile->domain);
    }

    if (grdcfile->password && grdcfile->password[0] != '\0')
    {
        argv[argc++] = g_strdup ("-p");
        argv[argc++] = g_strdup (grdcfile->password);
    }

    if (grdcfile->colordepth && grdcfile->colordepth[0] != '\0')
    {
        argv[argc++] = g_strdup ("-a");
        argv[argc++] = g_strdup (grdcfile->colordepth);
    }

    if (grdcfile->clientname && grdcfile->clientname[0] != '\0')
    {
        argv[argc++] = g_strdup ("-n");
        argv[argc++] = g_strdup (grdcfile->clientname);
    }

    if (grdcfile->bitmapcaching)
    {
        argv[argc++] = g_strdup ("-P");
    }

    if (grdcfile->compression)
    {
        argv[argc++] = g_strdup ("-z");
    }

    argv[argc++] = g_strdup ("-g");
    g_snprintf (buf, sizeof (buf), "%ix%i", grdcfile->resolution_width, grdcfile->resolution_height);
    argv[argc++] = g_strdup (buf);

    argv[argc++] = g_strdup ("-X");
    g_snprintf (buf, sizeof (buf), "%i", gtk_socket_get_id (GTK_SOCKET (gp_rdp->socket)));
    argv[argc++] = g_strdup (buf);

    pos = g_strrstr (grdcfile->server, ":");
    if (pos)
    {
        argv[argc++] = g_strdup (grdcfile->server);
    }
    else
    {
        g_snprintf (buf, sizeof (buf), "%s:3389", grdcfile->server);
        argv[argc++] = g_strdup (buf);
    }
    argv[argc++] = NULL;

    ret = g_spawn_async_with_pipes (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        &gp_rdp->pid, /* child_pid */
        NULL, /* standard_input */
        &gp_rdp->output_fd, /* standard_output */
        &gp_rdp->error_fd, /* standard_error */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (ret)
    {
        g_child_watch_add (gp_rdp->pid, grdc_plug_rdp_child_exit, gp_rdp);
        return TRUE;
    }
    else
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            error->message);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return FALSE;
    }
}

static gboolean
grdc_plug_rdp_close_connection (GrdcPlug *gp)
{
    GPid pid;

    pid = GRDC_PLUG_RDP (gp)->pid;
    if (pid)
    {
        /* If pid exists, "disconnect" signal will be emitted in the child_exit callback */
        kill (GRDC_PLUG_RDP (gp)->pid, SIGTERM);
    }
    else
    {
        grdc_plug_emit_signal (gp, "disconnect");
    }
    return FALSE;
}

gboolean
grdc_plug_rdp_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    return FALSE;
}

void
grdc_plug_rdp_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, gpointer data)
{
}

static void
grdc_plug_rdp_class_init (GrdcPlugRdpClass *klass)
{
    klass->parent_class.open_connection = grdc_plug_rdp_open_connection;
    klass->parent_class.close_connection = grdc_plug_rdp_close_connection;
    klass->parent_class.query_feature = grdc_plug_rdp_query_feature;
    klass->parent_class.call_feature = grdc_plug_rdp_call_feature;
}

static void
grdc_plug_rdp_destroy (GtkWidget *widget, gpointer data)
{
}

static void
grdc_plug_rdp_init (GrdcPlugRdp *gp_rdp)
{
    gp_rdp->socket = gtk_socket_new ();
    gtk_widget_show (gp_rdp->socket);
    g_signal_connect (G_OBJECT (gp_rdp->socket), "plug-added",
        G_CALLBACK (grdc_plug_rdp_plug_added), gp_rdp);
    g_signal_connect (G_OBJECT (gp_rdp->socket), "plug-removed",
        G_CALLBACK (grdc_plug_rdp_plug_removed), gp_rdp);
    gtk_fixed_put (GTK_FIXED (gp_rdp), gp_rdp->socket, 0, 0);

    g_signal_connect (G_OBJECT (gp_rdp), "destroy", G_CALLBACK (grdc_plug_rdp_destroy), NULL);

    gp_rdp->pid = 0;
    gp_rdp->output_fd = 0;
    gp_rdp->error_fd = 0;
}

GtkWidget*
grdc_plug_rdp_new (void)
{
    return GTK_WIDGET (g_object_new (GRDC_TYPE_PLUG_RDP, NULL));
}

