/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include "grdcfile.h"
#include "grdcfilemanager.h"
#include "grdcfileeditor.h"
#include "grdcconnectionwindow.h"
#include "grdcabout.h"
#include "grdcpixmaps.h"
#include "grdcpref.h"
#include "grdcprefdialog.h"
#include "grdcmain.h"

typedef struct _GrdcMain
{
    GtkWidget *main_window;
    GtkWidget *file_list;
    GtkListStore *file_store;
    GtkWidget *toolbar;
    GtkWidget *statusbar;

    GtkActionGroup *main_group;
    GtkActionGroup *file_sensitive_group;

    gboolean initialized;

    gchar *selected_filename;
    gchar *selected_name;
} GrdcMain;

enum
{
    PROTOCOL_COLUMN,
    NAME_COLUMN,
    GROUP_COLUMN,
    SERVER_COLUMN,
    FILENAME_COLUMN,
    N_COLUMNS
};

typedef enum
{
    GRDC_LAUNCH_MAIN,
    GRDC_LAUNCH_QUICK,
    GRDC_LAUNCH_FILE
} GrdcLaunchType;

static void
grdc_main_launcher (GrdcLaunchType launch_type, const gchar *filename)
{
    gint argc;
    gchar *argv[50];
    gint i;
    GError *error = NULL;
    gboolean ret;
    GtkWidget *dialog;

    argc = 0;
    argv[argc++] = g_strdup ("grdc");

    switch (launch_type)
    {
    case GRDC_LAUNCH_MAIN:
        break;

    case GRDC_LAUNCH_QUICK:
        argv[argc++] = g_strdup ("-q");
        break;

    case GRDC_LAUNCH_FILE:
        argv[argc++] = g_strdup ("-c");
        argv[argc++] = g_strdup (filename);
        break;
    }    

    argv[argc++] = NULL;

    ret = g_spawn_async (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        NULL, /* child_pid */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (!ret)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            error->message);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }
}

static void
grdc_main_quit (GrdcMain *grdcmain)
{
    GtkWidget *dialog;
    gint cnt;

    cnt = grdc_connection_window_count ();
    if (cnt > 0)
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (grdcmain->main_window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
            ngettext(
                "Found %i active connection.\nAre you sure to quit?",
                "Found %i active connections.\nAre you sure to quit?",
                cnt), cnt);
        if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES)
        {
            grdc_connection_window_quit_all ();
            gtk_widget_destroy (grdcmain->main_window);
        }
        gtk_widget_destroy (dialog);
    }
    else
    {
        gtk_widget_destroy (grdcmain->main_window);
    }
}

static gboolean
grdc_main_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    grdc_main_quit ((GrdcMain*) data);
    return TRUE;
}

static void
grdc_main_destroy (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

static void
grdc_main_list_callback (gpointer data, gpointer user_data)
{
    GtkTreeIter iter;
    GtkListStore *store;
    GrdcFile *grdcfile;

    grdcfile = (GrdcFile*) data;
    store = GTK_LIST_STORE (user_data);

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
        PROTOCOL_COLUMN, (g_strcmp0 (grdcfile->protocol, "VNC") == 0 ? grdc_pixmap_vnc (24) : grdc_pixmap_rdp (24)),
        NAME_COLUMN, grdcfile->name,
        GROUP_COLUMN, grdcfile->group,
        SERVER_COLUMN, grdcfile->server,
        FILENAME_COLUMN, grdcfile->filename,
        -1);
}

static void
grdc_main_clear_selection_data (GrdcMain *grdcmain)
{
    g_free (grdcmain->selected_filename);
    g_free (grdcmain->selected_name);
    grdcmain->selected_filename = NULL;
    grdcmain->selected_name = NULL;

    gtk_action_group_set_sensitive (grdcmain->file_sensitive_group, FALSE);
}

static gboolean
grdc_main_selection_func (
    GtkTreeSelection *selection,
    GtkTreeModel     *model,
    GtkTreePath      *path,
    gboolean          path_currently_selected,
    gpointer          user_data)
{
    GrdcMain *grdcmain;
    guint context_id;
    GtkTreeIter iter;
    gchar buf[1000];

    grdcmain = (GrdcMain*) user_data;
    if (!path_currently_selected)
    {
        grdc_main_clear_selection_data (grdcmain);

        context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (grdcmain->statusbar), "status");
        if (gtk_tree_model_get_iter (model, &iter, path))
        {
            gtk_tree_model_get (model, &iter,
                NAME_COLUMN, &grdcmain->selected_name,
                FILENAME_COLUMN, &grdcmain->selected_filename,
                -1);
            g_snprintf (buf, sizeof (buf), "%s (%s)", grdcmain->selected_name, grdcmain->selected_filename);
            gtk_statusbar_pop (GTK_STATUSBAR (grdcmain->statusbar), context_id);
            gtk_statusbar_push (GTK_STATUSBAR (grdcmain->statusbar), context_id, buf);

            gtk_action_group_set_sensitive (grdcmain->file_sensitive_group, TRUE);
        }
    }
    return TRUE;
}

static void
grdc_main_load_file_list (GrdcMain *grdcmain)
{
    gint n;
    gchar buf[200];
    guint context_id;

    grdc_main_clear_selection_data (grdcmain);

    gtk_list_store_clear (GTK_LIST_STORE (grdcmain->file_store));
    n = grdc_file_manager_iterate (grdc_main_list_callback, grdcmain->file_store);
    g_snprintf (buf, 200, ngettext("Total %i item.", "Total %i items.", n), n);
    context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (grdcmain->statusbar), "status");
    gtk_statusbar_pop (GTK_STATUSBAR (grdcmain->statusbar), context_id);
    gtk_statusbar_push (GTK_STATUSBAR (grdcmain->statusbar), context_id, buf);
}

static void
grdc_main_action_file_quick_connect (GtkAction *action, GrdcMain *grdcmain)
{
    if (grdc_pref.separated_process)
    {
        grdc_main_launcher (GRDC_LAUNCH_QUICK, NULL);
    }
    else
    {
        grdc_file_editor_open_temp (GTK_WINDOW (grdcmain->main_window), FALSE);
    }
}

static void
grdc_main_action_file_connect (GtkAction *action, GrdcMain *grdcmain)
{
    if (grdc_pref.separated_process)
    {
        grdc_main_launcher (GRDC_LAUNCH_FILE, grdcmain->selected_filename);
    }
    else
    {
        grdc_connection_window_open (GTK_WINDOW (grdcmain->main_window), grdcmain->selected_filename, FALSE);
    }
}

static void
grdc_main_action_file_new (GtkAction *action, GrdcMain *grdcmain)
{
    gint ret;

    ret = grdc_file_editor_create (GTK_WINDOW (grdcmain->main_window));
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_main_load_file_list (grdcmain);
    }
}

static void
grdc_main_action_file_edit (GtkAction *action, GrdcMain *grdcmain)
{
    gint ret;

    if (grdcmain->selected_filename == NULL) return;
    ret = grdc_file_editor_open (GTK_WINDOW (grdcmain->main_window), grdcmain->selected_filename, FALSE);
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_main_load_file_list (grdcmain);
    }
}

static void
grdc_main_action_file_delete (GtkAction *action, GrdcMain *grdcmain)
{
    GtkWidget *dialog;

    if (grdcmain->selected_filename == NULL) return;

    dialog = gtk_message_dialog_new (GTK_WINDOW (grdcmain->main_window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        _("Are you sure to delete '%s'"), grdcmain->selected_name);
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES)
    {
        g_unlink (grdcmain->selected_filename);
        grdc_main_load_file_list (grdcmain);
    }
    gtk_widget_destroy (dialog);
}

static void
grdc_main_action_file_preferences (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_pref_dialog_run (GTK_WINDOW (grdcmain->main_window), 0);
}

static void
grdc_main_action_file_quit (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_main_quit (grdcmain);
}

static void
grdc_main_action_view_toolbar (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_widget_show (grdcmain->toolbar);
    }
    else
    {
        gtk_widget_hide (grdcmain->toolbar);
    }
    if (grdcmain->initialized)
    {
        grdc_pref.hide_toolbar = !toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_view_statusbar (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_widget_show (grdcmain->statusbar);
    }
    else
    {
        gtk_widget_hide (grdcmain->statusbar);
    }
    if (grdcmain->initialized)
    {
        grdc_pref.hide_statusbar = !toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_view_small_toolbutton (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_toolbar_set_icon_size (GTK_TOOLBAR (grdcmain->toolbar), GTK_ICON_SIZE_MENU);
    }
    else
    {
        gtk_toolbar_unset_icon_size (GTK_TOOLBAR (grdcmain->toolbar));
    }
    if (grdcmain->initialized)
    {
        grdc_pref.small_toolbutton = toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_help_about (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_about_open (GTK_WINDOW (grdcmain->main_window));
}

static const gchar *grdc_main_ui_xml = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu name='FileMenu' action='File'>"
"      <menuitem name='FileQuickConnectMenu' action='FileQuickConnect'/>"
"      <menuitem name='FileConnectMenu' action='FileConnect'/>"
"      <separator/>"
"      <menuitem name='FileNewMenu' action='FileNew'/>"
"      <menuitem name='FileEditMenu' action='FileEdit'/>"
"      <menuitem name='FileDeleteMenu' action='FileDelete'/>"
"      <separator/>"
"      <menuitem name='FilePreferencesMenu' action='FilePreferences'/>"
"      <separator/>"
"      <menuitem name='FileQuitMenu' action='FileQuit'/>"
"    </menu>"
"    <menu name='ViewMenu' action='View'>"
"      <menuitem name='ViewToolbarMenu' action='ViewToolbar'/>"
"      <menuitem name='ViewStatusbarMenu' action='ViewStatusbar'/>"
"      <separator/>"
"      <menuitem name='ViewSmallToolbuttonMenu' action='ViewSmallToolbutton'/>"
"    </menu>"
"    <menu name='HelpMenu' action='Help'>"
"      <menuitem name='HelpAboutMenu' action='HelpAbout'/>"
"    </menu>"
"  </menubar>"
"  <toolbar name='ToolBar'>"
"    <toolitem action='FileQuickConnect'/>"
"    <toolitem action='FileConnect'/>"
"    <separator/>"
"    <toolitem action='FileNew'/>"
"    <toolitem action='FileEdit'/>"
"    <toolitem action='FileDelete'/>"
"    <separator/>"
"    <toolitem action='FilePreferences'/>"
"    <toolitem action='FileQuit'/>"
"  </toolbar>"
"</ui>";

static const GtkActionEntry grdc_main_ui_menu_entries[] =
{
    { "File", NULL, N_("_File") },
    { "View", NULL, N_("_View") },
    { "Help", NULL, N_("_Help") },

    { "FileQuickConnect", GTK_STOCK_JUMP_TO, N_("Quick Connect"), "<control>Q",
        N_("Open a quick connection"),
        G_CALLBACK (grdc_main_action_file_quick_connect) },

    { "FileNew", GTK_STOCK_NEW, NULL, "<control>N",
        N_("Create a new remote desktop file"),
        G_CALLBACK (grdc_main_action_file_new) },

    { "FilePreferences", GTK_STOCK_PREFERENCES, NULL, "<control>P",
        N_("Open the preferences dialog"),
        G_CALLBACK (grdc_main_action_file_preferences) },

    { "FileQuit", GTK_STOCK_QUIT, NULL, "<control>X",
        N_("Quit Grdc"),
        G_CALLBACK (grdc_main_action_file_quit) },

    { "HelpAbout", GTK_STOCK_ABOUT, NULL, NULL,
        NULL,
        G_CALLBACK (grdc_main_action_help_about) }
};

static const GtkActionEntry grdc_main_ui_file_sensitive_menu_entries[] =
{
    { "FileConnect", GTK_STOCK_CONNECT, NULL, "<control>C",
        N_("Open the connection to the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_connect) },

    { "FileEdit", GTK_STOCK_EDIT, NULL, "<control>E",
        N_("Edit the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_edit) },

    { "FileDelete", GTK_STOCK_DELETE, NULL, "<control>D",
        N_("Delete the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_delete) }
};

static const GtkToggleActionEntry grdc_main_ui_toggle_menu_entries[] =
{
    { "ViewToolbar", NULL, N_("Toolbar"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_toolbar), TRUE },

    { "ViewStatusbar", NULL, N_("Statusbar"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_statusbar), TRUE },

    { "ViewSmallToolbutton", NULL, N_("Small Toolbar Buttons"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_small_toolbutton), FALSE }
};

static gboolean
grdc_main_file_list_on_button_press (GtkWidget *widget, GdkEventButton *event, GrdcMain *grdcmain)
{
    if (event->type == GDK_2BUTTON_PRESS && grdcmain->selected_filename)
    {
        grdc_main_action_file_connect (NULL, grdcmain);
    }
    return FALSE;
}

void
grdc_main_open (void)
{
    GrdcMain *grdcmain;
    GtkWidget *vbox;
    GtkWidget *menubar;
    GtkUIManager *uimanager;
    GtkActionGroup *action_group;
    GtkWidget *scrolledwindow;
    GtkWidget *tree;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GError *error;

    grdcmain = g_new (GrdcMain, 1);

    grdcmain->initialized = FALSE;
    grdcmain->selected_filename = NULL;
    grdcmain->selected_name = NULL;

    /* Create main window */
    grdcmain->main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    g_signal_connect (G_OBJECT (grdcmain->main_window), "delete-event", G_CALLBACK (grdc_main_delete_event), grdcmain);
    g_signal_connect (G_OBJECT (grdcmain->main_window), "destroy", G_CALLBACK (grdc_main_destroy), grdcmain);
    gtk_container_set_border_width (GTK_CONTAINER (grdcmain->main_window), 2);
    gtk_window_set_title (GTK_WINDOW (grdcmain->main_window), _("Remote Desktop Client"));
    gtk_window_set_default_size (GTK_WINDOW (grdcmain->main_window), 600, 400);
    gtk_window_set_position (GTK_WINDOW (grdcmain->main_window), GTK_WIN_POS_CENTER);
    gtk_window_set_icon (GTK_WINDOW (grdcmain->main_window), grdc_pixmap_logo ());

    /* Create the main container */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (grdcmain->main_window), vbox);
    gtk_widget_show (vbox);

    /* Create the menubar and toolbar */
    uimanager = gtk_ui_manager_new ();

    action_group = gtk_action_group_new ("GrdcMainActions");
    gtk_action_group_set_translation_domain (action_group, NULL);
    gtk_action_group_add_actions (action_group, grdc_main_ui_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_menu_entries), grdcmain);
    gtk_action_group_add_toggle_actions (action_group, grdc_main_ui_toggle_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_toggle_menu_entries), grdcmain);

    gtk_ui_manager_insert_action_group (uimanager, action_group, 0);
    g_object_unref (action_group);
    grdcmain->main_group = action_group;

    action_group = gtk_action_group_new ("GrdcMainFileSensitiveActions");
    gtk_action_group_set_translation_domain (action_group, NULL);
    gtk_action_group_add_actions (action_group, grdc_main_ui_file_sensitive_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_file_sensitive_menu_entries), grdcmain);

    gtk_ui_manager_insert_action_group (uimanager, action_group, 0);
    g_object_unref (action_group);
    grdcmain->file_sensitive_group = action_group;

    error = NULL;
    gtk_ui_manager_add_ui_from_string (uimanager, grdc_main_ui_xml, -1, &error);
    if (error)
    {
        g_message ("building menus failed: %s", error->message);
        g_error_free (error);
    }

    menubar = gtk_ui_manager_get_widget (uimanager, "/MenuBar");
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

    grdcmain->toolbar = gtk_ui_manager_get_widget (uimanager, "/ToolBar");
    gtk_box_pack_start (GTK_BOX (vbox), grdcmain->toolbar, FALSE, FALSE, 0);

    gtk_window_add_accel_group (GTK_WINDOW (grdcmain->main_window), gtk_ui_manager_get_accel_group (uimanager));

    /* Create the scrolled window for the file list */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

    /* Create the grdc file list */
    grdcmain->file_store = gtk_list_store_new (N_COLUMNS,
        GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);

    tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (grdcmain->file_store));

    column = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_title (column, _("Name"));
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, NAME_COLUMN);
    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_add_attribute (column, renderer, "pixbuf", PROTOCOL_COLUMN);
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_add_attribute (column, renderer, "text", NAME_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Group"),
        renderer, "text", GROUP_COLUMN, NULL);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, GROUP_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Server"),
        renderer, "text", SERVER_COLUMN, NULL);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, SERVER_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

    gtk_container_add (GTK_CONTAINER (scrolledwindow), tree);
    gtk_widget_show (tree);

    gtk_tree_selection_set_select_function (
        gtk_tree_view_get_selection (GTK_TREE_VIEW (tree)),
        grdc_main_selection_func, grdcmain, NULL);
    g_signal_connect (G_OBJECT (tree), "button-press-event",
        G_CALLBACK (grdc_main_file_list_on_button_press), grdcmain);

    grdcmain->file_list = tree;

    /* Create statusbar */
    grdcmain->statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), grdcmain->statusbar, FALSE, FALSE, 0);
    gtk_widget_show (grdcmain->statusbar);

    /* Prepare the data */
    grdc_main_load_file_list (grdcmain);

    /* Load the preferences */
    if (grdc_pref.hide_toolbar)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewToolbar")), FALSE);
    }
    if (grdc_pref.hide_statusbar)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewStatusbar")), FALSE);
    }
    if (grdc_pref.small_toolbutton)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewSmallToolbutton")), TRUE);
    }

    /* Show the main window */
    gtk_widget_show (grdcmain->main_window);

    grdcmain->initialized = TRUE;
}

