/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcfile.h"

GrdcFile*
grdc_file_new_temp (void)
{
    GrdcFile *gf;

    /* Try to load from the preference file for default settings first */
    gf = grdc_file_load (grdc_pref_file);

    if (!gf)
    {
        gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));

        gf->filename = NULL;
        gf->name = NULL;
        gf->group = NULL;
        gf->server = NULL;
        gf->protocol = g_strdup ("RDP");
        gf->username = NULL;
        gf->password = NULL;
        gf->domain = NULL;
        gf->clientname = NULL;
        gf->resolution = g_strdup ("AUTO");
        gf->colordepth = g_strdup ("8");
        gf->quality = g_strdup ("0");
        gf->bitmapcaching = FALSE;
        gf->compression = FALSE;
        gf->showcursor = FALSE;
        gf->viewonly = FALSE;
        gf->viewmode = 0;
    }

    g_free (gf->name);
    gf->name = g_strdup (_("New Connection"));

    return gf;
}

GrdcFile*
grdc_file_new (void)
{
    GrdcFile *gf;
    GTimeVal gtime;
    gchar buf[MAX_PATH_LEN];

    g_get_current_time (&gtime);
    g_snprintf (buf, MAX_PATH_LEN, "%s/.grdc/%li%03li.grdc", g_get_home_dir (),
        gtime.tv_sec, gtime.tv_usec / 1000);

    gf = grdc_file_new_temp ();
    gf->filename = g_strdup (buf);

    return gf;
}

GrdcFile*
grdc_file_load (const gchar *filename)
{
    GKeyFile *gkeyfile;
    GrdcFile *gf;

    gkeyfile = g_key_file_new ();

    if (!g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL)) return NULL;

    if (g_key_file_has_key (gkeyfile, "grdc", "name", NULL))
    {
        gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));

        gf->filename = g_strdup (filename);
        gf->name = g_key_file_get_string (gkeyfile, "grdc", "name", NULL);
        gf->group = g_key_file_get_string (gkeyfile, "grdc", "group", NULL);
        gf->server = g_key_file_get_string (gkeyfile, "grdc", "server", NULL);
        gf->protocol = g_key_file_get_string (gkeyfile, "grdc", "protocol", NULL);
        gf->username = g_key_file_get_string (gkeyfile, "grdc", "username", NULL);
        gf->password = g_key_file_get_string (gkeyfile, "grdc", "password", NULL);
        gf->domain = g_key_file_get_string (gkeyfile, "grdc", "domain", NULL);
        gf->clientname = g_key_file_get_string (gkeyfile, "grdc", "clientname", NULL);
        gf->resolution = g_key_file_get_string (gkeyfile, "grdc", "resolution", NULL);
        gf->colordepth = g_key_file_get_string (gkeyfile, "grdc", "colordepth", NULL);
        gf->quality = g_key_file_get_string (gkeyfile, "grdc", "quality", NULL);
        gf->bitmapcaching = g_key_file_get_boolean (gkeyfile, "grdc", "bitmapcaching", NULL);
        gf->compression = g_key_file_get_boolean (gkeyfile, "grdc", "compression", NULL);
        gf->showcursor = g_key_file_get_boolean (gkeyfile, "grdc", "showcursor", NULL);
        gf->viewonly = FALSE;
        gf->viewmode = g_key_file_get_integer (gkeyfile, "grdc", "viewmode", NULL);
    }
    else
    {
        gf = NULL;
    }

    g_key_file_free (gkeyfile);

    return gf;
}

void
grdc_file_save (GrdcFile *gf)
{
    GKeyFile *gkeyfile;
    gchar *content;
    gsize length;

    if (gf->filename == NULL) return;

    gkeyfile = g_key_file_new ();

    g_key_file_load_from_file (gkeyfile, gf->filename, G_KEY_FILE_NONE, NULL);

    g_key_file_set_string (gkeyfile, "grdc", "name", (gf->name ? gf->name : ""));
    g_key_file_set_string (gkeyfile, "grdc", "group", (gf->group ? gf->group : ""));
    g_key_file_set_string (gkeyfile, "grdc", "server", (gf->server ? gf->server : ""));
    g_key_file_set_string (gkeyfile, "grdc", "protocol", (gf->protocol ? gf->protocol : ""));
    g_key_file_set_string (gkeyfile, "grdc", "username", (gf->username ? gf->username : ""));
    g_key_file_set_string (gkeyfile, "grdc", "password", (gf->password ? gf->password : ""));
    g_key_file_set_string (gkeyfile, "grdc", "domain", (gf->domain ? gf->domain : ""));
    g_key_file_set_string (gkeyfile, "grdc", "clientname", (gf->clientname ? gf->clientname : ""));
    g_key_file_set_string (gkeyfile, "grdc", "resolution", (gf->resolution ? gf->resolution : ""));
    g_key_file_set_string (gkeyfile, "grdc", "colordepth", (gf->colordepth ? gf->colordepth : ""));
    g_key_file_set_string (gkeyfile, "grdc", "quality", (gf->quality ? gf->quality : ""));
    g_key_file_set_boolean (gkeyfile, "grdc", "bitmapcaching", gf->bitmapcaching);
    g_key_file_set_boolean (gkeyfile, "grdc", "compression", gf->compression);
    g_key_file_set_boolean (gkeyfile, "grdc", "showcursor", gf->showcursor);
    g_key_file_set_integer (gkeyfile, "grdc", "viewmode", gf->viewmode);

    content = g_key_file_to_data (gkeyfile, &length, NULL);
    g_file_set_contents (gf->filename, content, length, NULL);

    g_key_file_free (gkeyfile);
    g_free (content);
}

void
grdc_file_free (GrdcFile *grdcfile)
{
    if (grdcfile == NULL) return;

    g_free (grdcfile->filename);
    g_free (grdcfile->name);
    g_free (grdcfile->group);
    g_free (grdcfile->server);
    g_free (grdcfile->protocol);
    g_free (grdcfile->username);
    g_free (grdcfile->password);
    g_free (grdcfile->domain);
    g_free (grdcfile->clientname);
    g_free (grdcfile->resolution);
    g_free (grdcfile->colordepth);
    g_free (grdcfile->quality);
    g_free (grdcfile);
}

GrdcFile*
grdc_file_dup (GrdcFile *grdcfile)
{
    GrdcFile *gf;

    gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));
    gf->filename = g_strdup (grdcfile->filename);
    gf->name = g_strdup (grdcfile->name);
    gf->group = g_strdup (grdcfile->group);
    gf->server = g_strdup (grdcfile->server);
    gf->protocol = g_strdup (grdcfile->protocol);
    gf->username = g_strdup (grdcfile->username);
    gf->password = g_strdup (grdcfile->password);
    gf->domain = g_strdup (grdcfile->domain);
    gf->clientname = g_strdup (grdcfile->clientname);
    gf->resolution = g_strdup (grdcfile->resolution);
    gf->colordepth = g_strdup (grdcfile->colordepth);
    gf->quality = g_strdup (grdcfile->quality);
    gf->bitmapcaching = grdcfile->bitmapcaching;
    gf->compression = grdcfile->compression;
    gf->showcursor = grdcfile->showcursor;
    gf->viewmode = grdcfile->viewmode;

    return gf;
}

void
grdc_file_update_screen_resolution (GrdcFile *grdcfile)
{
    GdkScreen *screen;
    gchar *tmp, *pos;


    if (grdcfile->resolution == NULL ||
        g_strcmp0 (grdcfile->resolution, "AUTO") == 0 ||
        g_strrstr (grdcfile->resolution, "x") == NULL)
    {
        screen = gdk_screen_get_default ();
        grdcfile->resolution_width = gdk_screen_get_width (screen);
        grdcfile->resolution_height = gdk_screen_get_height (screen);
    }
    else
    {
        tmp = g_strdup (grdcfile->resolution);
        pos = g_strrstr (tmp, "x");
        *pos++ = '\0';
        grdcfile->resolution_width = MAX (100, MIN (4096, (gint) g_ascii_strtoll(tmp, NULL, 0)));
        grdcfile->resolution_height = MAX (100, MIN (4096, (gint) g_ascii_strtoll(pos, NULL, 0)));
    }
}

