/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>
#include <panel-applet.h>
#include "config.h"
#include "grdcpixmaps.h"
#include "grdcappletmenuitem.h"
#include "grdcstringarray.h"
#include "grdcavahi.h"

#define GRDC_APPLET_FACTORY_IID "OAFIID:Grdc_Applet_Factory"
#define GRDC_APPLET_IID         "OAFIID:Grdc_Applet"

#define MAX_PATH_LEN 255

typedef struct _GrdcAppletData
{
    PanelApplet *applet;

    GtkWidget *image;
    GtkWidget *popup_menu;
    guint32 popup_time;

    gint prev_size;

    gchar *menu_group;
    gint menu_group_count;
    GtkWidget *menu_group_widget;

    GrdcAvahi *avahi;
} GrdcAppletData;

typedef enum
{
    GRDC_LAUNCH_MAIN,
    GRDC_LAUNCH_PREF,
    GRDC_LAUNCH_QUICK,
    GRDC_LAUNCH_FILE,
    GRDC_LAUNCH_EDIT,
    GRDC_LAUNCH_NEW
} GrdcLaunchType;

static void
grdc_applet_launcher (GrdcLaunchType launch_type, const gchar *filename,
    const gchar *server, const gchar *protocol)
{
    gint argc;
    gchar *argv[50];
    gint i;
    GError *error = NULL;
    gboolean ret;
    GtkWidget *dialog;

    argc = 0;
    argv[argc++] = g_strdup ("grdc");

    switch (launch_type)
    {
    case GRDC_LAUNCH_MAIN:
        break;

    case GRDC_LAUNCH_PREF:
        argv[argc++] = g_strdup ("-p");
        argv[argc++] = g_strdup ("3");
        break;

    case GRDC_LAUNCH_QUICK:
        argv[argc++] = g_strdup ("-q");
        break;

    case GRDC_LAUNCH_FILE:
        argv[argc++] = g_strdup ("-c");
        argv[argc++] = g_strdup (filename);
        break;

    case GRDC_LAUNCH_EDIT:
        argv[argc++] = g_strdup ("-e");
        argv[argc++] = g_strdup (filename);
        break;

    case GRDC_LAUNCH_NEW:
        argv[argc++] = g_strdup ("-n");
        break;

    }

    if (server)
    {
        argv[argc++] = g_strdup ("-s");
        argv[argc++] = g_strdup (server);
    }
    if (protocol)
    {
        argv[argc++] = g_strdup ("-t");
        argv[argc++] = g_strdup (protocol);
    }

    argv[argc++] = NULL;

    ret = g_spawn_async (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        NULL, /* child_pid */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (!ret)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            error->message, NULL);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }
}

static gchar*
grdc_applet_get_pref_string (const gchar *key)
{
    gchar filename[MAX_PATH_LEN];
    GKeyFile *gkeyfile;
    gchar *value;

    g_snprintf (filename, sizeof (filename), "%s/.grdc/grdc.pref", g_get_home_dir ());
    gkeyfile = g_key_file_new ();
    if (g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL))
    {
        value = g_key_file_get_string (gkeyfile, "grdc_pref", key, NULL);
    }
    else
    {
        value = NULL;
    }
    g_key_file_free (gkeyfile);

    return value;
}

static gboolean
grdc_applet_get_pref_boolean (const gchar *key)
{
    gchar filename[MAX_PATH_LEN];
    GKeyFile *gkeyfile;
    gboolean value;

    g_snprintf (filename, sizeof (filename), "%s/.grdc/grdc.pref", g_get_home_dir ());
    gkeyfile = g_key_file_new ();
    if (g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL))
    {
        value = g_key_file_get_boolean (gkeyfile, "grdc_pref", key, NULL);
    }
    else
    {
        value = FALSE;
    }
    g_key_file_free (gkeyfile);

    return value;
}

static void
grdc_applet_set_pref_boolean (const gchar *key, gboolean value)
{
    gchar filename[MAX_PATH_LEN];
    GKeyFile *gkeyfile;
    gchar *content;
    gsize length;

    g_snprintf (filename, sizeof (filename), "%s/.grdc/grdc.pref", g_get_home_dir ());
    gkeyfile = g_key_file_new ();
    if (g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL))
    {
        g_key_file_set_boolean (gkeyfile, "grdc_pref", key, value);
        content = g_key_file_to_data (gkeyfile, &length, NULL);
        g_file_set_contents (filename, content, length, NULL);
        g_free (content);
    }
    g_key_file_free (gkeyfile);
}

static void
grdc_applet_destroy (GtkWidget *widget, GrdcAppletData *appdata)
{
    grdc_avahi_free (appdata->avahi);
    g_free (appdata);
}

static void
grdc_applet_size_allocate (GtkWidget *widget, GtkAllocation *allocation, GrdcAppletData *appdata)
{
    GdkPixbuf *scaled;
    gint size = 0;

    switch (panel_applet_get_orient (appdata->applet))
    {
    case PANEL_APPLET_ORIENT_UP:
    case PANEL_APPLET_ORIENT_DOWN:
        size = allocation->height;
        break;
    case PANEL_APPLET_ORIENT_LEFT:
    case PANEL_APPLET_ORIENT_RIGHT:
        size = allocation->width;
        break;
    }

    if (size < 10) return;
    if (size == appdata->prev_size) return;

    appdata->prev_size = size;

    scaled = grdc_pixmap (GRDC_PIXMAP_TYPE_LOGO, size);
    gtk_image_set_from_pixbuf (GTK_IMAGE (appdata->image), scaled);
    g_object_unref (scaled);
}

static void
grdc_applet_menu_open_main (GrdcAppletMenuItem *menuitem, gpointer data)
{
    grdc_applet_launcher (GRDC_LAUNCH_MAIN, NULL, NULL, NULL);
}

static void
grdc_applet_menu_open_pref (GrdcAppletMenuItem *menuitem, gpointer data)
{
    grdc_applet_launcher (GRDC_LAUNCH_PREF, NULL, NULL, NULL);
}

static void
grdc_applet_menu_open_quick (GrdcAppletMenuItem *menuitem, GdkEventButton *event, GrdcAppletData *appdata)
{
    grdc_applet_launcher (GRDC_LAUNCH_QUICK, NULL, NULL, NULL);
}

static void
grdc_applet_menu_open_file (GrdcAppletMenuItem *menuitem, GdkEventButton *event, GrdcAppletData *appdata)
{
    grdc_applet_launcher (event->button == 3 ? GRDC_LAUNCH_EDIT : GRDC_LAUNCH_FILE, menuitem->filename, NULL, NULL);
}

static void
grdc_applet_menu_open_discovered (GrdcAppletMenuItem *menuitem, GdkEventButton *event, GrdcAppletData *appdata)
{
    grdc_applet_launcher (event->button == 3 ? GRDC_LAUNCH_NEW : GRDC_LAUNCH_QUICK, NULL, menuitem->name, menuitem->protocol);
}

static void
grdc_applet_popdown_menu (GtkWidget *widget, GrdcAppletData *appdata)
{
    appdata->popup_menu = NULL;
    gtk_widget_set_state (GTK_WIDGET (appdata->applet), GTK_STATE_NORMAL);
    if (gtk_get_current_event_time () - appdata->popup_time <= 500)
    {
        grdc_applet_menu_open_main (NULL, NULL);
    }
}

static void
grdc_applet_popup_menu_position (GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer data)
{
    GdkScreen *screen;
    GtkRequisition req;
    gint tx, ty, size;
    GrdcAppletData *appdata;

    appdata = (GrdcAppletData*) data;
    size = GTK_WIDGET (appdata->image)->allocation.width;
    gdk_window_get_origin (GTK_WIDGET (appdata->applet)->window, &tx, &ty);
    gtk_widget_size_request (GTK_WIDGET (menu), &req);

    switch (panel_applet_get_orient (appdata->applet))
    {
    case PANEL_APPLET_ORIENT_UP:
        ty -= req.height;
        break;
    case PANEL_APPLET_ORIENT_DOWN:
        ty += size;
        break;
    case PANEL_APPLET_ORIENT_LEFT:
        tx -= req.width;
        break;
    case PANEL_APPLET_ORIENT_RIGHT:
        tx += size;
        break;
    }

    screen = gdk_screen_get_default ();
    tx = MIN (gdk_screen_get_width (screen) - req.width - 1, tx);
    ty = MIN (gdk_screen_get_height (screen) - req.height - 1, ty);

    *x = tx;
    *y = ty;
    *push_in = TRUE;
}

static void
grdc_applet_popup_menu_update_group (GrdcAppletData* appdata)
{
    gchar *label;

    if (appdata->menu_group)
    {
        if (!grdc_applet_get_pref_boolean ("applet_hide_count"))
        {
            label = g_strdup_printf ("%s (%i)", appdata->menu_group, appdata->menu_group_count);
            gtk_label_set_text (GTK_LABEL (gtk_bin_get_child (GTK_BIN (appdata->menu_group_widget))), label);
            g_free (label);
        }
        appdata->menu_group = NULL;
        appdata->menu_group_count = 0;
    }
}

static void
grdc_applet_popup_menu_add_item (gpointer data, gpointer user_data)
{
    GrdcAppletData *appdata;
    GrdcAppletMenuItem *item;
    GtkWidget *submenu;
    GtkWidget *image;

    appdata = (GrdcAppletData*) user_data;
    item = GRDC_APPLET_MENU_ITEM (data);

    if (item->group && item->group[0] != '\0')
    {
        if (!appdata->menu_group || g_strcmp0 (appdata->menu_group, item->group) != 0)
        {
            grdc_applet_popup_menu_update_group (appdata);

            appdata->menu_group = item->group;

            appdata->menu_group_widget = gtk_image_menu_item_new_with_label (appdata->menu_group);
            gtk_widget_show (appdata->menu_group_widget);
            gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), appdata->menu_group_widget);

            image = gtk_image_new_from_icon_name ((item->item_type == GRDC_APPLET_MENU_ITEM_DISCOVERED ?
                "folder-remote" : "folder"), GTK_ICON_SIZE_MENU);
            gtk_widget_show (image);
            gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (appdata->menu_group_widget), image);

            submenu = gtk_menu_new ();
            gtk_widget_show (submenu);
            gtk_menu_item_set_submenu (GTK_MENU_ITEM (appdata->menu_group_widget), submenu);
        }
        else
        {
            submenu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (appdata->menu_group_widget));
        }
        gtk_menu_shell_append (GTK_MENU_SHELL (submenu), GTK_WIDGET (item));
        appdata->menu_group_count++;
    }
    else
    {
        grdc_applet_popup_menu_update_group (appdata);
        gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), GTK_WIDGET (item));
    }
}

static void
grdc_applet_popup_menu (GtkWidget *widget, GdkEventButton *event, GrdcAppletData *appdata)
{
    GtkWidget *menuitem;
    GPtrArray *menuitem_array;
    gint button, event_time;
    gchar dirname[MAX_PATH_LEN];
    gchar filename[MAX_PATH_LEN];
    GDir *dir;
    gchar *groups;
    const gchar *name;
    gboolean quick_ontop;
    GHashTableIter iter;
    gchar *tmp;

    quick_ontop = grdc_applet_get_pref_boolean ("applet_quick_ontop");

    appdata->popup_time = gtk_get_current_event_time ();
    appdata->popup_menu = gtk_menu_new ();

    if (quick_ontop)
    {
        menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_QUICK);
        g_signal_connect (menuitem, "button-press-event",
            G_CALLBACK (grdc_applet_menu_open_quick), appdata);
        gtk_widget_show (menuitem);
        gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), menuitem);

        menuitem = gtk_separator_menu_item_new ();
        gtk_widget_show (menuitem);
        gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), menuitem);
    }

    g_snprintf (dirname, MAX_PATH_LEN, "%s/.grdc", g_get_home_dir ());
    dir = g_dir_open (dirname, 0, NULL);
    if (dir != NULL)
    {
        menuitem_array = g_ptr_array_new ();

        /* Iterate all remote desktop profiles */
        while ((name = g_dir_read_name (dir)) != NULL)
        {
            if (!g_str_has_suffix (name, ".grdc")) continue;
            g_snprintf (filename, MAX_PATH_LEN, "%s/%s", dirname, name);

            menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_FILE, filename);
            g_signal_connect (G_OBJECT (menuitem), "button-press-event",
                G_CALLBACK (grdc_applet_menu_open_file), appdata);
            gtk_widget_show (menuitem);

            g_ptr_array_add (menuitem_array, menuitem);
        }

        /* Iterate all discovered services from Avahi */
        if (appdata->avahi)
        {
            g_hash_table_iter_init (&iter, appdata->avahi->discovered_services);
            while (g_hash_table_iter_next (&iter, NULL, (gpointer*) &tmp))
            {
                menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_DISCOVERED, tmp);
                g_signal_connect (G_OBJECT (menuitem), "button-press-event",
                    G_CALLBACK (grdc_applet_menu_open_discovered), appdata);
                gtk_widget_show (menuitem);

                g_ptr_array_add (menuitem_array, menuitem);
            }
        }

        tmp = grdc_applet_get_pref_string ("groups");
        groups = g_strdup_printf ("%s%s%s", tmp, (tmp && tmp[0] ? "," : ""), _("Discovered"));
        g_free (tmp);
        g_ptr_array_sort_with_data (menuitem_array, grdc_applet_menu_item_compare, groups);
        g_free (groups);

        appdata->menu_group = NULL;
        appdata->menu_group_count = 0;
        g_ptr_array_foreach (menuitem_array, grdc_applet_popup_menu_add_item, appdata);

        grdc_applet_popup_menu_update_group (appdata);

        g_ptr_array_free (menuitem_array, FALSE);
    }

    if (!quick_ontop)
    {
        menuitem = gtk_separator_menu_item_new ();
        gtk_widget_show (menuitem);
        gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), menuitem);

        menuitem = grdc_applet_menu_item_new (GRDC_APPLET_MENU_ITEM_QUICK);
        g_signal_connect (menuitem, "button-press-event",
            G_CALLBACK (grdc_applet_menu_open_quick), appdata);
        gtk_widget_show (menuitem);
        gtk_menu_shell_append (GTK_MENU_SHELL (appdata->popup_menu), menuitem);
    }

    if (event)
    {
        button = event->button;
        event_time = event->time;
    }
    else
    {
        button = 0;
        event_time = gtk_get_current_event_time ();
    }

    gtk_widget_set_state (GTK_WIDGET (appdata->applet), GTK_STATE_SELECTED);
    g_signal_connect (G_OBJECT (appdata->popup_menu), "deactivate", G_CALLBACK (grdc_applet_popdown_menu), appdata);

    gtk_menu_attach_to_widget (GTK_MENU (appdata->popup_menu), widget, NULL);
    gtk_widget_realize (appdata->popup_menu);
    gtk_menu_popup (GTK_MENU (appdata->popup_menu), NULL, NULL,
        grdc_applet_popup_menu_position, appdata, button, event_time);
}

static gboolean
grdc_applet_button_press_event (GtkWidget *widget, GdkEventButton *event, GrdcAppletData *appdata)
{
    if (event->button == 1)
    {
        switch (event->type)
        {
        case GDK_BUTTON_PRESS:
            grdc_applet_popup_menu (widget, event, appdata);
            break;
        default:
            break;
        }
        return TRUE;
    }
    return FALSE;
}

static void
grdc_applet_menu_enable_avahi (BonoboUIComponent *uic,
    const char                  *path,
    Bonobo_UIComponent_EventType type,
    const char                  *state,
    gpointer                     userdata)
{
    GrdcAppletData *appdata = (GrdcAppletData*) userdata;

    if (!appdata->avahi) return;

    if (state[0] == '1')
    {
        grdc_applet_set_pref_boolean ("applet_enable_avahi", TRUE);
        if (!appdata->avahi->started) grdc_avahi_start (appdata->avahi);
    }
    else
    {
        grdc_applet_set_pref_boolean ("applet_enable_avahi", FALSE);
        grdc_avahi_stop (appdata->avahi);
    }
}

static void
grdc_applet_menu_about (BonoboUIComponent *uic, GrdcAppletData *appdata, const char *verb)
{
    const gchar *authors[] =
    {
        "Vic Lee <llyzs@163.com>",
        NULL
    };
    const gchar *licenses[] =
    {
        N_("Grdc is free software; you can redistribute it and/or modify "
           "it under the terms of the GNU General Public License as published by "
           "the Free Software Foundation; either version 2 of the License, or "
           "(at your option) any later version."),
        N_("Grdc is distributed in the hope that it will be useful, "
           "but WITHOUT ANY WARRANTY; without even the implied warranty of "
           "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the "
           "GNU General Public License for more details."),
        N_("You should have received a copy of the GNU General Public License "
           "along with this program; if not, write to the Free Software "
           "Foundation, Inc., 59 Temple Place, Suite 330, "
           "Boston, MA 02111-1307, USA.")
    };
    gchar *license = g_strjoin ("\n\n", _(licenses[0]), _(licenses[1]), _(licenses[2]), NULL);

    gtk_show_about_dialog (NULL,
        "program-name", "Grdc Applet",
        "version", PACKAGE_VERSION,
        "comments", _("GTK+/Gnome Remote Desktop Client Applet"),
        "authors", authors,
        "translator-credits", _("translator-credits"),
        "copyright", "Copyright (C) 2009 - Vic Lee",
        "license", license,
        "wrap-license", TRUE,
        "logo", GRDC_PIXMAP_LOGO,
        "website", "http://grdc.sourceforge.net",
        NULL);

    g_free (license);
}

static const BonoboUIVerb grdc_applet_menu_verbs [] =
{   
    BONOBO_UI_UNSAFE_VERB ("OpenMain", grdc_applet_menu_open_main),
    BONOBO_UI_UNSAFE_VERB ("OpenPref", grdc_applet_menu_open_pref),
    BONOBO_UI_UNSAFE_VERB ("About", grdc_applet_menu_about),
    BONOBO_UI_VERB_END
};

static const gchar *grdc_applet_menu_xml = 
"    <popup name='button3'>"
"      <menuitem name='OpenMain' verb='OpenMain' _label='%s' pixtype='stock' pixname='gtk-execute'  />"
"      <menuitem name='OpenPref' verb='OpenPref' _label='%s' pixtype='stock' pixname='gtk-preferences'  />"
"      <menuitem name='About'    verb='About'    _label='%s' pixtype='stock' pixname='gtk-about' />"
#ifdef HAVE_LIBAVAHI_CLIENT
"      <separator />"
"      <menuitem name='EnableAvahi' type='toggle' id='EnableAvahi' _label='%s' />"
#endif
"    </popup>"
;

static void
grdc_applet_create (PanelApplet *applet)
{
    BonoboUIComponent *popup_component;
    gchar buf[1000];
    GrdcAppletData *appdata;

    appdata = g_new0 (GrdcAppletData, 1);
    appdata->applet = applet;
    appdata->popup_menu = NULL;
    appdata->popup_time = 0;
    appdata->prev_size = 0;
    appdata->avahi = grdc_avahi_new ();

    appdata->image = gtk_image_new_from_pixbuf (NULL);
    gtk_widget_show (appdata->image);

    gtk_container_add (GTK_CONTAINER (applet), appdata->image);

    g_signal_connect (GTK_WIDGET (applet), "destroy", G_CALLBACK (grdc_applet_destroy), appdata);
    g_signal_connect (GTK_WIDGET (applet), "button-press-event", G_CALLBACK (grdc_applet_button_press_event), appdata);
    g_signal_connect (appdata->image, "size-allocate", G_CALLBACK (grdc_applet_size_allocate), appdata);

    g_snprintf (buf, sizeof (buf), grdc_applet_menu_xml, _("Open Main Window"), _("Preferences"),
        _("About")
#ifdef HAVE_LIBAVAHI_CLIENT
        , _("Enable Service Discovery")
#endif
        );
    panel_applet_setup_menu (applet, buf, grdc_applet_menu_verbs, appdata);

    popup_component = panel_applet_get_popup_component (applet);
    bonobo_ui_component_add_listener (popup_component, "EnableAvahi", grdc_applet_menu_enable_avahi, appdata);

    panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);
    panel_applet_set_background_widget (applet, GTK_WIDGET (applet));

    gtk_widget_show (GTK_WIDGET (applet));

    if (grdc_applet_get_pref_boolean ("applet_enable_avahi"))
    {
        bonobo_ui_component_set_prop (popup_component, "/commands/EnableAvahi", "state", "1", NULL);
        grdc_avahi_start (appdata->avahi);
    }
}

static gboolean
grdc_applet_factory (PanelApplet *applet, const gchar *iid, gpointer data)
{
    if (strcmp (iid, GRDC_APPLET_IID) != 0) return FALSE;

    bindtextdomain (GETTEXT_PACKAGE, GRDC_GNOME_LOCALEDIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);

    grdc_pixmap_init ();

    grdc_applet_create (applet);

    return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY (GRDC_APPLET_FACTORY_IID,
                            PANEL_TYPE_APPLET,
                            _("Grdc Remote Desktop Client Applet"),
                            "0", 
                            grdc_applet_factory, 
                            NULL)

