/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.caching.configuration

import org.gradle.integtests.fixtures.AbstractIntegrationSpec
import spock.lang.Unroll

class BuildCacheConfigurationIntegrationTest extends AbstractIntegrationSpec {
    String cacheDir = temporaryFolder.file("cache-dir").createDir().absoluteFile.toURI().toString()
    String buildSrcCacheDir = temporaryFolder.file("buildSrc-cache-dir").createDir().absoluteFile.toURI().toString()

    def setup() {
        buildFile << """
            task assertLocalCacheConfigured {
                doLast {
                    assert gradle.services.get(BuildCacheConfiguration).local.directory == "$cacheDir"
                }
            }
        """
    }

    def "can configure with settings.gradle"() {
        settingsFile << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = '$cacheDir'
                }
            }
        """
        expect:
        succeeds("assertLocalCacheConfigured")
    }

    def "can configure with init script"() {
        def initScript = file("initBuildCache.gradle") << """
            gradle.settingsEvaluated { settings ->
                settings.buildCache {
                    local(DirectoryBuildCache) {
                        directory = '$cacheDir'
                    }
                }
            }
        """
        expect:
        executer.usingInitScript(initScript)
        succeeds("assertLocalCacheConfigured")
    }

    def "configuration in init script wins over settings.gradle"() {
        def initScript = file("initBuildCache.gradle") << """
            gradle.settingsEvaluated { settings ->
                settings.buildCache {
                    local(DirectoryBuildCache) {
                        directory = '$cacheDir'
                    }
                }
            }
        """
        settingsFile << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = "wrong"
                }
            }
        """
        expect:
        executer.usingInitScript(initScript)
        succeeds("assertLocalCacheConfigured")
    }

    def "buildSrc and project builds configured separately"() {
        def configuration = { path ->
            """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = "$path"
                }
            }
            """
        }
        settingsFile << configuration(cacheDir)
        file("buildSrc/settings.gradle") << configuration(buildSrcCacheDir)
        file("buildSrc/build.gradle") << """
            apply plugin: 'groovy'

            task assertLocalCacheConfigured {
                doLast {
                    assert gradle.services.get(BuildCacheConfiguration).local.directory == "$buildSrcCacheDir"
                }
            }
            
            build.dependsOn assertLocalCacheConfigured
        """
        expect:
        succeeds("assertLocalCacheConfigured")
    }

    @Unroll
    def "last #cache cache configuration wins"() {
        settingsFile << """
            import org.gradle.caching.internal.NoOpBuildCacheService

            class CustomBuildCache extends AbstractBuildCache {}
            class AnotherBuildCache extends AbstractBuildCache {}

            class CustomBuildCacheFactory implements BuildCacheServiceFactory<CustomBuildCache> {
                @Override BuildCacheService createBuildCacheService(CustomBuildCache configuration, Describer describer) { 
                    new NoOpBuildCacheService() 
                }
            }

            class AnotherBuildCacheFactory implements BuildCacheServiceFactory<AnotherBuildCache> {
                @Override BuildCacheService createBuildCacheService(AnotherBuildCache configuration, Describer describer) { 
                    new NoOpBuildCacheService() 
                }
            }
            
            buildCache {
                registerBuildCacheService(CustomBuildCache, CustomBuildCacheFactory)
                registerBuildCacheService(AnotherBuildCache, AnotherBuildCacheFactory)

                $cache(CustomBuildCache)
                $cache(AnotherBuildCache)
            }
            
            assert buildCache.$cache instanceof AnotherBuildCache
        """
        expect:
        succeeds("help")

        where:
        cache << ["local", "remote"]
    }

    def "disables remote cache with --offline"() {
        settingsFile << """
            import org.gradle.caching.internal.NoOpBuildCacheService
            class CustomBuildCache extends AbstractBuildCache {}
            
            class CustomBuildCacheFactory implements BuildCacheServiceFactory<CustomBuildCache> {
                @Override BuildCacheService createBuildCacheService(CustomBuildCache configuration, Describer describer) { 
                    new NoOpBuildCacheService() 
                }
            }
            
            buildCache {
                registerBuildCacheService(CustomBuildCache, CustomBuildCacheFactory)
                
                remote(CustomBuildCache)
            }            
        """
        expect:
        succeeds("help", "--build-cache", "--offline", "--info")
        result.output.contains("Remote build cache is disabled when running with --offline.")
    }

    def "unregistered build cache type is reported even when disabled"() {
        settingsFile << """
            class CustomBuildCache extends AbstractBuildCache {}
            
            buildCache {
                remote(CustomBuildCache) {
                    enabled = false
                }
            }            
        """
        expect:
        fails("help")
        failureHasCause("Build cache type 'CustomBuildCache' has not been registered.")
    }

    def "emits a useful incubating message when using the build cache"() {
        when:
        executer.withBuildCacheEnabled()
        succeeds("tasks", "--info")
        then:
        result.assertOutputContains("Using local directory build cache")
    }

    def "emits cache descriptions for buildSrc and main build"() {
        settingsFile << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = file("local-cache")
                }
            }
        """
        file("buildSrc/settings.gradle") << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = file("local-cache")
                }
            }
        """
        when:
        executer.withBuildCacheEnabled()
        succeeds("tasks", "--info")
        then:
        result.assertOutputContains "Using local directory build cache for build ':buildSrc' (location = ${file("buildSrc/local-cache")}, targetSize = 5 GB)."
        result.assertOutputContains "Using local directory build cache for the root build (location = ${file("local-cache")}, targetSize = 5 GB)."
    }

    def "command-line --no-build-cache wins over system property"() {
        file("gradle.properties") << """
            org.gradle.caching=true
        """
        executer.withArgument("--no-build-cache")
        when:
        succeeds("tasks", "--info")
        then:
        !result.output.contains("Using local directory build cache")
    }

    def "command-line --build-cache wins over system property"() {
        file("gradle.properties") << """
            org.gradle.caching=false
        """
        executer.withArgument("--build-cache")
        when:
        succeeds("tasks", "--info")
        then:
        result.assertOutputContains("Using local directory build cache")
    }

    def "does not use the build cache when it is not enabled"() {
        given:
        buildFile << customTaskCode()
        when:
        // Disable the local build cache
        settingsFile << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = '$cacheDir'
                    enabled = false
                }
            }
        """
        executer.withBuildCacheEnabled()
        succeeds("customTask")
        then:
        result.assertOutputContains("Task output caching is enabled, but no build caches are configured or enabled.")
        and:
        file("local-cache").assertDoesNotExist()
    }

    def "does not populate the build cache when we cannot push to it"() {
        given:
        buildFile << customTaskCode()
        when:
        // Disable pushing to the local build cache
        settingsFile << """
            buildCache {
                local(DirectoryBuildCache) {
                    directory = file("local-cache")
                    push = false
                }
            }
        """
        executer.withBuildCacheEnabled()
        succeeds("customTask", "--info")
        then:
        result.assertOutputContains("Using local directory build cache for the root build (pull-only, location = ${file("local-cache")}, targetSize = 5 GB).")
        and:
        !file("local-cache").listFiles().any { it.name ==~ /\p{XDigit}{32}/}
    }

    private static String customTaskCode() {
        """
            @CacheableTask
            class CustomTask extends DefaultTask {
                @OutputFile
                File outputFile = new File(temporaryDir, "output.txt")

                @TaskAction
                void generate() {
                    outputFile.text = "done"
                }
            }

            task customTask(type: CustomTask)
        """
    }
}
