#include "gpsshogi/gui/editboard.h"
#include "gpsshogi/gui/util.h"
#include "osl/record/csa.h"
#include <QMenu>
#include <qcursor.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qimage.h>
#include <qlabel.h>
#include <qbitmap.h>
#include <qfiledialog.h>
#include <qfile.h>
#include <QMouseEvent>

#include <iostream>

const int SUB_STAND_X = 380;
const int ADD_STAND_X = 440;

const int BUTTONS_Y = 400;
const int BUTTONS_SIZE = 60;


/* ShowPicture
 *  画像を表示し，その
 */
class ShowPicture : public QDialog
{
public:
  ShowPicture(QWidget *parent,
	      const char *name,
	      const QString file,
	      QRect& area)
    : QDialog(parent),
      area(area)
  {
    pixmap.load(file);
    setMinimumSize(pixmap.width(), pixmap.height());

    QCursor cursor(Qt::CrossCursor);
    setCursor(cursor);
  }

  void paintEvent(QPaintEvent *)
  {
    QPainter painter(this);
    painter.drawPixmap(0, 0, pixmap);

    //painter.setPen(QPen(red, 1, Qt::DotLine));
    //painter.drawRect(area);
  }

  void mousePressEvent(QMouseEvent *e)
  {
    area.setTopLeft(e->pos());
  }

  void mouseReleaseEvent(QMouseEvent *e)
  {
    area.setBottomRight(e->pos());
    if (QMessageBox::Ok ==
	QMessageBox::question(this, "Crop Area", "Is this the correct area?",
			      QMessageBox::Ok | QMessageBox::Default,
			      QMessageBox::Cancel | QMessageBox::Escape))
    {
      accept();
    }
  }

  void mouseMoveEvent(QMouseEvent *e)
  {
    area.setBottomRight(e->pos());
    update();
  }

private:
  QPixmap pixmap;
  QRect& area;
};

void gpsshogi::gui::EditBoard::
setInitialPieces()
{
  for (int i = 0; i < 40; i++)
  {
    initial_pieces[i] = state.pieceOf(i);
  }
}


gpsshogi::gui::EditBoard::
EditBoard(osl::state::SimpleState state,
	  QWidget *parent)
  : Board(state, parent)
{
  setInitialPieces();
}

void gpsshogi::gui::EditBoard::
reset()
{
  update_state(initial_pieces);
  update();
}

void gpsshogi::gui::EditBoard::
readPict()
{
  QString s = QFileDialog::getOpenFileName(this, "", QString(),
					   "Images (*.png *.xpm *.jpg *.JPG)",
					   this);
  QImage image;
  if (FALSE == image.load(s)) return;

  /* 画像を表示し，どの部分を切り出すか選んでもらう.
   */
  QRect area;
  ShowPicture sp(this, "", s, area);
  sp.exec();

  int o_width = area.width();
  double h_ratio = o_width / 360.0;
  int n_left = (int)(-80 * h_ratio) + area.left();
  int n_width = (int)(520 * h_ratio);

  int o_height = area.height();
  double v_ratio = o_height / 360.0;
  int n_top = area.top();
  int n_height = (int)(520 * v_ratio);

  QImage board_image = image.copy(area);
  board_image = board_image.smoothScale(900, 900);
  reset();

  image = image.copy(n_left, n_top, n_width, n_height);
  image.setAlphaBuffer(TRUE);
  image = image.convertDepth(32);


  /* 0xc0 を閾値にしたフィルタにかける.
   */
  for (int y = 0; y < image.height(); y++)
  {
    for (int x = 0; x < image.width(); x++)
    {
      QRgb rgb = image.pixel(x, y);

      if (qRed(rgb) > 0xc0 &&
	  qGreen(rgb) > 0xc0 &&
	  qBlue(rgb) > 0xc0)
      {
	image.setPixel(x, y, qRgba(0xff, 0xff, 0xff, 0x0));
      }
      else
      {
	image.setPixel(x, y, qRgba(0x00, 0xff, 0x00, 0x30));
      }
    }
  }
  image = image.smoothScale(width(), height());
  background.convertFromImage(image, 0);
  update();
}

void gpsshogi::gui::EditBoard::
togglePlayer()
{
  state.changeTurn();
  emit(statusChanged());
  update();
}

osl::Ptype gpsshogi::gui::EditBoard::
get_ptype(osl::Player& player,
	  bool to_stand)
{
  static const osl::Ptype ptype_list[] =
    {
      osl::PAWN,    osl::LANCE,    osl::KNIGHT,    osl::SILVER,
      osl::GOLD,    osl::BISHOP,    osl::ROOK,    osl::KING
    };

#if 0
  QMenu menu(this);
  for (int i = 0; i < 8; i++)
  {
    const osl::Ptype ptype = ptype_list[i];
    Q3PopupMenu *child = new Q3PopupMenu(&menu);

    if (to_stand)
    {
      menu.insertItem(Util::getKanjiPiece(ptype), ptype * 2);
    }
    else
    {
      child->insertItem(pieceToImage(osl::BLACK, ptype), ptype * 2);
      child->setItemEnabled(ptype * 2, available(osl::BLACK, ptype));
      child->insertItem(pieceToImage(osl::WHITE, ptype), ptype* 2 + 1);
      child->setItemEnabled(ptype * 2 + 1, available(osl::WHITE, ptype));
      if (osl::canPromote(ptype))
      {
	const osl::Ptype pptype = osl::promote(ptype);
	child->insertItem(pieceToImage(osl::BLACK, pptype), pptype * 2);
	child->setItemEnabled(pptype * 2, available(osl::BLACK, pptype));
	child->insertItem(pieceToImage(osl::WHITE, pptype), pptype * 2 + 1);
	child->setItemEnabled(pptype * 2 + 1, available(osl::WHITE, pptype));
      }
      menu.insertItem(Util::getKanjiPiece(ptype), child);
    }
  }

  int ret = menu.exec(QCursor::pos());
  player = (osl::Player)-(ret % 2);
  ret /= 2;
#endif

  return osl::PAWN;
}


bool gpsshogi::gui::EditBoard::
available(osl::Player player,
	  osl::Ptype ptype) const
{
  if (ptype == osl::KING)
  {
    return state.hasPieceOnStand(player, ptype);
  }

  if (!state.hasPieceOnStand(osl::WHITE, ptype))
  {
    return false;
  }

  return true;
}

void gpsshogi::gui::EditBoard::
mousePressEvent(QMouseEvent *e)
{
  const int x = e->x();
  const int y = e->y();

  osl::Square pos = getSquare(x, y);
  osl::Piece piece = state.pieceAt(pos);

  if (!pos.isPieceStand())
  {
    if (!piece.isEmpty())
    {
#if 0
      /* 駒のあるマスが選ばれた(削除するか？).
       */
      Q3PopupMenu menu(this);
      menu.insertItem("Remove", 1);
      menu.insertItem("Cancel", 0);
      
      int ret = menu.exec(QCursor::pos());
      
      if (1 == ret)
      {
	remove_piece(piece);
      }
#endif
    }
    else
    {
      /* 盤面上の空のマスが選ばれた.
       */
      osl::Player player;
      osl::Ptype ptype;
      ptype = get_ptype(player, false);
      
      if (osl::isPiece(ptype))
      {
	place_piece(player, pos, ptype);
      }
    }
  }
  else if (y >= BUTTONS_Y && y <= BUTTONS_Y + BUTTONS_SIZE)
  {
    if (x >= SUB_STAND_X && x <= SUB_STAND_X + BUTTONS_SIZE)
    {
      /* 先手の持駒を後手の駒台へ戻す.
       */
      osl::Player player;
      osl::Ptype ptype = get_ptype(player, true);

      if (isPiece(ptype))
      {
	for (int i = 0; i < 40; i++)
	{
	  osl::Piece piece = state.pieceOf(i);
	  if (piece.owner() == osl::BLACK &&
	      piece.ptype() == ptype &&
	      !piece.isOnBoard())
	  {
	    remove_piece(piece);
	    return;
	  }
	}
	//not removed
      }
      return;
    }
    else if (x >= ADD_STAND_X && x <= ADD_STAND_X + BUTTONS_SIZE)
    {
      /* 先手の駒台へ駒を加える.
       */
      osl::Player player;
      osl::Ptype ptype = get_ptype(player, true);
      if (isPiece(ptype))
      {
	place_piece(osl::BLACK, pos, ptype);
      }
    }
  }
  update();
}

void gpsshogi::gui::EditBoard::
paintEvent(QPaintEvent *pe)
{
  Board::paintEvent(pe);

  QPixmap pixmap(width(), height());
  bitBlt(&pixmap, 0, 0, this, 0, 0, width(), height());

  QPainter painter(&pixmap);

  {
    QBrush brush(white);
    painter.fillRect(SUB_STAND_X, BUTTONS_Y, BUTTONS_SIZE, BUTTONS_SIZE, brush);
    painter.setPen(black);
    painter.drawText(SUB_STAND_X, BUTTONS_Y + BUTTONS_SIZE, "Sub");
  }

  {
    QBrush brush(black);
    painter.fillRect(ADD_STAND_X, BUTTONS_Y, BUTTONS_SIZE, BUTTONS_SIZE, brush);
    painter.setPen(white);
    painter.drawText(ADD_STAND_X, BUTTONS_Y + BUTTONS_SIZE, "Add");
  }

  bitBlt(this, 0, 0, &pixmap, 0, 0, width(), height());
  bitBlt(this, 0, 0, &background, 0, 0, width(), height());
}

void gpsshogi::gui::EditBoard::
remove_piece(osl::Piece piece)
{
  osl::Piece pieces[40];

  for (int i = 0; i < 40; i++)
  {
    osl::Piece piece = state.pieceOf(i);
    pieces[i] = piece;
  }

  osl::Player player = osl::WHITE;
  if (piece.ptype() == osl::KING)
  {
    player = piece.owner();
  }

  piece = osl::Piece(player,
		     piece.ptype(),
		     piece.number(),
		     osl::Square());

  pieces[piece.number()] = piece;

  update_state(pieces);
}

void gpsshogi::gui::EditBoard::
place_piece(osl::Player player,
	    osl::Square pos,
	    osl::Ptype ptype)
{
  osl::Piece pieces[40];

  bool placed = false;
  const osl::Player current_owner = ((ptype == osl::KING)
				     ? player : osl::WHITE);
  for (int i = 0; i < 40; i++)
  {
    osl::Piece piece = state.pieceOf(i);
    if (!placed &&
	piece.owner() == current_owner &&
	piece.ptype() == unpromote(ptype) &&
	!piece.isOnBoard())
    {
      pieces[i] = osl::Piece(player, ptype, i, pos);
      placed = true;
    }
    else
    {
      pieces[i] = piece;
    }
  }

  if (!placed)
  {
    QMessageBox::warning(this, "Invalid", "Failed to place piece.",
			 QMessageBox::Ok, 0);
    return;
  }

  update_state(pieces);
}

void gpsshogi::gui::EditBoard::
update_state(osl::Piece pieces[40])
{
  state.init();

  for (int i = 0; i < 40; i++)
  {
    //bool ret = 
    state.setPiece(pieces[i].owner(),
		   pieces[i].square(),
		   pieces[i].ptype());
#if 0
    if (ret == false)
    {
      QString message = tr("Cannot set %1%2")
	.arg(pieces[i].owner())
	.arg(Util::getKanjiPiece(pieces[i].ptype()));
      QMessageBox::warning(this, "Failed",
			   message, QMessageBox::Ok, 0);
    }
#endif
  }
}

const QPixmap gpsshogi::gui::
EditBoard::pieceToImage(osl::Player owner,
			osl::Ptype ptype) const
{
  const osl::Piece piece(owner, ptype, 0, osl::Square::STAND());
  return Board::pieceToImage(piece);
}
