/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satpref-rot.c:  Gnome Predict configuration module (rotators).

  Copyright (C)  2002-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <hamlib/rig.h>
#include "satconfig-rot.h"
#include "extra-widgets.h"
#include "satpref-rot.h"
#include "satpref.h"
#include "hamlib-widgets.h"
#include "rotlist-util.h"
#include "satlog.h"


extern GConfClient *client;     /* shared client connection to GConf in main.c */

/* widgets */
static GtkWidget *newb,*editb,*delb,*defb,*testb,*clist;

/* other */
static gchar *titles[] = {
	N_("#"),
	N_("Brand"),
	N_("Model"),
	N_("Port"),
	N_("Speed"),
	N_("Default")
};

/* private function prototypes */
static gboolean satpref_rot_exists         (guint i);
static void     satpref_rot_row_clicked_cb (GtkCList *clist,
					    gint row,
					    gint column,
					    GdkEventButton *button,
					    gpointer data);
static void     satpref_rot_add_cb         (GtkWidget *button, gpointer clist);
static void     satpref_rot_delete_cb      (GtkWidget *button, gpointer clist);
static void     satpref_rot_edit_cb        (GtkWidget *button, gpointer clist);
static void     satpref_rot_default_cb     (GtkWidget *button, gpointer clist);
static void     satpref_rot_test_cb        (GtkWidget *button, gpointer clist);


GtkWidget *
satpref_rot_create ()
{
	GtkWidget *butbox,*hbox;
	GtkWidget *swin;
	guint i,defrot;
	gchar *p1,*p2,*p3,*p4,*buff,**bufv;

	/* CList widget */
	clist = gtk_clist_new_with_titles (6, titles);
	gtk_clist_column_titles_passive (GTK_CLIST (clist));

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
			    GTK_SIGNAL_FUNC (satpref_rot_row_clicked_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (clist), "unselect-row",
			    GTK_SIGNAL_FUNC (satpref_rot_row_clicked_cb),
			    NULL);

	/* adjust columns automatically */
	for (i=0; i<6; i++)
		gtk_clist_set_column_auto_resize (GTK_CLIST (clist), i, TRUE);

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), clist);

	/* default radio */
	p1 = g_strdup_printf ("%s/default", SAT_CONFIG_ROT_DIR);
	defrot = gconf_client_get_int (client, p1, NULL);
	g_free (p1);

	/* number of radios */
//	max = gconf_client_get_int (client, GRIG_CONFIG_RIG_DIR "/number", NULL);

	/* read radios */
	for (i=0; /*(i<max) &&*/ (satpref_rot_exists (i)); i++) {
		p1 = g_strdup_printf ("%s/rot%d_brand", SAT_CONFIG_ROT_DIR, i);
		p2 = g_strdup_printf ("%s/rot%d_model", SAT_CONFIG_ROT_DIR, i);
		p3 = g_strdup_printf ("%s/rot%d_port", SAT_CONFIG_ROT_DIR, i);
		p4 = g_strdup_printf ("%s/rot%d_speed", SAT_CONFIG_ROT_DIR, i);
		if (gconf_client_get_int (client, p4, NULL)) {
			/* user defined serial speed */
			buff = g_strdup_printf ("%i;%s;%s;%s;%d; ", i,
						gconf_client_get_string (client, p1, NULL),
						gconf_client_get_string (client, p2, NULL),
						gconf_client_get_string (client, p3, NULL),
						gconf_client_get_int (client, p4, NULL));
		}
		else {
			/* default hamlib serial speed */
			buff = g_strdup_printf ("%i;%s;%s;%s;DEF; ", i,
						gconf_client_get_string (client, p1, NULL),
						gconf_client_get_string (client, p2, NULL),
						gconf_client_get_string (client, p3, NULL));
		}
		bufv = g_strsplit (buff, ";", 0);
		gtk_clist_append (GTK_CLIST (clist), bufv);
		g_strfreev (bufv);
		g_free (buff);
		g_free (p1);
		g_free (p2);
		g_free (p3);
		g_free (p4);

		/* attach rotid to row */
		p1 = g_strdup_printf ("%s/rot%d_id", SAT_CONFIG_ROT_DIR, i);
		gtk_clist_set_row_data (GTK_CLIST (clist), i,
					GINT_TO_POINTER (gconf_client_get_int (client,
									       p1,
									       NULL)));
		g_free (p1);

		/* Is this the default radio? */
		if (defrot == i)
			gtk_clist_set_text (GTK_CLIST (clist), i, 5, "YES");
		else
			gtk_clist_set_text (GTK_CLIST (clist), i, 5, " ");
	}


	/* buttons */
	newb = gpredict_stock_pixmap_button (_("Add"), GNOME_STOCK_PIXMAP_ADD,
					     _("Add a new radio to the list"));
	delb = gpredict_stock_pixmap_button (_("Delete"), GNOME_STOCK_PIXMAP_REMOVE,
					     _("Delete the selected radio from the list"));
	editb = gpredict_stock_pixmap_button (_("Edit"), GNOME_STOCK_PIXMAP_PROPERTIES,
					      _("Edit the selected radio"));
	defb = gpredict_stock_pixmap_button (_("Default"), GNOME_STOCK_PIXMAP_ABOUT,
					     _("Make the selected radio default"));
	testb = gpredict_stock_pixmap_button (_("Test"), GNOME_STOCK_PIXMAP_EXEC,
					      _("Test the selected radio")); 

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (newb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_rot_add_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (delb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_rot_delete_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (editb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_rot_edit_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (defb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_rot_default_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (testb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_rot_test_cb),
			    clist);

	/* should only be clickable when a row is selected */
	gtk_widget_set_sensitive (delb, FALSE);
	gtk_widget_set_sensitive (editb, FALSE);
	gtk_widget_set_sensitive (defb, FALSE);
	gtk_widget_set_sensitive (testb, FALSE);

	/* vertical button box */
	butbox = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (butbox), GTK_BUTTONBOX_SPREAD);

	gtk_box_pack_start_defaults (GTK_BOX (butbox), newb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), delb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), editb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), defb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), testb);


	/* main horizontal box */
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);

	gtk_box_pack_start (GTK_BOX (hbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (hbox), butbox, FALSE, FALSE, 0);

	return hbox;
}

void
satpref_rot_close ()
{
	gtk_widget_destroy (newb);
	gtk_widget_destroy (delb);
	gtk_widget_destroy (editb);
	gtk_widget_destroy (defb);
	gtk_widget_destroy (testb);
	gtk_widget_destroy (clist);
}


void
satpref_rot_apply ()
{
	gint i,rows;
	gchar *key,*buff[] = { " " };
	gboolean has_default = FALSE;
	gboolean ok = TRUE;

	/* save number of rots */
	rows = (GTK_CLIST (clist)->rows);
/*	key = g_strdup_printf ("%s/number", SAT_CONFIG_ROT_DIR);
	gconf_client_set_int (client, key, rows, NULL);
	g_free (key);
*/

	/* save contents from the clist */
	for (i=0; i<rows; i++) {
		/* rotid */
		key = g_strdup_printf ("%s/rot%d_id", SAT_CONFIG_ROT_DIR, i);
		ok = ok && gconf_client_set_int (client, key,
						 GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (clist), i)),
						 NULL);
		g_free (key);

		/* brand */
		gtk_clist_get_text (GTK_CLIST (clist), i, 1, buff);
		key = g_strdup_printf ("%s/rot%d_brand", SAT_CONFIG_ROT_DIR, i);
		ok = ok && gconf_client_set_string (client, key, buff[0], NULL);
		g_free (key);

		/* Model */
		gtk_clist_get_text (GTK_CLIST (clist), i, 2, buff);
		key = g_strdup_printf ("%s/rot%d_model", SAT_CONFIG_ROT_DIR, i);
		ok = ok && gconf_client_set_string (client, key, buff[0], NULL);
		g_free (key);

		/* port */
		gtk_clist_get_text (GTK_CLIST (clist), i, 3, buff);
		key = g_strdup_printf ("%s/rot%d_port", SAT_CONFIG_ROT_DIR, i);
		ok = ok && gconf_client_set_string (client, key, buff[0], NULL);
		g_free (key);

		/* serial speed */
		gtk_clist_get_text (GTK_CLIST (clist), i, 4, buff);
		key = g_strdup_printf ("%s/rot%d_speed", SAT_CONFIG_ROT_DIR, i);
		if (g_strcasecmp (buff[0], "DEF")) {
			/* user has seleceted non-default serial speed */
			ok = ok && gconf_client_set_int (client, key, (gint) g_strtod (buff[0], NULL), NULL);
		}
		else {
			ok = ok && gconf_client_set_int (client, key, 0, NULL);
		}
		g_free (key);

		/* is this one the default? */
		gtk_clist_get_text (GTK_CLIST (clist), i, 5, buff);
		if (!g_strcasecmp (buff[0], "YES")) {
			key = g_strdup_printf ("%s/default", SAT_CONFIG_ROT_DIR);
			ok = ok && gconf_client_set_int (client, key, i, NULL);
			g_free (key);
			has_default = TRUE;
		}
		if (!ok) {
			key = g_strdup_printf ("%s: Error saving preferences!", __FUNCTION__);
			satlog_log (SAT_LOG_ERROR, key);
			g_free (key);
			ok = TRUE;
		}
	}
	
	/* set default rot to 0 if not already set */
	if (!has_default) {
		key = g_strdup_printf ("%s/default", SAT_CONFIG_ROT_DIR);
		gconf_client_set_int (client, key, 0, NULL);
		g_free (key);
	}

	/* Delete the rest from the Gconf config */
	for (i=rows; satpref_rot_exists (i); i++) {
		/* rotid */
		key = g_strdup_printf ("%s/rot%d_id", SAT_CONFIG_ROT_DIR, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* brand */
		key = g_strdup_printf ("%s/rot%d_brand", SAT_CONFIG_ROT_DIR, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* Model */
		key = g_strdup_printf ("%s/rot%d_model", SAT_CONFIG_ROT_DIR, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* port */
		key = g_strdup_printf ("%s/rot%d_port", SAT_CONFIG_ROT_DIR, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* serial speed */
		key = g_strdup_printf ("%s/rot%d_speed", SAT_CONFIG_ROT_DIR, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);
	}

	gconf_client_suggest_sync (client, NULL);
}


void
satpref_rot_cancel ()
{
/*  	if (newb) */
/*  		gtk_widget_destroy (newb); */
/*  	if (delb) */
/*  		gtk_widget_destroy (delb); */
/*  	if (editb) */
/*  		gtk_widget_destroy (editb); */
/*  	if (defb) */
/*  		gtk_widget_destroy (defb); */
/*  	if (testb) */
/*  		gtk_widget_destroy (testb); */
/*  	if (clist) */
/*  		gtk_widget_destroy (clist); */
}

void
satpref_rot_help ()
{
}



static gboolean
satpref_rot_exists (guint i)
{
	/* This function checks whether the
	   configuration contains a rotator with rot
	   number 'i'. 
	*/
	gchar *path;
	gint result;

	path = g_strdup_printf ("%s/rot%d_id", SAT_CONFIG_ROT_DIR, i);

	/* ROT Id is non-zero (Dummy has ID 1) so result = 0 means
	   that there is no radio with that number.
	*/
	result = gconf_client_get_int (client, path, NULL);
	g_free (path);

	return result;
}


static void
satpref_rot_row_clicked_cb (GtkCList *clist,
			    gint row,
			    gint column,
			    GdkEventButton *button,
			    gpointer data)
{
	/* This function is called when a row is selected
	   or unselected. It check whether the CList has a
	   valid selection and enables the buttons if so.
	   NOTE: The Delete button is enabled only if there
	   are more than one radios in the list.
	*/
	GList *selection = clist->selection;

	if (selection) {
		/* If only one radio is in list don't enable Delete button */
		if (clist->rows > 1) {
			gtk_widget_set_sensitive (delb, TRUE);
		}
		else {
			gtk_widget_set_sensitive (delb, FALSE);
		}
		gtk_widget_set_sensitive (editb, TRUE);
		gtk_widget_set_sensitive (defb, TRUE);
/*  		gtk_widget_set_sensitive (testb, TRUE); */
	}
	else {
		gtk_widget_set_sensitive (delb, FALSE);
		gtk_widget_set_sensitive (editb, FALSE);
		gtk_widget_set_sensitive (defb, FALSE);
/*  		gtk_widget_set_sensitive (testb, FALSE); */
	}
}


static void   
satpref_rot_add_cb         (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Add" button. It opens a new dialog to select a
	   new radio.
	*/
	GtkWidget *dialog,*rotclist,*swin,*entry,*portframe;
	GtkWidget *speedw,*speedframe,*hbox,*activemenu;
	GSList *rotlist;
	gint diabut=-1,row=-1,rotid,i;
	guint speed;
	gchar *newrot[6],*buff[] = {" "};

	/* get list of drivers */
	rotlist = rotlist_get_list ();

	/* create a columned list */
	rotclist = hamlib_rotlist_widget_new (rotlist);

	/* pack the clist into a scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), rotclist);
	gtk_widget_show_all (swin);

	/* Entry for port */
	entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (entry), "/dev/ttyS0");
	portframe = gtk_frame_new (_("Port"));
	gtk_container_add (GTK_CONTAINER (portframe), entry);

	/* speed */
	speedw = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (speedw), hamlib_widgets_create_serial_rate_menu ());
	gtk_option_menu_set_history (GTK_OPTION_MENU (speedw), 0);
	speedframe = gtk_frame_new (_("Speed"));
	gtk_container_add (GTK_CONTAINER (speedframe), speedw);

	/* put frames into a horizontal box */
	hbox = gtk_hbox_new (TRUE, 5);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), portframe);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), speedframe);
	gtk_widget_show_all (hbox);

	/* create dialog */
	dialog = gnome_dialog_new (_("Select a radio"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);

	/* make list more visible */
	gtk_widget_set (dialog, "height", 300, NULL);

	diabut = gnome_dialog_run (GNOME_DIALOG (dialog));

	switch (diabut) {
	case 0: /* OK */
		if (GTK_CLIST (rotclist)->selection)
			row = GPOINTER_TO_INT (GTK_CLIST (rotclist)->selection->data);
		else
			break;

		/* get rot info from rotclist */

		/* row number */
		newrot[0] = g_strdup_printf ("%d", (GTK_CLIST (clist)->rows));

		/* brand */
		gtk_clist_get_text (GTK_CLIST (rotclist), row, 1, buff);
		newrot[1] = g_strdup (buff[0]);

		/* model */
		gtk_clist_get_text (GTK_CLIST (rotclist), row, 2, buff);
		newrot[2] = g_strdup (buff[0]);

		/* port */
		newrot[3] = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

		/* speed */
		activemenu = gtk_menu_get_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (speedw))));
		speed = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (activemenu), "speed"));
		if (speed) {
			newrot[4] = g_strdup_printf ("%d", speed);
		}
		else
			newrot[4] = g_strdup ("DEF");

		/* default */
		newrot[5] = g_strdup (" ");

		/* rotid */
		rotid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rotclist), row));

		/* add new rot */
		row = gtk_clist_append (GTK_CLIST (clist), newrot);
		gtk_clist_set_row_data (GTK_CLIST (clist), row,
					GINT_TO_POINTER (rotid));
		/* free some memory */
		for (i=0; i<6; i++)
			if (newrot[i])
			    g_free (newrot[i]);

		/* trigger config changed */
		satpref_changed ();
		break;
	case 1: /* Cancel */
		break;
	default: /* unknown button... */
		break;
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}



static void
satpref_rot_delete_cb      (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Delete" button. It deletes the selected radio entry
	   from the clist.
	*/
	GList *selection = (GTK_CLIST (clist))->selection;
	gint num;
	gchar *buff,*dummy[] = { " "," "," "," "," "," "};
	gboolean def = FALSE;

	if (selection) {
		/* is this the default rot? */
		gtk_clist_get_text (GTK_CLIST (clist),
				    GPOINTER_TO_UINT (selection->data), 5,
				    dummy);
		g_print ("%s\n", dummy[0]);
		def = !g_strcasecmp (dummy[0], "YES");

		gtk_clist_remove (GTK_CLIST (clist),
				  GPOINTER_TO_UINT (selection->data));
		num = (GTK_CLIST (clist)->rows);
		while (--num >= 0) {
			buff = g_strdup_printf ("%d", num);
			gtk_clist_set_text (GTK_CLIST (clist), num, 0, buff);
			g_free (buff);
			/* make radio 0 the default if needed */
			if (!num && def)
				gtk_clist_set_text (GTK_CLIST (clist), num, 5, "YES");
		}
		/* trigger signal */
		satpref_changed ();
	}
}


static void
satpref_rot_edit_cb        (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Edit" button. It opens a new dialog to edit the
	   selected radio.
	*/
	GtkWidget *dialog,*rotclist,*swin,*entry,*frame;
	GtkWidget *spframe,*spmenu,*hbox,*activemenu;
	GSList *rotlist;
	gint crow,crotid;
	guint speed;
	gint diabut=-1,row=-1,rotid,i;
	gchar *newrot,*buff[] = {" "};

	/* get selected row and rotid (needed to select the rot in the popup dialog */
	crow = GPOINTER_TO_INT (GTK_CLIST (clist)->selection->data);
	crotid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (clist), crow));

	/* get list of drivers */
	rotlist = rotlist_get_list ();

	/* create a columned list */
	rotclist = hamlib_rotlist_widget_new (rotlist);

	/* select the current rot */
	for (i=0; i<(GTK_CLIST (rotclist))->rows; i++) {
		if (crotid == GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rotclist), i))) {
			gtk_clist_select_row (GTK_CLIST (rotclist), i, 0);
			/* stop iteration */
			i = (GTK_CLIST (rotclist))->rows;
		}
	}
	/* pack the clist into a scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), rotclist);
	gtk_widget_show_all (swin);

	/* radio port */
	entry = gtk_entry_new ();
	gtk_clist_get_text (GTK_CLIST (clist), crow, 3, buff);
	gtk_entry_set_text (GTK_ENTRY (entry), buff[0]);
	frame = gtk_frame_new (_("Port"));
	gtk_container_add (GTK_CONTAINER (frame), entry);

	/* serial speed */
	spmenu = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (spmenu), hamlib_widgets_create_serial_rate_menu ());
	spframe = gtk_frame_new (_("Speed"));
	gtk_container_add (GTK_CONTAINER (spframe), spmenu);
	gtk_clist_get_text (GTK_CLIST (clist), crow, 4, buff);
	/* select the correct speed */
	if (!g_strcasecmp (buff[0], "DEF")) {
		gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 0);
	}
	else {
		switch ((gint) g_strtod (buff[0], NULL)) {
		case 300:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 1);
			break;
		case 1200:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 2);
			break;
		case 2400:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 3);
			break;
		case 4800:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 4);
			break;
		case 9600:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 5);
			break;
		case 19200:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 6);
			break;
		case 38400:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 7);
			break;
		case 57600:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 8);
			break;
		case 115200:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 9);
			break;
		default:
			gtk_option_menu_set_history (GTK_OPTION_MENU (spmenu), 0);
			break;
		}
	}

	/* pack the frames into a horizontal box */
	hbox = gtk_hbox_new (TRUE, 5);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), frame);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), spframe);
	gtk_widget_show_all (hbox);

	/* create dialog */
	dialog = gnome_dialog_new (_("Select a radio"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);

	/* make list more visible */
	gtk_widget_set (dialog, "height", 300, NULL);

	diabut = gnome_dialog_run (GNOME_DIALOG (dialog));

	switch (diabut) {
	case 0: /* OK */
		if (GTK_CLIST (rotclist)->selection)
			row = GPOINTER_TO_INT (GTK_CLIST (rotclist)->selection->data);
		else
			break;

		/* get rot info from rotclist */

		/* brand */
		gtk_clist_get_text (GTK_CLIST (rotclist), row, 1, buff);
		newrot = g_strdup (buff[0]);
		gtk_clist_set_text (GTK_CLIST (clist), crow, 1, newrot);
		g_free (newrot);

		/* model */
		gtk_clist_get_text (GTK_CLIST (rotclist), row, 2, buff);
		newrot = g_strdup (buff[0]);
		gtk_clist_set_text (GTK_CLIST (clist), crow, 2, newrot);
		g_free (newrot);

		/* port */
		newrot = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
		gtk_clist_set_text (GTK_CLIST (clist), crow, 3, newrot);
		g_free (newrot);

		/* speed */
		activemenu = gtk_menu_get_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (spmenu))));
		speed = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (activemenu), "speed"));
		if (speed)
			newrot = g_strdup_printf ("%d", speed);
		else
			newrot = g_strdup ("DEF");
		gtk_clist_set_text (GTK_CLIST (clist), crow, 4, newrot);
		g_free (newrot);

		/* rotid */
		rotid = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (rotclist), row));
		gtk_clist_set_row_data (GTK_CLIST (clist), crow,
					GINT_TO_POINTER (rotid));

		/* trigger config changed */
		satpref_changed ();
		break;
	case 1: /* Cancel */
		break;
	default:
		break;
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}


static void 
satpref_rot_default_cb     (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Default" button. It makes the selected radio the
	   default radio.
	*/
	GList *selection = (GTK_CLIST (clist)->selection);
	gint row,i;

	if (selection) {
		row = GPOINTER_TO_INT (selection->data);
		for (i=0; i < (GTK_CLIST (clist)->rows); i++)
			gtk_clist_set_text (GTK_CLIST (clist), i, 5,
					    (i == row) ? "YES" : " ");
		/* trigger signal */
		satpref_changed ();
	} 
}


static void
satpref_rot_test_cb        (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Test" button. It performs a test on the selected
	   radio.
	*/
}
