/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satmodules.c: Maintains a list of open modules. This list is saved when the
                program exits and is read when the program starts. The modules
		should call the add/remove functions with the module name as
		argument. satmodules_add_module should be called when the
		module is created, and satmodules_remove_module should be
		called when the module is closed.

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include "satlog.h"
#include "satmodules.h"


/************  FIXME: Should be ported to GConf *************/


/* config paths */
const gchar *cfgpath[] = {
	"/gpredict/openmodules/lists",
	"/gpredict/openmodules/maps",
	"/gpredict/openmodules/skys",
	"/gpredict/openmodules/globes"
};

/* string representation of module types */
const gchar *modtypes[] = { "satlist", "satmap", "satsky", "satglobe" };


guint satmodules_init (void)
{
	return 0;
}


gboolean satmodules_test_module (gchar *module, guint type)
{
	/* This function test whether the specified module exists
	   or not. It can be used to avoid program crash, when the
	   the program tries to open a module that has been manually
	   deleted by the user. See bug #554785 for more info on
	   this problem.
	*/
	gchar *path;
	gboolean result;

	if (type < SATMOD_TYPE_MAX) {
		/* create gnome_vfs_uri */
		path = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".gpredict",
				    G_DIR_SEPARATOR_S, module, ".", modtypes[type],
				    NULL);
		result = g_file_test (path, G_FILE_TEST_ISFILE);
		g_free (path);
	}
	else {
		satlog_log (SAT_LOG_CRITICAL,
			    _("SATMODULES: test_module: type out of range!"));
		result = FALSE;
	}

	return result;
}

gboolean satmodules_test_open_modules (gchar *module, guint type)
{
	/* This function can be used to test whether a specific
	   module is already open. The function returns TRUE if
	   the specified module is open, FALSE otherwise.
	*/
	gchar **argv;
	gint argc=0,i;

	if (type < SATMOD_TYPE_MAX)
		gnome_config_get_vector (cfgpath[type], &argc, &argv);
	else
		satlog_log (SAT_LOG_CRITICAL,
			    _("SATMODULES: test_open_module: type out of range!"));

	if (!argc) /* No list! */
		return FALSE; 
	for (i=0; i<argc; i++) {
		if (!g_strcasecmp (module, argv[i])) {
			g_strfreev (argv);
			return TRUE;
		}
	}
	g_strfreev (argv);
	return FALSE;
}


void  satmodules_add_module (gchar *module, guint type)
{
	/* This function should be called by a module when it
	   is created. The module's name and type should be
	   used as argument.
	*/
	gchar *buff1,*buff2,**argv,*path;
	gint argc;

	/* get config path */
	if (type < SATMOD_TYPE_MAX)
		path = g_strdup (cfgpath[type]);
	else {
		satlog_log (SAT_LOG_CRITICAL, _("SATMODULES: add_module: Invalid module type!"));
		return;
	}

	if ((module) && !(satmodules_test_open_modules (module, type))) {
		/* add to list */
		gnome_config_get_vector (path, &argc, &argv);
		if (argc) {
			buff1 = g_strjoinv (";", argv);
			buff2 = g_strconcat (buff1, ";", module, NULL);
			g_strfreev (argv);
			g_free (buff1);
		}
		else
			buff2 = g_strdup (module);
		argc++;
		argv = g_strsplit (buff2, ";", argc);
		g_free (buff2);
		gnome_config_set_vector (path, argc, argv);
		gnome_config_sync ();
		g_strfreev (argv);
	}
	else
		satlog_log (SAT_LOG_WARNING,
			    _("SATMODULES: add_module: module already open or (NULL) argument"));
	g_free (path);
}


void  satmodules_remove_module (gchar *module, guint type)
{
	/* This function should be called by a module when
	   it is closed. The module's name should be used
	   as argument.
	*/
	gchar *buff1=NULL,*buff2=NULL,**argv,*path;
	gint argc;
	
	if (type < SATMOD_TYPE_MAX)
		path = g_strdup (cfgpath[type]);
	else {
		satlog_log (SAT_LOG_CRITICAL,
			    _("SATMODULES: remove_module: Invalid module type!"));
		return;
	}

	if (satmodules_test_open_modules (module, type)) {
		gnome_config_get_vector (path, &argc, &argv);
		buff1 = g_strjoinv (";", argv);
		g_strfreev (argv);
		/* Remove the module name by splitting the string using "module;"
		   as separator. */
		buff2 = g_strconcat (module, ";", NULL);
		argv = g_strsplit (buff1, buff2, 2);
		g_free (buff1);
		g_free (buff2);
		buff1 = g_strjoinv (NULL, argv);
		g_strfreev (argv);
		argc--;
		if (argc) {
			argv = g_strsplit (buff1, ";", argc);
			g_free (buff1);
			gnome_config_set_vector (path, argc, argv);
			gnome_config_sync ();
			g_strfreev (argv);
		}
		else {
			/* This is the correct way to clear the module list,
			   see bug #522484. */
			gnome_config_clean_key (path); 
			gnome_config_sync ();
		}
		g_free (path);
	}
	else
		satlog_log (SAT_LOG_WARNING,
			    _("SATMODULES: remove_module: module not open or (NULL) argument"));

}


void satmodules_get_modules (guint type, gint *argc, gchar ***argv)
{
	/* This function returns a vector of strings
	   containing the names of the open modules.
	   The vector should be freed when no longer
	   needed.
	*/
	gchar *path;

	if (type < SATMOD_TYPE_MAX)
		path = g_strdup (cfgpath[type]);
	else {
		satlog_log (SAT_LOG_CRITICAL,
			    _("SATMODULES: get_modules: Invalid module type!"));
		return;
	}
	gnome_config_get_vector (path, argc, argv);
	g_free (path);
}



void satmoudules_close ()
{
	/* ? */
}

