/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*

  qth.c:  Groundstation related stuff.

  Copyright (C)  2001-2005  Alexandru Csete.

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <stdio.h>
#include "satlog.h"
#include "defaults.h"
#include "qth.h"
#include "engine.h"
#include "util.h"


#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif


qth_struc qth;              /* QTH structure */
gint qth_cnx_id = 0;       /* Gconf client notification ID */
extern GConfClient *client; /* Initialized GConf client in main.c */


static void qth_changed (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data);

/* returns 1 on error */
gint qth_init_old()
{
	FILE   *qthf;
	gchar  *buff;

	if ( !(buff = g_strdup (gnome_config_get_string (QTH_FILE_PATH))) ) {
		buff = g_strconcat (__FUNCTION__, _("NO QTH PATH!"));
		satlog_log (SAT_LOG_ERROR, buff);
		g_free (buff);
		return 1;
	}
	if ( !(qthf=fopen(buff,"r")) )
		return 1;
	g_free (buff);

	/* read in QTH info - replace trailing "\n" */
	buff = g_malloc (80);
	buff = fgets (buff, 80, qthf);
	g_strdelimit (buff, "\n",' ');
	qth.name = g_strdup (buff);
	buff = fgets (buff, 80, qthf);
	g_strdelimit (buff, "\n",' ');
	qth.desc = g_strdup (buff);
	buff = fgets (buff, 80, qthf);
	g_strdelimit (buff, "\n",' ');
	qth.loc  = g_strdup (buff);
	buff = fgets (buff, 80, qthf);	
	qth.lat  = g_strtod (buff, NULL);
	buff = fgets (buff, 80, qthf);	
	qth.lon  = g_strtod (buff, NULL);
	buff = fgets (buff, 80, qthf);
	qth.alt  = (gint) g_strtod (buff, NULL);
	
	g_free (buff);
	fclose (qthf);

	return 0;
}


gint qth_init ()
{
	/* This function reads the specified location of the
	   observer which is used by the tracking engine. The
	   user might have had configured several locations, in
	   which case only the default location will be read.
	*/
	gchar *key;
	gint defqth = 0; /* default qth */

	key = g_strconcat (QTH_ROOT_PATH, "/default", NULL);
	defqth = gconf_client_get_int (client, key, NULL);
	g_free (key);

	/* QTH name */
	key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, defqth);
	qth.name = g_strdup (gconf_client_get_string (client, key, NULL));
	g_free (key);

	/* Description */
	key = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, defqth);
	qth.desc = g_strdup (gconf_client_get_string (client, key, NULL));
	g_free (key);
	
	/* Location */
	key = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, defqth);
	qth.loc = g_strdup (gconf_client_get_string (client, key, NULL));
	g_free (key);

	/* Weather station */
	key = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, defqth);
	if (gconf_client_get_string (client, key, NULL))
		qth.wx = g_strdup (gconf_client_get_string (client, key, NULL));
	g_free (key);

	/* QRA Locator */
	key = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, defqth);
	if (gconf_client_get_string (client, key, NULL))
		qth.qra = g_strdup (gconf_client_get_string (client, key, NULL));
	g_free (key);

	/* Latitude */
	key = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, defqth);
	qth.lat = gconf_client_get_float (client, key, NULL);
	g_free (key);

	/* Longitude */
	key = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, defqth);
	qth.lon = gconf_client_get_float (client, key, NULL);
	g_free (key);

	/* Altitude */
	key = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, defqth);
	qth.alt = gconf_client_get_int (client, key, NULL);
	g_free (key);

	/* Add GConf notifier which can update the program settings
	   automatically, if a key below QTH_ROOT_PATH is changed.
	 */
	if (!qth_cnx_id) {
		key = g_strconcat (QTH_ROOT_PATH, "/default", NULL);
		qth_cnx_id = gconf_client_notify_add (client, key, qth_changed, NULL, NULL, NULL);
		g_free (key);
	}

	return 0;
}

void qth_close ()
{
	if (qth.name)
		g_free (qth.name);
	if (qth.loc)
		g_free (qth.loc);
	if (qth.desc)
		g_free (qth.desc);
	if (qth.qra)
		g_free (qth.qra);
	if (qth.wx)
		g_free (qth.wx);

	/* Remove notification function */
	if (qth_cnx_id >= 0) {
		gconf_client_notify_remove (client, qth_cnx_id);
		qth_cnx_id = 0;
	}
}


extern gboolean update_satmap_qth;

void
qth_changed (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	/* This function is called by the GConf engine when the QTH
	   setting are changed. It stops the engine, reloads the
	   QTH settings and restarts the engine.
	*/
	engine_stop ();
	qth_close ();
	qth_init ();
	PreCalcAll (CurrentDaynum());
	CalcAll ();
	engine_start ();
}
