/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * GtkGauge: A simple gauge plotter
 * Copyright (C) 1999 Uwe Steinmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "gtkgauge.h"

#ifndef MIN
#define MIN(a,b) (((a)<(b))?(a):(b))
#endif

enum {
  ARG_0,
  ARG_QUANTISATION,
  ARG_AUTORANGE
};

static void gtk_gauge_class_init	(GtkGaugeClass	 *klass);
static void gtk_gauge_init		(GtkGauge	 *gauge);
static void gtk_gauge_set_arg		(GtkObject      *object,
					 GtkArg         *arg,
					 guint           arg_id);
static void gtk_gauge_get_arg		(GtkObject      *object,
					 GtkArg         *arg,
					 guint           arg_id);
static void gtk_gauge_destroy		(GtkObject	 *object);
static void gtk_gauge_size_request (GtkWidget	 *widget,
				    GtkRequisition *requisition);
static void gtk_gauge_size_allocate	(GtkWidget	*widget,
					 GtkAllocation	*allocation);
static void gtk_gauge_realize		(GtkWidget	*widget);
static gboolean gtk_gauge_expose		(GtkWidget	*widget,
					GdkEventExpose      *event);
static void gtk_gauge_update(GtkGauge *gauge);
static void gtk_gauge_adjustment_changed(GtkAdjustment *adjustment, gpointer data);
static void gtk_gauge_adjustment_value_changed(GtkAdjustment *adjustment, gpointer data);

static GtkMiscClass *parent_class = NULL;

enum {
  GRAPH_COLOR_ON,
  GRAPH_COLOR_BACK,
  GRAPH_COLOR_GRID
};


GtkType
gtk_gauge_get_type ()
{
  static GtkType gauge_type = 0;
  
  if (!gauge_type)
  {
    static const GTypeInfo gauge_info =
    {
      sizeof (GtkGaugeClass),
			NULL,
			NULL,
      (GClassInitFunc) gtk_gauge_class_init,
			NULL,
			NULL,
      sizeof (GtkGauge),
			0,
      (GInstanceInitFunc) gtk_gauge_init
    };
    
    gauge_type = g_type_register_static (GTK_TYPE_WIDGET, "GtkGauge", &gauge_info, 0);
  }
  
  return gauge_type;
}

void
gtk_gauge_class_init (GtkGaugeClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  
  parent_class = gtk_type_class (gtk_misc_get_type ());

  gtk_object_add_arg_type ("GtkGauge::autorange", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_AUTORANGE); 
  gtk_object_add_arg_type ("GtkGauge::quantisation", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_QUANTISATION); 

  object_class->destroy = gtk_gauge_destroy;
  object_class->set_arg = gtk_gauge_set_arg;
  object_class->get_arg = gtk_gauge_get_arg;
  
  widget_class->size_request  = gtk_gauge_size_request;
  widget_class->size_allocate = gtk_gauge_size_allocate;
  widget_class->realize       = gtk_gauge_realize;
  widget_class->expose_event  = gtk_gauge_expose;
}

static void
gtk_gauge_set_arg (GtkObject      *object,
                   GtkArg         *arg,
                   guint           arg_id)
{
  GtkGauge *gauge;

  gauge = GTK_GAUGE (object);

  switch (arg_id)
    {
    case ARG_AUTORANGE:
      gtk_gauge_set_autorange (gauge, GTK_VALUE_BOOL (*arg));
      break;
    case ARG_QUANTISATION:
      gtk_gauge_set_quantisation (gauge, GTK_VALUE_INT (*arg));
      break;
    default:
      break;
    }
}

static void
gtk_gauge_get_arg (GtkObject      *object,
                   GtkArg         *arg,
                   guint           arg_id)
{
  GtkGauge *gauge;

  gauge = GTK_GAUGE (object);

  switch (arg_id)
    {
    case ARG_AUTORANGE:
      GTK_VALUE_BOOL (*arg) = gauge->autorange;
      break;
    case ARG_QUANTISATION:
      GTK_VALUE_INT (*arg) = gauge->quantisation;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

void
gtk_gauge_init (GtkGauge *gauge)
{
	GTK_WIDGET_SET_FLAGS (gauge, GTK_NO_WINDOW);
  
	gauge->value             = 0;
	gauge->min               = 2000;
	gauge->max               = 2000;
	gauge->autorange         = 0;
	gauge->gc                = NULL;
	gauge->font              = gdk_font_load("fixed");
	gauge->adjustment        = NULL;

}

GtkWidget*
gtk_gauge_new (GtkAdjustment *adjustment)
{
  GtkGauge *gauge;
 
	gauge = g_object_new (gtk_gauge_get_type (), NULL);

	if (!adjustment)
		adjustment = (GtkAdjustment*) gtk_adjustment_new (0.0, 0.0, 0.0, 0.0, 0.0, 0.0);
	gtk_gauge_set_adjustment (gauge, adjustment);

  return GTK_WIDGET (gauge);
}

void
gtk_gauge_set_colors (GtkGauge *gauge, GdkColor *active, GdkColor *inactive, GdkColor *grid)
{
	GdkColormap  *cmap;

	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	gauge->fg[GRAPH_COLOR_ON] = *(active);
	gauge->fg[GRAPH_COLOR_BACK] = *(inactive);
	gauge->fg[GRAPH_COLOR_GRID] = *(grid);

	cmap = gtk_widget_get_colormap (GTK_WIDGET(gauge));
      
	if (!(&(gauge->fg[GRAPH_COLOR_ON])))
		gdk_color_parse ("#00E0E0", &(gauge->fg[GRAPH_COLOR_ON]));
	gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_ON]));
	if (!(&(gauge->fg[GRAPH_COLOR_BACK])))
		gdk_color_parse ("#E0E0E0", &(gauge->fg[GRAPH_COLOR_BACK]));
	gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_BACK]));
	if (!(&(gauge->fg[GRAPH_COLOR_GRID])))
		gdk_color_parse ("#A0A0F0", &(gauge->fg[GRAPH_COLOR_GRID]));
	gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_GRID]));
}

void
gtk_gauge_set_state (GtkGauge	*gauge,
		   GtkStateType widget_state,
		   gint	value)
{
  g_return_if_fail (gauge != NULL);
  g_return_if_fail (GTK_IS_GAUGE (gauge));

  gtk_widget_set_state (GTK_WIDGET (gauge), widget_state);
  gtk_gauge_set_value (gauge, value);
}

void
gtk_gauge_set_value (GtkGauge	     *gauge,
		gint     value)
{
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	gauge->value = value;
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

gint
gtk_gauge_get_value(GtkGauge  *gauge)
{
	g_return_val_if_fail (gauge != NULL, -1);
	g_return_val_if_fail (GTK_IS_GAUGE (gauge), -1);
  
	return gauge->value;
}

void
gtk_gauge_set_range (GtkGauge	     *gauge,
		gint     xmin,
		gint     xmax)
{
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	gauge->min = xmin;
	gauge->max = xmax;
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

void
gtk_gauge_set_autorange (GtkGauge	     *gauge,
		gint     value)
{
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	gauge->autorange = value;
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

void
gtk_gauge_set_quantisation (GtkGauge	     *gauge,
		gint     value)
{
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	gauge->quantisation = value;
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

static void
gtk_gauge_destroy (GtkObject *object)
{
  GtkGauge *gauge;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_GAUGE (object));
  
  gauge = GTK_GAUGE (object);

	if (gauge->adjustment) {
		g_object_unref (GTK_OBJECT (gauge->adjustment));
		gauge->adjustment = NULL;
	}

	gdk_font_unref(gauge->font);  

  if (GTK_WIDGET (object)->parent &&
      GTK_WIDGET_MAPPED (object))
    gtk_widget_unmap (GTK_WIDGET (object));
  
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkAdjustment*
gtk_gauge_get_adjustment (GtkGauge *gauge)
{
	g_return_val_if_fail (gauge != NULL, NULL);
	g_return_val_if_fail (GTK_IS_GAUGE (gauge), NULL);

	return gauge->adjustment;
}

void
gtk_gauge_set_adjustment (GtkGauge *gauge, GtkAdjustment *adjustment)
{
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));
	
	if (gauge->adjustment) {
		g_signal_handlers_disconnect_by_func (GTK_OBJECT (gauge->adjustment), NULL, (gpointer) gauge);
		g_object_unref (GTK_OBJECT (gauge->adjustment));
	}
	
	gauge->adjustment = adjustment;
	g_object_ref (GTK_OBJECT (gauge->adjustment));
	
	g_signal_connect (GTK_OBJECT (adjustment), "changed",
								    GTK_SIGNAL_FUNC (gtk_gauge_adjustment_changed),
								    (gpointer) gauge);
	g_signal_connect (GTK_OBJECT (adjustment), "value_changed",
								    GTK_SIGNAL_FUNC (gtk_gauge_adjustment_value_changed),
								    (gpointer) gauge);
	
	gauge->old_value = adjustment->value;
	gauge->old_lower = adjustment->lower;
	gauge->old_upper = adjustment->upper;
	
	gtk_gauge_update (gauge);
}

static void
gtk_gauge_adjustment_changed (GtkAdjustment *adjustment,
			      gpointer       data)
{
	GtkGauge *gauge;
	
	g_return_if_fail (adjustment != NULL);
	g_return_if_fail (data != NULL);
	
	gauge = GTK_GAUGE (data);
	
	if ((gauge->old_value != adjustment->value) ||
	    (gauge->old_lower != adjustment->lower) ||
	    (gauge->old_upper != adjustment->upper)) {
		gtk_gauge_update (gauge);
		
		gauge->old_value = adjustment->value;
		gauge->old_lower = adjustment->lower;
		gauge->old_upper = adjustment->upper;
	}
}

static void
gtk_gauge_adjustment_value_changed (GtkAdjustment *adjustment,
				    gpointer       data)
{
	GtkGauge *gauge;
	
	g_return_if_fail (adjustment != NULL);
	g_return_if_fail (data != NULL);
	
	gauge = GTK_GAUGE (data);
	
	if (gauge->old_value != adjustment->value) {
		gtk_gauge_update (gauge);
	
		gauge->old_value = adjustment->value;
	}
}


static void
gtk_gauge_size_allocate (GtkWidget	     *widget,
		      GtkAllocation *allocation)
{
	GtkGauge *gauge;
  
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_GAUGE (widget));
	g_return_if_fail (allocation != NULL);
  
	gauge = GTK_GAUGE (widget);
  
	widget->allocation = *allocation;

	if (gauge->pixmap) {
		gdk_pixmap_unref (gauge->pixmap);
		gauge->pixmap = NULL;
	}
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

static void
gtk_gauge_size_request (GtkWidget	     *widget,
		      GtkRequisition *requisition)
{
	GtkGauge *gauge;
  
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_GAUGE (widget));
	g_return_if_fail (requisition != NULL);
  
	gauge = GTK_GAUGE (widget);
  
	requisition->width = GRAPH_WIDTH;
	requisition->height = GRAPH_HEIGHT;
}

static void 
gtk_gauge_realize (GtkWidget *widget)
{
	GtkGauge       *gauge;
	GdkColormap  *cmap;
	GdkWindowAttr attributes;
	gint attributes_mask;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_GAUGE (widget));

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
	gauge = GTK_GAUGE (widget);

/*
	attributes.x = gauge->allocation.x;
	attributes.y = gauge->allocation.y;
	attributes.width = gauge->allocation.width;
	attributes.height = gauge->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.event_mask = gtk_widget_get_events (widget) | 
		    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | 
		    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
		    GDK_POINTER_MOTION_HINT_MASK;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	widget->window = gdk_window_new (widget->parent->window, &attributes, attributes_mask);
*/

	widget->window = gtk_widget_get_parent_window (widget);
	gdk_window_ref (widget->window);

	widget->style = gtk_style_attach (widget->style, widget->window);
	if (!gauge->gc)
	{
		cmap = gtk_widget_get_colormap (widget);
      
		gdk_color_parse ("#0000FF", &(gauge->fg[GRAPH_COLOR_ON]));
		gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_ON]));
		gdk_color_parse ("#D0D0D0", &(gauge->fg[GRAPH_COLOR_BACK]));
		gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_BACK]));
		gdk_color_parse ("#7070FF", &(gauge->fg[GRAPH_COLOR_GRID]));
		gdk_color_alloc (cmap, &(gauge->fg[GRAPH_COLOR_GRID]));
		gauge->gc = gdk_gc_new (widget->window);
		gdk_gc_copy (gauge->gc, widget->style->white_gc);
	}
}

static void
gtk_gauge_update(GtkGauge *gauge)
{
	gfloat new_value;
		  
	g_return_if_fail (gauge != NULL);
	g_return_if_fail (GTK_IS_GAUGE (gauge));

	g_signal_emit_by_name (GTK_OBJECT (gauge->adjustment), "value_changed");
	gtk_widget_queue_draw (GTK_WIDGET (gauge));
}

#define BORDER 2
static gboolean
gtk_gauge_expose(GtkWidget *widget, GdkEventExpose *event) {
	GtkGauge   *gauge;
	GdkColor *win_bg;
	gint     yradius, xradius, value, strwth;
	float    angle;
	char buffer[10];

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_GAUGE (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	if (event->count > 0)
		return FALSE;

	gauge = GTK_GAUGE (widget);

	if (!gauge->pixmap)
		gauge->pixmap = gdk_pixmap_new (widget->window,
		            widget->allocation.width, widget->allocation.height,
		            gtk_widget_get_visual (widget)->depth);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		if ((widget->allocation.width >= widget->requisition.width) &&
		    (widget->allocation.height >= widget->requisition.height)) {
			guint x, y;
			win_bg = &(gauge->fg[GRAPH_COLOR_BACK]);
			gdk_gc_set_foreground (gauge->gc, win_bg);
			x = widget->allocation.x;
			y = widget->allocation.y;

//fprintf(stderr, "gtkgauge: x = %d, y = %d, height = %d, width = %d\n", x, y, widget->allocation.height, widget->allocation.width);
			if(widget->allocation.height < 5)
				return;

			yradius = widget->allocation.height-2*BORDER;	

			/* Clear the pixmap */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_BACK]));
			gdk_draw_rectangle (gauge->pixmap,
			  gauge->gc,
			  TRUE,
			  0, 0,
			  widget->allocation.width,
			  widget->allocation.height);

			/* Draw grid with long lines */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_GRID]));
			for(value=-1500; value<=1500; value+=300) {
				angle = ((float)(value-gauge->min)/(gauge->max-gauge->min)*120+30)/180.0*M_PI;
				gdk_draw_line (gauge->pixmap, gauge->gc,
				        widget->allocation.width/2, widget->allocation.height-BORDER,
				        widget->allocation.width/2-(int)(yradius*cos(angle)),
				        widget->allocation.height-(int)(yradius*sin(angle))-BORDER);
			}
			/* Overpaint the inner part */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_BACK]));
			gdk_draw_arc(gauge->pixmap, gauge->gc,
			        TRUE,
			        (int)(widget->allocation.width/2-0.8*yradius+0.5),
			        (int)(widget->allocation.height-3-0.8*yradius+0.5),
							(int) (2*yradius*0.8), (int) (2*yradius*0.8),
							10*64, 150*64);

			/* Output the current in the middle of the gauge */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_GRID]));
			sprintf(buffer, "%d mA", gauge->value);
			strwth = gdk_string_width(gauge->font, buffer);
			gdk_draw_string(gauge->pixmap, gauge->font, gauge->gc,
			                widget->allocation.width/2 - strwth/2, 27, buffer);

			angle = ((float)(gauge->value-gauge->min)/(gauge->max-gauge->min)*120+30)/180.0*M_PI;
/*			fprintf(stderr, "%d %d %d %d %f %d %d %d\n", widget->allocation.width/2, widget->allocation.height-3,
              widget->allocation.width/2-(int)(yradius*cos(angle)),
              yradius-(int)(yradius*sin(angle)),
			        angle,
							gauge->min, gauge->max, gauge->value);
*/
			/* Draw a rectangle around */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_ON]));
/*			gdk_draw_rectangle (gauge->pixmap,
			  gauge->gc,
			  FALSE,
			  0, 0,
			  widget->allocation.width-1,
			  widget->allocation.height-1);
*/
			/* Draw the pointer */
			gdk_gc_set_line_attributes(gauge->gc, 2, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_MITER);
			gdk_draw_line (gauge->pixmap, gauge->gc,
			        widget->allocation.width/2, widget->allocation.height-BORDER,
			        widget->allocation.width/2-(int)(yradius*cos(angle)),
			        widget->allocation.height-(int)(yradius*sin(angle))-BORDER);
			gdk_gc_set_line_attributes(gauge->gc, 1, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_MITER);

			/* Draw the inner arc which covers the pointer */
			gdk_gc_set_foreground(gauge->gc, &(gauge->fg[GRAPH_COLOR_GRID]));
			gdk_draw_arc(gauge->pixmap, gauge->gc,
			        TRUE,
			        (int)(widget->allocation.width/2-0.3*yradius+0.5),
			        (int)(widget->allocation.height-3-0.3*yradius+0.5),
							(int) (2*yradius*0.3), (int) (2*yradius*0.3),
							0*64, 180*64);

			gdk_draw_pixmap (widget->window,
			      gauge->gc,
			      gauge->pixmap,
			      0, 0,
			      x, y,
			      widget->allocation.width,
			      widget->allocation.height);
		}
	}
	return FALSE;
}
/* EOF */

