
/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8 c-style: "K&R" -*- */

/**************************************************************************
**       Title: grab one gray image using libdc1394
**    $RCSfile: rec_image.c,v $
**   $Revision: 1.4 $$Name:  $
**       $Date: 2006/01/31 14:18:03 $
**   Copyright: LGPL $Author: gerber $
** Description:
**
**    Get one gray image using libdc1394 and store it as portable gray map
**    (pgm). Based on 'samplegrab' from Chris Urmson 
**
**-------------------------------------------------------------------------
**
**  $Log: rec_image.c,v $
**  Revision 1.4  2006/01/31 14:18:03  gerber
**  version 0.3.2
**
**  Revision 1.3  2005/02/26 16:29:19  gerber
**  parameter flags (parameter_logic) defined as gboolean, added scale_type to struct __GpivPostPar for inverse scaling
**
**  Revision 1.2  2005/01/19 15:45:58  gerber
**  all executables start with gpiv_
**
**  Revision 1.1  2005/01/19 10:36:03  gerber
**  Initiation of Data Acquisition; triggering of lasers and camera, recording images
**
**  Revision 1.2  2001/09/14 08:10:41  ronneber
**  - some cosmetic changes
**
**  Revision 1.1  2001/07/24 13:50:59  ronneber
**  - simple test programs to demonstrate the use of libdc1394 (based
**    on 'samplegrab' of Chris Urmson
**
**
**************************************************************************/

#include <stdio.h>
#ifndef DISABLE_DAC
#include <libraw1394/raw1394.h>
#include <libdc1394/dc1394_control.h>
#endif /* DISABLE_DAC */
#include <stdlib.h>
#include <getopt.h>
#include <gpiv.h>
#include "system.h"
#ifndef DISABLE_DAC
#include "rec_image.h"
#endif /* DISABLE_DAC */
#include <unistd.h> /* unsigned int sleep(unsigned int seconds); */


#define IMAGE_NAME_DEFAULT "Image.pgm"
#define EXIT_FAILURE 1
#define RCSID "$Id: rec_image.c,v 1.4 2006/01/31 14:18:03 gerber Exp $"

#ifndef DISABLE_DAC

char *xmalloc ();
char *xrealloc ();
char *xstrdup ();


char *program_name;

/* Option flags and variables */

int want_verbose;		/* --verbose */
char *fname;                    /* use file instead of stdout */
gboolean fname_logic;

static struct option const long_options[] =
{
        {"help", no_argument, 0, 'h'},
        {"version", no_argument, 0, 'v'},
        {"filename", required_argument, 0, 'f'},
        {"verbose", no_argument, 0, 'V'},
        {NULL, 0, NULL, 0}
};



static void
usage (int status
       )
/*-----------------------------------------------------------------------------
 */ 
{
        if (want_verbose) printf("entering usage");
        printf (_("%s - \
captures images from a IIDC-compliant camera with IEE1394 connection\n"), 
                program_name);

        printf (_("Usage: %s [OPTION]... [FILE]...\n"), program_name);

        printf (_("\
Options:\n\
  -f, --filename             use file for image storage instead of stdout \n\
  -h, --help                 display this help and exit \n\
  -v, --version              output version information and exit \n\
  -V, --verbose              print more information \n\
"));

        if (want_verbose) printf("leaving usage");

        exit (status);
}



static int
decode_switches (int argc, 
                 char **argv
                 )
/*-----------------------------------------------------------------------------
 */ 
{
        int c;


        while ((c = getopt_long (argc, argv, 
                                 "h"	/* help */
                                 "v"	/* version */
                                 "f"        /* use file instead of stdout */
                                 "V",	/* verbose */
                                 long_options, (int *) 0)) != EOF) {
                switch (c)
                        {
                        case 'v':
                                printf("%s\n%s\n", program_name, RCSID);
                                exit (0);
                                
                        case 'h':
                                usage (0);
                                
                        case 'f':
                                fname = xstrdup(optarg);
                                fname_logic = TRUE;
                                break;
                                
                        case 'V':
                                want_verbose = 1;
                                break;
                                
                        default:
                                usage (EXIT_FAILURE);
                        }
        }
        
        return optind;
}




int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
 */ 
{
        FILE* imagefile;
        dc1394_cameracapture capture;
        quadlet_t value;
        GpivCamVar cam_var;
        gint i;
        gchar *err_msg = NULL;
/*
 * from coriander:
 */
/*         chain_t **image_pipes; */
/*         Format7Info *format7_info; */
/*         UIInfo *uiinfos; */
/*         SelfIdPacket_t *selfids; */


        program_name = argv[0];
        if (want_verbose) printf("starting %s", program_name);
        i = decode_switches (argc, argv);
 
        if ((err_msg = gpiv_dac_get_camvar(&cam_var)) != NULL) {
                g_error("from: rec_image->main\n from: gpiv_dac_get_dacvar\n %s", 
                        err_msg);
        }

/*
 * coriander: IsoStartThread->on_button_receive
 */
        if (dc1394_setup_capture(cam_var.camera[0].handle, 
                                 cam_var.camera[0].id, 
                                 cam_var.misc_info[0].iso_channel, 
                                 cam_var.misc_info[0].format, 
                                 cam_var.misc_info[0].mode, 
                                 /* maxspeed */ SPEED_400,
                                 /* cam_var.misc_info[0].framerate */ FRAMERATE_7_5, 
                                 &capture)
            != DC1394_SUCCESS) {
/*                         info->receive_method=RECEIVE_METHOD_RAW1394; */
                dc1394_release_camera(cam_var.handle, &capture);
                raw1394_destroy_handle(cam_var.handle);
                g_error("unable to setup camera-\n"
                        "check line %d of %s to make sure\n"
                        "that the video mode,framerate and format are\n"
                        "supported by your camera\n",
                        __LINE__,__FILE__);
        }  else {
                if (want_verbose) g_message("dc1394_setup_capture: succes");
        }

/* 
 * set trigger mode
 * BUGFIX: disabled temporarly
 */
        if( dc1394_set_trigger_mode(cam_var.handle, 
                                    /* capture.node */ cam_var.camera[0].id, 
                                    TRIGGER_MODE_0)
            != DC1394_SUCCESS) {
/*     dc1394_release_camera(cam_var.handle,&capture); */
/*     raw1394_destroy_handle(cam_var.handle); */
                g_warning("unable to set camera trigger mode\n");
        } else {
                if (want_verbose) g_message("dc1394_set_trigger_mode: succes");
        }
  
  
/*
 * report camera's feature set
 */
        dc1394_print_feature_set(&cam_var.feature_set[0]);
    

/*
 * test on single feature
 */
        if (want_verbose 
            && dc1394_query_basic_functionality(cam_var.handle, 
                                                capture.node, 
                                                &value)
            != DC1394_SUCCESS) {
                g_warning("dc1394_query_basic_functionality: failed\n");
        } else {
                if ((value>>(8*sizeof(quadlet_t)-19-1)) != 0) {
                        g_warning("single shot supperted");
                } else {
                        g_warning("single shot _NOT_ supperted");
                }
        }

/*
 * have the camera start sending us data
 * on_iso_start_clicked
 */
        if (dc1394_start_iso_transmission(cam_var.handle, capture.node)
            != DC1394_SUCCESS) {
                dc1394_release_camera(cam_var.handle, &capture);
                raw1394_destroy_handle(cam_var.handle);
                g_error("unable to start camera iso transmission");
        } else {
                if (want_verbose) g_message("dc1394_start_iso_transmission: succes");
        }

        sleep(1);
/*
 * capture one frame
 * on_service_iso_toggled->IsoStartThread
 */
        if (dc1394_single_capture(cam_var.handle, &capture) != DC1394_SUCCESS) {
                dc1394_release_camera(cam_var.handle, &capture);
                raw1394_destroy_handle(cam_var.handle);
                g_error("unable to capture a frame");
        } else {
                if (want_verbose) g_message("dc1394_single_capture: succes");
        }

/*
 * Stop data transmission
 */
        if (dc1394_stop_iso_transmission(cam_var.handle, capture.node) 
            != DC1394_SUCCESS) {
                g_warning("couldn't stop the camera?\n");
        } else {
                if (want_verbose) g_message("dc1394_stop_iso_transmissio: succes");
        }

/*
 * save image as 'Image.pgm'
 */
        imagefile = fopen(IMAGE_NAME_DEFAULT, "w");

        if( imagefile == NULL) {
                dc1394_release_camera(cam_var.handle, &capture);
                raw1394_destroy_handle(cam_var.handle);
                g_error( "Can't create '" IMAGE_NAME_DEFAULT "'");
        } else {
                if (want_verbose) g_message("imagefile == NULL: succes");
        }
  
    
        if (want_verbose) g_message("width = %d  height = %d",
                                    capture.frame_width, 
                                    capture.frame_height);

        fprintf(imagefile, "P5\n%u %u 255\n", capture.frame_width,
                capture.frame_height );
        fwrite((const char *)capture.capture_buffer, 1,
               capture.frame_height*capture.frame_width, imagefile);
        fclose(imagefile);
        g_message("wrote: " IMAGE_NAME_DEFAULT "\n");

/*
 *  Close camera
 */
        dc1394_release_camera(cam_var.handle, &capture);

        free(cam_var.camera);
        free(cam_var.feature_set);
        free(cam_var.misc_info);
/*         free(image_pipes); */
/*         free(format7_info); */
/*         free(uiinfos); */
/*         free(selfids); */

        raw1394_destroy_handle(cam_var.handle);
        return 0;
}



void
GetFormat7Capabilities(raw1394handle_t handle, 
                       nodeid_t node, 
                       Format7Info *info
)
/*-----------------------------------------------------------------------------
 */
{
        int i, f, err;
        quadlet_t value;
  
        err = dc1394_query_supported_modes(handle, node, 
                                           FORMAT_SCALABLE_IMAGE_SIZE, &value);
        if (!err) {
                g_error("Could not query Format7 supported modes");
        } else {
                for (i=0,f=MODE_FORMAT7_MIN;f<MODE_FORMAT7_MAX;f++,i++) {
                        info->mode[i].present = (value & (0x1<<(31-i)) );
/*
 *check for mode presence before query
 */

                        if (info->mode[i].present) {
                                err=1;
                                err *= dc1394_query_format7_max_image_size
                                        (handle,node,f,
                                         &info->mode[i].max_size_x,
                                         &info->mode[i].max_size_y);
                                err *= dc1394_query_format7_unit_size
                                        (handle,node,f,&info->mode[i].step_x,
                                         &info->mode[i].step_y);
                                err *= dc1394_query_format7_image_position
                                        (handle,node,f,&info->mode[i].pos_x,
                                         &info->mode[i].pos_y);
                                err *= dc1394_query_format7_image_size
                                        (handle,node,f,&info->mode[i].size_x,
                                         &info->mode[i].size_y);
                                
                                err *= dc1394_query_format7_pixel_number
                                        (handle,node,f,&info->mode[i].pixnum);
                                err *= dc1394_query_format7_byte_per_packet
                                        (handle,node,f,&info->mode[i].bpp);
                                err *= dc1394_query_format7_packet_para
                                        (handle,node,f,&info->mode[i].min_bpp,
                                         &info->mode[i].max_bpp);
                                err *= dc1394_query_format7_total_bytes
                                        (handle,node,f,
                                         &info->mode[i].total_bytes);
                                
/*
 * TODO: get color coding id
 */
                                err *= dc1394_query_format7_color_coding
                                        (handle,node,f,
                                         &info->mode[i].
                                         color_coding);
                                if (!err) g_error("Got a problem querying format7 capabilitie");
                        }
                }
        }

        info->edit_mode = MODE_FORMAT7_MIN;
}

#else

int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
 */ 
{
        g_error("%s: this is dead code as DISABLE_DAC is set",
                argv[0]);
        return -1;
}

#endif /* DISABLE_DAC */
