/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------

gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
  libraries.

Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

This file is part of gpiv.

Gpiv is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
* (callback) functions for the display
* $Log: display.c,v $
* Revision 1.19  2006/01/31 14:28:12  gerber
* version 0.3.0
*
* Revision 1.18  2005/06/15 15:03:54  gerber
* Optional Anti Aliased canvas for viewer
*
* Revision 1.17  2005/06/15 09:40:40  gerber
* debugged, optimized
*
* Revision 1.16  2005/02/26 09:17:13  gerber
* structured of interrogate function by using gpiv_piv_isiadapt
*
* Revision 1.15  2005/01/19 15:53:41  gerber
* Initiation of Data Acquisition (DAC); trigerring of lasers and camera
* by using RTAI and Realtime Linux, recording images from IEEE1394
* (Firewire) IIDC compliant camera's
*
* Revision 1.14  2004/10/15 19:24:05  gerber
* GPIV_ and Gpiv prefix to defines and structure names of libgpiv
*
* Revision 1.13  2004/06/14 21:19:23  gerber
* Image depth up to 16 bits.
* Improvement "single int" and "drag int" in Eval tab.
* Viewer's pop-up menu.
* Adaption for gpiv_matrix_* and gpiv_vector_*.
* Resizing console.
* See Changelog for further info.
*
* Revision 1.12  2003/09/01 11:17:14  gerber
* improved monitoring of interrogation process
*
* Revision 1.11  2003/08/22 15:24:52  gerber
* interactive spatial scaling
*
* Revision 1.10  2003/07/31 11:43:26  gerber
* display images in gnome canvas (HOERAreset)
*
* Revision 1.9  2003/07/13 14:38:18  gerber
* changed error handling of libgpiv
*
* Revision 1.8  2003/07/12 21:21:15  gerber
* changed error handling libgpiv
*
* Revision 1.6  2003/07/10 11:56:07  gerber
* added man page
*
* Revision 1.5  2003/07/05 13:14:57  gerber
* drag and drop of a _series_ of filenames from NAUTILUS
*
* Revision 1.4  2003/07/04 10:47:00  gerber
* cleaning up
*
* Revision 1.3  2003/07/03 17:08:02  gerber
* display ruler adjusted for scaled data
*
* Revision 1.2  2003/06/27 13:47:26  gerber
* display ruler, line/point evaluation
*
* Revision 1.1.1.1  2003/06/17 17:10:52  gerber
* Imported gpiv
*
*/

#include "gpiv_gui.h"
#include "utils.h"
#include "display_interface.h"
#include "display.h"
#include "piveval.h"
#include "dialog_interface.h"


enum DataType {
    DATA_TYPE__INTREG,
    DATA_TYPE__PIV
};

static gboolean shift_pressed, ctrl_pressed, alt_pressed;
static gfloat x_offset, x_lower, x_upper, y_offset, y_lower, y_upper;
static gint enable_col_start, enable_col_end, enable_row_start, 
    enable_row_end;
static GdkCursor *cursor;


/*
 * Private display functions
 */


static void
search_nearest_index(Display * disp, 
                     gint x, 
                     gint y, 
                     gint * index_x, 
                     gint * index_y,
                     gint data_type
                     )
/*-----------------------------------------------------------------------------
* Search nearest index belongin to x an y points
*/
{
    gint i, j, x_min = 10e4, y_min = 10e4, dif;

    if (data_type == DATA_TYPE__INTREG) {
        for (i = 0, j = 0; i < disp->intreg.data.ny; i++) {
            dif = abs(y - (int) disp->intreg.data.point_y[i][j]);
            if (dif < y_min) {
                y_min = dif;
                *index_y = i;
            }
        }

        for (i = 0, j = 0; j < disp->intreg.data.nx; j++) {
            dif = abs(x - (int) disp->intreg.data.point_x[i][j]);
            if (dif < x_min) {
                x_min = dif;
                *index_x = j;
            }
        }


    } else if (data_type == DATA_TYPE__PIV) {
        for (i = 0, j = 0; i < disp->gpd.piv_data.ny; i++) {
            dif = abs(y - (int) disp->gpd.piv_data.point_y[i][j]);
            if (dif < y_min) {
                y_min = dif;
                *index_y = i;
            }
        }

        for (i = 0, j = 0; j < disp->gpd.piv_data.nx; j++) {
            dif = abs(x - (int) disp->gpd.piv_data.point_x[i][j]);
            if (dif < x_min) {
                x_min = dif;
                *index_x = j;
            }
        }

    } else {
        g_warning("search_nearest_index: Inavalid Data Type");
    }
    
}



static void
highlight_intreg(gint index_y, 
                 gint index_x, 
                 Display * disp
                 )
/*-----------------------------------------------------------------------------
* Highlights first and second interrogation area's
*/
{
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");

        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active
        && index_y <= disp->intreg.data.ny 
        && index_x <= disp->intreg.data.nx) {
        
        if (disp->intreg.gci_intreg2[disp->index_y_old][disp->index_x_old] 
            != NULL)
            gnome_canvas_item_set(disp->intreg.gci_intreg2
                                  [disp->index_y_old][disp->index_x_old],
                                  "outline_color", "yellow", 
                                  NULL);
        
        if (disp->intreg.gci_intreg1[disp->index_y_old][disp->index_x_old] 
            != NULL)
            gnome_canvas_item_set(disp->intreg.gci_intreg1
                                  [disp->index_y_old][disp->index_x_old], 
                                  "outline_color", "blue", 
                                  NULL);
        
        
        if (disp->intreg.gci_intreg2[index_y][index_x] != NULL) {
            gnome_canvas_item_set(disp->intreg.gci_intreg2[index_y][index_x],
                                  "outline_color", "red",
                                  NULL);
            gnome_canvas_item_raise_to_top(disp->intreg.
                                           gci_intreg2[index_y][index_x]);
        }
        

        
        if (disp->intreg.gci_intreg1[index_y][index_x] != NULL) {
            gnome_canvas_item_set(disp->intreg.gci_intreg1[index_y][index_x],
                                  "outline_color", "green", 
                                  NULL);
            gnome_canvas_item_raise_to_top(disp->intreg.
                                           gci_intreg1[index_y][index_x]);
        }
    }
}



static void
create_msg_display_with_pivdata(GpivPivData piv_data, 
                                gint index_y, 
                                gint index_x,
                                gint scale
                                )
/*-----------------------------------------------------------------------------
 * Displays message with piv data values
*/
{

    if (scale) {
        if (piv_data.snr[index_y][index_x] == GPIV_SNR_NAN) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.2fmm yp=%3.2fmm %s",
                       piv_data.point_x[index_y][index_x] * 1e3,
                       piv_data.point_y[index_y][index_x] * 1e3,
                       _("Disabled: not a number"));
        } else if (piv_data.snr[index_y][index_x] == GPIV_SNR_DISABLE) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.2fmm yp=%3.2fmm %s",
                       piv_data.point_x[index_y][index_x] * 1e3,
                       piv_data.point_y[index_y][index_x] * 1e3,
                       _("Disabled manually"));
        } else {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.2fmm yp=%3.2fmm U=%3.2fm/s V=%3.2fm/s"
                       " snr=%3.2f peak #%d",
                       piv_data.point_x[index_y][index_x] * 1e3,
                       piv_data.point_y[index_y][index_x] * 1e3,
                       piv_data.dx[index_y][index_x],
                       piv_data.dy[index_y][index_x],
                       piv_data.snr[index_y][index_x],
                       piv_data.peak_no[index_y][index_x]);
        }

    } else {
        if (piv_data.snr[index_y][index_x] == GPIV_SNR_NAN) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.0fpx yp=%3.0fpx %s",
                       piv_data.point_x[index_y][index_x],
                       piv_data.point_y[index_y][index_x],
                       _("Disabled: not a number"));
        } else if (piv_data.snr[index_y][index_x] == GPIV_SNR_DISABLE) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.0fpx yp=%3.0fpx %s",
                       piv_data.point_x[index_y][index_x],
                       piv_data.point_y[index_y][index_x],
                       _("Disabled manually"));
        } else {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%3.0fpx yp=%3.0fpx dx=%3.2fpx dy=%3.2fpx"
                       " snr=%3.2f peak #%d",
                       piv_data.point_x[index_y][index_x],
                       piv_data.point_y[index_y][index_x],
                       piv_data.dx[index_y][index_x],
                       piv_data.dy[index_y][index_x],
                       piv_data.snr[index_y][index_x],
                       piv_data.peak_no[index_y][index_x]);
        }
    }

}



static void
create_msg_display_with_scdata(GpivScalarData sc_data,
                               gchar *sc_type,
                               gint index_y, 
                               gint index_x,
                               gint scale
                               )
/*-----------------------------------------------------------------------------
 * Displays message with scalar data values
 */
{
    if (scale) {
        g_snprintf(msg_display, GPIV_MAX_CHARS, 
                   "xp=%3.2fmm yp=%3.2fmm %s=%3.2f1/s",
                   sc_data.point_x[index_y][index_x] * 1e3,
                   sc_data.point_y[index_y][index_x] * 1e3,
                   sc_type,
                   sc_data.scalar[index_y][index_x]);
    } else {
        g_snprintf(msg_display, GPIV_MAX_CHARS, 
                   "xp=%3.0fpx yp=%3.0fpx %s=%3.2f",
                   sc_data.point_x[index_y][index_x],
                   sc_data.point_y[index_y][index_x],
                   sc_type,
                   sc_data.scalar[index_y][index_x]);
    }
}



static void
create_msg_display_with_pivscdata(GpivPivData piv_data,
                                  GpivScalarData sc_data,
                                  gchar *sc_type,
                                  gint index_y, 
                                  gint index_x,
                                  gint scale
                                  )
/*-----------------------------------------------------------------------------
 * Displays message with piv and scalar data values
 */
{
    if (scale) {
        g_snprintf(msg_display, GPIV_MAX_CHARS, 
                   "xp=%3.2fmm yp=%3.2fmm U=%3.2fm/s V=%3.2fm/s "
                   "snr=%3.2f peak #%d %s=%3.2f1/s",
                   piv_data.point_x[index_y][index_x] * 1e3,
                   piv_data.point_y[index_y][index_x] * 1e3,
                   piv_data.dx[index_y][index_x],
                   piv_data.dy[index_y][index_x],
                   piv_data.snr[index_y][index_x],
                   piv_data.peak_no[index_y][index_x],
                   sc_type,
                   sc_data.scalar[index_y][index_x]);
    } else {
        g_snprintf(msg_display, GPIV_MAX_CHARS, 
                   "xp=%3.0fpx yp=%3.0fpx dx=%3.2fpx dy=%3.2fpx "
                   "snr=%3.2f peak #%d %s=%3.2f",
                   piv_data.point_x[index_y][index_x],
                   piv_data.point_y[index_y][index_x],
                   piv_data.dx[index_y][index_x],
                   piv_data.dy[index_y][index_x],
                   piv_data.snr[index_y][index_x],
                   piv_data.peak_no[index_y][index_x],
                   sc_type,
                   sc_data.scalar[index_y][index_x]);
    }
}



static void
create_line(Display *disp, 
            gint x1, 
            gint y1, 
            gint x2, 
            gint y2
            )
/*-----------------------------------------------------------------------------
 */
{
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(2);
    
    if (disp != NULL
        && gci_line == NULL) {
    
        points->coords[0] = x1;
        points->coords[1] = y1;
        points->coords[2] = x2;
        points->coords[3] = y2;
        gci_line = 
            gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(disp->canvas)),
                                  gnome_canvas_line_get_type(),
                                  "points", points,
                                  "fill_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
        
        gnome_canvas_points_free(points);
    }
}


static void
update_line(gint x1, 
            gint y1, 
            gint x2, 
            gint y2
            )
/*-----------------------------------------------------------------------------
 */
{
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(2);
    
    if (gci_line != NULL) {
    
        points->coords[0] = x1;
        points->coords[1] = y1;
        points->coords[2] = x2;
        points->coords[3] = y2;
        
        gnome_canvas_item_set(GNOME_CANVAS_ITEM(/* disp-> */gci_line),
                              "points", points,
                              "fill_color", "yellow",
                              "width_units", (double) THICKNESS,
                              NULL);
        
        gnome_canvas_points_free(points);
    }
}


static void
destroy_line()
/*-----------------------------------------------------------------------------
 */
{
    if (gci_line != NULL) {
    
        gtk_object_destroy(GTK_OBJECT(gci_line));
        gci_line = NULL;
    }
}



static void
create_rect(Display *disp, 
            gint x, 
            gint y
            )
/*-----------------------------------------------------------------------------
 */
{
    if (disp != NULL
        && gci_aoi == NULL) {
    
        gci_aoi = 
            gnome_canvas_item_new(gnome_canvas_root
                                  (GNOME_CANVAS(disp->canvas)),
                                  gnome_canvas_rect_get_type(),
                                  "x1", (double) x,
                                  "y1", (double) y,
                                  "x2", (double) x,
                                  "y2", (double) y,
                                  "outline_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
        
    }
}



static void
update_rect(gint x, 
            gint y
            )
/*-----------------------------------------------------------------------------
 */
{
    if (gci_aoi != NULL) {
    
        gnome_canvas_item_set(GNOME_CANVAS_ITEM(gci_aoi),
                              "x2", (double) x,
                              "y2", (double) y, 
                              NULL);
    }
}



static void
destroy_rect()
/*-----------------------------------------------------------------------------
 */
{
    if (gci_aoi != NULL) {
    
        gtk_object_destroy(GTK_OBJECT(gci_aoi));
        gci_aoi = NULL;
    }
}



static void
set__NO_MS(GpivConsole *gpiv,
           Display *disp
           )
/*-----------------------------------------------------------------------------
 * Reset mouse selection to inactive
 */
{
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                (gpiv->piveval->radiobutton_mouse_1), 
                                TRUE);
    cursor = gdk_cursor_new(GDK_DOTBOX);
    gdk_window_set_cursor(disp->mwin->window, cursor);
}



static void
canvas_display_motion_notify__NO_MS(Display *disp, 
                                    gint x, 
                                    gint y, 
                                    gint *index_x, 
                                    gint *index_y
                                    )
/*-----------------------------------------------------------------------------
 * Moves pointer over the display canvas if no mouse selection (NO_MS) is set
 */
{
    GpivPivData piv_data = disp->gpd.piv_data;
    GpivPivData piv_data_scaled = disp->gpd.piv_data_scaled;
    GpivScalarData vor_data = disp->gpd.vor_data;
    GpivScalarData vor_data_scaled = disp->gpd.vor_data_scaled;
    GpivScalarData sstrain_data = disp->gpd.sstrain_data;
    GpivScalarData sstrain_data_scaled = disp->gpd.sstrain_data_scaled;
    GpivScalarData nstrain_data = disp->gpd.nstrain_data;
    GpivScalarData nstrain_data_scaled = disp->gpd.nstrain_data_scaled;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");
    
    *index_y = 0;
    *index_x = 0;
    
/*
 * BUGFIX REPAIRED 26.5.03
 * if area of interest is created without displaying intregs, no intregs are
 * created, resulting into no displaying of piv values
 * Check on existance of intregs here

 * Changed policy 8..4.05; if no intregs exist and shown, piv values will be 
 * displayed, 
 */
    if ((disp->gpd.exist_piv && disp->display_piv
         && GTK_CHECK_MENU_ITEM(view_piv_display0)->active
         && piv_data.ny == disp->intreg.data.ny 
         && piv_data.nx == disp->intreg.data.nx
         && piv_data.point_y[0][0] == disp->intreg.data.point_y[0][0]
         && piv_data.point_x[0][0] == disp->intreg.data.point_x[0][0]
         && piv_data.point_y[piv_data.ny - 1][piv_data.nx - 1] == 
         disp->intreg.data.point_y[disp->intreg.data.ny - 1]
         [disp->intreg.data.nx - 1]
         && piv_data.point_x[piv_data.ny - 1][piv_data.nx - 1] == 
         disp->intreg.data.point_x[disp->intreg.data.ny - 1]
         [disp->intreg.data.nx - 1] )
        || (disp->gpd.exist_piv && disp->display_piv
            && !GTK_CHECK_MENU_ITEM(view_piv_display0)->active)
        ) {
        
        search_nearest_index(disp, x, y, index_x, index_y, DATA_TYPE__PIV);
        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
            highlight_intreg(*index_y, *index_x, disp);
        }
        

        if (disp->gpd.exist_vor && disp->display_vor) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_pivscdata(piv_data_scaled, 
                                                  vor_data_scaled,
                                                  "vor",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_pivscdata(piv_data, 
                                                  vor_data,
                                                  "vor",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            }
            
        } else if (disp->gpd.exist_sstrain && disp->display_sstrain) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_pivscdata(piv_data_scaled, 
                                                  sstrain_data_scaled,
                                                  "sstrain",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_pivscdata(piv_data, 
                                                  sstrain_data,
                                                  "sstrain",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            }
            
        } else if (disp->gpd.exist_nstrain && disp->display_nstrain) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_pivscdata(piv_data_scaled, 
                                                  nstrain_data_scaled,
                                                  "nstrain",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_pivscdata(piv_data, 
                                                  nstrain_data,
                                                  "nstrain",
                                                  *index_y, 
                                                  *index_x,
                                                  disp->gpd.scaled_piv);
            }
            
        } else {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_pivdata(piv_data_scaled,
                                                *index_y, 
                                                *index_x,
                                                disp->gpd.scaled_piv);
                
            } else {
                create_msg_display_with_pivdata(piv_data,
                                                *index_y, 
                                                *index_x,
                                                disp->gpd.scaled_piv);
            }
        }
        

/*
 * piv data exist, but are not displayed
 * displays eventually piv-derived scalar data
 */
    } else if ((disp->gpd.exist_piv 
                && !disp->display_piv
                && GTK_CHECK_MENU_ITEM(view_piv_display0)->active
                && piv_data.ny == disp->intreg.data.ny 
                && piv_data.nx == disp->intreg.data.nx
                && piv_data.point_y[0][0] == disp->intreg.data.point_y[0][0]
                && piv_data.point_x[0][0] == disp->intreg.data.point_x[0][0]
                && piv_data.point_y[piv_data.ny - 1][piv_data.nx - 1] == 
                disp->intreg.data.point_y[disp->intreg.data.ny - 1]
                [disp->intreg.data.nx - 1]
                && piv_data.point_x[piv_data.ny - 1][piv_data.nx - 1] == 
                disp->intreg.data.point_x[disp->intreg.data.ny - 1]
                [disp->intreg.data.nx - 1]
                && ((disp->gpd.exist_vor && disp->display_vor)
                    || (disp->gpd.exist_sstrain && disp->display_sstrain)
                    || (disp->gpd.exist_nstrain && disp->display_nstrain))
                )
               || (disp->gpd.exist_piv 
                   && !disp->display_piv
                   && !GTK_CHECK_MENU_ITEM(view_piv_display0)->active
                   && ((disp->gpd.exist_vor && disp->display_vor)
                       || (disp->gpd.exist_sstrain && disp->display_sstrain)
                       || (disp->gpd.exist_nstrain && disp->display_nstrain))
                   )
               ) {
        
        search_nearest_index(disp, x, y, index_x, index_y, 
                                 DATA_TYPE__PIV);
        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
            highlight_intreg(*index_y, *index_x, disp);
        }

        if (disp->gpd.exist_vor && disp->display_vor) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_scdata(vor_data_scaled,
                                               "vor",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_scdata(vor_data,
                                               "vor",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            }
            
        } else if (disp->gpd.exist_sstrain && disp->display_sstrain) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_scdata(sstrain_data_scaled,
                                               "sstrain",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_scdata(sstrain_data,
                                               "sstrain",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            }
            
        } else if (disp->gpd.exist_nstrain && disp->display_nstrain) {
            if (disp->gpd.scaled_piv) {
                create_msg_display_with_scdata(nstrain_data_scaled,
                                               "nstrain",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            } else {
                create_msg_display_with_scdata(nstrain_data,
                                               "nstrain",
                                               *index_y, 
                                               *index_x,
                                               disp->gpd.scaled_piv);
            }
        }
        
/*
 * PIV data (and resulting derivatives) do not exist are and not displayed
 * Interrogation area's are displayed
 */
    } else if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        search_nearest_index(disp, x, y, index_x, index_y, DATA_TYPE__INTREG);
        highlight_intreg(*index_y, *index_x, disp);
        
        g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%3.2f y=%3.2f i=%d j=%d ",
                   disp->intreg.data.point_x[*index_y][*index_x],
                   disp->intreg.data.point_y[*index_y][*index_x], 
                   *index_y, *index_x);
        
        
/*
 * Only image is displayed
 */
    } else if (disp->display_img1) {
        if (disp->gpd.scaled_piv) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%d px  ==>  %5.4f m   yp=%d px  ==>  %5.4f m   img #1: pixval=%d",
                       x,
                       disp->img.image_par.s_scale * x * 1e-3
                       + disp->img.image_par.z_off_x, 
                       y,
                       disp->img.image_par.s_scale * y * 1e-3
                       + disp->img.image_par.z_off_y,
                       disp->img.img1[y][x]
                       );
        } else {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%d px yp=%d px   img #1: pixval=%d",
                       x, y,
                       disp->img.img1[y][x]
                       );
        }
        
    } else if (disp->display_img2) {
        if (disp->gpd.scaled_piv) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%d px  ==>  %5.4f m   yp=%d px  ==>  %5.4f m   img #1: pixval=%d",
                       x,
                       disp->img.image_par.s_scale * x  * 1e-3
                       + disp->img.image_par.z_off_x, 
                       y,
                       disp->img.image_par.s_scale * y * 1e-3
                       + disp->img.image_par.z_off_y,
                       disp->img.img2[y][x]
                       );
        } else {
            g_snprintf(msg_display, GPIV_MAX_CHARS, 
                       "xp=%d px yp=%d px   img #2: pixval=%d",
                       x, y,
                       disp->img.img2[y][x]
                       );
        }
        
    } else {
        g_snprintf(msg_display, GPIV_MAX_CHARS, "No data are displayed");
    }
    
}



static void
canvas_display_motion_notify__DRAG_MS(Display *disp, 
                                      gint x, 
                                      gint y,
                                      gint *index_x, 
                                      gint *index_y
                                      )
/*-----------------------------------------------------------------------------
 * displaces interrogation area nearest to pointer
 */
{
    if (disp->intreg.exist_int) {
        if (!disp->index_old) {
            disp->index_x_old = *index_x;
            disp->index_y_old = *index_y;
            disp->index_old = TRUE;
        }
        
        if (*index_x == disp->index_x_old 
            && *index_y == disp->index_y_old) {
            
            gnome_canvas_item_set 
                (GNOME_CANVAS_ITEM(disp->intreg.
                                   gci_intreg1[*index_y][*index_x]), 
                 "x1", (double) x - piv_eval_par.int_size_1 / 2,
                 "y1", (double) y - piv_eval_par.int_size_1 / 2,
                 "x2", (double) x + piv_eval_par.int_size_1 / 2,
                 "y2", (double) y + piv_eval_par.int_size_1 / 2,
                 NULL);
            
            gnome_canvas_item_set 
                (GNOME_CANVAS_ITEM(disp->intreg.
                                   gci_intreg2[*index_y][*index_x]), 
                 "x1", (double) x - piv_eval_par.int_size_2 / 
                 2 + piv_eval_par.pre_shift_col,
                 "y1", (double) y - piv_eval_par.int_size_2 / 
                 2 + piv_eval_par.pre_shift_row,
                 "x2", (double) x + piv_eval_par.int_size_2 / 
                 2 + piv_eval_par.pre_shift_col,
                 "y2", (double) y + piv_eval_par.int_size_2 / 
                 2 + piv_eval_par.pre_shift_row,
                 NULL);
/*
 * put the interrogation area back to its original location
 */
        } else {
            update_intreg1(disp, disp->index_y_old, disp->index_x_old);
            update_intreg2(disp, disp->index_y_old, disp->index_x_old);
        }
    }
}


static void
canvas_display_motion_notify__SINGLE_POINT_MS(Display *disp, 
                                              gint x, 
                                              gint y
                                              )
/*-----------------------------------------------------------------------------
* Interrogates at a single arbitrary point in the image.
* I am using the first interrogation area ([0][0]) temporarly for 
* displaying
*/
{
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");

    if (disp->intreg.gci_intreg1[0][0] != NULL && 
        disp->intreg.gci_intreg2[0][0] != NULL
        ) {
        if (!GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {

           if (disp->intreg.gci_intreg1[0][0] != NULL) {
             gnome_canvas_item_show 
                (GNOME_CANVAS_ITEM(disp->intreg.gci_intreg1[0][0]));
           }
           if (disp->intreg.gci_intreg2[0][0] != NULL) {
               gnome_canvas_item_show 
                   (GNOME_CANVAS_ITEM(disp->intreg.gci_intreg2[0][0]));
           }
           disp->display_intregs = TRUE;
        }
        
        if (disp->gpd.exist_piv && disp->gpd.scaled_piv) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%3.2f mm  y=%3.2f mm", 
                       (gfloat) x * image_par.s_scale *10e2
                       + (gfloat) image_par.z_off_x *10e2,
                       (gfloat) y * image_par.s_scale *10e2
                       + (gfloat) image_par.z_off_y *10e2);
        } else {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", x, y);
        }
        
        gnome_canvas_item_set 
            (GNOME_CANVAS_ITEM(disp->intreg.gci_intreg1[0][0]), 
             "x1", (double) x - piv_eval_par.int_size_1 / 2,
             "y1", (double) y - piv_eval_par.int_size_1 / 2,
             "x2", (double) x + piv_eval_par.int_size_1 / 2,
             "y2", (double) y + piv_eval_par.int_size_1 / 2,
             NULL);
        
        gnome_canvas_item_set 
            (GNOME_CANVAS_ITEM(disp->intreg.gci_intreg2[0][0]), 
             "x1", (double) x - piv_eval_par.int_size_1 / 2 
             + piv_eval_par.pre_shift_col,
             "y1", (double) y - piv_eval_par.int_size_1 / 2 
             + piv_eval_par.pre_shift_row,
             "x2", (double) x + piv_eval_par.int_size_1 / 2 
             + piv_eval_par.pre_shift_col,
             "y2", (double) y + piv_eval_par.int_size_1 / 2 
             + piv_eval_par.pre_shift_row,
             NULL);
        
    } else {
        create_intreg1(disp, 0, 0);
        create_intreg2(disp, 0, 0);
    }
    
}



static char *
canvas_display_button_press__SINGLE_AREA_MS(GpivConsole *gpiv,
                                            Display *disp,
                                            gint x, 
                                            gint y
                                            )
/*-----------------------------------------------------------------------------
* Performs action on mouse button press when SINGLE_AREA_MS has been enabled
*/
{
    char * err_msg = NULL;
    gint index_x = 0, index_y = 0;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");
    
    if (disp->gpd.exist_piv
/*         && GTK_CHECK_MENU_ITEM(view_piv_display0)->active */
/*         && disp->intreg.data.nx == disp->gpd.piv_data.nx */
/*         && disp->intreg.data.ny == disp->gpd.piv_data.ny */

        ) {

/*         search_nearest_index(disp, x, y, &index_x, &index_y,  */
/*                              DATA_TYPE__INTREG); */
        search_nearest_index(disp, x, y, &index_x, &index_y, 
                             DATA_TYPE__PIV);

        
        if (m_select == SINGLE_AREA_MS) {
/*             if (disp->intreg.data.point_x[index_y][index_x] ==  */
/*                 disp->gpd.piv_data.point_x[index_y][index_x] */
/*                 && disp->intreg.data.point_y[index_y][index_x] ==  */
/*                 disp->gpd.piv_data.point_y[index_y][index_x] */
/*                 ) { */
                m_select_index_x = index_x;
                m_select_index_y = index_y;
                gpiv_warning("canvas_display_button_press__SINGLE_AREA_MS:: index_x=%d index_y = %d", index_x, index_y);
/*             } else { */
/*                 set__NO_MS(gpiv, disp); */
/*                 err_msg = "Interrogation area's have to be at the same positions \nas the already existing piv data"; */
/*                 warning_gpiv(err_msg); */
/*                 return err_msg; */
/*             } */
/*         g_message("x=%d point_x=%d y=%d point_x=%d */

        } else if (m_select == DRAG_MS) {
            m_select_index_x = index_x;
            m_select_index_y = index_y;
            disp->gpd.piv_data.point_x[m_select_index_y]
                [m_select_index_x] = (float) x;
            disp->gpd.piv_data.point_y[m_select_index_y]
                [m_select_index_x] = (float) y;
/* BUGFIXED ? */
            if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
                disp->intreg.data.point_x[m_select_index_y]
                    [m_select_index_x] = (float) x;
                disp->intreg.data.point_y[m_select_index_y]
                    [m_select_index_x] = (float) y;
            }
        } else {
            err_msg = "canvas_display_button_press__SINGLE_AREA_MS: should not arrive here";
            error_gpiv(err_msg);
        }
        
    }
    
    return err_msg;
}



static char *
canvas_display_button_release__SINGLE_AREA_MS(GpivConsole *gpiv,
                                              Display *disp,
                                              gint x, 
                                              gint y
                                              )
/*-----------------------------------------------------------------------------
* Performs action on mouse button release when SINGLE_AREA_MS has been enabled
*/
{
    char * err_msg = NULL;
    gint index_x = 0, index_y = 0;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");
    
    if (disp->gpd.exist_piv
/*         && GTK_CHECK_MENU_ITEM(view_piv_display0)->active */
/*         && disp->intreg.data.nx == disp->gpd.piv_data.nx */
/*         && disp->intreg.data.ny == disp->gpd.piv_data.ny */

        ) {
/*
 * There will only be action if the pointer, during pressing the button (as
 * defined in canvas_display_button_press__SINGLE_AREA_MS), is at identical
 * position as during releasing the button.
 */
        search_nearest_index(disp, x, y, &index_x, &index_y, 
                             DATA_TYPE__PIV);
        gpiv_warning("canvas_display_button_release__SINGLE_AREA_MS:: index_x=%d index_y = %d", index_x, index_y);



        if (m_select_index_x == index_x
            && m_select_index_y == index_y
            ) {
            interrogate(&disp->gpd.piv_data, disp->img.img1, 
                        disp->img.img2, image_par, piv_eval_par, gpiv);
            
            if (disp->gpd.gci_vector[m_select_index_y][m_select_index_x] 
                != NULL) {
                update_vector(&disp->gpd, m_select_index_y, m_select_index_x);
            } else {
                create_vector(&disp->gpd, m_select_index_y, m_select_index_x);
            }
            
            /*                 disp->gpd.exist_piv = TRUE; */
            disp->display_piv = TRUE;
            if (m_select == DRAG_MS) {
                if (disp->intreg.
                    gci_intreg1[m_select_index_y][m_select_index_x] 
                    != NULL)
                    update_intreg1(disp, m_select_index_y, m_select_index_x);
                if (disp->intreg.
                    gci_intreg2[m_select_index_y][m_select_index_x] 
                    != NULL) 
                    update_intreg2(disp, m_select_index_y, m_select_index_x);
            }
            
        } else {
            err_msg = "moved pointer too far away from \nintar to be interrogated";
            return err_msg;
        }
        
    } else {
/*         err_msg = "Interrogation area's and piv data must already exist \n and must be of identic quantity!"; */
        err_msg = "Piv data must exist!";
        warning_gpiv(err_msg);
        return err_msg;
    }
    
    
    if (!shift_pressed) set__NO_MS(gpiv, disp);    
    return err_msg;
}



static void
canvas_display_button_press__SINGLE_POINT_MS(GpivConsole *gpiv,
                                             Display *disp,
                                             gint x, 
                                             gint y
                                             )
/*-----------------------------------------------------------------------------
* Performs action on mouse button release when SINGLE_POINT_MS has 
* been enabled
*/
{
    GtkWidget *view_piv_display0 = 
                        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                                            "view_piv_display0");

    if (disp->gpd.exist_piv) {
        destroy_all_vectors(&disp->gpd);
        gpiv_free_pivdata (&disp->gpd.piv_data);
        disp->gpd.exist_piv = FALSE;
        if (disp->gpd.scaled_piv) {
            gpiv_free_pivdata(&disp->gpd.piv_data_scaled);
            disp->gpd.scaled_piv = FALSE;
        }
    }
    
    free_post_bufmems(disp);
    if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        destroy_all_intregs(disp);
    }
    m_select_index_x = 0;
    m_select_index_y = 0;
    piv_eval_par.int_point_col = (float) x;
    piv_eval_par.int_point_row = (float) y;
    disp->gpd.piv_data.nx = 1; 
    disp->gpd.piv_data.ny = 1; 
    gpiv_alloc_pivdata(&disp->gpd.piv_data);
    disp->gpd.exist_piv = TRUE;
    
    disp->gpd.piv_data.point_x[0][0] = (float) x;
    disp->gpd.piv_data.point_y[0][0] = (float) y;
    
    disp->intreg.data.nx = 1; 
    disp->intreg.data.ny = 1; 
    create_all_intregs(disp);
    disp->intreg.exist_int = TRUE;
    
    disp->intreg.data.point_x[0][0] = (float) x; 
    disp->intreg.data.point_y[0][0] = (float) y;
    show_all_intregs(disp);
    disp->display_intregs = TRUE;
    
    interrogate(&disp->gpd.piv_data, 
                disp->img.img1, 
                disp->img.img2,
                image_par, 
                piv_eval_par,
                gpiv);
    disp->display_piv = TRUE;
    
    create_all_vectors(&disp->gpd);
    gnome_canvas_update_now(GNOME_CANVAS(disp->canvas));
    
    if (!shift_pressed) set__NO_MS(gpiv, disp);
}



static void
canvas_display_button_press__DISABLE_POINT_MS(Display *disp, 
                                              gint x, 
                                              gint y
                                              )
/*-----------------------------------------------------------------------------
* Performs action on mouse button press when DISABLE_POINT_MS has 
* been enabled
*/
{
    gchar *err_msg;
    gint index_x = 0, index_y = 0;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");
    
    if ( m_select == DISABLE_POINT_MS ) {
        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
            search_nearest_index(disp, x, y, &index_x, &index_y,
                                 DATA_TYPE__INTREG);
        } else {
            search_nearest_index(disp, x, y, &index_x, &index_y,
                                 DATA_TYPE__PIV);
        }

        disp->gpd.piv_data.peak_no[index_y][index_x] = -1;
        disp->gpd.piv_data.snr[index_y][index_x] = GPIV_SNR_DISABLE;
        
        if (piv_post_par.set == TRUE) {
            disp->gpd.piv_data.dx[index_y][index_x] = 
                piv_post_par.set_dx;
            disp->gpd.piv_data.dy[index_y][index_x] = 
                piv_post_par.set_dx;
        }
        update_vector(&disp->gpd, index_y, index_x);
        
    } else if ( m_select == ENABLE_AREA_MS ) {
        enable_col_start = x;
        enable_row_start = y;
        assert( gci_aoi == NULL);
        gci_aoi = 
            gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(disp->canvas)
						    ),
                                  gnome_canvas_rect_get_type(),
                                  "x1", (double) x,
                                  "y1", (double) y,
                                  "x2", (double) x,
                                  "y2", (double) y,
                                  "outline_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
        
        
    } else if ( m_select == DISABLE_AREA_MS) {
        enable_col_start = x;
        enable_row_start = y;
        assert( gci_aoi == NULL);
        gci_aoi = 
            gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(disp->canvas)
                                                    ),
                                  gnome_canvas_rect_get_type(),
                                  "x1", (double) x,
                                  "y1", (double) y,
                                  "x2", (double) x,
                                  "y2", (double) y,
                                  "outline_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
    } else {
        err_msg = _("no image or piv data");
        warning_gpiv(err_msg);
    }
    
}



static void
canvas_display_button_release__AOI_MS(Display *disp, 
                                      GpivConsole *gpiv,
                                      GtkWidget * view_piv_display0,
                                      gint x, 
                                      gint y
                                      )
/*-----------------------------------------------------------------------------
* Performs action on mouse button release when AOI_MS has been enabled
*/
{    
    if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        destroy_all_intregs(disp);
    }

    if (x >= disp->xgrab_first) {
        piv_eval_par.col_start = disp->xgrab_first;
        piv_eval_par.col_end = x;
    } else {
        piv_eval_par.col_end = disp->xgrab_first;
        piv_eval_par.col_start = x;
    }
    
    if (y >= disp->ygrab_first) {
        piv_eval_par.row_start = disp->ygrab_first;
        piv_eval_par.row_end = y;
    } else {
        piv_eval_par.row_end = disp->ygrab_first;
        piv_eval_par.row_start = y;
        
    }
    
    if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        create_all_intregs(disp);
    }

    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_colstart), 
                              piv_eval_par.col_start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_rowstart), 
                              piv_eval_par.row_start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_colend), 
                              piv_eval_par.col_end);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_rowend), 
		      piv_eval_par.row_end);
    if (gci_aoi != NULL) destroy_rect();
    
}



static void
canvas_display_button_release__HV_LINE_MS(Display *disp, 
                                         GpivConsole *gpiv,
                                         GtkWidget * view_piv_display0, 
                                         gint x, 
                                         gint y
                                         )
/*-----------------------------------------------------------------------------
* Performs action on mouse button release when H_LINE_MS or V_LINE_MS has 
* been enabled
*/
{
    if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        destroy_all_intregs(disp);
    }

/*     if (disp->gpd.exist_piv) { */
/*         destroy_all_vectors(&disp->gpd); */
/*         disp->display_piv = FALSE; */
/*     } */
    
    
    m_select_index_x = 0;
    m_select_index_y = 0;
    
/*
 * Vertical line selected
 */
    if (m_select == V_LINE_MS) {
        piv_eval_par.int_line_row_end = y;
        
        piv_eval_par.row_start = piv_eval_par.int_line_row_start;
        piv_eval_par.row_end = piv_eval_par.int_line_row_end;
        piv_eval_par.col_start = piv_eval_par.int_line_col -
            piv_eval_par.int_size_2 / 2 
/* + piv_eval_par.pre_shift_col */
            + 1;
        
        piv_eval_par.col_end = piv_eval_par.int_line_col +
            piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_col ;
        
/*
 * Horizontal line selected
 */
    } else if (m_select == H_LINE_MS) {

        piv_eval_par.int_line_col_end = x;
        
        piv_eval_par.col_start = piv_eval_par.int_line_col_start;
        piv_eval_par.col_end = piv_eval_par.int_line_col_end;
        piv_eval_par.row_start = piv_eval_par.int_line_row -
	      piv_eval_par.int_size_2 / 2
/*  + piv_eval_par.pre_shift_row */
            + 1;
        
        piv_eval_par.row_end = piv_eval_par.int_line_row +
            piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_col;
    } else {
        g_warning("canvas_display_button_release__HV_LINE_MS: H_LINE_MS or V_LINE_MS inactive");
    }

    if (gci_line != NULL) destroy_line();
    if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
        create_all_intregs(disp);
    }
/*
* updating entry's for col/row_start_end
*/
      
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_colstart), 
                              piv_eval_par.col_start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_rowstart), 
                              piv_eval_par.row_start);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_colend), 
                              piv_eval_par.col_end);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON
                              (gpiv->piveval->spinbutton_rowend), 
                              piv_eval_par.row_end);
}



/*
* Public display functions
*/


void 
create_img(Display * disp
           )
/* ----------------------------------------------------------------------------
* Creates image in gnome canvas
* row stride; each row is a 4-byte buffer array
*/
{

    GpivImagePar image_par = disp->img.image_par;
    guchar *pos1 = NULL, *pos2 = NULL;
    gint i, j;
/*     GdkPixbuf *pixbuf1 = NULL, *pixbuf2 = NULL; */
    GdkPixbuf *px;
    guint16 fact = 1;
    gint depth = 8;

    assert (disp != NULL);
    assert (disp->img.img1[0] != NULL);
    assert (disp->img.exist_img);
    
    disp->img.pixbuf1 = NULL;
    disp->img.pixbuf2 = NULL;
    
    fact = fact << (image_par.depth - depth);

/*
 * BUGFIX: this works for rowstride
 */
    disp->img.rgb_img_width = gpiv_par.img_width * 3;
    while ((disp->img.rgb_img_width) % 4 != 0) {
        disp->img.rgb_img_width++;
    } 
    
/*
 * this is a more formal way to do it
 */
    px = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
                         FALSE,
                         depth,
                         gpiv_par.img_width,
                         gpiv_par.img_height);

    gdk_pixbuf_unref (px);
/*     g_message(":: rowstride = %d, rgb_img_width = %d", */
/*               gdk_pixbuf_get_rowstride(px), */
/*               disp->img.rgb_img_width */
/*               ); */


/*     disp->img.rgbbuf_img1 = g_malloc(disp->img.rgb_img_width * 3 *  */
/*                                      gpiv_par.img_height); */
    disp->img.rgbbuf_img1 = g_malloc(gdk_pixbuf_get_rowstride(px) *
                                     gpiv_par.img_height);
    
    disp->img.pixbuf1 = gdk_pixbuf_new_from_data(disp->img.rgbbuf_img1,
                                                 GDK_COLORSPACE_RGB,
                                                 FALSE,          /* gboolean has_alpha */
                                                 depth,          /* image_par.depth */
                                                 gpiv_par.img_width, 
                                                 gpiv_par.img_height,
                                                 disp->img.rgb_img_width, /* rowstride */ 
                                                 NULL, 
                                                 NULL);
    
    if (disp->img.gci_img1 != NULL) {
        destroy_img(disp);
    }
    
    disp->img.gci_img1 = 
        gnome_canvas_item_new( gnome_canvas_root( GNOME_CANVAS
                                                  (disp->canvas)),
                               gnome_canvas_pixbuf_get_type (),
                               "pixbuf", disp->img.pixbuf1,
                               NULL);
    
    pos1 = disp->img.rgbbuf_img1;
    
    
    for (i = 0; i < gpiv_par.img_height; i++) {
	for (j = 0; j < gpiv_par.img_width; j++) {
/*             loc_img = disp->img.img1[i][j]; */
/* g_message("create_img:: img1[%d][%d] = %d (guchar) depth = %d img.depth = %d fact = %d", */
/*           i, j, disp->img.img1[i][j],  */
/*           depth, image_par.depth, fact); */
	    *pos1++ =  (guchar) (disp->img.img1[i][j] / fact);
	    *pos1++ =  (guchar) (disp->img.img1[i][j] / fact);
	    *pos1++ =  (guchar) (disp->img.img1[i][j] / fact);
	}
    }

/*     gdk_pixbuf_unref (pixbuf1); */
    gdk_pixbuf_ref (disp->img.pixbuf1);

    
    
    
    if(image_par.x_corr) {
        pos2 = NULL;
        
        disp->img.rgbbuf_img2 = g_malloc(disp->img.rgb_img_width * 3 *
                                         gpiv_par.img_height);
        
        disp->img.pixbuf2 = gdk_pixbuf_new_from_data(disp->img.rgbbuf_img2,
                                                     GDK_COLORSPACE_RGB,
                                                     FALSE,      /* gboolean has_alpha */
                                                     depth,      /* image_par.depth */
                                                     gpiv_par.img_width, 
                                                     gpiv_par.img_height,
                                                     disp->img.rgb_img_width, /* rowstride */ 
                                                     NULL, 
                                                     NULL);
        
        if (disp->img.gci_img2 != NULL) {
            destroy_img(disp);
        }
        
        disp->img.gci_img2 = 
            gnome_canvas_item_new( gnome_canvas_root( GNOME_CANVAS
                                                      (disp->canvas)),
                                   gnome_canvas_pixbuf_get_type (),
                                   "pixbuf", disp->img.pixbuf2,
                                   NULL);

        
        pos2 = disp->img.rgbbuf_img2;
        
        for (i = 0; i < gpiv_par.img_height; i++) {
            for (j = 0; j < gpiv_par.img_width; j++) {
                *pos2++ = (guchar) (disp->img.img2[i][j] / fact);
                *pos2++ = (guchar) (disp->img.img2[i][j] / fact);
                *pos2++ = (guchar) (disp->img.img2[i][j] / fact);
            }
        }
        
/*         gdk_pixbuf_unref (pixbuf2); */
        gdk_pixbuf_ref (disp->img.pixbuf2);
        
    } else {
        disp->img.gci_img2 = disp->img.gci_img2;
    }
    
/*     g_message(":: freeing rgbbuf_img1/2"); */
/*     g_free(disp->img.rgbbuf_img1); */
/*     disp->img.rgbbuf_img1 = NULL; */
/*     g_free(disp->img.rgbbuf_img2); */
/*     disp->img.rgbbuf_img2 = NULL; */
/*     g_message(":: rgbbuf_img1/2 freed SUCCESSFULL"); */
}



void
hide_img1(Display * disp
          )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp != NULL);

    if (disp->img.exist_img && disp->img.gci_img1 != NULL) {
        gnome_canvas_item_hide(GNOME_CANVAS_ITEM(disp->img.gci_img1));
        disp->display_img1 = FALSE;
    }
}



void
show_img1(Display * disp
          )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp != NULL);

    if (disp->img.exist_img && disp->img.gci_img1 != NULL) {
        gnome_canvas_item_show(GNOME_CANVAS_ITEM(disp->img.gci_img1));
        disp->display_img1 = TRUE;
    }
}



void
hide_img2(Display * disp
          )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp != NULL);

    if (disp->img.exist_img && disp->img.gci_img2 != NULL) {
        gnome_canvas_item_hide(GNOME_CANVAS_ITEM(disp->img.gci_img2));
        disp->display_img2 = FALSE;
    }
}


void
show_img2(Display * disp
          )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp != NULL);

    if (disp->img.exist_img && disp->img.gci_img2 != NULL) {
        gnome_canvas_item_show(GNOME_CANVAS_ITEM(disp->img.gci_img2));
        disp->display_img2 = TRUE;
    }
}



void 
destroy_img(Display * disp
            )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp->img.gci_img1 != NULL);
    gtk_object_destroy(GTK_OBJECT
                       (disp->img.gci_img1));
    /*     g_free(disp->img.gci_img1); */
    disp->img.gci_img1 = NULL;
    
    if(image_par.x_corr) {
        gtk_object_destroy(GTK_OBJECT
                           (disp->img.gci_img2));
        disp->img.gci_img2 = NULL;
    }
}



void 
create_background(Display *disp
                  )
/* ----------------------------------------------------------------------------
 *  Displays backgroundcolor
 */
{
    gchar *color = NULL;

/*     g_warning("create_background:: background = %d",gpiv_par.background); */
    if (gpiv_par.background == SHOW_BG_DARKBLUE) {
        color = "darkblue";
    } else if (gpiv_par.background == SHOW_BG_BLACK) {
        color = "black";
    }

    if ( disp->gci_bg != NULL) {
/*         g_warning("create_background:: disp->gci_bg != NULL ==> destroying"); */
        destroy_background(disp);
    }

    disp->gci_bg = 
        gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(disp->canvas)),
                              gnome_canvas_rect_get_type(),
                              "x1", (double) 0,
                              "y1", (double) 0,
                              "x2", (double) gpiv_par.img_width,
                              "y2", (double) gpiv_par.img_height,
                              "fill_color", color,
                              "width_units", 1.0, 
                              NULL);
}



void 
destroy_background(Display *disp
                   )
/*-----------------------------------------------------------------------------
 */
{
    assert (disp->gci_bg != NULL);
    gtk_object_destroy(GTK_OBJECT
                       (disp->gci_bg));
    disp->gci_bg = NULL;
}



void 
create_all_intregs(Display * disp
                   )
/*-----------------------------------------------------------------------------
 *  Creates all interrogation areas
 */
{
    char *err_msg = NULL;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");


    if (disp != NULL) {
/*         if (GTK_CHECK_MENU_ITEM(view_piv_display0)->disabled) { */
        if (!disp->intreg.exist_int) {
            if ((err_msg = create_intreg_data(disp)) != NULL) {
                g_warning ("create_all_intregs: %s", err_msg);
                return;
            }
        }

        create_all_intregs2(disp);
        create_all_intregs1(disp);

        disp->intreg.row_start_old = 0;
        disp->intreg.row_start = piv_eval_par.row_start;
        disp->intreg.row_end = piv_eval_par.row_end;
        disp->intreg.col_start_old = 0;
        disp->intreg.col_start = piv_eval_par.col_start;
        disp->intreg.col_end = piv_eval_par.col_end;
        disp->intreg.int_size_1 = piv_eval_par.int_size_1;
        disp->intreg.int_size_2 = piv_eval_par.int_size_2;
        disp->intreg.int_shift = piv_eval_par.int_shift;
        disp->intreg.pre_shift_row = piv_eval_par.pre_shift_row;
        disp->intreg.pre_shift_col = piv_eval_par.pre_shift_col;
    } else {
        error_gpiv("create_all_intregs: disp != NULL failed");
    }
}



void 
destroy_all_intregs(Display * disp
                    )
/*-----------------------------------------------------------------------------
 * Destroys all interrogation areas
 */
{
    if (disp != NULL) {
        destroy_all_intregs1(disp);
        destroy_all_intregs2(disp);
        destroy_intreg_data(disp);
        disp->display_intregs = FALSE;
    }
}



void 
show_all_intregs(Display * disp
                 )
/*-----------------------------------------------------------------------------
 * Shows all interrogation areas
 */
{
    if (disp != NULL
        && !disp->intreg.exist_int) {
        create_all_intregs(disp);
    }
    show_all_intregs2(disp);
    show_all_intregs1(disp);
}



void 
hide_all_intregs(Display * disp
                 )
/*-----------------------------------------------------------------------------
 * Hides all interrogation areas
 */
{
    if (disp != NULL) {
        hide_all_intregs2(disp);
        hide_all_intregs1(disp);
    }
}



char *
create_intreg_data(Display * disp
                   )
/*-----------------------------------------------------------------------------
 */
{
    char *err_msg = NULL;
    GpivPivData data;
    data.point_x = NULL;
    data.point_y = NULL;
    data.dx = NULL;
    data.dy = NULL;
    data.snr = NULL;
    data.peak_no = NULL;

    if ((err_msg = 
         gpiv_piv_count_pivdata_fromimage(&data, image_par, piv_eval_par))
        != NULL) {
        g_warning ("create_intreg_data: %s", err_msg);
        return err_msg;
    }

    gpiv_alloc_pivdata(&data);
    disp->intreg.exist_int = TRUE;
    if ((err_msg = 
         gpiv_piv_select_int_point(&data, image_par, piv_eval_par))
        != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

    disp->intreg.data = data;
    return err_msg;
}



void
destroy_intreg_data(Display * disp
                    )
/*-----------------------------------------------------------------------------
 */
{
    if (disp != NULL
        && disp->intreg.gci_intreg1[0][0] == NULL
        && disp->intreg.gci_intreg2[0][0] == NULL) {
        gpiv_free_pivdata(&disp->intreg.data);
        disp->intreg.exist_int = FALSE;
    }
}



void 
create_all_intregs1(Display * disp
                    )
/* ----------------------------------------------------------------------------
 * Displays all first interrogation areas
 */
{
    int i = 0, j;
    gint nx = 0;
    gint ny = 0;

    if (disp != NULL
        && !disp->intreg.exist_int)
        create_intreg_data(disp);
    nx = disp->intreg.data.nx;
    ny = disp->intreg.data.ny;
    if (verbose) 
        g_message("create_all_intregs1: creating %d I.A.'s",  nx * ny);
    for (i = 0; i < ny; i++) {
        for (j = 0; j < nx ; j++) {
            create_intreg1(disp, i, j);
        }
    }
}



void 
create_intreg1(Display * disp,
               gint i, 
               gint j
               )
/* ----------------------------------------------------------------------------
 * Displays first interrogation area
 */
{
    int start_x = 0, start_y = 0, end_x = 0, end_y = 0;
    float x, y;

    if (disp != NULL
        && disp->intreg.exist_int) {
        x = disp->intreg.data.point_x[i][j];
        y = disp->intreg.data.point_y[i][j];


/*
 * Using centre points of interr regs
 */
        start_x = (int) x - piv_eval_par.int_size_1 / 2;
        start_y = (int) y - piv_eval_par.int_size_1 / 2;
        end_x = (int) x + piv_eval_par.int_size_1 / 2;
        end_y = (int) y + piv_eval_par.int_size_1 / 2;
        
        if (disp->intreg.gci_intreg1[i][j] != NULL) {
            destroy_intreg1(disp, i, j);
        }
        
        disp->intreg.gci_intreg1[i][j] =
            gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS
                                                    (disp->canvas)),
                                  gnome_canvas_rect_get_type(),
                                  "x1", (double) start_x,
                                  "y1", (double) start_y,
                                  "x2", (double) end_x,
                                  "y2", (double) end_y,
                                  "outline_color", "blue",
                                  "width_units", (double) THICKNESS,
                                  NULL);
    }
}



void 
update_intreg1(Display * disp,
               gint i, 
               gint j
               )
/* ----------------------------------------------------------------------------
 * Updates first interrogation area
 */
{
    int start_x = 0, start_y = 0, end_x = 0, end_y = 0;
    float x, y;

    if (disp != NULL
        && disp->intreg.gci_intreg1[i][j] != NULL) {
        x = disp->intreg.data.point_x[i][j];
        y = disp->intreg.data.point_y[i][j];
    
/*
 * Using centre points of interr regs
 */
        start_x = (int) x - piv_eval_par.int_size_1 / 2;
        start_y = (int) y - piv_eval_par.int_size_1 / 2;
        end_x = (int) x + piv_eval_par.int_size_1 / 2;
        end_y = (int) y + piv_eval_par.int_size_1 / 2;
        
        gnome_canvas_item_set(GNOME_CANVAS_ITEM
                              (disp->intreg.gci_intreg1[i][j]),
                              "x1", (double) start_x,
                              "y1", (double) start_y,
                              "x2", (double) end_x,
                              "y2", (double) end_y,
                              "outline_color", "blue",
                              "width_units", (double) THICKNESS,
                              NULL);
    }
}



void
destroy_intreg1(Display * disp,
                gint i, 
                gint j
                )
/* ----------------------------------------------------------------------------
 * Destroys 1st interrogation area at index i, j
 */
{
    if (disp != NULL
        && disp->intreg.gci_intreg1[i][j] != NULL) {
        gtk_object_destroy(GTK_OBJECT
                           (disp->intreg.gci_intreg1[i][j]));
        disp->intreg.gci_intreg1[i][j] = NULL;
    }
}



void 
show_all_intregs1(Display * disp
                  )
/* ----------------------------------------------------------------------------
 * Shows 1st interrogation areas
 */
{
    int i = 0, j = 0;
    gint nx = 0, ny = 0;

    if (disp != NULL
        && disp->intreg.exist_int) {
    
/*     if (!disp->intreg.exist_int) */
/*         create_intreg_data(disp); */
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;

        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                assert (disp->intreg.gci_intreg1[i][j] != NULL);
                gnome_canvas_item_show(GNOME_CANVAS_ITEM
                                       (disp->intreg.gci_intreg1[i][j]));
            }
        }
    }
}



void 
hide_all_intregs1(Display * disp
                  )
/* ----------------------------------------------------------------------------
 * Hides 1st interrogation areas
 */
{
    int i = 0, j = 0;
    gint nx = 0, ny = 0;

    if (disp != NULL
        && disp->intreg.exist_int) {
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                if (disp->intreg.gci_intreg1[i][j] != NULL) {
                   gnome_canvas_item_hide(GNOME_CANVAS_ITEM
                                           (disp->intreg.gci_intreg1[i][j]));
                }
            }
        }
    }
}


void 
destroy_all_intregs1(Display * disp
                     )
/* ----------------------------------------------------------------------------
 * Destroys 1st interrogation areas
 */
{
    int i = 0, j = 0;
    int nx = 0, ny = 0;

    if (disp != NULL
        && disp->intreg.exist_int) {
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
        if (verbose) 
            g_message("destroy_all_intregs1: destroying %d I.A.'s",  nx * ny);
        
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                destroy_intreg1(disp, i, j);
            }
        }
    }

}



void 
create_all_intregs2(Display * disp
                    )
/* ----------------------------------------------------------------------------
 * Displays all second interrogation areas
 */
{
    int i, j;
    int nx, ny;

    if (disp != NULL
        && !disp->intreg.exist_int)
        create_intreg_data(disp);

    nx = disp->intreg.data.nx;
    ny = disp->intreg.data.ny;
    if (verbose) 
        g_message("create_all_intregs2: creating %d I.A.'s",  nx * ny);
    
    for (i = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            create_intreg2(disp, i, j);
        }
    }
}



void 
create_intreg2(Display * disp,
               gint i, 
               gint j
               )
/* ----------------------------------------------------------------------------
 - Displays second interrogation area
 */
{
    int start_x, start_y, end_x, end_y;
    float x, y;

    if (disp != NULL
        && disp->intreg.exist_int) {
        x = disp->intreg.data.point_x[i][j];
        y = disp->intreg.data.point_y[i][j];

/*
 * with lines
 */
        start_x = (int) x - piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_col;
        start_y = (int) y - piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_row;
        end_x = (int) x + piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_col;
        end_y = (int) y + piv_eval_par.int_size_2 / 2 + 
            piv_eval_par.pre_shift_row;
        
        if (disp->intreg.gci_intreg2[i][j] != NULL) {
            destroy_intreg2(disp, i, j);
        }

        disp->intreg.gci_intreg2[i][j] =
            gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS
                                                    (disp->canvas)), 
                                  gnome_canvas_rect_get_type(),
                                  "x1", (double) start_x,
                                  "y1", (double) start_y,
                                  "x2", (double) end_x,
                                  "y2", (double) end_y,
                                  "outline_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
    }
}



void 
update_intreg2(Display * disp,
               gint i, 
               gint j
               )
/* ----------------------------------------------------------------------------
 * Updates second interrogation area
 */
{
    int start_x, start_y, end_x, end_y;
    float x, y;

    if (disp != NULL
        && disp->intreg.gci_intreg1[i][j] != NULL) {

        x = disp->intreg.data.point_x[i][j] + piv_eval_par.pre_shift_col;
        y = disp->intreg.data.point_y[i][j] + piv_eval_par.pre_shift_row;

/*
 * Using centre points of interr regs
 */
        start_x = (int) x - piv_eval_par.int_size_2 / 2;
        start_y = (int) y - piv_eval_par.int_size_2 / 2;
        end_x = (int) x + piv_eval_par.int_size_2 / 2;
        end_y = (int) y + piv_eval_par.int_size_2 / 2;
        
        if (disp->intreg.gci_intreg2[i][j] != NULL) {
            gnome_canvas_item_set(GNOME_CANVAS_ITEM
                                  (disp->intreg.gci_intreg2[i][j]),
                                  "x1", (double) start_x,
                                  "y1", (double) start_y,
                                  "x2", (double) end_x,
                                  "y2", (double) end_y,
                                  "outline_color", "yellow",
                                  "width_units", (double) THICKNESS,
                                  NULL);
        }
    }
}



void
destroy_intreg2(Display * disp,
                gint i, 
                gint j
                )
/* ----------------------------------------------------------------------------
 * Destroys 2nd interrogation area at index i, j
 */
{
    if (disp != NULL
        && disp->intreg.gci_intreg2[i][j] != NULL) {
        gtk_object_destroy(GTK_OBJECT
                           (disp->intreg.gci_intreg2[i][j]));
        disp->intreg.gci_intreg2[i][j] = NULL;
    }
}



void 
show_all_intregs2(Display * disp
                  )
/* ----------------------------------------------------------------------------
 * Shows 2nd interrogation areas
 */
{
    int i, j;
    int nx, ny;

    if (disp != NULL
        && disp->intreg.exist_int) {

        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
    
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                assert (disp->intreg.gci_intreg2[i][j] != NULL);
                gnome_canvas_item_show(GNOME_CANVAS_ITEM
                                       (disp->intreg.gci_intreg2[i][j]));
            }
        }
    }
}



void 
hide_all_intregs2(Display * disp
                  )
/* ----------------------------------------------------------------------------
 * Hides 2nd interrogation areas
 */
{
    int i, j;
    int nx, ny;

    if (disp != NULL
        && disp->intreg.exist_int) {
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
    
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                assert (disp->intreg.gci_intreg2[i][j] != NULL);
                gnome_canvas_item_hide(GNOME_CANVAS_ITEM
                                       (disp->intreg.gci_intreg2[i][j]));
            }
        }
    }
}



void 
destroy_all_intregs2(Display * disp
                     )
/* ----------------------------------------------------------------------------
 * Destroys 2nd interrogation areas
 */
{
    int i, j;
    int nx, ny;
    
    if (disp != NULL
        && disp->intreg.exist_int) {    
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
        if (verbose) 
        g_message("destroy_all_intregs2: destroying %d I.A.'s",  nx * ny);
    
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                destroy_intreg2(disp, i, j);
            }
        }
    }
}



gfloat
dxdy_max(GpivPivData piv_data
         )
/*-----------------------------------------------------------------------------
 * Calculates maximum particle displacement from dx and dy of a piv data-set
 */
{
    gint i, j;
    gfloat dl_abs = 0.0, dl_max = 0.0;
    
/*     assert(piv_data.nx != 0); */
/*     assert(piv_data.ny != 0); */
    

    for (i = 0; i < piv_data.ny; i++) {
        for (j = 0; j < piv_data.nx; j++) {
            if (piv_data.peak_no[i][j] >= 1) {
                dl_abs = sqrt(piv_data.dx[i][j] * piv_data.dx[i][j] + 
                              piv_data.dy[i][j] * piv_data.dy[i][j]);
                if (dl_abs > dl_max) {
                    dl_max = dl_abs;
                }
            }
        }
    }

    return dl_max;
}



gfloat
dxdy_min(GpivPivData piv_data
         )
/*-----------------------------------------------------------------------------
 * Calculates maximum particle displacement from dx and dy of a piv data-set
 */
{
    gint i, j;
    gfloat dl_abs = 0.0, dl_min = 10.0e+9;
    
/*     assert(piv_data.nx != 0); */
/*     assert(piv_data.ny != 0); */
    

    for (i = 0; i < piv_data.ny; i++) {
        for (j = 0; j < piv_data.nx; j++) {
            if (piv_data.peak_no[i][j] >= 1) {
                dl_abs = sqrt(piv_data.dx[i][j] * piv_data.dx[i][j] + 
                              piv_data.dy[i][j] * piv_data.dy[i][j]);
                if (dl_abs < dl_min) {
                    dl_min = dl_abs;
                }
            }
        }
    }

    return dl_min;
}



#ifdef CANVAS_AA
/*
 * BUGFIX repair color representation for canvas_aa
 */

guint32
create_vector_color(gint peak_no, 
                    gfloat snr, 
                    gfloat dl
                    )
/* ----------------------------------------------------------------------------
 * Create vector color for in canvas
 */
{
    guint32 color = 0;
    GdkColor *color_val = NULL;
    gint shift_factor;

    color_val = (GdkColor *)g_malloc(sizeof(GdkColor));

    if (gpiv_par.vector_color == SHOW_PEAKNR) {
        if (peak_no == -1) {
             color_val->red = BYTEVAL;
             color_val->green = 0;
             color_val->blue = 0;

        } else if (peak_no == 0) {
             color_val->red = 0;
             color_val->green = 0;
             color_val->blue = BYTEVAL;

        } else if (peak_no == 1) {
             color_val->red = 0;
             color_val->green = BYTEVAL;
             color_val->blue = 0;

        } else if (peak_no == 2) {
             color_val->red = 0;
             color_val->green = BYTEVAL;
             color_val->blue = BYTEVAL;

	 } else  {
/* if (peak_no[i][j] == 3) */
             color_val->red = BYTEVAL / 2;
             color_val->green = BYTEVAL / 2;
             color_val->blue = BYTEVAL / 2;
	 }


        color = GNOME_CANVAS_COLOR(color_val->red, 
                                   color_val->green, 
                                   color_val->blue);

    } else if (gpiv_par.vector_color == SHOW_SNR) {
        if (snr >= piv_valid_par.residu_max) {
/*             color = "red"; */
             color_val->red = BYTEVAL;
             color_val->green = 0;
             color_val->blue = 0;
        } else {
/*             color = "green";     */
             color_val->red = 0;
             color_val->green = BYTEVAL;
             color_val->blue = 0;
        }

        color = GNOME_CANVAS_COLOR(color_val->red, 
                                   color_val->green, 
                                   color_val->blue);

    } else if (gpiv_par.vector_color == SHOW_MAGNITUDE_GRAY) {
        if (peak_no >= 1) {
            color_val->red = (gint) (BYTEVAL * (dl - gpiv_var.dl_min) / 
                                     (gpiv_var.dl_max - gpiv_var.dl_min));
            color_val->green = (gint) (BYTEVAL * (dl - gpiv_var.dl_min) / 
                                       (gpiv_var.dl_max - gpiv_var.dl_min));
            color_val->blue = (gint) (BYTEVAL * (dl - gpiv_var.dl_min) / 
                                      (gpiv_var.dl_max - gpiv_var.dl_min));
        } else {
            color_val->red = 0;
            color_val->green = 0;
            color_val->blue = 0;
        }

        color = GNOME_CANVAS_COLOR(color_val->red, 
                                   color_val->green, 
                                   color_val->blue);

    } else if (gpiv_par.vector_color == SHOW_MAGNITUDE) {
        if (peak_no >= 1) {
            shift_factor = (gint) (28.0 * (dl - gpiv_var.dl_min) / 
                                   (gpiv_var.dl_max - gpiv_var.dl_min));
            color = (( 0xFFFF << shift_factor) | 0xFF);


        } else {
            color = GNOME_CANVAS_COLOR(0, 0, 0);
        }
    }
    
    g_free(color_val);
    return color;
}



guint
create_scalar_color(gint flag,
                    gfloat sc_min, 
                    gfloat sc_max, 
                    gfloat sc, 
                    gfloat scale_factor
                    )
/* ----------------------------------------------------------------------------
 * Create scalar color for in canvas
 */
{
    guint color = 0;
    GdkColor *color_val = NULL;

    color_val = (GdkColor *)g_malloc(sizeof(GdkColor));
    if (flag != -1) {
        if (sc_min < 0) {
            if (sc <0) {
                color_val->red = (gint) (-sc * BYTEVAL * scale_factor);
                color_val->green = 0;
                color_val->blue =  0;
            } else {
                color_val->red =  0;
                color_val->green = 0;
                color_val->blue = (int) (sc * BYTEVAL * scale_factor);
            }
        } else {
            color_val->red = 0;
            color_val->green = 0;
            color_val->blue =  (int) (sc * BYTEVAL * scale_factor);
        }
    } else {
                color_val->red = 0;
                color_val->green = BYTEVAL / 2;
                color_val->blue = 0;
    }

    color = GNOME_CANVAS_COLOR(color_val->red, 
                               color_val->green, 
                               color_val->blue); 
    g_free(color_val);
    return color;
}


#else /* CANVAS_AA */

guint32
create_vector_color(gint peak_no, 
                    gfloat snr, 
                    gfloat dl
                    )
/* ----------------------------------------------------------------------------
 * Create vector color for in canvas
 */
{
    guint32 color_val = 0;
    gint shift_factor;

    if (gpiv_par.vector_color == SHOW_PEAKNR) {
	 if (peak_no == -1) {
/* 	      color = "red"; */
        color_val =  (gint) (BYTEVAL);
        color_val =  (color_val << BITSHIFT_RED);
	 } else if (peak_no == 0) {
/* 	      color = "lightblue"; */
        color_val =  (gint) (BYTEVAL);
        color_val =  (color_val << BITSHIFT_BLUE);
	 } else if (peak_no == 1) {
/* 	      color = "green"; */
        color_val =  (gint) (BYTEVAL);
        color_val =  (color_val << BITSHIFT_GREEN);
	 } else if (peak_no == 2) {
/* 	      color = "yellow"; */
        color_val =  (gint) (BYTEVAL);
        color_val =  (color_val << BITSHIFT_GREEN) 
            + (color_val << BITSHIFT_BLUE);
	 } else  {
/* if (peak_no == 3) */
/* 	      color = "gray";     */
        color_val =  (gint) (127);
        color_val =  (color_val << BITSHIFT_RED)
            + (color_val << BITSHIFT_GREEN) 
            + (color_val << BITSHIFT_BLUE);
	 }


    } else if (gpiv_par.vector_color == SHOW_SNR) {
        if (snr >= piv_valid_par.residu_max) {
/*             color = "red"; */
            color_val =  (gint) (BYTEVAL);
            color_val =  (color_val << BITSHIFT_RED);
        } else {
/*             color = "green";     */
            color_val =  (gint) (BYTEVAL);
            color_val =  (color_val << BITSHIFT_GREEN);
        }


    } else if (gpiv_par.vector_color == SHOW_MAGNITUDE_GRAY) {
        if (peak_no >= 1) {
            color_val =  (gint) (BYTEVAL * (dl - gpiv_var.dl_min) / 
                                 (gpiv_var.dl_max - gpiv_var.dl_min));
            color_val =  (color_val << BITSHIFT_RED) + 
                (color_val << BITSHIFT_GREEN) + (color_val << BITSHIFT_BLUE);
        } else {
            color_val =  0;
        }
    } else if (gpiv_par.vector_color == SHOW_MAGNITUDE) {
        if (peak_no >= 1) {
            shift_factor = (gint) (28.0 * (dl - gpiv_var.dl_min) / 
                                   (gpiv_var.dl_max - gpiv_var.dl_min));
            color_val = ( 0xFFFF << shift_factor);
        } else {
            color_val =  0;
        }
    }


    return color_val;
}

guint
create_scalar_color(gint flag,
                    gfloat sc_min, 
                    gfloat sc_max, 
                    gfloat sc, 
                    gfloat scale_factor
                    )
/* ----------------------------------------------------------------------------
 * Create scalar color for in canvas
 */
{
    guint color_val = 0;

    if (flag != -1) {
        if (sc_min < 0) {
            if (sc <0) {
                color_val =  (int) (-sc * BYTEVAL * scale_factor);
                color_val =  (color_val << BITSHIFT_RED);
            } else {
                color_val =  (int) (sc * BYTEVAL * scale_factor);
                color_val =  (color_val << BITSHIFT_BLUE);
            }
        } else {
            color_val =  (int) (sc * BYTEVAL * scale_factor);
            color_val =  (color_val << BITSHIFT_BLUE);
        }
    } else {
        color_val = 128 << BITSHIFT_GREEN;
    }


    return color_val;
}


#endif /* CANVAS_AA */


void 
create_vector(GpivData * gpd,
              gint i, 
              gint j
              )
/* ----------------------------------------------------------------------------
 * Displays a single PIV vector on a Gnome canvas
 */
{
    GnomeCanvasPoints *points;
    Display *disp = display_act;

    float **point_x = gpd->piv_data.point_x;
    float **point_y = gpd->piv_data.point_y;
    float **dx = gpd->piv_data.dx, **dy = gpd->piv_data.dy;
    int **peak_no =  gpd->piv_data.peak_no;
    float **snr = gpd->piv_data.snr;
    float dl = sqrt(dx[i][j] * dx[i][j] + dy[i][j] * dy[i][j]);
    guint32 color_val = 0;
    points = gnome_canvas_points_new(2);

/*
 * Fill out the points
 */

    points->coords[0] = point_x[i][j];
    points->coords[1] = point_y[i][j];
    points->coords[2] = point_x[i][j] + dx[i][j] * gpiv_par.vector_scale;
    points->coords[3] = point_y[i][j] + dy[i][j] * gpiv_par.vector_scale;


    color_val = create_vector_color(peak_no[i][j], snr[i][j], dl);

    if (gpd->gci_vector[i][j] != NULL) {
/*         g_warning("create_vector:: gci_vector[%d][%d] != NULL ==> destroying", */
/*                     i, j); */
        destroy_vector(gpd, i, j);
    }

    gpd->gci_vector[i][j] =
        gnome_canvas_item_new(gnome_canvas_root
                              (GNOME_CANVAS(disp->canvas)),
                              gnome_canvas_line_get_type(),
                              "points", points,
                              "fill_color_rgba", color_val,
                              "width_units", (double) THICKNESS,
                              "last_arrowhead", TRUE,
                              "arrow_shape_a", (double) ARROW_LENGTH * 
                              ARROW_FACT * dl * gpiv_par.vector_scale + 
                              ARROW_ADD,
                              "arrow_shape_b", (double) ARROW_EDGE *
                              ARROW_FACT * dl * gpiv_par.vector_scale + 
                              ARROW_ADD,
                              "arrow_shape_c", (double) ARROW_WIDTH *
                              ARROW_FACT * dl * gpiv_par.vector_scale + 
                              ARROW_ADD,
                              NULL);

    gnome_canvas_points_free(points);
    
}



void 
update_vector(GpivData * gpd,
              gint i, 
              gint j
              )
/* ----------------------------------------------------------------------------
 * Updates a single PIV vector on a Gnome canvas
 */
{
    GnomeCanvasPoints *points;

    float **point_x = gpd->piv_data.point_x, **point_y =
	gpd->piv_data.point_y;
    float **dx = gpd->piv_data.dx, **dy = gpd->piv_data.dy;
    float **snr = gpd->piv_data.snr;
    int **peak_no = gpd->piv_data.peak_no;
    float dl = sqrt(dx[i][j] * dx[i][j] + dy[i][j] * dy[i][j]);
    guint32 color_val = 0;
    points = gnome_canvas_points_new(2);

/*
 * Fill out the points
 */
    points->coords[0] = point_x[i][j];
    points->coords[1] = point_y[i][j];
    points->coords[2] = point_x[i][j] + dx[i][j] * gpiv_par.vector_scale;
    points->coords[3] = point_y[i][j] + dy[i][j] * gpiv_par.vector_scale;

    color_val = create_vector_color(peak_no[i][j], snr[i][j], dl);

    if (gpd->gci_vector[i][j] != NULL) {
        gnome_canvas_item_set(GNOME_CANVAS_ITEM(gpd->gci_vector[i][j]),
                              "points", points,
                              "fill_color_rgba", color_val,
                              "width_units", (double) THICKNESS,
                              "last_arrowhead", TRUE,
                              "arrow_shape_a", (double) ARROW_LENGTH *
                              ARROW_FACT * dl * gpiv_par.vector_scale
                              + ARROW_ADD,
                              "arrow_shape_b", (double) ARROW_EDGE *
                              ARROW_FACT * dl * gpiv_par.vector_scale
                              + ARROW_ADD,
                              "arrow_shape_c", (double) ARROW_WIDTH * 
                              ARROW_FACT * dl * gpiv_par.vector_scale
                              + ARROW_ADD,
                              NULL);
    }

    gnome_canvas_points_free(points);
}



void 
destroy_vector(GpivData * gpd,
               gint i, 
               gint j
               )
/* ----------------------------------------------------------------------------
 * Detroys a single PIV vector on a Gnome canvas
 */
{
     if (gpd->gci_vector[i][j] != NULL) {
	  gtk_object_destroy(GTK_OBJECT (gpd->gci_vector[i][j]));
	  gpd->gci_vector[i][j] = NULL;
     }

}



void 
create_all_vectors(GpivData * gpd
                    )
/* ---------------------------------------------------------------------------
 * Displays all PIV vectors on a Gnome canvas
 */
{
    int i, j;
    gint nx = gpd->piv_data.nx, ny = gpd->piv_data.ny;
    
    if (gpd->exist_vec) {
        destroy_all_vectors(gpd);
    }
    gpiv_var.dl_max = dxdy_max(gpd->piv_data);
    gpiv_var.dl_min = dxdy_min(gpd->piv_data);
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    create_vector(gpd, i, j);
            
	}
    }
    gpd->exist_vec = TRUE;
}



void 
show_all_vectors(GpivData * gpd
                 )
/* ----------------------------------------------------------------------------
 * Shows all PIV vectors on a Gnome canvas
 */
{
    int i, j;
    gint nx = gpd->piv_data.nx, ny = gpd->piv_data.ny;

    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
            if (gpd->gci_vector[i][j] != NULL) {
                gnome_canvas_item_show(GNOME_CANVAS_ITEM
                                       (gpd->gci_vector[i][j]));
            }
	}
    }
}



void 
hide_all_vectors(GpivData * gpd
                 )
/* ----------------------------------------------------------------------------
 * Hides all PIV vectors on a Gnome canvas
 */
{
    int i, j;
    gint nx = gpd->piv_data.nx, ny = gpd->piv_data.ny;

    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
            if (gpd->gci_vector[i][j] != NULL) {
                gnome_canvas_item_hide(GNOME_CANVAS_ITEM
                                       (gpd->gci_vector[i][j]));
            }
	}
    }
}



void 
update_all_vectors(GpivData * gpd
                   )
/* ----------------------------------------------------------------------------
 * Scales PIV vectors for Gnome canvas
 */
{
    int i, j;
    int nx = gpd->piv_data.nx, ny = gpd->piv_data.ny;

    gpiv_var.dl_max = dxdy_max(gpd->piv_data);
    gpiv_var.dl_min = dxdy_min(gpd->piv_data);
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    update_vector(gpd, i, j);
	}
    }

}



void 
destroy_all_vectors(GpivData * gpd
                    )
/* ----------------------------------------------------------------------------
 * Destroys all PIV vectors on a Gnome canvas
 */
{
    int i, j;
    gint nx = gpd->piv_data.nx, ny = gpd->piv_data.ny;

    if (gpd->exist_vec) {
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                destroy_vector(gpd, i, j);
            }
        }
/*     } else { */
/*         g_warning("destroy_all_vectors: exist_vec = FALSE"); */
    }
    gpd->exist_vec = FALSE;

}



static void
destroy_filledrect(GnomeCanvasItem * filled_rect
                   )
/* ----------------------------------------------------------------------------
 * Destroys a single filled rectangular canvas item
 */
{
    if (filled_rect != NULL) {
        gtk_object_destroy(GTK_OBJECT(filled_rect));
        filled_rect = NULL;
    }
}



static void
show_filledrect(GnomeCanvasItem * filled_rect
                )
/* ----------------------------------------------------------------------------
 * Shows a single filled rectangular canvas item
 */
{
    if (filled_rect != NULL) {
        gnome_canvas_item_show
            (GNOME_CANVAS_ITEM(filled_rect));
    }
}



static void
hide_filledrect(GnomeCanvasItem * filled_rect
                )
/* ----------------------------------------------------------------------------
 * Hides a single filled rectangular canvas item
 */
{
    if (filled_rect != NULL) {
        gnome_canvas_item_hide
            (GNOME_CANVAS_ITEM(filled_rect));
    }
}



static void 
create_vor(Display * disp,
           gint i, 
           gint j,
           guint col_val
           )
/* ----------------------------------------------------------------------------
 * Creates vorticity gnome canvas item by coloring the interrogation area
 */
{
    float **x, **y;
    int start_x, start_y, end_x, end_y;
 
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(5);


/*
 * Using centre points of interr regs
 */
    x = disp->gpd.vor_data.point_x;
    y = disp->gpd.vor_data.point_y;
    start_x = (int) x[i][j] - piv_eval_par.int_size_1 / 2;
    start_y = (int) y[i][j] - piv_eval_par.int_size_1 / 2;
    end_x = (int) x[i][j] + piv_eval_par.int_size_1 / 2;
    end_y = (int) y[i][j] + piv_eval_par.int_size_1 / 2;
    
    disp->gpd.gci_scalar_vor[i][j] = 
        gnome_canvas_item_new(gnome_canvas_root
                              (GNOME_CANVAS(display_act->canvas)),
                              gnome_canvas_rect_get_type(),
                              "x1", (double) start_x,
                              "y1", (double) start_y,
                              "x2", (double) end_x,
                              "y2", (double) end_y,
                              "fill_color_rgba", col_val,
                              NULL);

    gnome_canvas_points_free(points);
}



static void 
create_sstrain(Display * disp,
               gint i, 
               gint j,
               guint col_val
               )
/* ----------------------------------------------------------------------------
 * Creates shear strain gnome canvas item by coloring the interrogation area
 */
{
    float **x, **y;
    int start_x, start_y, end_x, end_y;
 
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(5);


/*
 * Using centre points of interr regs
 */
    x = disp->gpd.sstrain_data.point_x;
    y = disp->gpd.sstrain_data.point_y;
    start_x = (int) x[i][j] - piv_eval_par.int_size_1 / 2;
    start_y = (int) y[i][j] - piv_eval_par.int_size_1 / 2;
    end_x = (int) x[i][j] + piv_eval_par.int_size_1 / 2;
    end_y = (int) y[i][j] + piv_eval_par.int_size_1 / 2;
    
    disp->gpd.gci_scalar_sstrain[i][j] = 
        gnome_canvas_item_new(gnome_canvas_root
                              (GNOME_CANVAS(display_act->canvas)),
                              gnome_canvas_rect_get_type(),
                              "x1", (double) start_x,
                              "y1", (double) start_y,
                              "x2", (double) end_x,
                              "y2", (double) end_y,
                              "fill_color_rgba", col_val,
                              NULL);
    
    gnome_canvas_points_free(points);
}



static void 
create_nstrain(Display * disp,
               gint i, 
               gint j,
               guint col_val
               )
/* ----------------------------------------------------------------------------
 * Creates normal strain gnome canvas item by coloring the interrogation area
 */
{
    float **x, **y;
    int start_x, start_y, end_x, end_y;
 
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(5);


/*
 * Using centre points of interr regs
 */

    x = disp->gpd.nstrain_data.point_x;
    y = disp->gpd.nstrain_data.point_y;
    start_x = (int) x[i][j] - piv_eval_par.int_size_1 / 2;
    start_y = (int) y[i][j] - piv_eval_par.int_size_1 / 2;
    end_x = (int) x[i][j] + piv_eval_par.int_size_1 / 2;
    end_y = (int) y[i][j] + piv_eval_par.int_size_1 / 2;
    
    disp->gpd.gci_scalar_nstrain[i][j] = 
        gnome_canvas_item_new(gnome_canvas_root
                              (GNOME_CANVAS(display_act->canvas)),
                              gnome_canvas_rect_get_type(),
                              "x1", (double) start_x,
                              "y1", (double) start_y,
                              "x2", (double) end_x,
                              "y2", (double) end_y,
                              "fill_color_rgba", col_val,
                              NULL);

    gnome_canvas_points_free(points);    
}



void 
create_all_scalars(/* GpivScalarData scalar_data,  */
                   Display * disp,
                   gint type
                   )
/* ----------------------------------------------------------------------------
 * Displays all scalar gnome canvas items
 */
{
    int i, j;
    int nx = 0, ny = 0, **flag = NULL;

    float **sc = NULL;
    float sc_min = 10000.0, sc_max = -10000.0;
    guint col_val = 0;
    GpivScalarData * scalar_data;    
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");
    GtkWidget * view_piv_display1 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display1");
    float scale_factor = 0.0;

     if (type == GPIV_VORTICITY) {
       scalar_data = &disp->gpd.vor_data;
        sc = scalar_data->scalar;
        nx = disp->gpd.vor_data.nx;
        ny = disp->gpd.vor_data.ny;
        flag = disp->gpd.vor_data.flag;
/*         scalar_colval(disp); */

    } else if (type == GPIV_S_STRAIN) {
        scalar_data = &disp->gpd.sstrain_data;
        sc = scalar_data->scalar;
        nx = disp->gpd.sstrain_data.nx;
        ny = disp->gpd.sstrain_data.ny;
        flag = disp->gpd.sstrain_data.flag;

    } else if (type == GPIV_N_STRAIN) {
        scalar_data = &disp->gpd.nstrain_data;
        sc = scalar_data->scalar;
        nx = disp->gpd.nstrain_data.nx;
        ny = disp->gpd.nstrain_data.ny;
        flag = disp->gpd.nstrain_data.flag;
    } else {
        g_warning(_("create_all_scalars: unexisting type"));
    }

/*
 * normalizing data between 0 and 1 to define color value col_val
 */
/* scalar_colval(Display * disp) { */
    for (i = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            if (flag[i][j] != -1) {
                if (sc[i][j] < sc_min) sc_min = sc[i][j];
                if (sc[i][j] > sc_max) sc_max =  sc[i][j];
            }
        }
    }

    if (sc_min < 0) {
        if (-sc_min >= sc_max) {
            scale_factor = 1.0 / -sc_min;
        } else {
            scale_factor = 1.0 / sc_max;
        }
    } else {
        scale_factor = 1.0 / sc_max;
    }


     for (i = 0; i < ny; i++) {
	  for (j = 0; j < nx; j++) {
              col_val = create_scalar_color(flag[i][j], sc_min, sc_max, 
                                            sc[i][j], scale_factor);
              if (type == GPIV_VORTICITY) {
                  create_vor(disp, i, j, col_val);
              } else if (type == GPIV_S_STRAIN) {
                  create_sstrain(disp, i, j, col_val);
              } else if (type == GPIV_N_STRAIN) {
                  create_nstrain(disp, i, j, col_val);
              } else {
                  g_warning(_("create_all_scalars: unexisting type"));
              }
          }
     }
     
     if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
         for (i = 0; i < ny; i++) {
             for (j = 0; j < nx; j++) {
                 gnome_canvas_item_raise_to_top
                     (disp->intreg.gci_intreg1[i][j]);
                 gnome_canvas_item_raise_to_top
                     (display_act->intreg.gci_intreg2[i][j]);
             }
         }
     }

     if (GTK_CHECK_MENU_ITEM(view_piv_display1)->active) {
         if (disp->gpd.exist_piv && disp->display_piv) {
             for (i = 0; i < ny; i++) {
                 for (j = 0; j < nx; j++) {
                     gnome_canvas_item_raise_to_top
                         (disp->gpd.gci_vector[i][j]);
                 }
             }
         }
     }

}



void 
show_all_scalars(Display * disp,
                 gint type
                 )
/* ----------------------------------------------------------------------------
 * Shows scalar gnome canvas items
 */
{
    int i = 0, j = 0;
    int nx = 0, ny = 0;

    if (type == GPIV_VORTICITY) {
        nx = disp->gpd.vor_data.nx;
        ny = disp->gpd.vor_data.ny;
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
/*               if (i ==0 && j==0)  */
                show_filledrect(display_act->gpd.gci_scalar_vor[i][j]);
            }
        }
        display_act->display_vor = TRUE;
    }

    if (type == GPIV_S_STRAIN) {
        nx = disp->gpd.sstrain_data.nx;
        ny = disp->gpd.sstrain_data.ny;
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                show_filledrect(display_act->gpd.gci_scalar_sstrain[i][j]);
            }
        }
        display_act->display_sstrain = TRUE;
    }

    if (type == GPIV_N_STRAIN) {
        nx = disp->gpd.nstrain_data.nx;
        ny = disp->gpd.nstrain_data.ny;
        for (i = 0; i < ny; i++) {
            for (j = 0; j < nx; j++) {
                show_filledrect(display_act->gpd.gci_scalar_nstrain[i][j]);
            }
        }
        display_act->display_nstrain = TRUE;
    }

}



void 
hide_all_scalars(/* GpivScalarData * scalar_data,  */
                 Display * disp,
                 gint type
                 )
/* ----------------------------------------------------------------------------
 * Hides all scalar gnome canvas items
 */
{
    int i, j;
/*     int nx = disp->gpd.scalar_data.nx, ny = disp->gpd.scalar_data.ny; */

   if (type == GPIV_VORTICITY) {
        for (i = 0; i < disp->gpd.vor_data.ny; i++) {
            for (j = 0; j < disp->gpd.vor_data.nx; j++) {
/*               if (i ==0 && j==0)  */
                hide_filledrect(disp->gpd.gci_scalar_vor[i][j]);
            }
        }
        disp->display_vor = FALSE;
    }

    if (type == GPIV_S_STRAIN) {
        for (i = 0; i < disp->gpd.sstrain_data.ny; i++) {
            for (j = 0; j < disp->gpd.sstrain_data.nx; j++) {
                hide_filledrect(disp->gpd.gci_scalar_sstrain[i][j]);
           }
        }
        disp->display_sstrain = FALSE;
    }

    if (type == GPIV_N_STRAIN) {
        for (i = 0; i < disp->gpd.nstrain_data.ny; i++) {
            for (j = 0; j < disp->gpd.nstrain_data.nx; j++) {
                hide_filledrect(disp->gpd.gci_scalar_nstrain[i][j]);
            }
        }
        disp->display_nstrain = FALSE;
      }


}



void 
destroy_all_scalars(Display * disp,
                    gint type
                    )
/* ----------------------------------------------------------------------------
 * Destroys scalar canvas items
 */
{
    int i, j;

    if (type == GPIV_VORTICITY) {
        for (i = 0; i < disp->gpd.vor_data.ny; i++) {
            for (j = 0; j < disp->gpd.vor_data.nx; j++) {
/*               if (i ==0 && j==0)  */
                destroy_filledrect(disp->gpd.gci_scalar_vor[i][j]);
            }
        }

    } else if (type == GPIV_S_STRAIN) {
        for (i = 0; i < disp->gpd.sstrain_data.ny; i++) {
            for (j = 0; j < disp->gpd.sstrain_data.nx; j++) {
                destroy_filledrect(disp->gpd.gci_scalar_sstrain[i][j]);
            }
        }

    } else if (type == GPIV_N_STRAIN) {
        for (i = 0; i < disp->gpd.nstrain_data.ny; i++) {
            for (j = 0; j < disp->gpd.nstrain_data.nx; j++) {
                destroy_filledrect(disp->gpd.gci_scalar_nstrain[i][j]);
            }
        }
    } else {
        g_warning(_("destroy_all_scalars: unexisting type"));
    }
    
    
}



/*
 * Callback display functions
 */


gint
on_button_display_origin_press_event(GtkWidget *widget, 
                                    GdkEventButton *event, 
                                    gpointer data
                                     )
/*-----------------------------------------------------------------------------
 */
{
  gtk_menu_popup(GTK_MENU(display_act->display_menu), NULL, NULL, NULL, NULL,
		 event->button, event->time);
  return FALSE;
}



void 
on_display_set_focus(GtkWidget * widget, 
                     gpointer data
                     )
/*-----------------------------------------------------------------------------
 */
{
    Display *disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint i = 0, row_selected = 0;
    gchar * text, labelno[GPIV_MAX_CHARS];
    gtk_clist_set_selection_mode(GTK_CLIST (gpiv->clist_buf),
                                 GTK_SELECTION_SINGLE);

    g_snprintf(labelno, GPIV_MAX_CHARS,"%d", disp->id);


    for (i = 0; i < nbufs; i++) {
        gtk_clist_get_text(GTK_CLIST(gpiv->clist_buf), i, 0, &text);

        if (strcmp(labelno, text) == 0) {
            row_selected = i;
        }
    }


    gtk_clist_select_row(GTK_CLIST(gpiv->clist_buf), row_selected, 0);

/* g_warning("on_display_set_focus:: disp->id=%d row_selected=%d",  */
/* disp->id, row_selected); */

    gtk_clist_set_selection_mode(GTK_CLIST (gpiv->clist_buf),
                                 GTK_SELECTION_EXTENDED);
}



void
delete_display(GtkWidget *widget,
               GdkEvent  *event,
               gpointer   data
               )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    Display *disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    gint row, ibuf;

    ibuf = disp->id;
    close_buffer(gpiv, disp, &ibuf);

/*
 * BUGFIX: Cleaning up clist
 */
/*     for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row;  */
/*          row++) { */
/*         display_act = gtk_clist_get_row_data */
/*             (GTK_CLIST(gpiv->clist_buf), gpiv->first_selected_row);  */
/*         ibuf = display_act->id; */
/*         if (display[ibuf] == NULL) { */
/*             gtk_clist_remove(GTK_CLIST(gpiv->clist_buf),  */
/*                              gpiv->first_selected_row); */
/*             nbufs--; */
/*         } else { */
/*             gpiv->first_selected_row++; */
/*         } */
/*     } */

}



void 
on_scrolledwindow_display_adj_changed(GtkAdjustment *adj, 
                                      gpointer data
                                      )
/*-----------------------------------------------------------------------------
 * Adjusts x_offset, x_lower and x_upper when scrolling
 * Ruler adjustments for zooming and scaling of data
 *
 * Disp->stretch_window_tmp is a flag that is set when the window is at 
 * full-scale by invoking stretch window or by picking a smaller zoom-scale
 * than the acual one.
 */
{
    Display * disp = gtk_object_get_data(GTK_OBJECT(adj), "disp");
    gfloat x_start = 0, x_end = 0, y_start = 0, y_end = 0;
    enum variable_type {
        X_ADJ = 0,
        Y_ADJ = 1
    } var_type;

    var_type = atoi(gtk_object_get_data(GTK_OBJECT(adj), "var_type"));

    if (disp->stretch_window_tmp) {
       if (var_type == X_ADJ) {
            x_offset = (gint) adj->value;
            x_lower = (gint) adj->lower;
            x_upper = (gint) adj->upper;


            if (disp->gpd.exist_piv && display_act->gpd.scaled_piv) {
                x_start = (gfloat) x_offset * image_par.s_scale + 
                    image_par.z_off_x *10e2;
                x_end = (gfloat) gpiv_par.img_width * image_par.s_scale - 1.0
                    + image_par.z_off_x *10e2;
                
            } else {
                
                x_start = 0.0;
                x_end = (gfloat) (gpiv_par.img_width - 1);
            }

            gtk_ruler_set_range(GTK_RULER(disp->hruler),
                                x_start,
                                x_end,
                                x_start,
                                x_end);

            gtk_widget_set_size_request (disp->hruler,
                                         (gint) (disp->zoom_factor * 
                                                 gpiv_par.img_width),
                                         RULER_WIDTH);
            gtk_widget_set_uposition (disp->hruler, RULER_WIDTH, 0);
            }

        if (var_type == Y_ADJ) {
            y_offset = (gint) adj->value;
            y_lower = (gint) adj->lower;
            y_upper = (gint) adj->upper;

            if (disp->gpd.exist_piv && display_act->gpd.scaled_piv) {
                y_start = (gfloat) y_offset  * image_par.s_scale + 
                    image_par.z_off_y *10e2;

                y_end = (gfloat) gpiv_par.img_height * image_par.s_scale - 1.0
                    + image_par.z_off_y *10e2;
                
            } else {
                
                y_start = 0.0;
                y_end = (gfloat) (gpiv_par.img_height - 1);
            }

            gtk_ruler_set_range(GTK_RULER(disp->vruler),
                                y_start,
                                y_end,
                                y_start,
                                y_end);
            gtk_widget_set_size_request (disp->vruler,
                                         RULER_WIDTH,
                                         (gint) (disp->zoom_factor * 
                                                 gpiv_par.img_height));
            gtk_widget_set_uposition (disp->vruler, 0, RULER_WIDTH);

        }


    } else {
        if (var_type == X_ADJ) {
            x_offset = (gint) adj->value;
            x_lower = (gint) adj->lower;
            x_upper = (gint) adj->upper;

            if (disp->gpd.exist_piv && display_act->gpd.scaled_piv) {
                x_start = (gfloat) x_offset / (gfloat) disp->zoom_factor
                    * image_par.s_scale *10e2 + image_par.z_off_x *10e2;
                x_end =  (gfloat) x_offset / (gfloat) disp->zoom_factor
                    * image_par.s_scale *10e2 + image_par.z_off_x *10e2 
                    + (gfloat) (gpiv_par.img_width - 1.0) 
                    * image_par.s_scale *10e2 
                    * (gfloat) disp->zoom_factor_old 
                    / (gfloat) disp->zoom_factor;

            } else {

                x_start = (gfloat) (x_offset) / (gfloat) disp->zoom_factor;
                x_end = (gfloat) x_offset  
                    / (gfloat) disp->zoom_factor  
                    + (gfloat) (gpiv_par.img_width - 1) 
                    * (gfloat) disp->zoom_factor_old 
                    / (gfloat) disp->zoom_factor;
            }

           gtk_ruler_set_range(GTK_RULER(disp->hruler),
                                x_start,
                                x_end,
                                x_start,
                                x_end);
        }

        if (var_type == Y_ADJ) {
            y_offset = (gint) adj->value;
            y_lower = (gint) adj->lower;
            y_upper = (gint) adj->upper;
            
            if (disp->gpd.exist_piv && display_act->gpd.scaled_piv) {
                y_start = (gfloat) y_offset / (gfloat) disp->zoom_factor
                    * image_par.s_scale *10e2 + image_par.z_off_y *10e2;
                y_end =  (gfloat) y_offset / (gfloat) disp->zoom_factor
                    * image_par.s_scale *10e2 + image_par.z_off_y *10e2  
                    + (gfloat) (gpiv_par.img_height - 1.0) 
                    * image_par.s_scale *10e2 
                    * (gfloat) disp->zoom_factor_old 
                    / (gfloat) disp->zoom_factor;

            } else {

                y_start = (gfloat) (y_offset) / (gfloat) disp->zoom_factor;
                y_end = (gfloat) y_offset  
                    / (gfloat) disp->zoom_factor  
                    + (gfloat) (gpiv_par.img_height - 1) 
                    * (gfloat) disp->zoom_factor_old 
                    / (gfloat) disp->zoom_factor;
            }

            gtk_ruler_set_range(GTK_RULER(disp->vruler),
                                y_start,
                                y_end,
                                y_start,
                                y_end);
            
        }
    }

    g_message("display_adj_changed:: x_offset = %f x_lower = %f x_upper = %f",
              x_offset, x_lower, x_upper);

    g_message("display_adj_changed:: y_offset = %f y_lower = %f y_upper = %f",
              y_offset, y_lower, y_upper);
}




/*
 * Callback functions for pop-up menus
 */

/* gint */
/* on_displaypopup_handler(GtkWidget *widget,  */
/*                         GdkEvent *event */
/*                         ) */
/* /*----------------------------------------------------------------------------- */
/*  */
/* { */
/*     GtkMenu *menu; */
/*     GdkEventButton *event_button; */

/*     g_return_val_if_fail (widget != NULL, FALSE); */
/*     g_return_val_if_fail (GTK_IS_MENU (widget), FALSE); */
/*     g_return_val_if_fail (event != NULL, FALSE); */

/* /* The "widget" is the menu that was supplied when  */
/*  * gtk_signal_connect_object was called. */
/*  */
/*     g_warning("on_displaypopup_handler:: event->type == %d", event->type); */
/*     menu = GTK_MENU (widget); */

/*     if (event->type == GDK_BUTTON_PRESS) */
/*         { */
/*             event_button = (GdkEventButton *) event; */
/*             if (event_button->button == 3) */
/*                 { */
/*                     gtk_menu_popup (menu, NULL, NULL, NULL, NULL,  */
/*                                     event_button->button, event_button->time); */
/*                     return TRUE; */
/*                 } */
/*         } */
    
/*     return FALSE; */
/* } */



void 
select_zoomscale(gpointer data, 
                 guint action, 
                 GtkWidget * widget
                 )
/*-----------------------------------------------------------------------------
 */
{
    Display *disp = display_act;
    gint zoom_index = action;

/* g_message("select_zoomscale:: zoom_index=%d", zoom_index); */
    zoom_display(disp, zoom_index);
}



void
zoom_display(Display *disp, 
             gint zoom_index
             )
/*-----------------------------------------------------------------------------
 */
{
    gint screen_width = gdk_screen_width();
    gint screen_height = gdk_screen_height();
    gint width = 0, height = 0;
    gint new_width = 0, new_height = 0;
 
    if (disp != NULL) {
        disp->zoom_factor = zfactor[zoom_index];
        new_width = (gint) (disp->zoom_factor * gpiv_par.img_width + 
                            VIEW_HMARGE);
        new_height = (gint) (disp->zoom_factor * gpiv_par.img_height + 
                             VIEW_VMARGE);
        gtk_window_get_size(GTK_WINDOW (disp->mwin), &width, &height);
        gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(disp->canvas), 
                                         disp->zoom_factor);

        if (new_width < width && new_height < height) {
/*             gtk_window_resize(GTK_WINDOW (disp->mwin),  */
/*                               new_width,  */
/*                               new_height); */

/*             gtk_widget_set_size_request (GTK_WIDGET (disp->mwin),  */
/*                                          new_width,  */
/*                                          new_height); */
            gtk_widget_set_size_request (disp->canvas, 
                                         (gint) (disp->zoom_factor * 
                                                 gpiv_par.img_width), 
                                         (gint) (disp->zoom_factor * 
                                                 gpiv_par.img_height));
            
            disp->stretch_window_tmp = TRUE;
            disp->zoom_factor_old = disp->zoom_factor;
        }
    }

}



gboolean
canvas_display_enter_notify(GtkWidget * widget, 
                            GdkEventMotion * event,
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    Display *disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GtkWidget *view_piv_display0 = 
                        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                                            "view_piv_display0");

    disp->index_x_old = 0;
    disp->index_y_old = 0;
    disp->index_old = FALSE;


    if (m_select == NO_MS) {
	cursor = gdk_cursor_new(GDK_DOTBOX);



    } else if (m_select == SINGLE_AREA_MS
               || m_select == ENABLE_POINT_MS 
               || m_select == DISABLE_POINT_MS) {
	cursor = gdk_cursor_new(GDK_CROSSHAIR);



    } else if ( m_select == DRAG_MS ) {
	cursor = gdk_cursor_new(GDK_CROSS);



    } else if (m_select == SINGLE_POINT_MS
               || m_select == V_LINE_MS
               || m_select == H_LINE_MS
               ) {
	cursor = gdk_cursor_new(GDK_CROSS);
	if (display_act->gpd.exist_piv && disp->display_piv) {
	    hide_all_vectors(&display_act->gpd);
	}

        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
            hide_all_intregs(disp);
	}



    } else if (m_select == AOI_MS 
               || m_select == ENABLE_AREA_MS 
               || m_select == DISABLE_AREA_MS) {
	cursor = gdk_cursor_new(GDK_FLEUR);
        if (m_select == AOI_MS) {
            if (display_act->gpd.exist_piv && disp->display_piv) {
                hide_all_vectors(&display_act->gpd);
            }

            if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
                hide_all_intregs(disp);
            }
        }



    } else if (m_select == SPANLENGTH_MS
               || m_select == V_SPANLENGTH_MS
               || m_select == H_SPANLENGTH_MS
               ) {
	cursor = gdk_cursor_new(GDK_CROSS);
    }

    gdk_window_set_cursor(disp->mwin->window, cursor);

    return TRUE;
}



gboolean
canvas_display_motion_notify(GtkWidget * widget, 
                             GdkEventMotion * event,
                             gpointer data
                             )
/*-----------------------------------------------------------------------------
 */
{
    Display *disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    
    gint x, y;
    gint index_y = 0, index_x = 0;
    
    GdkModifierType state;
    GnomeCanvasPoints *points;
    points = gnome_canvas_points_new(2);
    
    
    if (event->is_hint) {
	gdk_window_get_pointer(event->window, &x, &y, &state);
/* BUGFIX: seems to wordk, but not for displaying vectors */
/* 	x = x / disp->zoom_factor; */
/* 	y = y / disp->zoom_factor; */
	x = (x  + x_offset) / disp->zoom_factor;
	y = (y  + y_offset) / disp->zoom_factor;
    } else {
/* 	x = (event->x  + x_offset) / disp->zoom_factor; */
/* 	y = (event->y  + y_offset) / disp->zoom_factor; */
	x = event->x / disp->zoom_factor;
	y = event->y / disp->zoom_factor;
	state = event->state;
    }
/*     g_message("canvas_display_motion_notify:: xp =%d yp=%d", x, y); */
    
    if (x >= 0 && x < disp->img.image_par.ncolumns
        && y >= 0 && y < disp->img.image_par.nrows) {

/*
 * display particle displacements / velocities and its attributes,
 */
        if (m_select == NO_MS 
            || m_select == SINGLE_AREA_MS 
            || m_select == DRAG_MS
            || m_select == ENABLE_POINT_MS 
            || m_select == DISABLE_POINT_MS) {


/*
 * display locations of interrogation area's and estimators,
 * only display locations of interrogation area's,
 * or nothing
 */
            canvas_display_motion_notify__NO_MS(disp, x, y, &index_x, 
                                                &index_y);
            if ( m_select == DRAG_MS ) {
                canvas_display_motion_notify__DRAG_MS(disp, x, y, &index_x, 
                                                      &index_y);
            }
        
            disp->index_x_old = index_x;
            disp->index_y_old = index_y;


/*
 * Only prints pointer position
 */
        } else if (m_select == AOI_MS 
                   || m_select == ENABLE_AREA_MS 
                   || m_select == DISABLE_AREA_MS ) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px",
                       x, y);
        
        
        } else if ( m_select == SINGLE_POINT_MS) {
            canvas_display_motion_notify__SINGLE_POINT_MS(disp, x, y);
        
/*
 * Update a line
 */
        } else if (m_select == V_LINE_MS) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                       x, y);
        
            if (state & GDK_BUTTON1_MASK && gci_line != NULL ) {
                g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                           piv_eval_par.int_line_col, y);
                update_line(piv_eval_par.int_line_col, 
                            piv_eval_par.int_line_row_start,
                            piv_eval_par.int_line_col,
                            y);
            }


        } else if (m_select == H_LINE_MS) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                       x, y);
            
            if (state & GDK_BUTTON1_MASK && gci_line != NULL ) {
                g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d y=%d", 
                           x, piv_eval_par.int_line_row);
                update_line(piv_eval_par.int_line_col_start, 
                            piv_eval_par.int_line_row,
                            x,
                            piv_eval_par.int_line_row);
            }
        
        
        } else if (m_select == SPANLENGTH_MS) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                       x, y);
        
            if (state & GDK_BUTTON1_MASK && gci_line != NULL ) {
                g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d y=%d", 
                           x, y);
                update_line(disp->xgrab_first, 
                            disp->ygrab_first,
                            x,
                            y);
            }

        } else if (m_select == V_SPANLENGTH_MS) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                       x, y);
        
            if (state & GDK_BUTTON1_MASK && gci_line != NULL ) {
                g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d y=%d", 
                           disp->xgrab_first, y);
                update_line(disp->xgrab_first, 
                            disp->ygrab_first,
                            disp->xgrab_first,
                            y);
            }
        
        } else if (m_select == H_SPANLENGTH_MS) {
            g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d px y=%d px", 
                       x, y);
        
            if (state & GDK_BUTTON1_MASK && gci_line != NULL ) {
                g_snprintf(msg_display, GPIV_MAX_CHARS, "x=%d y=%d", 
                           x, disp->ygrab_first);
                update_line(disp->xgrab_first, 
                            disp->ygrab_first,
                            x,
                            disp->ygrab_first);
                
            }
        }

    
    
        gnome_appbar_push(GNOME_APPBAR(disp->appbar), msg_display);
        gnome_appbar_push(GNOME_APPBAR(gpiv->appbar), msg_display);
    
        if (state & GDK_BUTTON1_MASK) {
            if ((m_select == AOI_MS
                 || m_select == ENABLE_AREA_MS 
                 || m_select == DISABLE_AREA_MS)
                && gci_aoi != NULL) {
                update_rect(x, y);
            }
        }
    
/*     } else { */
/*         g_message("canvas_display_motion_notify: out of image borders"); */
    }

    gnome_canvas_points_free(points);
    
    return TRUE;
}



gboolean
canvas_display_button_press(GtkWidget * widget, 
                            GdkEventButton * event,
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    Display * disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint x, y;
    gint index_x = 0, index_y = 0;
    GdkModifierType state;
    GnomeCanvasPoints *points;
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");

    points = gnome_canvas_points_new(2);
    GtkMenu *menu = GTK_MENU (disp->display_menu);

    gdk_window_get_pointer(event->window, &x, &y, &state);
/* BUGFIX: */
/*     x = x / disp->zoom_factor; */
/*     y = y / disp->zoom_factor; */
    x = (x  + x_offset)/ disp->zoom_factor;
    y = (y  + y_offset) / disp->zoom_factor;

    if (event->button == 1) {
/*
 * No action
 */
        if (m_select == NO_MS
            ) {


/*
 * select Area Of Interest
 */
        } else if (m_select == AOI_MS) {
/*
 * storige of original AOI
 */
            disp->intreg.col_start_old = piv_eval_par.col_start;
            disp->intreg.row_start_old = piv_eval_par.row_start;
            
            disp->xgrab_first = x;
            disp->ygrab_first = y;
            create_rect(disp, x, y);

/*
 * analyse at single interrogation area
 */
        } else if ((m_select == SINGLE_AREA_MS || m_select == DRAG_MS) &&
                   disp->img.exist_img) {
            canvas_display_button_press__SINGLE_AREA_MS(gpiv, disp, x, y);

/*
 * analyse at single point
 */
        } else if (m_select == SINGLE_POINT_MS && disp->img.exist_img) {
            canvas_display_button_press__SINGLE_POINT_MS(gpiv, disp, x, y); 


        } else if (m_select == V_LINE_MS && display_act->img.exist_img) {
            piv_eval_par.int_line_col = x;
            piv_eval_par.int_line_row_start = y;
            if (gci_line != NULL) destroy_line();
            create_line(disp, x, y, x, y);
             

        } else if (m_select == H_LINE_MS && disp->img.exist_img) {
            piv_eval_par.int_line_col_start = x;
            piv_eval_par.int_line_row = y;
            if (gci_line != NULL) destroy_line();
            create_line(disp, x, y, x, y);
            
            
        } else if ( m_select == ENABLE_POINT_MS ) {
/*
 * Only active with piv data
 */
            if (disp->gpd.exist_piv && disp->display_piv 
/*                && GTK_CHECK_MENU_ITEM(view_piv_display0)->active */
                ) {
                search_nearest_index(disp, x, y, &index_x, &index_y,
/*                                      DATA_TYPE__INTREG); */
                                     DATA_TYPE__PIV);
                disp->gpd.piv_data.peak_no[index_y][index_x] = 1;
                update_vector(&disp->gpd, index_y, index_x);
            }


/*
 * Enabling and disabling only active with piv data
 */
        } else if ((m_select == DISABLE_POINT_MS
                    ||  m_select == ENABLE_AREA_MS
                    ||  m_select == DISABLE_AREA_MS)
                   && disp->gpd.exist_piv && disp->display_piv) {
            canvas_display_button_press__DISABLE_POINT_MS(disp, x, y); 


        } else if (m_select == SPANLENGTH_MS 
                   && disp->img.exist_img) {
            disp->xgrab_first = x;
            disp->ygrab_first = y;       
            if (gci_line != NULL) destroy_line();
            create_line(disp, x, y, x, y);
            
            
            
        } else if (m_select == V_SPANLENGTH_MS 
                   && disp->img.exist_img) {
            disp->xgrab_first = x;
            disp->ygrab_first = y;       
            if (gci_line != NULL) destroy_line();
            create_line(disp, x, y, x, y);
            
            
            
        } else if (m_select == H_SPANLENGTH_MS  
                   && disp->img.exist_img) {
            disp->xgrab_first = x;
            disp->ygrab_first = y;       
            if (gci_line != NULL) destroy_line();
            create_line(disp, x, y, x, y);
            
            
        } else {
            g_warning(_("canvas_display_button_press: should not arrive here; m_select=%d"), 
                      m_select);
        }
        

    } else if (event->button == 3) {
        gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
                        event->button, event->time);
    }

    gnome_canvas_points_free(points);
    return TRUE;
}


gboolean
canvas_display_button_release(GtkWidget * widget, 
                              GdkEventButton * event,
                              gpointer data
                              )
/*-----------------------------------------------------------------------------
 */
{
    Display *disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    GtkWidget *view_piv_display0 = 
                        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                                            "view_piv_display0");
    gint x, y, i, j;
    GdkModifierType state;

    shift_pressed = event->state & GDK_SHIFT_MASK;
    gdk_window_get_pointer(event->window, &x, &y, &state);
/* BUGFIX */
/*     x = x / disp->zoom_factor; */
/*     y = y / disp->zoom_factor; */
    x = (x  + x_offset)/ disp->zoom_factor;
    y = (y  + y_offset) / disp->zoom_factor;



    if (event->button == 1) {

	if (m_select == AOI_MS && gci_aoi != NULL) {
            canvas_display_button_release__AOI_MS(disp, gpiv,
                                                  view_piv_display0,
                                                  x, y);



        } else if ((m_select == V_LINE_MS || m_select == H_LINE_MS) 
            && disp->img.exist_img) {
            canvas_display_button_release__HV_LINE_MS(disp, gpiv,
                                                  view_piv_display0,
                                                  x, y);
              


/*
 * analyse at single interrogation area
 */
        } else if ((m_select == SINGLE_AREA_MS || m_select == DRAG_MS) &&
                   disp->img.exist_img) {
            canvas_display_button_release__SINGLE_AREA_MS(gpiv, disp, x, y);



	} else if (m_select == ENABLE_AREA_MS && gci_aoi != NULL) {
            enable_col_end = x;
            enable_row_end = y;
            if (disp->gpd.exist_piv) {
                for (i = 0; i < disp->gpd.piv_data.ny; i++) {
                    for (j = 0; j < disp->gpd.piv_data.nx; j++) {
                        if (disp->gpd.piv_data.point_x[i][j] >= 
                            enable_col_start
                            && disp->gpd.piv_data.point_x[i][j] < 
                            enable_col_end
                            && disp->gpd.piv_data.point_y[i][j] >= 
                            enable_row_start 
                            && disp->gpd.piv_data.point_y[i][j] < 
                            enable_row_end) {
                            disp->gpd.piv_data.peak_no[i][j] = 1;
                            if (disp->display_piv)                         
                                update_vector(&disp->gpd, i, j);
                            
                        }
                    }
                }
            }
	    if (gci_aoi != NULL) destroy_rect();



	} else if (m_select == DISABLE_AREA_MS && gci_aoi != NULL) {
            enable_col_end = x;
            enable_row_end = y;
            if (disp->gpd.exist_piv) {
                for (i = 0; i < disp->gpd.piv_data.ny; i++) {
                    for (j = 0; j < disp->gpd.piv_data.nx; j++) {
                        if (disp->gpd.piv_data.point_x[i][j] >= 
                            enable_col_start
                            && disp->gpd.piv_data.point_x[i][j] < 
                            enable_col_end 
                            && disp->gpd.piv_data.point_y[i][j] >= 
                            enable_row_start 
                            && disp->gpd.piv_data.point_y[i][j] < 
                            enable_row_end) {
                            disp->gpd.piv_data.peak_no[i][j] = -1;
                            disp->gpd.piv_data.snr[i][j] = GPIV_SNR_DISABLE;

                            if (piv_post_par.set == TRUE) {
                                disp->gpd.piv_data.dx[i][j] = 
                                    piv_post_par.set_dx;
                                disp->gpd.piv_data.dy[i][j] = 
                                    piv_post_par.set_dx;
                            }

                            if (disp->display_piv)                         
                                update_vector(&disp->gpd, i, j);
                            
                        }
                    }
                }
            }
	    if (gci_aoi != NULL) destroy_rect();



        } else if (m_select == SPANLENGTH_MS 
                   && disp->img.exist_img
                   && gci_line != NULL
                   ) {
            gpiv_var.img_span_px = 
                sqrt((x - disp->xgrab_first) * (x - disp->xgrab_first) 
                     + (y - disp->ygrab_first) * (y - disp->ygrab_first));
            if (gci_line != NULL) destroy_line ();

/*             g_warning("canvas_display_button_release:: Hor distance = %d Vert distance = %d; Abs distance = %f",  */
/*                       abs(x - disp->xgrab_first), */
/*                       abs(y - disp->ygrab_first), */
/*                       gpiv_var.img_span_px); */
           gtk_spin_button_set_value(GTK_SPIN_BUTTON
                                     (gpiv->imgh->spinbutton_sscale_px), 
                                     gpiv_var.img_span_px
                                     );



        } else if (m_select == V_SPANLENGTH_MS 
                   && disp->img.exist_img
                   && gci_line != NULL
                   ) {
            gpiv_var.img_span_px = abs(y - disp->ygrab_first);
            if (gci_line != NULL) destroy_line ();
/*             g_warning("canvas_display_button_release:: distance_y = %d", (gint) gpiv_var.img_span_px); */
           gtk_spin_button_set_value(GTK_SPIN_BUTTON
                                     (gpiv->imgh->spinbutton_sscale_px), 
                                     gpiv_var.img_span_px);



        } else if (m_select == H_SPANLENGTH_MS  
                   && disp->img.exist_img
                   && gci_line != NULL
                   ) {
            gpiv_var.img_span_px = abs(x - disp->xgrab_first);
            if (gci_line != NULL) destroy_line ();
/*             g_warning("canvas_display_button_release:: distance_x = %d", (gint) gpiv_var.img_span_px); */
           gtk_spin_button_set_value(GTK_SPIN_BUTTON
                                     (gpiv->imgh->spinbutton_sscale_px), 
                                     gpiv_var.img_span_px);


        
/*
 * No action
 */
        } else if (m_select == GPIV_NONE 
                   || m_select == SINGLE_POINT_MS
                   || m_select == ENABLE_POINT_MS
                   || m_select == DISABLE_POINT_MS
                   ) {

        } else {
            g_warning(_("canvas_display_button_release: should not arrive here; m_select=%d"), 
                      m_select);
        }   
    }

    return TRUE;
}




gboolean 
canvas_display_leave_notify(GtkWidget * widget, 
                            GdkEventMotion * event,
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    Display * disp = gtk_object_get_data(GTK_OBJECT(widget), "disp");
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    GtkWidget * view_piv_display0 = 
        gtk_object_get_data(GTK_OBJECT(disp->mwin), 
                            "view_piv_display0");

    gnome_appbar_push(GNOME_APPBAR(disp->appbar), 
                      disp->msg_display_default);
    gnome_appbar_push(GNOME_APPBAR(gpiv->appbar), msg_default);


    if (m_select == SINGLE_POINT_MS 
        && GTK_CHECK_MENU_ITEM(view_piv_display0)->active
        ) {
        if (!disp->intreg.exist_int) {
            update_intreg1(disp, m_select_index_y, m_select_index_x);
            update_intreg2(disp, m_select_index_y, m_select_index_x);
        }

        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
            show_all_intregs(disp);
        }


       if (disp->gpd.exist_piv && disp->display_piv) {
            show_all_vectors(&disp->gpd);
        }


    } else if (m_select == DRAG_MS) {
        if (!disp->intreg.exist_int) {
            update_intreg1(disp, disp->index_y_old, disp->index_x_old);
            update_intreg2(disp, disp->index_y_old, disp->index_x_old);
        }

    } else if (m_select == SINGLE_AREA_MS && disp->intreg.exist_int) {
        if (!disp->intreg.exist_int) {
            update_intreg1(disp, 0, 0);
            update_intreg2(disp, 0, 0);
        }


    } else if (m_select == AOI_MS
        || m_select == V_LINE_MS
        || m_select == H_LINE_MS) {
        if (GTK_CHECK_MENU_ITEM(view_piv_display0)->active) {
           show_all_intregs(disp);
        }

        if (disp->gpd.exist_piv && disp->display_piv) {
            show_all_vectors(&disp->gpd);
        }

        if (gci_line != NULL) {
            gtk_object_destroy(GTK_OBJECT(gci_line));
            gci_line = NULL;
        }

    }



    if (GDK_BUTTON1_MASK) {
/* 
 * Cancel AOI_MS selection
 */
	if (m_select == AOI_MS) {
	    if (gci_aoi != NULL) {
		piv_eval_par.col_start = disp->intreg.col_start_old;
		piv_eval_par.row_start = disp->intreg.row_start_old;
		gtk_spin_button_set_value(GTK_SPIN_BUTTON
					  (gpiv->piveval->spinbutton_colstart),
					  piv_eval_par.col_start);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON
					  (gpiv->piveval->spinbutton_rowstart),
					  piv_eval_par.row_start);
                if (gci_aoi != NULL) destroy_rect();
	    }

/*             show_all_intregs(disp); */

	} else if ( m_select == ENABLE_AREA_MS 
                    || m_select == DISABLE_AREA_MS) { 
             enable_col_start = 0;
             enable_row_start = 0;
             enable_col_end = 0;
             enable_row_end = 0;
	    if (gci_aoi != NULL) destroy_rect();

/*
 * Cancel V_LINE_MS, LINE_MS selection
 * (V_, H_) SPANLENGTH_MS
 */
        } else if (m_select == V_LINE_MS 
                   || m_select == H_LINE_MS
                   || m_select == SPANLENGTH_MS
                   || m_select == V_SPANLENGTH_MS
                   || m_select == H_SPANLENGTH_MS
                   ) {
            if (gci_line != NULL) destroy_line();
        }
    }




/*
 * General
 */
    if (cursor != NULL) {
	gdk_cursor_destroy(cursor);
	cursor = NULL;
    }

    if (disp->intreg.gci_intreg2[disp->index_y_old][disp->index_x_old] 
        != NULL) {
        update_intreg2(disp, disp->index_y_old, disp->index_x_old);
    }

    if (disp->intreg.gci_intreg1[disp->index_y_old][disp->index_x_old] 
        != NULL) {
        update_intreg1(disp, disp->index_y_old, disp->index_x_old);
    }

    disp->index_x_old = 0;
    disp->index_y_old = 0;

    return TRUE;
}



/*
 * Callback functions for display
 */

gboolean
canvas_event(GtkWidget * widget, 
             GdkEvent *event,
             gpointer data
             )
/*-----------------------------------------------------------------------------
 */
{
/*   gchar keystate[30]; */
    shift_pressed = event->button.state & GDK_SHIFT_MASK;
    ctrl_pressed = event->button.state & GDK_CONTROL_MASK;
    alt_pressed = event->button.state & GDK_MOD1_MASK;
    
    switch ((int)event->type)
        {
        case GDK_ENTER_NOTIFY:
            canvas_display_enter_notify(widget, &event->motion, NULL);
            return TRUE;

        case GDK_MOTION_NOTIFY:
            canvas_display_motion_notify(widget, &event->motion, NULL);
            return TRUE;

        case GDK_BUTTON_PRESS:
            canvas_display_button_press(widget, &event->button, NULL);
            return TRUE;

        case GDK_BUTTON_RELEASE:
            canvas_display_button_release(widget, &event->button, NULL);
            return TRUE;

        case GDK_LEAVE_NOTIFY:
            canvas_display_leave_notify(widget, &event->motion, NULL);
            return TRUE;
            
        }

/*
 * Event not handled; try parent item
 */
    return FALSE;
/*   return TRUE; */
}


   
void 
view_toggle_stretch_display(GtkWidget * widget, 
                            gpointer data
                            )
/*-----------------------------------------------------------------------------
 */
{
    Display * disp = display_act;
    gint screen_width = gdk_screen_width();
    gint screen_height = gdk_screen_height();

    gint new_width = (gint) (disp->zoom_factor * gpiv_par.img_width + 
                             VIEW_HMARGE);
    gint new_height = (gint) (disp->zoom_factor * gpiv_par.img_height + 
                              VIEW_VMARGE);

    if (new_width >= screen_width - VIEW_HMARGE 
        || new_height >= screen_height - VIEW_VMARGE) {
        new_width = screen_width - VIEW_HMARGE;
        new_height = screen_height - VIEW_VMARGE;
    }

/*     gtk_window_resize (GTK_WINDOW (disp->mwin), */
/*                                    new_width, */
/*                                    new_height); */

/*     gtk_widget_set_size_request (GTK_WIDGET (disp->mwin),  */
/*                                  new_width,  */
/*                                  new_height); */

    gtk_widget_set_size_request (disp->canvas, 
                                 (gint) (disp->zoom_factor * 
                                         gpiv_par.img_width), 
                                 (gint) (disp->zoom_factor * 
                                         gpiv_par.img_height));

    disp->stretch_window_tmp = TRUE;
    disp->zoom_factor_old = disp->zoom_factor;
    g_message("view_toggle_stretch_display:: 1/1");
}


void
select_view_background(gpointer data, 
                       guint action, 
                       GtkWidget *widget)
/*-----------------------------------------------------------------------------
 */
{
    gchar *color = NULL;
    Display *disp = display_act;

    gpiv_par.background =  action;

    hide_img1(disp);
    hide_img2(disp);

    if (action == SHOW_BG_DARKBLUE) {
        color = "darkblue";
        if (disp->gci_bg != NULL) {
            gnome_canvas_item_set(GNOME_CANVAS_ITEM(disp->gci_bg),
                                  "fill_color", color,
                                  NULL);
        }

    } else if (action == SHOW_BG_BLACK) {
        color = "black";
        if (disp->gci_bg != NULL) {
            gnome_canvas_item_set(GNOME_CANVAS_ITEM(disp->gci_bg),
                                  "fill_color", color,
                                  NULL);
        }

    } else if (action == SHOW_BG_IMG1) {
        show_img1(display_act);

    } else if (action == SHOW_BG_IMG2) {
            show_img2(display_act);

    } else {
        color = "black";
        if (disp->gci_bg != NULL) {
            gnome_canvas_item_set(GNOME_CANVAS_ITEM(disp->gci_bg),
                                  "fill_color", color,
                                  NULL);
        } else {
            g_warning(_("select_view_background: should not arrive here"));
        }
    }

}



/* void  */
/* view_toggle_img1(GtkWidget * widget,  */
/*                  gpointer data */
/*                  ) */
/*-----------------------------------------------------------------------------
 */
/* { */
/*     if (display_act->img.exist_img) { */
/* 	if (GTK_CHECK_MENU_ITEM(widget)->active) { */
/* 	    display_act->display_img1 = TRUE; */
/*             show_img1(display_act); */
/* 	} else { */
/* 	    display_act->display_img1 = FALSE; */
/*             hide_img1(display_act); */
/* 	} */
/*     } */

/* } */



/* void  */
/* view_toggle_img2(GtkWidget * widget,  */
/*                       gpointer data */
/*                       ) */
/*-----------------------------------------------------------------------------
 */
/* { */
/*     if (display_act->img.exist_img) { */
/* 	if (GTK_CHECK_MENU_ITEM(widget)->active) { */
/* 	    display_act->display_img2 = TRUE; */
/*             show_img2(display_act); */
/* 	} else { */
/* 	    display_act->display_img2 = FALSE; */
/*             hide_img2(display_act); */
/* 	} */
/*     } */

/* } */



void 
view_toggle_intregs(GtkWidget * widget, 
                         gpointer data
                         )
/*-----------------------------------------------------------------------------
 */
{
	if (GTK_CHECK_MENU_ITEM(widget)->active
            ) {
/*
 * view_toggle_intregs_flag is set false during creation of display and
 * pop-up menu in order to avoid double creating of intregs
 */
            if (view_toggle_intregs_flag) {
                create_all_intregs(display_act);
                show_all_intregs(display_act);
            }
            display_act->display_intregs = TRUE;
	} else {
            destroy_all_intregs(display_act);
            display_act->display_intregs = FALSE;
	}
}



void view_toggle_piv(GtkWidget * widget, 
                     gpointer data
                     )
/*-----------------------------------------------------------------------------
 */
{
    if (display_act->gpd.exist_piv) {
	if (GTK_CHECK_MENU_ITEM(widget)->active) {
	    display_act->display_piv = TRUE;
/*             show_all_vectors(&display_act->gpd); */
            create_all_vectors(&display_act->gpd);
	} else {
	    display_act->display_piv = FALSE;
/*             hide_all_vectors(&display_act->gpd); */
            destroy_all_vectors(&display_act->gpd);
	}
    }
}



void
select_view_scalardata(gpointer data, 
                       guint action, 
                       GtkWidget *widget
                       )
/*-----------------------------------------------------------------------------
 */
{
    gchar *color = NULL;
    Display *disp = display_act;
    
    gpiv_par.background =  action;
    
    hide_all_scalars(disp, GPIV_VORTICITY);
    hide_all_scalars(disp, GPIV_S_STRAIN);
    hide_all_scalars(disp, GPIV_N_STRAIN);

    if (action == SHOW_SC_NONE) {
        hide_all_scalars(disp, GPIV_VORTICITY);
        hide_all_scalars(disp, GPIV_S_STRAIN);
        hide_all_scalars(disp, GPIV_N_STRAIN);

    } else if (action == SHOW_SC_VORTICITY) {
        show_all_scalars(display_act, GPIV_VORTICITY);
        
    } else if (action == SHOW_SC_SSTRAIN) {
        show_all_scalars(display_act, GPIV_S_STRAIN);
        
    } else if (action == SHOW_SC_NSTRAIN) {
        show_all_scalars(display_act, GPIV_N_STRAIN);

    } else {
        color = "black";
        if (disp->gci_bg != NULL) {
            gnome_canvas_item_set(GNOME_CANVAS_ITEM(disp->gci_bg),
                                  "fill_color", color,
                                  NULL);
        } else {
            g_warning(_("select_view_scalardata: should not arrive here"));
        }
    }

}



/* void view_toggle_vor(GtkWidget * widget,  */
/*                      gpointer data */
/*                      ) */
/*-----------------------------------------------------------------------------
 */
/* { */
/*     if (display_act->gpd.exist_vor) { */
/* 	if (GTK_CHECK_MENU_ITEM(widget)->active) { */
/* 	    display_act->display_vor = TRUE; */
/*             show_all_scalars(display_act, GPIV_VORTICITY); */
/*         } else { */
/* 	    display_act->display_vor = FALSE; */
/*             hide_all_scalars(display_act, GPIV_VORTICITY); */
/*         } */
/*     } */
/* } */



/* void */
/* view_toggle_sstrain(GtkWidget *widget,  */
/*                     gpointer data */
/*                     ) */
/*-----------------------------------------------------------------------------
 */
/* { */
/*     if (display_act->gpd.exist_sstrain) { */
/* 	if (GTK_CHECK_MENU_ITEM(widget)->active) { */
/* 	    display_act->display_sstrain = TRUE; */
/*             show_all_scalars(&display_act->gpd.sstrain_data, GPIV_S_STRAIN); */
/*         } else { */
/* 	    display_act->display_sstrain = FALSE; */
/*             hide_all_scalars(&display_act->gpd.sstrain_data, GPIV_S_STRAIN); */
/*         } */
/*     } */
/* } */



/* void */
/* view_toggle_nstrain(GtkWidget *widget,  */
/*                     gpointer data */
/*                     ) */
/*-----------------------------------------------------------------------------
 */
/* { */
/*     if (display_act->gpd.exist_nstrain) { */
/* 	if (GTK_CHECK_MENU_ITEM(widget)->active) { */
/* 	    display_act->display_nstrain = TRUE; */
/*             show_all_scalars(&display_act->gpd.nstrain_data, GPIV_N_STRAIN); */
/*         } else { */
/* 	    display_act->display_nstrain = FALSE; */
/*             hide_all_scalars(&display_act->gpd.nstrain_data, GPIV_N_STRAIN); */
/*         } */
/*     } */
/* } */



void 
select_vectorscale(gpointer data, 
                   guint action, 
                   GtkWidget * widget
                   )
/*-----------------------------------------------------------------------------
  Setting vector_scale from display pop-up menu */
{
    /* if (display () == NULL) return; */
    gpiv_par.vector_scale = action;
    update_all_vectors(&display_act->gpd);
}



void 
select_vectorcolor(gpointer data, 
                   guint action, 
                   GtkWidget * widget
                   )
/*-----------------------------------------------------------------------------
  Setting vector_color from display pop-up menu */
{
    /* if (display () == NULL) return; */
    gpiv_par.vector_color = action;
    update_all_vectors(&display_act->gpd);
}



void
nav_popup_click_handler(GtkWidget * widget, 
                        GdkEventButton * event,
			gpointer data
                        )
/*-----------------------------------------------------------------------------
 */
{
    /* Seee gimp: nav_window.c */
}



void 
on_view_options_clicked(GtkButton * button, 
                        gpointer user_data
                        )
/*-----------------------------------------------------------------------------
 */
{

}



/*
 * img_display callbacks
 */
/* gboolean */
/* on_darea_img_expose(GtkWidget * widget, */
/* 		    GdkEventExpose * event, gpointer user_data) */
/* { */
/*     gdk_draw_gray_image(widget->window, */
/* 			widget->style->fg_gc[GTK_STATE_NORMAL],  */
/*                         0,  */
/*                         0, */
/*
 * 4-byte aligned width
 */
/*                         display_act->img.rgb_img_width,  */
/*                         gpiv_par.img_height,  */
/*                         GDK_RGB_DITHER_MAX, */
/* 		        display_act->img.graybuf_img,  */
/*
 * WIDTH
 */
/*                         display_act->img.rgb_img_width); */

/*     return TRUE; */
/* } */

