# gozerbot/db.py
#
#

""" mysql interface """

__copyright__ = 'this file is in the public domain'

from gozerbot.config import config
from gozerbot.generic import rlog, lockdec, tolatin1, handle_exception
from gozerbot.datadir import datadir
import thread, os, time

dblock = thread.allocate_lock()
dblocked = lockdec(dblock)

class Db(object):

    """ this class implements an database connection. it connects to the 
        database on initialisation.
    """

    def __init__(self, doconnect=True, dbtype=None):
        self.dbname = ""
        self.dbhost = ""
        self.dbuser = ""
        self.dbpasswd = ""
        self.connection = None
        self.timeout = 15
        self.oldstyle = ""
        self.dbtype = dbtype or config['dbtype'] or 'mysql'
        if doconnect:
            self.connect()

    @dblocked
    def connect(self, dbname=None, dbhost=None, dbuser=None, dbpasswd=None, \
timeout=15, oldstyle=False):
        """ connect to the database """
        self.dbname = dbname or config['dbname']
        self.dbhost = dbhost or config['dbhost']
        self.dbuser = dbuser or config['dbuser']
        self.dbpasswd = dbpasswd or config['dbpasswd']
        self.timeout = timeout
        self.oldstyle = oldstyle or config['dboldstyle']
        if self.dbtype == 'mysql':
            import MySQLdb
            self.connection = MySQLdb.connect(db=self.dbname, \
host=self.dbhost, user=self.dbuser, passwd=self.dbpasswd, \
connect_timeout=self.timeout, charset='utf8')
        elif self.dbtype == 'sqlite':
            import sqlite
            self.connection = sqlite.connect(datadir + os.sep + self.dbname)
        elif self.dbtype == 'postgres':
            import psycopg2
            rlog(1000, 'db', 'NOTE THAT POSTGRES IS NOT FULLY SUPPORTED')
            self.connection = psycopg2.connect(database=self.dbname, \
host=self.dbhost, user=self.dbuser, password=self.dbpasswd)
        else:
            rlog(100, 'db', 'unknown database type %s' % self.dbtype)
            return 0
        rlog(5, 'db', "database ok")
        return 1

    def reconnect(self):
        """ reconnect to the mysql server """
        if self.dbtype == 'mysql':
            import MySQLdb
            self.connection = MySQLdb.connect(db=self.dbname, \
host=self.dbhost, user=self.dbuser, passwd=self.dbpasswd, \
connect_timeout=self.timeout, charset='utf8')
        elif self.dbtype == 'sqlite':
            import sqlite
            self.connection = sqlite.connect(self.dbname)
        elif self.dbtype == 'postgres':
            import psycopg2
            self.connection = psycopg2.connect(database=self.dbname, \
host=self.dbhost, user=self.dbuser, password=self.dbpasswd)
        else:
            rlog(100, 'db', 'unknown database type %s' % self.dbtype)
            return 0
        rlog(10, 'db', 'reconnect done')
        return 1

    @dblocked
    def execute(self, execstr, args=None):
        """ execute string on database """
        time.sleep(0.001)
        result = None
        execstr = execstr.strip()
        # first to ping to see if connection is alive .. if not reconnect
        if self.dbtype == 'mysql':
            try:
                self.ping()
            except Exception, ex:
                rlog(10, 'db', "can't ping database: %s" % str(ex))
                rlog(10, 'db', 'reconnecting')
                try:
                    self.reconnect()
                except Exception, ex:
                    rlog(10, 'db', 'failed reconnect: %s' % str(ex))
                    return
        # get cursor
        cursor = self.cursor()
        # excecute string on cursor
        nr = 0
        if args:
            if self.oldstyle:
                nargs = []
                for i in args:
                    nargs.append(tolatin1(i))
                args = nargs
            rlog(-2, 'db', 'exec %s %s' % (execstr, args))
            try:
                if type(args) == tuple or type(args) == list:
                    nr = cursor.execute(execstr, args)
                else:
                    nr = cursor.execute(execstr, (args, ))
            except:
                if self.dbtype == 'postgres':
                    cursor.execute(""" ROLLBACK """)
                raise
        else:
            rlog(-2, 'db', 'exec %s' % execstr)
            nr = cursor.execute(execstr)
        # see if we need to commit the query
        got = False
        if execstr.startswith('INSERT') or execstr.startswith('UPDATE'):
            nr = cursor.lastrowid or nr
            got = True
        elif execstr.startswith('DELETE'):
            nr = cursor.rowcount
            got = True
        if got:
            self.commit()
        # fetch results
        result = None
        try:
            result = cursor.fetchall()
            if not result:
                result = nr
        except Exception, ex:
            if 'no results to fetch' in str(ex):
                pass
            else:
                handle_exception()
            result = nr
        cursor.close()
        return result

    def cursor(self):
        """ return cursor to the database """
        return self.connection.cursor()

    def commit(self):
        """ do a commit on the datase """
        self.connection.commit()

    def ping(self):
        """ do a ping """
        return self.connection.ping()

    def close(self):
        """ close database """
        self.connection.close()

# create default database if enabled
cfg = config['dbenable']
if cfg:
    db = Db()
else:
    db = None
