# gozerbot/irc.py
#
#

""" an Irc object handles the connection to the irc server .. receiving,
    sending, connect and reconnect code """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import rlog, handle_exception, getrandomnick, todefenc, \
fix_format, splittxt, waitforqueue
from gozerbot.wait import Wait
from gozerbot.config import config
from gozerbot.ircevent import Ircevent
from gozerbot.monitor import saymonitor
from gozerbot.less import Less
from gozerbot.ignore import shouldignore
from gozerbot.pdod import Pdod
from gozerbot.datadir import datadir
from gozerbot.eventhandler import Outputhandler
from gozerbot.fleet import fleet

import gozerbot.thr as thr
import time, thread, socket, threading, os, Queue

class AlreadyConnected(Exception):

    """ already connected exception """

    pass

class AlreadyConnecting(Exception):

    """ bot is already connecting exception """

    pass

class Irc(object):

    """ the irc class, provides interface to irc related stuff """

    def __init__(self, name='main'):
        self.name = name 
        self.type = 'irc'
        self.wait = Wait()
        self.lastoutput = 0
        self.outputlock = thread.allocate_lock()
        self.connectok = threading.Event()
        self.outputhandler = Outputhandler(self)
        self.fsock = None
        self.sock = None
        self.nick = config['nick'] or 'gb1'
        self.orignick = config['orignick'] or self.nick
        self.server = 'localhost'
        self.port = 6667
        self.password = None
        self.ipv6 = None
        self.stopped = 0
        self.nolimiter = config['nolimiter']
        self.reconnectcount = 0
        self.pongcheck = 0
        self.blocking = config['blocking'] or 1
        self.nickchanged = 0
        self.noauto433 = 0
        self.connecting = False
        self.less = Less(5)
        self.state = Pdod(datadir + os.sep + '%s.state' % self.name)
        if not self.state.has_key('alternick'):
            self.state['alternick'] = config['alternick']
        if not self.state.has_key('no-op'):
            self.state['no-op'] = []
        if not self.state.has_key('nick'):
            self.state['nick'] = config['nick'] or 'gb1'
        if not self.state.has_key('confignick'):
            self.state['confignick'] = config['nick']
        if config['nick'] != self.state['confignick']:
            self.state['nick'] = config['nick']
            self.state['confignick'] = config['nick']
            self.state.save()
        self.nick = self.state['nick']
        self.outputhandler.start()

    def connect(self, nick, server, port=6667, password=None, ipv6=None, \
reconnect=True):
        """ connect to server/port using nick .. connect can timeout so catch
            exception .. reconnect if enabled """
        if self.connecting:
            rlog(10, self.name, 'already connecting')
            raise AlreadyConnecting()
        if self.connectok.isSet():
            rlog(10, self.name, 'already connected')
            raise AlreadyConnected()
        try:
            return self.doconnect(nick, server, port, password, ipv6)
        except Exception, ex:
            if self.stopped:
                return 0
            rlog(10, self.name, str(ex))
            if reconnect:
                return self.reconnect()
        if not fleet.byname(self.name):
            fleet.addbot(self)

    def doconnect(self, nick, server, port, password, ipv6):
        """ connect to server/port using nick """
        self.connecting = True
        self.connectok.clear()
        self.nick = str(nick)
        self.orignick = self.nick
        self.server = str(server)
        self.port = int(port)
        self.password = password
        self.ipv6 = ipv6
        # create socket
        if ipv6:
            rlog(1, self.name, 'creating ipv6 socket')
            self.sock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
            self.ipv6 = 1
        else:
            rlog(0, self.name, 'creating ipv4 socket')
            self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # optional bind
        elite = config['bindhost']
        if elite:
            try:
                self.sock.bind((elite, 0))
            except socket.gaierror:
                rlog(10, self.name, "can't bind to %s" % elite)
        # do the connect .. set timeout to 30 sec upon connecting
        rlog(10, self.name, 'connecting to ' + self.server)
        self.sock.settimeout(30)
        self.sock.connect((self.server, int(self.port)))
        # we are connected
        rlog(10, self.name, 'connection ok')
        self.stopped = 0
        # make file socket
        self.fsock = self.sock.makefile("r")
        # set blocking
        self.sock.setblocking(self.blocking)
        self.fsock._sock.setblocking(self.blocking)
        # set socket time out
        if self.blocking:
            socktimeout = config['socktimeout']
            if not socktimeout:
                socktimeout = 301.0
            else:
                socktimeout = float(socktimeout)
            self.sock.settimeout(socktimeout)
            self.fsock._sock.settimeout(socktimeout)
        # start readloop
        self.outputhandler.start()
        rlog(0, self.name, 'starting readloop')
        thr.start_new_thread(self.readloop, ())
        # execute onconnect file .. code that can be executed after login
        self.logon()
        self.onconnect()
        # init 
        self.reconnectcount = 0
        self.nickchanged = 0
        self.outputhandler.nooutput = False
        self.connecting = False
        return 1

    def logon(self):
        """ log on to the network """
        # if password is provided send it
        if self.password:
            rlog(10, self.name ,'sending password')
            self.send("PASS %s" % self.password)
        # register with irc server
        rlog(10, self.name, 'registering with server %s using nick: %s' % \
(self.server, self.nick))
        rlog(10, self.name, 'this may take a while')
        # check for username and realname
        username = config['username'] or self.nick
        realname = config['realname'] or username
        # first send nick
        self.send("NICK %s" % self.nick)
        # send USER
        self.send("USER %s localhost localhost :%s" % (username, realname))
        # wait on login
        self.connectok.wait()
        # registration ok
        rlog(10, self.name, 'logged on !')

    def onconnect(self):
        """ run the onconnect-botname file after connect """
        onconnectfile = 'onconnect-%s' % self.name
        try:
            execfile(onconnectfile, globals(), locals())
            rlog(5, self.name, '%s done' % onconnectfile)
        except Exception, ex:
            try:
                (errnr, error) = ex
                if errnr == 2:
                    pass
                else:
                    rlog(5, self.name, 'error excecuting %s: %s' % \
(onconnectfile, str(error)))
            except ValueError:
                rlog(5, self.name, 'error excecuting %s: %s' % \
(onconnectfile, str(ex)))

    def readloop(self):
        """ loop on the socketfile """
        self.stopped = 0
        doreconnect = 0
        timeout = 1
        while not self.stopped:
            try:
                time.sleep(0.001)
                res = todefenc(self.fsock.readline())
                # if res == "" the other side has disconnected
                if not res and not self.stopped:
                    doreconnect = 1
                    break
                if self.stopped:
                    break 
                res = res.rstrip()
                rlog(2, self.name, res)
                # parse txt read into an ircevent
                try:
                    ievent = Ircevent().parse(self, res)
                except Exception, ex:
                    handle_exception()
                    continue
                # call handle_ievent 
                if ievent:
                    self.handle_ievent(ievent)
                timeout = 1
            except socket.timeout:
                # timeout occured .. first time send ping .. reconnect if
                # second timeout follows
                if self.stopped:
                    break
                timeout += 1
                if timeout > 2:
                    doreconnect = 1
                    rlog(10, self.name, 'no pong received')
                    break
                rlog(1, self.name, "socket timeout")
                pingsend = self.ping()
                if not pingsend:
                    doreconnect = 1
                    break
                continue
            except Exception, ex:
                if self.stopped:
                    break
                err = ex
                try:
                    (errno, msg) = ex
                except:
                    errno = -1
                    msg = err
                # check for temp. unavailable error .. raised when using
                # nonblocking socket .. 35 is FreeBSD 11 is Linux
                if errno == 35 or errno == 11:
                    time.sleep(0.5)
                    continue
                rlog(10, self.name, "error in readloop: %s" % msg)
                self.outputhandler.nooutput = True
                doreconnect = 1
                break
        rlog(10, self.name, 'readloop stopped')
        # see if we need to reconnect
        self.connectok.clear()
        self.connecting = False
        if doreconnect:
            self.reconnect()

    def shutdown(self):
        """ shutdown sockets """
        rlog(10, self.name, 'shutdown')
        self.stopped = 1
        try:
            self.sock.shutdown(2)
        except:
            pass
        try:
            time.sleep(1)
            self.sock.close()
            self.fsock.close()
        except:
            pass
        self.connecting = False
        self.connectok.clear()

    def exit(self):
        """ exit the bot """
        self.stopped = 1
        self.outputhandler.stop()
        self.shutdown()

    def reconnect(self):
        """ reconnect to the irc server """
        if self.stopped:
            return 0
        self.shutdown()
        rlog(10, self.name, 'reconnecting')
        self.stopped = 0
        time.sleep(5)
        # determine how many seconds to sleep
        if self.reconnectcount > 0:
            reconsleep = self.reconnectcount*60
            rlog(10, self.name, 'sleeping %s seconds' % reconsleep)
            time.sleep(reconsleep)
        if self.stopped:
            return 0
        self.reconnectcount += 1
        return self.connect(self.nick, self.server, self.port, \
self.password, self.ipv6)

    def handle_pong(self, ievent):
        """ set pongcheck on received pong """
        rlog(1, self.name, 'received server pong')
        self.pongcheck = 1

    def sendraw(self, txt):
        """ send raw text to the server """
        if not txt:
            return
        rlog(2, self.name + '.sending', txt)
        try:
            self.sock.send(txt + '\n')
        except Exception, ex:
            rlog(10, self.name, "ERROR: can't send %s" % str(ex))

    def fakein(self, txt):
        """ do a fake ircevent """
        if not txt:
            return
        rlog(1, self.name + '.fakein', txt)
        self.handle_ievent(Ircevent().parse(self, txt))

    def say(self, printto, what, who=None, how='msg', fromm=None, speed=5):
        """ say what to printto """
        if not printto or not what:
            return
        # check if printto is a socket
        if type(printto) == socket.SocketType:
            try:
                printto.send(what + '\n')
            except Exception, ex :
                time.sleep(0.01)
            return
        # if who is set add "who: " to txt
        if who:
            what = "%s: %s" % (who, what)
        self.out(printto, what, who, how, fromm, speed) 

    def out(self, printto, what, who=None, how='msg', fromm=None, speed=5):
        """ output the first 375 chars .. put the rest into cache """
        # first do mombo to get txt converted to ascii
        try:
            what = todefenc(what.rstrip())
        except Exception, ex:
            rlog(10, self.name, "can't output: %s" % str(ex))
            return
        if not what:
            return
        # split up in parts of 375 chars overflowing on word boundaries
        txtlist = splittxt(what)
        size = 0
        # see if we need to store output in less cache
        if len(txtlist) > 1:
            if not fromm:
                self.less.add(printto, txtlist)
            else:
                self.less.add(fromm, txtlist)
            size = len(txtlist) - 1
            result = txtlist[:1][0]
            if size:
                result += " (+%s)" % size
        else:
            result = txtlist[0]
        self.outputhandler.put(speed, self, printto, result, how, who, fromm)

    def output(self, printto, what, how='msg' , who=None, fromm=None):
        """ first output .. then call saymonitor """
        self.outputnolog(printto, what, how, who, fromm)
        saymonitor.callcb(self.name, printto, what, who, how, fromm)
        
    def outputnolog(self, printto, what, how, who=None, fromm=None):
        """ do output to irc server .. rate limit to 3 sec """
        if fromm and shouldignore(fromm):
            return
        self.outputlock.acquire()
        try:
            what = fix_format(what)
            now = time.time()
            timetosleep = 3 - (now - self.lastoutput)
            if timetosleep > 0 and not self.nolimiter:
                rlog(1, self.name, 'flood protect')
                time.sleep(timetosleep)
            self.lastoutput = time.time()
            if what:
                if how == 'msg':
                    self.privmsg(printto, what)
                elif how == 'notice':
                    self.notice(printto, what)
                elif how == 'ctcp':
                    self.ctcp(printto, what)
        except Exception, ex:
            handle_exception()
        self.outputlock.release()

    def donick(self, nick, setorig=0, save=0):
        """ change nick .. do not change orignick """
        if not nick:
            return
        self.noauto433 = 1
        queue = Queue.Queue()
        self.wait.register('NICK', self.nick, queue, 5)
        self.send('NICK %s' % nick)
        result = waitforqueue(queue)
        self.noauto433 = 0
        if not result:
            return 0
        self.nick = nick
        if setorig:
            self.orignick = nick
        if save:
            self.state['nick'] = nick
            self.state.save()
            if fleet.getmainbot() == self:
                config.set('nick', nick)
        return 1

    def join(self, channel, password=None):
        """ join channel with optional password """
        if not channel:
            return
        q = Queue.Queue()
        self.wait.register('353', channel, q, 5)
        if password:
            self.send('JOIN %s %s' % (channel, password))
        else:
            self.send('JOIN %s' % channel)
        result = waitforqueue(q, 5)
        if not result:
            return 'failed to join %s' % channel
        else:
            return 1

    def part(self, channel):
        """ leave channel """
        if not channel:
            return
        q = Queue.Queue()
        self.wait.register('PART', channel, q, 5)
        self.send('PART %s' % channel)
        result = waitforqueue(q, 5)
        if not result:
            return 0
        else:
            return 1

    def who(self, who):
        """ send who query """
        if not who:
            return
        self.send('WHO %s' % who.strip())

    def names(self, channel):
        """ send names query """
        if not channel:
            return
        self.send('NAMES %s' % channel)

    def whois(self, who):
        """ send whois query """
        if not who:
            return
        self.send('WHOIS %s' % who)

    def privmsg(self, printto, what):
        """ send privmsg to irc server """
        if not printto or not what:
            return
        self.send('PRIVMSG %s :%s' % (printto, what))

    def send(self, txt):
        """ send text to irc server """
        if not txt:
            return
        if self.stopped:
            return
        rlog(2, self.name + '.send', txt)
        try:
            self.sock.send(txt[:510]+'\n')
        except Exception, ex:
            # check for broken pipe error .. if so ignore
            try:
                (errno, errstr) = ex
                if errno != 32:
                    raise
                else:
                    rlog(10, self.name, 'broken pipe error .. ignoring')
            except:
                rlog(10, self.name, "ERROR: can't send %s" % str(ex))

    def voice(self, channel, who):
        """ give voice """
        if not channel or not who:
            return
        self.send('MODE %s +v %s' % (channel, who))
 
    def doop(self, channel, who):
        """ give ops """
        if not channel or not who:
            return
        self.send('MODE %s +o %s' % (channel, who))

    def delop(self, channel, who):
        """ de-op user """
        if not channel or not who:
            return
        self.send('MODE %s -o %s' % (channel, who))

    def quit(self, reason='http://r8.cg.nu'):
        """ send quit message """
        rlog(10, self.name, 'sending quit')
        try:
            self.sock.send('QUIT :%s\n' % reason)
        except IOError:
            pass

    def notice(self, printto, what):
        """ send notice """
        if not printto or not what:
            return
        self.send('NOTICE %s :%s' % (printto, what))
 
    def ctcp(self, printto, what):
        """ send ctcp privmsg """
        if not printto or not what:
            return
        self.send("PRIVMSG %s :\001%s\001" % (printto, what))

    def ctcpreply(self, printto, what):
        """ send ctcp notice """
        if not printto or not what:
            return
        self.send("NOTICE %s :\001%s\001" % (printto, what))

    def action(self, printto, what):
        """ do action """
        if not printto or not what:
            return
        self.send("PRIVMSG %s :\001ACTION %s\001" % (printto, what))

    def handle_ievent(self, ievent):
        """ handle ircevent .. dispatch to 'handle_command' method """ 
        try:
            # see if the irc object has a method to handle the ievent
            method = getattr(self,'handle_' + ievent.cmnd.lower())
            # try to call method
            try:
                method(ievent)
            except:
                handle_exception()
        except AttributeError:
            # no command method to handle event
            pass
        try:
            # see if there are wait callbacks
            self.wait.check(ievent)
        except:
            handle_exception()

    def handle_433(self, ievent):
        """ handle nick already taken """
        if self.noauto433:
            return
        nick = ievent.arguments[1]
        # check for alternick
        alternick = self.state['alternick']
        if alternick and not self.nickchanged:
            rlog(10, self.name, 'using alternick %s' % alternick)
            if self.donick(alternick):
                self.nickchanged = 1
                return
        # use random nick
        randomnick = getrandomnick()
        self.donick(randomnick)
        self.nick = randomnick
        rlog(100, self.name, 'ALERT: nick %s already in use/unavailable .. \
using randomnick %s' % (nick, randomnick))
        self.nickchanged = 1

    def handle_ping(self, ievent):
        """ send pong response """
        if not ievent.txt:
            return
        self.send('PONG :%s' % ievent.txt)

    def handle_001(self, ievent):
        """ we are connected  """
        self.connectok.set()

    def handle_privmsg(self, ievent):
        """ check if msg is ctcp or not .. return 1 on handling """
        if ievent.txt and ievent.txt[0] == '\001':
            self.handle_ctcp(ievent)
            return 1

    def handle_notice(self, ievent):
        """ handle notice event .. check for version request """
        if ievent.txt and ievent.txt.find('VERSION') != -1:
            self.say(ievent.nick, config['version'], None, 'notice')
            return 1

    def handle_ctcp(self, ievent):
        """ handle client to client request .. version and ping """
        if ievent.txt.find('VERSION') != -1:
            self.ctcpreply(ievent.nick, 'VERSION %s' % config['version'])
        if ievent.txt.find('PING') != -1:
            try:
                pingtime = ievent.txt.split()[1]
                pingtime2 = ievent.txt.split()[2]
                if pingtime:
                    self.ctcpreply(ievent.nick, 'PING ' + pingtime + ' ' + \
pingtime2)
            except IndexError:
                pass

    def handle_error(self, ievent):
        """ show error """
        if ievent.txt.startswith('Closing'):
            rlog(10, self.name, ievent.txt)
        else:
            rlog(10, self.name + '.ERROR', ievent.txt)

    def ping(self):
        """ ping the irc server """
        rlog(1, self.name, 'sending ping')
        try:
            self.sock.send('PING :%s\n' % self.server)
            return 1
        except Exception, ex:
            rlog(10, self.name, "can't send ping: %s" % str(ex))
            return 0
