//go:build ignore
// +build ignore

package main

import (
	"bytes"
	"fmt"
	"io"
	"log"
	"os"
	"os/exec"

	tcellterm "git.sr.ht/~rockorager/tcell-term"
	"github.com/gdamore/tcell/v2"
	"github.com/gdamore/tcell/v2/views"
)

type model struct {
	term      *tcellterm.Terminal
	s         tcell.Screen
	termView  views.View
	title     *views.TextBar
	titleView views.View
}

// Update is the main event handler. It should only be called by the main thread
func (m *model) Update(ev tcell.Event) bool {
	switch ev := ev.(type) {
	case *tcell.EventKey:
		switch ev.Key() {
		case tcell.KeyCtrlC:
			m.term.Close()
			m.s.Clear()
			m.s.Fini()
			return true
		}
		if m.term != nil {
			return m.term.HandleEvent(ev)
		}
	case *tcell.EventResize:
		if m.term != nil {
			m.termView.Resize(0, 2, -1, -1)
			m.term.Resize()
		}
		m.titleView.Resize(0, 0, -1, 2)
		m.title.Resize()
		m.s.Sync()
		return true
	case *views.EventWidgetContent:
		m.term.Draw()
		m.title.Draw()

		vis, x, y, style := m.term.GetCursor()
		if vis {
			m.s.ShowCursor(x, y+2)
			m.s.SetCursorStyle(style)
		} else {
			m.s.HideCursor()
		}
		m.s.Show()
		return true
	case *tcellterm.EventClosed:
		m.s.Clear()
		m.s.Fini()
		return true
	case *tcell.EventPaste:
		return m.term.HandleEvent(ev)
	case *tcell.EventMouse:
		// Translate the coordinates to our global coordinates (y-2)
		x, y := ev.Position()
		if y-2 < 0 {
			// Event is outside our view
			return false
		}
		e := tcell.NewEventMouse(x, y-2, ev.Buttons(), ev.Modifiers())
		return m.term.HandleEvent(e)
	case *tcellterm.EventMouseMode:
		m.s.EnableMouse(ev.Flags()...)
	}
	return false
}

// HandleEvent is used to handle events from underlying widgets. Any events
// which redraw must be executed in the main goroutine by posting the event back
// to tcell
func (m *model) HandleEvent(ev tcell.Event) bool {
	m.s.PostEvent(ev)
	return false
}

func main() {
	var err error
	f, _ := os.Create(".log")
	defer f.Close()
	logbuf := bytes.NewBuffer(nil)
	log.SetOutput(io.MultiWriter(f, logbuf))
	log.SetFlags(log.LstdFlags | log.Lshortfile)

	m := &model{}
	m.s, err = tcell.NewScreen()
	if err != nil {
		fmt.Fprintf(os.Stderr, "%v\n", err)
		os.Exit(1)
	}
	if err = m.s.Init(); err != nil {
		fmt.Fprintf(os.Stderr, "%v\n", err)
		os.Exit(1)
	}
	m.s.EnablePaste()

	m.title = views.NewTextBar()
	m.title.SetCenter(
		"Welcome to tcell-term",
		tcell.StyleDefault.Foreground(tcell.ColorBlue).
			Bold(true).
			Underline(true),
	)

	m.titleView = views.NewViewPort(m.s, 0, 0, -1, 2)
	m.title.Watch(m)
	m.title.SetView(m.titleView)

	m.termView = views.NewViewPort(m.s, 0, 2, -1, -1)
	m.term = tcellterm.New(tcellterm.WithRecorder("recording.log"))
	m.term.Watch(m)
	m.term.SetView(m.termView)

	cmd := exec.Command(os.Getenv("SHELL"))
	err = m.term.Start(cmd)
	if err != nil {
		panic(err)
	}
	for {
		ev := m.s.PollEvent()
		if ev == nil {
			break
		}
		m.Update(ev)
	}
	os.Stdout.Write(logbuf.Bytes())
}
