package nvd

import (
	"bytes"
	"compress/gzip"
	"encoding/xml"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"strconv"
	"time"

	"github.com/cheggaaa/pb"
	c "github.com/kotakanbe/go-cve-dictionary/config"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	"github.com/kotakanbe/go-cve-dictionary/util"
	"github.com/parnurzeal/gorequest"
)

// Nvd is array of Entry
type Nvd struct {
	Entries []Entry `xml:"entry"`
}

// Entry is Root Element
type Entry struct {
	CveID            string      `xml:"id,attr" json:"id"`
	PublishedDate    time.Time   `xml:"published-datetime"`
	LastModifiedDate time.Time   `xml:"last-modified-datetime"`
	Cvss             Cvss        `xml:"cvss>base_metrics" json:"cvss"`
	Products         []string    `xml:"vulnerable-software-list>product"` //CPE
	Summary          string      `xml:"summary"`
	References       []Reference `xml:"references"`
	Cwe              Cwe         `xml:"cwe"`
}

// Cvss is Cvss Score
type Cvss struct {
	Score                 string    `xml:"score"`
	AccessVector          string    `xml:"access-vector"`
	AccessComplexity      string    `xml:"access-complexity"`
	Authentication        string    `xml:"authentication"`
	ConfidentialityImpact string    `xml:"confidentiality-impact"`
	IntegrityImpact       string    `xml:"integrity-impact"`
	AvailabilityImpact    string    `xml:"availability-impact"`
	Source                string    `xml:"source"`
	GeneratedOnDate       time.Time `xml:"generated-on-datetime"`
}

// Cwe has Cwe ID
type Cwe struct {
	ID string `xml:"id,attr"`
}

// Reference is additional information about the CVE
type Reference struct {
	Type   string `xml:"reference_type,attr"`
	Source string `xml:"source"`
	Link   Link   `xml:"reference"`
}

// Link is additional information about the CVE
type Link struct {
	Value string `xml:",chardata" json:"value"`
	Href  string `xml:"href,attr" json:"href"`
}

// FetchFiles Fetch CVE vulnerability informatino from JVN
func FetchFiles(years []int) (entries []Entry, err error) {
	urls := makeFeedURLs(years)
	nvds, err := fetchFeedFileConcurrently(urls)
	if err != nil {
		return entries,
			fmt.Errorf("Failed to fetch cve data from NVD. err: %s", err)
	}
	for _, nvd := range nvds {
		entries = append(entries, nvd.Entries...)
	}
	return entries, nil
}

func makeFeedURLs(years []int) (urls []string) {
	//  http://static.nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-2016.xml.gz
	formatTemplate := "https://static.nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-%s.xml.gz"
	for _, year := range years {
		feed := ""
		if year == -8 {
			feed = "modified"
		} else {
			feed = strconv.Itoa(year)
		}
		urls = append(urls, fmt.Sprintf(formatTemplate, feed))
	}
	return
}

func fetchFeedFileConcurrently(urls []string) (nvds []Nvd, err error) {
	reqChan := make(chan string, len(urls))
	resChan := make(chan Nvd, len(urls))
	errChan := make(chan error, len(urls))
	defer close(reqChan)
	defer close(resChan)
	defer close(errChan)

	go func() {
		for _, url := range urls {
			reqChan <- url
		}
	}()

	concurrency := len(urls)
	tasks := util.GenWorkers(concurrency)
	for range urls {
		tasks <- func() {
			select {
			case url := <-reqChan:
				log.Infof("Fetching... %s", url)
				nvd, err := fetchFeedFile(url)
				if err != nil {
					errChan <- err
					return
				}
				resChan <- nvd
			}
		}
	}

	errs := []error{}
	bar := pb.New(len(urls))
	if c.Conf.Quiet {
		bar.Output = ioutil.Discard
	} else {
		bar.Output = os.Stderr
	}
	bar.Start()
	timeout := time.After(10 * 60 * time.Second)
	for range urls {
		select {
		case nvd := <-resChan:
			nvds = append(nvds, nvd)
		case err := <-errChan:
			errs = append(errs, err)
		case <-timeout:
			return nvds, fmt.Errorf("Timeout Fetching Nvd")
		}
		bar.Increment()
	}
	bar.Finish()
	//  bar.FinishPrint("Finished to fetch CVE information from JVN.")
	if 0 < len(errs) {
		return nvds, fmt.Errorf("%s", errs)
	}
	return nvds, nil
}

func fetchFeedFile(url string) (nvd Nvd, err error) {
	var body string
	var errs []error
	var resp *http.Response

	resp, body, errs = gorequest.New().Proxy(c.Conf.HTTPProxy).Get(url).End()
	//  defer resp.Body.Close()
	if len(errs) > 0 || resp == nil || resp.StatusCode != 200 {
		return nvd, fmt.Errorf(
			"HTTP error. errs: %v, url: %s", errs, url)
	}

	b := bytes.NewBufferString(body)
	reader, err := gzip.NewReader(b)
	defer reader.Close()
	if err != nil {
		return nvd, fmt.Errorf(
			"Failed to decompress NVD feedfile. url: %s, err: %s", url, err)
	}

	bytes, err := ioutil.ReadAll(reader)
	if err != nil {
		return nvd, fmt.Errorf(
			"Failed to Read NVD feedfile. url: %s, err: %s", url, err)
	}

	if err = xml.Unmarshal(bytes, &nvd); err != nil {
		return nvd, fmt.Errorf(
			"Failed to unmarshal. url: %s, err: %s", url, err)
	}
	return nvd, nil
}
