# go-cve-dictionary

This is tool to build a local copy of the NVD (National Vulnerabilities Database) [1]
and the Japanese JVN [2], which contain security vulnerabilities according to their
CVE identifiers [3] including exhaustive information and a risk score. The local
copy is generated in sqlite format, and the tool has a server mode for easy querying.

[1] https://en.wikipedia.org/wiki/National_Vulnerability_Database  
[2] https://en.wikipedia.org/wiki/Common_Vulnerabilities_and_Exposures  
[3] http://jvndb.jvn.jp/apis/termsofuse.html

## Installation

### Install requirements

go-cve-dictionary requires the following packages.

- SQLite3, MySQL, PostgreSQL or Redis
- git
- gcc
- go v1.7.1 or later
    - https://golang.org/doc/install

Here's an example for Amazon EC2 server.

```bash
$ ssh ec2-user@52.100.100.100  -i ~/.ssh/private.pem
$ sudo yum -y install sqlite git gcc
$ wget https://storage.googleapis.com/golang/go1.7.1.linux-amd64.tar.gz
$ sudo tar -C /usr/local -xzf go1.7.1.linux-amd64.tar.gz
$ mkdir $HOME/go
```

Put these lines into /etc/profile.d/goenv.sh

```bash
export GOPATH=$HOME/go
export PATH=$PATH:/usr/local/go/bin:$GOPATH/bin
```

Set the OS environment variable to current shell

```bash
$ source /etc/profile.d/goenv.sh
```

### Deploy go-cve-dictionary

To install:

```bash
$ mkdir -p $GOPATH/src/github.com/kotakanbe
$ cd $GOPATH/src/github.com/kotakanbe
$ git clone https://github.com/kotakanbe/go-cve-dictionary.git
$ cd go-cve-dictionary
$ make install
```

Create a log output directory.
You can use another directory on the command line option (-log-dir).

```bash
$ sudo mkdir /var/log/vuls
$ sudo chown ec2-user /var/log/vuls
$ sudo chmod 700 /var/log/vuls
```

Fetch vulnerability data from NVD.
It takes about 10 minutes (on AWS).

```bash
$ for i in `seq 2002 $(date +"%Y")`; do go-cve-dictionary fetchnvd -years $i; done
... snip ...
$ ls -alh cve.sqlite3
-rw-r--r-- 1 ec2-user ec2-user 7.0M Mar 24 13:20 cve.sqlite3
```

Now we have vulnerability data.
Start go-cve-dictionary as server mode.

```bash
$ go-cve-dictionary server
[Mar 24 15:21:55]  INFO Opening DB. datafile: /home/ec2-user/cve.sqlite3
[Mar 24 15:21:55]  INFO Migrating DB
[Mar 24 15:21:56]  INFO Starting HTTP Sever...
[Mar 24 15:21:56]  INFO Listening on 127.0.0.1:1323
```

### Update go-cve-dictionary

If the DB schema was changed, please specify new SQLite3, MySQL, PostgreSQL or Redis DB file.

```bash
$ cd $GOPATH/src/github.com/kotakanbe/go-cve-dictionary
$ git pull
$ rm -r vendor
$ make install
```

Binary files are created under $GOPATH/bin

----

## Sample data sources

### Hello HeartBleed

```bash
$ curl http://127.0.0.1:1323/cves/CVE-2014-0160 | jq "."
{
  "CveID": "CVE-2014-0160",
  "Nvd": {
    "Summary": "The (1) TLS and (2) DTLS implementations in OpenSSL 1.0.1 before 1.0.1g do not properly handle Heartbeat Extension packets, which allows remote attackers to obtain sensitive information from process memory via crafted packets that trigger a buffer over-read, as demonstrated by reading private keys, related to d1_both.c and t1_lib.c, aka the Heartbleed bug.",
    "Score": 5,
    "AccessVector": "NETWORK",
    "AccessComplexity": "LOW",
    "Authentication": "NONE",
    "ConfidentialityImpact": "PARTIAL",
    "IntegrityImpact": "NONE",
    "AvailabilityImpact": "NONE",
    "Cpes": null,
    "References": [
      {
        "Source": "CERT",
        "Link": "http://www.us-cert.gov/ncas/alerts/TA14-098A"
      },
      ...snip...
    ],
    "PublishedDate": "2014-04-07T18:55:03.893-04:00",
    "LastModifiedDate": "2015-10-22T10:19:38.453-04:00"
  },
  "Jvn": {
    "Title": "OpenSSL の heartbeat 拡張に情報漏えいの脆弱性",
    "Summary": "OpenSSL の heartbeat 拡張の実装には、情報漏えいの脆弱性が存在します。TLS や DTLS 通信において OpenSSL のコードを実行しているプロセスのメモリ内容が通信相手に漏えいする可能性があります。",
    "JvnLink": "http://jvndb.jvn.jp/ja/contents/2014/JVNDB-2014-001920.html",
    "JvnID": "JVNDB-2014-001920",
    "Score": 5,
    "Severity": "Medium",
    "Vector": "(AV:N/AC:L/Au:N/C:P/I:N/A:N)",
    "References": [
      {
        "Source": "AT-POLICE",
        "Link": "http://www.npa.go.jp/cyberpolice/detect/pdf/20140410.pdf"
      },
      ...snip...
    ],
    "Cpes": null,
    "PublishedDate": "2014-04-08T16:13:59+09:00",
    "LastModifiedDate": "2014-04-08T16:13:59+09:00"
  }
}

```

### Hello Ruby on Rails 4.0.2

```bash
$ curl -v -H "Accept: application/json" -H "Content-type: application/json" -X POST -d '{"name": "cpe:/a:rubyonrails:ruby_on_rails:4.0.2:-"}' http://localhost:1323/cpes | jq "."
[
  {
    "CveID": "CVE-2016-0751",
    "Nvd": {
      "CveDetailID": 345,
      "Summary": "actionpack/lib/action_dispatch/http/mime_type.rb in Action Pack in Ruby on Rails before 3.2.22.1, 4.0.x and 4.1.x before 4.1.14.1, 4.2.x before 4.2.5.1, and 5.x before 5.0.0.beta1.1 does not properly restrict use of the MIME type cache, which allows remote attackers to cause a denial of service (memory consumption) via a crafted HTTP Accept header.",
      "Score": 5,
      "AccessVector": "NETWORK",
      "AccessComplexity": "LOW",
      "Authentication": "NONE",
      "ConfidentialityImpact": "NONE",
      "IntegrityImpact": "NONE",
      "AvailabilityImpact": "PARTIAL",
      "Cpes": null,
      "References": [
        {
          "Source": "MLIST",
          "Link": "https://groups.google.com/forum/message/raw?msg=ruby-security-ann/9oLY_FCzvoc/5CDXbvpYEgAJ"
        },
        {
          "Source": "MLIST",
          "Link": "http://www.openwall.com/lists/oss-security/2016/01/25/9"
        }
      ],
      "PublishedDate": "2016-02-15T21:59:05.877-05:00",
      "LastModifiedDate": "2016-03-18T21:02:43.817-04:00"
    },
    "Jvn": {
      "Title": "",
      "Summary": "",
      "JvnLink": "",
      "JvnID": "",
      "Score": 0,
      "Severity": "",
      "Vector": "",
      "References": null,
      "Cpes": null,
      "PublishedDate": "0001-01-01T00:00:00Z",
      "LastModifiedDate": "0001-01-01T00:00:00Z"
    }
  },
  ... snip ...
]
```

----

## Usage

### Usage: List subcommands

```bash
$ go-cve-dictionary -help
Usage: go-cve-dictionary <flags> <subcommand> <subcommand args>

Subcommands:
        commands         list all command names
        flags            describe all known top-level flags
        help             describe subcommands and their syntax

Subcommands for fetchjvn:
        fetchjvn         Fetch Vulnerability dictionary from JVN

Subcommands for fetchnvd:
        fetchnvd         Fetch Vulnerability dictionary from NVD

Subcommands for server:
        server           Start CVE dictionary HTTP server


Use "go-cve-dictionary flags" for a list of top-level flags
```

go-cve-dictionary has four subcommands

- fetchnvd
  Fetch vulnerability data from NVD(English)
- fetchjvn
  Fetch vulnerability data from JVN(Japanese)
- server
  Start HTTP server

----

### Usage: Fetch NVD Data

```bash
$ go-cve-dictionary fetchnvd -help
fetchnvd:
        fetchnvd
                [-last2y]
                [-years] 2015 2016 ...
                [-dbtype=mysql|postgres|sqlite3|redis]
                [-dbpath=$PWD/cve.sqlite3 or connection string]
                [-http-proxy=http://192.168.0.1:8080]
                [-debug]
                [-debug-sql]
                [-quiet]
                [-log-dir=/path/to/log]

For the first time, run the blow command to fetch data for entire period. (It takes about 10 minutes)
   $ for i in `seq 2002 $(date +"%Y")`; do go-cve-dictionary fetchnvd -years $i ; done

  -dbpath string
        /path/to/sqlite3 or SQL connection string (default "$PWD/cve.sqlite3")
  -dbtype string
        Database type to store data in (sqlite3, mysql ,postgres or redis supported) (default "sqlite3")
  -debug
        debug mode
  -debug-sql
        SQL debug mode
  -http-proxy string
        http://proxy-url:port (default: empty)
  -last2y
        Refresh NVD data in the last two years.
  -quiet
        quiet mode (no output)
  -log-dir string
        /path/to/log (default "/var/log/vuls")
  -years
        Refresh NVD data of specific years.
```

- Fetch data for entire period.

    ```bash
    for i in `seq 2002 $(date +"%Y")`; do go-cve-dictionary fetchnvd -years $i; done
    ```

- Fetch data in the last two years

    ```bash
    $ go-cve-dictionary fetchnvd -last2y
    ```

- Fetch data of specific years

    ```bash
    $ go-cve-dictionary fetchnvd -years 2002 2003 2016
    ```

----

### Usage: Fetch JVN Data

```bash
$ go-cve-dictionary fetchjvn -h
fetchjvn:
        fetchjvn
                [-latest]
                [-last2y]
                [-years] 1998 1999 ...
                [-dbpath=$PWD/cve.sqlite3 or connection string]
                [-dbtype=mysql|postgres|sqlite3|redis]
                [-http-proxy=http://192.168.0.1:8080]
                [-debug]
                [-debug-sql]
                [-quiet]
                [-log-dir=/path/to/log]

  -dbpath string
        /path/to/sqlite3 or SQL connection string (default "$PWD/cve.sqlite3")
  -dbtype string
        Database type to store data in (sqlite3, mysql, postgres or redis supported) (default "sqlite3")
  -debug
        debug mode
  -debug-sql
        SQL debug mode
  -http-proxy string
        http://proxy-url:port (default: empty)
  -last2y
        Refresh JVN data in the last two years.
  -latest
        Refresh JVN data for latest.
  -quiet
        quiet mode (no output)
  -log-dir string
        /path/to/log (default "/var/log/vuls")
  -years
        Refresh JVN data of specific years.

```

- Fetch data for entire period

    ```bash
    for i in `seq 1998 $(date +"%Y")`; do go-cve-dictionary fetchjvn -years $i; done
    ```

- Fetch data in the last two years

    ```bash
    $ go-cve-dictionary fetchjvn -last2y
    ```

- Fetch data of specific years

    ```bash
    $ go-cve-dictionary fetchjvn -years 2002 2003 2016
    ```

- Fetch data for latest

    ```bash
    $ go-cve-dictionary fetchjvn -latest
    ```

----

### Usage: Run HTTP Server

```bash
$ go-cve-dictionary server -h
server:
        server
                [-bind=127.0.0.1]
                [-port=8000]
                [-dbpath=$PWD/cve.sqlite3 or connection string]
                [-dbtype=mysql|postgres|sqlite3|redis]
                [-debug]
                [-debug-sql]
                [-quiet]
                [-log-dir=/path/to/log]

  -bind string
        HTTP server bind to IP address (default: loop back interface) (default "127.0.0.1")
  -dbpath string
        /path/to/sqlite3 or SQL connection string (default : $PWD/cve.sqlite3)
  -dbtype string
        Database type to store data in (sqlite3, mysql, postgres or redis supported) (default "sqlite3")
  -debug
        debug mode (default: false)
  -debug-sql
        SQL debug mode (default: false)
  -quiet
        quiet mode (no output)
  -log-dir string
        /path/to/log (default "/var/log/vuls")
  -port string
        HTTP server port number (default: 1323) (default "1323")

```

----

### Usage: Use MySQL as a DB storage back-end

- fetchnvd

    ```bash
    $ go-cve-dictionary fetchnvd -last2y \
          -dbtype mysql \
          -dbpath "user:pass@tcp(localhost:3306)/dbname?parseTime=true"
    ```

- fetchjvn

    ```bash
    $ go-cve-dictionary fetchjvn -last2y \
          -dbtype mysql \
          -dbpath "user:pass@tcp(localhost:3306)/dbname?parseTime=true"
    ```

- server

    ```bash
    $ go-cve-dictionary server \
          -dbtype mysql \
          -dbpath "user:pass@tcp(localhost:3306)/dbname?parseTime=true"
    ```

### Usage: Use Postgres as a DB storage back-end

- fetchnvd

    ```bash
    $ go-cve-dictionary fetchnvd -last2y \
          -dbtype postgres \
          -dbpath "host=myhost user=user dbname=dbname sslmode=disable password=password"
    ```

- fetchjvn

    ```bash
    $ go-cve-dictionary fetchjvn -last2y \
          -dbtype postgres \
          -dbpath "host=myhost user=user dbname=dbname sslmode=disable password=password"
    ```

- server

    ```bash
    $ go-cve-dictionary server \
          -dbtype postgres \
          -dbpath "host=myhost user=user dbname=dbname sslmode=disable password=password"
    ```

### Usage: Use Redis as a DB storage back-end

- fetchnvd

    ```bash
    $ go-cve-dictionary fetchnvd -last2y \
          -dbtype redis \
          -dbpath "redis://localhost/0"
    ```

- fetchjvn

    ```bash
    $ go-cve-dictionary fetchjvn -last2y \
          -dbtype redis \
          -dbpath "redis://localhost/0"
    ```

- server

    ```bash
    $ go-cve-dictionary server \
          -dbtype redis \
          -dbpath "redis://localhost/0"
    ```

----

## Misc

- HTTP Proxy Support  
If your system at behind HTTP proxy, you have to specify -http-proxy option.

- How to daemonize go-cve-dictionary  
Use Systemd, Upstart or supervisord, daemontools...

- How to update vulnerability data automatically  
Use job scheduler like cron (with -last2y or -latest option).

- How to cross compile

    ```bash
    $ cd /path/to/your/local-git-repository/go-cve-dictionary
    $ GOOS=linux GOARCH=amd64 go build -o cvedict.amd64
    ```

- Logging  
go-cve-dictionary writes a log under -log-path specified directory (default is /var/log/vuls/).

- Debug  
Run with -debug, -sql-debug option.

----

## Data Source

- [NVD](https://nvd.nist.gov/)
- [JVN(Japanese)](http://jvndb.jvn.jp/apis/myjvn/)

----

## Authors

kotakanbe ([@kotakanbe](https://twitter.com/kotakanbe)) created go-cve-dictionary and [these fine people](https://github.com/future-architect/go-cve-dictionary/graphs/contributors) have contributed.

----

## How to Contribute

1. fork a repository: github.com/kotakanbe/go-cve-dictionary to github.com/you/repository
1. get original code: github.com/kotakanbe/go-cve-dictionary
1. work on original code
1. add remote to your repository: git remote add myfork https://github.com/you/repo.git
1. push your changes: git push myfork
1. create a new Pull Request

- see [GitHub and Go: forking, pull requests, and go-getting](http://blog.campoy.cat/2014/03/github-and-go-forking-pull-requests-and.html)

----

## Change Log

Please see [CHANGELOG.md](https://github.com/kotakanbe/go-cve-dictionary/blob/master/CHANGELOG.md).

----

## Licence

Please see [LICENSE](https://github.com/kotakanbe/go-cve-dictionary/blob/master/LICENSE).

----

## Additional License

- [NVD](https://nvd.nist.gov/faq)

> How can my organization use the NVD data within our own products and services?
> All NVD data is freely available from our XML Data Feeds. There are no fees, licensing restrictions, or even a requirement to register. All NIST publications are available in the public domain according to Title 17 of the United States Code. Acknowledgment of the NVD  when using our information is appreciated. In addition, please email nvd@nist.gov to let us know how the information is being used.

- [JVN](http://jvndb.jvn.jp/apis/termsofuse.html)
