#!/usr/bin/env python
# Copyright 2012 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street,
# Boston, MA 02110-1301, USA.
# 

from gnuradio import gr, gr_unittest
try: import pmt
except: from gruel import pmt
import digital_swig as digital

class qa_digital_carrier_allocator_cvc (gr_unittest.TestCase):

    def setUp (self):
        self.tb = gr.top_block ()

    def tearDown (self):
        self.tb = None

    def test_001_t (self):
        """
        pretty simple
        """
        fft_len = 6
        tx_symbols = (1, 2, 3)
        pilot_symbols = ((1j,),)
        occupied_carriers = ((0, 1, 2),)
        pilot_carriers = ((3,),)
        expected_result = (1, 2, 3, 1j, 0, 0)
        tag_name = "len"
        tag = gr.gr_tag_t()
        tag.offset = 0
        tag.key = pmt.pmt_string_to_symbol(tag_name)
        tag.value = pmt.pmt_from_long(len(tx_symbols))
        src = gr.vector_source_c(tx_symbols, False, 1, (tag,))
        alloc = digital.ofdm_carrier_allocator_cvc(fft_len,
                       occupied_carriers,
                       pilot_carriers,
                       pilot_symbols,
                       tag_name)
        sink = gr.vector_sink_c(fft_len)
        self.tb.connect(src, alloc, sink)
        self.tb.run ()
        self.assertEqual(sink.data(), expected_result)

    def test_002_t (self):
        """
        same, but using negative carrier indices
        """
        fft_len = 6
        tx_symbols = (1, 2, 3)
        pilot_symbols = ((1j,),)
        occupied_carriers = ((-1, 1, 2),)
        pilot_carriers = ((3,),)
        expected_result = (0, 2, 3, 1j, 0, 1)
        tag_name = "len"
        tag = gr.gr_tag_t()
        tag.offset = 0
        tag.key = pmt.pmt_string_to_symbol(tag_name)
        tag.value = pmt.pmt_from_long(len(tx_symbols))
        src = gr.vector_source_c(tx_symbols, False, 1, (tag,))
        alloc = digital.ofdm_carrier_allocator_cvc(fft_len,
                       occupied_carriers,
                       pilot_carriers,
                       pilot_symbols,
                       tag_name)
        sink = gr.vector_sink_c(fft_len)
        self.tb.connect(src, alloc, sink)
        self.tb.run ()
        self.assertEqual(sink.data(), expected_result)

    def test_003_t (self):
        """
        more advanced:
        - 6 symbols per carrier
        - 2 pilots per carrier
        - have enough data for nearly 3 OFDM symbols
        - send that twice
        - add some random tags
        """
        tx_symbols = range(1, 16); # 15 symbols
        pilot_symbols = ((1j, 2j), (3j, 4j))
        occupied_carriers = ((1, 3, 4, 11, 12, 14), (1, 2, 4, 11, 13, 14),)
        pilot_carriers = ((2, 13), (3, 12))
        expected_result = (0, 1,  1j,  2,  3, 0, 0, 0, 0, 0, 0, 4,  5,  2j, 6,  0,
                           0, 7,  8,  3j,  9, 0, 0, 0, 0, 0, 0, 10, 4j, 11, 12, 0,
                           0, 13, 1j, 14, 15, 0, 0, 0, 0, 0, 0, 0,  0,  2j, 0,  0)
        fft_len = 16
        tag_name = "len"
        tag1 = gr.gr_tag_t()
        tag1.offset = 0
        tag1.key = pmt.pmt_string_to_symbol(tag_name)
        tag1.value = pmt.pmt_from_long(len(tx_symbols))
        tag2 = gr.gr_tag_t()
        tag2.offset = len(tx_symbols)
        tag2.key = pmt.pmt_string_to_symbol(tag_name)
        tag2.value = pmt.pmt_from_long(len(tx_symbols))
        testtag1 = gr.gr_tag_t()
        testtag1.offset = 0
        testtag1.key = pmt.pmt_string_to_symbol('tag1')
        testtag1.value = pmt.pmt_from_long(0)
        testtag2 = gr.gr_tag_t()
        testtag2.offset = 7 # On the 2nd OFDM symbol
        testtag2.key = pmt.pmt_string_to_symbol('tag2')
        testtag2.value = pmt.pmt_from_long(0)
        testtag3 = gr.gr_tag_t()
        testtag3.offset = len(tx_symbols)+1 # First OFDM symbol of packet 2
        testtag3.key = pmt.pmt_string_to_symbol('tag3')
        testtag3.value = pmt.pmt_from_long(0)
        testtag4 = gr.gr_tag_t()
        testtag4.offset = 2*len(tx_symbols)-1 # Last OFDM symbol of packet 2
        testtag4.key = pmt.pmt_string_to_symbol('tag4')
        testtag4.value = pmt.pmt_from_long(0)
        src = gr.vector_source_c(tx_symbols * 2, False, 1, (tag1, tag2, testtag1, testtag2, testtag3, testtag4))
        alloc = digital.ofdm_carrier_allocator_cvc(fft_len,
                       occupied_carriers,
                       pilot_carriers,
                       pilot_symbols,
                       tag_name)
        sink = gr.vector_sink_c(fft_len)
        self.tb.connect(src, alloc, sink)
        self.tb.run ()
        self.assertEqual(sink.data(), expected_result * 2)
        tags_found = {'tag1': False, 'tag2': False, 'tag3': False, 'tag4': False}
        correct_offsets = {'tag1': 0, 'tag2': 1, 'tag3': 3, 'tag4': 5}
        for tag in sink.tags():
            key = pmt.pmt_symbol_to_string(tag.key)
            if key in tags_found.keys():
                tags_found[key] = True
                self.assertEqual(correct_offsets[key], tag.offset)
            if key == tag_name:
                self.assertTrue(tag.offset == 0 or tag.offset == 3)
                self.assertTrue(pmt.pmt_to_long(tag.value) == 3)
        self.assertTrue(all(tags_found.values()))


if __name__ == '__main__':
    gr_unittest.run(qa_digital_carrier_allocator_cvc, "qa_digital_carrier_allocator_cvc.xml")

