# Copyright (C) 2009, 2010, 2011  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


from datetime import date
from datetime import datetime
from os.path import abspath

from icalendar import Calendar

from timelinelib.db.exceptions import TimelineIOError
from timelinelib.db.objects import Event
from timelinelib.utilities.observer import Observable
from timelinelib.db.search import generic_event_search
from timelinelib.db.utils import IdCounter
from timelinelib.time.gregoriantime import GregorianTimeType
from timelinelib.utils import ex_msg
from timelinelib.calendar.gregorian import Gregorian
import timelinelib.calendar.gregorian as gregorian

class IcsTimeline(Observable):

    def __init__(self, path):
        Observable.__init__(self)
        self.path = path
        self.event_id_counter = IdCounter()
        self.cals = []
        self.import_timeline(self.path)

    def get_time_type(self):
        return GregorianTimeType()

    def is_read_only(self):
        return True

    def supported_event_data(self):
        return []

    def search(self, search_string):
        return generic_event_search(self._get_events(), search_string)

    def get_events(self, time_period):
        def decider(event):
            return event.inside_period(time_period)
        return self._get_events(decider)

    def get_all_events(self):
        def decider(event):
            return True
        return self._get_events(decider)

    def get_first_event(self):
        events = self._get_events()
        if events:
            return min(events, key=lambda x: x.time_period.start_time)
        else:
            return None

    def get_last_event(self):
        events = self._get_events()
        if events:
            return max(events, key=lambda x: x.time_period.end_time)
        else:
            return None

    def save_event(self, event):
        pass

    def delete_event(self, event_or_id):
        pass

    def get_categories(self):
        return []

    def save_category(self, category):
        pass

    def delete_category(self, category_or_id):
        pass

    def load_view_properties(self, view_properties):
        pass

    def save_view_properties(self, view_properties):
        pass

    def find_event_with_id(self, id):
        events = self._get_events()
        for e in events:
            if e.id == id:
                return e
        return None

    def _get_events(self, decider_fn=None):
        self.events = []
        for cal in self.cals:
            for event in cal.walk("VEVENT"):
                start, end = extract_start_end(event)
                txt = ""
                if event.has_key("summary"):
                    txt = event["summary"]
                elif event.has_key("description"):
                    txt = event["description"]
                else:
                    txt == "Unknown"
                e = Event(self.get_time_type(), start, end, txt)
                e.set_id(event["timeline_id"])
                if event.has_key("description"):
                    e.set_data("description", event["description"])
                if decider_fn is None or decider_fn(e):
                    self.events.append(e)
        return self.events

    def import_timeline(self, path):
        try:
            ics_file = open(path, "rb")
            try:
                file_contents = ics_file.read()
                try:
                    cal = Calendar.from_ical(file_contents)
                    for event in cal.walk("VEVENT"):
                        event["timeline_id"] = self.event_id_counter.get_next()
                    self.cals.append(cal)
                except Exception, pe:
                    msg1 = _("Unable to read timeline data from '%s'.")
                    msg2 = "\n\n" + ex_msg(pe)
                    raise TimelineIOError((msg1 % abspath(path)) + msg2)
            finally:
                ics_file.close()
        except IOError, e:
            msg = _("Unable to read from file '%s'.")
            whole_msg = (msg + "\n\n%s") % (abspath(self.path), e)
            raise TimelineIOError(whole_msg)


def extract_start_end(vevent):
    start = convert_to_datetime(vevent.decoded("dtstart"))
    if vevent.has_key("dtend"):
        end = convert_to_datetime(vevent.decoded("dtend"))
    elif vevent.has_key("duration"):
        end = start + vevent.decoded("duration")
    else:
        end = convert_to_datetime(vevent.decoded("dtstart"))
    return (start, end)


def convert_to_datetime(d):
    if isinstance(d, datetime):
        return Gregorian(d.year, d.month, d.day, d.hour, d.minute, d.second).to_time()
    elif isinstance(d, date):
        return gregorian.from_date(d.year, d.month, d.day).to_time()
    else:
        raise TimelineIOError("Unknown date.")
