/*
**  Utilities.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "Utilities.h"

#include "EditWindowController.h"
#include "ExtendedAttachmentCell.h"
#include "ExtendedTextView.h"
#include "FolderNode.h"
#include "FolderNodePopUpItem.h"
#include "GNUMail.h"
#include "GNUMail/GNUMailBundle.h"
#include "Constants.h"
#include "MailboxManagerController.h"
#include "MailHeaderCell.h"
#include "MailWindowController.h"
#include "MessageViewWindowController.h"
#include "MimeType.h"
#include "MimeTypeManager.h"
#include "NSAttributedString+TextEnriched.h"
#include "NSUserDefaults+Extensions.h"
#include "PasswordPanelController.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Flags.h>
#include <Pantomime/Folder.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeMultipart.h>
#include <Pantomime/MimeUtility.h>
#include <Pantomime/NSRegEx.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/Part.h>
#include <Pantomime/Store.h>
#include <Pantomime/URLName.h>

#include <limits.h>


// Our static vars
static NSMutableDictionary *passwordCache = nil;

//
// Our dictionary of colors for quoting and functions to work with them.
//
static NSMutableArray *quoteLevelColors = nil;

#define MAX_LEVEL 4

static NSColor* colorForLevel(int theLevel)
{  
  if ( !quoteLevelColors )
    {
      quoteLevelColors = [[NSMutableArray alloc] initWithCapacity: MAX_LEVEL];
    }
  
  if ( [quoteLevelColors count] == 0 )
    {
      NSUserDefaults *aUserDefaults;
      NSColor *aColor;

      aUserDefaults = [NSUserDefaults standardUserDefaults];
      
      // We look at the user preferences to get the colors...
      aColor = [aUserDefaults colorForKey: @"QUOTE_COLOR_LEVEL_1"];
      if ( aColor )
	{
	  [quoteLevelColors addObject: aColor];
	}	
      else
	{
	  [quoteLevelColors addObject: [NSColor blueColor]];
	}
      
      aColor = [aUserDefaults colorForKey: @"QUOTE_COLOR_LEVEL_2"];
      if ( aColor )
	{
	  [quoteLevelColors addObject: aColor];
	}	
      else
	{
	  [quoteLevelColors addObject: [NSColor redColor]];
	}
      
      aColor = [aUserDefaults colorForKey: @"QUOTE_COLOR_LEVEL_3"];
      if ( aColor )
	{
	  [quoteLevelColors addObject: aColor];
	}	
      else
	{
	  [quoteLevelColors addObject: [NSColor greenColor]];
	}
      
      aColor = [aUserDefaults colorForKey: @"QUOTE_COLOR_LEVEL_4"];
      if ( aColor )
	{
	  [quoteLevelColors addObject: aColor];
	}
      else
	{
	  [quoteLevelColors addObject: [NSColor cyanColor]];
	}
    }
  
  return [quoteLevelColors objectAtIndex: theLevel];
}


//
//
//
static int levelFromString(NSString *theString, int start, int end)
{
  int i, level;
  unichar c;
  
  for (i = start,level = 0; i < end; i++)
    {
      c = [theString characterAtIndex: i];

      if (c == '>')
	{
	  level++;
	}
      else if (c > 32)
	{
	  break;
	}
    }

  return level;
}

//
// Useful macros
//
#define INITIALIZE_MESSAGE(message) ({ \
 if ( ![message isInitialized] ) \
   { \
     [message setInitialized: YES]; \
     [message setProperty: [AUTORELEASE([[NSDate alloc] init]) addTimeInterval: RETAIN_PERIOD]  forKey: @"EXPIRE_DATE"]; \
   } \
})


//
//
//
@implementation Utilities

+ (void) initialize
{
  if ( !passwordCache )
    {
      passwordCache = [[NSMutableDictionary alloc] init];
    }
}


//
//
//
+ (NSFont *) fontFromFamilyName: (NSString *) theName
			  trait: (int) theTrait
			   size: (int) theSize
{
  NSArray *allFontNames;
  NSString *aFontName;
  NSFont *aFont;
  int i;
  
  allFontNames = [[NSFontManager sharedFontManager] availableMembersOfFontFamily: theName];
  aFontName = nil;

  if ( theName )
    {
      for (i = 0; i < [allFontNames count]; i++)
	{
	  NSArray *attributes;
	  
	  attributes = [allFontNames objectAtIndex: i];
	  
	  // We verify if the font name has the trait we are looking for
	  if ( [[attributes objectAtIndex: 3] intValue] == theTrait )
	    {
	      aFontName = [attributes objectAtIndex: 0];
	      break;
	    }
	}
    }

  if ( aFontName )
    {
      aFont = [NSFont fontWithName: aFontName
		      size: theSize];
    }
  else
    {
      switch ( theTrait )
	{
	case NSFixedPitchFontMask:
	  aFont = [NSFont userFixedPitchFontOfSize: theSize];
	  break;

	case NSBoldFontMask:
	  aFont = [NSFont boldSystemFontOfSize: theSize];
	  break;
	  
	case NSUnboldFontMask:
	default:
	  aFont = [NSFont systemFontOfSize: theSize];
	  break;
	}
    }

  return aFont;
}


//
//
//
+ (void) appendAddress: (NSArray *) theAddress
	   toTextField: (NSTextField *) theTextField
{
  NSString *aString;
  NSRange aRange;

  if ([theAddress objectAtIndex: 0] && [[theAddress objectAtIndex: 0] length])
    {
      aString = [NSString stringWithFormat: @"%@ <%@>", [theAddress objectAtIndex: 0], [theAddress objectAtIndex: 1]];
    }
  else
    {
      aString = [theAddress objectAtIndex: 1];
    }

  aRange = [[theTextField stringValue] rangeOfString: aString
				       options: NSCaseInsensitiveSearch];
  
  if ( aRange.location != NSNotFound )
    {
      return; 
    }

  if ( [[theTextField stringValue] length] )
    {
      [theTextField setStringValue: [NSString stringWithFormat: @"%@, %@",
					      [theTextField stringValue],
					      aString]];
    }
  else
    {
      [theTextField setStringValue: aString];
    }
}


//
// This method returns a NSAttributedString object that has been built
// from the content of the message.
//
+ (NSAttributedString *) attributedStringFromContentForPart: (Part *) thePart
{
  NSMutableAttributedString *maStr;
  NSMutableDictionary *tAttr;
  
  tAttr = [[NSMutableDictionary alloc] init];
  
  [tAttr setObject: [Utilities fontFromFamilyName: [[NSUserDefaults standardUserDefaults] 
						     objectForKey: @"MESSAGE_FONT_NAME"]
			       trait: NSUnboldFontMask
			       size: [[NSUserDefaults standardUserDefaults] floatForKey: @"MESSAGE_FONT_SIZE"]]
	 forKey: NSFontAttributeName];
  
  maStr = [[NSMutableAttributedString alloc] init];
  
  if ( [[thePart content] isKindOfClass: [MimeMultipart class]] )
    {
      // We first verify if our multipart object is a multipart alternative.
      // If yes, we represent the best representation of the part.
      if ( [thePart isMimeType: @"multipart" : @"alternative"] )
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
	  
	  // We then append the best representation from this multipart/alternative object
	  [maStr appendAttributedString: [Utilities bestRepresentationFromMultipartAlternative:
						      (MimeMultipart *)[thePart content]] ];
	}
      // Then we verify if our multipart object is a multipart appledouble.
      else if ( [thePart isMimeType: @"multipart" : @"appledouble"] )
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
	  
	  // We then append the best representation from this multipart/appledouble object
	  [maStr appendAttributedString: [Utilities bestRepresentationFromMultipartAppleDouble:
						      (MimeMultipart *)[thePart content]] ];
	}
      // We have a multipart/mixed or multipart/related (or an other, unknown multipart/* object)
      else
	{
	  MimeMultipart *aMimeMultipart;
	  Part *aPart;
	  int i;
      
	  aMimeMultipart = (MimeMultipart*)[thePart content];
      
	  for (i = 0; i < [aMimeMultipart count]; i++)
	    {
	      // We get our part
	      aPart = [aMimeMultipart bodyPartAtIndex: i];
	      
	      // We recursively call our method to show all parts
	      [maStr appendAttributedString: [Utilities attributedStringFromContentForPart: aPart]];
	    }
	}
    }
  // We have a message with Content-Type: application/* OR audio/* OR image/* OR video/*
  // We can also have a text/* part that was base64 encoded, but, we skip it. It'll be
  // treated as a NSString object. (See below).
  else if ( [[thePart content] isKindOfClass: [NSData class]] )
    {
      NSTextAttachment *aTextAttachment;
      ExtendedAttachmentCell *cell;
      NSFileWrapper *aFileWrapper;
      
      MimeType *aMimeType;
      NSImage *anImage;
      
      NSRect rectOfTextView;
      NSSize imageSize;
      
      
      aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: (NSData *)[thePart content]];
      
      if ( ![thePart filename] )
	{
	  [aFileWrapper setPreferredFilename: @"unknown"];
	}
      else
	{
	  [aFileWrapper setPreferredFilename: [thePart filename]];
	}

      // We get the righ Mime-Type object for this part
      aMimeType = [[MimeTypeManager singleInstance] 
		    mimeTypeForFileExtension: [[aFileWrapper preferredFilename] pathExtension]];
      
      if ( aMimeType && [aMimeType view] == DISPLAY_AS_ICON )
	{
	  anImage = [[MimeTypeManager singleInstance] bestIconForMimeType: aMimeType
						      pathExtension: [[aFileWrapper preferredFilename] pathExtension]];
	}
      else
	{
	  // If the image is a TIFF, we create it directly from a NSData
	  if ( [thePart isMimeType: @"image": @"tiff"] )
	    {
	      anImage = [[NSImage alloc] initWithData:(NSData *)[thePart content]];
	      AUTORELEASE(anImage);
	    }
	  // If not, we temporary write it to a file and we init everything from
	  // the content of the file. Libwraster will be used here to load the image
	  // from the file. We simply remove the file when we're done.
	  //
	  // NOTE: Currently, GNUstep doesn't allow creating images other than TIFF from a NSData object.
	  //       This is a limitation that will be fixed in the future.
	  //       We currently write the data object to a file and initialize an image from
	  //       the content of this file. Libwraster's functions will be used by the back (art or x11)
	  //       backend to decode formats other than TIFF.
	  //       - Ludo
	  else if ( [thePart isMimeType: @"image": @"*"] || 
		    (aMimeType && [aMimeType view] == DISPLAY_IF_POSSIBLE) )
	    {
	      NSString *aString;
	      
	      aString = [NSString stringWithFormat:@"%@/%d_%@", GNUMailTemporaryDirectory(), 
				  [[NSProcessInfo processInfo] processIdentifier],
				  [[thePart filename] lowercaseString]];
	      
	      [(NSData *)[thePart content] writeToFile: aString
			 atomically: YES];
	      
	      anImage = [[NSImage alloc] initWithContentsOfFile: aString];
	      
	      // The "display if possible", failed. We at least try to get the right icon.
	      if ( !anImage )
		{
		  anImage = [[MimeTypeManager singleInstance] bestIconForMimeType: aMimeType
							      pathExtension: [[aFileWrapper preferredFilename] 
									       pathExtension]];
		}
	      else
		{
		  AUTORELEASE(anImage);
		}
	      
	      [[NSFileManager defaultManager] removeFileAtPath: aString
					      handler: nil];
	    }
	  //
	  // It's not an image, we query our MIME type manager in order to see 
	  // to get the right icon.
	  //
	  else
	    {
	      anImage = [[MimeTypeManager singleInstance] bestIconForMimeType: aMimeType
							  pathExtension: [[aFileWrapper preferredFilename] 
									   pathExtension]];
	    }
	}
      
      // If the image has been loaded sucessfully, it become our icon for our filewrapper
      if ( anImage )
	{
	  [aFileWrapper setIcon: anImage];
	} 	
      
      // We now rescale the attachment if it doesn't fit in the text view. That could happen
      // very often for images.
      rectOfTextView = [[[[GNUMail lastMailWindowOnTop] windowController] textView] frame];
      imageSize = [[aFileWrapper icon] size];
      
      if ( imageSize.width > rectOfTextView.size.width )
	{
	  double delta =  1.0 / ( imageSize.width / rectOfTextView.size.width );
	  double dy = 15*delta;

	  [[aFileWrapper icon] setScalesWhenResized: YES];
	  [[aFileWrapper icon] setSize: NSMakeSize(((imageSize.width-15) * delta), (imageSize.height-dy) * delta)];
	}
      
      // We now create our text attachment with our file wrapper
      aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
      
      // We add this attachment to our 'Save Attachment' menu
      [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
        
      cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
					     size: [(NSData *)[thePart content] length] ];
      [cell setPart: thePart];
      
      [aTextAttachment setAttachmentCell: cell];
      
      // Cocoa bug
#ifdef MACOSX
      [cell setAttachment: aTextAttachment];
      [cell setImage: [aFileWrapper icon]];
#endif
      RELEASE(cell);
      RELEASE(aFileWrapper);
      
      // We separate the text attachment from any other line previously shown
      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						attributes: nil] ];
      
      [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment: aTextAttachment]];
      RELEASE(aTextAttachment);
    }
  //
  // We have a message/rfc822 as the Content-Type.
  //
  else if ( [[thePart content] isKindOfClass: [Message class]] )
    {
      Message *aMessage;
      
      aMessage = (Message *)[thePart content];
            
      // We must represent this message/rfc822 part as an attachment
      if ( [thePart contentDisposition] &&
	   ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame) )
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;
	  
	  aData = [aMessage rawSource];
	  
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  [aFileWrapper setPreferredFilename: @"message/rfc822 attachment"];
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
          
	  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						 size: [aData length] ];
	  [cell setPart: thePart];
	  
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell); 
	  RELEASE(aFileWrapper);
	  
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // Its inline..
      else
	{	      
	  [maStr appendAttributedString: [Utilities attributedStringFromHeadersForMessage: aMessage
						    showAllHeaders: NO
						    useMailHeaderCell: NO] ];
	  [maStr appendAttributedString: [Utilities attributedStringFromContentForPart: aMessage]];
	}
    }
  //
  // We have a message with Content-Type: text/*
  // or at least, a part that we can represent directly in our textView.
  // 
  else if ( [[thePart content] isKindOfClass: [NSString class]] )
    { 
      NSString *aString;
      
      aString = (NSString *)[thePart content];

      // We must represent our NSString as an text attachment in our textView.
      if ( [thePart contentDisposition] &&
	   ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame) )
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;
	  
	  aData = [aString dataUsingEncoding: NSUTF8StringEncoding];
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  if ( ![thePart filename] )
	    {
	      [aFileWrapper setPreferredFilename: @"unknown"];
	    }
	  else
	    {
	      [aFileWrapper setPreferredFilename: [thePart filename]];
	    }
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];

	  cell = [[ExtendedAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						 size: [aData length] ];
	  [cell setPart: thePart];
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell);
	  RELEASE(aFileWrapper);
	  
	  // We separate the text attachment from any other line previously shown
	  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n" 
						    attributes: nil] ];
		      
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // It's inline...
      else
	{
	  [maStr appendAttributedString: [Utilities _attributedStringFromTextForPart: thePart]];
	}
    }
  //
  // We have something that we probably can't display.
  // Let's inform the user about this situation.
  // 
  else
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"A part of this E-Mail can't be represented. Please report this as a bug."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
    }
  
  RELEASE(tAttr);

  return AUTORELEASE(maStr);
}


//
// This method returns a NSAttributedString object that has been built
// from the headers of the message.
//
+ (NSAttributedString *) attributedStringFromHeadersForMessage: (Message *) theMessage
						showAllHeaders: (BOOL) showAllHeaders
					     useMailHeaderCell: (BOOL) useMailHeaderCell
{
  NSMutableDictionary *headerNameAttribute, *headerValueAttribute;
  NSMutableAttributedString *maStr; 

  NSMutableAttributedString *aMutableAttributedString;

  NSDictionary *allHeaders;
  NSArray *headersToShow;
  int i;

  maStr = [[NSMutableAttributedString alloc] init];
  

  // Attributes for our header names
  headerNameAttribute = [[NSMutableDictionary alloc] init];

  [headerNameAttribute setObject: [Utilities fontFromFamilyName: [[NSUserDefaults standardUserDefaults] 
								   objectForKey: @"HEADER_NAME_FONT_NAME"]
					     trait: NSBoldFontMask
					     size: [[NSUserDefaults standardUserDefaults] 
						     floatForKey: @"HEADER_NAME_FONT_SIZE"]]
		       forKey: NSFontAttributeName];


  // Attributes for our header values
  headerValueAttribute = [[NSMutableDictionary alloc] init];

  [headerValueAttribute setObject: [Utilities fontFromFamilyName: [[NSUserDefaults standardUserDefaults]
								    objectForKey: @"HEADER_VALUE_FONT_NAME"]
					      trait: NSUnboldFontMask
					      size: [[NSUserDefaults standardUserDefaults] 
						      floatForKey: @"HEADER_VALUE_FONT_SIZE"]]
			forKey: NSFontAttributeName];
			
  
  // We get all the message's headers
  allHeaders = [theMessage allHeaders];
  
  // We verify which headers of the message we show show.
  if ( showAllHeaders )
    {
      headersToShow = [allHeaders allKeys];
    }
  else
    {
      headersToShow = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNHEADERS"];
    }

  if ( headersToShow )
    {  
      for (i = 0; i < [headersToShow count]; i++)
	{
	  NSString *anHeader = [headersToShow objectAtIndex: i];
	  
	  if ( [anHeader caseInsensitiveCompare: @"Date"] == NSOrderedSame &&
	       [theMessage receivedDate] )
	    {
	      if ( [theMessage receivedDate] )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Date: "
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage receivedDate] description]
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"From"] == NSOrderedSame &&
		    [theMessage from] )
	    {
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"From: "
							attributes: headerNameAttribute] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage from] unicodeStringValue]
							attributes: headerValueAttribute] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							attributes: nil] ];
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Bcc"] == NSOrderedSame )
	    {
	      NSString *bccStr = [MimeUtility stringFromRecipients: [theMessage recipients]
					      type: BCC];
	      
	      if ( [bccStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Bcc: "
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [bccStr substringToIndex: ([bccStr length] - 2)]
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Cc"] == NSOrderedSame )
	    {
	      NSString *ccStr= [MimeUtility stringFromRecipients: [theMessage recipients]
					    type: CC];
	      
	      if ( [ccStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Cc: "
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [ccStr substringToIndex: ([ccStr length] - 2)]
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Reply-To"] == NSOrderedSame )
	    { 
	      InternetAddress *anInternetAddress = [theMessage replyTo];
	      
	      if ( anInternetAddress )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"Reply-To: "
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [[theMessage replyTo] unicodeStringValue]
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}	
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"To"] == NSOrderedSame )
	    {
	      NSString *toStr = [MimeUtility stringFromRecipients: [theMessage recipients]
					     type: TO];
	      
	      if ( [toStr length] > 0 )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"To: "
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [toStr substringToIndex: ([toStr length] - 2)]
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	  else if ( [anHeader caseInsensitiveCompare: @"Content-Type"] == NSOrderedSame )
	    {
	      NSString *aString;
	      
	      if ( [theMessage charset] )
		{
		  aString = [NSString stringWithFormat: @"%@; charset=%@",
				      [theMessage contentType], [theMessage charset]];
		}
	      else
		{
		  aString = [theMessage contentType];
		}
	      
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"Content-Type: "
							attributes: headerNameAttribute] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: aString
							attributes: headerValueAttribute] ];
	      [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							attributes: nil] ];
	    }
	  else
	    {
	      NSArray *allKeys = [allHeaders allKeys];
	      NSString *valueString = nil;
	      int j, count;
	      
	      count = [allKeys count];
	      
	      for (j = 0; j < count; j++)
		{
		  if ( [[[allKeys objectAtIndex: j] uppercaseString] isEqualToString: [anHeader uppercaseString]] )
		    {
		      if ( [[allHeaders objectForKey: [allKeys objectAtIndex: j]] 
			     isKindOfClass: [InternetAddress class]] )
			{
			  valueString = [[allHeaders objectForKey: [allKeys objectAtIndex: j]] unicodeStringValue];
			}
		      else
			{
			  valueString = [allHeaders objectForKey: [allKeys objectAtIndex: j]];
			}
		      break;
		    }
		}
		  
	      if ( valueString )
		{
		  [maStr appendAttributedString: [Utilities attributedStringWithString: [NSString stringWithFormat: @"%@: ", anHeader]
							    attributes: headerNameAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: valueString
							    attributes: headerValueAttribute] ];
		  [maStr appendAttributedString: [Utilities attributedStringWithString: @"\n"
							    attributes: nil] ];
		}
	    }
	} // for (..) 
      
    } // if ( headersToShow )
  
  if ( useMailHeaderCell )
    {
      NSTextAttachment *aTextAttachment;
      MailHeaderCell *theMailHeaderCell;
      
      theMailHeaderCell = [[[GNUMail lastMailWindowOnTop] windowController] mailHeaderCell];
      [theMailHeaderCell setAttributedStringValue: maStr];
      [theMailHeaderCell resize: nil];
      
      // We now "embed" the header cell into a NSTextAttachment object
      // so we can add it to our mutable attributed string.
      aTextAttachment = [[NSTextAttachment alloc] init];
      [aTextAttachment setAttachmentCell: theMailHeaderCell];
      
      aMutableAttributedString = [[NSMutableAttributedString alloc] init];
      
      [aMutableAttributedString appendAttributedString: 
				  [NSMutableAttributedString attributedStringWithAttachment: aTextAttachment]];
      RELEASE(aTextAttachment);

#ifdef MACOSX      
      [aMutableAttributedString appendAttributedString: [Utilities attributedStringWithString: @"\n\n"
#else
      [aMutableAttributedString appendAttributedString: [Utilities attributedStringWithString: @"\n"
#endif

								   attributes: nil]];
    }
  else
    {
      aMutableAttributedString = [[NSMutableAttributedString alloc] initWithString: @"\n"];
      [aMutableAttributedString appendAttributedString: maStr];
      [aMutableAttributedString appendAttributedString: [Utilities attributedStringWithString: @"\n"
								   attributes: nil] ];
    }
  
  RELEASE(maStr);
  RELEASE(headerNameAttribute);
  RELEASE(headerValueAttribute);

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSAttributedString *) attributedStringWithString: (NSString *) theString
					 attributes: (NSDictionary *) theAttributes
{
  if ( !theAttributes )
    {
      NSMutableDictionary *aMutableAttributedString;
      NSMutableDictionary *attributes;

      attributes = [[NSMutableDictionary alloc] init];
      [attributes setObject: [NSFont systemFontOfSize: 0]
		  forKey: NSFontAttributeName];
      
      aMutableAttributedString = [[NSAttributedString alloc] initWithString: theString
							     attributes: attributes];
      RELEASE(attributes);
      
      return AUTORELEASE( aMutableAttributedString );
    }
  else 
    {
      return AUTORELEASE( [[NSAttributedString alloc] initWithString: theString
						      attributes: theAttributes] );
    }
}


//
//
//
+ (NSAttributedString *) bestRepresentationFromMultipartAlternative: (MimeMultipart *) theMimeMultipart
{
  Part *aPart;
  NSUserDefaults *aUserDefaults;
  NSString *aSubtype;
  int i, index;

  // We get the preferred subtype
  aUserDefaults = [NSUserDefaults standardUserDefaults];

  if ( [aUserDefaults objectForKey: @"DEFAULT_MULTIPART_ALTERNATIVE_TYPE"] &&
       [aUserDefaults integerForKey: @"DEFAULT_MULTIPART_ALTERNATIVE_TYPE"] == TYPE_HTML )
    {
      aSubtype = @"html";
    }
  else
    {
      aSubtype = @"plain";
    }

  index = -1;

  // We search for our preferred part (text/html or text/plain) depending on aSubtype
  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aPart = [theMimeMultipart bodyPartAtIndex: i];
      
      if ( [aPart isMimeType: @"text": aSubtype] )
	{
	  index = i;
	  break;
	}
    }

  // If we have found our preferred part, we use that one
  if ( index >= 0 )
    {
      return [Utilities _attributedStringFromTextForPart: [theMimeMultipart bodyPartAtIndex: index]];
    }
  // We haven't, we show the first one!
  else
    {
      if ( [theMimeMultipart count] > 0 )
	{
	  return [Utilities _attributedStringFromTextForPart: [theMimeMultipart bodyPartAtIndex: 0]];
	}
    }
  
  return [Utilities _attributedStringFromTextForPart: nil];
}


//
//
//
+ (NSAttributedString *) bestRepresentationFromMultipartAppleDouble: (MimeMultipart *) theMimeMultipart
{
  NSMutableAttributedString *aMutableAttributedString;
  NSMutableDictionary *attributes;
  
  Part *aPart;
  int i;
  
  // We create a set of attributes (base font, color red)
  attributes = [[NSMutableDictionary alloc] init];
  [attributes setObject: [NSColor redColor]
	      forKey: NSForegroundColorAttributeName];

  aMutableAttributedString = [[NSMutableAttributedString alloc] init];

  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aPart = [theMimeMultipart bodyPartAtIndex: i];
    
      if ( [aPart isMimeType: @"application": @"applefile"] )
	{
	  [aMutableAttributedString appendAttributedString:
				      [Utilities attributedStringWithString: _(@"(Decoded Apple file follows...)")
						 attributes: attributes]];
	}
      else
	{
	  // We first add a \n between our applefile description and the 'representaiton' of
	  // the attachment
	  [aMutableAttributedString appendAttributedString:
				      [Utilities attributedStringWithString: @"\n" attributes: nil]];
	  
	  // We add the representation of our attachment
	  [aMutableAttributedString appendAttributedString:
				      [Utilities attributedStringFromContentForPart: aPart]];
	} 
      
    }
  
  // We add a \n to separate everything.
  [aMutableAttributedString appendAttributedString:
			      [Utilities attributedStringWithString: @"\n" attributes: nil]];

  RELEASE(attributes);

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSAttributedString *) formattedAttributedStringFromAttributedString: (NSAttributedString *) theAttributedString
{
  NSMutableAttributedString *aMutableAttributedString;
  NSRange aRange;
  
  id attribute;
  int index;
  
  // If we only have one char (or less), we don't do anything.
  // This could append for example when we only have a image/tiff as the Content-Type of an E-Mail.
  if ( [theAttributedString length] <= 1 )
    {
      return theAttributedString;
    }
  
  aMutableAttributedString = nil;
  index = 0;

  attribute = [theAttributedString attribute: NSAttachmentAttributeName
				   atIndex: index
				   effectiveRange: &aRange];
  
  // We if have NO text attachments at all... we just return our param
  if ( !attribute && 
       (aRange.length == [theAttributedString length]) )
    {
      return theAttributedString;
    }
  
  
  // We initialize our mutable string
  aMutableAttributedString = [[NSMutableAttributedString alloc] initWithAttributedString: theAttributedString];
 
  while ( index < [aMutableAttributedString length] )
    {
      // We found a text attachment...
      if ( attribute )
	{
	  // We search for <<filename>> OR <filename> and if we got one of them,
	  // we simply replace that string with the actual attachment.
	  ExtendedAttachmentCell *cell;
	  
	  cell = [attribute attachmentCell];

	  if ( [cell respondsToSelector: @selector(part)] )
	    {
	      NSString *aString;
	      NSRange r;
	      aString = [aMutableAttributedString string];
	      
	      // We first search for <<filename>>
	      r = [aString rangeOfString: [NSString stringWithFormat: @"<<%@>>", [[cell part] filename]]];
	      
	      // If not found, we search for <filename>
	      if ( r.length == 0 )
		{
		  r = [aString rangeOfString: [NSString stringWithFormat: @"<%@>", [[cell part] filename]]];
		}
	      
	      // We found something, let's replace <filename> (or <<filename>>) by the actual
	      // attachment cell.
	      if ( r.length )
		{
		  NSAttributedString *attachment;
		  
		  // We get the attributed that contains our attachment
		  attachment = [aMutableAttributedString attributedSubstringFromRange: aRange];
		  RETAIN(attachment);
	      
		  // FIXME - we should verify if the range of that attachment > of the range of
		  // the <filename in case the disposition of our E-Mail is 'wrong'.
		  
		  // We remove it from our string
		  [aMutableAttributedString deleteCharactersInRange: aRange];
		  
		  // We replace
		  [aMutableAttributedString replaceCharactersInRange: r
					    withAttributedString: attachment];
		  
		  RELEASE(attachment);
		  
		  // We adjust our index for the search
		  index = r.location + 1;
		}
	      else
		{
		  index = NSMaxRange(r);
		}
	    }
	  else
	    {
	      index = NSMaxRange(aRange);
	    }
	}
      else // of if ( attribute ) 
	{
	  index = NSMaxRange(aRange);
	}
      
      if (index >= [aMutableAttributedString length])
      	{
      	  break;
      	}

      attribute = [aMutableAttributedString attribute: NSAttachmentAttributeName
					    atIndex: index
					    effectiveRange: &aRange];
    }

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSString *) encryptPassword: (NSString *) thePassword
                       withKey: (NSString *) theKey
{
  NSMutableString *key;
  NSMutableData *encryptedPassword;
  NSString *result;
  int i;
  unichar p, k, e;

  // The length of the key must be greater (or equal) than
  // the length of the password
  key = [[NSMutableString alloc] init];
  
  while ( [key length] < [thePassword length] )
    {
      [key appendString: theKey];
    }

  encryptedPassword = [[NSMutableData alloc] init];
 
  for ( i = 0; i < [thePassword length]; i++ )
    {
      p = [thePassword characterAtIndex: i];
      k = [key characterAtIndex: i];

      e = p ^ k;
      
      [encryptedPassword appendBytes: (void *)&e length: 2];
    }

  result = AUTORELEASE([[NSString alloc] initWithData: [MimeUtility encodeBase64: encryptedPassword  lineLength: 0]
					 encoding: NSASCIIStringEncoding]);

  RELEASE(encryptedPassword);
  RELEASE(key);

  return result;
}


//
//
//
+ (NSString *) decryptPassword: (NSString *) thePassword
                       withKey: (NSString *) theKey
{
  NSMutableString *key;
  NSMutableString *password;
  NSData *dec;
  unsigned char *decryptedPassword;
  NSString *result;
  int i;
  unichar p, k, d;

  if ( nil == thePassword || nil == theKey )
    {
      return nil;
    }

  // We 'verify' if the password is not encoded in base64
  // We should not rely on this method but it's currently the best guess we could make
  if ( [thePassword length] == 0 || 
       ([thePassword length] & 0x03) || 
       [theKey length] == 0)
    {
      return thePassword;
    }

  // The length of the key must be greater (or equal) than
  // the length of the password
  key = [[NSMutableString alloc] init];
  
  while ( [key length] < [thePassword length] )
    {
      [key appendString: theKey];
    }

  password = [[NSMutableString alloc] init];

  dec = [MimeUtility decodeBase64: [thePassword dataUsingEncoding: NSASCIIStringEncoding]];
  decryptedPassword = (unsigned char *)[dec bytes];
  
  for ( i = 0; i < [dec length]; i += 2 )
    {
      d = decryptedPassword[i] | decryptedPassword[i+1];
      k = [key characterAtIndex: i/2];

      p = d ^ k;
      [password appendString: [NSString stringWithCharacters: &p  length: 1]];
    }

  result = [[NSString alloc] initWithString: password];

  RELEASE(password);
  RELEASE(key);

  return AUTORELEASE(result);
}


//
//
//
+ (void) loadAccountsInPopUpButton: (NSPopUpButton *) thePopUpButton 
			    select: (NSString *) theAccount
{
  NSString *aDefaultAccount, *aKey;
  NSEnumerator *theEnumerator;
  NSDictionary *allAccounts;
  NSArray *allKeys;
 
  int i, index;
  
  allAccounts = [Utilities allEnabledAccounts];
  allKeys = [[allAccounts allKeys] sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
  aDefaultAccount = nil;

  if ( theAccount )
    {
      aDefaultAccount = theAccount;
    }
  else
    {
      for (i = 0; i < [allKeys count]; i++)
	{
	  if ( [[[allAccounts objectForKey: [allKeys objectAtIndex: i]] objectForKey: @"DEFAULT"] boolValue] )
	    {
	      aDefaultAccount = [allKeys objectAtIndex: i];
	      break;
	    }
	}
    }

  // We initialize our popup button
  [thePopUpButton removeAllItems];
  
  theEnumerator = [allKeys objectEnumerator];
  i = index = 0;

  while ( (aKey = [theEnumerator nextObject]) )
    {      
      if ( aDefaultAccount && 
	   [aKey isEqualToString: aDefaultAccount] ) 
	{
	  index = i;
	}
      
      [thePopUpButton insertItemWithTitle: [NSString stringWithFormat: @"%@ (%@)",
						     [[[allAccounts objectForKey: aKey] objectForKey: @"PERSONAL"]
						       objectForKey: @"EMAILADDR"], aKey]
		      atIndex: i];
      i++;
    }
  
  [thePopUpButton selectItemAtIndex: index];
  [thePopUpButton synchronizeTitleAndSelectedItem];
}


//
//
//
+ (void) loadTransportMethodsInPopUpButton: (NSPopUpButton *) thePopUpButton
{
  NSArray *allKeys;
  int i;

  // We initialize our popup button used to select the transport methods
  [thePopUpButton removeAllItems];

  allKeys = [[Utilities allEnabledAccounts] allKeys];

  for (i = 0; i < [allKeys count]; i++)
    {
      NSDictionary *allValues;
      NSString *aString;
         
      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: [allKeys objectAtIndex: i]] objectForKey: @"SEND"];
      
      if ( [[allValues objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP)
	{
	  aString = [NSString stringWithFormat: @"SMTP (%@)", [allValues objectForKey: @"SMTP_HOST"]];
	}
      else
	{
	  aString = [NSString stringWithFormat: @"Mailer (%@)", [allValues objectForKey: @"MAILER_PATH"]];
	}
      
      [thePopUpButton insertItemWithTitle: aString  atIndex: 0];
    }
}


//
//
//
+ (NSString *) accountNameForFolder: (Folder *) theFolder
{
  if ( [theFolder isKindOfClass: [IMAPFolder class]] )
    {
      NSString *aUsername, *aServerName;
      IMAPStore *aStore;
      
      aStore = (IMAPStore *)[theFolder store];
      aUsername = [aStore username];
      aServerName = [aStore name];

      return [self accountNameForServerName: aServerName
                       username: aUsername];
  }

  return nil;
}


//
//
//
+ (NSString *) accountNameForItemTitle: (NSString *) theString
{
  NSString *theAccountName;
  NSRange aRange;
  
  // We first extract the profile name from the popup title.
  aRange = [theString rangeOfString: @"("
		      options: NSBackwardsSearch];
  
  theAccountName = [theString substringWithRange: NSMakeRange(aRange.location + 1,
							      [theString length] - aRange.location - 2)];
  
  return theAccountName;
}


//
//
//
+ (NSString *) accountNameForTransportMethodItemTitle: (NSString *) theString
{
  NSString *aString, *theAccountName;
  NSEnumerator *theEnumerator;
  NSDictionary *allAccounts;
  NSRange aRange;
  
  // We first extract transport method value from the popup title.
  aRange = [theString rangeOfString: @"("
		      options: NSBackwardsSearch];
  
  aString = [theString substringWithRange: NSMakeRange(aRange.location + 1,
						       [theString length] - aRange.location - 2)];

  allAccounts = [Utilities allEnabledAccounts];
  theEnumerator = [allAccounts keyEnumerator];
  
  while ( (theAccountName = [theEnumerator nextObject]) )
    {
      NSDictionary *allValues;

      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: theAccountName] objectForKey: @"SEND"];

      if ( [[allValues objectForKey: @"SMTP_HOST"] isEqualToString: aString] ||
	   [[allValues objectForKey: @"MAILER_PATH"] isEqualToString: aString] )
	{
	  break;
	}
    }

  return theAccountName;
}

//
//
//
+ (NSString *) accountNameForServerName: (NSString *) theServerName
			       username: (NSString *) theUsername
{
  NSEnumerator *theEnumerator;
  NSString *theAccountName;

  theEnumerator = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] keyEnumerator];

  while ( (theAccountName = [theEnumerator nextObject]) )
    {
      NSDictionary *allValues;

      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: theAccountName] objectForKey: @"RECEIVE"];

      if ( [[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP && 
	   [[allValues objectForKey: @"USERNAME"] isEqualToString: theUsername] &&
	   [[allValues objectForKey: @"SERVERNAME"] isEqualToString: theServerName] )
	{
	  return theAccountName;
	}
    }
  
  return nil;
}


//
// Returns the name of the default account
//
+ (NSString *) defaultAccountName
{
    NSString *aDefaultAccount;
    NSDictionary *allAccounts;
    NSArray *allKeys;

    int i;

    allAccounts = [Utilities allEnabledAccounts];
    allKeys = [allAccounts allKeys];
    aDefaultAccount = nil;

    for (i = 0; i < [allKeys count]; i++)
    {
	if ( [[[allAccounts objectForKey: [allKeys objectAtIndex: i]] objectForKey: @"DEFAULT"] boolValue] )
	{
	    return ([allKeys objectAtIndex: i]);
	}
    }

    return (nil);
}

//
// This method returns the window associated to the folder & store.
// If the folder name is nil, it returns the first window that uses the 'store'.
//
+ (id) windowForFolderName: (NSString *) theName
		     store: (Store *) theStore
{
  NSArray *allWindows;
	      
  // We get all opened windows
  allWindows = [GNUMail allMailWindows];
  
  if ( allWindows )
    {
      Folder *aFolder;
      id aWindow;
      int i;
      
      for (i = 0; i < [allWindows count]; i++)
	{
	  aWindow = [allWindows objectAtIndex: i];
	  aFolder = [(MailWindowController *)[aWindow windowController] folder];
	  
	  // If we found our opened folder
	  if ( theName &&
	       [[aFolder name] isEqualToString: theName] && 
	       [aFolder store] == theStore )
	    {
	      return aWindow;
	    }
	  else if ( theName == nil &&
		    [aFolder store] == theStore )
	    {
	      return aWindow;
	    }
	}
    }
  
  return nil;
}


//
//
//
+ (FolderNode *) folderNodesFromFolders: (NSEnumerator *) theFolders
			      separator: (NSString *) theSeparator
{
  NSString *aName, *aString;
  NSRange aRange;
  int length;
 
  FolderNode *root;
 
  root = [[FolderNode alloc] init];
  [root setParent: nil];

  if ( !theSeparator )
    {
      theSeparator = [NSString stringWithString: @"/"];
    }

  length = [theSeparator length];

  while ( ((aString = [theFolders nextObject])) )
    {
      aRange = [aString rangeOfString: theSeparator];
      
      if ( aRange.length > 0)
	{ 
	  FolderNode *parent;
	  int mark;
	  
	  parent = root;
	  mark = 0;
	  
	  while ( aRange.length > 0 )
	    {
	      if ( mark == aRange.location )
		{
		  mark += length; 
		}
	      else
		{	  
		  aName = [aString substringWithRange: NSMakeRange(mark, aRange.location - mark)];
		  
		  if ( ![parent childWithName: aName] )
		    {
		      [parent addChild: [FolderNode folderNodeWithName: aName
						    parent: parent]];
		    }
		  
		  parent = [parent childWithName: aName];
		  mark = aRange.location + aRange.length;
		}
	      
	      aRange = [aString rangeOfString: theSeparator
				options: 0
				range: NSMakeRange(mark, [aString length] - mark)];
	    }
	  
	  aName = [aString substringFromIndex: mark];

	  if ( ![parent childWithName: aName] )
	    {
	      [parent addChild: [FolderNode folderNodeWithName: aName
					    parent: parent]];
	    }
	}
      else
	{
	  if ( ![root childWithName: aString] )
	    {
	      [root addChild: [FolderNode folderNodeWithName: aString
					  parent: root]];
	    }
	}
    }

  return AUTORELEASE(root);
}


//
// This method returns a node using a thePath to search
// and rootNode as the starting point in it's search.
//
// The path to the node MUST always use @"/" as the folder separator
// if no other separtor can be specified. Otherwise, it should use
// the result of the -folderSeparator method from IMAPStore.
//
+ (FolderNode *) folderNodeForPath: (NSString *) thePath
			     using: (FolderNode *) rootNode
			 separator: (NSString *) theSeparator
{
  NSArray *pathComponents;
  FolderNode *aFolderNode;
  int i, j;

  pathComponents = [thePath componentsSeparatedByString: theSeparator];
  aFolderNode = rootNode;

  for (i = 0; i < [pathComponents count]; i++)
    {
      NSString *aPathComponent;

      aPathComponent = [pathComponents objectAtIndex: i];
      
      if ( [aPathComponent length] == 0 )
	{
	  continue;
	}

      for (j = 0; j < [aFolderNode childCount]; j++)
	{
	  if ( [[[aFolderNode childAtIndex: j] name] isEqualToString: aPathComponent] )
	    {
	      aFolderNode = [aFolderNode childAtIndex: j];
	      break;
	    }
	}
    }
  
  return aFolderNode;
}


//
// This method build a complete path of a node. It will stop
// building the path when parent == nil is reached.
//
// We always return a /<Store name>/folder/subfolder/subsubfolder
// or                 /<Store name>/folder.subfolder.subsubfolder
//
+ (NSString *) completePathForFolderNode: (FolderNode *) theFolderNode
			       separator: (NSString *) theSeparator
{
  NSMutableString *aMutableString;
  FolderNode *parent;

  aMutableString = [[NSMutableString alloc] init];
  parent = theFolderNode;

  if ( !theSeparator )
    {
      theSeparator = @"/";
    }

  // We don't loop for no reason if we root was passed as the parameter
  if ( ![parent parent] )
    {
      return [NSString stringWithFormat: @"/%@/", [parent name]];
    }
  
  while ( parent != nil )
    {
      [aMutableString insertString: [parent name]
		      atIndex: 0];
      
      // We verify if the parent of that node has a parent.
      // If it doesn't, that means we must add our /<Store name>/
      // and break the loop.
      if ( [[parent parent] parent] )
	{
	  [aMutableString insertString: theSeparator
			  atIndex: 0];
	}
      else
	{
	  [aMutableString insertString: [NSString stringWithFormat: @"/%@/", [[parent parent] name]]
			  atIndex: 0];
	  break;
	}
      
      parent = [parent parent];
    }
  
  return AUTORELEASE(aMutableString);
}


//
// We skip the first <foo bar> in  /<foo bar>/folder/subfolder/mbox
//                             or  /<foo bar>/folder.subfolder.mbox
// in order to only get the full path of the mbox.
//
// If the no mailbox was selected, we return nil.
//
+ (NSString *) pathOfFolderFromFolderNode: (FolderNode *) theFolderNode
				separator: (NSString *) theSeparator
{
  NSString *aString;
  NSRange aRange;
  
  if ( !theSeparator )
    {
      theSeparator = @"/";
    }

  // We build our full path (including /<Store>/) to our folder
  aString = [Utilities completePathForFolderNode: theFolderNode
		       separator: theSeparator];
  
  // We trim the /<Store>/ part.
  aRange = [aString rangeOfString: @"/"
		    options: 0
		    range: NSMakeRange(1, [aString length] - 1)];
  
  if ( aRange.length )
    {
      return [aString substringFromIndex: aRange.location + 1];
    }
  
  return nil;
}


//
//
//
+ (NSString *) flattenPathFromString: (NSString *) theString
			   separator: (NSString *) theSeparator
{
  NSMutableString *aMutableString;
  NSRange aRange;
  
  // If the separator is undefined, we assume a default one.
  if ( !theSeparator )
    {
      theSeparator = @"/";
    }
  
  aMutableString = [[NSMutableString alloc] initWithString: theString];

  aRange = [aMutableString rangeOfString: theSeparator];

  while ( aRange.length )
    {
      [aMutableString replaceCharactersInRange: aRange
		      withString: @"_"];

      aRange = [aMutableString rangeOfString: theSeparator
			       options: 0
			       range: NSMakeRange(aRange.location + 1,
						  [aMutableString length] - aRange.location - 1)];
    }

  return AUTORELEASE(aMutableString);
}


//
// Calculates the store key from a node. The key returned is:
//
// <username> @ <store name>.
//
// The usage of this method only makes sense for FolderNode objects
// used to represent an IMAP store and its folders.
//
+ (NSString *) storeKeyForFolderNode: (FolderNode *) theFolderNode
			  serverName: (NSString **) theServerName
			    username: (NSString **) theUsername
{  
  NSString *aString = nil;
  
  if ( theFolderNode )
    {
      NSString *aServerName, *aUsername;
      NSRange aRange;
   
      aString = [Utilities completePathForFolderNode: theFolderNode
			   separator: @"/"];
      
      aRange = [aString rangeOfString: @"/"
			options: 0
			range: NSMakeRange(1, [aString length] - 1)];
  
      if ( aRange.length )
	{
	  aString = [aString substringWithRange: NSMakeRange(1, aRange.location - 1)];
	}
      else
	{
	  aString = [aString substringFromIndex: 1];
	}
      
      aString = [aString stringByTrimmingWhiteSpaces];

      // We ensure that we really received a "IMAP value"
      if ( [aString isEqualToString: _(@"Local")] )
	{
	   // Seems we got a Local value.
	  aServerName = nil;
	  aUsername = NSUserName();
	}
      else
	{
	  NSDictionary *allValues;

	  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
			 objectForKey: aString] objectForKey: @"RECEIVE"];
	  
	  aServerName = [allValues objectForKey: @"SERVERNAME"];
	  aUsername = [allValues objectForKey: @"USERNAME"];
	  aString = [NSString stringWithFormat: @"%@ @ %@", aUsername, aServerName];
	}
      
      if ( theServerName != NULL )
	{
	  *theServerName = aServerName;
	}
      
      if ( theUsername != NULL )
	{
	  *theUsername = aUsername;
	}
    }
  
  return aString;
}


//
//
//
+ (BOOL) URLWithString: (NSString *) theString
           matchFolder: (Folder *) theFolder
{
  URLName *theURLName;
  
  theURLName = [[URLName alloc] initWithString: theString
				path: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];

  if ( [[theFolder name] isEqualToString: [theURLName foldername]] )
    {
      // If's a local folder, we simply compare the protocol.
      if ( [theFolder isKindOfClass: [LocalFolder class]] )
	{
	  if ( [[theURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
	    {
	      RELEASE(theURLName);
	      return YES;
	    }
	}
      // It's an IMAP folder, we must compare the hostname and the username.
      else
	{
	  IMAPStore *aStore;

	  aStore = (IMAPStore *)[theFolder store];
	  
	  if ( [[aStore name] isEqualToString: [theURLName host]] &&
	       [[aStore username] isEqualToString: [theURLName username]] )
	    {
	      RELEASE(theURLName);
	      return YES;
	    }
	}
    }

  RELEASE(theURLName);
  return NO;
}


//
//
//
+ (BOOL) stringValueOfURLName: (NSString *) theString
	            isEqualTo: (NSString *) theName
{
  NSEnumerator *theEnumerator;
  
  NSString *theAccountName;
  
  theEnumerator = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] keyEnumerator];

  while ( (theAccountName = [theEnumerator nextObject]) )
    {
      NSDictionary *allValues;

      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: theAccountName] objectForKey: @"MAILBOXES"];

      if ( [[allValues objectForKey: theName] isEqualToString: theString] )
	{
	  return YES;
	}
    }
  
  return NO;
}


//
//
//
+ (NSString *) stringValueOfURLNameFromFolder: (Folder *) theFolder
{
  NSString *aString;

  
  if ( [theFolder isKindOfClass: [LocalFolder class]] )
    {
      aString = [NSString stringWithFormat: @"local://%@/%@", 
			  [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
			  [theFolder name]];
    }
  else
    {
      aString = [NSString stringWithFormat: @"imap://%@@%@/%@", 
			  [((IMAPStore *)[theFolder store]) username],
			  [((IMAPStore *)[theFolder store]) name],
			  [theFolder name]];
    }

  return aString;
}


//
//
//
+ (NSString *) stringValueOfURLNameFromFolderNode: (FolderNode *) theFolderNode
				       serverName: (NSString *) theServerName
					 username: (NSString *) theUsername
{
  NSString *aString;

  aString = [Utilities pathOfFolderFromFolderNode: theFolderNode
		       separator: @"/"];


  //
  // If it's a Local mailbox...
  //
  if ( [aString hasPrefix: _(@"Local Mailboxes")] )
    {
      // We have something like "Local Mailboxes/subfolderA/subfolderB/mbox"
      NSRange aRange;

      aRange = [aString rangeOfString: @"/"];

      // We build our URL
      aString = [NSString stringWithFormat: @"local://%@/%@",
			  [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"],
			  [aString substringFromIndex: aRange.location + 1]];
    }
  //
  // It's an IMAP mailbox.
  //
  else
    {
      NSString  *aPathToFolder;
      NSRange aRange;
      
      // We have something like <Account name>/subfolderA/subfolderB/mbox
      aRange = [aString rangeOfString: @"/"];  // we search for: <Account name>/  <- the /
      aPathToFolder = [aString substringFromIndex: aRange.location + 1];
      
      if ( theServerName && theUsername )
	{
	  aString = [NSString stringWithFormat: @"imap://%@@%@/%@", theUsername, theServerName, aPathToFolder];
	}
      else
	{
	  NSString *aServername, *aUsername, *theAccountName;
	  NSDictionary *allValues;

	  theAccountName = [aString substringToIndex: aRange.location];      
	  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
			 objectForKey: theAccountName] objectForKey: @"RECEIVE"];
	  aUsername = [allValues objectForKey: @"USERNAME"];
	  aServername = [allValues objectForKey: @"SERVERNAME"];
	  
	  aString = [NSString stringWithFormat: @"imap://%@@%@/%@", aUsername, aServername, aPathToFolder];
	}
    }
  
  return aString;
}


//
//
//
+ (FolderNode *) initializeFolderNodesUsingAccounts: (NSDictionary *) theAccounts
{
  FolderNode *allNodes, *nodes;
  LocalStore *aStore;
  NSArray *allKeys;

  int i;

  allNodes = [[FolderNode alloc] init];
  
  //
  // Local
  //
  aStore = [[MailboxManagerController singleInstance] storeForName: @"GNUMAIL_LOCAL_STORE"
						      username: NSUserName()];  
  
  nodes = [Utilities folderNodesFromFolders: [aStore folderEnumerator]
		     separator: @"/"];
  
  if ( [nodes childCount] > 0 )
    {
      [nodes setName: _(@"Local Mailboxes")];
      [allNodes addChild: nodes];
      [nodes setParent: allNodes];
    }

  //
  // IMAP
  //
  allKeys = [[theAccounts allKeys] sortedArrayUsingSelector: @selector(compare:)];
  
  for (i = 0; i < [allKeys count]; i++)
    {
      NSDictionary *allValues;
      NSArray *theArray;

      allValues = [[theAccounts objectForKey: [allKeys objectAtIndex: i]] objectForKey: @"RECEIVE"];
      theArray = [allValues objectForKey: @"SUBSCRIBED_FOLDERS"];
      
      if ( theArray && [theArray count] > 0  )
	{
	  nodes = [Utilities folderNodesFromFolders: [theArray objectEnumerator]
			     separator: @"/"];
	  
	  [nodes setName: [allKeys objectAtIndex: i]];
	  
	  [allNodes addChild: nodes];
	  [nodes setParent: allNodes];
	}
    }
  
  return AUTORELEASE(allNodes);
}


//
// Usually, theFolderNodes will be a pointer to a FolderNode object
// representing the "tree" of all our Local and/or IMAP nodes.
// This "tree" is usually build using:
//
// Utilities: +initializeFolderNodesUsingAccounts:
//
// We will ALWAYS have at least one node.
//
+ (void) addItemsToPopUpButton: (NSPopUpButton *) thePopUpButton
              usingFolderNodes: (FolderNode *) theFolderNodes
{
  int i;

  [thePopUpButton removeAllItems];
  [thePopUpButton setAutoenablesItems: NO];
    
  // We now add all our nodes
  for (i = 0; i < [theFolderNodes childCount]; i++)
    {
      [Utilities addItem: [theFolderNodes childAtIndex: i]
		 level: 0
		 toPopUpButton: thePopUpButton];
    }
  
  [thePopUpButton selectItemAtIndex: 0];
}


//
// This method add new items to the popup button.
// It respects the level parameter for indenting properly
// the items in order to represent a hierarchy.
//
+ (void) addItem: (FolderNode *) theFolderNode
	   level: (int) theLevel
   toPopUpButton: (NSPopUpButton *) thePopUpButton
{
  NSMutableString *aMutableString;
  FolderNodePopUpItem *theItem;
  int i;

  aMutableString = [[NSMutableString alloc] init];

  for (i = 0; i < theLevel; i++)
    {
      [aMutableString appendString: @"     "];
    }

  [aMutableString appendString: [theFolderNode name]];
  
  theItem = [[FolderNodePopUpItem alloc] initWithTitle: aMutableString
					 action: NULL
					 keyEquivalent: @""];
  [theItem setFolderNode: theFolderNode];
  RELEASE(aMutableString);

  // We enable / disable our item
  if ( [theFolderNode childCount] > 0 )
    {    
      [theItem setAction: NULL];
      [theItem setEnabled: NO];
    }
  else
    {
      [theItem setAction: @selector(foo:)];
      [theItem setEnabled: YES];
    }

  // We finally add our item
  [[thePopUpButton menu] addItem: theItem];
  RELEASE(theItem);

  for (i = 0; i < [theFolderNode childCount]; i++)
    {
      [Utilities addItem: [theFolderNode childAtIndex: i]
		 level: (theLevel + 1)
		 toPopUpButton: thePopUpButton];
    }
}


//
//
//
+ (FolderNodePopUpItem *) folderNodePopUpItemForFolderNode: (FolderNode *) theFolderNode
					       popUpButton: (NSPopUpButton *) thePopUpButton
{
  FolderNodePopUpItem *theItem;
  int i;

  for (i = 0; i < [thePopUpButton numberOfItems]; i++)
    {
      theItem = [thePopUpButton itemAtIndex: i];

      if ( [theItem folderNode] == theFolderNode )
	{
	  return theItem;
	}
    }

  return nil;
}


//
//
//
+ (FolderNodePopUpItem *) folderNodePopUpItemForURLNameAsString: (NSString *) theString
					       usingFolderNodes: (FolderNode *) theFolderNodes
						    popUpButton: (NSPopUpButton *) thePopUpButton
							account: (NSString *) theAccountName
{
  FolderNodePopUpItem *aPopUpItem;
  FolderNode *aFolderNode;
  URLName *aURLName;

  aURLName = [[URLName alloc] initWithString: theString
			      path: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];
  
  if ( [[aURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
    {
      aFolderNode = [Utilities folderNodeForPath: [NSString stringWithFormat: @"%@/%@", _(@"Local Mailboxes"),
							    [aURLName foldername]]
			       
			       using: theFolderNodes
			       separator: @"/"];
    }
  else
    {
      if ( !theAccountName )
	{
	  theAccountName = [Utilities accountNameForServerName: [aURLName host]
				      username: [aURLName username]];
	}

      aFolderNode = [Utilities folderNodeForPath: [NSString stringWithFormat: @"%@/%@", theAccountName, [aURLName foldername]]
			       using: theFolderNodes
			       separator: @"/"];
    }
  
  aPopUpItem = [Utilities folderNodePopUpItemForFolderNode: aFolderNode
			  popUpButton: thePopUpButton];

  RELEASE(aURLName);

  return aPopUpItem;
}


//
// When type is POP3 or IMAP, theKey is directly the key in our
// RECEIVING dictionary.
//
// When type is SMTP, the key is equal to a NSNumber representing
// the index in our SENDING array.
//
+ (NSString *) passwordForKey: (id) theKey
			 type: (int) theType
		       prompt: (BOOL) aBOOL
{
  NSString *aPassword, *usernameKey, *passwordKey, *serverNameKey;
  NSDictionary *allValues;
  
  if ( theType == POP3 || theType == IMAP )
    {
      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: theKey] objectForKey: @"RECEIVE"];
      usernameKey = @"USERNAME";
      passwordKey = @"PASSWORD";
      serverNameKey = @"SERVERNAME";
    }
  else
    {
      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		     objectForKey: theKey] objectForKey: @"SEND"];
      usernameKey = @"SMTP_USERNAME";
      passwordKey = @"SMTP_PASSWORD";
      serverNameKey = @"SMTP_HOST";
    }

  // We define a new key
  theKey = [NSString stringWithFormat: @"%@ @ %@", [allValues objectForKey: usernameKey],
		     [allValues objectForKey: serverNameKey]];

  // We verify in the user defaults
  aPassword = [Utilities decryptPassword: [allValues objectForKey: passwordKey]
			 withKey: theKey];  
 
  // We verify in our cache
  if ( !aPassword )
    {
      aPassword = [passwordCache objectForKey: theKey];
    }

  // If we must prompt for the password
  if ( !aPassword && aBOOL )
    {
      PasswordPanelController *theController; 
      int result;
      
      theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
      [[theController window] setTitle: [NSString stringWithFormat: @"%@ @ %@",
						  [allValues objectForKey: usernameKey],
						  [allValues objectForKey: serverNameKey]]];
      
      result = [NSApp runModalForWindow: [theController window]];
      
      // If the user has entered a password...
      if (result == NSRunStoppedResponse)
	{
	  aPassword = [theController password];
	  
	  // Let's cache this password...
	  [passwordCache setObject: aPassword
			 forKey: theKey];
	}
      else
	{
	  aPassword = nil;
	}
      
      RELEASE(theController);
    }
  
  return aPassword;
}


//
//
//
+ (NSMutableDictionary *) passwordCache
{
  return passwordCache;
}


//
// Creates a reply
//
+ (void) replyToSender: (Message *) theMessage
		folder: (Folder *) theFolder
	    replyToAll: (BOOL) theBOOL
{
  EditWindowController *theEditWindowController;
  NSString *theAccountName, *theAccountAddress;
  Message *aMessage;
  
  BOOL shouldReplyToAll, shouldReplyToList;
  int i;
  
  if ( !theMessage )
    {
      NSBeep();
      return;
    }

  // We initialize our message, just to be safe. It SHOULD already be
  // initialized since it's selected (since the user wants to reply to
  // this selected mail)
  INITIALIZE_MESSAGE(theMessage);
 
  if ( IMAPSTORE_IS_DISCONNECTED([[theMessage folder] store]) )
    {
      [[MailboxManagerController singleInstance] connectionWasLost: [[theMessage folder] store]];
      [[MailboxManagerController singleInstance] setStore: nil
						 name: [(IMAPStore *)[[theMessage folder] store] name]
						 username: [(IMAPStore *)[[theMessage folder] store] username]];
      return;
    }
 
  theAccountName = [self _guessAccountNameFromMessage: theMessage];
  theAccountAddress = nil;
  shouldReplyToAll = shouldReplyToList = NO;
  
  if ( theAccountName )
    {
      theAccountAddress = [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: theAccountName]
			     objectForKey: @"PERSONAL"] objectForKey: @"EMAILADDR"];
    }
  
  //
  // We verify for the List-Post header. This is definied in RFC-2369. We use it to offer
  // a "reply to list" option to the user.
  //
  if ( [[theMessage allHeaders] objectForKey: @"List-Post"] &&
       [[[[theMessage allHeaders] objectForKey: @"List-Post"] stringByTrimmingWhiteSpaces] caseInsensitiveCompare: @"NO"] != NSOrderedSame )
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"List Reply..."),
			       _(@"Would you like to reply to the mailing list only?"),
			       _(@"Yes"), // default
			       _(@"No"),  // alternate
			       nil);
      
      if ( choice == NSAlertDefaultReturn )
        {
	  shouldReplyToList = YES;
	}
    }
  
  if ( shouldReplyToList )
    {
      // We do only that (ie., we don't offer the "Reply to all" option)
    }
  else if ( theBOOL )
    {
      shouldReplyToAll = YES;
    }
  else if ( [theMessage recipientsCount] > 1 )
    {
      int choice;
      
      choice = NSRunAlertPanel(_(@"Reply..."),
			       _(@"Would you like to reply to all recipients?"),
			       _(@"No"),  // default
			       _(@"Yes"), // alternate
			       nil);
      
      if ( choice == NSAlertAlternateReturn )
        {
	  shouldReplyToAll = YES;
	}
      else
	{
	  shouldReplyToAll = NO;
	}
    }  
  
  // We create our window controller
  theEditWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];

  if ( theEditWindowController )
    {
      [[theEditWindowController window] setTitle: _(@"Reply to a message...")];
      [theEditWindowController setSignaturePosition: 
				 [[NSUserDefaults standardUserDefaults]
				   integerForKey: @"SIGNATURE_REPLY_POSITION"] ];
      [theEditWindowController setShowCc: shouldReplyToAll];
      
      // We set the original message
      [theEditWindowController setUnmodifiedMessage: theMessage];
      
      // We create a replied copy of our message and we retain it
      aMessage = [theMessage replyWithReplyToAll: shouldReplyToAll];
      RETAIN(aMessage);
      
      // If we are in the Sent folder, we replace the recipient with the original recipients
      // of the messages.
      if ( [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: theFolder]  
		      isEqualTo: @"SENTFOLDERNAME"] )
	{
          [aMessage setRecipients: [[theEditWindowController unmodifiedMessage] recipients]];
	}
      
      // Remove original recipient from recipient list. As a small optimization, we don't do that
      // if we reply only to a mailing list since the recepients will get replaced below.
      if ( !shouldReplyToList && shouldReplyToAll && theAccountAddress )
	{
	  for (i = ([aMessage recipientsCount] - 1); i >= 0; i--)
	    {
	      if ( [[(InternetAddress*)[[aMessage recipients] objectAtIndex: i] address] 
		     caseInsensitiveCompare: theAccountAddress] == NSOrderedSame )
		{
		  [aMessage removeFromRecipients: (InternetAddress*)[[aMessage recipients] objectAtIndex: i]];
		  break;
		}
	    }		  
	}
      
      // We want to include only the list address on list reply
      if ( shouldReplyToList )
	{
	  InternetAddress *theInternetAddress;
	  NSMutableString *aMutableString;

	  aMutableString = [NSMutableString stringWithString: [theMessage headerValueForName: @"List-Post"]];
	  [aMutableString deleteCharactersInRange: [aMutableString rangeOfString: @"mailto:"]];
	  
	  theInternetAddress = [[InternetAddress alloc] initWithString: aMutableString];
	  [theInternetAddress setType: TO];
	  [aMessage setRecipients: [NSArray arrayWithObject: theInternetAddress]];
	  RELEASE(theInternetAddress);
	}
      
      [theEditWindowController setMessage: aMessage];
      RELEASE(aMessage);
      
      // We set the appropriate account and show our window
      [theEditWindowController setAccountName: theAccountName];
      [theEditWindowController showWindow: self];
    }
}


//
//
//
+ (void) forwardMessage: (Message *) theMessage
{
  EditWindowController *theEditWindowController;
  NSString *theAccountName;
  Message *aMessage;
  
  if ( !theMessage ) 
    {
      NSBeep();
      return;
    }

  // We initialize our message, just to be safe. It SHOULD already be
  // initialized since it's selected (since the user wants to forward
  // this selected mail)
  INITIALIZE_MESSAGE(theMessage);
  
  if ( IMAPSTORE_IS_DISCONNECTED([[theMessage folder] store]) )
    {
      [[MailboxManagerController singleInstance] connectionWasLost: [[theMessage folder] store]];
      [[MailboxManagerController singleInstance] setStore: nil
						 name: [(IMAPStore *)[[theMessage folder] store] name]
						 username: [(IMAPStore *)[[theMessage folder] store] username]];
      return;
    }

  // guess the profile we should be using
  theAccountName = [self _guessAccountNameFromMessage: theMessage];
  
  // We create a forwarded copy of our message and we retain it
  aMessage = [theMessage forward];
  RETAIN(aMessage);
  
  theEditWindowController = [[EditWindowController alloc] initWithWindowNibName: @"EditWindow"];
  
  if ( theEditWindowController )
    {
      [[theEditWindowController window] setTitle: _(@"Forward a message...")];
      [theEditWindowController setSignaturePosition: 
				 [[NSUserDefaults standardUserDefaults] 
				   integerForKey: @"SIGNATURE_FORWARD_POSITION"] ];
      [theEditWindowController setMessage: aMessage];
      [theEditWindowController setShowCc: NO];
      
      // We set the appropriate account and show our window
      [theEditWindowController setAccountName: theAccountName];
      [theEditWindowController showWindow: self];
    }
  
  RELEASE(aMessage);
}


//
// This method displays a message in the textView.
//
+ (void) showMessage: (Message *) theMessage
	      target: (NSTextView *) theTextView 
      showAllHeaders: (BOOL) headersFlag
{
  if ( theMessage )
    {
      Flags *theFlags;
      id aDelegate;
      int i;

      // If the content of the message has neven been parsed before, we do it now!
      INITIALIZE_MESSAGE(theMessage);
      
      if ( IMAPSTORE_IS_DISCONNECTED([[theMessage folder] store]) )
	{
	  return;
	}

#ifdef MACOSX
      [[theTextView textStorage] beginEditing];
#endif

      // We clear our 'Save Attachment' menu
      [(GNUMail *)[NSApp delegate] removeAllItemsFromMenu: [(GNUMail *)[NSApp delegate] saveAttachmentMenu]];
      
      // We begin by clearing what we have in our text view
      [[theTextView textStorage] deleteCharactersInRange:
				    NSMakeRange(0, [[theTextView textStorage] length])];
      
      // We inform our bundles that the message WILL BE shown in the text view
      for (i = 0; i < [[GNUMail allBundles] count]; i++)
	{
	  id<GNUMailBundle> aBundle;
	  
	  aBundle = [[GNUMail allBundles] objectAtIndex: i];

	  if ( [aBundle respondsToSelector: @selector(messageWillBeDisplayed:inView:)] )
	    {
	      [aBundle messageWillBeDisplayed: theMessage
		       inView: theTextView];
	    }
	}
      
      // Then, we add our headers and our content
      [[theTextView textStorage] appendAttributedString: 
				   [Utilities attributedStringFromHeadersForMessage: theMessage
					      showAllHeaders: headersFlag
					      useMailHeaderCell: YES] ];
      
      [[theTextView textStorage] appendAttributedString:
				   [Utilities _quoteMessageContentFromAttributedString:			     
						[Utilities formattedAttributedStringFromAttributedString:
							     [Utilities attributedStringFromContentForPart: 
									  theMessage]]] ]; 
      
      // We update the Flags of our message (ie., we add SEEN)
      theFlags = [theMessage flags];
      
      // If the message was a new one, let's change the app icon back to GNUMail.tiff
      if ( ![theFlags contain: SEEN] )
	{
	  [theFlags add: SEEN];
	}
      
      // We remove the potential RECENT IMAP flag.
      [theFlags remove: RECENT];
      
      // We ensure that the selected table row is properly refreshed after changing the flags
      // FIXME: find a more elegant solution.
      aDelegate = [[GNUMail lastMailWindowOnTop] delegate];
      
      if ( aDelegate )
	{
	  id aDataView;
	  
	  aDataView = nil;

	  if ( [aDelegate isKindOfClass: [MailWindowController class]] )
	    {
	      aDataView = [aDelegate dataView];
	    }
	  else if ( [aDelegate isKindOfClass: [MessageViewWindowController class]] )
	    {
	      aDataView = [[aDelegate mailWindowController] dataView];
	    }
	  
	  if ( aDataView )
	    {
	      // FIXME: We could optimize this by getting all the parent of the 'row' and
	      //        redisplay only those rows (since for example, we could read an unread
	      //        message at the bottom of a thread.
	      [aDataView setNeedsDisplay: YES];
	      //[aDataView setNeedsDisplayInRect: [aDataView rectOfRow: [aDataView selectedRow]]];		  
	    }
	}
      
      // We finally highlight the URLs in our message, if we want to
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] &&
	   [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIGHLIGHT_URL"] intValue] == NSOnState )
	{
          [Utilities _highlightAndActivateURLs: theTextView];

          // We update the rects of our cursors
          [[theTextView window] invalidateCursorRectsForView: theTextView];
	}

      // We inform our bundles that the message HAS BEEN shown in the text view
      for (i = 0; i < [[GNUMail allBundles] count]; i++)
	{
	  id<GNUMailBundle> aBundle;
	  
	  aBundle = [[GNUMail allBundles] objectAtIndex: i];
	  
	  if ( [aBundle respondsToSelector: @selector(messageWasDisplayed:inView:)] )
	    {
	      [aBundle messageWasDisplayed: theMessage
		       inView: theTextView];
	    }
	}

      // If we have more than one attachement, we create a 'save all' menu at the top
      if ( [[(GNUMail *)[NSApp delegate] saveAttachmentMenu] numberOfItems] > 1 )
        {
          NSMenuItem *aMenuItem;

	  aMenuItem = [[NSMenuItem alloc] init];
          [aMenuItem setTitle: _(@"Save all")];
	  [aMenuItem setTarget: [NSApp delegate]];
          [aMenuItem setAction: @selector(saveAllAttachments:)];
          [aMenuItem setKeyEquivalent: @""];
          [[(GNUMail *)[NSApp delegate] saveAttachmentMenu] insertItem: aMenuItem
							    atIndex: 0];
	  
          RELEASE(aMenuItem);
        }

#ifdef MACOSX
      [[theTextView textStorage] endEditing];
#endif
    }
  else
    {
      NSDebugLog(@"Unable to find the message in the hashtable!");
    }
  
  // We scroll to the beginning of the message and we remove any previous text selection
  [theTextView scrollRangeToVisible: NSMakeRange(0,0)];
  [theTextView setSelectedRange: NSMakeRange(0,0)];
}


//
//
//
+ (void) showMessageRawSource: (Message *) theMessage 
		       target: (NSTextView *) theTextView
{
  if ( theMessage && theTextView )
    {
      IMAPFolder *aFolder;
      IMAPStore *aStore;
      NSString *aString;
      NSData *aData;

      aFolder = (IMAPFolder *)[theMessage folder];
      aStore = (IMAPStore *)[aFolder store];

      aData = [theMessage rawSource];

      if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
	{
	  [[MailboxManagerController singleInstance] setStore: nil
							 name: [aStore name]
						     username: [aStore username]];
	  return;
	}
	
      aString = [[NSString alloc] initWithData: aData
				  encoding: NSASCIIStringEncoding];
      
      if ( aString )
	{
	  NSAttributedString *theAttributedString;
	  NSDictionary *attributes;
	  
	  attributes = [NSDictionary dictionaryWithObject: [NSFont userFixedPitchFontOfSize: 0]
				     forKey: NSFontAttributeName];
	  
	  theAttributedString = [[NSAttributedString alloc] initWithString: aString
							    attributes: attributes];
	  
	  [[theTextView textStorage] setAttributedString: theAttributedString];
	  
	  RELEASE(theAttributedString);
	  RELEASE(aString);

	  // We scroll to the beginning of the message and we remove any previous text selection
	  [theTextView scrollRangeToVisible: NSMakeRange(0,0)];
	  [theTextView setSelectedRange: NSMakeRange(0,0)];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
+ (void) clickedOnCell: (id <NSTextAttachmentCell>) attachmentCell
	        inRect: (NSRect) cellFrame
               atIndex: (unsigned) charIndex
		sender: (id) sender
{
  NSTextAttachment *attachment;
  NSFileWrapper *filewrapper;
  MimeType *aMimeType;
  NSString *aString;

#ifdef MACOSX
  NSWindow *aWindow;
#endif  

  // If it's our header cell, we immediately return
  if ( [attachmentCell isKindOfClass: [MailHeaderCell class]] )
    {
      return;
    }
  
  attachment = [attachmentCell attachment];
  filewrapper = [attachment fileWrapper];
  aMimeType = nil;

  aMimeType = [[MimeTypeManager singleInstance] mimeTypeForFileExtension:
						  [[filewrapper preferredFilename] pathExtension]];
  
  if ( !aMimeType || [aMimeType action] == PROMPT_SAVE_PANEL || sender == [NSApp delegate] )
    {
      NSSavePanel *aSavePanel;

      aSavePanel = [NSSavePanel savePanel];
      [aSavePanel setAccessoryView: nil];
      [aSavePanel setRequiredFileType: @""];
      
#ifdef MACOSX
      if ( [sender respondsToSelector:@selector(window)] )
	{
	  aWindow = [sender window];
	}
      else
	{
	  aWindow = [GNUMail lastMailWindowOnTop];
	}
      
      [aSavePanel beginSheetForDirectory: [GNUMail currentWorkingPath] 
		  file: [filewrapper preferredFilename] 
		  modalForWindow: aWindow
		  modalDelegate: self 
		  didEndSelector: @selector(_savePanelDidEnd: returnCode: contextInfo:) 
		  contextInfo: filewrapper];
#else
      [self _savePanelDidEnd: aSavePanel 
	    returnCode: [aSavePanel runModalForDirectory: [GNUMail currentWorkingPath]
				    file: [filewrapper preferredFilename]]
	    contextInfo: filewrapper];
#endif
    }
  else if ( [aMimeType action] == OPEN_WITH_WORKSPACE )
    {
      aString = [NSString stringWithFormat:@"%@/%d_%@", GNUMailTemporaryDirectory(), 
			  [[NSProcessInfo processInfo] processIdentifier],
			  [filewrapper preferredFilename]];
      
      if ( [filewrapper writeToFile: aString
			atomically: YES
			updateFilenames: NO] )
	{
	  //
	  // If we successfully open the file with the Workspace, it will be 
	  // removed when the application exists in GNUMail+Extensions: -removeTemporaryFiles.
	  //
	  if ( ![[NSWorkspace sharedWorkspace] openFile: aString] )
	    {
	      [[NSFileManager defaultManager] removeFileAtPath: aString
					      handler: nil];
	    }
	}
    }
  else
    { 
      if ( ![[NSFileManager defaultManager] fileExistsAtPath: [aMimeType dataHandlerCommand]] )
	{
	  
	  NSRunAlertPanel(_(@"Error!"),
			  _(@"The external program (%@) for opening this MIME-Type (%@) can't be found."),
			  _(@"OK"),
			  NULL,
			  NULL,
			  [aMimeType dataHandlerCommand], [aMimeType mimeType]);
	  return;
	}
      
      aString = [NSString stringWithFormat:@"%@/%d_%@", GNUMailTemporaryDirectory(), 
			  [[NSProcessInfo processInfo] processIdentifier],
			  [filewrapper preferredFilename]];
      
      if ( [filewrapper writeToFile: aString
			atomically: YES
			updateFilenames: NO] )
	{
	  NSDictionary *aDictionary;
	  
	  aDictionary = [NSDictionary dictionaryWithObjectsAndKeys: [aMimeType dataHandlerCommand], @"command",
				      [NSArray arrayWithObjects: aString, nil], @"arguments",
				      aString, @"file", nil];
	  
	  [NSThread detachNewThreadSelector: @selector(_launchExternalProgram:)
		    toTarget: self
		    withObject: aDictionary];
	}
      else
	{
	  NSBeep();
	}
    }
}


//
//
//
+ (NSDictionary *) allEnabledAccounts
{
  NSMutableDictionary *aMutableDictionary;

  aMutableDictionary = nil;

  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] )
    {
      NSArray *allKeys;
      int i;

      aMutableDictionary = [[NSMutableDictionary alloc] initWithDictionary: [[NSUserDefaults standardUserDefaults] 
									      objectForKey: @"ACCOUNTS"]];
      AUTORELEASE(aMutableDictionary);

      allKeys = [aMutableDictionary allKeys];
      
      for (i = 0; i < [allKeys count]; i++)
	{
	  if ( [[[aMutableDictionary objectForKey: [allKeys objectAtIndex: i]]
		  objectForKey: @"ENABLED"] boolValue] == NO )
	    {
	      [aMutableDictionary removeObjectForKey: [allKeys objectAtIndex: i]];
	    }
	}
    }

  return aMutableDictionary;
}


//
//
//
+ (NSMutableArray *) quoteLevelColors
{
  return quoteLevelColors;
}

@end



//
// Private implementation for Utilities
//
@implementation Utilities (Private)

//
//
//
+ (void) _savePanelDidEnd: (NSSavePanel *) theSavePanel
	       returnCode: (int) theReturnCode
	      contextInfo: (void *) theContextInfo
{
  // If successful, save file under designated name
  if (theReturnCode == NSOKButton)
    {
      if ( ![(NSFileWrapper *)theContextInfo writeToFile: [theSavePanel filename]
			      atomically: YES
			      updateFilenames: YES] )
	{
	  NSBeep();
	}
      [GNUMail setCurrentWorkingPath: [[theSavePanel filename] stringByDeletingLastPathComponent]];
    }
}


//
//
//
+ (NSMutableAttributedString *) _quoteMessageContentFromAttributedString: (id) theAttributedString
{
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"COLOR_QUOTED_TEXT"] == NSOffState )
    {
      return theAttributedString;
    }
  else
    {   
      NSMutableAttributedString *aMutableAttributedString;
      NSDictionary *attributes;
      NSString *aString;
 
      int i, j, len, level;
     
      aMutableAttributedString = [[NSMutableAttributedString alloc] initWithAttributedString: theAttributedString];
      aString = [theAttributedString string];
      len = [aString length];
      i = j = level = 0;
      
      for (; i < len; i++)
	{
	  if ( [aString characterAtIndex: i]=='\n' )
	    {
	      if (i > j)
		{
		  level = levelFromString(aString, j, i);
		  
		  if (level)
		    {
		      attributes = [NSDictionary dictionaryWithObjectsAndKeys: 
						   colorForLevel((level-1)%MAX_LEVEL), 
						 NSForegroundColorAttributeName, 
						 nil];
		      [aMutableAttributedString addAttributes: attributes
						range: NSMakeRange(j,i-j)];
		    }
		}
	      
	  j = i+1;
	    }
	}
      
      if (i > j)
	{
	  level = levelFromString(aString, j, i);
	  if (level)
	    {
	      attributes = [NSDictionary dictionaryWithObjectsAndKeys: 
					   colorForLevel((level-1)%MAX_LEVEL), 
					 NSForegroundColorAttributeName, 
					 nil];
	      [aMutableAttributedString addAttributes: attributes
					range: NSMakeRange(j,i-j)];
	    }
	}
      
      return AUTORELEASE(aMutableAttributedString);
    }
}


//
//
//
+ (void) _highlightAndActivateURLs: (NSTextView *) theTextView
{
  NSRange searchRange, foundRange;
  NSString *aString, *aPrefix;
  NSEnumerator *theEnumerator;
  NSArray *allPrefixes;

  int len;
  char c;

  allPrefixes = [NSArray arrayWithObjects: @"www.", @"http://", @"https://", @"ftp://", @"file://", nil];
  theEnumerator = [allPrefixes objectEnumerator];
    
  aString = [theTextView string];
  len = [aString length];

  while ( (aPrefix = (NSString *)[theEnumerator nextObject]) )
    {
      searchRange = NSMakeRange(0, len);
      
      do
	{
	  foundRange = [aString rangeOfString: aPrefix
				options: 0
				range: searchRange];
	  
	  // If we found an URL...
	  if ( foundRange.length > 0 )
	    {
	      NSDictionary *linkAttributes;
	      NSURL *anURL;
	      int end;

	      // Restrict the searchRange so that it won't find the same string again
	      searchRange.location = end = NSMaxRange(foundRange);
	      searchRange.length = len - searchRange.location;
	      
	      // We assume the URL ends with whitespace
	      while ( (end < len) && ((c = [aString characterAtIndex: end]) != '\n' && c != ' ' && c != '\t'))
		{
		  end++;
		}

	      // Set foundRange's length to the length of the URL
	      foundRange.length = end - foundRange.location;

	      // If our URL is ended with a ".", "!", "," or ">", we trim it
	      c = [aString characterAtIndex: (end - 1)];
	      if ( c == '.' || c == '!' || c == ',' || c == '?' || c == '>')
		{
		  foundRange.length--;
		}

	      // We create our URL object from our substring
              // if we found just "www", we prepend "http://"
              if ( [aPrefix caseInsensitiveCompare: @"www."] == NSOrderedSame )
		{
		  anURL = [NSURL URLWithString: [NSString stringWithFormat: @"http://%@", 
							  [aString substringWithRange: foundRange]]];
		}
	      else
		{
		  anURL = [NSURL URLWithString: [aString substringWithRange: foundRange]];
		}
	      
	      // Make the link attributes
	      linkAttributes= [NSDictionary dictionaryWithObjectsAndKeys: anURL, NSLinkAttributeName,
					    [NSNumber numberWithInt: NSSingleUnderlineStyle], NSUnderlineStyleAttributeName,
					    [NSColor blueColor], NSForegroundColorAttributeName,
					    NULL];
	      
	      // Finally, apply those attributes to the URL in the text
	      [[theTextView textStorage] addAttributes: linkAttributes
					 range: foundRange];
	    }
	  
	} while ( foundRange.length != 0 );
    }
}


//
// Given a message, guesses the profile it is associated with by comparing
// recipients with profile addresses. Currently compares address domains
// only (string after '@') to handle email aliases. Once email alias support
// has been added to profiles, we will compare full email addresses.
//
+ (NSString *) _guessAccountNameFromMessage: (Message *) theMessage
{
  InternetAddress *theInternetAddress;
  NSArray *allKeys, *allRecipients;
  NSString *theAccountAddress;
  NSDictionary *theAccount;
  int i, j;
  
  // We get all the message recipients
  allRecipients = [theMessage recipients];
  
  // We get all the keys for the personal profiles
  allKeys = [[[Utilities allEnabledAccounts] allKeys]
	      sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)];
  
  //
  // We go through all enabled account to first try to match one that has the exact recipient
  // defined in our message's recipients list.
  //
  for (i = 0; i < [allKeys count]; i++)
    {
      theAccount = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: [allKeys objectAtIndex: i]];

      // We get the address for this account
      theAccountAddress = [[theAccount objectForKey: @"PERSONAL"] objectForKey: @"EMAILADDR"];

      if ( theAccountAddress && allRecipients )
	{
	  // Walk through the recipients and check against profile address
	  for (j = 0; j < [allRecipients count]; j++)
	    {
	      theInternetAddress = (InternetAddress *)[allRecipients objectAtIndex: j];

	      if ( [[theAccountAddress stringByTrimmingWhiteSpaces] caseInsensitiveCompare: [theInternetAddress address]] == NSOrderedSame)
		{
		  NSDebugLog(@"Profile to be used = %@", [allKeys objectAtIndex: i]);
		  
		  // We found a matching profile, we return it.
		  return [allKeys objectAtIndex: i];
		}
	    }
	}
    }

  //
  // We haven't found one. At least, let's now try to match one that has only the domain part
  // in the message's recipients list.
  //
  for (i = 0; i < [allKeys count]; i++)
    {
      NSString *theAccountDomain;
      NSRange aRange;

      theAccount = [[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: [allKeys objectAtIndex: i]];
      
      // We get the address for this account
      theAccountAddress = [[theAccount objectForKey: @"PERSONAL"] objectForKey: @"EMAILADDR"];
      
      if ( theAccountAddress && allRecipients )
	{
	  // Walk through the recipients and check against profile address
	  for (j = 0; j < [allRecipients count]; j++)
	    {
	      // We check only the domain part of profile address (after '@').
	      aRange = [theAccountAddress rangeOfString: @"@"
					  options: NSBackwardsSearch];
	      
	      if ( aRange.location == NSNotFound )
		{
		  continue;
		}
	      
	      theAccountDomain = [theAccountAddress substringFromIndex: NSMaxRange(aRange)];
	      theInternetAddress = (InternetAddress *)[allRecipients objectAtIndex: j];
	      aRange = [[theInternetAddress address] rangeOfString: theAccountDomain
						     options: NSCaseInsensitiveSearch];
	      
	      if ( aRange.length > 0 )
		{
		  NSDebugLog(@"Profile to be used = %@", [allKeys objectAtIndex: i]);
		  
		  // We found a matching profile, we return it.
		  return [allKeys objectAtIndex: i];
		}
	    }
	}
    }
    
  // No match according to the recipients. Now let's check if we can find out the account
  // that this message belongs to. If we can't, we will simply return nil, which means no
  // profiles were found.
  return [self accountNameForFolder: [theMessage folder]];
}


//
//
//
+ (NSAttributedString *) _attributedStringFromTextForPart: (Part *) thePart
{
  NSMutableDictionary *textMessageAttributes;
  NSAttributedString *aAttributedString;
  NSString *aString;

  aAttributedString = nil;

  // Initial and trivial verifications.
  // We MUST be sure the content isn't nil and also that it's a valid NSString object.
  // Pantomime will return the raw source (NSData object) if the decoding operation has failed.
  if ( !thePart ||
       ![thePart content] || 
       ![[thePart content] isKindOfClass: [NSString class]] )
    {
      goto done;
    }

  // Initializations of the local vars
  textMessageAttributes = [[NSMutableDictionary alloc] init];
  aString = (NSString *)[thePart content];
  
  [textMessageAttributes setObject: [Utilities fontFromFamilyName: [[NSUserDefaults standardUserDefaults] 
								     objectForKey: @"MESSAGE_FONT_NAME"]
					       trait: NSUnboldFontMask
					       size: [[NSUserDefaults standardUserDefaults] floatForKey: @"MESSAGE_FONT_SIZE"]]
			 forKey: NSFontAttributeName];


  //
  // text/html
  //
  if ( [thePart isMimeType: @"text": @"html"] )
    {
#ifdef MACOSX
      NSData *aData;
      
      aData = [aString dataUsingEncoding: [MimeUtility stringEncodingForPart: thePart]];      
      aAttributedString = [[NSAttributedString alloc] initWithHTML: aData
						      documentAttributes: nil];
      AUTORELEASE(aAttributedString);
#else
      aString = [MimeUtility plainTextContentFromPart: thePart];
      aAttributedString = [Utilities attributedStringWithString: aString
				     attributes: textMessageAttributes];
#endif
    }
  //
  // text/enriched
  //
  else if ( [thePart isMimeType: @"text": @"enriched"] )
    {
      aAttributedString = [NSAttributedString attributedStringFromTextEnrichedString: aString];
    }
  //
  // text/rtf
  //
  else if ( [thePart isMimeType: @"text": @"rtf"] )
    {
      NSData *aData;
	      
      aData = [aString dataUsingEncoding: [MimeUtility stringEncodingForPart: thePart]];
      
      aAttributedString = [[NSAttributedString alloc] initWithRTF: aData
						      documentAttributes: NULL];
      AUTORELEASE(aAttributedString); 
    }
  //
  // We surely got a text/plain part
  //
  else
    {
      NSMutableDictionary *plainTextMessageAttributes;
      
      if ( [[NSUserDefaults standardUserDefaults] 
	     objectForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] &&
	   [[NSUserDefaults standardUserDefaults] 
	     integerForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] == NSOnState )
	{
	  plainTextMessageAttributes = [[NSMutableDictionary alloc] init];
	  AUTORELEASE(plainTextMessageAttributes);
	  
	  [plainTextMessageAttributes setObject:
					[Utilities fontFromFamilyName: 
						     [[NSUserDefaults standardUserDefaults] 
						       objectForKey: @"PLAIN_TEXT_MESSAGE_FONT_NAME"]
						   trait: NSFixedPitchFontMask
						   size: [[NSUserDefaults standardUserDefaults]
							   floatForKey: @"PLAIN_TEXT_MESSAGE_FONT_SIZE"]]
				      forKey: NSFontAttributeName];
	}
      else
	{
	  plainTextMessageAttributes = textMessageAttributes;
	}
      
      aAttributedString = [Utilities attributedStringWithString: aString
				     attributes: plainTextMessageAttributes];
    }

  RELEASE(textMessageAttributes);
  
 done:
  // We haven't found a text/plain part. Report this as a bug in GNUMail.app for not supporting
  // the other kind of parts.
  if ( !aAttributedString )
    {
      aAttributedString = [Utilities attributedStringWithString: 
				       _(@"No text part found. Please report this bug since GNUMail.app doesn't support this kind of part.")
				     attributes: nil];
    }
  
  return aAttributedString;
}


//
//
//
+ (void) _launchExternalProgram: (NSDictionary *) theDictionary
{
  NSAutoreleasePool *pool;
  NSMutableString *aPath;
  NSTask *aTask;
  
  pool = [[NSAutoreleasePool alloc] init];

  aTask = [[NSTask alloc] init];
  AUTORELEASE(aTask);
  
  // Construct the launch path. If is an MacOS app wrapper, add the full path to the binary
  aPath = [[NSMutableString alloc] initWithString: [theDictionary objectForKey: @"command"]];
#ifdef MACOSX
  if ( [[NSWorkspace sharedWorkspace] isFilePackageAtPath: aPath])
    {
      [aPath appendString: [NSString stringWithFormat: @"/Contents/MacOS/%@", 
				     [[aPath stringByDeletingPathExtension] lastPathComponent]]];
    }
#endif

  // Launch task and look for exceptions
  NS_DURING
    {
      [aTask setLaunchPath: aPath];
      [aTask setArguments: [theDictionary objectForKey: @"arguments"]];
      [aTask launch];
      [aTask waitUntilExit];
    }
  NS_HANDLER
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"There was an error launching the external program (%@) for opening this attachment (%@)!\nException: %@"),
		      _(@"OK"),
		      NULL,
		      NULL,
		      aPath, [[theDictionary objectForKey: @"arguments"] objectAtIndex: 0], localException);
    }
  NS_ENDHANDLER

  // Remove the temporary file.
  [[NSFileManager defaultManager] removeFileAtPath: [theDictionary objectForKey: @"file"]
				  handler: nil];



  RELEASE(aPath);
  RELEASE(pool);
}

@end


//
// C functions
//
NSComparisonResult CompareVersion(NSString *theCurrentVersion, NSString *theLatestVersion)
{
  NSArray *currentVersion, *latestVersion;
  int i, currentCount, latestCount;
  
  currentVersion = [theCurrentVersion componentsSeparatedByString: @"."];
  currentCount = [currentVersion count];

  latestVersion = [theLatestVersion componentsSeparatedByString: @"."];
  latestCount = [latestVersion count];
  
  //
  // Note: Version 1.0 < 1.0.0 < 1.0.1
  //
  for (i = 0; i < currentCount && i < latestCount; i++)
    {
      int c, l;
      
      c = [[currentVersion objectAtIndex: i] intValue];
      l = [[latestVersion objectAtIndex: i] intValue];
      
      if ( c < l )
	{
	  return NSOrderedAscending;
	}
      
      if ( c > l )
	{
	  return NSOrderedDescending;
	}
    }
  
  if ( i < latestCount )
    {
      return NSOrderedAscending;
    }
  
  return NSOrderedSame;
}


//
//
//
NSString *GNUMailTemporaryDirectory()
{
  NSFileManager *aFileManager;
  NSString *aString;
  
  aString = [NSString stringWithFormat: @"%@/GNUMail", NSTemporaryDirectory()];
  aFileManager = [NSFileManager defaultManager];

  if ( ![aFileManager fileExistsAtPath: aString] )
    {
      [aFileManager createDirectoryAtPath: aString  
		    attributes: [aFileManager fileAttributesAtPath: NSTemporaryDirectory()  traverseLink: NO]];
    }

  return aString;
}


//
//
//
NSString *GNUMailUserLibraryPath()
{
  return [NSString stringWithFormat: @"%@/GNUMail", 
		   [NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, NSUserDomainMask, YES)
						       objectAtIndex: 0] ];
}


//
//
//
NSString *GNUMailVersion()
{
#ifdef MACOSX
  return [[[NSBundle bundleForClass: [[NSApp delegate] class]] infoDictionary] objectForKey: @"CFBundleVersion"];
#else
  return [[[NSBundle mainBundle] infoDictionary] objectForKey: @"ApplicationRelease"];
#endif
}
