/*
**  PreferencesWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "PreferencesWindowController.h"

#include "ConsoleWindowController.h"
#include "Constants.h"
#include "GNUMail.h"
#include "MailWindowController.h"

#include <GNUMail/GNUMailBundle.h>
#include <GNUMail/PreferencesModule.h>

#ifndef MACOSX
#include "PreferencesWindow.h"
#endif

static PreferencesWindowController *singleInstance = nil;

@implementation PreferencesWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
  NSDictionary *allPreferences;

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  PreferencesWindow *preferencesWindow;

  preferencesWindow = [[PreferencesWindow alloc] initWithContentRect: NSMakeRect(250,250,472,520)
						 styleMask: NSTitledWindowMask
						 backing: NSBackingStoreRetained
						 defer: NO];

  self = [super initWithWindow: preferencesWindow];
  
  [preferencesWindow layoutWindow];
  [preferencesWindow setDelegate: self];
 
  // We link our outlets
  matrix = preferencesWindow->matrix;
  scrollView = preferencesWindow->scrollView;
  box = preferencesWindow->box;
  expert = preferencesWindow->expert;  

  RELEASE(preferencesWindow);
#endif

  // We copy the current preferences to a volatile domain
  allPreferences = [NSDictionary dictionaryWithDictionary: [[NSUserDefaults standardUserDefaults] dictionaryRepresentation]];
  // FIXME - This cause a segfault on OS X when reloading a 2nd time the preferences panel
#ifndef MACOSX
  [[NSUserDefaults standardUserDefaults] removeVolatileDomainForName: @"PREFERENCES"];
#endif
  [[NSUserDefaults standardUserDefaults] setVolatileDomain: allPreferences  forName: @"PREFERENCES"];

  // We set our window title
  [[self window] setTitle: _(@"Preferences Panel")];

  // We set our mode
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"PREFERENCES_MODE"] )
    {
      [self setMode: [[NSUserDefaults standardUserDefaults] integerForKey: @"PREFERENCES_MODE"]];
    }
  else
    {
      [self setMode: MODE_STANDARD];
    }


#ifndef MACOSX
  // We adjust our window size
  [self _adjustWindowSize];
#endif

  // We initialize our matrix with the standard modules
  [self initializeWithStandardModules];

  // We then add our additional modules
  [self initializeWithOptionalModules];

  // We finally set our autosave window frame name and restore the one from the user's defaults.
  // FIXME - Reactivate later.
  //[[self window] setFrameAutosaveName: @"PreferencesWindow"];
  //[[self window] setFrameUsingName: @"PreferencesWindow"];

  return self;
}


//
//
//
- (void) dealloc
{
  [self _releaseLoadedBundles];

  RELEASE(allModules);

  [super dealloc];
}


//
// delegate methods
//
- (void) windowDidLoad
{
  // We maintain an array of opened modules
  allModules = [[NSMutableDictionary alloc] initWithCapacity: 10];
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{  
  // We save our current preferences setting (expert/normal)
  [[NSUserDefaults standardUserDefaults] setInteger: mode  forKey: @"PREFERENCES_MODE"];
  AUTORELEASE(self);
  singleInstance = nil;
}


//
//
//
- (void) handleCellAction: (id) sender
{  
  id aModule;
  
  aModule = [allModules objectForKey: [[matrix selectedCell] title]];

  if ( aModule )
    {
      [self addModuleToView: aModule];
    }
  else
    {
      NSLog(@"Unable to load the %@ bundle.", [[matrix selectedCell] title]);
    }
}


//
// action methods
//
- (IBAction) cancelClicked: (id) sender
{
  [self close];
}


//
//
//
- (IBAction) expertClicked: (id) sender
{
  NSString *titleOfSelectedCell;
  
  titleOfSelectedCell = [[matrix selectedCell] stringValue];

  if ( [self mode] == MODE_STANDARD )
    {
      [self setMode: MODE_EXPERT];
    }
  else
    {
      [self setMode: MODE_STANDARD];  
    }

  // We initialize our matrix with the standard modules
  [self initializeWithStandardModules];

  // We then add our additional modules
  [self initializeWithOptionalModules];

  // We reselect the right cell
  [self _selectCellWithTitle: titleOfSelectedCell];

#ifndef MACOSX
  // We adjust our window size
  [self _adjustWindowSize];
#endif
}



//
//
//
- (IBAction) saveAndClose: (id) sender
{
  [self savePreferences: nil];
  [self close];
}


//
//
//
- (IBAction) savePreferences: (id) sender
{
  NSArray *allNames;
  id<PreferencesModule> aModule;
  int i;

  allNames = [allModules allKeys];

  for (i = 0; i < [allNames count]; i++)
    {
      aModule = [allModules objectForKey: [allNames objectAtIndex: i]];

      if ( [aModule hasChangesPending] )
	{
	  [aModule saveChanges];
	}
    }
  
  [[NSUserDefaults standardUserDefaults] synchronize];
}


//
// other methods
//
- (void) addModuleToView: (id<PreferencesModule>) aModule
{    
  if ( aModule == nil )
    {
      return;
    }

  if ( [box contentView] != [aModule view] )
    {
#ifdef MACOSX
      NSRect aFrame;
      float delta;

      // Compute the height delta between the current module view and the new module view
      aFrame = [[box contentView] frame];
      delta = [[aModule view] frame].size.height - aFrame.size.height;

      [box setContentView: blankView];

      // Resize the box
      aFrame = [box frame];
      aFrame.origin.y -= delta;
      aFrame.size.height += delta;

      [box setFrame: aFrame];

      // Resize the window
      aFrame = [NSWindow contentRectForFrameRect: [[self window] frame]
			 styleMask: [[self window] styleMask]];

      aFrame.origin.y -= delta;
      aFrame.size.height += delta;

      aFrame = [NSWindow frameRectForContentRect: aFrame
			 styleMask: [[self window] styleMask]];
      [[self window] setFrame: aFrame  display: YES  animate: YES];
#endif      
      [box setContentView: [aModule view]];
      [box setTitle: [aModule name]];
    }
}


//
//
//
- (void) initializeWithStandardModules
{
  if ( [self mode] == MODE_STANDARD )
    {
      [matrix renewRows: 1  columns: 6];

      [self _initializeModuleWithName: @"Account"   atIndex: 0];
      [self _initializeModuleWithName: @"Viewing"   atIndex: 1];
      [self _initializeModuleWithName: @"Receiving" atIndex: 2];
      [self _initializeModuleWithName: @"Compose"   atIndex: 3];
      [self _initializeModuleWithName: @"Fonts"     atIndex: 4];
      [self _initializeModuleWithName: @"Colors"    atIndex: 5];
    }
  else
    {
      [matrix renewRows: 1  columns: 10];
      [self _initializeModuleWithName: @"Account"   atIndex: 0];
      [self _initializeModuleWithName: @"Viewing"   atIndex: 1];
      [self _initializeModuleWithName: @"Sending"   atIndex: 2];
      [self _initializeModuleWithName: @"Receiving" atIndex: 3];
      [self _initializeModuleWithName: @"Compose"   atIndex: 4];
      [self _initializeModuleWithName: @"Fonts"     atIndex: 5];
      [self _initializeModuleWithName: @"Colors"    atIndex: 6];
      [self _initializeModuleWithName: @"MIME"      atIndex: 7];
      [self _initializeModuleWithName: @"Filtering" atIndex: 8];
      [self _initializeModuleWithName: @"Advanced"  atIndex: 9];
    }
}


//
//
//
- (void) initializeWithOptionalModules
{
  int i;
  
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasPreferencesPanel] )
	{
	  id<PreferencesModule> aModule;
	  NSButtonCell *aButtonCell;
	  int column;

	  // We add our column
	  [matrix addColumn];
	  column = ([matrix numberOfColumns] - 1);

	  // We get our Preferences module and we add it to our matrix.
	  aModule = (id<PreferencesModule>)[aBundle preferencesModule];

	  [allModules setObject: aModule
		      forKey: [aModule name]];
	  
	  aButtonCell = [matrix cellAtRow: 0
				column: column];
	  
	  [aButtonCell setTag: column];
	  [aButtonCell setTitle: [aModule name]];
#ifdef MACOSX
	  [aButtonCell setFont: [NSFont systemFontOfSize: 10]];
          [aButtonCell setButtonType: NSOnOffButton];
          [aButtonCell setBezelStyle: 0]; // not documented but I assume it means None :)
          [aButtonCell setBordered: NO];
          [aButtonCell setGradientType: NSGradientNone];
#else
	  [aButtonCell setFont: [NSFont systemFontOfSize: 8]];
#endif
	  [aButtonCell setImage: [aModule image]];
	}
    }

  [matrix sizeToCells];
  [matrix setNeedsDisplay: YES];
}


//
// access/mutation methods
//
- (NSMatrix *) matrix
{
  return matrix;
}


//
//
//
- (int) mode
{
  return mode;
}


//
//
//
- (void) setMode: (int) theMode 
{
  mode = theMode;

  if ( [self mode] == MODE_EXPERT )
    {
      [expert setTitle: _(@"Standard")];
    }
  else
    {
      [expert setTitle: _(@"Expert")];
    }
}



//
// class methods
//
+ (id) singleInstance
{
  if ( !singleInstance )
    {
      singleInstance = [[PreferencesWindowController alloc] initWithWindowNibName: @"PreferencesWindow"];

      // We select the first cell in our matrix
      [[singleInstance matrix] selectCellAtRow: 0  column: 0];
      [singleInstance handleCellAction: [singleInstance matrix]];
    }
  else
    {
      return nil;
    }

  return singleInstance;
}

@end


//
// Private interface
//
@implementation PreferencesWindowController (Private)

- (void) _adjustWindowSize
{
  NSRect frame, rect;

  frame = [[self window] frame];

  // We adjust our box
  rect = [box frame];
  [box setFrame: NSMakeRect(rect.origin.x, rect.origin.y, 455, 260)];
  
  // We adjust our matrix/scrollView
  [matrix setFrame: NSMakeRect(rect.origin.x, rect.origin.y + 265, 455, 86)];
  [scrollView setFrame: NSMakeRect(rect.origin.x, rect.origin.y + 265, 455, 86)];
  [matrix setCellSize: NSMakeSize(64,64)];
  
  // We adjust our window
  [[self window] setFrame: NSMakeRect(frame.origin.x, frame.origin.y, 472, 420)
		 display: YES];
}


//
//
//
- (void) _initializeModuleWithName: (NSString *) theName
			   atIndex: (int) theIndex
{
  id<PreferencesModule> aModule;
  NSButtonCell *aButtonCell;

  aModule = [self _instanceForBundleWithName: theName];

  if ( !aModule )
    {
      NSLog(@"Unable to initialize module %@", theName);
      return;
    }

  [allModules setObject: aModule
	      forKey: _(theName)];
  
  aButtonCell = [matrix cellAtRow: 0
			column: theIndex];
      
  [aButtonCell setTag: theIndex];
  [aButtonCell setTitle: [aModule name]];
#ifdef MACOSX
  [aButtonCell setFont: [NSFont systemFontOfSize: 10]];
#else
  [aButtonCell setFont: [NSFont systemFontOfSize: 8]];
#endif
  [aButtonCell setImage: [aModule image]];
}


//
// Can be:  $GNUSTEP_INSTALATION_DIR/{Local,Network,System}/Apps/GNUMail.app/Resources or
//     or:  ANY_OTHER_DIRECTORY/GNUMail/GNUMail.app/Resources
//
- (id) _instanceForBundleWithName: (NSString *) theName
{
  NSString *aString;
  NSBundle *aBundle;
  Class aClass;

#ifdef MACOSX
  aString = [[[NSBundle mainBundle] builtInPlugInsPath] 
	      stringByAppendingPathComponent: [theName 
						stringByAppendingPathExtension: @"prefs"]];
#else
  aString = [[[NSBundle mainBundle] resourcePath] stringByDeletingLastPathComponent];
  aString = [aString stringByDeletingPathExtension];

  // We verify if we must load the bundles in the GNUstep's Local, Network or System dir.
  // We must also be careful in case poeple are using GNUstep with --enable-flattened
  if ( [aString hasSuffix: @"/Applications/GNUMail"] )
    {
      aString = [NSString stringWithFormat: @"%@/Library/GNUMail/%@.prefs",
			  [[aString stringByDeletingLastPathComponent] 
			    stringByDeletingLastPathComponent],
			  theName];
    }
  else
    {
      aString = [NSString stringWithFormat: @"%@/Bundles/%@/%@.prefs",
			  [aString stringByDeletingLastPathComponent],
			  theName, theName];
    }
#endif
  
  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									   _(@"Loading preferences bundle at path %@."),
									 aString]];
  
  aBundle = [NSBundle bundleWithPath: aString];

  aClass = [aBundle principalClass];
  
  if ( [aClass conformsToProtocol: @protocol(PreferencesModule)] )
    {
      return [aClass singleInstance];
    }

  // Reached in case of an error.
  return nil;
}


//
//
//
- (void) _releaseLoadedBundles
{
  NSEnumerator *aEnumerator;
  id aModule;
  
  aEnumerator = [allModules objectEnumerator];
 
  while ( (aModule = [aEnumerator nextObject]) )
    {
      RELEASE(aModule);
    }
}


//
//
//
- (void) _selectCellWithTitle: (NSString *) theTitle
{
  int i;

  for (i = 0; i < [matrix numberOfColumns]; i++)
    {
      if ( [theTitle isEqualToString: [[matrix cellAtRow: 0  column: i] stringValue]] )
        {
          [matrix selectCellAtRow: 0  column: i];
          [self addModuleToView: [allModules objectForKey: theTitle]];
          return;
        }
    }

  // No cell found, we select the first one and perform the action
  [[singleInstance matrix] selectCellAtRow: 0  column: 0];
  [singleInstance handleCellAction: matrix];
  [self addModuleToView: [allModules objectForKey: [[matrix selectedCell] title]]];
}


@end
