/*
**  MailWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "MailWindowController.h"

#include "ConsoleWindowController.h"
#include "Constants.h"
#include "EditWindowController.h"
#include "ExtendedCell.h"
#include "ExtendedOutlineView.h"
#include "ExtendedTableView.h"
#include "ExtendedWindow.h"
#include "GNUMail.h"
#include "GNUMail/GNUMailBundle.h"
#include "LabelWidget.h"

#ifndef MACOSX
#include "MailWindow.h"
#else
#include "ImageTextCell.h"
#endif

#include "Filter.h"
#include "FilterManager.h"
#include "LocalMailDelivery.h"
#include "MailboxManagerCache.h"
#include "MailboxManagerController.h"
#include "MailHeaderCell.h"
#include "MessageViewWindowController.h"
#include "MimeType.h"
#include "MimeTypeManager.h"
#include "NSUserDefaults+Extensions.h"
#include "Utilities.h"

#include "GNUMail+TaskManager.h"
#include "Task.h"
#include "TaskManager.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Container.h>
#include <Pantomime/Flags.h>
#include <Pantomime/Folder.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/InternetAddress.h>
#include <Pantomime/LocalFolderCacheManager.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/POP3Folder.h>
#include <Pantomime/POP3Store.h>
#include <Pantomime/NSData+Extensions.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/POP3CacheManager.h>
#include <Pantomime/TCPConnection.h>
#include <Pantomime/URLName.h>


//
// FIXME: Remove those categories when GNUstep -gui is fixed.
//
#ifndef MACOSX
@implementation NSActionCell (Correction)
- (void) setControlView: (NSView *) cv
{
  _control_view = cv;
}
@end

@implementation NSTableColumn (Correction)
- (void) setTableView: (NSTableView*) aTableView
{
  _tableView = aTableView;
  if ([_dataCell respondsToSelector: @selector(setControlView:)])
    {
      [(id)_dataCell setControlView: aTableView];
    }
}
@end
#endif


//
//
//
@implementation MailWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
  int scrollerSize;

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  MailWindow *aMailWindow;
  
  aMailWindow = [[MailWindow alloc] initWithContentRect: NSMakeRect(150,100,612,595)
				    styleMask: NSClosableWindowMask|NSTitledWindowMask|
				    NSMiniaturizableWindowMask|NSResizableWindowMask
				    backing: NSBackingStoreRetained
				    defer: NO];

  self = [super initWithWindow: aMailWindow];
  
  [aMailWindow layoutWindow];
  [aMailWindow setDelegate: self];

  // We link our outlets
  tableScrollView = aMailWindow->tableScrollView;
  textScrollView = aMailWindow->textScrollView;
  splitView = aMailWindow->splitView;
  textView = aMailWindow->textView;
  icon = aMailWindow->icon;
  delete = aMailWindow->delete;
  mailboxes = aMailWindow->mailboxes;
  compose = aMailWindow->compose;
  forward = aMailWindow->forward;
  reply = aMailWindow->reply;
  addresses = aMailWindow->reply;
  find = aMailWindow->find;
  get = aMailWindow->get;
  label = (NSTextField *)aMailWindow->label;

  RELEASE(aMailWindow);
#endif

  // We set our window title
  [[self window] setTitle: @""];

  
  //
  // We create all table columns
  //
  statusColumn = [[NSTableColumn alloc] initWithIdentifier: @"Status"];
  [statusColumn setEditable: NO];
  [statusColumn setResizable: YES];
  [[statusColumn headerCell] setStringValue: _(@"Status")];
  [statusColumn setMinWidth: 50];
  [statusColumn setMaxWidth: 50];

  idColumn = [[NSTableColumn alloc] initWithIdentifier: @"#"];
  [idColumn setEditable: NO];
  [idColumn setResizable: YES];
  [[idColumn headerCell] setStringValue: @"#"];
  [idColumn setMinWidth: 40];
  [idColumn setMaxWidth: 40];
  
  dateColumn = [[NSTableColumn alloc] initWithIdentifier: @"Date"];
  [dateColumn setEditable: NO];
  [dateColumn setResizable: YES];
  [[dateColumn headerCell] setStringValue: _(@"Date")];
  [dateColumn setMinWidth: 85];
  [[dateColumn headerCell] setAlignment: NSLeftTextAlignment];
  
  fromColumn = [[NSTableColumn alloc] initWithIdentifier: @"From"];
  [fromColumn setEditable: NO];
  [fromColumn setResizable: YES];
  [[fromColumn headerCell] setStringValue: _(@"From")];
#ifdef MACOSX
  [fromColumn setMinWidth: 120];
  [fromColumn setWidth: 120];
#else
  [fromColumn setMinWidth: 155];
#endif
  [[fromColumn headerCell] setAlignment: NSLeftTextAlignment];
  
  subjectColumn = [[NSTableColumn alloc] initWithIdentifier: @"Subject"];
  [subjectColumn setEditable: NO];
  [subjectColumn setResizable: YES];
  [[subjectColumn headerCell] setStringValue: _(@"Subject")];
  [subjectColumn setMinWidth: 195];
#ifdef MACOSX
  [subjectColumn setWidth: 260];
#endif
  [subjectColumn setWidth: 195];
  [[subjectColumn headerCell] setAlignment: NSLeftTextAlignment];

  sizeColumn = [[NSTableColumn alloc] initWithIdentifier: @"Size"];
  [sizeColumn setEditable: NO];
  [sizeColumn setResizable: YES];
  [[sizeColumn headerCell] setStringValue: _(@"Size")];
  [sizeColumn setMinWidth: 50];
  [sizeColumn setMaxWidth: 70];
  [[sizeColumn headerCell] setAlignment: NSRightTextAlignment];


  // We create our mail header cell and our observer for resize notifications
  mailHeaderCell = [[MailHeaderCell alloc] init];
  
  [[NSNotificationCenter defaultCenter] addObserver: mailHeaderCell
					selector: @selector (resize:) 
					name: @"NSViewFrameDidChangeNotification" 
					object: textView];

  // We initialize our standard fonts
  [self _initializeFonts];
  
  // We set our custom cell
  [statusColumn setDataCell: AUTORELEASE([[ExtendedCell alloc] init])];

  // We set our data view type
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"MAILWINDOW_DATA_VIEW_TYPE"] <= 1 )
    {
      [self setDataViewType: TYPE_TABLEVIEW];
    }
  else
    {
      [self setDataViewType: TYPE_OUTLINEVIEW];
    }      

  // We load our accessory views
  [self _loadAccessoryViews];

  // We restore our split view knob position
  [self _restoreSplitViewSize];

  // We restore our sorting order
  [self _restoreSortingOrder];

  // We set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"MailWindow"];
  [[self window] setFrameUsingName: @"MailWindow"];
    
  // Set the sizes for the scroll bars
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"SCROLLER_SIZE"] == NSRegularControlSize )
    {
      scrollerSize = NSRegularControlSize;
    }
  else
    {
      scrollerSize = NSSmallControlSize;
    }
  
  [[tableScrollView verticalScroller] setControlSize: scrollerSize];
  [[tableScrollView horizontalScroller] setControlSize: scrollerSize];
  [[textScrollView verticalScroller] setControlSize: scrollerSize];
  [[textScrollView horizontalScroller] setControlSize: scrollerSize];
  
  // Set our textview to non-editable
  [textView setEditable: NO];
  
  // Set ourselves up as the delegate
  [textView setDelegate: self];

  // We set our animation ivar to nil
  animation = nil;

  // Add ourselves as an observer to reload message list notification
  [[NSNotificationCenter defaultCenter] 
    addObserver: self
    selector: @selector(reloadMessageList:)
    name: ReloadMessageList
    object: nil];
 
  return self;
}


//
//
//
#ifdef MACOSX
- (void) awakeFromNib
{
  NSToolbar *aToolbar;
  
  aToolbar = [[NSToolbar alloc] initWithIdentifier: @"MailWindowToolbar"];
  [aToolbar setDelegate: self];
  [aToolbar setAllowsUserCustomization: YES];
  [aToolbar setAutosavesConfiguration: YES];
  [[self window] setToolbar: aToolbar];
  RELEASE(aToolbar);
}
#endif


//
//
//
- (void) dealloc
{
  NSDebugLog(@"MailWindowController: -dealloc");
  
  // We first remove all our observers
  [[NSNotificationCenter defaultCenter] 
    removeObserver: mailHeaderCell
    name: @"NSViewFrameDidChangeNotification" 
    object: textView];
  
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ReloadMessageList
    object: nil];  

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: ReplyToMessageWasSuccessful
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: FontValuesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: TableColumnsHaveChanged
    object: nil];

  // We release our standard fonts
  RELEASE(deletedMessageFont);
  RELEASE(recentMessageFont);
  RELEASE(seenMessageFont);

  // We relaser our header cell and our array holding all MessageViewWindow:s
  RELEASE(mailHeaderCell);
  RELEASE(allMessageViewWindowControllers);

  // We release our NSDrawer's extended outline view
#ifdef MACOSX
  RELEASE(delete);
  RELEASE(get);
  RELEASE(outlineView);
#endif

  // We cleanup the ivars used for our dataView's data source  
  TEST_RELEASE(allMessages);
  TEST_RELEASE(allContainers);
  TEST_RELEASE(currentOrder);
  TEST_RELEASE(previousOrder);

  // We release our table columns
  RELEASE(statusColumn);
  RELEASE(idColumn);
  RELEASE(dateColumn);
  RELEASE(fromColumn);
  RELEASE(subjectColumn);
  RELEASE(sizeColumn);

  // We finally release our folder
  RELEASE(folder);
  
  [super dealloc];
}


//
// action methods
//
- (IBAction) doubleClickedOnDataView: (id) sender
{
  // We ignore a double-click on a table column
  if ( (sender != self) && [dataView clickedRow] < 0 )
    {
      return;
    }
  
  // If we are in the Draft folder, we re-opened the selected mail for editing
  if ( [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: folder]  
		  isEqualTo: @"DRAFTSFOLDERNAME"] )
    {
      [[NSApp delegate] restoreDraft: nil];
    }
  // Or, we just 'reply' to the mail or open it in a separate window.
  else
    {
      if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"DOUBLECLICKACTION"] && 
	   [[NSUserDefaults standardUserDefaults] integerForKey: @"DOUBLECLICKACTION"] == 0 )
      	{
	  [self replyMessage: nil];
	}
      else if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"DOUBLECLICKACTION"] ||
		[[NSUserDefaults standardUserDefaults] integerForKey: @"DOUBLECLICKACTION"] == 1 )
      	{
	  [self viewMessageInWindow: nil];
	  [self updateStatusLabel];
	}
    }
}


// 
//
//
- (IBAction) deleteMessage: (id) sender
{
  // If we have no element (or no selection), we return!
  if ( [folder count] == 0 || [dataView numberOfSelectedRows] == 0 )
    {
      NSBeep();
      return;
    } 
  else
    {
      NSArray *selectedRows;      
      Message *theMessage;
      Flags *theFlags;
      NSNumber *aRow;

      int i, last_row, first_row;
      BOOL firstFlagOfList;
      
      selectedRows = [[dataView selectedRowEnumerator] allObjects];
      firstFlagOfList = NO;
      first_row = -1;
      last_row = 0;
      
      for (i = 0; i < [selectedRows count]; i++)
	{
	  aRow = [selectedRows objectAtIndex: i];

	  if ( first_row < 0 ) first_row = [aRow intValue];

	  theMessage = [self messageAtRow: [aRow intValue]];

	  // We set the flag Deleted (or not) to the message
	  theFlags = [[theMessage flags] copy];

          if ( i == 0 )
            {
              // This is the first message of the list we want to {un}delete
              // We must save the flag.
              if ( [theFlags contain: DELETED] && ![sender isKindOfClass: [ExtendedWindow class]] )
                {
                  [theFlags remove: DELETED];
                  firstFlagOfList = NO;
                }
              else
                {
                  [theFlags add: DELETED];
                  firstFlagOfList = YES;
                }

	      [(GNUMail*)[NSApp delegate] updateMenuItemsForMessage: theMessage];
            }
          else
            {
              if ( !firstFlagOfList && [theFlags contain: DELETED] && ![sender isKindOfClass: [ExtendedWindow class]] )
                { 
                  [theFlags remove: DELETED];
                }
              else if ( firstFlagOfList && (![theFlags contain: DELETED]) )
                {
                  [theFlags add: DELETED];
                }
            }
	  
	  last_row = [aRow intValue];

	  // If we are {un}deleting more than one message,
	  // lets optimize things (mosly for IMAP)
	  if ( [selectedRows count] > 1 )
	    {
	      [[self folder] setFlags: theFlags
			     messages: [self selectedMessages]];
	      last_row = [[selectedRows lastObject] intValue];
	      i = [selectedRows count];
	      break;
	    }
	  
	  // We finally set our new flags
	  [theMessage setFlags: theFlags];
	  RELEASE(theFlags);
	}
            
      // We always refresh our dataView after a delete operation
      [self reloadMessageList: nil];
      
      // We now select the row right after the message(s) beeing deleted
      // If we're in reverse order, we select the previous row, otherwise, the next one.
      if ( sender == delete || sender == self || [sender isKindOfClass: [ExtendedWindow class]] )
	{
	  int count, row_to_select;
	  
	  count = [dataView numberOfRows];
	  row_to_select = last_row;

	  if ( count > 0 )
	    {
	      if ( isReverseOrder )
		{
		  row_to_select--;

		  if ( [[self folder] showDeleted] )
		    {
		      row_to_select = --first_row;
		    }
		}
	      else
		{
		  // If we show the mails marked as DELETE, we jump to the next mail.
		  // Otherwise, we just try to show the same index again.
		  if ( [[self folder] showDeleted] )
		    {
		      row_to_select = ++last_row;
		    }
		  if (i > 1)
		    {
		      row_to_select = (last_row - i);
		    }
		}
	      
	      // We ensure ourself row_to_select is inbounds.
	      if ( row_to_select >= count )
		{
		  row_to_select = (count - 1);
		}
	      else if (row_to_select < 0)
		{
		  row_to_select = 0;
		}

	      [dataView selectRow: row_to_select 
			byExtendingSelection: NO];
	      [dataView scrollRowToVisible: row_to_select];
	    }

	  // We at least post our notification
	  [[NSNotificationCenter defaultCenter]
            postNotificationName: SelectionOfMessageHasChanged
            object: nil
            userInfo: nil];
	}


      // We update the status label
      [self updateStatusLabel];
    }
}


//
// This method selects the message after the current
// selected message and displays it.
//
- (IBAction) nextMessage: (id) sender
{
  int row;
  
  row = [dataView selectedRow];
  
  if (row == -1 ||
      row >= ([dataView numberOfRows] - 1) ) 
    {
      NSBeep();
    }
  else
    {
      [dataView selectRow: (row+1)  byExtendingSelection: NO];
      [dataView scrollRowToVisible: (row+1)];
      [dataView setNeedsDisplay: YES];
    }
}


//
//
//
- (IBAction) nextUnreadMessage: (id) sender
{
  if ( [self dataViewType] == TYPE_OUTLINEVIEW )
    {
      // FIXME: TODO
    }
  else
    {
      int row, i;

      row = [dataView selectedRow];

      if ( row == -1 )
	{
	  NSBeep();
	}
      else
	{
	  for (i = row; i < [[self allMessages] count]; i++)
	    {
	      if ( ![[[self messageAtRow: i] flags] contain: SEEN] )
		{
		  [dataView selectRow: i  byExtendingSelection: NO];
		  [dataView scrollRowToVisible: i];
		  [dataView setNeedsDisplay: YES];
		  return;
		}
	    }

	  // We haven't found an unread message, simply call -nextMessage
	  [self nextMessage: sender];
	}
    }
}


//
//
//
- (IBAction) firstMessage: (id) sender
{
  if ([dataView numberOfRows] > 0)
    {
      [dataView selectRow: 0  byExtendingSelection: NO];
      [dataView scrollRowToVisible: 0];
      [dataView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) lastMessage: (id) sender
{
  if ([dataView numberOfRows] > 0)
    {
      [dataView selectRow: ([dataView numberOfRows] - 1)  byExtendingSelection: NO];
      [dataView scrollRowToVisible: ([dataView numberOfRows] - 1)];
      [dataView setNeedsDisplay: YES];
    }
}


//
//
//
- (IBAction) pageDownMessage: (id) sender
{
  NSRect aRect;
  double origin;

  aRect = [textScrollView documentVisibleRect];
  origin = aRect.origin.y;
  
  aRect.origin.y += aRect.size.height - [textScrollView verticalPageScroll];
  [textView scrollRectToVisible: aRect];
  
  aRect = [textScrollView documentVisibleRect];
  
  if (aRect.origin.y == origin)
    {
      [self nextMessage: nil];
    } 
}


//
//
//
- (IBAction) pageUpMessage: (id) sender
{
  NSRect aRect;
  double origin;

  aRect = [textScrollView documentVisibleRect];
  origin = aRect.origin.y;

  aRect.origin.y -= aRect.size.height - [textScrollView verticalPageScroll];
  [textView scrollRectToVisible: aRect];

  aRect = [textScrollView documentVisibleRect];

  if (aRect.origin.y == origin)
    {
      [self previousMessage: nil];
    }
}


//
// This method selects the message before the current
// selected message and displays it.
//
- (IBAction) previousMessage: (id) sender
{
  int row;

  row = [dataView selectedRow];
  
  if (row > 0)
    {
      [dataView selectRow: (row-1)  byExtendingSelection: NO];
      [dataView scrollRowToVisible: (row-1)];
      [dataView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) previousUnreadMessage: (id) sender
{
  if ( [self dataViewType] == TYPE_OUTLINEVIEW )
    {
      // FIXME: TODO
    }
  else
    {
      int row, i;

      row = [dataView selectedRow];

      if ( row == -1 )
	{
	  NSBeep();
	}
      else
	{
	  for (i = row; i >= 0; i--)
	    {
	      if ( ![[[self messageAtRow: i] flags] contain: SEEN] )
		{
		  [dataView selectRow: i  byExtendingSelection: NO];
		  [dataView scrollRowToVisible: i];
		  [dataView setNeedsDisplay: YES];
		  return;
		}
	    }

	  // We haven't found an unread message, simply call -previousMessage
	  [self previousMessage: sender];
	}
    }
}


//
//
//
- (IBAction) collapseItem: (id) sender
{
  int row;

  row = [dataView selectedRow];

  if ( [self dataViewType] == TYPE_OUTLINEVIEW &&
       row > 0 &&
       [dataView numberOfSelectedRows] == 1 )
    {
      [dataView collapseItem: [dataView itemAtRow: row]]; 
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) expandItem: (id) sender
{
  int row;
  
  row = [dataView selectedRow];

  if ( [self dataViewType] == TYPE_OUTLINEVIEW &&
       row > 0 &&
       [dataView numberOfSelectedRows] == 1 )
    {
      [dataView expandItem: [dataView itemAtRow: row]]; 
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) forwardMessage: (id) sender
{
  Message *aMessage;
  
  if ( [dataView selectedRow] < 0 ) 
    {
      NSBeep();
      return;
    }
  
  // We obtain the selected entry in the table for the forward information
  aMessage = [self selectedMessage];
  
  // Forward this message
  [Utilities forwardMessage: aMessage];
}


//
// If the sender is the application's delegate, we reply to all
// recipients. It's only invoked that way from GNUMail: -replyAllMessage.
//
- (IBAction) replyMessage: (id) sender
{
  Message *aMessage;
  
  if ( [dataView selectedRow] < 0 ) 
    {
      NSBeep();
      return;
    }
  
  // We obtain the selected entry in the table for the reply informations
  aMessage = [self selectedMessage];
  
  // Invoke the message utility class method to reply to this message.
  [Utilities replyToSender: aMessage
	     folder: [self folder]
	     replyToAll: (sender == [NSApp delegate] ? YES : NO)];  
}


//
// This opens a new window and displays the message in it.
//
- (IBAction) viewMessageInWindow: (id) sender
{
  MessageViewWindowController *aViewWindowController;
  Message *aMessage;
  
  if ( [dataView selectedRow] < 0 )
    {
      NSBeep();
      return;
    }
  
  // We obtain the selected entry in the table for the display informations
  aMessage = [self selectedMessage];
  
  // We create our window controller
  aViewWindowController = [[MessageViewWindowController alloc] initWithWindowNibName: @"MessageViewWindow"];
  
  // set our message and folder
  [aViewWindowController setMessage: aMessage];
  [aViewWindowController setFolder: [self folder]];
  [aViewWindowController setMailWindowController: self];  
  
  // show the window and the message
  [aViewWindowController showWindow: self];
  [allMessageViewWindowControllers addObject: aViewWindowController];

  [Utilities showMessage: aMessage
	     target: [aViewWindowController textView]
	     showAllHeaders: [self showAllHeaders]];

  // On MacOS X, if the mail window is not active, double-clicking on an unselected message
  // causes the message not to draw itself in the new window that pops up.
  // Let's force it to draw itself.
#ifdef MACOSX
  [[aViewWindowController textView] setNeedsDisplay: YES];
#endif

}


//
// This method returns the folder associated to this MailWindow.
//
- (Folder *) folder
{
  return folder;
}


//
// This method sets the folder associated to this MailWindow.
//
// NOTE: This method DOES NOT close the folder. It'll release it
//       but the folder SHOULD BE CLOSED FIRST.
//
- (void) setFolder: (Folder *) theFolder
{
  NSUserDefaults *aUserDefaults;
  
  aUserDefaults = [NSUserDefaults standardUserDefaults];
	
  RETAIN(theFolder);
  RELEASE(folder);
  folder = theFolder;

  // We close all MessageViewWindows
  [self _closeAllMessageViewWindows];

  // We now set the window title
  if ( !folder )
    {
      [[self window] setTitle: _(@"No mailbox selected")];
      [self dataViewShouldReloadData];
    }
  else if ( [folder isKindOfClass: [LocalFolder class]] )
    {
      [[self window] setTitle: [NSString stringWithFormat: _(@"Local - %@"), [folder name]] ];
    }
  else
    {
      IMAPStore *aStore;
      
      aStore = (IMAPStore *)[folder store];
      
      // We set the delegate on our folder
      [(IMAPFolder *)folder setDelegate: self];

      [[self window] setTitle: [NSString stringWithFormat: _(@"IMAP on %@ - %@"), [aStore name], 
					 [folder name]] ];
    }
  
  if ( [folder count] > 0 ) 
    {
      int i, count;

      if ( [self dataViewType] == TYPE_OUTLINEVIEW )
	{
	  // We do the message threading
	  [folder thread];
	}

      // We reload the data our of dataView (since it now has some)
      [self dataViewShouldReloadData];
      
      count = [dataView numberOfRows];
      
      // We search for the first message without the SEEN flag and we select it
      for (i = 0; i < count; i++)
	{
	  Message *aMessage;
	  
	  aMessage = [self messageAtRow: i];
	  
	  if ( ![[aMessage flags] contain: SEEN] )
	    {
	      break;
	    }
	}
      
      if (i == count)
	{
	  if ( isReverseOrder )
	    {
	      i = 0;
	    }
	  else
	    {
	      i--;
	    }
	}
      
      // We scroll to and select our found row
      [dataView scrollRowToVisible: i];
      [dataView selectRow: i 
		byExtendingSelection: NO];
    }

  // We verify if we need to rename our From column to "To" in case we are in the Sent
  // or Drafts folder.
  if ( [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: folder]  
		  isEqualTo: @"DRAFTSFOLDERNAME"] ||
       [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: folder]  
		  isEqualTo: @"SENTFOLDERNAME"] )
    {
      [[fromColumn headerCell] setStringValue: _(@"To")];
      draftsOrSentFolder = YES;
    }
  else
    {
      [[fromColumn headerCell] setStringValue: _(@"From")];
      draftsOrSentFolder = NO;
    }

  [[dataView headerView] setNeedsDisplay: YES];

  [self _restoreImage];

  // We always update our status label, if we have or not messages in our folder
  [self updateStatusLabel];
}


//
// NSOutlineView's delegate/datasource methods
//
- (id) outlineView: (NSOutlineView *) outlineView
	     child: (int) index
	    ofItem: (id) item
{  
  if ( !item || item == allContainers )
    {
      //NSDebugLog(@"outlineView: child: ofItem: Index = %d", index);
      return [allContainers objectAtIndex: index];
    }
  
  if ( [item isKindOfClass: [Container class]] )
    {
      //NSDebugLog(@"outlineView: child: ofItem: returning child, index = %d", index);
      return [(Container *)item childAtIndex: index];
   }
  
  //NSDebugLog(@"returning nil");
  
  return nil;
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
    isItemExpandable: (id) item
{
  if ( !item || item == allContainers )
    {
      return YES;
    }

  if ( [item isKindOfClass: [Container class]] )
    {
      return ([(Container *)item count] > 0 ? YES : NO);
    }

  return NO;
}


//
//
//
- (int)        outlineView: (NSOutlineView *) outlineView 
    numberOfChildrenOfItem: (id) item
{
  if ( !item || item == allContainers )
    {
      //NSDebugLog(@"returning count = %d", [allContainers count]);
      return [allContainers count];
    }

  if ( [item isKindOfClass: [Container class]] )
    {
      //NSDebugLog(@"Child count = %d", [(Container *)item count]);
      return [(Container *)item count];
    }

  //NSDebugLog(@"returning 0");
  
  return 0;
}


//
//
//
- (id)         outlineView: (NSOutlineView *) outlineView 
 objectValueForTableColumn: (NSTableColumn *) aTableColumn 
		    byItem: (id) item
{
  if ( [item isKindOfClass: [Container class]] )
    {
      Message *aMessage;

      aMessage = ((Container *)item)->message;

      if ( !aMessage )
	{
	  return @"";
	}
      
      return [self _objectValueForTableColumn: aTableColumn
		   message: aMessage
		   item: item];
    }

  // Should never be reached...
  return @"";
}


//
//
//
- (void) outlineView: (NSOutlineView *) aOutlineView
     willDisplayCell: (id) aCell 
      forTableColumn: (NSTableColumn *) aTableColumn 
		item: (id) item
{
  [self _willDisplayCellForMessage: ((Container *)item)->message
	cell: aCell
	tableColumn: aTableColumn
	row: 0
	item: item];
}


//
//
//
#ifdef MACOSX
- (void)    outlineView: (NSOutlineView *) aOutlineView 
 willDisplayOutlineCell: (id) aCell 
	 forTableColumn: (NSTableColumn *) aTbleColumn 
		   item: (id)item
{
  [aCell setImage: [NSImage imageNamed: @"sort_right"]];
  [aCell setAlternateImage: [NSImage imageNamed: @"sort_down"]];
  [aCell setImagePosition: NSImageAbove];
}
#endif


//
//
//
- (void) outlineViewSelectionDidChange: (NSNotification *) notification
{
  [self _selectionDidChange];
}


//
//
//
- (void) outlineViewItemDidCollapse: (NSNotification *) notification
{
  [self _zeroIndexOffset];
}


//
//
//
- (void) outlineViewItemDidExpand: (NSNotification *) notification
{
  [self _zeroIndexOffset];
}


//
// NSOutlineView drag and drop methods
//
- (BOOL) outlineView: (NSOutlineView *) aOutlineView
	  writeItems: (NSArray *) items
	toPasteboard: (NSPasteboard *) pboard
{
  NSMutableArray *aMutableArray;
  int i;

  aMutableArray = [[NSMutableArray alloc] initWithCapacity: [items count]];
  AUTORELEASE(aMutableArray);
  
  for (i = 0; i < [items count]; i++)
    {
      [aMutableArray addObject: [NSNumber numberWithInt: [aOutlineView rowForItem: [items objectAtIndex: i]]]];
    }
  
  return [self tableView: (NSTableView *)aOutlineView
	       writeRows: aMutableArray
	       toPasteboard: pboard];
}


//
//
//
- (NSDragOperation) outlineView: (NSOutlineView *) aOutlineView 
		   validateDrop: (id<NSDraggingInfo>) info 
		   proposedItem: (id) item
	     proposedChildIndex: (int) index
{
  return [self tableView: (NSTableView *)aOutlineView
	       validateDrop: info
	       proposedRow: [dataView rowForItem: item]
	       proposedDropOperation: 0];
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) aOutlineView 
	  acceptDrop: (id<NSDraggingInfo>) info
		item: (id) item 
	  childIndex: (int) index
{
  return [self tableView: (NSTableView *) aOutlineView
	       acceptDrop: info
	       row: [dataView rowForItem: item]
	       dropOperation: 0];
}


//
//
//
// NSTableView delegate/datasource methods
//
//
//
- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [folder count];
}


//
//
//
- (id)           tableView: (NSTableView *) aTableView
 objectValueForTableColumn: (NSTableColumn *) aTableColumn
                       row: (int) rowIndex
{
  Message *aMessage;

  aMessage = [allMessages objectAtIndex: rowIndex];

  return [self _objectValueForTableColumn: aTableColumn
	       message: aMessage
	       item: nil];
}


//
//
//
- (void) tableView: (NSTableView *)aTableView
   willDisplayCell: (id)aCell
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  [self _willDisplayCellForMessage: [allMessages objectAtIndex: rowIndex]
	cell: aCell
	tableColumn: aTableColumn
	row: rowIndex
	item: nil];
}


//
//
//
- (void) tableViewSelectionDidChange: (NSNotification *) aNotification
{
  [self _selectionDidChange];
}


//
//
//
- (void)   tableView: (NSTableView *) aTableView
 didClickTableColumn: (NSTableColumn *) aTableColumn
{
  NSString *newOrder;

  newOrder = [aTableColumn identifier];
  
  if ( ([newOrder isEqualToString: @"#"] == NO)
       && ([newOrder isEqualToString: @"Date"] == NO)
       && ([newOrder isEqualToString: @"From"] == NO)
       && ([newOrder isEqualToString: @"Subject"] == NO)
       && ([newOrder isEqualToString: @"Size"] == NO) )
    {
      return;
    }
  
  [aTableView setHighlightedTableColumn: aTableColumn];
  [self setPreviousOrder: [self currentOrder]];

  if ([[self currentOrder] isEqualToString: newOrder])
    {
      if (isReverseOrder == YES)
	{
	  NSDebugLog(@"non reverse");
	}
      else
	{
	  NSDebugLog(@"reverse");
	}
      
      isReverseOrder = !isReverseOrder;
    }
  else
    {
      [self setCurrentOrder: newOrder];
      isReverseOrder = NO;
    }
  
  [self _setIndicatorImageForTableColumn: aTableColumn];
  
  [[NSUserDefaults standardUserDefaults] setObject: [self currentOrder]
					 forKey: @"SORTINGORDER"];

  [[NSUserDefaults standardUserDefaults] setInteger: isReverseOrder
					 forKey: @"SORTINGSTATE"];

  [self dataViewShouldReloadData];
}


//
//
//
-  (void) textView: (NSTextView *) aTextView
     clickedOnCell: (id <NSTextAttachmentCell>) attachmentCell
	    inRect: (NSRect) cellFrame
	   atIndex: (unsigned) charIndex
  
{
  [Utilities clickedOnCell: attachmentCell
	     inRect: cellFrame
	     atIndex: charIndex
	     sender: self];
}


//
//
//
- (BOOL) textView: (NSTextView *) textView
    clickedOnLink: (id) link 
	  atIndex: (unsigned) charIndex
{
  NSDebugLog(@"Opening %@...", [link description]);
  return [[NSWorkspace sharedWorkspace] openURL: link];
}


//
// NSTableDataSource Drag and drop
//
- (BOOL) tableView: (NSTableView *) aTableView
	 writeRows: (NSArray *) rows
      toPasteboard: (NSPasteboard *) pboard
{
  NSMutableArray *propertyList;
  Message *aMessage;
  int i, row;

  propertyList = [[NSMutableArray alloc] initWithCapacity: [rows count]];
  
  for (i = 0; i < [rows count]; i++)
    {
      // For each implicated row, retrieve message and insert its
      // raw representation in a dictionary with two keys: 'flags' and 'message'
      NSMutableDictionary *aDictionary;
      NSData *flags, *rawSource;
      
      aDictionary = [[NSMutableDictionary alloc] initWithCapacity: 2];
      row = [[rows objectAtIndex: i] intValue];
      aMessage = [allMessages objectAtIndex: row];
      
      flags = [NSArchiver archivedDataWithRootObject: [aMessage flags]];
      [aDictionary setObject: flags forKey: @"flags"];

      rawSource = [NSData dataWithData: [aMessage rawSource]];
      [aDictionary setObject: rawSource forKey: @"message"];
      
      [propertyList addObject: aDictionary];
      RELEASE(aDictionary);
    }

  // Set property list of paste board
  [pboard declareTypes: [NSArray arrayWithObject: MessagePboardType] owner: self];
  [pboard setPropertyList: propertyList forType: MessagePboardType];
  RELEASE(propertyList);
  
  return YES;
}


//
// NSTableDataSource Drag and drop
//
- (NSDragOperation) tableView: (NSTableView *) aTableView
		 validateDrop: (id <NSDraggingInfo>) info
		  proposedRow: (int) row
	proposedDropOperation: (NSTableViewDropOperation) operation

{
  if ([info draggingSource] == dataView)
    {
      // We don't allow drag'n'drop to the same dataView
      return NSDragOperationNone;
    }

  if ([info draggingSourceOperationMask] & NSDragOperationGeneric)
    {
      return NSDragOperationGeneric;
    }
  else if ([info draggingSourceOperationMask] & NSDragOperationCopy)
    {
      return NSDragOperationCopy;
    }
  else
    {
      return NSDragOperationNone;
    }
}


//
// NSTableDataSource Drag and drop
//
- (BOOL) tableView: (NSTableView *) aTableView
	acceptDrop: (id <NSDraggingInfo>) info
	       row: (int) row
     dropOperation: (NSTableViewDropOperation) operation
{
  NSArray *propertyList;
  id sourceDelegate;
  int i, count;
  
  if ([info draggingSource] == dataView)
    {
      // We don't allow drag'n'drop to the same dataView
      return NO;
    }
  
  // We retrieve property list of messages from paste board
  propertyList = [[info draggingPasteboard] propertyListForType: MessagePboardType];

  if ( !propertyList )
    {
      return NO;
    }

  count = [propertyList count];
  
  for (i = 0; i < count; i++)
    {
      // We retrieve message from property list
      NSDictionary *aDictionary;
      Flags *flags;
      NSData *rawSource;
      
      aDictionary = (NSDictionary*)[propertyList objectAtIndex: i];
      
      rawSource = [aDictionary objectForKey: @"message"];
      
      flags = (Flags*)[NSUnarchiver unarchiveObjectWithData: (NSData*)[aDictionary objectForKey: @"flags"]];
      
      if (rawSource && flags)
	{
	  if ( ![[MailboxManagerController singleInstance] transferMessageFromRawSource: rawSource
							   flags: flags
							   toFolderWithName: [[self folder] name]
							   orToFolder: [self folder]
							   store: [[self folder] store]] )
	    {
	      // The transfer failed; we exit the loop
	      break;
	    }
	}
    }
  
  if (count > 0)
    {
      // We refresh the table view
      [self dataViewShouldReloadData];
      
      // We update the status label
      [self updateStatusLabel];
  
      // If the message is moved (and not copied),
      // we inform the dragging source that the selected messages have been transferred
      sourceDelegate = [[info draggingSource] delegate];
      
      // We mark messages as transferred if append succeed
      if (count == i &&
	  [info draggingSourceOperationMask] & NSDragOperationGeneric && 
	  [sourceDelegate respondsToSelector: @selector(setSelectedMessagesAsTransferred)])
	{
	  [sourceDelegate setSelectedMessagesAsTransferred];
	}
    }
  
  return YES;
}


//
//
// MailWindowController delegate methods
//
//
- (BOOL) windowShouldClose: (id) sender
{
  int choice;

  // We first verify if the animation is running, if it is,
  // we are currently sending a mail so we inform the user about this.
  if ( animation && [animation isValid] )
    {
      NSRunInformationalAlertPanel(_(@"Closing..."),
				   _(@"GNUMail.app is currently receiving E-Mails. Please wait."),
				   _(@"OK"), 
				   NULL, 
				   NULL);
      return NO;
    }

  // If it is an IMAP folder, we offer the user the option to close the connection
  // to the IMAP store
  if ( [[self folder] isKindOfClass: [IMAPFolder class]] )
    {
      IMAPStore *aStore;

      aStore = (IMAPStore *)[[self folder] store];
      choice = NSRunAlertPanel( [NSString stringWithFormat: _(@"Closing connection on %@ @ %@..."), [aStore username], [aStore name]],
			       _(@"Would you like to close the connection with the IMAP server?"),
			       _(@"Cancel"), // default
			       _(@"Yes"),    // alternate
			       _(@"No"));    // other return
      
      // We don't want to close the window
      if ( choice == NSAlertDefaultReturn )
	{
	  return NO;
	}
      // Yes we want to close it, and we also want to close the connection with the IMAP Store
      else if ( choice == NSAlertAlternateReturn )
	{
	  shouldCloseStore = YES;
	  return YES;
	}
      
    }
  
  shouldCloseStore = NO;
      
  return YES;
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  NSMutableArray *columnSizes, *shownTableColumns;
  NSString *theIdentifier;
  int i;
  
  //
  // Ideally, this should be handled by the NSTableView's autosave feature.
  // But it seems to be messing with the dynamic addition and removal of columns.
  //
  // Check whether we have archived the column sizes
  //
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MAIL_WINDOW_TABLE_COLUMN_SIZES"] )
    {
      columnSizes = [NSMutableArray arrayWithArray: [[NSUserDefaults standardUserDefaults] 
						      objectForKey: @"MAIL_WINDOW_TABLE_COLUMN_SIZES"]];
    }
  else
    {
      columnSizes = [[NSMutableArray alloc] initWithCapacity: 6];
      for (i = 0; i < 6; i++)
	{
	  [columnSizes addObject: [NSNumber numberWithFloat: 0.0]];
	}
      AUTORELEASE(columnSizes);
    }
  
  [columnSizes replaceObjectAtIndex: 0  withObject: [NSNumber numberWithFloat: [statusColumn width]]];
  [columnSizes replaceObjectAtIndex: 1  withObject: [NSNumber numberWithFloat: [idColumn width]]];
  [columnSizes replaceObjectAtIndex: 2  withObject: [NSNumber numberWithFloat: [dateColumn width]]];
  [columnSizes replaceObjectAtIndex: 3  withObject: [NSNumber numberWithFloat: [fromColumn width]]];
  [columnSizes replaceObjectAtIndex: 4  withObject: [NSNumber numberWithFloat: [subjectColumn width]]];
  [columnSizes replaceObjectAtIndex: 5  withObject: [NSNumber numberWithFloat: [sizeColumn width]]];
  
  [[NSUserDefaults standardUserDefaults] setObject: columnSizes  
					 forKey: @"MAIL_WINDOW_TABLE_COLUMN_SIZES"];
  
  // We also save the table column's order
  shownTableColumns = [[NSMutableArray alloc] init];

  for (i = 0; i < [[dataView tableColumns] count]; i++)
    {
      theIdentifier = [[[dataView tableColumns] objectAtIndex: i] identifier];
      [shownTableColumns addObject: ([theIdentifier isEqualToString: @"#"] ? @"Number" : (id)theIdentifier)];
    }
  
  [[NSUserDefaults standardUserDefaults] setObject: shownTableColumns
					 forKey: @"SHOWNTABLECOLUMNS"];
  RELEASE(shownTableColumns);
  

  // We save the height of our textScrollView
  [[NSUserDefaults standardUserDefaults] setInteger: [textScrollView bounds].size.height
					 forKey: @"TEXTSCROLLVIEW_HEIGHT"];
  
  // We close all MessageViewWindows.
  [self _closeAllMessageViewWindows];
  
  // We update our last mail window on top if it was the current selected one
  // OR if we decided to re-use the MailWindow - we must set it to nil
  if ( [GNUMail lastMailWindowOnTop] == [self window] || 
       ([[NSUserDefaults standardUserDefaults] objectForKey: @"REUSE_MAILWINDOW"] &&
	[[[NSUserDefaults standardUserDefaults] objectForKey: @"REUSE_MAILWINDOW"] intValue] == NSOnState) )
    {
      [GNUMail setLastMailWindowOnTop: nil];
    }
  
  // We update our current super view for bundles (we set it to nil) and we
  // inform our bundles that the viewing view will be removed from the super view
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  [aBundle setCurrentSuperview: nil];

	  if ( [aBundle viewingViewAccessoryType] == ViewingViewTypeHeaderCell )
	    {
	      [aBundle viewingViewAccessoryWillBeRemovedFromSuperview: mailHeaderCell];
	    }
	  else
	    {
	      [aBundle viewingViewAccessoryWillBeRemovedFromSuperview: [[self window] contentView]];
	    }
	}
    }


  // If we must compact the mbox on close, let's do it rigth now.
  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"COMPACT_MAILBOX_ON_CLOSE"] )
    {
      if ( !IMAPSTORE_IS_DISCONNECTED([folder store]) )
	{
	  [folder expunge: NO];
	}

      // We call this method just to update our cache, even if we do a little more work for nothing,
      // it's worth it :-)
      [self updateStatusLabel];
    }
  
  // We definitively close our folder.
  if ( !IMAPSTORE_IS_DISCONNECTED([folder store]) ) 
    {
      [folder close];
    }

  // We add a message to our Console
  if ( [folder isKindOfClass: [LocalFolder class]] )
    {
      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									       _(@"Closed local folder %@."),
									     [[self folder] name]]];
    }
  else
    {
      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									       _(@"Closed IMAP folder %@ on %@."),
									     [[self folder] name], 
									     [(IMAPStore *)[[self folder] store] name]]];
    }

  // We clear our 'Save Attachment' menu
  [(GNUMail *)[NSApp delegate] removeAllItemsFromMenu: [(GNUMail *)[NSApp delegate] saveAttachmentMenu]];

  // We remove our window from our list of opened windows
  [GNUMail removeMailWindow: [self window]];

  // We update our dock's menu, on OS X
#ifdef MACOSX
  [[NSApp delegate] updateDockMenu];
#endif

  // We finally verify if we must close our IMAP Store.
  if ( shouldCloseStore )
    {
      NSString *aUsername, *aServerName;

      aServerName = [(IMAPStore *)[[self folder] store] name];
      aUsername = [(IMAPStore *)[[self folder] store] username];

      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									       _(@"Closing IMAP connection on %@."),
									     aServerName]];
  
      [(IMAPStore *)[[self folder] store] close];
      [[MailboxManagerController singleInstance] setStore: nil
						 name: aServerName
						 username: aUsername];

      
#ifndef MACOSX
      // FIXME - We should only collapse the item...
      [[[MailboxManagerController singleInstance] outlineView] reloadData];
#endif
    }

#ifdef MACOSX
  // We finally unset the mailbox manager's current outline view
  [[MailboxManagerController singleInstance] setCurrentOutlineView: nil];
#endif

  AUTORELEASE(self);
}


//
//
//
- (void) windowDidLoad
{
#ifdef MACOSX
  //
  // We set up our NSDrawer's contentView.
  //
  NSTableColumn *mailboxColumn, *messagesColumn;
  NSScrollView *scrollView;
  id aCell;
    
  mailboxColumn = [[NSTableColumn alloc] initWithIdentifier: @"Mailbox"];
  [mailboxColumn setEditable: NO];
  [[mailboxColumn headerCell] setStringValue: _(@"Mailbox")];

  aCell =  [[ImageTextCell alloc] init];
  [mailboxColumn setDataCell: aCell];
  AUTORELEASE(aCell);

  messagesColumn = [[NSTableColumn alloc] initWithIdentifier: @"Messages"];
  [messagesColumn setEditable: NO];
  [[messagesColumn headerCell] setStringValue: _(@"Messages")];
  
  outlineView = [[ExtendedOutlineView alloc] initWithFrame: NSZeroRect];
  [outlineView addTableColumn: mailboxColumn];
  [outlineView addTableColumn: messagesColumn];
  [outlineView setOutlineTableColumn: mailboxColumn];
  [outlineView setDrawsGrid: NO];
  [outlineView setIndentationPerLevel: 10];
  [outlineView setAutoresizesOutlineColumn: YES];
  [outlineView setIndentationMarkerFollowsCell: YES];
  [outlineView setAllowsColumnSelection: NO];
  [outlineView setAllowsColumnReordering: NO];
  [outlineView setAutoresizesAllColumnsToFit: YES];
  [outlineView sizeLastColumnToFit];
  [outlineView setDataSource: [MailboxManagerController singleInstance]];
  [outlineView setDelegate: [MailboxManagerController singleInstance]];
  [outlineView setTarget: [MailboxManagerController singleInstance]];
  [outlineView setDoubleAction: @selector(open:)];

  // We register the outline view for dragged types
  [outlineView registerForDraggedTypes: [NSArray arrayWithObject: MessagePboardType]];
  
  // We set our autosave name for our outline view
  [outlineView setAutosaveName: @"MailboxManager"];
  [outlineView setAutosaveTableColumns: YES];

  scrollView = [[NSScrollView alloc] initWithFrame: NSZeroRect];
  [scrollView setDocumentView: outlineView];
  [scrollView setHasHorizontalScroller: NO];
  [scrollView setHasVerticalScroller: YES];
  [scrollView setBorderType: NSBezelBorder];
  [scrollView setAutoresizingMask: NSViewWidthSizable|NSViewHeightSizable];
  
  [drawer setContentView: scrollView];
  RELEASE(scrollView);
  RELEASE(mailboxColumn);
  RELEASE(messagesColumn);
  
  //
  // We set up our various toolbar items
  //
  [icon setTarget: [NSApp delegate]];
  [icon setAction: @selector(showConsoleWindow:)];

  [mailboxes setTarget: [NSApp delegate]];
  [mailboxes setAction: @selector(showMailboxManager:)];
  
  [addresses setTarget: [NSApp delegate]];
  [addresses setAction: @selector(showAddressBook:)];
  
  [find setTarget: [NSApp delegate]];
  [find setAction: @selector(showFindWindow:)];

  [dataView setDoubleAction: @selector(doubleClickedOnDataView:)];
#endif

  // We finally add our observer if the filters changed
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(filtersHaveChanged:)
    name: FiltersHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(updateMessageFlags:)
    name: ReplyToMessageWasSuccessful
    object: nil];
  
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_fontValuesHaveChanged)
    name: FontValuesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_reloadTableColumns:)
    name: TableColumnsHaveChanged
    object: nil];

  // We initialize some values
  [self setShowAllHeaders: NO];
  [GNUMail setLastMailWindowOnTop: [self window]];
  
  // We add our window from our list of opened windows
  [GNUMail addMailWindow: [self window]];

  // We initialize some ivars
  allMessageViewWindowControllers = [[NSMutableArray alloc] init];
  shouldCloseStore = NO;
}


//
//
//
- (void) windowDidBecomeKey: (NSNotification *) aNotification
{
  int i;
  
  // We set the last window on top
  [GNUMail setLastMailWindowOnTop: [self window]];
  
  // We set the current superview of our bundle having providing
  // a viewing accessory view.
  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;
      
      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  [aBundle setCurrentSuperview: [[self window] contentView]];
	}
    }

  [[NSApp delegate] setEnableSaveInDraftsMenuItem: NO];
  [[NSApp delegate] setShowRawSourceMenuItem: ![self showRawSource]];
  [[NSApp delegate] updateThreadOrUnthreadMenuItem: ([self dataViewType] == TYPE_TABLEVIEW)];

#ifdef MACOSX
  // We set the current outline view for our mailbox manager
  [[MailboxManagerController singleInstance] setCurrentOutlineView: outlineView];

  // We update our dock's menu, on OS X
  [[NSApp delegate] updateDockMenu];
#endif

  [[self window] makeFirstResponder: dataView];
}


//
//
//
- (Message *) selectedMessage
{
  int index;
  
  index = [dataView selectedRow];
  
  if (index < 0)
    {
      return nil;
    }

  return [self messageAtRow: index];
}


//
//
//
- (NSArray *) selectedMessages
{
  if ( [dataView numberOfSelectedRows] == 0 )
    {
      NSBeep();
    }
  else
    {
      NSMutableArray *aMutableArray;
      NSEnumerator *anEnumerator;
      NSNumber *aRow;
      
      Message *aMessage;

      aMutableArray = [[NSMutableArray alloc] initWithCapacity: [dataView numberOfSelectedRows]];
      
      anEnumerator = [dataView selectedRowEnumerator];
      
      while ( (aRow = [anEnumerator nextObject]) )
	{
	  aMessage = [self messageAtRow: [aRow intValue]];
	  
	  // We guard ourself against broken threads
	  if ( aMessage )
	    {
	      [aMutableArray addObject: aMessage];
	    }
	}
      
      return AUTORELEASE(aMutableArray);
    }
  
  // Reached in case of error.
  return nil;
}


//
//
//
- (id) dataView
{
  return dataView;
}


//
//
//
- (int) dataViewType
{
  return dataViewType;
}


//
//
//
- (void) setDataViewType: (int) theType
{
  id aDataView;
  NSRect aRect;

#ifndef MACOSX
  NSSize aSize;
#endif

  dataViewType = theType;
  aRect = [tableScrollView frame];
  
  // We set the data source / delegate / target of our previous
  // view to nil - just to be safe.
  aDataView = [tableScrollView documentView];

  if ( aDataView )
    {
      [aDataView setDataSource: nil]; 
      [aDataView setDelegate: nil];
      [aDataView setTarget: nil];
    }

  switch ( theType )
    {
    case 2:
      //
      // NSOutlineView
      //
      // FIXME - Should we really use an ExtendedOutlineView?
      dataView = [[ExtendedOutlineView alloc] initWithFrame: aRect];
#ifndef MACOSX
      [dataView setCollapsedImage: [NSImage imageNamed: @"sort_right.tiff"]];
      [dataView setExpandedImage: [NSImage imageNamed: @"sort_down.tiff"]];
#endif
      [dataView addTableColumn: statusColumn];
      [dataView addTableColumn: idColumn];
      [dataView addTableColumn: dateColumn];
      [dataView addTableColumn: fromColumn];
      [dataView addTableColumn: subjectColumn];
      [dataView addTableColumn: sizeColumn];;
      [dataView setOutlineTableColumn: subjectColumn];
      [dataView setIndentationPerLevel: 10];
      [dataView setIndentationMarkerFollowsCell: YES]; 
      [dataView setAutoresizesOutlineColumn: YES];
      break;

    case 1:
    default:
      //
      // NSTableView
      //
      dataView = [[ExtendedTableView alloc] initWithFrame: aRect];
      [dataView addTableColumn: statusColumn];
      [dataView addTableColumn: idColumn];
      [dataView addTableColumn: dateColumn];
      [dataView addTableColumn: fromColumn];
      [dataView addTableColumn: subjectColumn];
      [dataView addTableColumn: sizeColumn]; 
    }

  // General methods that apply to both of them
  [dataView setDrawsGrid: NO];
  [dataView setAllowsColumnSelection: NO];
  [dataView setAllowsColumnReordering: YES];
  [dataView setAllowsColumnResizing: YES];
  [dataView setAllowsEmptySelection: YES];
  [dataView setAllowsMultipleSelection: YES];
  [dataView setIntercellSpacing: NSZeroSize];
  [dataView sizeLastColumnToFit];

  [dataView setDataSource: self]; 
  [dataView setDelegate: self];
  [dataView setTarget: self];
  [dataView setDoubleAction: @selector(doubleClickedOnDataView:)];

  // We add it to our document view and we can now safely release it
  // since the scrollview will retain it.
  [tableScrollView setDocumentView: dataView];

  // We register the table view for dragged types
  [dataView registerForDraggedTypes: [NSArray arrayWithObject: MessagePboardType]];

  // We set any vertical mouse motion has being dragging
  [dataView setVerticalMotionCanBeginDrag: NO];
   
  // FIXME: Should we really make that work under OS X?
  //        Find the right * y ratio
#ifndef MACOSX
  // We set the table row height, depending on the current font
  aSize = [seenMessageFont maximumAdvancement];
  [dataView setRowHeight: aSize.height];
#endif

  // We load the right set of columns
  [self _reloadTableColumns: self];

  // We set our table view background color
  if ( [[NSUserDefaults standardUserDefaults] colorForKey: @"MAILWINDOW_TABLE_COLOR"] )
    {
      [dataView setBackgroundColor: [[NSUserDefaults standardUserDefaults]
				      colorForKey: @"MAILWINDOW_TABLE_COLOR"]];
      [tableScrollView setBackgroundColor: [[NSUserDefaults standardUserDefaults]
					     colorForKey: @"MAILWINDOW_TABLE_COLOR"]];
    }

  RELEASE(dataView);
}


//
//
//
- (NSTextView *) textView
{
  return textView;
}


//
//
//
- (NSString *) currentOrder
{
  return currentOrder;
}


//
//
//
- (void) setCurrentOrder: (NSString *) theCurrentOrder
{
  RETAIN(theCurrentOrder);
  RELEASE(currentOrder);
  currentOrder = theCurrentOrder;
}


//
//
//
- (NSString *) previousOrder
{
  return previousOrder;
}


//
//
//
- (void) setPreviousOrder: (NSString *) thePreviousOrder
{
  RETAIN(thePreviousOrder);
  RELEASE(previousOrder);
  previousOrder = thePreviousOrder;
}


//
//
//
- (MailHeaderCell *) mailHeaderCell
{
  return mailHeaderCell;
}


//
//
//
- (NSMutableArray *) allMessageViewWindowControllers
{
  return allMessageViewWindowControllers;
}


//
//
//
- (Message *) messageAtRow: (int) theRow
{
  if ( [dataView isKindOfClass: [NSOutlineView class]] )
    {
      Container *aContainer;
      
      aContainer = (Container *)[dataView itemAtRow: theRow];
      
      return aContainer->message;
    }
  else
    {
      return [allMessages objectAtIndex: theRow];
    }
  
  // Never reached.
  return nil;
}


//
//
//
- (NSArray *) allMessages
{
  return allMessages;
}


//
//
//
- (NSArray *) allContainers
{
  return allContainers;
}


//
//
//
- (BOOL) showAllHeaders
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] )
    {
      return ([[[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWALLHEADERS"] intValue] == NSOnState ? YES
	      : showAllHeaders);
    }

  return showAllHeaders;
}


//
//
//
- (void) setShowAllHeaders: (BOOL) aBOOL
{
  showAllHeaders = aBOOL;
}

- (BOOL) showRawSource
{
  return showRawSource;
}

- (void) setShowRawSource: (BOOL) aBool
{
  showRawSource = aBool;
}


//
//
//
- (IBAction) filtersHaveChanged : (id) sender
{
  [dataView setNeedsDisplay: YES];
}


//
//
//
- (IBAction) updateMessageFlags: (id) sender
{
  if (sender)
    {
      Message *aMessage;
      
      aMessage = [[sender userInfo] objectForKey: @"Message"];

      if ( aMessage && [[self folder]->allMessages containsObject: aMessage] )
	{
	  Flags *theFlags;

	  theFlags = [[aMessage flags] copy];
	  [theFlags add: ANSWERED];
	  [aMessage setFlags: theFlags];
	  RELEASE(theFlags);

	  [dataView setNeedsDisplay: YES];
	}
    }
}


//
//
//
- (IBAction) getNewMessages: (id) sender
{
  // If the animation is running and this method is called, that means
  // that we want to stop the transfer.
  // FIXME: If the menu item has been clicked, we must not do anything
  if ( animation )
    {
      [[TaskManager singleInstance] stopCurrentTask];
      return;
    }
  
  [[NSApp delegate] checkForNewMail: sender
		    controller: self];
}


//
//
//
#ifdef MACOSX
- (IBAction) openOrCloseDrawer: (id) sender
{
  if ( [drawer state] == NSDrawerOpenState )
    {
      [drawer close];
    }
  else
    {
      [drawer open];
    }
}
#endif

//
//
//
- (void) startAnimation
{
#ifdef MACOSX 
  if ( [[[self window] toolbar] customizationPaletteIsRunning] )
    {
      return;
    }
#endif

  // If our animation is already started, we do nothing.
  if ( animation == nil )
    {
#ifdef MACOSX
      [get setImage: [NSImage imageNamed: @"stop_32.tiff"]];
      [get setLabel: _(@"Stop")];
      [progressIndicator startAnimation: self];
#else
      [get setImage: [NSImage imageNamed: @"stop_48.tiff"]];
      [get setTitle: _(@"Stop")];
#endif     

      animation_index = 1;
      animation = [NSTimer scheduledTimerWithTimeInterval: 0.1
			   target: self
			   selector: @selector(_updateAnimatedIcon:)
			   userInfo: nil
			   repeats: YES];
      RETAIN(animation);
    }
}


//
// 
//
- (void) stopAnimation
{
#ifdef MACOSX 
  if ( [[[self window] toolbar] customizationPaletteIsRunning] )
    {
      return;
    }
#endif

  if ( animation != nil )
    {
#ifdef MACOSX
      [get setImage: [NSImage imageNamed: @"retrieve_32.tiff"]];
      [get setLabel: _(@"Get")];
      [progressIndicator stopAnimation: self];
#else
      [get setImage: [NSImage imageNamed: @"retrieve_48.tiff"]];
      [get setTitle: _(@"Get")];
#endif
      [animation invalidate];
      DESTROY(animation);
      
      [self _restoreImage];
      [self updateStatusLabel];
    }
}



//
//
//
- (void) setSelectedMessagesAsTransferred
{
  NSEnumerator *anEnumerator;
  Message *theMessage;
  NSNumber *aRow;
  Flags *theFlags;
  
  anEnumerator = [dataView selectedRowEnumerator];
  
  while ( (aRow = [anEnumerator nextObject]) )
    {
      theMessage = [self messageAtRow: [aRow intValue]];
      theFlags = [[theMessage flags] copy];
      
      // We set the flag DELETED to the message
      [theFlags add: DELETED];
      [theMessage setFlags: theFlags];
      RELEASE(theFlags);
    }
  
  // We update the messages cache
  [[self folder] updateCache];

  // We refresh our dataView
  [self dataViewShouldReloadData];

  // We update the status label
  [self updateStatusLabel];
}


//
//
//
- (void) dataViewShouldReloadData
{
  if ( [self dataViewType] == TYPE_OUTLINEVIEW )
    {
      [self _outlineViewShouldReloadData];

      if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"AUTOMATICALLY_EXPAND_THREADS"] == NSOnState )
	{
	  int i;
	  
	  for (i = 0; i < [dataView numberOfRows]; i++)
	    {
	      [dataView expandItem: [dataView itemAtRow: i]  expandChildren: YES];
	    }
	}
    }
  else
    {
      [self _tableViewShouldReloadData];
    }
}


//
// 
//
- (void) updateStatusLabel
{
  NSString *aString, *aStoreName, *aUsername, *aSeparator, *aFolderName;
  NSEnumerator *enumerator;
  Message *aMessage;
  Flags *theFlags;
  id anObject;

  int totalSize, unreadCount, unreadSize, selectedSize, deletedCount, deletedSize;
  int i, count, aSize, numberOfSelectedRows;
  

  if ( animation )
    {
      return;
    }
  
  totalSize = unreadCount = unreadSize = deletedCount = deletedSize = 0;
  count = [folder count];
  for (i = 0; i < count; i++)
    {
      // FIXME: We should be sure to call [folder allMessages] and not
      //        simply use the allMessages ivar (of this class), especially
      //        when using message threading.
      aMessage = [[folder allMessages] objectAtIndex: i];
      theFlags = [aMessage flags];
      aSize = [aMessage size];
      totalSize += aSize;

      if (![theFlags contain: SEEN])
	{
	  unreadCount++;
	  unreadSize += aSize;
	}
      if ([theFlags contain: DELETED])
	{
	  deletedCount++;
	  deletedSize += aSize;
	}
    }
  
  numberOfSelectedRows = [dataView numberOfSelectedRows];
  selectedSize = 0;
  
  if (numberOfSelectedRows > 0)
    {
      enumerator = [dataView selectedRowEnumerator];
      
      while ((anObject = [enumerator nextObject]))
	{
	  aMessage = [self messageAtRow: [anObject intValue]];
	 
	  // We guard ourself against broken message threads
	  if ( aMessage )
	    {
	      selectedSize += [aMessage size];
	    }
	}
    }
  
  aString = [NSString stringWithFormat: _(@"%d messages (%dKB) - %d unread (%dKB) - %d selected (%0.1fKB) - %d deleted (%0.1fKB)"),
		      count, (totalSize/1024),
		      unreadCount, (unreadSize/1024), 
		      numberOfSelectedRows, ((float)selectedSize/(float)1024),
		      deletedCount, ((float)deletedSize/(float)1024)];
  
  [label setStringValue: aString];
  [label setNeedsDisplay: YES];

  // If the unread count is 0, set the application icon to the normal icon.
  if ( unreadCount == 0 )
    {
      [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail.tiff"]];
    }

  // We update our cache
  if ( [(id<NSObject>)[[self folder] store] isKindOfClass: [LocalStore class]] )
    {
      aStoreName = @"GNUMAIL_LOCAL_STORE";
      aUsername = NSUserName();
      aSeparator = @"/";
    }
  else
    {
      aStoreName = [(IMAPStore *)[[self folder] store] name];
      aUsername = [(IMAPStore *)[[self folder] store] username];
      aSeparator = [(IMAPStore *)[[self folder] store] folderSeparator];
    }

  aFolderName = [[self folder] name];

  [[[MailboxManagerController singleInstance] mailboxManagerCache]
    setAllValuesForStoreName: aStoreName
    folderName: [aFolderName stringByReplacingOccurrencesOfString: aSeparator
			     withString: @"/"]
    username: aUsername
    nbOfMessages: count
    nbOfUnreadMessages: unreadCount];

  [[[MailboxManagerController singleInstance] outlineView] setNeedsDisplay: YES];

  // We finally update our dock's menu, on OS X
#ifdef MACOSX
  [[NSApp delegate] updateDockMenu];
#endif
}


//
//
//
- (void) updateStatusLabelWithMessage: (NSString *) theString
{
  NSString *aString;

  aString = [NSString stringWithFormat: _(@"%d messages - %@"),  [folder count], theString];
  
  [label setStringValue: aString];
  [label setNeedsDisplay: YES];
}


//
// IMAPFolder's delegate methods se we are informed when messages were received,
// expunged or when the flags for messages have changed.
//

//
//
//
- (void) messagesWereReceived: (id) sender
{
  [self dataViewShouldReloadData];
  [self updateStatusLabel];

  // We set the application icon to GNUMail_Full.tiff
  [NSApp setApplicationIconImage: [NSImage imageNamed: @"GNUMail_Full.tiff"]];
}


//
//
//
- (void) messagesWereExpunged: (id) sender
{
  [self dataViewShouldReloadData];
  [self updateStatusLabel];
}


//
//
//
-(void) messagesFlagsHaveChanged: (id) sender
{
  [self dataViewShouldReloadData];
  [self updateStatusLabel];
}


//
// reload the message list
//
- (void) reloadMessageList: (NSNotification *) aNotification
{
  if ( [[self folder] showDeleted] )
    {
      NSDebugLog(@"Showing deleted messages...");
      [dataView setNeedsDisplay: YES];
    }
  else
    {
      NSDebugLog(@"NOT Showing deleted messages...");
      [[self folder] updateCache];
      [self dataViewShouldReloadData];
    }
}

@end


//
// Private interface for MailWindowContrller
//
@implementation MailWindowController (Private)

- (void) _closeAllMessageViewWindows
{
  int i;

  // No need to actually remove the object from the array since that will
  // already be done in MessageViewWindowController: -windowWillClose.
  for (i = ([allMessageViewWindowControllers count] - 1); i >= 0; i--)
    {
      [[allMessageViewWindowControllers objectAtIndex: i] close];
    }

}


//
//
//
- (void) _fontValuesHaveChanged
{
  [self _initializeFonts];
  [self _showMessage: self];
}


//
//
//
- (void) _initializeFonts
{
#ifndef MACOSX
  NSSize aSize;
#endif

  DESTROY(seenMessageFont);
  DESTROY(recentMessageFont);
  DESTROY(deletedMessageFont);

  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MESSAGE_LIST_FONT_NAME"] )
    {
      NSString *aName;
      int size;

      aName =  [[NSUserDefaults standardUserDefaults] stringForKey: @"MESSAGE_LIST_FONT_NAME"];
      size = [[[NSUserDefaults standardUserDefaults] objectForKey: @"MESSAGE_LIST_FONT_SIZE"] intValue];

      seenMessageFont = [Utilities fontFromFamilyName: aName  trait: NSUnboldFontMask  size: size];
      recentMessageFont = [Utilities fontFromFamilyName: aName  trait: NSBoldFontMask  size: size];
      deletedMessageFont = [[NSFontManager sharedFontManager] convertFont: seenMessageFont  toHaveTrait: NSItalicFontMask];
    }
  else
    {
      //
      // SEEN message's font
      //
#ifdef MACOSX
      seenMessageFont = [NSFont systemFontOfSize: [NSFont smallSystemFontSize]];
#else
      seenMessageFont = [NSFont systemFontOfSize: 0];
#endif

      //
      // "UNSEEN" message's font
      //
#ifdef MACOSX
      recentMessageFont = [NSFont boldSystemFontOfSize: [NSFont smallSystemFontSize]];
#else
      recentMessageFont = [NSFont boldSystemFontOfSize: 0];
#endif
      
      //
      // DELETED message's font
      //
#ifdef MACOSX
      deletedMessageFont = [[NSFontManager sharedFontManager] convertFont: [NSFont systemFontOfSize:
										     [NSFont smallSystemFontSize]]
							      toHaveTrait: NSItalicFontMask];
#else
      deletedMessageFont = [[NSFontManager sharedFontManager] convertFont: [NSFont systemFontOfSize: 0]
							      toHaveTrait: NSItalicFontMask];
#endif
    }
  
  // FIXME: Should we really make that work under OS X?
  //        Find the right * y ratio
#ifndef MACOSX
  // We set the table row height, depending on the current font
  aSize = [seenMessageFont maximumAdvancement];
  [dataView setRowHeight: aSize.height];
#endif

  RETAIN(seenMessageFont);
  RETAIN(recentMessageFont);
  RETAIN(deletedMessageFont);
}


//
//
//
- (void) _loadAccessoryViews
{
  int i, index;

  index = 0;

  for (i = 0; i < [[GNUMail allBundles] count]; i++)
    {
      id<GNUMailBundle> aBundle;

      aBundle = [[GNUMail allBundles] objectAtIndex: i];
      
      if ( [aBundle hasViewingViewAccessory] )
	{
	  id aView;
	  
	  aView = [aBundle viewingViewAccessory];
	  
	  if ( [aBundle viewingViewAccessoryType] == ViewingViewTypeHeaderCell )
	    {
	      NSDebugLog(@"Adding ViewingViewTypeHeaderCell type of Bundle...");
	      [mailHeaderCell addView: aView];
	    }
	  else
	    {
	      NSRect aRect;
	      
	      aRect = [find frame];
	      aRect.origin.x += 72;

	      [aView setAutoresizingMask: NSViewMinYMargin];
	      [aView setFrame: aRect];
	      
	      [[[self window] contentView] addSubview: aView];
	    }
	}

      // We also set the current superview
      [aBundle setCurrentSuperview: [[self window] contentView]];
    }
}


//
//
//
- (id) _objectValueForTableColumn: (NSTableColumn *) aTableColumn
                          message: (Message *) theMessage
			     item: (id) theItem
{
  InternetAddress *anInternetAddress;

  // We then provide the real data for our row
  if (aTableColumn == idColumn)
    {
      return [NSString stringWithFormat: @"%d", [theMessage messageNumber]];
    }
  else if (aTableColumn == dateColumn)
    {   
      NSCalendarDate *date;

      date = [theMessage receivedDate];

      if ( !date )
	{
	  return @"";
	}
      else
	{
	  NSUserDefaults *aUserDefaults; 
	  NSString *aString;
	  int day, today;
	  
	  aUserDefaults = [NSUserDefaults standardUserDefaults];      
	  
	  [date setTimeZone: [NSTimeZone localTimeZone]];
	  day = [date dayOfCommonEra];
	  today = [[NSCalendarDate calendarDate] dayOfCommonEra];
	  
	  if ( day == today )
	    {
	      aString = [aUserDefaults objectForKey: NSTimeFormatString];
	    }
	  else if ( day == today-1 )
	    {
	      aString = [NSString stringWithFormat: @"%@ %@",
				  [[aUserDefaults objectForKey: NSPriorDayDesignations] objectAtIndex: 0],
				  [aUserDefaults objectForKey: NSTimeFormatString]];
	    }
	  else
	    { 
	      aString = [aUserDefaults objectForKey: NSShortDateFormatString];
	    }
	  
	  if ( !aString )
	    {
	      aString = @"%b %d %Y";
	    }
	  
	  return [date descriptionWithCalendarFormat: aString
		       timeZone: [date timeZone]
		       locale: nil];
	}
    }
  else if (aTableColumn == fromColumn)
    {
      NSUserDefaults *aUserDefaults;
      
      aUserDefaults = [NSUserDefaults standardUserDefaults];

      // If we are in Sent or Drafts, we show the first To recipient
      if ( draftsOrSentFolder )
	{
	  if ( [theMessage recipientsCount] > 0 )
	    {
	      anInternetAddress = [[theMessage recipients] objectAtIndex: 0];
	    }
	  else
	    {
	      return @"";
	    }
	}
      else
	{
	  anInternetAddress = [theMessage from];
	}

      if ( !anInternetAddress )
	{
	  return @"";
	}
      else if ( [anInternetAddress personal] == nil || 
		[[anInternetAddress personal] length] == 0 )
	{
	  return [anInternetAddress address];
	}
      else
	{
	  return [anInternetAddress personal];
	}
    }
  else if (aTableColumn == subjectColumn)
    {
      NSUserDefaults *aUserDefaults;
      
      aUserDefaults = [NSUserDefaults standardUserDefaults];
      
      if ( [self dataViewType] == TYPE_OUTLINEVIEW &&
	   [aUserDefaults objectForKey: @"MAILWINDOW_REPEAT_SUBJECT"] &&
	   [aUserDefaults boolForKey: @"MAILWINDOW_REPEAT_SUBJECT"] == NO &&
	   ((Container *)theItem)->parent )
	{
	  return @"";
	}
      
      return [theMessage subject];
    }
  else if (aTableColumn == sizeColumn) 
    {
      return [NSString stringWithFormat: @"%.1fKB ", ((float)[theMessage size]/(float)1024)];
    }

  return @"";
}


//
//
//
- (void) _outlineViewShouldReloadData
{
  int selectedRow;
  SEL sortingSel;

  DESTROY(allContainers);

  selectedRow = [dataView selectedRow];
  sortingSel = NULL;
  
  if ( [self currentOrder] == nil )
    {
      [self setPreviousOrder: @"#"];
      [self setCurrentOrder: @"#"];
    }
  
  //
  // Sort by #.
  //
  if ( [[self currentOrder] isEqualToString: @"#"] )
    {
      NSDebugLog(@"#");
      if ( isReverseOrder )
	{
	  sortingSel = @selector(reverseCompareAccordingToNumber:);
	}
      else
	{
	  sortingSel = NULL;
	}
    }
  //
  // Sort by Date.
  //
  else if ( [[self currentOrder] isEqualToString: @"Date"] )
    {
      NSDebugLog(@"Date");
      if ( isReverseOrder )
	{
	  sortingSel = @selector(reverseCompareAccordingToDate:);;
	}
      else
	{
	  sortingSel = @selector(compareAccordingToDate:);
	}
    }
  //
  // Sort by From.
  //
  else if ( [[self currentOrder] isEqualToString: @"From"] )
    {
      NSDebugLog(@"From");
      if ( isReverseOrder )
	{
	  sortingSel = @selector(reverseCompareAccordingToSender:);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSender:);
	}
    }
  //
  // Sort by Subject.
  //
  else if ( [[self currentOrder] isEqualToString: @"Subject"] )
    {
      NSDebugLog(@"Subject");
      if ( isReverseOrder )
	{
	  sortingSel = @selector(reverseCompareAccordingToSubject:);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSubject:);
	}
    }
  //
  // Sort by Size.
  //
  else if ( [[self currentOrder] isEqualToString: @"Size"] )
    {
      NSDebugLog(@"Size");
      if ( isReverseOrder )
	{
	  sortingSel = @selector(reverseCompareAccordingToSize:);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSize:);
	}
    }
  
  // We sort using our new selector
  if ( sortingSel )
    {
      allContainers = RETAIN([[[self folder] allContainers] sortedArrayUsingSelector: sortingSel]);
    }
  else
    {
      allContainers = RETAIN([[self folder] allContainers]);
    }

  [dataView deselectAll: self];
  [dataView reloadData];
  
  if ( selectedRow >= 0 && selectedRow < [dataView numberOfRows] )
    {
      [dataView selectRow: selectedRow
		byExtendingSelection: NO];
    }
  
  [self _selectionDidChange];
}


//
//
//
- (void) _reloadTableColumns: (id) sender
{
  NSArray *shownTableColumns, *selectedRows;
  NSMutableArray *columnSizes;
  int i;

  shownTableColumns = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNTABLECOLUMNS"];

  // If the value doesn't exist in the user's defaults, we show all table columns.
  if ( !shownTableColumns )
    {
      return;
    }

  // We backup our selected rows
  selectedRows = [[[self dataView] selectedRowEnumerator] allObjects];
  RETAIN(selectedRows);

  [[self dataView] removeTableColumn: statusColumn];
  [[self dataView] removeTableColumn: idColumn];
  [[self dataView] removeTableColumn: dateColumn];
  [[self dataView] removeTableColumn: fromColumn];

  //
  // We don't remove our Subject column if we are using message threading
  // as it causes problems under OS X (an exception is generated when trying
  // to remove the outlineTableColumn from the NSOutlineView)
  //
  if ( [self dataViewType] == TYPE_TABLEVIEW )
    { 
      [[self dataView] removeTableColumn: subjectColumn];
    }

  [[self dataView] removeTableColumn: sizeColumn];


  // Check whether we have archived the column sizes
  // Ideally, setting column size should be handled by the NSTableView's autosave feature.
  // But it seems to be messing with the dynamic addition and removal of columns.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"MAIL_WINDOW_TABLE_COLUMN_SIZES"] )
    {
      columnSizes =  [[NSUserDefaults standardUserDefaults] 
		       objectForKey: @"MAIL_WINDOW_TABLE_COLUMN_SIZES"];
    }
  else
    {
      columnSizes = [[NSMutableArray alloc] initWithCapacity: 6];
      for (i = 0; i < 6; i++)
	{
	  [columnSizes addObject: [NSNumber numberWithFloat: 0.0]];
	}
      AUTORELEASE(columnSizes);
    }

  if ([(NSNumber *)[columnSizes objectAtIndex: 0] floatValue] > 0.0)
    {
      [statusColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 0] floatValue]];
    }
  if ([(NSNumber *)[columnSizes objectAtIndex: 1] floatValue] > 0.0)
    {
      [idColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 1] floatValue]];
    }
  if ([(NSNumber *)[columnSizes objectAtIndex: 2] floatValue] > 0.0)
    {
      [dateColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 2] floatValue]];
    }
  if ([(NSNumber *)[columnSizes objectAtIndex: 3] floatValue] > 0.0)
    {
      [fromColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 3] floatValue]];
    }
  if ([(NSNumber *)[columnSizes objectAtIndex: 4] floatValue] > 0.0)
    {
      [subjectColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 4] floatValue]];
    }
  if ([(NSNumber *)[columnSizes objectAtIndex: 5] floatValue] > 0.0)
    {
      [sizeColumn setWidth: [(NSNumber *)[columnSizes objectAtIndex: 5] floatValue]];
    }

  for (i = 0; i < [shownTableColumns count]; i++)
    {
      NSString *column;

      column = [shownTableColumns objectAtIndex: i];

      if ( [column isEqualToString: @"Status"] )
	{
	  [[self dataView] addTableColumn: statusColumn];
	}
      // FIXME: In 1.1.2, Id was renamed to Number, remove the || condition later.
      else if ( [column isEqualToString: @"Number"] || [column isEqualToString: @"Id"] )
	{
	  [[self dataView] addTableColumn: idColumn];
	}
      else if ( [column isEqualToString: @"Date"] )
	{
	  [[self dataView] addTableColumn: dateColumn];
	}
      else if ( [column isEqualToString: @"From"] )
	{
	  [[self dataView] addTableColumn: fromColumn];
	}
      else if ( [column isEqualToString: @"Subject"] )
	{
	  if ( [self dataViewType] == TYPE_TABLEVIEW )
	    {
	      [[self dataView] addTableColumn: subjectColumn];
	    }
	  else
	    {
	      [[self dataView] moveColumn: 0  toColumn: i];
	    }
	}
      else if ( [column isEqualToString: @"Size"] )
	{
	  [[self dataView] addTableColumn: sizeColumn];
	}
    }
  
    
  // We restore the list of selected rows
  for (i = 0; i < [selectedRows count]; i++)
    {
      [[self dataView] selectRow: [[selectedRows objectAtIndex: i] intValue] 
		       byExtendingSelection: YES];
            
      // If we had at least one row selected, we scroll to it
      if ( i == ([selectedRows count]-1) )
        {
          [[self dataView] scrollRowToVisible: [[selectedRows objectAtIndex: i] intValue]];
        }
    }
    
  RELEASE(selectedRows);
}


//
//
//
- (void) _restoreImage
{
  // We verify if we are using a secure (SSL) connection or not
  if ( [[self folder] isKindOfClass: [IMAPFolder class]] &&
       [(IMAPStore *)[[self folder] store] tcpConnection] && 
       ![(NSObject *)[(IMAPStore *)[[self folder] store] tcpConnection] isKindOfClass: [TCPConnection class]] )
    {
      [icon setImage: [NSImage imageNamed: @"pgp-mail-small.tiff"]];
    }
  else
    {
      [icon setImage: nil];
    }
}


//
//
//
- (void) _restoreSortingOrder
{
  // We get our default sorting order
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"SORTINGORDER"] )
    {
      NSString *aString;

      aString = [[NSUserDefaults standardUserDefaults] stringForKey: @"SORTINGORDER"];

      // FIXME: Eventually remove that if (). It was renamed in 1.1.0pre1.
      if ( aString && [aString isEqualToString: @"Id"] )
	{
	  aString = @"#";
	}

      [self setCurrentOrder: aString];
      
      isReverseOrder = [[NSUserDefaults standardUserDefaults] integerForKey: @"SORTINGSTATE"];
      
      if ( [[self currentOrder] isEqualToString: @"Date"] )
	{
	  [[self dataView] setHighlightedTableColumn: dateColumn];
	}
      else if ( [[self currentOrder] isEqualToString: @"From"] )
	{
	  [[self dataView] setHighlightedTableColumn: fromColumn];
	}
      else if ( [[self currentOrder] isEqualToString: @"Subject"] )
	{
	  [[self dataView] setHighlightedTableColumn: subjectColumn];
	}
      else if ( [[self currentOrder] isEqualToString: @"Size"] )
	{
	  [[self dataView] setHighlightedTableColumn: sizeColumn];
	}
      else
	{
	  [[self dataView] setHighlightedTableColumn: idColumn];
	}
    }
  else
    {
      [[self dataView] setHighlightedTableColumn: idColumn];
    }
    
  [self _setIndicatorImageForTableColumn: [[self dataView] highlightedTableColumn]]; 
}


//
//
//
- (void) _restoreSplitViewSize
{
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"TEXTSCROLLVIEW_HEIGHT"] ) 
    {
      NSRect aRect;
      
      aRect = [textScrollView frame];
      aRect.size.height = [[NSUserDefaults standardUserDefaults] integerForKey: @"TEXTSCROLLVIEW_HEIGHT"];
      
      [textScrollView setFrame: aRect];
      
      [splitView adjustSubviews];
      [splitView setNeedsDisplay: YES];
    }
}


//
//
//
- (void) _selectionDidChange
{
  if ( isReloading )
    {
      return;
    }

  // If we have more than one selected rows or no selection at all, 
  // we clear up the text view.
  if ( [dataView numberOfSelectedRows] > 1 ||  [dataView selectedRow] < 0)
    {
      [textView setString: @""];
      
      // We redisplay our dataview since "selectAll" doesn't do it for us.
      [dataView setNeedsDisplay: YES];
    }
  else 
    {
      NSRect r1, r2;

      // We zero all our index's offset
      [self _zeroIndexOffset];
	  
      // We show our message!
      [self _showMessage: self];

      if ( IMAPSTORE_IS_DISCONNECTED([[self folder] store]) )
	{
	  [[MailboxManagerController singleInstance] setStore: nil
						     name: [(IMAPStore *)[[self folder] store] name]
						     username: [(IMAPStore *)[[self folder] store] username]];
	  return;
	}
      
      //
      // We now autoscroll, intelligently our dataView.
      //
      r1 = [dataView rectOfRow: [dataView selectedRow]];
      r2 = [dataView convertRect: r1  toView: tableScrollView];
      
      if ( r2.origin.y < (2*[dataView rowHeight]) )
	{
	  r1.origin.y -= (2*[tableScrollView verticalPageScroll]);
	  [dataView scrollRectToVisible: r1];
	}
      else if ( r2.origin.y > [tableScrollView contentSize].height )
	{
	  r1.origin.y += (2*[tableScrollView verticalPageScroll]);
	  [dataView scrollRectToVisible: r1];
	}
    }

  [self updateStatusLabel];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: SelectionOfMessageHasChanged
    object: nil
    userInfo: nil];
}


//
//
//
- (void) _setIndicatorImageForTableColumn: (NSTableColumn *) aTableColumn
{
  NSArray *tableColumns;
  int i;
  
  tableColumns = [dataView tableColumns];
 
  for (i = 0; i < [tableColumns count]; i++)
    {
      [dataView setIndicatorImage: nil  inTableColumn: [tableColumns objectAtIndex: i]];
    }

  if ( isReverseOrder )
    {
#ifdef MACOSX
      [dataView setIndicatorImage: [NSImage imageNamed: @"NSAscendingSortIndicator"]  inTableColumn: aTableColumn];
#else
      [dataView setIndicatorImage: [NSImage imageNamed: @"sort_up.tiff"]  inTableColumn: aTableColumn];
#endif
    }
  else
    {
#ifdef MACOSX
      [dataView setIndicatorImage: [NSImage imageNamed: @"NSDescendingSortIndicator"]  inTableColumn: aTableColumn];
#else
      [dataView setIndicatorImage: [NSImage imageNamed: @"sort_down.tiff"]  inTableColumn: aTableColumn];
#endif
    }
}


//
//
//
- (void) _showMessage: (id) sender
{
  // Don't render the message if user won't be able to see it.
  if ( [textScrollView frame].size.height == 0 )
    {
      return;
    }
  
  [Utilities showMessage: [self selectedMessage]
	     target: [self textView]
	     showAllHeaders: [self showAllHeaders]];
}


//
// FIXME: refactor like _outlineViewShouldReloadData
//
- (void) _tableViewShouldReloadData
{
  NSArray *previousArray;
  SEL sortingSel;

  previousArray = allMessages;
  sortingSel = NULL;
  
  if ( [self currentOrder] == nil )
    {
      [self setPreviousOrder: @"#"];
      [self setCurrentOrder: @"#"];
    }
  
  //
  // Sort by #.
  //
  if ( [[self currentOrder] isEqualToString: @"#"] )
    {
      NSDebugLog(@"#");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToNumber:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToNumber:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToNumber:);
	  allMessages = [[NSArray alloc]
			  initWithArray: [folder allMessages]];
	}
    }
  //
  // Sort by Date.
  //
  else if ( [[self currentOrder] isEqualToString: @"Date"] )
    {
      NSDebugLog(@"Date");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToDate:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToDate:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToDate:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToDate:)]);
	}
    }
  //
  // Sort by From.
  //
  else if ( [[self currentOrder] isEqualToString: @"From"] )
    {
      NSDebugLog(@"From");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSender:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSender:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSender:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSender:)]);
	}
    }
  //
  // Sort by Subject.
  //
  else if ( [[self currentOrder] isEqualToString: @"Subject"] )
    {
      NSDebugLog(@"Subject");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSubject:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSubject:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSubject:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSubject:)]);
	}
    }
  //
  // Sort by Size.
  //
  else if ( [[self currentOrder] isEqualToString: @"Size"] )
    {
      NSDebugLog(@"Size");
      if (isReverseOrder == YES)
	{
	  sortingSel = @selector(reverseCompareAccordingToSize:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(reverseCompareAccordingToSize:)]);
	}
      else
	{
	  sortingSel = @selector(compareAccordingToSize:);
	  allMessages = 
	    RETAIN([[folder allMessages] 
		     sortedArrayUsingSelector:
		       @selector(compareAccordingToSize:)]);
	}
    }
  
  //
  // We now select all the messages that were previously selected
  // in the previous order.
  //
  if ( previousArray && [self folder] )
    {
      NSMutableArray *sm;
      NSArray *sc;
      id aMessage;
      
      int i, index, selectedRow, count, newCount;
      BOOL newSelection;
      NSRange range;

      sc = [[dataView selectedRowEnumerator] allObjects];
      selectedRow = [dataView selectedRow];
      
      count = [sc count];
      newCount = [allMessages count];
      range = NSMakeRange(0, newCount);
      
      newSelection = NO;
      
      sm = [[NSMutableArray alloc] initWithCapacity: newCount];
      
      // We get all the previous selected messages (Message objects)
      for ( i = 0; i < count; i++ )
	{
	  [sm addObject: [previousArray objectAtIndex: [[sc objectAtIndex: i] intValue]]];
	}
      
      [sm sortUsingSelector: sortingSel];
      
      isReloading = YES;
      
      [dataView deselectAll: self];
      [dataView reloadData];

      for ( i = 0; i < count; i++ )
	{
	  aMessage = [sm objectAtIndex: i];
			      
	  index = [allMessages indexOfObject: aMessage
			       inRange: range];

	  if (index != NSNotFound)
	    {
	      [dataView selectRow: index  byExtendingSelection: YES];
	      range = NSMakeRange(index + 1, newCount - index - 1);
	    }
	  else
	    {
	      newSelection = YES;
	    }
	}
      
      if (selectedRow != -1)
	{
	  aMessage = [previousArray objectAtIndex: selectedRow];
	  index = [allMessages indexOfObject: aMessage];
	  
	  if (index != NSNotFound)
	    {
	      [dataView selectRow: index
			 byExtendingSelection: YES];
	    }
	}
      
      isReloading = NO;
      
      // If the selection has changed over the previous reload
      if (newSelection == YES)
	{
	  [self tableViewSelectionDidChange: nil];
	}
      
      // We scroll back to a selected row
      if ([dataView selectedRow] != -1)
	{
	  [dataView scrollRowToVisible: [dataView selectedRow]];
	}
    }

  TEST_RELEASE(previousArray);

  [self setPreviousOrder: [self currentOrder]];

  [dataView reloadData];
  
  // We verify if we have at least one selected row, in case we don't, we just 
  // clear our textView
  if ( [dataView numberOfSelectedRows] != 1 )
    {
      [textView setString: @""];
    }
}


//
// This method does absolutely nothing under OS X
//
- (IBAction) _updateAnimatedIcon: (id) sender
{
#ifndef MACOSX
  if (animation_index == 9)
    {
      animation_index = 1;
    }
  
  [icon setImage: [NSImage imageNamed: [NSString stringWithFormat: @"anim-logo-%d.tiff", animation_index]]];
  
  animation_index += 1;
#endif
}

//
//
//
- (void) _willDisplayCellForMessage: (Message *) theMessage
			       cell: (id) theCell
			tableColumn: (NSTableColumn *) theTableColumn
				row: (int) rowIndex
			       item: (id) theItem
{ 
  Flags *theFlags;
  BOOL aBOOL;

  // We get the message's flags
  theFlags = [theMessage flags]; 

  // We verify for a coloring filter. We also don't draw the background color if 
  // the row is selected in the dataView.
  if ( [dataView selectedRow] != rowIndex )
    {
      NSColor *aColor;
      
      aColor = [[FilterManager singleInstance] colorForMessage: theMessage];
 
      // We if have a special color coming from our filter, we set it for this cell
      if ( aColor )
	{
	  [theCell setDrawsBackground: YES];
	  [theCell setBackgroundColor: aColor];
	}
      else
	{
	  [theCell setDrawsBackground: NO];
	}
    }
  else
    {
      [theCell setDrawsBackground: NO];
    }

  // If we are using message threading, we must drill-down into all children
  // to be sure we don't have a message marked as unread. If it's the case,
  // we mark all it's parent as unread IIF they are collapsed.
  if ( [self dataViewType] == TYPE_OUTLINEVIEW )
    {
      // If our message (containing children) is already unread, we don't
      // need to iterate into its children list to see if we must mark it
      // as unread.
      aBOOL = [theFlags contain: SEEN];

      if ( aBOOL )
	{
	  NSEnumerator *aEnumerator;
	  Container *aContainer;
	  
	  aEnumerator = [theItem childrenEnumerator];
	  
	  while ( (aContainer = [aEnumerator nextObject]) )
	    {
	      if ( ![[aContainer->message flags] contain: SEEN] )
		{
		  aBOOL = NO;
		  break;
		}
	    }
	}
    }
  else
    {
      aBOOL = [theFlags contain: SEEN];
    }

  // If it's a new message, we set the cell's text to bold
  if ( aBOOL )
    {
      [theCell setFont: seenMessageFont];
    }
  else
    {
      [theCell setFont: recentMessageFont];
    }

  // If it's a deleted message, we set the cell's text to italic
  if ( [theFlags contain: DELETED] )
    {
      [theCell setTextColor: [NSColor darkGrayColor]];
      [theCell setFont: deletedMessageFont];
    }
  else
    {
      [theCell setTextColor: [NSColor blackColor]];
    }

  // We set the right aligment for our last (ie., Size) column.
  if ( theTableColumn == sizeColumn )
    {
      [theCell setAlignment: NSRightTextAlignment];
    }
  else
    {
      [theCell setAlignment: NSLeftTextAlignment];
    }

  // We set the image of our status cell
  if ( theTableColumn == statusColumn )
    {
      ExtendedCell *cell;
      
      cell = (ExtendedCell *)[statusColumn dataCell];
      [cell setFlags: theFlags];
      [cell setIsMimeMessage: ([theMessage mimeVersion] == nil ? NO : YES)];
    }
}


//
//
//
- (void) _zeroIndexOffset
{
  int i;

  for (i = 0; i < [[self allMessageViewWindowControllers] count]; i++)
    {
      [[allMessageViewWindowControllers objectAtIndex: i] setIndexOffset: 0];
    }
}

@end
