/*
**  FindWindowController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "FindWindowController.h"

#ifndef MACOSX
#include "FindWindow.h"
#endif

#include "GNUMail.h"
#include "ConsoleWindowController.h"
#include "Constants.h"
#include "MailboxManagerController.h"
#include "MailWindowController.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Container.h>
#include <Pantomime/Folder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/MimeMultipart.h>
#include <Pantomime/NSString+Extensions.h>

static FindWindowController *singleInstance = nil;


@implementation FindWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  FindWindow *theWindow;
  
  theWindow = [[FindWindow alloc] initWithContentRect: NSMakeRect(200,200,340,175)
				  styleMask: NSTitledWindowMask|NSClosableWindowMask|
				  NSMiniaturizableWindowMask|NSResizableWindowMask
				  backing: NSBackingStoreBuffered
				  defer: YES];
  
  [super initWithWindow: theWindow];

  [theWindow layoutWindow];
  [theWindow setDelegate: self];
  [theWindow setMinSize: [theWindow frame].size];

  // We link our outlets
  findField = theWindow->findField;
  foundLabel = theWindow->foundLabel;
  ignoreCaseButton = theWindow->ignoreCaseButton;
  regularExpressionButton = theWindow->regularExpressionButton;
  matrix = theWindow->matrix;

  RELEASE(theWindow);
#endif

  [[self window] setTitle: _(@"Find")];

  // We finally set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"FindWindow"];
  [[self window] setFrameUsingName: @"FindWindow"];

  return self;
}


//
//
//
- (void) dealloc
{
  RELEASE(indexes);
  
  [super dealloc];
}


//
// action methods
//
- (IBAction) findAll: (id) sender
{
  NSString *aString;

  aString = [findField stringValue];
  location = 0;
  
  if ( [GNUMail lastMailWindowOnTop] && 
       ([[aString stringByTrimmingWhiteSpaces] length] > 0)) 
    {
      MailWindowController *aMailWindowController;
      NSAutoreleasePool *pool;
      NSArray *allResults;
      Folder *aFolder;
      
      int mask, options;
      
      pool = [[NSAutoreleasePool alloc] init];
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
    
      aFolder = [aMailWindowController folder];
      
      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									       _(@"Searching for %@..."), aString]];
      
      //
      // We get our mask
      //
      if ( [[matrix cellAtRow: 0  column: 0] state] == NSOnState )
	{
	  mask = PantomimeFrom;
	}
      else if ( [[matrix cellAtRow: 1  column: 0] state] == NSOnState )
	{
	  mask = PantomimeTo;
	}
      else if ( [[matrix cellAtRow: 1  column: 1] state] == NSOnState )
	{
	  mask = PantomimeContent;
	}
      else
	{
	  mask = PantomimeSubject;
	}

      //
      // We get our options
      //
      options = 0;

      if ( [ignoreCaseButton state] == NSOnState )
	{
	  options = options|PantomimeCaseInsensitiveSearch;
	}
      
      if ( [regularExpressionButton state] == NSOnState )
	{
	  options = options|PantomimeRegularExpression;
	}
      
      allResults = [aFolder search: aString
			    mask: mask
			    options: options];
      
      if ( IMAPSTORE_IS_DISCONNECTED([aFolder store]) )
	{
	  [[MailboxManagerController singleInstance] setStore: nil
						     name: [(IMAPStore *)[aFolder store] name]
						     username: [(IMAPStore *)[aFolder store] username]];
	  RELEASE(pool);
	  return;
	}

      // We get all the indexes from our messages found
      if ( [allResults count] > 0 )
	{
	  id dataView;
	  
	  int previous_row_selected;

	  dataView = [aMailWindowController dataView];	  
	  previous_row_selected = [dataView selectedRow];
	  
	  [dataView deselectAll: nil];
	  [indexes removeAllObjects];
	  
	  // We add the index of our rows.. and we select the rows..
	  [self _selectIndexesFromResults: allResults
		controller: aMailWindowController];

	  // If we found only one result, we automatically scroll to that row
	  if ( [allResults count] == 1 && [indexes count] > 0)
	    {
	      [dataView scrollRowToVisible: [[indexes objectAtIndex: 0] intValue]];
	    }

	  [dataView setNeedsDisplay:YES];
	}
      else
	{
	  NSBeep();
	}

      [self updateLabel: foundLabel
	    withValue: [allResults count]];
      
      [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									       _(@"Done searching. %d results found."), 
									     [allResults count]]];

      RELEASE(pool);
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) nextMessage: (id) sender
{
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *aMailWindowController;
      id dataView; 

      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      dataView = [aMailWindowController dataView];
      
      if ([indexes count] < 2)
	{
	  NSBeep();
	  return;
	}
      else
	{
	  [dataView selectRow: [[indexes objectAtIndex:location] intValue] 
		    byExtendingSelection: NO];
	  [dataView scrollRowToVisible: [[indexes objectAtIndex:location] intValue] ];
	  location += 1;
	  
	  if (location == [indexes count])
	    {
	      location = 0;
	    }
	  
	  [dataView setNeedsDisplay:YES];
	}
    }
}


//
//
//
- (IBAction) previousMessage: (id) sender
{
  
  if ( [GNUMail lastMailWindowOnTop] )
    {
      MailWindowController *aMailWindowController;
      id dataView; 
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      dataView = [aMailWindowController dataView];
      
      if ([indexes count] < 2)
	{
	  NSBeep();
	  return;
	}
      else
	{
	  [dataView selectRow: [[indexes objectAtIndex:location] intValue] 
		    byExtendingSelection: NO];
	  [dataView scrollRowToVisible: [[indexes objectAtIndex:location] intValue] ];
	  location -= 1;
	  
	  if (location < 0)
	    {
	      location = [indexes count] - 1;
	    }

	  [dataView setNeedsDisplay:YES];
	}
    }
}


//
// delegate methods
//
- (void) windowDidLoad
{
  indexes = [[NSMutableArray alloc] init];
}


//
// access / mutation
//
- (void) updateLabel: (NSTextField *) theLabel
           withValue: (int) theValue
{
  [theLabel setStringValue: [NSString stringWithFormat: @"%i found", theValue]];
}


//
//
//
- (NSTextField *) findField
{
  return findField;
}


//
// class methods
//
+ (id) singleInstance
{
  if ( !singleInstance )
    {
      singleInstance = [[FindWindowController alloc] initWithWindowNibName: @"FindWindow"];
    }
  
  return singleInstance;
}

@end


//
// FindWindowController Private implementation
//
@implementation FindWindowController (Private)

- (void) _selectIndexesFromResults: (NSArray *) theResults
                        controller: (MailWindowController *) theMailWindowController
{
  int i, index, count;
  id dataView;
 
  dataView = [theMailWindowController dataView];

  // FIXME: We should optimize this since it runs in O(n^2)
  if ( [theMailWindowController dataViewType] == TYPE_OUTLINEVIEW )
    {
      count = [dataView numberOfRows];

      for (i = 0; i < count; i++)
	{
	  index = [theResults indexOfObject: ((Container *)[dataView itemAtRow: i])->message];

	  if ( index != NSNotFound )
	    {
	      [indexes addObject: [NSNumber numberWithInt: i]];
	      [dataView selectRow: i  byExtendingSelection: YES];
	    }
	}
    }
  else
    {
      NSArray *allMessages;

      allMessages = [theMailWindowController allMessages];
      count = [theResults count];

      for (i = 0; i < count; i++)
	{
	  index = [allMessages indexOfObject: [theResults objectAtIndex: i]];

	  if ( index != NSNotFound )
	    {
	      [indexes addObject: [NSNumber numberWithInt: index]];
	      [dataView selectRow: index  byExtendingSelection: YES];
	    }
	}
    }
}

@end
