/*
**  ExtendedOutlineView.m
**
**  Copyright (c) 2002, 2003
**
**  Author: Francis Lachapelle <francis@Sophos.ca>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ExtendedOutlineView.h"

#ifdef MACOSX
#include "MailboxManagerController.h"
#endif

/*!
 * @class ExtendedOutlineView
 * @abstract This class overwrites some methods of NSOutlineView.
 * @discussion This class is used by the MailboxManager class so
 *             all items look like folders un Mac OS X and items
 *             can be dragged on items and not inbetween nodes.
 *             
 */
@implementation ExtendedOutlineView: NSOutlineView

#ifndef MACOSX
- (id) initWithFrame: (NSRect) aRect
{
  self = [super initWithFrame: aRect];

  collapsed = RETAIN([NSImage imageNamed: @"closedFolder"]);
  expanded = RETAIN([NSImage imageNamed: @"openFolder"]);

  return self;
}

- (void) dealloc
{
  RELEASE(collapsed);
  RELEASE(expanded);

  [super dealloc];
}

- (void) setCollapsedImage: (NSImage *) theImage
{
  RETAIN(theImage);
  RELEASE(collapsed);
  collapsed = theImage;
}

- (void) setExpandedImage: (NSImage *) theImage
{
  RETAIN(theImage);
  RELEASE(expanded);
  expanded = theImage;
}

#endif

//
//
//
- (BOOL) acceptsFirstMouse: (NSEvent *) theEvent
{
  return YES;
}


#ifdef MACOSX
/*!
 * @method isExpandable:
 * @abstract Specifies if a node can be expanded.
 * @discussion
 * @param item The node of the outline view.
 * @result Always returns YES, so that every node appears
 *         as expandable.
 */
- (BOOL) isExpandable: (id) item
{
  return YES;
}

- (NSMenu *) menuForEvent: (NSEvent *) theEvent
{
  int row;
  id item;
  
  row = [self rowAtPoint:[self convertPoint:[theEvent locationInWindow] fromView:nil]];
  
  if (row >= 0)
    {
      [self abortEditing];
      
      item = [self itemAtRow:row];
      
      if (item)
	{
	  id delegate;
	  
	  delegate = [self delegate];
	  
	  if ( [delegate respondsToSelector: @selector(outlineView:shouldSelectItem:)] && 
	       [delegate outlineView: self  shouldSelectItem: item] )
	    {
	      [self selectRow: row  byExtendingSelection: NO];
	    }
	  
	  if ( [delegate respondsToSelector: @selector(outlineView:contextMenuForItem:)] ) 
	    {
	      return [delegate outlineView :self contextMenuForItem: item];
	    }
	}
    }
  else
    {
      [self deselectAll: self];
      return [self menu];
    }
  return nil;
}
#endif


/*!
 * @method setDropItem: dropChildIndex:
 * @abstract Retargets drop on child index.
 * @discussion This method sets the behaviour of the outline
 *             view so that items dragged over a node can be
 *             dropped on the node and inbetween nodes.
 * @param item The item involved.
 * param index The index of the item.
 */
- (void) setDropItem: (id) item
      dropChildIndex: (int) index
{
  [super setDropItem: item
	 dropChildIndex: NSOutlineViewDropOnItemIndex];
}


/*!
 * @method shouldCollapseAutoExpandedItemsForDeposited:
 * @abstract Indicate if auto expanded items should return to
 *           their original collapsed state. 
 * @discussion
 * @param deposited Tells whether or not the drop terminated due
 *                  to a successful drop.
 * @result Always returns NO, so that items never return to
 *         their original collapsed state.
 */
- (BOOL)shouldCollapseAutoExpandedItemsForDeposited: (BOOL) deposited
{
  return NO;
}

#ifdef MACOSX
- (NSImage *) dragImageForRows: (NSArray *) dragRows
			 event: (NSEvent *) dragEvent 
	       dragImageOffset: (NSPointPointer) dragImageOffset
#else
- (NSImage *) dragImageForRows: (NSArray *) dragRows
			 event: (NSEvent *) dragEvent 
	       dragImageOffset: (NSPoint *) dragImageOffset
#endif
{
  
  if ( [dragRows count] > 1 )
    {
      return [NSImage imageNamed: @"drag_mails.tiff"];
    }
  else
    {
      return [NSImage imageNamed: @"drag_mail.tiff"];
    }

  return [super dragImageForRows: dragRows
  		event: dragEvent 
		dragImageOffset: dragImageOffset];
}


//
// We define a category for NSOutlineView under GNUstep to override
// the normal behavior of -drawRow:
//
#ifndef MACOSX
- (void) drawRow: (int) rowIndex 
	clipRect: (NSRect) aRect
{
  NSRect drawingRect, imageRect;
  NSTableColumn *aTableColumn;
  NSCell *cell, *imageCell;

  int i, startingColumn, endingColumn;
  float x_pos;
  
  if (_dataSource == nil)
    {
      return;
    }
  
  imageCell = nil;

  /* Using columnAtPoint: here would make it called twice per row per drawn 
     rect - so we avoid it and do it natively */
  if (rowIndex >= _numberOfRows)
    {
      return;
    }
  
  /* Determine starting column as fast as possible */
  x_pos = NSMinX (aRect);
  i = 0;
  
  while ((x_pos > _columnOrigins[i]) && (i < _numberOfColumns))
    {
      i++;
    }
  startingColumn = (i - 1);
  
  if (startingColumn == -1)
    {
      startingColumn = 0;
    }
  
  /* Determine ending column as fast as possible */
  x_pos = NSMaxX (aRect);
  
  // Nota Bene: we do *not* reset i
  while ((x_pos > _columnOrigins[i]) && (i < _numberOfColumns))
    {
      i++;
    }
  endingColumn = (i - 1);

  if (endingColumn == -1)
    {
      endingColumn = _numberOfColumns - 1;
    }
  
  /* Draw the row between startingColumn and endingColumn */
  for (i = startingColumn; i <= endingColumn; i++)
    {
      if (i != _editedColumn || rowIndex != _editedRow)
	{
	  id item = [self itemAtRow: rowIndex];
	  
	  aTableColumn = [_tableColumns objectAtIndex: i];
	  cell = [aTableColumn dataCellForRow: rowIndex];
	  
	  if ([_delegate respondsToSelector: @selector(outlineView:willDisplayCell:forTableColumn:item:)])
	    {
	      [_delegate outlineView: self   
			 willDisplayCell: cell 
			 forTableColumn: aTableColumn   
			 item: item];
	    }

	  [cell setObjectValue: [_dataSource outlineView: self
					     objectValueForTableColumn: aTableColumn
					     byItem: item]]; 
	  drawingRect = [self frameOfCellAtColumn: i
			      row: rowIndex];

	  if (aTableColumn == _outlineTableColumn)
	    {
	      NSImage *image = nil;
	      int level = 0;
	      float indentationFactor = 0.0;
	      // float originalWidth = drawingRect.size.width;

	      // display the correct arrow...
	      if ([self isItemExpanded: item])
		{
		  image = expanded;
		}
	      else
		{
		  image = collapsed;
		}
	      
	      if (![self isExpandable: item])
		{
		  image = nil;
		}
	      
	      level = [self levelForItem: item];
	      indentationFactor = _indentationPerLevel * level;
	      
	      if ( image )
		{
		  imageCell = [[NSCell alloc] initImageCell: image];
		}
	      
	      if (_indentationMarkerFollowsCell)
		{
		  imageRect.origin.x = drawingRect.origin.x + indentationFactor;
		  imageRect.origin.y = drawingRect.origin.y;
		}
	      else
		{
		  imageRect.origin.x = drawingRect.origin.x;
		  imageRect.origin.y = drawingRect.origin.y;
		}
	      
	      if ( image )
		{
		  imageRect.size.width = [image size].width;
		  imageRect.size.height = [image size].height;
		  
		  [imageCell drawWithFrame: imageRect inView: self];
		  
		  drawingRect.origin.x += indentationFactor + [image size].width + 5;
		  drawingRect.size.width -= indentationFactor + [image size].width + 5;
		  RELEASE(imageCell);
		}
	      else
		{
		  drawingRect.origin.x += indentationFactor;
		  drawingRect.size.width -= indentationFactor;
		}
	    }

	  [cell drawWithFrame: drawingRect inView: self];
	}
    }
}
#endif

@end
