/*
**  AccountEditorWindowController.m
**
**  Copyright (c) 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "AccountEditorWindowController.h"

#include "FolderNode.h"
#include "FolderNodePopUpItem.h"
#include "GNUMail.h"
#include "Constants.h"
#include "PasswordPanelController.h"
#include "Utilities.h"

#ifndef MACOSX
#include "AccountEditorWindow.h"
#include "IMAPView.h"
#include "MailboxesView.h"
#include "PersonalView.h"
#include "POP3View.h"
#include "ReceiveView.h"
#include "SendView.h"
#include "UNIXView.h"
#endif

#include <Pantomime/Constants.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/SMTP.h>

#define IMAP_SERVICE_PORT  143
#define POP3_SERVICE_PORT  110
#define IMAPS_SERVICE_PORT 993
#define POP3S_SERVICE_PORT 995

#define SMTP_PORT  25
#define SSMTP_PORT 465

#define VERIFY_SELECTION(popup, name) ({ \
  [popup synchronizeTitleAndSelectedItem]; \
  if ( [popup indexOfSelectedItem] == 0 ) \
    { \
      NSRunInformationalAlertPanel(_(@"Error!"), \
				   _(@"You must select a valid %@ in the Mailboxes tab."), \
				   _(@"OK"), \
				   NULL, \
				   NULL, \
				   name); \
      return NO; \
    } \
})

#ifndef MACOSX
#define SSL_BUNDLE_IS_INSTALLED ({ \
  NSArray *allPaths; \
  BOOL b; \
  int i; \
  \
  b = NO; \
  allPaths = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, \
						 NSLocalDomainMask|NSNetworkDomainMask|NSSystemDomainMask|NSUserDomainMask, \
						 YES); \
  for (i = 0; i < [allPaths count]; i++) \
    { \
	if ( [NSBundle bundleWithPath: [NSString stringWithFormat: @"%@/Pantomime/TCPSSLConnection.bundle", \
						 [allPaths objectAtIndex: i]]] ) \
	  { \
	      b = YES; \
	  } \
    } \
  b; \
})
#else
#define SSL_BUNDLE_IS_INSTALLED YES
#endif

//
//
//
@implementation AccountEditorWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifndef MACOSX
  AccountEditorWindow *aWindow;
#endif
    
  // First initializations of some variables
  allNodes = [Utilities initializeFolderNodesUsingAccounts: [[[NSUserDefaults standardUserDefaults] 
							       volatileDomainForName: @"PREFERENCES"] 
							      objectForKey: @"ACCOUNTS"]];
  RETAIN(allNodes);

  allFolders = [[FolderNode alloc] init];
  [allFolders setName: _(@"All Mailboxes")];
  store = nil;

#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  aWindow = [[AccountEditorWindow alloc] initWithContentRect: NSMakeRect(200,200,410,400)
					 styleMask: NSTitledWindowMask|NSMiniaturizableWindowMask
					 backing: NSBackingStoreBuffered
					 defer: YES];
  
  self = [super initWithWindow: aWindow];
  
  [aWindow layoutWindow];
  [aWindow setDelegate: self];

  // We link our standard outlets
  tabView = aWindow->tabView;

  // We load our Personal view
  personalView = [[PersonalView alloc] initWithParent: self];
  [personalView layoutView];
  personalAccountNameField = ((PersonalView *)personalView)->personalAccountNameField;
  personalNameField = ((PersonalView *)personalView)->personalNameField;
  personalEMailField = ((PersonalView *)personalView)->personalEMailField;
  personalReplyToField = ((PersonalView *)personalView)->personalReplyToField;
  personalOrganizationField = ((PersonalView *)personalView)->personalOrganizationField;
  personalSignaturePopUp = ((PersonalView *)personalView)->personalSignaturePopUp;
  personalSignatureField = ((PersonalView *)personalView)->personalSignatureField;
  personalLocationButton = ((PersonalView *)personalView)->personalLocationButton;
  personalLocationLabel = (NSTextField *)((PersonalView *)personalView)->personalLocationLabel;
  
  // We load our Receive view
  receiveView = [[ReceiveView alloc] initWithParent: self];
  [receiveView layoutView];
  receiveServerNameField = ((ReceiveView *)receiveView)->receiveServerNameField;
  receiveServerPortField = ((ReceiveView *)receiveView)->receiveServerPortField;
  receiveUsernameField = ((ReceiveView *)receiveView)->receiveUsernameField;
  receivePopUp = ((ReceiveView *)receiveView)->receivePopUp;
  receivePasswordSecureField = ((ReceiveView *)receiveView)->receivePasswordSecureField;
  receiveRememberPassword = ((ReceiveView *)receiveView)->receiveRememberPassword;
  receiveCheckOnStartup = ((ReceiveView *)receiveView)->receiveCheckOnStartup;
  receiveUseSecureConnection = ((ReceiveView *)receiveView)->receiveUseSecureConnection;
  receiveMatrix = ((ReceiveView *)receiveView)->receiveMatrix;
  receiveMinutesField = ((ReceiveView *)receiveView)->receiveMinutesField;
  
  // IMAPView
  imapView = [[IMAPView alloc] initWithParent: self];
  [imapView layoutView];

  imapSupportedMechanismsPopUp = ((IMAPView *)imapView)->imapSupportedMechanismsPopUp;
  imapViewMailboxColumn = ((IMAPView *)imapView)->imapViewMailboxColumn;
  imapSubscriptionColumn = ((IMAPView *)imapView)->imapSubscriptionColumn;
  imapOutlineView = ((IMAPView *)imapView)->imapOutlineView;
  imapMatrix = ((IMAPView *)imapView)->imapMatrix;

  // POP3View
  pop3View = [[POP3View alloc] initWithParent: self];
  [pop3View layoutView];
  pop3LeaveOnServer = ((POP3View *)pop3View)->pop3LeaveOnServer;
  pop3DaysField = ((POP3View *)pop3View)->pop3DaysField;
  pop3UseAPOP = ((POP3View *)pop3View)->pop3UseAPOP;

  // UNIXView
  unixView = [[UNIXView alloc] initWithParent: self];
  [unixView layoutView];
  unixMailspoolFileField = ((UNIXView *)unixView)->unixMailspoolFileField;

  // We load our Send view
  sendView = [[SendView alloc] initWithParent: self];
  [sendView layoutView];
  sendTransportMethodPopUpButton = ((SendView *)sendView)->sendTransportMethodPopUpButton;

  sendMailerView = [[SendMailerView alloc] initWithParent: self];
  [sendMailerView layoutView];
  sendMailerField = ((SendMailerView *)sendMailerView)->sendMailerField;

  sendSMTPView = [[SendSMTPView alloc] initWithParent: self];
  [sendSMTPView layoutView];
  sendSMTPHostField = ((SendSMTPView *)sendSMTPView)->sendSMTPHostField;
  sendSMTPPortField = ((SendSMTPView *)sendSMTPView)->sendSMTPPortField;
  sendSMTPUsernameField = ((SendSMTPView *)sendSMTPView)->sendSMTPUsernameField;
  sendSMTPPasswordSecureField = ((SendSMTPView *)sendSMTPView)->sendSMTPPasswordSecureField;
  sendRememberPassword = ((SendSMTPView *)sendSMTPView)->sendRememberPassword;
  sendUseSecureConnection = ((SendSMTPView *)sendSMTPView)->sendUseSecureConnection;
  sendAuthenticateUsingButton = ((SendSMTPView *)sendSMTPView)->sendAuthenticateUsingButton;
  sendSupportedMechanismsButton = ((SendSMTPView *)sendSMTPView)->sendSupportedMechanismsButton;
  sendSupportedMechanismsPopUp = ((SendSMTPView *)sendSMTPView)->sendSupportedMechanismsPopUp;

  // We load our Mailboxes view
  mailboxesView = [[MailboxesView alloc] initWithParent: self];
  [mailboxesView layoutView];
  mailboxesDefaultInboxPopUpButton = ((MailboxesView *)mailboxesView)->mailboxesDefaultInboxPopUpButton;
  mailboxesDefaultSentPopUpButton = ((MailboxesView *)mailboxesView)->mailboxesDefaultSentPopUpButton;
  mailboxesDefaultDraftsPopUpButton = ((MailboxesView *)mailboxesView)->mailboxesDefaultDraftsPopUpButton;
  mailboxesDefaultTrashPopUpButton = ((MailboxesView *)mailboxesView)->mailboxesDefaultTrashPopUpButton;
  mailboxesDefaultInboxLabel = (NSTextField *)((MailboxesView *)mailboxesView)->mailboxesDefaultInboxLabel;
  mailboxesRefreshButton = ((MailboxesView *)mailboxesView)->mailboxesRefreshButton;

  RELEASE(aWindow);
#endif

  [[self window] setTitle: _(@"")];
  
  // We add all our views
#ifndef MACOSX
  [[tabView tabViewItemAtIndex: 0] setView: personalView];
  [[tabView tabViewItemAtIndex: 1] setView: receiveView];
  [[tabView tabViewItemAtIndex: 3] setView: sendView];
  [[tabView tabViewItemAtIndex: 4] setView: mailboxesView];
#endif
  [[tabView tabViewItemAtIndex: 2] setView: pop3View];

  // We populate our popup buttons
  [Utilities addItemsToPopUpButton: mailboxesDefaultInboxPopUpButton   usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultSentPopUpButton    usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultDraftsPopUpButton  usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultTrashPopUpButton   usingFolderNodes: allNodes];

  // We select a default item in our receive popup and we refresh the view
  [receivePopUp selectItemWithTitle: _(@"POP3")];
  [self setType: nil];

  // We set the supported IMAP AUTH mechanism
  [imapSupportedMechanismsPopUp removeAllItems];
  [imapSupportedMechanismsPopUp addItemWithTitle: @"Password"]; // Do NOT localize

  // We select a default item in our send popup and we refresh the view
  [sendTransportMethodPopUpButton selectItemWithTitle: _(@"SMTP")];
  [self sendTransportMethodHasChanged: nil];

  // We set the supported SMTP AUTH mechanisms
  [sendSupportedMechanismsPopUp removeAllItems];
  [sendSupportedMechanismsPopUp addItemWithTitle: _(@"None")];

  // We initialize some ivars
  allVisibleFolders = [[NSMutableArray alloc] init];
  store = nil;

  // If the SSL Bundle is NOT installed, lets disable the related buttons.
  if ( !SSL_BUNDLE_IS_INSTALLED )
    {
      [receiveUseSecureConnection setEnabled: NO];
      [sendUseSecureConnection setEnabled: NO];
    }

  return self;
}


//
//
//
- (void) dealloc
{
  NSDebugLog(@"AccountEditorWindowController: -dealloc");
  
  // We close our IMAP connection, if it was open
  if ( store && [store isConnected] )
    {
      [store close];
      RELEASE(store);
    }

  // We release our ivars
  RELEASE(allVisibleFolders);
  RELEASE(key);
  RELEASE(allFolders);
  RELEASE(allNodes);

#ifndef MACOSX
  // We release all our views
  RELEASE(mailboxesView);
  RELEASE(personalView);
  RELEASE(receiveView);
  RELEASE(sendView);
  RELEASE(sendMailerView);
  RELEASE(sendSMTPView);
  RELEASE(imapView);
  RELEASE(pop3View);
  RELEASE(unixView);
#else
  [imapOutlineView setDataSource: nil];
#endif
  [super dealloc];
}


//
// action methods
//
- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [self close];
}


//
// Dummy method so the supportedAuthenticationMechanisms NSPopUpButton doesn't disable
// itself when selecting an enabled item in it.
//
-(void) foo: (id) sender {}


//
//
//
- (IBAction) imapList: (id) sender
{  
  NSEnumerator *theEnumerator;
  FolderNode *nodes;
  
  [self _connectToIMAPServer];

  if ( !store )
    {
      return;
    }

  [allFolders setChildren: nil];

  theEnumerator = [store folderEnumerator];
  
  if ( !theEnumerator )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"An error occured while listing the folders on this IMAP server."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }
  
  nodes = [Utilities folderNodesFromFolders: theEnumerator
		     separator: [store folderSeparator]];
  RETAIN(nodes);
  
  [allFolders setChildren: [nodes children]];
  RELEASE(nodes);
  
  [self _rebuildListOfSubscribedFolders];

  [imapOutlineView reloadData];
}


//
//
//
- (IBAction) imapSubscribe: (id) sender
{
  FolderNode *aFolderNode;
  NSString *aString;
  int row;

  if ( !store )
    {
      NSBeep();
      return;
    }

  row = [imapOutlineView selectedRow];
  aFolderNode = (FolderNode *)[imapOutlineView itemAtRow: row];

  if ( [aFolderNode subscribed] )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"You are already subscribed to this IMAP folder."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }

  aString = [Utilities pathOfFolderFromFolderNode: aFolderNode
		       separator: [store folderSeparator]];

  //
  // RFC3501 does NOT explicitely say we cannot subscribe to a \Noselect mailbox but we
  // assume we can't since we cannot SELECT it, nor APPEND messages to it.
  //
  if ( aFolderNode == allFolders ||
       ([store folderTypeForFolderName: aString] & PantomimeNoSelect) == PantomimeNoSelect )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"You cannot subscribe to this folder."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }
   
  if ( [store subscribeToFolderWithName: aString] )
    {
      if ( ![allVisibleFolders containsObject: aString] )
	{ 
	  [allVisibleFolders addObject: aString];
	}
      
      [aFolderNode setSubscribed: YES];  
      [imapOutlineView setNeedsDisplay: YES];
    }
  else
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"An error occured while subscribing to folder:\n%@."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   aString);
    }
}


//
//
//
//
- (IBAction) imapSupportedMechanismsButtonClicked: (id) sender
{
  NSMenuItem *theItem;
  NSArray *theArray;
  NSString *aString;
  int i;

  [self _connectToIMAPServer];

  if ( !store )
    {
      return;
    }

  [imapSupportedMechanismsPopUp removeAllItems];

  // The standard "password" authentication
  theItem = [[NSMenuItem alloc] initWithTitle: @"Password"
				action: NULL
				keyEquivalent: @""];
  [theItem setAction: @selector(foo:)];
  [theItem setEnabled: YES];
  [theItem setTarget: self];
  [[imapSupportedMechanismsPopUp menu] addItem: theItem];
  
  theArray = [store supportedMechanisms];
  
  for (i = 0; i < [theArray count]; i++)
    {
      aString = [theArray objectAtIndex: i];

      theItem = [[NSMenuItem alloc] initWithTitle: aString
				    action: NULL
				    keyEquivalent: @""];
      
      if ( [aString caseInsensitiveCompare: @"CRAM-MD5"] == NSOrderedSame || 
	   [aString caseInsensitiveCompare: @"LOGIN"] == NSOrderedSame )
	{
	  [theItem setAction: @selector(foo:)];
	  [theItem setEnabled: YES];
	}
      else
	{
	  [theItem setAction: NULL];
	  [theItem setEnabled: NO];
	}

      [theItem setTarget: self];
      [[imapSupportedMechanismsPopUp menu] addItem: theItem];
      RELEASE(theItem);
    }

  [imapSupportedMechanismsPopUp selectItemAtIndex: 0];
}


//
//
//
- (IBAction) imapUnsubscribe: (id) sender
{
  FolderNode *aFolderNode;
  NSString *aString;
  int row;

  if ( !store )
    {
      NSBeep();
      return;
    }

  row = [imapOutlineView selectedRow];
  aFolderNode = (FolderNode *)[imapOutlineView itemAtRow: row];

  if ( ![aFolderNode subscribed] )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"You are already unsubscribed from this IMAP folder."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }

  
  aString = [Utilities pathOfFolderFromFolderNode: aFolderNode
		       separator: [store folderSeparator]];
  
  if ( [store unsubscribeToFolderWithName: aString] )
    {
      NSString *pathToFile;

      [allVisibleFolders removeObject: aString];
      [aFolderNode setSubscribed: NO];  
      [imapOutlineView setNeedsDisplay: YES];

      // We remove the cache file.
      pathToFile = [NSString stringWithFormat: @"%@/IMAPCache_%@_%@", 
			     GNUMailUserLibraryPath(),
			     [store name],
			     [Utilities flattenPathFromString: aString
					separator: [store folderSeparator]] ];
      
      NS_DURING
	[[NSFileManager defaultManager] removeFileAtPath: pathToFile
					handler: nil];
      NS_HANDLER
	// Under GNUstep, if we pass something that can't be converted to a cString
	// to -removeFileAtPath, it throws an exception.
	NSDebugLog(@"Exception occured while removing the cache file.");
      NS_ENDHANDLER
    }
  else
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"An error occured while unsubscribing to folder:\n%@."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   aString);
    }
}


//
// FIXME: This method won't work too well if the account's key has changed.
//
- (IBAction) mailboxesRefreshButtonClicked: (id) sender
{
  NSMutableDictionary *aMutableDictionary;
  FolderNode *aFolderNode, *nodes;
  FolderNodePopUpItem *aPopUpItem;
 
  [self _connectToIMAPServer];

  if ( !store )
    {
      return;
    }

  if ( ![self _accountNameIsValid] )
    {
      return;
    }

  // Get the list of subscribed folders
  [allVisibleFolders removeAllObjects];

  if ( [[imapMatrix selectedCell] tag] == 1 )
    {
      [allVisibleFolders addObjectsFromArray: [[store subscribedFolderEnumerator] allObjects]];
    }
  else
    { 
      [allVisibleFolders addObjectsFromArray: [[store folderEnumerator] allObjects]];
    }
  
  // If INBOX isn't in that list, subscribe to Inbox and add it to the
  // list of subscribed folder. We assume we always sucessfully subscribe 
  // to this mailbox.
  if ( ![allVisibleFolders containsObject: @"INBOX"] )
    {
      [store subscribeToFolderWithName: @"INBOX"];
      [allVisibleFolders addObject: @"INBOX"];
      [self imapList: self];
    }

  // Reload the popups with the list of subscribed folders. We first populate our "allNodes" array
  // with the newly fetched list of folders.
  aFolderNode = [allNodes childWithName: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];

  // This account is a newly created one
  if ( !aFolderNode )
    {
      nodes = [Utilities folderNodesFromFolders: [allVisibleFolders objectEnumerator]
			 separator: @"/"];
      
      [nodes setName: [personalAccountNameField stringValue]];
      [allNodes addChild: nodes];
      [nodes setParent: allNodes];
    }
  // We replace the values
  else
    {
      RETAIN(aFolderNode);
      [allNodes removeChild: aFolderNode];

      nodes = [Utilities folderNodesFromFolders: [allVisibleFolders objectEnumerator]
			 separator: @"/"];
      [nodes setName: [aFolderNode name]];
      [allNodes addChild: nodes];
      [nodes setParent: allNodes];
      RELEASE(aFolderNode);
    }

  //
  // We save the list of selected mailboxes
  //
  aMutableDictionary = [[NSMutableDictionary alloc] init];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultSentPopUpButton
	name: @"SENTFOLDERNAME"
	dictionary: aMutableDictionary];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultDraftsPopUpButton
	name: @"DRAFTSFOLDERNAME"
	dictionary: aMutableDictionary];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultTrashPopUpButton
	name: @"TRASHFOLDERNAME"
	dictionary: aMutableDictionary];

  // We reload our all subscribed mailboxes
  [Utilities addItemsToPopUpButton: mailboxesDefaultInboxPopUpButton   usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultSentPopUpButton    usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultDraftsPopUpButton  usingFolderNodes: allNodes];
  [Utilities addItemsToPopUpButton: mailboxesDefaultTrashPopUpButton   usingFolderNodes: allNodes];

  // For the Inbox popup, select INBOX
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [NSString stringWithFormat: @"imap://%@@%@/INBOX", 
									   [receiveUsernameField stringValue],
									   [receiveServerNameField stringValue]]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultInboxPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  [mailboxesDefaultInboxPopUpButton selectItem: aPopUpItem];
  
  // Sent
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [aMutableDictionary objectForKey: @"SENTFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultSentPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  [mailboxesDefaultSentPopUpButton selectItem: aPopUpItem];

  // Drafts
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [aMutableDictionary objectForKey: @"DRAFTSFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultDraftsPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  [mailboxesDefaultDraftsPopUpButton selectItem: aPopUpItem];
  
  // Deleted
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [aMutableDictionary objectForKey: @"TRASHFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultTrashPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  [mailboxesDefaultTrashPopUpButton selectItem: aPopUpItem];

  RELEASE(aMutableDictionary);
}


//
//
//
- (IBAction) okClicked: (id) sender
{
  NSMutableDictionary *aMutableDictionary, *allValues, *allPreferences, *allAccounts;
  NSNumber *serverTypeValue;
  NSString *theKey;
  int value;

  // Before doing anything, we verify if we got any invalid preferences set
  if ( ![self _accountNameIsValid] )
    {
      return;
    }

  if ( ![self _receiveInformationIsValid] )
    {
      return;
    }

  if ( ![self _sendInformationIsValid] )
    {
      return;
    }

  if ( ![self _mailboxesSelectionIsValid] )
    {
      return;
    }
  

  // We are now ready to save the new account (or the edited account);
  allPreferences = [[NSMutableDictionary alloc] init];
  allAccounts = [[NSMutableDictionary alloc] init];

  [allPreferences addEntriesFromDictionary: [[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"]];

  if ( [allPreferences objectForKey: @"ACCOUNTS"] )
    {
      [allAccounts addEntriesFromDictionary: [allPreferences objectForKey: @"ACCOUNTS"]];
    }

  if ( [self key] )
    {
      allValues = AUTORELEASE([[NSMutableDictionary alloc] initWithDictionary: [allAccounts objectForKey: [self key]]]);
    }
  else
    {
      allValues = AUTORELEASE([[NSMutableDictionary alloc] init]);
    }
  
  //
  // PERSONAL
  //
  aMutableDictionary = [[NSMutableDictionary alloc] init];
  [personalSignaturePopUp synchronizeTitleAndSelectedItem];

  [aMutableDictionary setObject: [personalNameField stringValue]  forKey: @"NAME"];
  [aMutableDictionary setObject: [personalEMailField stringValue]  forKey: @"EMAILADDR"];
  [aMutableDictionary setObject: [personalReplyToField stringValue]  forKey: @"REPLYTOADDR"];
  [aMutableDictionary setObject: [personalOrganizationField stringValue]  forKey: @"ORGANIZATION"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [personalSignaturePopUp indexOfSelectedItem]]  forKey: @"SIGNATURE_SOURCE"];
  [aMutableDictionary setObject: [personalSignatureField stringValue]  forKey: @"SIGNATURE"];
  [allValues setObject: aMutableDictionary  forKey: @"PERSONAL"];
  RELEASE(aMutableDictionary);
  
  //
  // RECEIVE
  //
  aMutableDictionary = [[NSMutableDictionary alloc] init];
  [receivePopUp synchronizeTitleAndSelectedItem];

  // We set the type of our server and our port
  if ( [[receivePopUp titleOfSelectedItem] isEqualToString: _(@"POP3")] )
    {
      serverTypeValue = [NSNumber numberWithInt: POP3];
      
      if ([receiveServerPortField intValue] > 0)
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveServerPortField intValue]]  forKey: @"PORT"];
	}
      else
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: POP3_SERVICE_PORT]  forKey: @"PORT"];
	}
    }
  else if ( [[receivePopUp titleOfSelectedItem] isEqualToString: _(@"IMAP")] )
    {
      serverTypeValue = [NSNumber numberWithInt: IMAP];
      
      if ([receiveServerPortField intValue] > 0)
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveServerPortField intValue]]  forKey: @"PORT"];
	}     
      else
	{
	  [aMutableDictionary setObject: [NSNumber numberWithInt: IMAP_SERVICE_PORT]  forKey: @"PORT"];
	}
    }
  else
    {
      serverTypeValue = [NSNumber numberWithInt: UNIX];
    }

  
  // We get the "new" key
  theKey = [personalAccountNameField stringValue];
  
  //
  // Before potentially removing the other values of our configuration (if for example,
  // theKey is different from [self key]), we save our list of subscribed IMAP folders.
  //
  if ( [allAccounts objectForKey: [self key]] &&
       [[allValues objectForKey: @"RECEIVE"] objectForKey: @"SUBSCRIBED_FOLDERS"] &&
       [allVisibleFolders count] == 0 )
    {
      [allVisibleFolders addObjectsFromArray: [[allValues objectForKey: @"RECEIVE"] objectForKey: @"SUBSCRIBED_FOLDERS"]];
    }

  if ( ![theKey isEqualToString: [self key]] )
    {
      // We don't try to remove it if it's not in there!
      if ( [allAccounts objectForKey: [self key]] )
    	{
    	  [allAccounts removeObjectForKey: [self key]];
    	}
      [self setKey: theKey];
    }
  
  // We set the SERVER TYPE pref value
  [aMutableDictionary setObject: serverTypeValue  forKey: @"SERVERTYPE"];
  
  // We set the rest of the informations
  [aMutableDictionary setObject: [receiveServerNameField stringValue]  forKey: @"SERVERNAME"];
  [aMutableDictionary setObject: [receiveUsernameField stringValue]  forKey: @"USERNAME"];
  
  // We save the password or we remove it from the defaults database
  if ( [receiveRememberPassword state] == NSOnState )
    {
      [aMutableDictionary setObject: [Utilities encryptPassword: [receivePasswordSecureField stringValue]
						withKey: [NSString stringWithFormat: @"%@ @ %@", [receiveUsernameField stringValue],
								   [receiveServerNameField stringValue]]]  
			  forKey: @"PASSWORD"];
    }
  else
    {
      // We don't try to remove it if it's not in there!
      if ( [aMutableDictionary objectForKey: @"PASSWORD"] )
	{
	  [aMutableDictionary removeObjectForKey: @"PASSWORD"];
	}
    }

  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveRememberPassword state]]  forKey: @"REMEMBERPASSWORD"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveCheckOnStartup state]]  forKey: @"CHECKONSTARTUP"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveUseSecureConnection state]]  forKey: @"USESECURECONNECTION"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [receiveMatrix selectedRow]]  forKey: @"RETRIEVEMETHOD"];  
  
  value = [receiveMinutesField intValue];

  if ( value <= 0 )
    {
      value = 1;
    }

  [aMutableDictionary setObject: [NSNumber numberWithInt: value]  forKey: @"RETRIEVEMINUTES"];

  // Our POP3 defaults
  [aMutableDictionary setObject: [NSNumber numberWithInt: [pop3LeaveOnServer state]]  forKey: @"LEAVEONSERVER"];

  value = [pop3DaysField intValue];
  [aMutableDictionary setObject: [NSNumber numberWithInt: (value <= 0 ? 365 : value)]  forKey: @"RETAINPERIOD"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [pop3UseAPOP state]]  forKey: @"USEAPOP"];


  // Our IMAP defaults - authentication mechanism and our subscribed list of folders for IMAP
  [aMutableDictionary setObject: [imapSupportedMechanismsPopUp titleOfSelectedItem] forKey: @"AUTH_MECHANISM"];
  [aMutableDictionary setObject: allVisibleFolders  forKey: @"SUBSCRIBED_FOLDERS"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [[imapMatrix selectedCell] tag]]
		      forKey: @"SHOW_WHICH_MAILBOXES"];

  // Our UNIX defaults
  [aMutableDictionary setObject: [unixMailspoolFileField stringValue]  forKey: @"MAILSPOOLFILE"];
  
  // We now save the new defaults pour this server
  [allValues setObject: aMutableDictionary  forKey: @"RECEIVE"];
  RELEASE(aMutableDictionary);
  

  //
  // SEND
  //
  aMutableDictionary = [[NSMutableDictionary alloc] init];
  [sendTransportMethodPopUpButton synchronizeTitleAndSelectedItem];
  [sendSupportedMechanismsPopUp synchronizeTitleAndSelectedItem];

  [aMutableDictionary setObject: [NSNumber numberWithInt: ([sendTransportMethodPopUpButton indexOfSelectedItem] + 1)] forKey: @"TRANSPORT_METHOD"];
  [aMutableDictionary setObject: [sendMailerField stringValue] forKey: @"MAILER_PATH"];
  [aMutableDictionary setObject: [sendSMTPHostField stringValue] forKey: @"SMTP_HOST"];
  
  // SMTP port
  value = [sendSMTPPortField intValue];
  
  if ( value <= 0 )
    {
      value = SMTP_PORT;
    }
  
  [aMutableDictionary setObject: [NSNumber numberWithInt: value] forKey: @"SMTP_PORT"];
  
  // SMTP username and password
  [aMutableDictionary setObject: [sendSMTPUsernameField stringValue] forKey: @"SMTP_USERNAME"];
  
  if ( [sendRememberPassword state] == NSOnState )
    {
      [aMutableDictionary setObject: [Utilities encryptPassword: [sendSMTPPasswordSecureField stringValue]
						withKey: [NSString stringWithFormat: @"%@ @ %@", [sendSMTPUsernameField stringValue],
								   [sendSMTPHostField stringValue]]]
			  forKey: @"SMTP_PASSWORD"];
    }
  else
    {
      [aMutableDictionary removeObjectForKey: @"SMTP_PASSWORD"];
    }
  
  [aMutableDictionary setObject: [NSNumber numberWithInt: [sendRememberPassword state]] forKey: @"REMEMBERPASSWORD"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [sendUseSecureConnection state]] forKey: @"USESECURECONNECTION"];
  [aMutableDictionary setObject: [NSNumber numberWithInt: [sendAuthenticateUsingButton state]] forKey: @"SMTP_AUTH"];
  [aMutableDictionary setObject: [sendSupportedMechanismsPopUp titleOfSelectedItem] forKey: @"SMTP_AUTH_MECHANISM"];
  [allValues setObject: aMutableDictionary  forKey: @"SEND"];
  RELEASE(aMutableDictionary);


  //
  // MAILBOXES
  //
  aMutableDictionary = [[NSMutableDictionary alloc] init];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultInboxPopUpButton
	name: @"INBOXFOLDERNAME"
	dictionary: aMutableDictionary];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultSentPopUpButton
	name: @"SENTFOLDERNAME"
	dictionary: aMutableDictionary];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultDraftsPopUpButton
	name: @"DRAFTSFOLDERNAME"
	dictionary: aMutableDictionary];
  [self _saveChangesForMailboxesPopUpButton: mailboxesDefaultTrashPopUpButton
	name: @"TRASHFOLDERNAME"
	dictionary: aMutableDictionary];
  [allValues setObject: aMutableDictionary  forKey: @"MAILBOXES"];
  RELEASE(aMutableDictionary);
  

  // We now save back all the accounts in the volatile domain
  [allAccounts setObject: allValues  forKey: [self key]];
  [allPreferences setObject: allAccounts  forKey: @"ACCOUNTS"];

  // FIXME - This is causing a segfault under OS X
#ifndef MACOSX
  [[NSUserDefaults standardUserDefaults] removeVolatileDomainForName: @"PREFERENCES"];
#endif
  [[NSUserDefaults standardUserDefaults] setVolatileDomain: allPreferences
					 forName: @"PREFERENCES"];

  RELEASE(allAccounts);
  RELEASE(allPreferences);
  

  // We finally warn the user if he's adding an IMAP account showing
  // all folders. That could impact the performance of GNUMail.app.
  if ( [self operation] == ACCOUNT_ADD &&
       [serverTypeValue intValue] == IMAP &&
       [[imapMatrix selectedCell] tag] == IMAP_SHOW_ALL )
    {
      NSRunInformationalAlertPanel(_(@"Warning!"),
				   _(@"You have created a new IMAP account showing all mailboxes.\nDepending on the IMAP server, this could slow down GNUMail.app.\nYou might consider modifying the newly created account so it shows only\nsubscribed mailboxes by checking the appropriate option in the \"Receive options\" tab."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
    }

  [NSApp stopModal];
  [self close];
}


//
//
//
- (IBAction) personalLocationButtonClicked: (id) sender
{
  NSOpenPanel *oPanel;
  int result;
  
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection: NO];
  result = [oPanel runModalForDirectory: [GNUMail currentWorkingPath]
		   file: nil 
		   types: nil];
  
  if (result == NSOKButton)
    {
      NSArray *fileToOpen;
      int count;
      
      fileToOpen = [oPanel filenames];
      count = [fileToOpen count];
      
      if (count > 0)
	{
	  NSString *aString;

	  aString = [fileToOpen objectAtIndex: 0];
	  [personalSignatureField setStringValue: aString];
	  [GNUMail setCurrentWorkingPath: [aString stringByDeletingLastPathComponent]];
	}
    }
}


//
//
//
- (IBAction) receiveRememberPasswordClicked: (id) sender
{
  if ( [receiveRememberPassword state] == NSOnState )
    {
      [receivePasswordSecureField setEditable: YES];
      [receiveRememberPassword setNextKeyView: receivePasswordSecureField];
      [receivePasswordSecureField setNextKeyView: receiveCheckOnStartup];
    }
  else
    {
      [receivePasswordSecureField setEditable: NO];
      [receiveRememberPassword setNextKeyView: receiveCheckOnStartup];
    }
}


//
//
//
- (IBAction) receiveSetAutomatically: (id) sender
{
  [receiveMinutesField setEditable: YES];
}


//
//
//
- (IBAction) receiveSetManually: (id) sender
{
  [receiveMinutesField setEditable: NO];
}


//
//
//
- (IBAction) receiveUseSecureConnectionClicked: (id) sender
{
  int aState;

  aState = [receiveUseSecureConnection state];
  
  if ( [[receivePopUp titleOfSelectedItem] isEqualToString: _(@"POP3")] )
    {
      if ( NSOffState == aState &&
           POP3S_SERVICE_PORT == [receiveServerPortField intValue] )
        {
          [receiveServerPortField setIntValue: POP3_SERVICE_PORT];
        }
      else if ( NSOnState == aState &&
                POP3_SERVICE_PORT == [receiveServerPortField intValue] )
        {
          [receiveServerPortField setIntValue: POP3S_SERVICE_PORT];
        }
    }
  else
    {
      if ( NSOffState == aState &&
           IMAPS_SERVICE_PORT == [receiveServerPortField intValue] )
        {
          [receiveServerPortField setIntValue: IMAP_SERVICE_PORT];
        }
      else if ( NSOnState == aState &&
                IMAP_SERVICE_PORT == [receiveServerPortField intValue] )
        {
          [receiveServerPortField setIntValue: IMAPS_SERVICE_PORT];
        }
    }
}


//
//
//
- (IBAction) selectionInPersonalSignaturePopUpHasChanged: (id) sender
{
  [personalSignaturePopUp synchronizeTitleAndSelectedItem];

  if ( [personalSignaturePopUp indexOfSelectedItem] == 0 )
    {
      [personalLocationButton setTitle: _(@"Choose file")];
      [personalLocationLabel setStringValue: _(@"File location:")];
    }
  else
    {
      [personalLocationButton setTitle: _(@"Choose program")];
      [personalLocationLabel setStringValue: _(@"Program location:")];
    }
  
  [personalLocationButton setNeedsDisplay: YES];
  [personalLocationLabel setNeedsDisplay: YES];
}


//
//
//
- (IBAction) sendAuthenticateUsingButtonClicked: (id) sender
{
  BOOL aBOOL;

  aBOOL = ([sendAuthenticateUsingButton state] == NSOnState ? YES : NO);
  
  [sendSMTPUsernameField setEditable: aBOOL];
  [sendRememberPassword setEnabled: aBOOL];

  if ( aBOOL && [sendRememberPassword state] == NSOnState )
    {
      [sendSMTPPasswordSecureField setEditable: YES];
    }
  else
    {
      [sendSMTPPasswordSecureField setEditable: NO];
    }

  [sendSupportedMechanismsButton setEnabled: aBOOL];
  [sendSupportedMechanismsPopUp setEnabled: aBOOL];

  //
  // We adjust the next key views
  //
  if ( aBOOL )
    {
      [sendAuthenticateUsingButton setNextKeyView: sendSupportedMechanismsPopUp];
      [sendSupportedMechanismsPopUp setNextKeyView: sendSupportedMechanismsButton];
      [sendSupportedMechanismsButton setNextKeyView: sendSMTPUsernameField];
      [sendSMTPUsernameField setNextKeyView: sendRememberPassword];
      [sendRememberPassword setNextKeyView: sendSMTPPasswordSecureField];
      [sendSMTPPasswordSecureField setNextKeyView: sendSMTPHostField];
    }
  else
    {
      [sendAuthenticateUsingButton setNextKeyView: sendSMTPHostField];
    }
}


//
//
//
- (IBAction) sendRememberPasswordClicked: (id) sender
{
  if ( [sendRememberPassword state] == NSOnState )
    {
      [sendSMTPPasswordSecureField setEditable: YES];
    }
  else
    {
      [sendSMTPPasswordSecureField setEditable: NO];
    }
}

//
//
//
- (IBAction) sendSupportedMechanismsButtonClicked: (id) sender
{
  NSMenuItem *theItem;
  NSArray *theArray;
  NSString *aString;
  SMTP *aSMTP;
  int i, value;

  [sendSupportedMechanismsPopUp removeAllItems];
  [sendSupportedMechanismsPopUp addItemWithTitle: _(@"None")];

  // We get our SMTP port value (can be other than 25!)
  value = [sendSMTPPortField intValue];

  if ( value <= 0 )
    {
      if ( [sendUseSecureConnection state] == NSOnState )
	{
	  value = SSMTP_PORT;
	}
      else
	{
	  value = SMTP_PORT;
	}
    }
  
  if ( [sendUseSecureConnection state] == NSOnState )
    {
      aSMTP = [[SMTP alloc] initSSLWithName: [sendSMTPHostField stringValue]
			    port: value];
    }
  else
    {
      aSMTP = [[SMTP alloc] initWithName: [sendSMTPHostField stringValue]
			    port: value];
    }
  
  theArray = [aSMTP supportedMechanisms];
  
  for (i = 0; i < [theArray count]; i++)
    {
      aString = [theArray objectAtIndex: i];

      theItem = [[NSMenuItem alloc] initWithTitle: aString
				    action: NULL
				    keyEquivalent: @""];
      
      if ( [aString caseInsensitiveCompare: @"CRAM-MD5"] == NSOrderedSame || 
	   [aString caseInsensitiveCompare: @"LOGIN"] == NSOrderedSame ||
	   [aString caseInsensitiveCompare: @"PLAIN"] == NSOrderedSame )
	{
	  [theItem setAction: @selector(foo:)];
	  [theItem setEnabled: YES];
	}
      else
	{
	  [theItem setAction: NULL];
	  [theItem setEnabled: NO];
	}

      [theItem setTarget: self];
      [[sendSupportedMechanismsPopUp menu] addItem: theItem];
      RELEASE(theItem);
    }

  [aSMTP close];
  RELEASE(aSMTP);
}


//
//
//
- (IBAction) sendTransportMethodHasChanged: (id) sender
{
  [sendTransportMethodPopUpButton synchronizeTitleAndSelectedItem];

  // If we selected "Mailer"
  if ( [sendTransportMethodPopUpButton indexOfSelectedItem] == 0 )
    {
      [sendSMTPView removeFromSuperviewWithoutNeedingDisplay];
      [sendView addSubview: sendMailerView];
      [sendView setNeedsDisplay: YES];
    }
  else
    {
      [sendMailerView removeFromSuperviewWithoutNeedingDisplay];
      [sendView addSubview: sendSMTPView];
      [sendView setNeedsDisplay: YES];
    }
}


//
//
//
- (IBAction) sendUseSecureConnectionClicked: (id) sender
{
  int state;

  state  = [sendUseSecureConnection state];
  
  if ( state == NSOffState &&
       SSMTP_PORT == [sendSMTPPortField intValue] )
    {
      [sendSMTPPortField setIntValue: SMTP_PORT];
    }
  else if ( state == NSOnState &&
	    SMTP_PORT == [sendSMTPPortField intValue] )
    {
      [sendSMTPPortField setIntValue: SSMTP_PORT];
    }
}


//
//
//
- (IBAction) setType: (id) sender
{
  [receivePopUp synchronizeTitleAndSelectedItem];
  
  if ( [[receivePopUp titleOfSelectedItem] isEqualToString: _(@"POP3")] )
    {
      [self _setEnableReceiveUIElements: YES]; 
      [[tabView tabViewItemAtIndex: 2] setView: pop3View];

      // If the number 'IMAP_SERVICE_PORT' was on the text field,
      // we set it to POP3_SERVICE_PORT. Else, we keep that custom port.
      if ( ([receiveServerPortField intValue] == IMAP_SERVICE_PORT || [receiveServerPortField intValue] == 0)  &&
           [receiveUseSecureConnection state] == NSOffState )
	{
	  [receiveServerPortField setIntValue: POP3_SERVICE_PORT];
	}
      else if ( [receiveServerPortField intValue] == IMAPS_SERVICE_PORT &&
                [receiveUseSecureConnection state] == NSOnState )
	{
	  [receiveServerPortField setIntValue: POP3S_SERVICE_PORT];
	}
      [mailboxesDefaultInboxLabel setStringValue: _(@"Save messages received from this account in this mailbox:")];
      [mailboxesDefaultInboxPopUpButton setEnabled: YES];
      [mailboxesRefreshButton setEnabled: NO];
    }
  else if ( [[receivePopUp titleOfSelectedItem] isEqualToString: _(@"IMAP")] )
    {
      [self _setEnableReceiveUIElements: YES];
      [[tabView tabViewItemAtIndex: 2] setView: imapView];

      // If the number 'POP3_SERVICE_PORT' was on the text field,
      // we set it to IMAP_SERVICE_PORT. Else, we keep that custom port.
      if ( [receiveServerPortField intValue] == POP3_SERVICE_PORT &&
           [receiveUseSecureConnection state] == NSOffState )
	{
	  [receiveServerPortField setIntValue: IMAP_SERVICE_PORT];
	}
      else if ( [receiveServerPortField intValue] == POP3S_SERVICE_PORT &&
                [receiveUseSecureConnection state] == NSOnState )
	{
	  [receiveServerPortField setIntValue: IMAPS_SERVICE_PORT];
	}
      [mailboxesDefaultInboxLabel setStringValue: _(@"Primary mailbox on this server (always \"INBOX\" for IMAP):")];
      [mailboxesDefaultInboxPopUpButton setEnabled: NO];
      [mailboxesRefreshButton setEnabled: YES];
    }
  else
    {
      [self _setEnableReceiveUIElements: NO];
      [[tabView tabViewItemAtIndex: 2] setView: unixView];
      [mailboxesDefaultInboxLabel setStringValue: _(@"Save messages received from this account in this mailbox:")];
      [mailboxesDefaultInboxPopUpButton setEnabled: YES];
      [mailboxesRefreshButton setEnabled: NO];
    }

  // We best guess the mail spool file
  [self _bestGuessMailspoolFile];
  
  [tabView setNeedsDisplay: YES];
}



//
//
//
- (IBAction) unixMailspoolFileButtonClicked: (id) sender
{
  NSArray *fileToOpen;
  NSOpenPanel *oPanel;
  NSString *fileName;  
  int count, result;
 
  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection:NO];
  result = [oPanel runModalForDirectory: NSHomeDirectory()  file: nil  types: nil];
  
  if (result == NSOKButton) {

    fileToOpen = [oPanel filenames];
    count = [fileToOpen count];

    if (count > 0) {
      fileName = [fileToOpen objectAtIndex:0];
      [unixMailspoolFileField setStringValue:fileName];
    }
  }
}



//
// Datasource/delegate methods for the outline view
//
- (BOOL)  outlineView: (NSOutlineView *) outlineView
shouldEditTableColumn: (NSTableColumn *) tableColumn
                 item: (id) item
{
  // We prevent any kind of editing in the IMAPView's outline view.
  return NO;
}

- (id) outlineView: (NSOutlineView *) outlineView
	     child: (int) index
	    ofItem: (id) item
{
  
  // root object
  if ( !item )
    {
      return allFolders;
    }
  
  
  if ( [item isKindOfClass: [FolderNode class]] )
    {
      return [(FolderNode *)item childAtIndex: index];
    }
  
  return nil;
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
    isItemExpandable: (id) item
{
  if ( [item isKindOfClass: [FolderNode class]] )
    {
      if ( [(FolderNode *)item childCount] > 0 )
	{
	  return YES;
	}
      else
	{
	  return NO;
	}
    }

  return NO;
}


//
//
//
- (int)        outlineView: (NSOutlineView *) outlineView 
    numberOfChildrenOfItem: (id) item
{
  // Root, always one element
  if ( !item )
    {
      return 1;
    }

  if ( [item isKindOfClass: [FolderNode class]] )
    {
      return [(FolderNode *)item childCount];
    }

  return 0;
}


//
//
//
- (id)         outlineView: (NSOutlineView *) outlineView 
 objectValueForTableColumn: (NSTableColumn *) tableColumn 
		    byItem: (id) item
{
  if ( tableColumn == imapViewMailboxColumn )
    {
      return [(FolderNode *)item name];
    }
  else
    {
      return ([(FolderNode *)item subscribed] == YES ? _(@"Y") : @""); 
    }

  // Never reached...
  return nil;
}


//
// delegate methods
//
- (void) windowDidLoad
{

}


//
// access/mutation methods
//
- (NSString *) key
{
  return key;
}

- (void) setKey: (NSString *) theKey
{
  if ( theKey )
    {
      RETAIN(theKey);
      RELEASE(key);
      key = theKey;
    }
  else
    {
      DESTROY(key);
    }
}


//
//
//
- (int) operation
{
  return operation;
}

- (void) setOperation: (int) theOperation
{
  operation = theOperation;

  if ( operation == ACCOUNT_ADD )
    {
      [[self window] setTitle: _(@"Add an Account...")];

      // We initialize the state of some UI elements so when adding a new account,
      // all UI elements are correctly enabled / disabled initially.
      [personalAccountNameField setStringValue: _(@"<Specify the account name here>")];
      [receivePasswordSecureField setEditable: NO];
      [sendSMTPPortField setIntValue: SMTP_PORT];
      [self sendAuthenticateUsingButtonClicked: self];
    }
  else
    {
      [[self window] setTitle: [NSString stringWithFormat: _(@"Edit the %@ account..."), [self key]]];
    }
}


//
// other methods
//
- (void) initializeFromDefaults
{
  NSNumber *serverTypeValue, *portValue;
  FolderNodePopUpItem *aPopUpItem;
  NSDictionary *allValues;
  NSString *aString;

  //
  // Account's name
  //
  [personalAccountNameField setStringValue: [self key]];

  //
  // PERSONAL
  //
  allValues = [[[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"] objectForKey: @"ACCOUNTS"]
		 objectForKey: [self key]]
		objectForKey: @"PERSONAL"];
  [personalNameField setStringValue: ((aString = [allValues objectForKey: @"NAME"]) ? aString : @"")];
  [personalEMailField setStringValue: ((aString = [allValues objectForKey: @"EMAILADDR"]) ? aString : @"")];
  [personalReplyToField setStringValue: ((aString = [allValues objectForKey: @"REPLYTOADDR"]) ? aString : @"")];
  [personalOrganizationField setStringValue: ((aString = [allValues objectForKey: @"ORGANIZATION"]) ? aString : @"")];
  
  if ( [allValues objectForKey: @"SIGNATURE_SOURCE"] )
    {
      [personalSignaturePopUp selectItemAtIndex: [(NSNumber *)[allValues objectForKey: @"SIGNATURE_SOURCE"] intValue]];
    }
  else
    {
      [personalSignaturePopUp selectItemAtIndex: 0];
    }

  [personalSignatureField setStringValue: ((aString = [allValues objectForKey: @"SIGNATURE"]) ? aString : @"")];

  
  //
  // RECEIVE (and options)
  //  
  // We now get all the data from the volatile user defaults
  allValues = [[[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"]  objectForKey: @"ACCOUNTS"] 
		 objectForKey: [self key]] 
		objectForKey: @"RECEIVE"];
  
  // We decode our type
  serverTypeValue = [allValues objectForKey: @"SERVERTYPE"];
  
  if (serverTypeValue && [serverTypeValue intValue] == IMAP)
    {
      [receivePopUp selectItemWithTitle: _(@"IMAP")];
    }
  else if ( !serverTypeValue ||
	    (serverTypeValue && [serverTypeValue intValue] == POP3) )
    {
      [receivePopUp selectItemWithTitle: _(@"POP3")];
    }
  else
    {
      [receivePopUp selectItemWithTitle: _(@"UNIX")];
    }
  
  // We decode our port
  portValue =  [allValues objectForKey: @"PORT"];
  
  if ( portValue )
    {
      [receiveServerPortField setIntValue: [portValue intValue]];
    }
  else
    {
      if (serverTypeValue && [serverTypeValue intValue] == IMAP)
	{
	  [receiveServerPortField setIntValue: IMAP_SERVICE_PORT];
	}
      else
	{
	  [receiveServerPortField setIntValue: POP3_SERVICE_PORT];
	}
    }

  // We decode the rest of the information. We begin with SERVERNAME.
  if ( allValues && [allValues objectForKey: @"SERVERNAME"] )
    {
      [receiveServerNameField setStringValue: [allValues objectForKey: @"SERVERNAME"] ];
    }
  else
    {
      [receiveServerNameField setStringValue: @""];
    }

  if ( allValues && [allValues objectForKey: @"USERNAME"] )
    {
      [receiveUsernameField setStringValue: [allValues objectForKey: @"USERNAME"] ];
    }
  else
    {
      [receiveUsernameField setStringValue: @""];
    }
  
  // We get our password, if we need to!
  if ( allValues && [allValues objectForKey: @"PASSWORD"] )
    {
      NSString *aPassword;
      
      aPassword = [Utilities decryptPassword: [allValues objectForKey: @"PASSWORD"]
			     withKey: [NSString stringWithFormat: @"%@ @ %@", [receiveUsernameField stringValue],
						[receiveServerNameField stringValue]]];
      
      if ( !aPassword )
	{
	  aPassword = @"";
	}

      [receivePasswordSecureField setStringValue: aPassword];
    }
  else
    {
      [receivePasswordSecureField setStringValue: @""];
    }
  
  // REMEMBERPASSWORD
  if ( allValues && [allValues objectForKey: @"REMEMBERPASSWORD"] )
    {
      [receiveRememberPassword setState: [[allValues objectForKey: @"REMEMBERPASSWORD"] intValue]];
    }
  else
    {
      [receiveRememberPassword setState: NSOffState];
    }
  
  // We update our editable/non-editable status of the password field
  [self receiveRememberPasswordClicked: self];


  // CHECKONSTARTUP
  if ( allValues && [allValues objectForKey: @"CHECKONSTARTUP"] )
    {
      [receiveCheckOnStartup setState: [[allValues objectForKey: @"CHECKONSTARTUP"] intValue]];
    }
  else
    {
      [receiveCheckOnStartup setState: NSOffState];
    }

  // USESECURECONNECTION
  if ( allValues && [allValues objectForKey: @"USESECURECONNECTION"] )
    {
      [receiveUseSecureConnection setState: [[allValues objectForKey: @"USESECURECONNECTION"] intValue]];
    }
  else
    {
      [receiveUseSecureConnection setState: NSOffState];
    }

  // RETRIEVEMETHOD and RETRIEVEMINUTES
  [receiveMatrix selectCellAtRow: [[allValues objectForKey: @"RETRIEVEMETHOD"] intValue]  column: 0];
  [receiveMinutesField setIntValue: [[allValues objectForKey: @"RETRIEVEMINUTES"] intValue]];;


  // POP3 - LEAVEONSERVER
  if ( allValues && [allValues objectForKey: @"LEAVEONSERVER"] )
    {
      [pop3LeaveOnServer setState: [[allValues objectForKey: @"LEAVEONSERVER"] intValue]];
    }
  else
    {
      [pop3LeaveOnServer setState: NSOnState];
    }
  
  // POP3 - RETAINPERIOD
  if ( allValues && [allValues objectForKey: @"RETAINPERIOD"] )
    {
      [pop3DaysField setIntValue: [[allValues objectForKey: @"RETAINPERIOD"] intValue]];
    }
  else
    {
      [pop3DaysField setIntValue: 365];
    }
  
  // POP3 - USEAPOP
  if ( allValues && [allValues objectForKey: @"USEAPOP"] )
    {
      [pop3UseAPOP setState: [[allValues objectForKey: @"USEAPOP"] intValue]];
    }
  else
    {
      [pop3UseAPOP setState: NSOffState];
    }

  // IMAP - AUTH MECHANISM
  if ( allValues && [allValues objectForKey: @"AUTH_MECHANISM"] )
    {    
      // If the method is not in the popup button, let's add it.
      if ( ![imapSupportedMechanismsPopUp itemWithTitle: [allValues objectForKey: @"AUTH_MECHANISM"]] )
	{
	  NSMenuItem *theItem;

	  theItem = [[NSMenuItem alloc] initWithTitle: [allValues objectForKey: @"AUTH_MECHANISM"]
					action: NULL
					keyEquivalent: @""];
	  [theItem setTarget: self];
 	  [theItem setAction: @selector(foo:)];
	  [theItem setEnabled: YES];
	  [[imapSupportedMechanismsPopUp menu] addItem: theItem];
	  RELEASE(theItem);
	}

      [imapSupportedMechanismsPopUp selectItemWithTitle: [allValues objectForKey: @"AUTH_MECHANISM"]];
    }

  // IMAP - SHOW_WHICH_MAILBOXES
  if ( allValues && [[allValues objectForKey: @"SHOW_WHICH_MAILBOXES"] intValue] == 1 )
    {
      [imapMatrix selectCellAtRow: 0  column: 1];
    }
  else
    {
      [imapMatrix selectCellAtRow: 0  column: 0];
    }

  // UNIX - mail spool file
  if ( allValues && [allValues objectForKey: @"MAILSPOOLFILE"] )
    {
      NSString *aString;

      aString = [allValues objectForKey: @"MAILSPOOLFILE"];
   
      if ( [aString length] > 0 )
	{
	  [unixMailspoolFileField setStringValue: aString];
	}
      else
	{
	  NSProcessInfo *processInfo;
	  
	  processInfo = [NSProcessInfo processInfo];
	  aString = [[processInfo environment] objectForKey: @"MAIL"];
	  
	  if ( aString )
	    {
	      [unixMailspoolFileField setStringValue: aString];
	    }
	  else
	    {
	      [unixMailspoolFileField setStringValue: @""];
	    }
	}
    }

  // We refresh our view
  [self setType: nil];

  //
  // SEND
  //
  allValues = [[[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"]  objectForKey: @"ACCOUNTS"] 
		 objectForKey: [self key]] 
		objectForKey: @"SEND"];
  
  [sendTransportMethodPopUpButton selectItemAtIndex: ([[allValues objectForKey: @"TRANSPORT_METHOD"] intValue] - 1) ];
  [sendMailerField setStringValue: ((aString = [allValues objectForKey: @"MAILER_PATH"]) ? aString : @"")];
  [sendSMTPHostField setStringValue: ((aString = [allValues objectForKey: @"SMTP_HOST"]) ? aString : @"")];
  
  if ( [allValues objectForKey: @"SMTP_PORT"] )
    {
      [sendSMTPPortField setIntValue: [[allValues objectForKey: @"SMTP_PORT"] intValue]];
    }
  else
    {
      [sendSMTPPortField setIntValue: SMTP_PORT];
    }

  [sendSMTPUsernameField setStringValue: ((aString = [allValues objectForKey: @"SMTP_USERNAME"]) ? aString : @"")];

  if ( [allValues objectForKey: @"SMTP_PASSWORD"] )
    {
      [sendSMTPPasswordSecureField setStringValue: 
				    [Utilities decryptPassword: [allValues objectForKey: @"SMTP_PASSWORD"]
					       withKey: [NSString stringWithFormat: @"%@ @ %@", [sendSMTPUsernameField stringValue],
								   [sendSMTPHostField stringValue]]]];
    }
  
  if ( [allValues objectForKey: @"SMTP_AUTH"] )
    {
      [sendAuthenticateUsingButton setState: [[allValues objectForKey: @"SMTP_AUTH"] intValue]];
      
      // If the method is not in the popup button, let's add it.
      if ( ![sendSupportedMechanismsPopUp itemWithTitle: [allValues objectForKey: @"SMTP_AUTH_MECHANISM"]] )
	{
	  NSMenuItem *theItem;

	  theItem = [[NSMenuItem alloc] initWithTitle: [allValues objectForKey: @"SMTP_AUTH_MECHANISM"]
					action: NULL
					keyEquivalent: @""];
	  [theItem setTarget: self];
 	  [theItem setAction: @selector(foo:)];
	  [theItem setEnabled: YES];
	  [[sendSupportedMechanismsPopUp menu] addItem: theItem];
	  RELEASE(theItem);
	}

      [sendSupportedMechanismsPopUp selectItemWithTitle: [allValues objectForKey: @"SMTP_AUTH_MECHANISM"]];
    }
  
  if ( [allValues objectForKey: @"REMEMBERPASSWORD"] )
    {
      [sendRememberPassword setState: [[allValues objectForKey: @"REMEMBERPASSWORD"] intValue]];
    }
  else
    {
      [sendRememberPassword setState: NSOffState];
    }
  
  if ( [allValues objectForKey: @"USESECURECONNECTION"] )
    {
      [sendUseSecureConnection setState: [[allValues objectForKey: @"USESECURECONNECTION"] intValue]];
    }
  else
    {
      [sendUseSecureConnection setState: NSOffState];
    }
  
  [self sendTransportMethodHasChanged: nil];
  [self sendAuthenticateUsingButtonClicked: nil];
  

  //
  // MAILBOXES
  //
  allValues = [[[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"] objectForKey: @"ACCOUNTS"] 
		 objectForKey: [self key]] 
		objectForKey: @"MAILBOXES"];
  
  //
  // Default Inbox
  //
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [allValues objectForKey: @"INBOXFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultInboxPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  
  if ( aPopUpItem )
    {
      [mailboxesDefaultInboxPopUpButton selectItem: aPopUpItem];
    }
  else
    {
      // FIXME
    }
  
  
  //
  // Default Sent
  //
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [allValues objectForKey: @"SENTFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultSentPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  
  if ( aPopUpItem )
    {
      [mailboxesDefaultSentPopUpButton selectItem: aPopUpItem];
    }
  else
    {
      // FIXME
    }
  

  //
  // Default Drafts
  //
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [allValues objectForKey: @"DRAFTSFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultDraftsPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  
  if ( aPopUpItem )
    {
      [mailboxesDefaultDraftsPopUpButton selectItem: aPopUpItem];
    }
  else
    {
      // FIXME
    }
  

  //
  // Default Trash
  //
  aPopUpItem = [Utilities folderNodePopUpItemForURLNameAsString: [allValues objectForKey: @"TRASHFOLDERNAME"]
			  usingFolderNodes: allNodes
			  popUpButton: mailboxesDefaultTrashPopUpButton
			  account: [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces]];
  
  if ( aPopUpItem )
    {
      [mailboxesDefaultTrashPopUpButton selectItem: aPopUpItem];
    }
  else
    {
      // FIXME
    }
}

@end


//
// Private implementation
//
@implementation AccountEditorWindowController (Private)

- (BOOL) _accountNameIsValid
{
  if ( [self operation] == ACCOUNT_ADD )
    {
      NSString *aString;

      aString = [[personalAccountNameField stringValue] stringByTrimmingWhiteSpaces];

      if ( [aString length] == 0 ||
	   [aString isEqualToString: _(@"<Specify the account name here>")] ||
	   [[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"] 
	      objectForKey: @"ACCOUNTS"] objectForKey: aString] )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"You must specify a valid account name."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       NULL);
	  
	  return NO;
	}  
    }

  return YES;
}

//
// If the user HAS NOT specified one or the MAIL environment variable isn't defined,
// we try to guess the best path to the mail spool file a user would use.
//
- (void) _bestGuessMailspoolFile
{
  if ( [[[unixMailspoolFileField stringValue] stringByTrimmingWhiteSpaces] length] == 0 )
    {
      BOOL isDir;
      
      if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/var/mail/%@", 
    								       NSUserName()]
    					   isDirectory: &isDir] && !isDir )
	{
	  [unixMailspoolFileField setStringValue: [NSString stringWithFormat: @"/var/mail/%@",
							       NSUserName()]];
	}
      else if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/var/spool/mail/%@",
    									    NSUserName()]
    						isDirectory: &isDir] && ! isDir )
	{
	  [unixMailspoolFileField setStringValue: [NSString stringWithFormat: @"/var/spool/mail/%@",
							    NSUserName()]];
	}
      else if ( [[NSFileManager defaultManager] fileExistsAtPath: [NSString stringWithFormat: @"/usr/spool/mail/%@",
    									    NSUserName()]
    						isDirectory: &isDir] && ! isDir )
	{
	  [unixMailspoolFileField setStringValue: [NSString stringWithFormat: @"/usr/spool/mail/%@",
							    NSUserName()]];
	}
      else
	{
	  [unixMailspoolFileField setStringValue: _(@"< Please choose a mail spool file >")];
	}
    }
}


//
//
//
- (void) _connectToIMAPServer
{
  NSString *aPassword;
  
  // If we already have a connection, do nothing.
  if ( store )
    {
      return;
    }

  // We must establish a new connection to the IMAP server...
  if ( [[receiveServerNameField stringValue] length] == 0 || 
       [[receiveUsernameField stringValue] length] == 0)
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You must specify a valid server name and username\nin the Receive tab."),
		      _(@"OK"), // default
		      NULL,     // alternate
		      NULL);
      return;
    }
  
  aPassword = [[receivePasswordSecureField stringValue] stringByTrimmingWhiteSpaces];
  
  if ( [aPassword length] == 0 )
    {
      PasswordPanelController *theController; 
      int result;
      
      theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
      [[theController window] setTitle: [NSString stringWithFormat: @"%@ @ %@",
						  [receiveUsernameField stringValue],
						  [receiveServerNameField stringValue]] ];
      
      result = [NSApp runModalForWindow: [theController window]];
      
      // If the user has entered a password...
      if (result == NSRunStoppedResponse)
	{
	  aPassword = [theController password];
	}
      else
	{
	  aPassword = @"";
	}
    }


  if ( !store )
    {
      if ( [receiveUseSecureConnection state] == NSOnState )
	{
	  store = [[IMAPStore alloc] initSSLWithName: [receiveServerNameField stringValue]
				     port: [receiveServerPortField intValue]];
	  
	}
      else
	{
	  store = [[IMAPStore alloc] initWithName: [receiveServerNameField stringValue]
				     port: [receiveServerPortField intValue]];
	  
	}
      if ( ![store authenticate: [receiveUsernameField stringValue]
		   password: aPassword
		   mechanism: nil] )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"Authentication error with the IMAP server."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       NULL);
	  
	  [store close];
	  DESTROY(store);
	  
	  return;
	}
    }
}


//
//
//
- (void) _rebuildListOfSubscribedFolders
{
  NSEnumerator *theEnumerator;
  FolderNode *aFolderNode;
  NSString *aPath;
  
  [allVisibleFolders removeAllObjects];

  if ( [[imapMatrix selectedCell] tag] == 1 )
    {
      [allVisibleFolders addObjectsFromArray: [[store subscribedFolderEnumerator] allObjects]];
    }
  else
    {
      [allVisibleFolders addObjectsFromArray: [[store folderEnumerator] allObjects]];
    }
      
  theEnumerator = [allVisibleFolders objectEnumerator];
  
  while ( (aPath = [theEnumerator nextObject]) )
    { 
      aFolderNode = [Utilities folderNodeForPath: aPath
			       using: allFolders
			       separator: [store folderSeparator]];

      if ( aFolderNode &&
	   aFolderNode != allFolders )
	{
	  [aFolderNode setSubscribed: YES];
	}
    }
}


//
// This medhod is used to verify that we don't have an other IMAP account
// with the same username @ servername defined in an other profile. That
// would cause troubles since we wouldn't know which profile to use
// if we have an URLName object with something like imap://ludovic@Sophos.ca/INBOX.
//
- (BOOL) _receiveInformationIsValid
{
  NSString *theAccountName, *aServerName, *aUsername;
  NSEnumerator *theEnumerator;
  
  // We don't need to do any verification if it's a POP3/UNIX account or
  // if we are editing.
  [receivePopUp synchronizeTitleAndSelectedItem];
  if ( [receivePopUp indexOfSelectedItem] != 1 ||
       [self operation] == ACCOUNT_EDIT )
    {
      return YES;
    }

  theEnumerator = [[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"] 
		     objectForKey: @"ACCOUNTS"] keyEnumerator];

  aUsername = [[receiveUsernameField stringValue] stringByTrimmingWhiteSpaces];
  aServerName = [[receiveServerNameField stringValue] stringByTrimmingWhiteSpaces];

  while ( (theAccountName = [theEnumerator nextObject]) )
    {
      NSDictionary *allValues;

      allValues = [[[[[NSUserDefaults standardUserDefaults] volatileDomainForName: @"PREFERENCES"] 
		      objectForKey: @"ACCOUNTS"] objectForKey: theAccountName] objectForKey: @"RECEIVE"];

      if ( [[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP && 
	   [[allValues objectForKey: @"USERNAME"] isEqualToString: aUsername] &&
	   [[allValues objectForKey: @"SERVERNAME"] isEqualToString: aServerName] )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"You already defined an IMAP account with the same\ninformation in the %@ account."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       theAccountName);
	  return NO;
	}
    }
  
  return YES;
}


//
// This method verifies that the information specified in the Send panel
// are valid.
//
- (BOOL) _sendInformationIsValid
{
  [sendTransportMethodPopUpButton synchronizeTitleAndSelectedItem];

  if ( ([sendTransportMethodPopUpButton indexOfSelectedItem]+1) == TRANSPORT_MAILER &&
       [[[sendMailerField stringValue] stringByTrimmingWhiteSpaces] length] == 0 )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"You must specify a valid Mailer path in the Send tab."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return NO;
    }
  else if ( ([sendTransportMethodPopUpButton indexOfSelectedItem]+1) == TRANSPORT_SMTP &&
	    [[[sendSMTPHostField stringValue] stringByTrimmingWhiteSpaces] length] == 0 )
    {
      NSRunInformationalAlertPanel(_(@"Error!"),
				   _(@"You must specify a valid SMTP host name in the Send tab."),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return NO;
    }

  return YES;
}


//
//
//
- (BOOL) _mailboxesSelectionIsValid
{
  VERIFY_SELECTION(mailboxesDefaultInboxPopUpButton, _(@"Inbox"));
  VERIFY_SELECTION(mailboxesDefaultSentPopUpButton, _(@"Sent"));
  VERIFY_SELECTION(mailboxesDefaultDraftsPopUpButton, _(@"Drafts"));
  VERIFY_SELECTION(mailboxesDefaultTrashPopUpButton, _(@"Trash"));

  return YES;
}


//
//
//
- (void) _saveChangesForMailboxesPopUpButton: (NSPopUpButton *) thePopUpButton
					name: (NSString *) theName
				  dictionary: (NSMutableDictionary *) theMutableDictionary
{
  FolderNode *aFolderNode;
  NSString *aString;
  
  [thePopUpButton synchronizeTitleAndSelectedItem];
    
  //
  // FIXME: We must verify for, at least, local store if the folder is "selectable". 
  //
  aFolderNode = [(FolderNodePopUpItem *)[thePopUpButton selectedItem] folderNode];
  
  if ( [aFolderNode parent] == allNodes )
    {
      NSDebugLog(@"Selected an invalid mailbox, ignoring.");
      return;
    }
  
  aString = [Utilities stringValueOfURLNameFromFolderNode: aFolderNode
		       serverName: [[receiveServerNameField stringValue] stringByTrimmingWhiteSpaces]
		       username: [[receiveUsernameField stringValue] stringByTrimmingWhiteSpaces]];
  [theMutableDictionary setObject: aString  forKey: theName];
}


//
//
//
- (void) _setEnableReceiveUIElements: (BOOL) aBOOL
{
  [receiveServerNameField setEditable: aBOOL];
  [receiveServerPortField setEditable: aBOOL];
  [receiveUsernameField setEditable: aBOOL];

  if ( aBOOL && [receiveRememberPassword state] == NSOnState )
    {
      [receivePasswordSecureField setEditable: YES];
    }
  else
    {
      [receivePasswordSecureField setEditable: NO];
    }

  [receiveRememberPassword setEnabled: aBOOL];
  [receiveCheckOnStartup setEnabled: aBOOL];

  if ( SSL_BUNDLE_IS_INSTALLED )
    {
      [receiveUseSecureConnection setEnabled: aBOOL];
    }
}

@end
