#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2005 Free Software Foundation
#
# FILE:
# GFDisplayHandler.py
#
# $Id$
#
#
"""
The base display handler for entries that deal with cursor based
input (aka text).

This class should not be used directly.  Other handlers should 
inherit from this one.
"""

__revision__ = "$Id$"

import sys
#from gnue.forms.input.displayHandlers import Base
from gnue.common import events

# TODO: When Base is used inherit from that instead
class BaseCursor(events.EventAware):
  """
  The base display handler for entries that deal with cursor based
  input (aka text).
  
  This class should not be used directly.  Other handlers should 
  inherit from this one.
  """

  def __init__(self, entry, eventHandler, subEventHandler):
    """
    Constructor

    @param entry: The GFEntry instance associated with this handler
    @param eventHandler: The 
    @param subEventHandler: The event handler this display helper will 
                            user to register it's listeners.
    """
    events.EventAware.__init__(self, eventHandler)
    
    self.entry = entry            # The GFEntry that this instance will support
    self.field = entry._field     # The GFField associated with that GFEntry
    self.editing = False          # Is handler in edit mode
    self.modified = False         # Have we been modified??
    self.value = None             # The latest db-compat value
    self.work = ""                # Our working value
    self.display = ""             # The latest display-formatted value
    self._loadedAllowedValues = False # Have allowed values been loaded
    self.subEventHandler = subEventHandler

    # Cursor based vars
    self._selection1 = None        # Start of highlight
    self._selection2 = None        # End of highlight
    self._cursor = 0               # Cursor position

    # TODO: replace w/an event that asks the
    # TODO: UIdriver if this should happen!
    self.handleCR = sys.platform == "win32"

    self.subEventHandler.registerEventListeners( {
    
                 # "Entry" events
                 'requestKEYPRESS'     : self._addText,
                 'requestCURSORLEFT'   : self._moveCursorLeft,
                 'requestCURSORRIGHT'  : self._moveCursorRight,
                 'requestCURSOREND'    : self._moveCursorToEnd,
                 'requestCURSORHOME'   : self._moveCursorToBegin,
                 'requestCURSORMOVE'   : self._moveCursor,
                 'requestBACKSPACE'    : self._backspace,
                 'requestDELETE'       : self._delete,
                 'beginEDITMODE'       : self._beginEdit,
                 'endEDITMODE'         : self._endEdit,
                 'requestENTER'        : self.__handleENTER,

                 # Selection/clipboard events
                 'requestSELECTWITHMOUSE' : self._selectWithMouse,
                 'requestSELECTALL'       : self._selectAll,
                 'requestSELECTTOHOME'    : self._selectToBegin,
                 'requestSELECTTOEND'     : self._selectToEnd,
                 'requestSELECTLEFT'      : self._selectLeft,
                 'requestSELECTRIGHT'     : self._selectRight,
                 'requestCOPY'            : self._clipboardCopy,
                 'requestCUT'             : self._clipboardCut,
                 'requestPASTE'           : self._clipboardPaste,

                 # Request for direct buffer manipulation
                 'requestINSERTAT'     : self._insertTextAt,
                 'requestDELETERANGE'  : self._deleteRange,
        })
  
  #####################
  #
  # General methods
  #

  def setValue(self, value):
    """
    Sets the stored db-compatible value and marks the handler as not modified.
    
    @param value: The new value 
    """
    self.modified = False
    self.value = value
    self._buildDisplay()


  def getValue(self):
    """
    Gets the stored db-compatible value from the handler
    """
    return self.value

  def getDisplayFiller(self, value):
    """
    Returns the value as it should be displayed in the entry.
    
    @param value: The value to be formatted for display
    """
    return self._buildDisplayHelper(value, False)

  def generateRefreshEvent(self):
    """
    Function to emit an event that will cause forms
    to update the UI.
    """
    assert gDebug (5, "generateRefreshEvent on %s '%s' %s" % \
        (self, self.display, self.entry))
    
    # TODO: this should probably actually happen in UIwxpython!
    if (self.handleCR and type(self.display)=='str'):
      self.dispatchEvent(events.Event('updateEntryEditor',
           object = self.field,
           display=self.display,
           cursor=self._cursor + len(self.display[:self._cursor+1].split('\n'))-1,
           selection=self.getSelectionArea(),
         ))
    else:
      self.dispatchEvent(events.Event('updateEntryEditor',
           object = self.entry,
           display=self.display,
           cursor=self._cursor,
           selection=self.getSelectionArea(),
         ))

  def __beep(self):
    """
    Generates an event requesting that the UI beep.
    """
    #TODO: Beep not working atm.
    self.dispatchEvent(events.Event('formBEEP'))

  def isPending(self):
    """
    Return True if the display handler is in edit mode and has modified text
    """
    return self.editing and self.modified

  #####################
  #
  # Editing methods
  #

  def _beginEdit(self, event):
    """
    Notifies the handler that it will be doing edits.
    
    Called when a widget first gets focus. It places the display handler into
    edit mode, syncs the current value with the GFField associated with this
    display handler, and creates the string to display in the form.
    """
    self.editing = self.field.isEditable(event._form.getCurrentMode())
    self.modified = False

    # TODO: Replace with formatter
    self.setValue(self.field.getValue())
    self.work = self._buildDisplayHelper(self.value, False)
    self._buildDisplay()

    self._cursor = len(self.display)
    # Ensure cursor is properly placed.
    self.generateRefreshEvent()

  def _endEdit(self, event):
    """
    Called when a widget loses focus or when ENTER is hit.
    """
    if not self.editing:
      return

    if not self._loadedAllowedValues:
      self.field.allowedValues()
      self._loadedAllowedValues = True


    # If this event returns __error__, then
    # the input was invalid and the entry is
    # still in editing mode.  __errortext__
    # will contain an appropriate error message.
    #
    # If no error, then event.modified is true
    # if the field was modified and the event's
    # result is the final entry value.

    self._selection1 = None

    if self.modified:
      # TODO: buildValue should raise an exception instead of return False
      if self._buildValue():
        if self.field._allowedValues and \
           not self.field._allowedValues.has_key("%s" % self.value):
          self.work = ""
          event.__error__ = True
          event.__errortext__ = u_("Invalid value '%s' for field") % self.value
          return

        self.editing = False
        event.__results__ = self.value
        event.modified = self.modified
        self.field.setValue(self.value)

        self._buildDisplay()

      else:
        self.dispatchEvent(events.Event('formALERT',
          u_("Invalid input: '%s'") % self.work,_form=event._form))
    else:
      self.editing = False


  def _addText(self, event):
    """
    Handles the adding of new text 
    """
    
    # Skip if this display handler isn't the one currently being edited
    if not self.editing:
      assert gDebug(5, 
        "Entry %s: Received request to add text but isn't in edit mode" % 
        self.entry.name )
      return

    # Get the text to be added forcing to specific case if necessary
    if self.field._lowercase:
      value = event.text.lower()
    elif self.field._uppercase:
      value = event.text.upper()
    else:
      value = event.text
    
    # -------------------------------------------------------------------------
    # Validate the input
    # TODO: This will be moved to the mask system!!
    # -------------------------------------------------------------------------
    if hasattr(self.field,'maxLength') and \
       len(self.work)  + len(value) > self.field.maxLength:
      # TODO: Should we beep?
      self.__beep()
      assert gDebug (6, "Entry %s: Max length reached"  % self.entry.name )
      return

    if ( self.field._numeric and \
         self.field._block.mode == 'normal' ):
      for char in value:
        if not (char.isdigit() or char in '.-') :
          # TODO: Should we beep?
          assert gDebug (6, "Entry %s: Invalid numeric input" % self.entry.name)
          return

    # -------------------------------------------------------------------------
    # Insert the text
    # -------------------------------------------------------------------------
    # To do overstrike, we'll fudge by first "highlighting"
    # the character to replace, then use the selection logic.

    if  getattr(event, 'overstrike', False) and self._selection1 is None:
      self._selection1 = self._cursor
      self._selection2 = self._selection1 + 1
        
    if self._selection1 is not None:
      # If text is selected, then we will replace

      minSelectionPos = min(self._selection1, self._selection2)
      maxSelectionPos = max(self._selection1, self._selection2)

      self.work = self.work[:minSelectionPos]  \
                   + value        \
                   + self.work[maxSelectionPos:]

      self._selection1 = None
      self._cursor = minSelectionPos + len(value)

    else:
      # Otherwise just graft the new text in place

      self.work = self.work[:self._cursor] \
                   + value                \
                   + self.work[self._cursor:]

      self._cursor += len(value)


    event.__dropped__ = True
    event.refreshDisplay = True
    self.modified = True
    self._buildDisplay()

    # Update the field. This means PRE-CHANGE and POST-CHANGE will get fired
    # now. For now, only do this here if we are a lookup.
    if hasattr(self.field, 'fk_source'):
      self._buildValue()
      self.field.setValue(self.value)

  def _insertTextAt(self, event):
    """
    Insert text at specified position
    """
    if not self.editing:
      assert gDebug(5, 
        "Entry %s: Received request to insert text but not in edit mode" % 
        self.entry.name )
      return

    # set cursor position
    self._cursor = event.position

    # add the event text
    self._addText(event)

  def _deleteRange(self, event):
    """
    Deletes the requested range of test from the entry
    
    @param event: The GFEvent making the request
    """
    if not self.editing:
      assert gDebug(5, 
        "Entry %s: Received request to delete text but not in edit mode" % 
        self.entry.name )
      return

    self._selection1 = event.start_pos
    self._selection2 = event.end_pos
    self._cursor     = event.position
    self._delete (event)

  def __handleENTER(self, event):
    """
    Private function to handle enter key presses in an multiline entry.
    
    @param event: The GFEvent making the request
    """
    if gConfigForms('EnterIsNewLine') and \
       hasattr(self.entry, 'Char__height') and \
       self.entry.Char__height > 1:
      event.text = '\n'
      self._addText(event)
      event.drop()

  # ===========================================================================
  # Cursor movement functions
  # ===========================================================================

  def _backspace(self, event):
    """
    Delete backwards one character
    
    @param event: The GFEvent making the request
    """
    if not self.editing:
      assert gDebug(5, 
        "Entry %s: Received request to backspace but not in edit mode" % 
        self.entry.name )
      return


    # If we have a "selection", then act like a "delete"
    if self._selection1 != None:
      self._delete(event)
      return

    precurs = self._cursor
    self._moveCursorLeft(event)

    if self._cursor != precurs:
      event.overstrike = True
      event.text = ""

      self._addText(event)

  def _delete(self, event):
    """
    Delete forward one character
    
    @param event: The GFEvent making the request
    """
    if not self.editing:
      assert gDebug(5, 
        "Entry %s: Received request to delete text but not in edit mode" % 
        self.entry.name )
      return

    event.overstrike = True
    event.text = ""

    self._addText(event)

  def _moveCursor(self, event, selecting=False):
    """
    Moves the cursor to the specified position optionally selecting the text.
    
    @param event: The GFEvent making the request
    @param selecting: Boolean indicating if text should be selected 
                      as part of the cursor move
    """
    if not selecting:
      self._selection1 = None

    self._cursor = min(event.position, len(self.display))
    event.refreshDisplay = True


  def _moveCursorLeft(self, event, selecting=False):
    """
    Moves the cursor to the left optionally selecting the text.
    
    @param event: The GFEvent making the request
    @param selecting: Boolean indicating if text should be selected 
                      as part of the cursor move
    """
    if not selecting:
      self._selection1 = None

    if self._cursor > 0:
      self._cursor -= 1
      event.refreshDisplay = True

  def _moveCursorRight(self, event, selecting=False):
    """
    Moves the cursor to the right optionally selecting the text.
    
    @param event: The GFEvent making the request
    @param selecting: Boolean indicating if text should be selected 
                      as part of the cursor move
    """    
    if not selecting:
      self._selection1 = None

    if self._cursor < len(self.display):
      self._cursor += 1
      event.refreshDisplay = True

  def _moveCursorToEnd(self, event, selecting=False):
    """
    Moves the cursor to the end optionally selecting the text.
    
    @param event: The GFEvent making the request
    @param selecting: Boolean indicating if text should be selected 
                      as part of the cursor move
    """
    if not selecting:
      self._selection1 = None

    self._cursor = len(self.display)
    event.refreshDisplay = True


  def _moveCursorToBegin(self, event, selecting=False):
    """
    Moves the cursor to the beginning optionally selecting the text.
    
    @param event: The GFEvent making the request
    @param selecting: Boolean indicating if text should be selected 
                      as part of the cursor move
    """
    if not selecting:
      self._selection1 = None

    self._cursor = 0
    event.refreshDisplay = True


  # ---------------------------------------------------------------------------
  # Selection Support
  # ---------------------------------------------------------------------------

  def setSelectionArea(self, cursor1, cursor2):
    """
    Set the selection area
    
    Starting and ending position can be passed in an any order.
    
    @param cursor1: A starting or ending position for the selection 
    @param cursor2: A starting or ending position for the selection
    """
    self._selection1 = min(cursor1, cursor2)
    self._selection2 = max(cursor1, cursor2)

  def getSelectionArea(self):
    """
    Return the selected area
    
    @return: The selected area as a tuple or None if no selection
    """
    if self._selection1 == None:
      return None
    else:
      return ( min(self._selection1, self._selection2),
               max(self._selection1, self._selection2) )


  def _selectWithMouse(self, event):
    """
    Select an area of text based upon the mouse
        
    @param event: The GFEvent making the request
    """
    self._selection1 = event.position1
    self._selection2 = event.position2
    if self._cursor == self._selection2:
      event.position = self._selection1
    else:
      event.position = self._selection2
    self._moveCursor(event, True)


  def _selectAll (self, event):
    """
    Select the entire text of the entry and move the cursor to the end
        
    @param event: The GFEvent making the request    
    """
    self._selection1 = 0
    self._moveCursorToEnd (event, True)
    self._selection2 = self._cursor


  def _selectLeft(self, event):
    """
    Move the selection cursor to the left one unit
        
    @param event: The GFEvent making the request
    """
    if self._selection1 == None:
      self._selection1 = self._cursor

    self._moveCursorLeft(event, True)
    self._selection2 = self._cursor


  def _selectRight(self, event):
    """
    Move the selection cursor to the right one unit
        
    @param event: The GFEvent making the request
    """ 
    if self._selection1 == None:
      self._selection1 = self._cursor

    self._moveCursorRight(event, True)
    self._selection2 = self._cursor


  
  def _selectToBegin(self, event):
    """
    Select from the current curson position to the beginning of the entry
        
    @param event: The GFEvent making the request
    """
    if self._selection1 == None:
      self._selection1 = self._cursor

    self._moveCursorToBegin(event, True)
    self._selection2 = self._cursor


  
  def _selectToEnd(self, event):
    """
    Select from the current curson position to the end of the entry
        
    @param event: The GFEvent making the request
    """

    if self._selection1 == None:
      self._selection1 = self._cursor

    self._moveCursorToEnd(event, True)
    self._selection2 = self._cursor

  # ---------------------------------------------------------------------------
  # Clipboard Support
  # ---------------------------------------------------------------------------

  # Copy to the clipboard
  def _clipboardCopy(self, event):
    """
    Copy the current selection to the clipboard
        
    @param event: The GFEvent making the request
    """
    if self._selection1 != None:
      sel1, sel2 = self.getSelectionArea ()
      self.dispatchEvent (events.Event ('setCLIPBOARD',
               text = self.display [sel1:sel2]))

    event.refreshDisplay = False


  def _clipboardCut(self, event):
    """
    Cut the current selection and place in the clipboard
    
    @param event: The GFEvent making the request    """
    if not self.editing:
      return self._clipboardCopy(event)

    self._clipboardCopy(event)
    edevent = events.Event("requestKEYPRESS", text="", _form=event._form)
    self.dispatchEvent(edevent)


  def _clipboardPaste(self, event):
    """
    Paste the current contents of the clipboard into the entry
    
    @param event: The GFEvent making the request
    """
    
    if not self.editing:
      return

    event.text = self.dispatchEvent(events.Event('getCLIPBOARD'))
    if event.text != None:
      self._addText(event)

  # --------------------------------------------------------------------------
  # Internal methods
  # --------------------------------------------------------------------------

  def _buildValue(self):
    """
    Private function that builds the db compatible value from 
    the currently working value.
    """
    if self.field._allowedValues:
      if self.work == "":
        self.value = '' # None
      else:
        if self.field._allowedValuesReverse.has_key (self.display):
          self.value = self.field._allowedValuesReverse [self.display]
        else:
          self.value = None
          return False
    else:
      self.value = self.work
    return True


  def _buildDisplayHelper(self, value, editing):
    """
    Builds a string that represents what should be 
    displayed for the field.
    
    An entry can have two different displayable values.  The value
    to be displayed when the entry isn't the one currently being 
    edited.  And the value that is displayed while the entry is being 
    edited.  An example of this would be entries displaying date
    information
    
    @param value: The raw value to be displayed
    @param editing: If true the format the display string as if it
                    were to be edited.
    """
    if self.field._allowedValues:
      # TODO: Verify this comment
      # The field associated with this field is most likely
      # a dropdown and so only valid dropdown values should be displayed
      if editing:
        val = value.lower ()
        
        for disp in self.field._allowedValuesDescr:
          if disp [:len (val)].lower () == val:
            revVal = self.field._allowedValuesReverse [disp]
            return self.field._allowedValues [revVal]

        return value

      if self.field._allowedValues.has_key ("%s" % value):
        return self.field._allowedValues ["%s" % value]
      else:
        return ""

    else:
      if value == None:
        return ""
      else:
        return "%s" % value

  def _buildDisplay(self):
    """
    Private function that rebuilds the display string.
    
    A wrapper function that calls the private _buildDisplayHelper 
    with the appropriate arguments for the mode the display handler is 
    currently in.  (Edit mode or not edit mode :)
    """
    if self.editing:
      self.display = self._buildDisplayHelper(self.work, True)
    else:
      self.display = self._buildDisplayHelper(self.value, False)
