# GNU Enterprise Forms - GF Class Hierarchy - Block
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFBlock.py 7983 2005-09-24 09:52:49Z reinhard $
"""
Classes making up the Block object
"""

__revision__ = "$Id: GFBlock.py 7983 2005-09-24 09:52:49Z reinhard $"

from gnue.forms.GFObjects.GFDataSource import GFDataSource

from gnue.common.apps import errors
from gnue.common.datasources import GConditions
from GFContainer import GFContainer
from gnue.common import events

class GFBlock(GFContainer, events.EventAware):
  """
  Class that represents the <block> tag in a gfd file.
  """
  def __init__(self, parent=None):
    GFContainer.__init__(self, parent, 'GFBlock')

    self.mode = 'normal'
    self._convertAsterisksToPercent = gConfigForms('AsteriskWildcard')

    self._inits = [self.__initialize]

    #self._datasource = None
    self._resultSet = None
    self._dataSourceLink = None

    self._currentRecord = 0
    self._recordCount = 0
    self._queryDefaults = {}
    self._queryValues = {}
    self._lastQueryValues = {}

    self._gap = 0
    self._rows = 1

    self._scrollbars= []

    # Populated by GFEntry's initialize
    self._entryList = []

    # Populated by GFField's initialize
    self._fieldMap = {}
    self._fieldList = []

    # Current top of visible portion
    self.__visibleStart = 0

    # Are we scrolling the visible portion along with record movement?
    self.__scrolling = False

    #
    # Trigger exposure
    #
    self._validTriggers = {
                  'ON-NEWRECORD':   'On-NewRecord',
                  'ON-RECORDLOADED':'On-RecordLoaded',
                  'PRE-COMMIT':     'Pre-Commit',
                  'POST-COMMIT':    'Post-Commit',
                  'PRE-QUERY':      'Pre-Query',
                  'POST-QUERY':     'Post-Query',
                  'PRE-MODIFY':     'Pre-Modify',
                  'PRE-INSERT':     'Pre-Insert',
                  'PRE-DELETE':     'Pre-Delete',
                  'PRE-UPDATE':     'Pre-Update',
                  'PRE-FOCUSOUT':   'Pre-FocusOut',
                  'POST-FOCUSOUT':  'Post-FocusOut',
                  'PRE-FOCUSIN':    'Pre-FocusIn',
                  'POST-FOCUSIN':   'Post-FocusIn',
                  'PRE-CHANGE':     'Pre-Change',
                  'POST-CHANGE':    'Post-Change',
                  }


    self._triggerGlobal = 1
    self._triggerFunctions={
        'clear':{'function':self.processClear,
                  'description':''},
        'getResultSet':{'function':self.getResultSet,
                  'description':''},
        'commit':{'function':self.commit,
                  'description':''},
        'newRecord':{'function':self.newRecord,
                      'description':''},
        'deleteRecord':{'function':self.deleteRecord,
                        'description':''},
        'duplicateRecord':{'function':self.duplicateRecord,
                        'description':'Duplicates the current (non-empty) record into a new record.'},
        'gotoRecord':{'function':self.jumpRecord,
                      'description':''},
        'firstRecord':{'function':self.firstRecord,
                        'description':'Navigates the block to the first record it contains.'},
        'isEmpty':{'function':self.isEmpty,
                    'description':'Returns True if block is empty.'},
        'isSaved':{'function':self.isSaved,
                    'description':'Depriciated: Returns True if block contains no modified records.'},
        'isPending':{'function':self.isPending,
                    'description':'Returns True if block contains modified records.'},
        'lastRecord':{'function':self.lastRecord,
                  'description':'Navigates the block to the last record it contains.'},
        'nextRecord':{'function':self.nextRecord,
                      'description':'Navigates the block to the next record in sequence.'},
        'prevRecord':{'function':self.prevRecord,
                      'description':'Navigates the block to the previous record in sequence.'},
        'jumpRecords':{'function':self.jumpRecords,
                      'description':'Navigates the specified number of records.'},
        'rollback':{'function':self.processRollback,
                    'description':'Clears all records regardless of state from the block'},
        'initQuery':{'function':self.initQuery,
                    'description':'Prepares the block for query input.'},
        'copyQuery':{'function':self.copyQuery,
                    'description':'Prepares the block for query input.'},
        'cancelQuery':{'function':self.cancelQuery,
                    'description':'Cancels query input.'},
        'executeQuery':{'function':self.processQuery,
                        'description':'Executes the current query.'},
        'call': {'function'   : self.callFunction,
                 'description': 'Executes a function of the datasource'},
        'update': {'function': self.updateCurrentRecordSet}
        }

    self._triggerProperties={
          'parent':        {'get':self.getParent},
          'autoCommit':    {'set':self.triggerSetAutoCommit,
                            'get':self.triggerGetAutoCommit },
          'queryable':     {'set':self.triggerSetQueryable,
                            'get':self.triggerGetQueryable },
          'editable':      {'set':self.triggerSetEditable,
                            'get':self.triggerGetEditable },
          'autoCreate':    {'set':self.triggerSetAutoCreate,
                            'get':self.triggerGetAutoCreate },
          'deletable':     {'set':self.triggerSetDeletable,
                            'get':self.triggerGetDeletable },
          'transparent':   {'set':self.triggerSetTransparent,
                            'get':self.triggerGetTransparent },
          'autoNextRecord':{'set':self.triggerSetAutoNextRecord,
                            'get':self.triggerGetAutoNextRecord },
       }


  # Iterator support (Python 2.2+)
  def __iter__(self):
    """
    Iterator support for Python 2.2+

    Allows you to do:
      for foo in myBlock:
        ....
    """

    return _BlockIter(self)


  def _buildObject(self):

    if hasattr(self, 'rows'):
      self._rows = self.rows

    if hasattr(self, 'rowSpacer'):
      self._gap = self.rowSpacer

    if hasattr(self, 'restrictDelete') and self.restrictDelete:
      self.deletable = False
      del self.__dict__['restrictDelete']

    if hasattr(self, 'restrictInsert') and self.restrictInsert:
      self.editable = 'update'
      del self.__dict__['restrictInsert']

    if hasattr(self,'datasource'):
      self.datasource = self.datasource.lower()

    return GFContainer._buildObject(self)

  #
  # Primary initialization
  #
  def __initialize(self):
    self._form = form = self.findParentOfType('GFForm')
    self._logic = logic = self.findParentOfType('GFLogic')

    self._lastValues = {}

    logic._blockList.append(self)
    logic._blockMap[self.name] = self

    # Initialize our events system
    events.EventAware.__init__(self, form._instance.eventController)

    # Create a stub/non-bound datasource if we aren't bound to one
    if not hasattr(self,'datasource') or not self.datasource:
      ds = GFDataSource(self._form)
      ds.type = 'unbound'
      self.datasource = ds.name = "__dts_%s" % id(self)
      form._datasourceDictionary[ds.name] = ds
      ds._buildObject()
      ds.phaseInit()

    self._dataSourceLink = form._datasourceDictionary.get (self.datasource)
    if self._dataSourceLink is None:
      raise errors.ApplicationError, \
          u_("Datasource '%(datasource)s' in block '%(block)s' not found") \
          % {'datasource': self.datasource,
             'block': self.name}

    # Register event handling functions
    self._dataSourceLink.registerEventListeners ({
        'dsResultSetActivated': self.__dsResultSetActivated,
        'dsResultSetChanged'  : self.__dsResultSetActivated, # sic!
        'dsCursorMoved'       : self.__dsCursorMoved,
        'dsRecordInserted'    : self.__dsRecordInserted,
        'dsRecordLoaded'      : self.__dsRecordLoaded,
        'dsRecordTouched'     : self.__dsRecordTouched,
        'dsCommitInsert'      : self.__dsCommitInsert,
        'dsCommitUpdate'      : self.__dsCommitUpdate,
        'dsCommitDelete'      : self.__dsCommitDelete})

    # Get min and max child rows, if applicable
    try:
      self._minChildRows = self._dataSourceLink.detailmin
    except AttributeError:
      self._minChildRows = 0
    try:
      self._maxChildRows = self._dataSourceLink.detailmax
    except AttributeError:
      self._maxChildRows = None

    self.walk(self.__setChildRowSettings)

  def __setChildRowSettings(self, object):
    if hasattr(object,'rows'):
      rows = object._rows = object.rows
    else:
      rows = object._rows = self._rows

    if hasattr(object,'rowSpacer'):
      object._gap = object.rowSpacer
    else:
      object._gap = self._gap


  # ---------------------------------------------------------------------------
  # Event handling functions for datasource events
  # ---------------------------------------------------------------------------

  def __dsResultSetActivated (self, event):
    self._resultSet = event.resultSet
    self._recordCount = self._resultSet.getRecordCount ()
    recno = self._resultSet.getRecordNumber ()
    if recno == -1:
      if not self._resultSet.firstRecord ():
        self.newRecord ()
    else:
      self.switchRecord (recno - self._currentRecord)

  # ---------------------------------------------------------------------------

  def __dsCursorMoved (self, event):
    # Blocks can't cope with current record #-1
    recno = self._resultSet.getRecordNumber ()
    if recno != -1:
      if self.__scrolling:
        self.switchRecord (0)
      else:
        self.switchRecord (recno - self._currentRecord)

  # ---------------------------------------------------------------------------

  def __dsRecordInserted (self, event):
    self._initializingRecord = event.record
    oldmode = self.mode
    self.mode = 'init'
    self.processTrigger ('ON-NEWRECORD')
    self.mode = oldmode

  # ---------------------------------------------------------------------------

  def __dsRecordLoaded (self, event):
    self._initializingRecord = event.record
    oldmode = self.mode
    self.mode = 'init'
    self.processTrigger ('ON-RECORDLOADED')
    self.mode = oldmode

  # ---------------------------------------------------------------------------

  def __dsRecordTouched (self, event):
    # This already gets called by GFField??
    # self.__fireRecordTrigger ('PRE-CHANGE')
    pass

  # ---------------------------------------------------------------------------

  def __dsCommitInsert (self, event):
    self.__fireRecordTrigger ('PRE-INSERT')
    self.__fireRecordTrigger ('PRE-COMMIT')

  # ---------------------------------------------------------------------------

  def __dsCommitUpdate (self, event):
    self.__fireRecordTrigger ('PRE-UPDATE')
    self.__fireRecordTrigger ('PRE-COMMIT')

  # ---------------------------------------------------------------------------

  def __dsCommitDelete (self, event):
    self.__fireRecordTrigger ('PRE-DELETE')
    self.__fireRecordTrigger ('PRE-COMMIT')

  # ---------------------------------------------------------------------------

  def __fireRecordTrigger (self, trigger):
    self.processTrigger (trigger)
    for field in self._fieldList:
      field.processTrigger (trigger)


  # ---------------------------------------------------------------------------
  #
  #
  def getFocusOrder(self):
    list = []
    for field in self._children:
      try:
        list += field._entryList
      except AttributeError:
        pass # Triggers, etc

    return GFContainer.getFocusOrder(self,list)

  #
  # isSaved
  #
  #
  def isSaved(self):
    """
    Returns True if the datasource the block is associated
    with is not pending any changes.
    """
    print "DEPRECIATION WARNING: the use of block.isSaved () is depreciated.", \
          "Please use block.isPending () instead."
    return not self.isPending()

  def isPending(self):
    """
    Returns True if the datasource the block is associated
    with is pending any changes.
    """
    return self._resultSet and self._resultSet.isPending()

  #
  # deleteRecord
  #
  # FIXME: This is a duplicate of GDataSource.__trigger_delete. Do we really
  # need both?
  def deleteRecord(self):
    """
    Doesn't really delete the record but marks it for
    deletion during next commit
    """
    self._resultSet.current.delete()

  #
  # undeleteRecord
  #
  def undeleteRecord(self):
    """
    Removes the delete mark from the record
    """
    self._resultSet.current.undelete()

  #
  # isEmpty()
  #
  def isEmpty(self):
    return self._resultSet.current.isEmpty()

  #
  # getResultSet()
  #
  def getResultSet(self):
    return self._resultSet


  #
  #
  #
  def switchRecord(self, adjustment):
    """
    Moves the proper record into editing position
    """
    newRecord = self._resultSet.getRecordNumber ()
    newRecordCount = self._resultSet.getRecordCount ()

    self.__visibleStart += newRecord - self._currentRecord - adjustment
    if self.__visibleStart > newRecord:
      self.__visibleStart = newRecord
    if self.__visibleStart < newRecord - self._rows + 1:
      self.__visibleStart = newRecord - self._rows + 1
    if self.__visibleStart < 0:
      self.__visibleStart = 0

    self._currentRecord = newRecord
    self._recordCount = newRecordCount

    for field in self._fieldList:
      field.gotNewCurrentRecord()
      for entry in field._entryList:
        # This loop is probably better somewhere else
        entry.recalculateVisible( adjustment, self._currentRecord, 
                                  self._recordCount)
      self._form.updateUIEntry(field)
      self._form.refreshUIEvents()

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar (self.__visibleStart,
          max (self._recordCount, self.__visibleStart + self._rows))

  #
  # newRecord
  #
  # Adds a record to the current records in memory
  #
  def newRecord(self):

    # Focus out
    self.processTrigger('PRE-FOCUSOUT')
    if self.autoCommit and self._resultSet.current:
      self._form.commit()
    self.processTrigger('POST-FOCUSOUT')

    if self._resultSet.insertRecord (self._lastValues):

      self._recordCount = self._resultSet.getRecordCount()

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def duplicateRecord(self, exclude=(), include=()):
    self._resultSet.duplicateRecord(exclude=exclude, include=include)

  def isLastRecord(self):
    return self._resultSet.isLastRecord()

  def isFirstRecord(self):
    return self._resultSet.isFirstRecord()

  def nextRecord(self):
    if not self._resultSet.isLastRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      if self.autoCommit:
        self._form.commit()
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.nextRecord()
      self._recordCount = self._resultSet.getRecordCount()

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

    elif self.autoCreate and not self.isEmpty() and \
         not self.editable in ('update','N'):
      self.newRecord()
      # Go to first field
      self._form.findAndChangeFocus(self._entryList[0])



  def lastRecord(self):
    if not self._resultSet.isLastRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.lastRecord()

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def firstRecord(self):
    if not self._resultSet.isFirstRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.firstRecord()

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def prevRecord(self):
    if not self._resultSet.isFirstRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.prevRecord()

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def commit(self):
    print "DEPRECIATION WARNING: the use of block.commit () is depreciated.", \
          "Please use form.commit () instead."
    self._form.commit ()


  def jumpRecord(self, recordNumber):
    # If recordNumber is negative, move relative to last record
    if recordNumber < 0:
      recordNumber += self._resultSet.getRecordCount()
    if recordNumber < 0:
      raise "Invalid record number"

    if recordNumber != self._resultSet.getRecordNumber():

      if self._form._currentEntry._block == self:
        # Focus out
        self.processTrigger('PRE-FOCUSOUT')
        self.processTrigger('POST-FOCUSOUT')

      if not self._resultSet.setRecord(recordNumber):
        self._resultSet.lastRecord()

      if self._form._currentEntry._block == self:
        # Focus in
        self.processTrigger('PRE-FOCUSIN')
        self.processTrigger('POST-FOCUSIN')
        # Move to correct record in grid
        self._form.findAndChangeFocus (self._form._currentEntry)
        self._form._instance.updateRecordCounter (self._form)


  def jumpRecords(self, adjustment):
    targetRecord = self._resultSet.getRecordNumber() + adjustment

    if targetRecord < 0:
      targetRecord = 0
    elif targetRecord > self._resultSet.getRecordCount():
      targetRecord = self._resultSet.getRecordCount()

    self.jumpRecord(targetRecord)


  # ---------------------------------------------------------------------------
  # Scroll a given number of records
  # ---------------------------------------------------------------------------

  def scrollRecords (self, adjustment):
    """
    Scroll the given number of records.
    """

    self.scrollToRecord (self.__visibleStart + adjustment)


  # ---------------------------------------------------------------------------
  # Scroll to given record number
  # ---------------------------------------------------------------------------

  def scrollToRecord (self, position):
    """
    Scrolls the block to the given position.

    The record number given in position will become the first visible record.
    """

    # First, scroll as far as possible without moving the record pointer
    newVisibleStart = position

    if newVisibleStart > self._currentRecord:
      newVisibleStart = self._currentRecord
    if newVisibleStart < self._currentRecord - self._rows + 1:
      newVisibleStart = self._currentRecord - self._rows + 1

    if newVisibleStart != self.__visibleStart:
      self.switchRecord (self.__visibleStart - newVisibleStart)

    # Now we have to move the record pointer to keep it in the visible portion
    if position != self.__visibleStart:
      self.__scrolling = True
      try:
        self.jumpRecords (position - self.__visibleStart)
      finally:
        self.__scrolling = False
    else:
      # If we didn't move the cursor, we have to update cursor position in UI
      if self._form._currentEntry._block == self:
        self._form.findAndChangeFocus (self._form._currentEntry)


  #
  # processCommit
  #
  def processCommit(self):
    assert gDebug (4, "processing commit on block %s"%self.name,1)

    self.mode='commit'

    self._resultSet.setRecord(self._precommitRecord)

    if self._getMasterBlock () is None:
      self._dataSourceLink.postAll ()

  #
  # Called after the commit on the backend is through.
  #
  def finalizeCommit (self, commit):

    # Synchronize backend -> resultset -> UI
    if self._getMasterBlock () is None:
      self._dataSourceLink.requeryAll (commit)

    # Our recordCount might have changed if records were deleted
    self._recordCount = self._resultSet.getRecordCount()

    # If all records were deleted, create an empty new one.
    if not self._recordCount:
      self.newRecord()

    self.mode='normal'


  #
  # processClear
  #
  def processClear(self):

    # Detail blocks cannot be cleared - they follow their master blindly.
    if self._getMasterBlock () is not None:
      return

    if self._dataSourceLink._connection is not None:
      self._dataSourceLink._connection.rollback ()

    self._dataSourceLink.createEmptyResultSet ()


  #
  # processRollback
  #
  # if recovering=False, then the user requested a rollback
  # if recovering=True, then a commit or such failed and we need to clean up
  # (but not lose state information)
  #
  def processRollback (self, recovering = False, backendRollback = True):

    # Detail blocks cannot be rolled back - they follow their master blindly.
    if self._getMasterBlock () is not None:
      return

    # if called from GFForm the connection's rollback () has been executed
    # already. But if called from a trigger code we do it here
    if backendRollback:
      if self._dataSourceLink._connection is not None:
        self._dataSourceLink._connection.rollback ()

    if not recovering:
      self._dataSourceLink.createEmptyResultSet ()


  #
  # initQuery and processQuery
  #
  def initQuery(self):

    # If Enter-Query is hit once, enter query mode
    # If Enter-Query is hit twice, bring back conditions from last query.
    # If Enter-Query is hit thrice, cancel the query and go into normal mode.

    if self.mode != 'query':
      self.mode = 'query'
      self._query2 = int(gConfigForms("RememberLastQuery"))
      self._queryValues = {}
      self._queryValues.update(self._queryDefaults)
      self.switchRecord(0)

  def copyQuery(self):
    self._query2 = 0
    self._queryValues = {}
    self._queryValues.update(self._lastQueryValues)
    self.switchRecord(0)

  def cancelQuery(self):
    self.mode = 'normal'
    self.switchRecord(0)

  def processQuery(self):
    # Set the maxList to a single master block
    # as a placeholder
    maxList = [self._getTopMasterBlock ()]

    # Find the longest master/detail chain that
    # contains query values.  This will become
    # the chain that is queried
    for block in self._logic._blockList:
      if block._queryValues.keys ():
        templist = [block]
        while (templist [-1])._getMasterBlock ():
          templist.append ((templist [-1])._getMasterBlock ())
        if len (maxList) < len (templist): maxList = templist

    # Store block states
    for block in self._logic._blockList:
      block.mode = 'normal'
      block._lastQueryValues = {}
      block._lastQueryValues.update(block._queryValues)

    # graft in the sloppy query stuff if needed
    for block in maxList:
      for entry in block._entryList:
        if hasattr(entry._field,'sloppyQuery') and \
           block._queryValues.has_key(entry._field):
          block._queryValues[entry._field] = "%"+ \
            "%".join(list(block._queryValues[entry._field]))+"%"

    # Find root block
    rootBlock = maxList [-1]

    # Condition for the master block
    conditions = _generateConditional(rootBlock)

    # Conditions for the detail block
    for block in maxList[:-1]:

      block.processTrigger('PRE-QUERY')
      for field in block._fieldList:
        field.processTrigger('PRE-QUERY')

      c = _generateConditional(block)
      exist = GConditions.GCexist()
      exist.table = block._dataSourceLink.table
      exist.masterlink = block._dataSourceLink.masterlink
      exist.detaillink = block._dataSourceLink.detaillink
      exist._children = [c]
      conditions = GConditions.combineConditions (conditions, exist)

    rootBlock._dataSourceLink.createResultSet(conditions)

    rootBlock._recordCount = rootBlock._resultSet.getRecordCount()

    for block in self._logic._blockList:
      block.processTrigger('POST-QUERY')
      for field in block._fieldList:
        field.processTrigger('POST-QUERY')

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar (self._currentRecord, self._recordCount)


  def registerScrollbar (self, sb):
    self._scrollbars.append (sb)


  # ---------------------------------------------------------------------------
  # Call a datasource's function
  # ---------------------------------------------------------------------------

  # FIXME: This is a duplicate of GDataSource.__trigger_call. Do we really need
  # both?
  def callFunction (self, name, parameters):
    try:
      # Remember the current record; the record pointer is not reliable between
      # postAll and requeryAll!
      current = self._resultSet.current
      self._dataSourceLink.postAll ()
      try:
        res = current.call (name, parameters)
      finally:
        self._dataSourceLink.requeryAll (False)
    finally:
      self.switchRecord (0)

    return res


  # ---------------------------------------------------------------------------
  # Update the current datasource's record set
  # ---------------------------------------------------------------------------

  # FIXME: This is a duplicate of GDataSource.__trigger_update. Do we really
  # need both?
  def updateCurrentRecordSet (self):
    try:
      self._dataSourceLink.postAll ()
      self._dataSourceLink.requeryAll (False)
    finally:
      self.switchRecord (0)


  # ---------------------------------------------------------------------------
  # Return the top level master block of this block
  # ---------------------------------------------------------------------------

  def _getTopMasterBlock (self):

    result = self
    master = result._getMasterBlock ()
    while master is not None:
      result = master
      master = result._getMasterBlock ()
    return result


  # ---------------------------------------------------------------------------
  # Return the master block of this block
  # ---------------------------------------------------------------------------

  def _getMasterBlock (self):

    if self._dataSourceLink.hasMaster ():
      ds = self._dataSourceLink.getMaster ()
      for block in self._logic._blockList:
        if block._dataSourceLink == ds:
          return block
      # return None in case our master is not bound to a block; e.g. if our
      # master is a dropdown
      return None
    else:
      return None


  ###################################################################
  #
  # Trigger settable stuff
  #
  def triggerSetAutoCommit(self, value):
    self.autoCommit = not not value # Force boolean

  def triggerGetAutoCommit(self):
    return self.autoCommit

  def triggerSetQueryable(self, value):
    self.queryable = not not value # Force boolean

  def triggerGetQueryable(self):
    return self.queryable

  def triggerSetEditable(self, value):
    assert (value in ('Y','new','update','N'))
    self.editable = value

  def triggerGetEditable(self):
    return self.editable

  def triggerSetAutoCreate(self, value):
    self.autoCreate = not not value # Force boolean

  def triggerGetAutoCreate(self):
    return self.autoCreate

  def triggerSetDeletable(self, value):
    self.deletable = not not value # Force boolean

  def triggerGetDeletable(self):
    return self.deletable

  def triggerSetTransparent(self, value):
    self.transparent = not not value # Force boolean

  def triggerGetTransparent(self):
    return self.transparent

  def triggerSetAutoNextRecord(self, value):
    self.autoNextRecord = not not value # Force boolean

  def triggerGetAutoNextRecord(self):
    return self.autoNextRecord



# -----------------------------------------------------------------------------
# Create a condition tree for a given block
# -----------------------------------------------------------------------------

def _generateConditional (block):
  """
  Create a condition tree based upon the values currently stored in the form.

  @param block: GFBlock instance to create a condition for
  @return: GCondition instance with the condition to use or an empty dictionary
      if no condition is needed.
  """

  # 'user input': [GCondition, pass value?]
  baseComparisons = { '>': ['gt', True],
                      '>=': ['ge', True],
                      '<': ['lt', True],
                      '<=': ['le', True],
                      '=': ['eq', True],
                      '!=': ['ne', True],
                      'like': ['like', True],
                      'empty': ['null', False],
                      'notempty': ['notnull', False],
                    }
  comparisonDelimeter = ":"
  
  condLike = {}
  condEq   = {}
  conditions = []
  # Get all the user-supplied parameters from the entry widgets
  for entry, val in block._queryValues.items ():
    if entry._bound and entry.isQueryable () and len (str (val)):
      
      # New : operator support
      match = False
      for comparison in baseComparisons.keys():
          
        if val[:2+len(comparison)].lower() == "%s%s%s" % \
          (comparisonDelimeter, comparison, comparisonDelimeter):
          value=val[2+len(comparison):]
          
          if baseComparisons[comparison][1]:
            conditions.append([ baseComparisons[comparison][0], 
                                ['field', entry.field], 
                                ['const', value] 
                              ])
          else:
            conditions.append([ baseComparisons[comparison][0], 
                                ['field', entry.field] 
                              ])
          match = True  
          break
      
      # Falls through to old behaviour if no : condition given or 
      # the : condition is unknown
      if not match:
        if entry.typecast == 'text':
          if block._convertAsterisksToPercent:
            try:
              val = str (val).replace ('*', '%')
            except ValueError:
              pass
  
          if (val.find ('%') >= 0 or val.find ('_') >= 0):
            condLike [entry.field] = val
          else:
            condEq [entry.field] = val
        else:
          condEq [entry.field] = val

  epf = [['eq', ['field', f], ['const', v]] for (f, v) in condEq.items ()]
  lpf = [['like', ['field', f], ['const', v]] for (f, v) in condLike.items ()]

  if epf or lpf or conditions:
    result = GConditions.buildConditionFromPrefix (['and'] + epf + 
                                                   lpf + conditions)

  else:
    result = {}

  return result


class _BlockIter:
  """A simple resultset iterator that lets you use ResultSets as:

    for record in myResultSet:
      blah
  """
  def __init__(self, block):
    self.block = block
    self.new = True
    self.done = False

  def __iter__(self):
    return self

  def next(self):
    if self.done:
      raise StopIteration
    elif self.new:
      self.block.jumpRecord(0)
      self.new = False
    elif self.block._resultSet.isLastRecord():
      self.done = True
      raise StopIteration
    else:
      self.block.nextRecord()

    if self.block.isEmpty() and self.block._resultSet.isLastRecord():
      self.done = True
      raise StopIteration

    return self.block

