#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# SplitDock.py
#
# DESCRIPTION:
# Provides "docks" for the main application window
# Handles resizing, etc.
#
# NOTES:
#

__all__ = ['SplitDock']

from wxPython.wx import *
from gnue.common.apps import RuntimeSettings

class SplitDock(wxPanel):
  def __init__(self, parent, docker, name):
    wxPanel.__init__(self, parent, -1)
    self.parent = parent
    self.name = name
    self.docker = docker
    self.panels = []
    self.child = None
    self.sashSize = 0
    self.sizer = MySizer()
    self.SetAutoLayout(1)
    self.SetSizer(self.sizer)
    self.__initialize = 1

  #
  # Rebuild our dock
  #
  def refresh(self):

    # Get the panels in the order the user wants
    panels = self.panels[:]
    self.panels = []
    panels.sort()

    # Remove old sizers
    while self.sizer.Remove(0):
      pass

    #
    # Now, reparent all the tools
    index = 0
    for foo, panel in panels:
      # Reparent...
      panel.Reparent(self)

      # Make sashes visible
      if index:
        panel.SetSashVisible(wxSASH_TOP,1)

      self.sizer.Add(panel, 1, wxEXPAND)
      panel.tool._dock_index = index
      self.panels.append((index, panel))
      panel.tool._dock_location = self.name


      index += 1

    # Restore settings from last exit
    # if this is first time refresh
    # has been called since startup.
    if self.__initialize:
      self.__initialize = 0
      baseName = '%s-Sash-' % self.name
      index = 0
      for foo, panel in panels:
        panel.SetSize((10, RuntimeSettings.getint(self.docker.runtime_section, baseName + str(index), 100)))
        index += 1


    # Redraw screen
    self.Layout()

##    self.sashSize = self.GetClientSizeTuple()[1] - \
##                    self.panels[0].GetWindow1().GetClientSizeTuple()[1] - \
##                    self.panels[0].GetWindow2().GetClientSizeTuple()[1]


##    self.titlebarSize = self.panels[0][1].minimumVerticalSize -2
##    self.minimumPaneSize = self.titlebarSize + self.sashSize

##    for splitter in self.panels:
##      splitter.SetMinimumPaneSize(self.minimumPaneSize)


  #
  # Add a tool panel to our control
  #
  def add(self, panel, indx = '0'):

    # We do it as (index, panel) so we can
    # later do a self.panels.sort().

    # the sort() routine will sort by
    # the first list element.
    self.panels.append((indx, panel))

    # Save state information on the tool panel
    panel.tool._dock_index = indx


  #
  # Save sizes.. used by RuntimeSettings
  #
  def saveSettings(self):
    baseName = '%s-Sash-' % self.name
    rs = {}
    index = 0
    for foo, panel in self.panels:
      rs[baseName + str(index)] = panel.GetSizeTuple()[1]
      index += 1
    return rs


  #
  # Remove a tool panel from our control
  #
  def remove(self, panel):
    self.panels.remove((panel.tool._dock_index, panel))
    self.refresh()

  #
  # Redock a tool panel
  #
  def redock(self, panel, relation):
    pass

  #
  # Minimize a tool, so only title bar is showing
  #
  def minimize(self, panel):
    numPanels = len(self.panels)
    # No splitters in use?  Nothing to do.
    # Already minimized? Ditto.
    if numPanels < 2 or not panel.tool._visible:
      return

    # One splitter?  Treat as special case...
    # Only one of the tools can be minimized.
    if numPanels == 2:
      other = self.panels[0][1] != panel and self.panels[0][1] or self.panels[1][1]
      other.tool._visible = 1
##      self.docker.instance.dispatchEvent
    else:

      # Find out how many are already minimized.
      # One must remain visible at all times.
      numMinimized = 0
      for foo, p in self.panels:
        if not p.tool._visible:
          numMinimized += 1
      if numMinimized == numPanels - 1:
        return


    # Save the current size so
    # tool can be restored later
    panel.lastSize = panel.GetSizeTuple()[1]
    panel.tool._visible = 0
    self.Layout()


  #
  # Restore a minimized tool
  #
  def restore(self, panel):
    # No splitters in use?  Nothing to do.
    # Already visible? Ditto.
    if len(self.panels) < 2 or panel.tool._visible:
      return

    # Restore to the last known size
    panel.tool._visible = 1
    self.sizer.RecalcSizes(panel, panel.lastSize)



  #
  # Resize a tool
  #
  def resize(self, panel, size):

    # This is used several times...
    numPanels = len(self.panels)

    # Not enought panels?  Nothing to do.
    if numPanels < 2:
      return

    above = None
    maxResize = panel.GetSizeTuple()[1]
    for foo, p in self.panels:
      if p == panel:
        break
      elif p.tool._visible:
        above = p
      maxResize += p.GetSizeTuple()[1] - p.minimumVerticalSize

    # Nothing to resize above us?
    if not above:
      return

    # Don't resize
    size = min(maxResize, size)

    self.sizer.RecalcSizes(panel, size, above)


  #
  # Change the "minimized" state of a tool.
  # i.e., if it is minimized, restore it;
  # if it is not minimized, minimize it.
  def toggleMinimized(self, panel):
    if not panel.tool._visible:
      self.restore(panel)
    else:
      self.minimize(panel)

  def finalize(self):
    pass

######################################################################
#
# Custom wxSizer class to handle our tool docks
#
class MySizer(wxPySizer):
  def __init__(self):
    wxPySizer.__init__(self)
    self.__objects = []
    self.__childHideSize = 10
    self.__oldw = 0
    self.__oldh = 0

  def Add(self, window, option=0, flag=0, border=0, userData = None):
    self.__objects.append(window)
    try:
      self.__childHideSize = window.minimumVerticalSize + 1
    except AttributeError:
      pass

  def CalcMin(self):
    return wxSize(50, self.__childHideSize * len(self.__objects))

  def RecalcSizes(self, absolute=None, absheight=0, absrelative=None):
    size = self.GetSize()
    w = size.x
    h = size.y

    allocated = 0
    sizes = {}
    variable = []
    for child in self.__objects:
      if not child.tool._visible:
        sizes[child] = self.__childHideSize
        allocated += self.__childHideSize
      elif child == absolute:
        ch =  child.GetSizeTuple()[1]
        sizes[child] = absheight
        allocated += absheight
        if absrelative:
          adj = absrelative.GetSizeTuple()[1] - (absheight - ch)
          sizes[absrelative] = adj
          allocated += adj
      elif not absrelative:
        variable.append(child)
      else:
        sizes[child] = child.GetSizeTuple()[1]


    if variable:
      each = int((h - allocated)/len(variable))
      first = h - allocated - each * (len(variable) - 1)

      for child in variable:
        if first:
          sizes[child] = first
          first = 0
        else:
          sizes[child] = each

    y = 0
    for child in self.__objects:
      h = sizes[child]
      child.SetSize((w, h))
      child.SetPosition((0, y))
      y += h

  def Remove(self, thingy):
    try:
      del self.__objects[thingy]
      return 1
    except:
      try:
        self.__objects.remove(thingy)
        return 1
      except:
        return 0

