#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# Simple.py
#
# DESCRIPTION:
# Implements a basic form template
#
# NOTES:
# While functional, the primary purpose of this wizard is
# as a "learning-by-example" tool.


from gnue.designer.forms.TemplateSupport import *
import string

#
# Steps
#
# 0. Get Title             Goto step 1
# 1. Select Connection     Goto step 2
# 2. Select Table          Goto step 3
# 3. Select Fields         Goto step 4 or 7
# 4. Select Master         Goto step 5
# 5. Select Master Keys    Goto step 6
# 6. Select Detail Keys    Goto step 7
# 7. Add another?          Goto step 1,2, or 8
# 8. Complete.
#


class FormBuilderTemplate(FormTemplate):


  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'
  TYPEMAP    = {'date'    : 'date',
                'time'    : 'date',
                'datetime': 'date',
                'boolean' : 'boolean',
                'number'  : 'number',
                'string'  : 'text'}



  ###############
  #
  # Initialize any runtime variables
  #
  def Start(self, form, current):
    self.form = form
    self.current = current

    # The first table can never have a master
    self.variables['hasmaster0'] = '0'
    self.variables['newPage0'] = '0'


  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, stepCode):

    try:
      step, iteration = string.split(stepCode,':')
      iteration = int(iteration)
    except ValueError:
      step = stepCode

    # Add another? loop logic
    if step == '1or8':

      self.variables['newPage%s' % iteration] = self.variables['newPage']
      # If continue is set to 1, then go to step 1; otherwise, step 8
      if self.variables['continue'] == '1':

        step = '1'

        # If we are reusing the same connection, record that now and skip a step
        if self.variables['singleconnection'] == '1':
          self.variables['connection%s' % iteration] = self.variables['connection0']
          step = '2'


      else:
        # We are done... record the total number of children
        self.variables['iterations'] = iteration
        step = '8'


    # Has Master? loop logic
    if step == '4or7':
      if self.variables['hasmaster%s' % iteration] == '1':
        step = '4'
      else:
        step = '7'


    #
    # Step #0 / Get Title, et al
    #
    if step == '0':

      return   { 'title': 'Basic Form Information',
                 'content': (WizardText('Welcome to the sample form wizard.'),
                             WizardText('To create your form, I need to know some basic information. \n\n'
                                        'First, what shall I call your form? This name will appear in '
                                        'the title bar.'),
                             WizardInput('title', label='Form Title:', required=1,
                                         size=40),
                            ),
                 'prev': None,
                 'next': '1:0' }

    #
    # Step #1 / Get Connection
    #
    if step == '1':

      if iteration == 0:
        # first time through ask about reusing same connection
        return   { 'title': 'Connection Information',
                   'content':  (WizardText('What connection contains the table you wish to the form?'),
                                WizardInput('connection%s'%iteration,label='Connection:', required=1,
                                            set=self.GetAvailableConnections()),
                                WizardInput('singleconnection',
                                         label='Reuse this connection for all tables?',
                                         required=1,set=(('1','Yes'),
                                                         ('0','No')
                                                         )),
                                WizardText('You may be asked to login to this connection.'),
                              ),
                   'prev': '0', # Loop back
                   'next': '2:%s' % iteration }

      else:

        return   { 'title': 'Connection Information',
                   'content':  (WizardText('What connection contains the table you wish to the form?'),
                                WizardInput('connection%s'%iteration,label='Connection:', required=1,
                                            set=self.GetAvailableConnections()),
                                WizardText('You may be asked to login to this connection.'),
                              ),
                   'prev': '7:%s' % (iteration - 1), # Loop back
                   'next': '2:%s' % iteration }


    #
    # Step #2 / Get Table
    #
    elif step == '2':

      # If using the same connection for all tables, skip question on Prev
      if iteration and self.variables['singleconnection'] == '1':
        prev = '1'
      elif iteration:
        prev = '7'
      else:
        prev = '0'

      # Do not ask about master/detail if we're iteration #1
      if iteration:
        return   { 'title': 'Select Table/Source',
                   'content': (WizardText('Now, please select a table for your form.'),
                               WizardInput('table%s' % iteration,
                                           label='Table:', required=1, lines=5,
                                           set=self.GetAvailableSources(self.variables['connection%s'%iteration])),
                               WizardInput('hasmaster%s' % iteration,
                                           label='Will this table have a master?',
                                           required=1,set=(('0','No'),
                                                           ('1','Yes'),
                                                           )),
                                            ),
                   'prev': '%s:%s' % (prev,iteration),
                   'next': '3:%s' % iteration }

      else:
        return   { 'title': 'Select Table/Source',
                   'content': (WizardText('Now, please select a table for your form.'),
                               WizardInput('table%s' % iteration,
                                           label='Table:', required=1, lines=5,
                                           set=self.GetAvailableSources(self.variables['connection%s'%iteration])),
                                            ),
                   'prev': '%s:%s' % (prev,iteration),
                   'next': '3:%s' % iteration }


    #
    # Step #3 / Select fields to include
    #
    elif step == '3':


      return   { 'title': 'Select Fields to Include',
                 'content': (WizardText('Which fields shall I include in your form as entries?'),
                             WizardInput('tablefields%s' % iteration,
                                         label='Columns:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection%s'%iteration],
                                                self.variables['table%s' % iteration])),
                             WizardInput('tablearrangement%s' % iteration,
                                         label='Arrangement Method:',
                                         required=1,
                                         set=(('left','Single Record with Labels to Left'),
                                              ('above','Single Record with Labels Above'),
                                              ('grid','Grid Format')))),
                 'prev': '2:%s' % iteration,
                 'next': '4or7:%s' % iteration }


    #
    # Step #4 / Select Master Table
    #
    elif step == '4':

      set = []
      tables = {}
      for i in range(iteration):

        table = self.variables['table%s' % i]

        # Append a "#2" to the table name if multiple sources use same table
        if tables.has_key(table):
          tables[table] += 1
          table += ' #%s' % tables[table]
        else:
          tables[table] = 1

        set.append( (i, table ) )



      # NOTE: mastertable# will be set to a NUMBER corresponding to the datasource to
      #       use. It is NOT set to an actual name!  (i.e., the table to use is
      #       self.variables['table%s' % mastertable#]

      return   { 'title': 'Link to Master Table',
                 'content': (WizardText('Now, please select the table to be the master of %s.' % self.variables['table%s'%iteration]),
                             WizardInput('mastertable%s'%iteration, label='Master Table:', required=1, lines=5,
                                         set=set), ),
                 'prev': '3:%s' % iteration,
                 'next': '5:%s' % iteration }


    #
    # Step #6 / Get Master Columns (primary keys)
    #
    elif step == '5':


      return   { 'title': 'Master key fields',
                 'content': (WizardText('Which master-level fields are the primary keys?'),
                             WizardInput('masterkeys%s' % iteration, label='Columns:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection%s' % self.variables['mastertable%s' % iteration]],
                                                self.variables['table%s' % self.variables['mastertable%s' % iteration]]))),
                 'prev': '4:%s' % iteration,
                 'next': '6:%s' % iteration }


    #
    # Step #6 / Get Detail Columns (detail keys)
    #
    elif step == '6':

      # TODO: If the table changed between steps,
      # TODO: variables['fields'] should be cleared.

      return   { 'title': 'Detail key fields',
                 'content': (WizardText('Which detail-level fields are the foreign keys?'),
                             WizardInput('detailkeys%s' % iteration, label='Columns:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection%s' % iteration],
                                                self.variables['table%s' % iteration]))),
                 'prev': '5:%s' % iteration,
                 'next': '7:%s' % iteration }

    #
    #
    # Step #7 / Get Master Columns to Include
    #
    elif step == '7':

      # If current table has no master, then skip the master/detail stuff
      if self.variables['hasmaster%s' % iteration] == '0':
        prev = '3'
      else:
        prev = '6'


      # TODO: If the table changed between steps,
      # TODO: variables['fields'] should be cleared.

      return   { 'title': 'Add Another Table?',
                 'content': (WizardInput('continue', label='Add Another Table?',
                                         required=1,set=(('1','Yes'),
                                                         ('0','No')
                                                         )),
#                             WizardInput('newPage%s' % (iteration+1),
                             WizardInput('newPage',
                                         label='If yes, put this table on a new page?',
                                         required=1,set=(('0','No'),
                                                         ('1','Yes')
                                                         )),),
                 'prev': '%s:%s' % (prev,iteration),
                 'next': '1or8:%s' % (iteration+1) }

    elif step == '8':
      return   { 'title': "That's it!",
                 'content': (WizardText('Click finish to create your form?'),),
                 'prev': '7:%s' % (iteration-1),
                 'next': None }



  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # The Simple wizard uses basic "required"
    # settings in the page markup, so doesn't
    # need any special validation.
    return None



  ###############
  #
  # We have all the data, so generate our form. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):
    self.instance.wizardName = TemplateInformation['Name']

    logic = self.current['logic']
    layout = self.current['layout']
    page = self.current['page']

    formwidth = 0
    formheight = 0
    basey = 0
    height = 0
    dataSourceArray = {}
    pageCounter = 1


    for count in range(self.variables['iterations']):
      if not page or self.variables['newPage%s' % count] == '1':
        pageCounter += 1
        page = self.AddElement('page', layout,
                               {  'name': 'pg%s' % pageCounter })
        basey=0
        height=0

      tableKey = string.upper(self.variables['table%s' % count])
      
      attrs = { 'connection': self.variables['connection%s' % count],
                'table': self.variables['table%s' % count],
                'type': 'object',
                'name': 'dts%s%s' %  (tableKey,count) }

      if self.variables['hasmaster%s' % count] == '1':
        attrs['master'] = dataSourceArray[self.variables['mastertable%s' % count]].name
        attrs['masterlink'] = string.join(self.variables['masterkeys%s' % count],',')
        attrs['detaillink'] = string.join(self.variables['detailkeys%s' % count],',')
                      
      datasource = self.AddElement('datasource', self.form, attrs)
      dataSourceArray[count] = datasource


      # Is this a grid style form?
      multirecord = self.variables['tablearrangement%s' % count] == 'grid'

      # Will labels appear above or to the left of the entries?
      # Note: this results in false for both "grid" and "above"
      leftlabels = self.variables['tablearrangement%s' % count] == 'left'


      # We will need a block to hold our entries...

      # The basic attributes for a block
      attrs = {'name' : 'blk%s' % tableKey,
               'datasource': datasource['name']}

      # If this is a multirecord form, set rows at the block level.
      if multirecord:
        attrs['rows'] = 10

      # Create the block
      block = self.AddElement('block', logic, attrs)

      schema = self.GetSourceSchema(self.variables['connection%s' % count],
                           self.variables['table%s' % count])


      # Create the entries and labels...


      # Make a map of all the field schemas we will need
      # We will not actually create the entries at this point
      # because we want to keep in the order that the user
      # specified
      fields = {}
      for field in schema.findChildrenOfType ('GSField', False, True):
        if field.name in self.variables['tablefields%s' % count]:
          fields[field.name] = field

      entryQueue = []
      labelQueue = []
      largestField = 0
      largestLabel = 0

      # First, let's make all the fields
      for name in self.variables['tablefields%s' % count]:

        field = fields[name]

        # We will use the field name as the basis for all our
        # entry and label names.  Capitalize the first letter
        # and strip all spaces...
        if len(field.name) == 1:
          fieldKey = string.upper(field.name)
        else:
          fieldKey = string.join(string.split(string.capwords( \
                       string.replace(field.name,'_',' '))),'')

        # Create a label. If the schema specified a label for a field,
        # use that as is.  If not, use the field name, replacing any '_'
        # with spaces and tacking on a colon.
        text = hasattr(field,'label') and label or \
               string.capwords(string.replace(field.name,'_',' ')) + ':'

        # Add text, x, y to labelQueue
        # The x,y will be replaced by logic later on...
        labelQueue.append([text,0,0])

        # Keep track of the greatest label width
        largestLabel = max(len(text),largestLabel)

        # Create an entry for this field.

        attrs={ 'name': "fld%s" % fieldKey,
                'field': field.name,
                'typecast': self.TYPEMAP.get (field.type, 'text')}

        # If we have a length for the field, use this as the maxLength
        # for the entry. Also, adjust the display width if necessary.
        if hasattr(field,'length'):
          ln = max(min(field.length, 40),0) or 10 
          attrs['maxLength'] = ln
          largestField = max(largestField, ln)
        else:
          largestField = max(largestField, 10)

        # Create the entry element
        fld = self.AddElement('field', block, attrs)

        # and queue it so we can create entry's later
        entryQueue.append([fld, 0, 0] )


      #
      # Rearrange the fields and labels to snugly fit the screen
      # based on the layout arrangement selected by the user.
      #

      # Grid/multirecord layout
      if multirecord:

        height += 13
        width = 1
        x = 1

        for i in range(len(entryQueue)):
          field = entryQueue[i][0]
          text = labelQueue[i][0]
          textLen = len(text)

          labelQueue[i][1] = x
          labelQueue[i][2] = basey + 1
          entryQueue[i][1] = x
          entryQueue[i][2] = basey + 2

          # If label width is larger than entry width, center the entry
          try:
            entryWidth = min(field['maxLength'],40)
          except KeyError:
            entryWidth = 10
          if entryWidth < textLen:
            entryQueue[i][1] += int((textLen - entryWidth)/2)

          # Calculate the starting x for the next label/entry
          dx = max(entryWidth, textLen) + 1

          # Increase the form width accordingly
          width += dx
          x += dx
        basey += 12

      # Single Record layout (regardless of label location)
      else:

        # Assign the starting (x,y) values for labels (l) and fields (f)
        # as well as the delta (x,y) values.
        if leftlabels:
          # Labels to the left of the entries
          lx, ly, ldx, ldy = (1, 1, 0, 1)
          fx, fy, fdx, fdy = (largestLabel + 2, 1, 0, 1)

          # Set form width
          width = largestField + largestLabel + 3

        else:
          # Labels above the entries
          lx, ly, ldx, ldy = (1, 1, 0, 3)
          fx, fy, fdx, fdy = (1, 2, 0, 3)

          # Set the form width
          width = 2 + max(largestField, largestLabel)

        # Rearrange
        height += fy+2-fdy
        by = basey
        for i in range(len(entryQueue)):
          labelQueue[i][1] = lx
          labelQueue[i][2] = by + ly
          entryQueue[i][1] = fx
          entryQueue[i][2] = by + fy

          ly = ly + ldy
          lx = lx + ldx
          fy = fy + fdy
          fx = fx + fdx
          height += fdy
          basey += fdy

      #
      # Finally, add the visual elements...
      #
      for i in range(len(entryQueue)):

        #
        # First, the label
        #
        text, x, y = labelQueue[i]
        self.AddElement( 'label', page,
                    {'Char:x': x,
                     'Char:y': y,
                     'name': "lbl%s" % fieldKey,
                     'text': text,
                     'Char:width': len(text)})

        #
        # And the entry...
        #
        field, x, y = entryQueue[i]
        attrs={'name': "ent%s" % field['name'][3:],
               'field': field['name'],
               'block': block['name'],
               'Char:x': x,
               'Char:y': y,
               'Char:width': 10 }

        # If we have a length for the field, use this as the maxLength
        # for the entry. Also, adjust the display width if necessary.
        try:
          attrs['Char:width'] = min(field['maxLength'], 40)
        except KeyError:
          pass

        # Create the entry element
        self.AddElement('entry', page, attrs)


      formwidth = max(formwidth, width)
      formheight = max(formheight, height)


    #
    # Set the basic attributes of the form
    #
    self.form['title'] = self.variables['title']
    layout['Char:width'] = formwidth
    layout['Char:height'] = formheight

    return 1



############
#
# Basic information about this template
#
TemplateInformation = {
    'Product': 'forms',
    'BaseID' : 'FormBuilder',
    'BaseClass' : FormBuilderTemplate,
    'Name' : 'Multi-Block, Page form building template',
    'Description' : 'Creates forms containing arbitrary number of pages/blocks',
    'Version' : '0.0.1',
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD
}

