/***************************************************************************
                          envt.hpp  -  the environment for each GDL module
                             -------------------
    begin                : July 22 2002
    copyright            : (C) 2002 by Marc Schellens
    email                : m_schellens@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ENVT_HPP_
#define ENVT_HPP_

#include <vector>
#include <cstdlib>

#include "typedefs.hpp"
#include "str.hpp"
#include "dpro.hpp"
#include "datatypes.hpp"
#include "dstructgdl.hpp"
#include "datalistt.hpp"
#include "extrat.hpp"

//#define GDL_DEBUG
#undef GDL_DEBUG

class DInterpreter;

namespace lib {
  BaseGDL* obj_new( EnvT* e);
  void obj_destroy( EnvT* e);
}

class EnvBaseT
{
protected:
  static DInterpreter* interpreter;
  DataListT            env;
  SizeT                parIx;     // ix of next parameter to put
  DSub*                pro;
  ProgNodeP            callingNode;
  bool                 obj;       // member subroutine?
  ExtraT*              extra;

  // finds the local variable pp points to
  int FindLocalKW( BaseGDL** pp) { return env.FindLocal( pp);}
  // used by the interperter returns the keyword index, used for UD functions
  int GetKeywordIx( const std::string& k);

  // for HEAP_GC
  void AddStruct( DPtrListT& ptrAccessible,  DPtrListT& objAccessible, 
		  DStructGDL* stru);
  void AddPtr( DPtrListT& ptrAccessible, DPtrListT& objAccessible, 
	       DPtrGDL* ptr);
  void AddObj( DPtrListT& ptrAccessible, DPtrListT& objAccessible, 
	       DObjGDL* obj);
  void Add( DPtrListT& ptrAccessible, DPtrListT& objAccessible, 
	    BaseGDL* p);

public:
  void AddEnv( DPtrListT& ptrAccessible, DPtrListT& objAccessible);

  virtual ~EnvBaseT() { delete extra;}

  EnvBaseT( ProgNodeP cN, DSub* pro_);

  // finds the global variable pp (used by arg_present function)
  int FindGlobalKW( BaseGDL** pp) { return env.FindGlobal( pp);}

  // checks if pp points to a local variable
  bool IsLocalKW( BaseGDL** pp) const { return env.InLoc( pp);}

  void RemoveLoc( BaseGDL* p) { env.RemoveLoc( p);}

  // called after parameter definition
  void Extra();
  friend class ExtraT;

  // used by compiler and from EnvT (for variable number of paramters)
  SizeT AddEnv()
  {
    SizeT s = env.size();
    env.resize(s+1);
    return s;
  }

  // the upper (calling) environment
  // a EnvT must have always a EnvUDT caller
  // i.e. library functions never call each other
  // with a new EnvT environment
  EnvBaseT* Caller();

  // returns environment data, by value (but that by C++ reference)
  BaseGDL*& GetKW(SizeT ix) { return env[ix];}

  // used by HELP
  SizeT EnvSize() { return env.size();}

  // next four are used by interpreter
  void SetNextPar( BaseGDL* const nextP); // by value (reset loc)
  void SetNextPar( BaseGDL** const nextP); // by reference (reset env)
  void SetKeyword( const std::string& k, BaseGDL* const val);  // value
  void SetKeyword( const std::string& k, BaseGDL** const val); // reference

  // to check if a lib function returned a variable of this env
  bool Contains( BaseGDL* p) const;

  BaseGDL** GetPtrTo( BaseGDL* p);

  DInterpreter* Interpreter() const { return interpreter;}

  bool  IsObject() const { return obj;}
  DSub* GetPro()   const { return pro;}

  ProgNodeP CallingNode() { return callingNode;}

  SizeT NParam( SizeT minPar = 0); //, const std::string& subName = "");

  const std::string GetProName() const
  {
    if( pro == NULL) return "";
    return pro->ObjectName();
  }
  
  // get the name of 'i'th parameter
  const std::string GetParString( SizeT i)
  {
    SizeT ix= i + pro->key.size();
    return GetString( ix);
  }

  // get the name of 'ix'th environment entry (asks 'pro')
  const std::string GetString( SizeT ix);
  
  // get name of 'p'
  const std::string GetString( BaseGDL*& p);

  virtual const std::string GetFilename() const=0;

  // converts parameter 'ix' if necessary and sets 'scalar' 
  void AssureLongScalarPar( SizeT ix, DLong& scalar);
  // get i'th parameter
  // throws if not defined (ie. never returns NULL)
  BaseGDL*& GetParDefined(SizeT i); //, const std::string& subName = "");
  bool KeywordPresent( SizeT ix)
  { return (env.Loc(ix)!=NULL)||(env.Env(ix)!=NULL);}
  bool GlobalKW( SizeT ix) 
  {
    if( ix >= env.size()) return false;
    return ( env.Env( ix) != NULL);
  }
};

// for UD subroutines (written in GDL) ********************************
class EnvUDT: public EnvBaseT
{
  ProgNodeP         ioError; 
  DLong             onError; // on_error setting
  BaseGDL**         catchVar;
  ProgNodeP         catchNode; 
  bool              lFun; // assignment paramter for functions as l_value
  SizeT             nJump; // number of jumps in current environment
  int               lastJump; // to which label last jump went
  
public:
  // UD pro/fun
  EnvUDT( ProgNodeP idN, DSub* pro_, bool lF = false);

  // member procedure
  EnvUDT( ProgNodeP idN, BaseGDL* self, 
	const std::string& parent="");
  // member function
  EnvUDT( BaseGDL* self, ProgNodeP idN, 
	const std::string& parent="", bool lF = false);


  // for obj_new and obj_destroy
  EnvUDT( EnvT* pEnv, DSub* newPro, BaseGDL** self); 

  DLong GetOnError() { return onError;}

  SizeT NJump() { return nJump;}
  int    LastJump() { return lastJump;}
  ProgNodeP GotoTarget( int ix)
  { 
    lastJump = ix; //static_cast<DSubUD*>( pro)->LabelOrd( ix);
    ++nJump;
    return static_cast<DSubUD*>( pro)->GotoTarget( ix);
  }
  bool LFun() const { return lFun;} // left-function

  void SetIOError( int targetIx) 
  { // this isn't a jump
    if( targetIx != -1)
      ioError = static_cast<DSubUD*>( pro)->GotoTarget( targetIx)->
	getNextSibling(); 
    else
      ioError = NULL;
  }
  ProgNodeP GetIOError() { return ioError;}

  const std::string GetFilename() const
  {
    DSubUD* subUD=static_cast<DSubUD*>( pro);
    return subUD->GetFilename();
  }

  friend class DInterpreter;
  friend class EnvT;
};

// for library subroutines **************************************
// this contains the library function API ***********************
class EnvT: public EnvBaseT
{
  // Please use non library API (see below) function with caution
  // (most of them can be ignored by library function authors)

protected:
  // stores all data which has to deleted upon destruction
  std::vector<BaseGDL*>  toDestroy;

public:
  ~EnvT()
  {
    for( std::vector<BaseGDL*>::iterator i=toDestroy.begin();
	 i != toDestroy.end(); ++i) 
      delete *i;
  }

  EnvT( ProgNodeP cN, DSub* pro_);

  // for obj_new and obj_destroy
  EnvT( EnvT* pEnv, DSub* newPro, BaseGDL** self); 

  void HeapGC( bool doPtr, bool doObj, bool verbose);
  void ObjCleanup( DObj actID);


  SizeT NewObjHeap( SizeT n=1, DStructGDL* v=NULL);
  SizeT NewHeap( SizeT n=1, BaseGDL* v=NULL);
  void FreeObjHeap( DObj id);
  void FreeHeap( DPtr id);
  void FreeHeap( DPtrGDL* p);
  DStructGDL* GetObjHeap( DObj ID);
  BaseGDL* GetHeap( DPtr ID);

  // used by obj_new (basic_fun.cpp)
  void PushNewEnv(  DSub* newPro, SizeT skipP, BaseGDL** newObj=NULL);
  void PushNewEnvUD(  DSub* newPro, SizeT skipP, BaseGDL** newObj=NULL);
  // for exclusive use by lib::on_error
  void OnError();
  // for exclusive use by lib::catch_pro
  void Catch();

  const std::string GetFilename() const
  {
    static const std::string internal("<INTERNAL_LIBRARY>");
    return internal;
  }

  // *************************
  // API for library functions
  // *************************

  // raise an exception from within a library function
  // automatically cares for adding line/column info and the
  // function name. 's' should be set to the 'raw' error message
  // saves some typing :-)
  void Throw( const std::string& s)
  { throw GDLException( CallingNode(), pro->ObjectName()+": "+s);}

  // 'guards' a newly created variable which should be deleted
  // upon library routines exit (normal or on error)
  // elimates the need of auto_ptr
  void Guard( BaseGDL* toGuard)
  { toDestroy.push_back( toGuard);}

  // returns environment data, by value (but that by C++ reference)
  BaseGDL*& GetKW(SizeT ix) { return env[ix];}

  // returns the ix'th parameter (NULL if not defined)
  BaseGDL*& GetPar(SizeT i);

  // get i'th parameter
  // throws if not defined (ie. never returns NULL)
  BaseGDL*& GetParDefined(SizeT i); //, const std::string& subName = "");

  // throw for STRING, STRUCT, PTR and OBJECT
  BaseGDL*& GetNumericParDefined( SizeT ix)
  {
    BaseGDL*& p0 = GetParDefined( ix);
    if( p0->Type() == STRING)
      Throw( "String expression not allowed in this context: "+GetParString(0));
    if( p0->Type() == STRUCT)
      Throw( "Struct expression not allowed in this context: "+GetParString(0));
    if( p0->Type() == PTR)
      Throw( "Pointer expression not allowed in this context: "+GetParString(0));
    if( p0->Type() == OBJECT)
      Throw( "Object reference not allowed in this context: "+GetParString(0));
    return p0;
  }

  // get i'th parameter
  // throws if not global (might be NULL), for assigning a new variable to
  // (write only)
  BaseGDL*& GetParGlobal(SizeT i); 

  // get the pIx'th paramter and converts it to T if necessary
  // implies that the parameter must be defined
  // if it converts it cares for the destruction of the copy
  // CAUTION: this is for *read only* data, as the returned data might
  // be a copy or not
  template <typename T> 
  T* GetParAs( SizeT pIx)
  {
    BaseGDL* p = GetParDefined( pIx);
    T* res = dynamic_cast<T*>( p);
    if( res != NULL) return res;
    res = static_cast<T*>( p->Convert2( T::t, BaseGDL::COPY));
    toDestroy.push_back( res);
    return res;
  }
  // same as before for keywords
  template <typename T> 
  T* GetKWAs( SizeT ix)
  {
    BaseGDL* p = GetKW( ix);
    if( p == NULL)
      Throw( "Keyword is undefined: "+GetString( ix));
    T* res = dynamic_cast<T*>( p);
    if( res != NULL) return res;
    res = static_cast<T*>( p->Convert2( T::t, BaseGDL::COPY));
    toDestroy.push_back( res);
    return res;
  }

  // next two same as last two, but return NULL if parameter/keyword is not defined
  template <typename T> 
  T* IfDefGetParAs( SizeT pIx)
  {
    BaseGDL* p = GetPar( pIx);
    if( p == NULL) return NULL;
    T* res = dynamic_cast<T*>( p);
    if( res != NULL) return res;
    res = static_cast<T*>( p->Convert2( T::t, BaseGDL::COPY));
    toDestroy.push_back( res);
    return res;
  }
  // same as before for keywords
  template <typename T> 
  T* IfDefGetKWAs( SizeT ix)
  {
    BaseGDL* p = GetKW( ix);
    if( p == NULL) return NULL;
    T* res = dynamic_cast<T*>( p);
    if( res != NULL) return res;
    res = static_cast<T*>( p->Convert2( T::t, BaseGDL::COPY));
    toDestroy.push_back( res);
    return res;
  }

  // returns the struct of a valid object reference or throws
  DStructGDL* GetObjectPar( SizeT pIx);

  // returns the actual number of paramters passed to a library function
  // minPar is the minimal number of paramters the function needs
  // (if less it throws), subName is used for error reporting
  SizeT NParam( SizeT minPar = 0); //, const std::string& subName = "");

  // for library functions (keyword must be exact)
  // returns the index of keyword k
  int KeywordIx( const std::string& k);

  // for use within library functions
  // consider to use (note: 'static' is the point here):
  // static int kwIx = env->KeywordIx( "KEYWORD");
  // bool kwSet = env->KeywordSet( kwIx);
  //
  // instead of:
  // bool kwSet = env->KeywordSet( "KEYWORD");
  //
  // this one adds some overhead, but is easy to use
  bool KeywordSet( const std::string& kw);
  // this one together with a static int holding the index is faster
  // (after the first call)
  bool KeywordSet( SizeT ix);

  bool KeywordPresent( SizeT ix)
  { return EnvBaseT::KeywordPresent( ix);}

  // local/global keyword/paramter
  bool LocalKW( SizeT ix) 
  {
    if( ix >= env.size()) return false;
    return ( env.Loc( ix) != NULL);
  }
  bool GlobalKW( SizeT ix) 
  {
    return EnvBaseT::GlobalKW( ix);
  }
  bool LocalPar( SizeT ix) { return LocalKW( ix + pro->key.size());}
  bool StealLocalPar( SizeT ix) 
  { 
    if( LocalKW( ix + pro->key.size()))
      {
	env.Clear( ix + pro->key.size());
	return true;
      }
    return false;
  }
  bool GlobalPar( SizeT ix) { return GlobalKW( ix + pro->key.size());}

  // next two to set keywords/paramters
  // note that the value MUST be created in the library function
  // with operator new
  // Before it must be tested with KeywordPresent() or NParam() if
  // the keyword/paramter is present 
  // this is not done automatically because its more effective, to 
  // create the data (newVal) only if its necessary
  // if the functions throw, they delete newVal before -> no
  // guarding of newVal is needed
  void SetKW( SizeT ix, BaseGDL* newVal);
  void SetPar( SizeT ix, BaseGDL* newVal);

  // Assure functions:
  // if name contains "Par" they must be used for paramters, else for keywords
  // (the error messages are defined for this usage and the indexing is 
  // done respectively)

  // next two: NO CONVERSION (throw if wrong type)
  // NOTE: only few functions need to be so restrictive
  // converts parameter to scalar, throws if parameter is of different type,
  // non-scalar or not defined
  template <typename T> 
  void AssureScalarPar( SizeT pIx, typename T::Ty& scalar)
  {
    BaseGDL* p = GetParDefined( pIx);
    T* tp= dynamic_cast<T*>(p);
    if( tp == NULL)
      Throw( "Variable must be a "+T::str+" in this context: "+
	     GetParString(pIx));
    if( !tp->Scalar( scalar))
      Throw("Variable must be a scalar in this context: "+
	    GetParString(pIx));
  }
  // same as before for keywords
  template <typename T> 
  void AssureScalarKW( SizeT ix, typename T::Ty& scalar)
  {
    BaseGDL* p = GetKW( ix);
    if( p == NULL)
      Throw("Keyword undefined: "+GetString(ix));
    T* tp= dynamic_cast<T*>(p);
    if( tp == NULL)
      Throw("Keyword must be a "+T::str+" in this context: "+
	    GetString(ix));
    if( !tp->Scalar( scalar))
      Throw("Keyword must be a scalar in this context: "+
	    GetString(ix));
  }

  void AssureGlobalPar( SizeT pIx);
  void AssureGlobalKW( SizeT ix);

  // if keyword 'kw' is not set, 'scalar' is left unchanged
  void AssureLongScalarKWIfPresent( const std::string& kw, DLong& scalar);
  void AssureLongScalarKWIfPresent( SizeT ix, DLong& scalar);
  // converts keyword 'kw' if necessary and sets 'scalar' 
  void AssureLongScalarKW( const std::string& kw, DLong& scalar);
  // converts ix'th keyword if necessary and sets 'scalar' 
  void AssureLongScalarKW( SizeT ix, DLong& scalar);
  // converts parameter 'ix' if necessary and sets 'scalar' 
  void AssureLongScalarPar( SizeT ix, DLong& scalar);

  // same as for Long
  void AssureDoubleScalarKWIfPresent( const std::string& kw, DDouble& scalar);
  void AssureDoubleScalarKWIfPresent( SizeT ix, DDouble& scalar);
  void AssureDoubleScalarKW( const std::string& kw, DDouble& scalar);
  void AssureDoubleScalarKW( SizeT ix, DDouble& scalar);
  void AssureDoubleScalarPar( SizeT ix, DDouble& scalar);

  // same as for Long
  void AssureFloatScalarKWIfPresent( const std::string& kw, DFloat& scalar);
  void AssureFloatScalarKWIfPresent( SizeT ix, DFloat& scalar);
  void AssureFloatScalarKW( const std::string& kw, DFloat& scalar);
  void AssureFloatScalarKW( SizeT ix, DFloat& scalar);
  void AssureFloatScalarPar( SizeT ix, DFloat& scalar);

  // same as for Long
  void AssureStringScalarKWIfPresent( const std::string& kw, DString& scalar);
  void AssureStringScalarKWIfPresent( SizeT ix, DString& scalar);
  void AssureStringScalarKW( const std::string& kw, DString& scalar);
  void AssureStringScalarKW( SizeT ix, DString& scalar);
  void AssureStringScalarPar( SizeT ix, DString& scalar);

  // to be extended on demand for other data types  
};


typedef std::deque<EnvBaseT*> EnvStackT;

#endif


