/*  Gnometab -- a guitar tablature editor for GNOME
    Copyright (C) 2001  William L. Guelker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "tab_canvas.h"
#include "tab_objects.h"
#include "rhythm_builder.h"

static void
dialog_destroyed (GtkObject *obj,  void **dialog_pointer)
{
    if (dialog_pointer != NULL)
    {
        g_free (*dialog_pointer);
        *dialog_pointer = NULL;
    }
}

static void
rhthm_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogRhythm *gtab_dialog)
{
    if (res_id == GTK_RESPONSE_OK) {
        create_tab_rhythm(gtab_dialog->tab_doc, gtab_dialog->rhythm, 
                                    gtab_dialog->x, gtab_dialog->y);
    }    
    gtk_widget_destroy(gtab_dialog->dialog);
}

static void 
rhythm_vert_line(GtabDialogRhythm *dialog, gdouble x, gdouble y)
{
    GnomeCanvasPoints   *points;
    GnomeCanvasItem     *item;
    
    x += 70.0;
    
    points = gnome_canvas_points_new(2);
    
    points->coords[0] = x;
    points->coords[1] = y;
    points->coords[2] = x;
    points->coords[3] = y - 180.0;
    
    item = gnome_canvas_item_new(dialog->objects,
                    gnome_canvas_line_get_type(),
                    "points", points,
                    "fill_color", "black",
                    "width_pixels", 2,
                    "first_arrowhead", FALSE,
                    "last_arrowhead", FALSE,
                    "smooth", TRUE,
                    NULL);
    
    gnome_canvas_points_unref(points);
    gnome_canvas_item_raise_to_top(item);
    
}

static void 
rhythm_head(GtabDialogRhythm *dialog, gdouble x, gboolean fill)
{
    GnomeCanvasItem *item;
    
    x += 70.0;
   
    if (fill) {
        item = gnome_canvas_item_new(dialog->objects,
                        gnome_canvas_ellipse_get_type(),
                        "x1", x,
                        "y1", 35.0,
                        "x2", x + 40.0,
                        "y2", 65.0,
                        "width_pixels", 2,
                        "fill_color", "black",
                        "outline_color", "black",
                        NULL);
    }
    else {
        item = gnome_canvas_item_new(dialog->objects,
                        gnome_canvas_ellipse_get_type(),
                        "x1", x,
                        "y1", 35.0,
                        "x2", x + 40.0,
                        "y2", 65.0,
                        "width_pixels", 2,
                        "fill_color", NULL,
                        "outline_color", "black",
                        NULL);
    }  
    gnome_canvas_item_raise_to_top(item);
}

static void
rhythm_horiz_line(GtabDialogRhythm *dialog, 
                    gdouble x, gdouble y, gboolean complete, 
                    gboolean flat, gint direction)
{
    GnomeCanvasPoints   *points;
    GnomeCanvasItem     *item;
    
    x += 70.0;
    
    points = gnome_canvas_points_new(2);
    
    points->coords[0] = x;
    points->coords[1] = y;
    
    switch (direction) {
        
        case -1:
            points->coords[2] = x - 40.0;
            points->coords[3] = y;
            break;
        
        default:
            
            if (complete) {
                points->coords[2] = x + 140.0;
                points->coords[3] = y;
            }
            else {
                
                if (flat) {
                    points->coords[2] = x + 40.0;
                    points->coords[3] = y;
                }
                else {
                    points->coords[2] = x + 40.0;
                    points->coords[3] = y - 40.0;
                }
            }
            break;
    }
    
    item = gnome_canvas_item_new(dialog->objects,
                    gnome_canvas_line_get_type(),
                    "points", points,
                    "fill_color", "black",
                    "width_pixels", 4,
                    "first_arrowhead", FALSE,
                    "last_arrowhead", FALSE,
                    "smooth", TRUE,
                    NULL);
    
    gnome_canvas_item_raise_to_top(item);
    
    gnome_canvas_points_unref(points);
}

static void 
rhythm_dot(GtabDialogRhythm *dialog, gdouble x, gdouble y)
{
    /* pass this function the coords of the base of the stem */
    
    GnomeCanvasItem *item;
    
    /* pad x 70 points to move notes to the middle of each pos */
    
    x += 80.0;
    
    y -= 20.0;
    
    item = gnome_canvas_item_new(dialog->objects,
                    gnome_canvas_ellipse_get_type(),
                    "x1", x,
                    "y1", y,
                    "x2", x + 5.0,
                    "y2", y - 5.0,
                    "fill_color", "black",
                    NULL);
    
    gnome_canvas_item_raise_to_top(item);
    
}

static void
render_rhythm(GtabDialogRhythm *dialog)
{    
    gdouble dot_y;
    gboolean flat = FALSE;
    
    if (GNOME_IS_CANVAS_ITEM(dialog->objects)) {
        /* clear the current rhythm */
        
        gtk_object_destroy(GTK_OBJECT(dialog->objects));
        dialog->objects = GNOME_CANVAS_GROUP(
                            gnome_canvas_item_new(dialog->root_group,
                                gnome_canvas_group_get_type(),
                                NULL));
    }
    
    /* this is undoubtedly the ugliest code ever written */
    
    if (dialog->rhythm.pos1.active == TRUE) {
        
        dot_y = 70.0;
        
        if (dialog->rhythm.pos1.value < 2 || 
            dialog->rhythm.pos1.value > 2) {
            rhythm_head(dialog, 20.0, TRUE);
        }
        else {
            rhythm_head(dialog, 20.0, FALSE);
        }
        
        if (dialog->rhythm.pos1.value >= 2) {
            
            dot_y = 230.0;
            rhythm_vert_line(dialog, 20.0, 230.0);
        }
        
        if (dialog->rhythm.pos1.value > 4)    {
            if (dialog->rhythm.pos2.value > 4 && dialog->rhythm.pos2.active == TRUE)    {
                rhythm_horiz_line(dialog, 20.0, 230.0, TRUE, TRUE, 1);
                flat = TRUE;
              }
              else    {
                rhythm_horiz_line(dialog, 20.0, 230.0, FALSE, FALSE, 1);
              }
        }
        if (dialog->rhythm.pos1.value > 8)    {
            dot_y = 210.0;
            
            if (dialog->rhythm.pos2.value > 8 && dialog->rhythm.pos2.active == TRUE)    {
                rhythm_horiz_line(dialog, 20.0, 210.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (flat) {
                    rhythm_horiz_line(dialog, 20.0, 210.0, FALSE, TRUE, 1);
                }
                else {
                    rhythm_horiz_line(dialog, 20.0, 210.0, FALSE, FALSE, 1);
                }
                
            }
        }
        if (dialog->rhythm.pos1.value > 16)    {
            dot_y = 190.0;
            
            if (dialog->rhythm.pos2.value > 16 && dialog->rhythm.pos2.active == TRUE)    {
                rhythm_horiz_line(dialog, 20.0, 190.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else {
                if (flat) {
                    rhythm_horiz_line(dialog, 20.0, 190.0, FALSE, TRUE, 1);
                }
                else {
                    rhythm_horiz_line(dialog, 20.0, 190.0, FALSE, FALSE, 1);
                }
                    
            }
        }
        if (dialog->rhythm.pos1.dotted == TRUE)    {
            if (dot_y == 70.0) {
                rhythm_dot(dialog, 65.0, dot_y);
            }
            else {
                if (!flat) dot_y -= 3.0;
                    
                rhythm_dot(dialog, 20.0, dot_y);
            }
        }
    }
    
    if (dialog->rhythm.pos2.active == TRUE)    {
        
        dot_y = 70.0;
        flat = FALSE;
        
        if (dialog->rhythm.pos2.value < 2 || 
            dialog->rhythm.pos2.value > 2) {
            rhythm_head(dialog, 160.0, TRUE);
        }
        else {
            rhythm_head(dialog, 160.0, FALSE);
        }
        
        if (dialog->rhythm.pos2.value >= 2) {
            dot_y = 230.0;
            rhythm_vert_line(dialog, 160.0, 230.0);
        }
        
        
        
        if (dialog->rhythm.pos2.value > 4)    {
            if (dialog->rhythm.pos3.value > 4 && dialog->rhythm.pos3.active == TRUE)    {
                rhythm_horiz_line(dialog, 160.0, 230.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (dialog->rhythm.pos1.active == FALSE || dialog->rhythm.pos1.value <= 4)    {
                      rhythm_horiz_line(dialog, 160.0, 230.0, FALSE, FALSE, 1);
                }
            }
        }
        if (dialog->rhythm.pos2.value > 8)    {
            dot_y = 210.0;
            
            if (dialog->rhythm.pos3.value > 8 && dialog->rhythm.pos3.active == TRUE)    {
                rhythm_horiz_line(dialog, 160.0, 210.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (dialog->rhythm.pos1.active == FALSE || dialog->rhythm.pos1.value <= 4)    {
                    if (flat) {  
                        rhythm_horiz_line(dialog, 160.0, 210.0, FALSE, TRUE, 1);
                    }
                    else {
                        rhythm_horiz_line(dialog, 160.0, 210.0, FALSE, FALSE, 1);
                    }
                }
                else {
                    if (dialog->rhythm.pos1.value <= 8) {
                        rhythm_horiz_line(dialog, 160.0, 210.0, FALSE, TRUE, -1);
                    }
                }
            }
        }
        if (dialog->rhythm.pos2.value > 16)    {
            dot_y = 190.0;
            
            if (dialog->rhythm.pos3.value > 16 && dialog->rhythm.pos3.active == TRUE)    {
                rhythm_horiz_line(dialog, 160.0, 190.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (dialog->rhythm.pos1.active == FALSE || dialog->rhythm.pos1.value <= 4)    {
                    if (flat) {
                        rhythm_horiz_line(dialog, 160.0, 190.0, FALSE, TRUE, 1);
                    }
                    else {
                        rhythm_horiz_line(dialog, 160.0, 190.0, FALSE, FALSE, 1);
                    }
                }
                else {
                    if (dialog->rhythm.pos1.value <= 16) {
                        if (dialog->rhythm.pos3.value <= 4 || 
                            dialog->rhythm.pos3.active == FALSE) {
                            rhythm_horiz_line(dialog, 160.0, 190.0, FALSE, TRUE, -1);
                        }
                        else {
                            rhythm_horiz_line(dialog, 160.0, 190.0, FALSE, TRUE, 1);
                        }
                    }
                }
            }
        }
        if (dialog->rhythm.pos2.dotted == TRUE)    {
            if (dot_y == 70.0) {
                rhythm_dot(dialog, 205.0, dot_y);
            }
            else {
                if (!flat) dot_y -= 3.0;
                rhythm_dot(dialog, 160.0, dot_y);
            }
        }
    }
    
    if (dialog->rhythm.pos3.active == TRUE)    {
        flat = FALSE;
        dot_y = 70.0;
        
        if (dialog->rhythm.pos3.value < 2 || 
            dialog->rhythm.pos3.value > 2) {
            rhythm_head(dialog, 300.0, TRUE);
        }
        else {
            rhythm_head(dialog, 300.0, FALSE);
        }
        
        if (dialog->rhythm.pos3.value >= 2) {
            dot_y = 230.0;
            
            rhythm_vert_line(dialog, 300.0, 230.0);
        }
        
        if (dialog->rhythm.pos3.value > 4)    {
            if (dialog->rhythm.pos4.value > 4 && dialog->rhythm.pos4.active == TRUE)    {
                rhythm_horiz_line(dialog, 300.0, 230.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (dialog->rhythm.pos2.active == FALSE || dialog->rhythm.pos2.value <= 4)    {
                      rhythm_horiz_line(dialog, 300.0, 230.0, FALSE, FALSE, 1);
                }
            }
        }
        if (dialog->rhythm.pos3.value > 8)    {
            dot_y = 210.0;
            
            if (dialog->rhythm.pos4.value > 8 && dialog->rhythm.pos4.active == TRUE)    {
                rhythm_horiz_line(dialog, 300.0, 210.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else    {
                if (dialog->rhythm.pos2.active == FALSE || dialog->rhythm.pos2.value <= 4)    {
                    if (flat) {
                        rhythm_horiz_line(dialog, 300.0, 210.0, FALSE, TRUE, 1);
                    }
                    else {
                        rhythm_horiz_line(dialog, 300.0, 210.0, FALSE, FALSE, 1);
                    }
                }
                else {
                    if (dialog->rhythm.pos2.value <= 8) {
                        rhythm_horiz_line(dialog, 300.0, 210.0, FALSE, TRUE, -1);
                    }
                }
            }
        }
        if (dialog->rhythm.pos3.value > 16)    {
            dot_y = 190.0;
            
            if (dialog->rhythm.pos4.value > 16 && dialog->rhythm.pos4.active == TRUE)    {
                rhythm_horiz_line(dialog, 300.0, 190.0, TRUE, TRUE, 1);
                flat = TRUE;
            }
            else {
                if (dialog->rhythm.pos2.active == FALSE || dialog->rhythm.pos2.value <= 4)    {
                    if (flat) {
                        rhythm_horiz_line(dialog, 300.0, 190.0, FALSE, TRUE, 1);
                    }
                    else {
                        rhythm_horiz_line(dialog, 300.0, 190.0, FALSE, FALSE, 1);
                    }
                }
                else {
                    if (dialog->rhythm.pos2.value <= 16) {
                        if (dialog->rhythm.pos4.value <= 4 || 
                            dialog->rhythm.pos4.active == FALSE) {
                            rhythm_horiz_line(dialog, 300.0, 190.0, FALSE, TRUE, -1);
                        }
                        else {
                            rhythm_horiz_line(dialog, 300.0, 190.0, FALSE, TRUE, 1);
                        }
                    }
                }
            }
        }
        if (dialog->rhythm.pos3.dotted == TRUE)    {
            if (dot_y == 70.0) {
                rhythm_dot(dialog, 345.0, dot_y);
            }
            else {
                if (!flat) dot_y -= 3.0;
                rhythm_dot(dialog, 300.0, dot_y);
            }
        }
    }
    
    if (dialog->rhythm.pos4.active == TRUE)    {
        dot_y = 70.0;
        
        if (dialog->rhythm.pos4.value < 2 || 
            dialog->rhythm.pos4.value > 2) {
            rhythm_head(dialog, 440.0, TRUE);
        }
        else {
            rhythm_head(dialog, 440.0, FALSE);
        }
        
        if (dialog->rhythm.pos4.value >= 2) {
            dot_y = 230.0;
            
            rhythm_vert_line(dialog, 440.0, 230.0);
        }
        
        if (dialog->rhythm.pos4.value > 4) {
            if (dialog->rhythm.pos3.value <= 4 || dialog->rhythm.pos3.active == FALSE) {
                rhythm_horiz_line(dialog, 440.0, 230.0, FALSE, FALSE, 1);
            }
        }
        
        if (dialog->rhythm.pos4.value > 8) {
            
            if (dialog->rhythm.pos3.value <= 4 || dialog->rhythm.pos3.active == FALSE) {
                dot_y = 210.0;
                rhythm_horiz_line(dialog, 440.0, 210.0, FALSE, FALSE, 1);
            }
            else {
                if (dialog->rhythm.pos3.value <= 8) {
                    rhythm_horiz_line(dialog, 440.0, 210.0, FALSE, TRUE, -1);
                }
            }
        }
        
        if (dialog->rhythm.pos4.value > 16) {
            if (dialog->rhythm.pos3.value <= 4 || dialog->rhythm.pos3.active == FALSE) {
                dot_y = 190.0;
                rhythm_horiz_line(dialog, 440.0, 190.0, FALSE, FALSE, 1);
            }
            else {
                if (dialog->rhythm.pos3.value <= 16) {
                    rhythm_horiz_line(dialog, 440.0, 190.0, FALSE, TRUE, -1);
                }
            }
        }
                     
        if (dialog->rhythm.pos4.dotted == TRUE)    {
            if (dot_y == 70.0) {
                rhythm_dot(dialog, 485.0, dot_y);
            }
            else {
                if (!flat) dot_y -= 3.0;
                rhythm_dot(dialog, 440.0, dot_y);
            }
        }
    }
    
}
     
static gint
get_highlight_pos(gdouble x1)
{
    if (x1 > 21.0  && x1 <= 161.0) return 2;
    if (x1 > 161.0 && x1 <= 301.0) return 3;
    if (x1 > 301.0 && x1 <= 441.0) return 4;
    return 1;
}

static void
move_popup_highlights(GtabDialogRhythm *dialog, gint key)
{
    double x1, y1, x2, y2;
    
    gnome_canvas_item_get_bounds(dialog->highlights, &x1, &y1, &x2, &y2);
    
    switch (key) {
        case GDK_Right:
            if (x1 < 439.0) {
                gnome_canvas_item_move(dialog->highlights, 140.0, 0.0);
            }
            else {
                gnome_canvas_item_move(dialog->highlights, -420.0, 0.0);
            }
                
            break;
            
        case GDK_Left:
            if (x1 > 21.0) {
                gnome_canvas_item_move(dialog->highlights, -140.0, 0.0);
            }
            else {
                gnome_canvas_item_move(dialog->highlights, 420.0, 0.0);
            }
            
            break;
        
        default:
            break;
    }
    
}

static gint
highlights_event(GnomeCanvasItem *highlights, GdkEvent *event,
                                                GtabDialogRhythm *dialog)
{
    gdouble x1, y1, x2, y2;
    
    switch (event->type) {
        case GDK_BUTTON_PRESS:
            if (event->button.button == 3) {
                gnome_canvas_item_get_bounds(dialog->highlights, &x1, &y1, &x2, &y2);
                
                dialog->menu->x = x1; 
                
                gtk_menu_popup(GTK_MENU(dialog->menu->menu),
                                        NULL, NULL, NULL, NULL, 1, 0);
            }
            return TRUE;
            break;
        
        default:
            break;
    }
    
    return FALSE;
}

static gint
rhythm_popup_bg_event(GnomeCanvasItem *background, GdkEvent *event, 
                                            GtabDialogRhythm *dialog)
{
    gdouble x1, y1, x2, y2;
    gint pos;
    
    switch (event->type) {
        case GDK_KEY_PRESS:
            gnome_canvas_item_get_bounds(dialog->highlights, &x1, &y1, &x2, &y2);
            pos = get_highlight_pos(x1);
        
            switch (event->key.keyval) {

                case GDK_Return:
                    /* trigger the callback */
                
                    rhthm_response(GTK_DIALOG(dialog->dialog), 
                                    GTK_RESPONSE_OK, dialog);
               
                    break;
                
                case GDK_Escape:
                    
                    rhthm_response(GTK_DIALOG(dialog->dialog),
                                    GTK_RESPONSE_CANCEL, dialog);
                    break;
                
                case GDK_Right:
                    move_popup_highlights(dialog, GDK_Right);
                    break;
                
                case GDK_Left:
                    move_popup_highlights(dialog, GDK_Left);
                    break;
                
                case GDK_1:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_1:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 1;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_2:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_2:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 2;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_4:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_4:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 4;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_8:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_8:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 8;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_6:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_6:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 16;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_3:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                
                case GDK_KP_3:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = TRUE;
                            dialog->rhythm.pos1.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = TRUE;
                            dialog->rhythm.pos2.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = TRUE;
                            dialog->rhythm.pos3.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = TRUE;
                            dialog->rhythm.pos4.value = 32;
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                case GDK_space:
                    switch (pos) {
                        case 1:
                            dialog->rhythm.pos1.active = FALSE;
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            dialog->rhythm.pos2.active = FALSE;
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            dialog->rhythm.pos3.active = FALSE;
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            dialog->rhythm.pos4.active = FALSE;
                            move_popup_highlights(dialog, GDK_Right);
                            
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                 
                case GDK_period:
                    switch (pos) {
                        case 1:
                            if (!dialog->rhythm.pos1.dotted) {
                                dialog->rhythm.pos1.dotted = TRUE;
                            }
                            else {
                                dialog->rhythm.pos1.dotted = FALSE;
                            }
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 2:
                            if (!dialog->rhythm.pos2.dotted) {
                                dialog->rhythm.pos2.dotted = TRUE;
                            }
                            else {
                                dialog->rhythm.pos2.dotted = FALSE;
                            }
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        case 3:
                            if (!dialog->rhythm.pos3.dotted) {
                                dialog->rhythm.pos3.dotted = TRUE;
                            }
                            else {
                                dialog->rhythm.pos3.dotted = FALSE;
                            }
                            
                            move_popup_highlights(dialog, GDK_Right);
                            break;
                        default:
                            if (!dialog->rhythm.pos4.dotted) {
                                dialog->rhythm.pos4.dotted = TRUE;
                            }
                            else {
                                dialog->rhythm.pos4.dotted = FALSE;
                            }
                            move_popup_highlights(dialog, GDK_Right);
                            
                            break;
                    }
                    render_rhythm(dialog);
                    break;
                    
                default:
                    break;
            }
            return TRUE;
            break;
            
        case GDK_BUTTON_PRESS:
            
            gnome_canvas_item_get_bounds(dialog->highlights, &x1, &y1, &x2, &y2);
                
            if (event->button.x >= 20.0 && event->button.x <= 160.0) {
                gnome_canvas_item_move(dialog->highlights, -(x1 - 20.0), 0.0);
            }
            if (event->button.x >= 160.0 && event->button.x <= 300.0) {
                gnome_canvas_item_move(dialog->highlights, -(x1 - 160.0), 0.0);
            }
            if (event->button.x >= 300.0 && event->button.x <= 440.0) {
                gnome_canvas_item_move(dialog->highlights, -(x1 - 300.0), 0.0);
            }
            if (event->button.x >= 440.0 && event->button.x <= 580.0) {
                gnome_canvas_item_move(dialog->highlights, -(x1 - 440.0), 0.0);
            }
            
            
            switch (event->button.button) {
                           
                case 3:
                    gnome_canvas_item_get_bounds(dialog->highlights, &x1, &y1, &x2, &y2);
                
                    dialog->menu->x = x1; 
                
                    gtk_menu_popup(GTK_MENU(dialog->menu->menu),
                                            NULL, NULL, NULL, NULL, 1, 0);
                    break;
                
                default:
                    
                    break;
            }
                
            return TRUE;
            break;

        default:
            break;
    }
   
    return FALSE;
       
}

static void
on_blank_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = FALSE;   
            break;
        case 2:
            dialog->rhythm.pos2.active = FALSE;
            break;
        case 3:
            dialog->rhythm.pos3.active = FALSE;
            break;
        default:
            dialog->rhythm.pos4.active = FALSE;
            break;
    }
    render_rhythm(dialog);      
    
}

static void
on_whole_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm        *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 1;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 1;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 1;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 1;
            break;
    }
    render_rhythm(dialog);
}

static void
on_half_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 2;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 2;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 2;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 2;
            break;
    }
    render_rhythm(dialog);
    
    
    
}

static void
on_quarter_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 4;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 4;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 4;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 4;
            break;
    }
    render_rhythm(dialog);
    
    
    
}

static void
on_8th_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 8;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 8;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 8;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 8;
            break;
    }
    render_rhythm(dialog);
}

static void
on_16th_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 16;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 16;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 16;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 16;
            break;
    }
    render_rhythm(dialog);
}

static void
on_32nd_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            dialog->rhythm.pos1.active = TRUE;
            dialog->rhythm.pos1.value = 32;
            break;
        case 2:
            dialog->rhythm.pos2.active = TRUE;
            dialog->rhythm.pos2.value = 32;
            break;
        case 3:
            dialog->rhythm.pos3.active = TRUE;
            dialog->rhythm.pos3.value = 32;
            break;
        default:
            dialog->rhythm.pos4.active = TRUE;
            dialog->rhythm.pos4.value = 32;
            break;
    }
    render_rhythm(dialog);
}

static void
on_dot_popup_activate                  (GtkMenuItem     *menuitem,
                                            GtabDialogRhythm         *dialog)
{
    switch (get_highlight_pos(dialog->menu->x)) {
        case 1:
            if (!dialog->rhythm.pos1.dotted) {
                dialog->rhythm.pos1.dotted = TRUE;
            }
            else {
                dialog->rhythm.pos1.dotted = FALSE;
            }
            
            break;
        case 2:
            if (!dialog->rhythm.pos2.dotted) {
                dialog->rhythm.pos2.dotted = TRUE;
            }
            else {
                dialog->rhythm.pos2.dotted = FALSE;
            }
            
            break;
        case 3:
            if (!dialog->rhythm.pos3.dotted) {
                dialog->rhythm.pos3.dotted = TRUE;
            }
            else {
                dialog->rhythm.pos3.dotted = FALSE;
            }
            
            break;
        default:
            if (!dialog->rhythm.pos4.dotted) {
                dialog->rhythm.pos4.dotted = TRUE;
            }
            else {
                dialog->rhythm.pos4.dotted = FALSE;
            }
            
            break;
    }
    render_rhythm(dialog);
}

static GnomeUIInfo rhythm_popup_menu_uiinfo[] =
{
    {
        GNOME_APP_UI_ITEM, N_("(_Blank)"),
        NULL,
        (gpointer) on_blank_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1 (Whole Note)"),
        NULL,
        (gpointer) on_whole_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1/2"),
        NULL,
        (gpointer) on_half_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1/4"),
        NULL,
        (gpointer) on_quarter_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1/8"),
        NULL,
        (gpointer) on_8th_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1/16"),
        NULL,
        (gpointer) on_16th_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("1/32"),
        NULL,
        (gpointer) on_32nd_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    {
        GNOME_APP_UI_ITEM, N_("Toggle Dot"),
        NULL,
        (gpointer) on_dot_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    GNOMEUIINFO_END
};

static void
popup_destroyed (GtkObject *obj,  void **popup_pointer)
{
    if (popup_pointer != NULL)
    {
        g_free (*popup_pointer);
        *popup_pointer = NULL;
    }
}

static GtabDocPopup*
create_rhythm_popup_menu(GtabDialogRhythm *dialog)
{
    GtabDocPopup *popup;

    popup = g_new0(GtabDocPopup, 1);
    
    popup->menu = gtk_menu_new ();
    
    gtk_container_set_border_width (GTK_CONTAINER (popup->menu), 2);
    gnome_app_fill_menu_with_data (GTK_MENU_SHELL (popup->menu), rhythm_popup_menu_uiinfo,
                           NULL, FALSE, 0, dialog);
    
    g_signal_connect(G_OBJECT(popup->menu), "destroy",
                          G_CALLBACK(popup_destroyed), &popup);
    
    popup->x = 0;
    popup->y = 0;
    
    return popup;
}

GtabDialogRhythm* 
create_rhythm2_popup     (GtabDoc *doc)
{
    static GtabDialogRhythm     *dialog = NULL;
    GtkWidget         *table;
    GtkWidget         *vbox;
    GtkWidget         *label1;
    GtkWidget         *label2;
    GnomeCanvasPoints *points;
    GnomeCanvasItem   *background;
    
    dialog = g_new0(GtabDialogRhythm, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Rhythm Builder"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_CANCEL);
    
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    
    table = gtk_table_new(2, 4, FALSE);
    gtk_widget_show(table);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);
    gtk_table_set_row_spacings (GTK_TABLE (table), 6);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
    
    gtk_widget_push_colormap (gdk_rgb_get_colormap ());
    dialog->canvas = gnome_canvas_new_aa ();
    gtk_widget_pop_colormap ();
    gtk_widget_show (dialog->canvas);
    gtk_table_attach(GTK_TABLE(table), dialog->canvas, 0, 4, 0, 1,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
    gnome_canvas_set_scroll_region (GNOME_CANVAS (dialog->canvas), 0, 0, 601, 251);
    gtk_widget_set_size_request(GTK_WIDGET(dialog->canvas), 601, 251);
    
    dialog->root_group = gnome_canvas_root(GNOME_CANVAS(dialog->canvas));
    points = gnome_canvas_points_new(4);
    
    points->coords[0] = 0.0;
    points->coords[1] = 0.0;
    points->coords[2] = 600.0;
    points->coords[3] = 0.0;
    points->coords[4] = 600.0;
    points->coords[5] = 250.0;
    points->coords[6] = 0.0;
    points->coords[7] = 250.0;
    
    background = gnome_canvas_item_new(dialog->root_group,
                    gnome_canvas_polygon_get_type(),
                    "points", points,
                    "width_pixels", 1,
                    "fill_color", "white",
                    "outline_color", "black",
                    NULL);
     
    g_signal_connect(GTK_OBJECT(background), 
                        "event", 
                        (GtkSignalFunc) rhythm_popup_bg_event, 
                        dialog);
                        
    gnome_canvas_points_unref(points);
    
    gnome_canvas_item_grab_focus(background);
    
    dialog->objects = GNOME_CANVAS_GROUP(gnome_canvas_item_new(dialog->root_group,
                        gnome_canvas_group_get_type(),
                        NULL));
                            
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(rhthm_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
 
    dialog->tab_doc = doc;
    
    dialog->rhythm.pos1.active = FALSE;
    dialog->rhythm.pos1.value = 4;
    dialog->rhythm.pos1.dotted = FALSE;
    dialog->rhythm.pos2.active = FALSE;
    dialog->rhythm.pos2.value = 4;
    dialog->rhythm.pos2.dotted = FALSE;
    dialog->rhythm.pos3.active = FALSE;
    dialog->rhythm.pos3.value = 4;
    dialog->rhythm.pos3.dotted = FALSE;
    dialog->rhythm.pos4.active = FALSE;
    dialog->rhythm.pos4.value = 4;
    dialog->rhythm.pos4.dotted = FALSE;
    
    points = gnome_canvas_points_new(4);
    points->coords[0] = 20.0;
    points->coords[1] = 15.0;
    points->coords[2] = 160.0;
    points->coords[3] = 15.0;
    points->coords[4] = 160.0;
    points->coords[5] = 235.0;
    points->coords[6] = 20.0;
    points->coords[7] = 235.0;
    
    dialog->highlights = gnome_canvas_item_new(dialog->root_group,
                                gnome_canvas_polygon_get_type(),
                                "points", points,
                                "width_pixels", 2,
                                "outline_color", NULL,
                                "fill_color", "light gray",
                                NULL);
    
    g_signal_connect(GTK_OBJECT(dialog->highlights), 
                        "event", 
                        (GtkSignalFunc) highlights_event, 
                        dialog);
                        
    gnome_canvas_points_unref(points);
    
    label1 = gtk_label_new("1 = Whole Note\n2 = Half Note\n4 = Quarter Note");
    gtk_label_set_justify(GTK_LABEL(label1), GTK_JUSTIFY_LEFT);
    gtk_widget_show(label1);
    gtk_table_attach(GTK_TABLE(table), label1, 0, 1, 1, 2,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
       
    label2 = gtk_label_new("6 = Sixteenth Note\n3 = ThirtySecond Note\n. = Toggle Dotted");
    gtk_label_set_justify(GTK_LABEL(label2), GTK_JUSTIFY_LEFT);
    gtk_widget_show(label2);
    gtk_table_attach(GTK_TABLE(table), label2, 1, 2, 1, 2,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
    
    label1 = gtk_label_new("Space = Clear\nRight Click = Context Menu");
    gtk_label_set_justify(GTK_LABEL(label1), GTK_JUSTIFY_LEFT);
    gtk_widget_show(label1);
    gtk_table_attach(GTK_TABLE(table), label1, 2, 3, 1, 2,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
                    
    label2 = gtk_label_new("Return = Accept\nEscape = Cancel");
    gtk_label_set_justify(GTK_LABEL(label2), GTK_JUSTIFY_LEFT);
    gtk_widget_show(label2);
    gtk_table_attach(GTK_TABLE(table), label2, 3, 4, 1, 2,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
     
    
    dialog->menu = create_rhythm_popup_menu(dialog);
    
    return dialog;
    
}
