/*********************************************************************
 * The Shortcuts is Copyright (C) 2016-2018 Kyle Robbertze
 * African Institute for Mathematical Sciences, South Africa
 *
 * Shortcuts is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation
 *
 * Shortcuts is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Shortcuts.  If not, see <http://www.gnu.org/licenses/>.
 **********************************************************************/

const GLib = imports.gi.GLib;
const GObject = imports.gi.GObject;
const Gio = imports.gi.Gio;
const Gtk = imports.gi.Gtk;

const Lang = imports.lang;
const Gettext = imports.gettext.domain("Shortcuts");
const _ = Gettext.gettext;

const ExtensionUtils = imports.misc.extensionUtils;
const Me = ExtensionUtils.getCurrentExtension();

const Config = imports.misc.config;

let Settings;

/**
 * Initialises the preferences widget
 */
function init() {
  ExtensionUtils.initTranslations("Shortcuts");
  Settings = ExtensionUtils.getSettings("org.gnome.shell.extensions.shortcuts");
}

/**
 * Builds the preferences widget
 */
function buildPrefsWidget() {
  let current_version = Config.PACKAGE_VERSION.split(".");
  if (current_version[0] == 3) {
    let widget = new ShortcutsPrefsWidget3();
    widget.show_all();
    return widget;
  } else {
    let widget = new ShortcutsPrefsWidget4();
    return widget;
  }
}

/**
 * Describes the widget that is shown in the extension settings section of
 * GNOME tweek.
 */
const ShortcutsPrefsWidget3 = new GObject.Class({
  Name: "Shortcuts.Prefs.Widget",
  GTypeName: "ShortcutsPrefsWidget3",
  Extends: Gtk.Grid,

  /**
   * Initalises the widget
   */
  _init: function (params) {
    this.parent(params);
    this.margin = 12;
    this.row_spacing = this.column_spacing = 6;
    this.set_orientation(Gtk.Orientation.VERTICAL);

    this.customShortcutsFileCheckButton = new Gtk.CheckButton({
      label: _("Custom Shortcuts File"),
    });
    this.attach(this.customShortcutsFileCheckButton, 0, 0, 2, 1);
    Settings.bind(
      "use-custom-shortcuts",
      this.customShortcutsFileCheckButton,
      "active",
      Gio.SettingsBindFlags.DEFAULT
    );
    this.shortcutsFile = new Gtk.FileChooserButton({
      title: _("Select shortcut file"),
      action: Gtk.FileChooserAction.OPEN,
    });
    this.shortcutsFile.select_uri(
      "file://" + Settings.get_string("shortcuts-file")
    );
    let shortcutsFileFilter = new Gtk.FileFilter();
    this.shortcutsFile.set_filter(shortcutsFileFilter);
    shortcutsFileFilter.add_mime_type("application/json");
    this.shortcutsFile.connect(
      "selection_changed",
      Lang.bind(this, function (sw, data) {
        let path = String(this.shortcutsFile.get_uri()).slice(7);
        Settings.set_string("shortcuts-file", path);
      })
    );
    this.attach(this.shortcutsFile, 3, 0, 1, 1);

    let showIconCheckButton = new Gtk.CheckButton({
      label: _("Show tray icon"),
      margin_top: 6,
    });
    Settings.bind(
      "show-icon",
      showIconCheckButton,
      "active",
      Gio.SettingsBindFlags.DEFAULT
    );
    this.attach(showIconCheckButton, 0, 1, 2, 1);
  },
});

const ShortcutsPrefsWidget4 = GObject.registerClass(
  {
    Name: "Shortcuts.Prefs.Widget4",
    GTypeName: "ShortcutsPrefsWidget4",
    Template: Me.dir.get_child("prefs.ui").get_uri(),
    InternalChildren: [
      "file_chooser",
      "file_chooser_button",
      "file_chooser_chkbutcusfile",
      "file_chooser_chkbuttrayicon",
      "file_chooser_chkbuttransp",
    ],
  },

  class ShortcutsPrefsWidget4 extends Gtk.Box {
    _init(params = {}) {
      super._init(params);

      this._file_chooser_chkbutcusfile.set_active(
        Settings.get_boolean("use-custom-shortcuts")
      );

      if (Settings.get_boolean("use-custom-shortcuts") == true) {
        this._file_chooser_button.label =
          "file://" + Settings.get_string("shortcuts-file");
      }

      this._file_chooser_chkbuttrayicon.set_active(
        Settings.get_boolean("show-icon")
      );

      this._file_chooser_chkbuttransp.set_active(
        Settings.get_boolean("use-transparency")
      );

      this.append(
        buildRange(
          "visibility",
          [0, 100, 1, 70],
          _("Visibility"),
          _("Visibility in percent")
        )
      );
    }

    _onChkCusFileClicked(chkbox) {
      let parent = chkbox.get_root();

      Settings.set_boolean("use-custom-shortcuts", chkbox.get_active());
    }

    _onChkTrayIconClicked(chkbox) {
      let parent = chkbox.get_root();
      Settings.set_boolean("show-icon", chkbox.get_active());
    }

    _onBtnClicked(btn) {
      let parent = btn.get_root();
      this._file_chooser.set_transient_for(parent);

      let shortcutsFileFilter = new Gtk.FileFilter();
      this._file_chooser.set_filter(shortcutsFileFilter);
      shortcutsFileFilter.add_mime_type("application/json");

      this._file_chooser.title = _("Select shortcut file");

      this._file_chooser.show();
    }

    _onChkTransparencyClicked(chkbox) {
      let parent = chkbox.get_root();
      Settings.set_boolean("use-transparency", chkbox.get_active());
    }

    _onFileChooserResponse(native, response) {
      if (response !== Gtk.ResponseType.ACCEPT) {
        return;
      }
      let fileURI = native.get_file().get_uri();
      this._file_chooser_button.set_label(fileURI);
      let fileURI2 = fileURI.replace("file://", "");
      Settings.set_string("shortcuts-file", fileURI2);
    }
  }
);

function buildRange(key, values, labeltext, tooltip) {
  let [min, max, step, defv] = values;
  let hbox = new Gtk.Box({
    orientation: Gtk.Orientation.HORIZONTAL,
    spacing: 10,
  });

  let label = new Gtk.Label({ label: labeltext, xalign: 0 });

  let range = Gtk.Scale.new_with_range(
    Gtk.Orientation.HORIZONTAL,
    min,
    max,
    step
  );

  range.tooltip = tooltip;
  range.set_value(Settings.get_int(key));
  range.set_draw_value(true);
  range.add_mark(defv, Gtk.PositionType.BOTTOM, null);
  range.set_size_request(200, -1);

  range.connect("value-changed", function (slider) {
    Settings.set_int(key, slider.get_value());
  });

  hbox.append(label);
  hbox.append(range);

  return hbox;
}
