/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2005 Imendio AB
 *
 * The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * Recent File Storage,
 * see http://freedesktop.org/Standards/recent-file-spec/
 *
 * This code is taken from libegg and has been adapted to the GIMP needs.
 * The original author is James Willcox <jwillcox@cs.indiana.edu>,
 * responsible for bugs in this version is Sven Neumann <sven@gimp.org>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/HTMLparser.h>

#include "lb-bookmark-list.h"
#include "lb-utils.h"


typedef struct {
        xmlParserCtxtPtr  xmlctxt;
        gint              res;
        gchar             chars[10];

	LbBookmark       *current_bookmark;
	GList            *bookmarks;
} HtmlParser;


static void
html_characters (gpointer       ctx,
		 const xmlChar *ch,
		 gint           len)
{
	HtmlParser *parser = ctx;

	if (parser->current_bookmark) {
		gchar *chars = g_strndup (ch, len);

		if (parser->current_bookmark->name) {
			gchar *tmp = g_strconcat (parser->current_bookmark->name,
						  chars, NULL);
			g_free (parser->current_bookmark->name);
			parser->current_bookmark->name = tmp;
		} else {
			parser->current_bookmark->name = g_strndup (ch, len);
		}
	}
}

static void
html_startElement (gpointer        ctx,
		   const xmlChar  *name,
		   const xmlChar **atts)
{
        HtmlParser *parser = ctx;

	if (! g_ascii_strcasecmp (name, "a")) {
		do {
			const xmlChar *att   = NULL;
			const xmlChar *value = NULL;

			g_assert (parser->current_bookmark == NULL);

			if (*atts) att   = *atts++;
			if (*atts) value = *atts++;

			if (att && value &&
			    ! g_ascii_strcasecmp (att, "href")) {

				/* FIXME: If we encounter "%s" in the
				   url, we ignore the bookmark. This
				   is because %s is used for Mozilla
				   smart bookmarks. It would be nice
				   to be able to enter smart bookmarks
				   directly sometime */

				if (strstr (value, "%s") != 0)
					break;

				parser->current_bookmark = g_new0 (LbBookmark, 1);
				parser->current_bookmark->href = g_strdup (value);

				parser->bookmarks =
					g_list_prepend (parser->bookmarks,
							parser->current_bookmark);

				break;
			}
		} while (*atts);
	}
}

static void
html_endElement (gpointer       ctx,
		 const xmlChar *name)
{
	HtmlParser *parser = ctx;

	if (! g_ascii_strcasecmp (name, "a")) {
		parser->current_bookmark = NULL;
	}
}

xmlSAXHandler SAXHandlerStruct = {
	NULL, /* internalSubset */
	NULL, /* isStandalone */
	NULL, /* hasInternalSubset */
	NULL, /* hasExternalSubset */
	NULL, /* resolveEntity */
	NULL, /* getEntity */
	NULL, /* entityDecl */
	NULL, /* notationDecl */
	NULL, /* attributeDecl */
	NULL, /* elementDecl */
	NULL, /* unparsedEntityDecl */
	NULL, /* setDocumentLocator */
	NULL, /* startDocument */
	NULL, /* endDocument */
	html_startElement, /* startElement */
	html_endElement, /* endElement */
	NULL, /* reference */
	html_characters, /* characters */
	NULL, /* ignorableWhitespace */
	NULL, /* processingInstruction */
	NULL, /* comment */
	NULL, /* xmlParserWarning */
	NULL, /* xmlParserError */
	NULL, /* xmlParserError */
	NULL, /* getParameterEntity */
	html_characters, /* cdataBlock; */
};

xmlSAXHandlerPtr SAXHandler = &SAXHandlerStruct;

static GList *
lb_bookmark_list_read (gint fd)
{
	HtmlParser  parser;
	GIOChannel *io;
	GIOStatus   status;
	guchar      buffer[4096];
	gsize       len = 0;
	GError     *error = NULL;

	lseek (fd, 0, SEEK_SET);

	parser.res      = 0;
	parser.chars[0] = '\0';
	parser.xmlctxt  = htmlCreatePushParserCtxt (SAXHandler, &parser,
						    parser.chars, parser.res,
						    NULL, 0);
	parser.bookmarks = NULL;
	parser.current_bookmark = NULL;

	io = g_io_channel_unix_new (fd);

	g_io_channel_set_encoding (io, "UTF-8", NULL);

	while (TRUE) {
		status = g_io_channel_read_chars (io,
						  buffer, sizeof (buffer),
						  &len, &error);

		switch (status) {
		case G_IO_STATUS_ERROR:
			goto done;
		case G_IO_STATUS_EOF:
			htmlParseChunk (parser.xmlctxt, NULL, 0, 1);
			goto done;
		case G_IO_STATUS_NORMAL:
		case G_IO_STATUS_AGAIN:
			break;
		}

		htmlParseChunk (parser.xmlctxt, buffer, len, 0);
	}

 done:

	g_io_channel_unref (io);

	xmlFreeParserCtxt (parser.xmlctxt);

	return g_list_reverse (parser.bookmarks);
}

GList *
lb_bookmark_list_get (const gchar *filename)
{
	gint   fd;
	GList *list = NULL;

	g_return_val_if_fail (filename != NULL, NULL);

	fd = open (filename, O_RDWR);

	if (fd < 0) {
		g_warning ("Could not open bookmarks file: %s", strerror (errno));
		return FALSE;
	}

	if (lb_file_lock (fd)) {
		list = lb_bookmark_list_read (fd);

		if (! lb_file_unlock (fd))
			g_warning ("Failed to unlock: %s", strerror (errno));
	} else {
		g_warning ("Failed to lock:  %s", g_strerror (errno));
		return FALSE;
	}

	close (fd);

	return list;
}

void
lb_bookmark_list_free (GList *list)
{
	GList *l;

	for (l = list; list; list = list->next) {
		LbBookmark *bookmark = list->data;

		g_free (bookmark->name);
		g_free (bookmark->href);
		g_free (bookmark);
	}
}
